// SPDX-License-Identifier: GPL-2.0-only
/*
 * debug.h - header file for debug and log messages.
 *
 * Copyright (C) 2025 Genexis AB.
 *
 * Author: anjan.chanda@genexis.eu
 */

#ifndef WIFIMNGR_DEBUG_H
#define WIFIMNGR_DEBUG_H

/* log feature categories */
enum {
	LOG_CMD,	/* command request and response */
	LOG_EVENT,	/* events */
	LOG_DEFAULT,	/* uncategorized messages */
};

/* log levels */
enum {
	LOGLEVEL_ERROR,
	LOGLEVEL_WARN,
	LOGLEVEL_INFO,
	LOGLEVEL_DEBUG,
	LOGLEVEL_TRACE,
};

#define LOG_FEATURE(x)	BIT(x)
#define LOG_FEATURE_ALL	BIT(LOG_DEFAULT) | (BIT(LOG_DEFAULT) - 1)

#define DEFAULT_LOGLEVEL	LOGLEVEL_INFO

uint32_t logfeature_to_enum(const char *s);
const char *logfeature_to_string(uint32_t e);

void restart_logging(void *opts);
void stop_logging(void);
void log_message(int feature, int level, const char *fmt, ...)
	__attribute__((__format__(__printf__, 3, 4)));

#define wm_err(f, fmt, ...)         log_message(f, 0, fmt, ## __VA_ARGS__)
#define wm_warn(f, fmt, ...)        log_message(f, 1, fmt, ## __VA_ARGS__)
#define wm_info(f, fmt, ...)        log_message(f, 2, fmt, ## __VA_ARGS__)
#define wm_dbg(f, fmt, ...)         log_message(f, 3, fmt, ## __VA_ARGS__)
#define wm_trace(f, fmt, ...)       log_message(f, 4, fmt, ## __VA_ARGS__)

#define wifimngr_err(fmt, ...)      log_message(LOG_DEFAULT, 0, fmt, ## __VA_ARGS__)
#define wifimngr_warn(fmt, ...)     log_message(LOG_DEFAULT, 1, fmt, ## __VA_ARGS__)
#define wifimngr_info(fmt, ...)     log_message(LOG_DEFAULT, 2, fmt, ## __VA_ARGS__)
#define wifimngr_dbg(fmt, ...)      log_message(LOG_DEFAULT, 3, fmt, ## __VA_ARGS__)
#define wifimngr_trace(fmt, ...)    log_message(LOG_DEFAULT, 4, fmt, ## __VA_ARGS__)

#endif /* WIFIMNGR_DEBUG_H */
