/*
 * Copyright (C) 2025 iopsys Software Solutions AB
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1
 * as published by the Free Software Foundation
 *
 *	  Author: Amin Ben Romdhane <amin.benromdhane@iopsys.eu>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libbbfdm-ubus/bbfdm-ubus.h>

#include "wifi.h"
#include "common.h"

#define SERVICE_NAME "wifidmd"
#define DEFAULT_LOG_LEVEL LOG_ERR

struct list_head dev_list;
bool mld_capable = false;
bool easymesh_enable = false;
bool device_supports_mlo = false;

/* ********** DynamicObj ********** */
DM_MAP_OBJ tDynamicObj[] = {
/* parentobj, nextobject, parameter */
{"Device.", tDeviceWiFiObj, NULL, init_wifi_legacy_module, NULL, sync_wifi_module},
{0}
};

static void usage(char *prog)
{
	fprintf(stderr, "Usage: %s [options]\n", prog);
	fprintf(stderr, "\n");
	fprintf(stderr, "options:\n");
	fprintf(stderr, "    -l  <0-7> Set the loglevel\n");
	fprintf(stderr, "    -d  Display the schema data model supported by micro-service\n");
	fprintf(stderr, "    -h  Display this help\n");
	fprintf(stderr, "\n");
}

static int init_radio_devices_info(void)
{
	struct uci_section *sec = NULL;
	struct bbfdm_ctx dm_ctx = {0};
	struct radio_info *node = NULL;
	char mlo_capable[5] = {0};

	memset(&dev_list, 0, sizeof(struct list_head));
	INIT_LIST_HEAD(&dev_list);

	memset(&dm_ctx, 0, sizeof(struct bbfdm_ctx));
	bbfdm_init_ctx(&dm_ctx);

	BBFDM_UCI_FOREACH_SECTION(&dm_ctx, "wireless", "wifi-device", sec) {
		char *dev = bbfdm_section_name(sec);
		if (strlen(dev) == 0)
			continue;

		node = (struct radio_info *)malloc(sizeof(struct radio_info));
		if (node == NULL)
			return -1;

		memset(node, 0, sizeof(struct radio_info));
		snprintf(node->radio_name, sizeof(node->radio_name), "%s", dev);
		bbfdm_uci_get_buf(&dm_ctx, "wireless", dev, "band", "", node->band, sizeof(node->band));
		bbfdm_uci_get_buf(&dm_ctx, "wireless", dev, "mlo_capable", "0", mlo_capable, sizeof(mlo_capable));

		INIT_LIST_HEAD(&node->list);
		list_add_tail(&node->list, &dev_list);
	}

	bbfdm_free_ctx(&dm_ctx);
	device_supports_mlo = dmuci_string_to_boolean(mlo_capable);

	return 0;
}

static void restore_default_options(void)
{
	struct uci_section *sec = NULL;
	struct bbfdm_ctx dm_ctx = {0};

	memset(&dm_ctx, 0, sizeof(struct bbfdm_ctx));
	bbfdm_init_ctx(&dm_ctx);

	BBFDM_UCI_FOREACH_SECTION(&dm_ctx, "wireless", "wifi-iface", sec) {
		char ap_mode[5] = {0};

		char *sec_name = bbfdm_section_name(sec);
		if (strlen(sec_name) == 0)
			continue;

		// Restore for multi_ap fronthaul AP sections
		bbfdm_uci_get_buf(&dm_ctx, "wireless", sec_name, "multi_ap", "", ap_mode, sizeof(ap_mode));
		if (DM_LSTRCMP(ap_mode, "2") != 0)
			continue;

		bbfdm_uci_set(&dm_ctx, "wireless", sec_name, "hidden", "");
		bbfdm_uci_set(&dm_ctx, "wireless", sec_name, "wmm", "");
		bbfdm_uci_set(&dm_ctx, "wireless", sec_name, "wmm_apsd", "");
		bbfdm_uci_set(&dm_ctx, "wireless", sec_name, "macfilter", "");
		bbfdm_uci_set(&dm_ctx, "wireless", sec_name, "maxassoc", "");
		bbfdm_uci_set(&dm_ctx, "wireless", sec_name, "isolate", "");
		bbfdm_uci_set(&dm_ctx, "wireless", sec_name, "maclist", "");
		bbfdm_uci_set(&dm_ctx, "wireless", sec_name, "acct_server", "");
		bbfdm_uci_set(&dm_ctx, "wireless", sec_name, "acct_secret", "");
		bbfdm_uci_set(&dm_ctx, "wireless", sec_name, "wps_pushbutton", "1");
	}

	bbfdm_uci_commit_package(&dm_ctx, "wireless");
	bbfdm_free_ctx(&dm_ctx);

	return;
}

static void remove_unused_mld(void)
{
	struct uci_section *sec = NULL, *s_tmp = NULL, *ap_sec = NULL, *m_sec = NULL;
	struct bbfdm_ctx dm_ctx = {0};
	struct bbfdm_ctx map_ctx = {0};

	memset(&dm_ctx, 0, sizeof(struct bbfdm_ctx));
	bbfdm_init_ctx(&dm_ctx);

	memset(&map_ctx, 0, sizeof(struct bbfdm_ctx));
	map_ctx.uci_confdir = "/etc/bbfdm/dmmap/";
	bbfdm_init_ctx(&map_ctx);

	BBFDM_UCI_FOREACH_SECTION_SAFE(&dm_ctx, "mapcontroller", "mld", s_tmp, sec) {
		char id[5] = {0};
		bool used = false;

		char *sec_name = bbfdm_section_name(sec);
		if (strlen(sec_name) == 0)
			continue;

		bbfdm_uci_get_buf(&dm_ctx, "mapcontroller", sec_name, "id", "", id, sizeof(id));
		if (DM_STRLEN(id) == 0)
			continue;

		BBFDM_UCI_FOREACH_SECTION(&dm_ctx, "mapcontroller", "ap", ap_sec) {
			char mld_id[5] = {0};

			char *ap_name = bbfdm_section_name(ap_sec);
			if (strlen(ap_name) == 0)
				continue;

			bbfdm_uci_get_buf(&dm_ctx, "mapcontroller", ap_name, "mld_id", "", mld_id, sizeof(mld_id));
			if (DM_STRCMP(id, mld_id) == 0) {
				used = true;
				break;
			}
		}

		if (used == true)
			continue;

		BBFDM_UCI_FOREACH_SECTION(&map_ctx, "WiFi", "SSID", m_sec) {
			char mld_id[5] = {0};

			char *s_name = bbfdm_section_name(m_sec);
			if (strlen(s_name) == 0)
				continue;

			bbfdm_uci_get_buf(&map_ctx, "WiFi", s_name, "mld_id", "", mld_id, sizeof(mld_id));
			if (DM_STRCMP(id, mld_id) == 0) {
				used = true;
				break;
			}
		}

		if (used == false)
			bbfdm_uci_delete_section(&dm_ctx, sec);
	}

	bbfdm_uci_commit_package(&dm_ctx, "mapcontroller");
	bbfdm_free_ctx(&dm_ctx);
	bbfdm_free_ctx(&map_ctx);

	return;
}

static void free_radio_devices_info(void)
{
	struct radio_info *node = NULL, *ntmp = NULL;

	list_for_each_entry_safe(node, ntmp, &dev_list, list) {
		list_del(&node->list);
		free(node);
	}
}

static void wifi_data_elements_cb(struct ubus_context *ctx, struct ubus_event_handler *ev,
			  const char *type, struct blob_attr *msg)
{
	BBF_DEBUG("Received WiFi DataElements event: '%s'", type);

	// Refresh References Data Base
	bbfdm_refresh_references(BBFDM_BOTH, "bbfdm."SERVICE_NAME);
}

int main(int argc, char **argv)
{
	struct bbfdm_context bbfdm_ctx = {0};
	int log_level = DEFAULT_LOG_LEVEL;
	int c = 0, dm_type = 0;
	struct ubus_event_handler ev_wifi_data_elements = {
		.cb = wifi_data_elements_cb,
	};


	while ((c = getopt(argc, argv, "hdl:")) != -1) {
		switch (c) {
		case 'l':
			log_level = (int)strtod(optarg, NULL);
			if (log_level < 0 || log_level > 7) {
				log_level = DEFAULT_LOG_LEVEL;
			}
			break;
		case 'd':
			dm_type++;
			break;
		case 'h':
			usage(argv[0]);
			return EXIT_SUCCESS;
		default:
			usage(argv[0]);
			return EXIT_FAILURE;
		}
	}

	memset(&bbfdm_ctx, 0, sizeof(struct bbfdm_context));

	bbfdm_ubus_set_service_name(&bbfdm_ctx, SERVICE_NAME);
	bbfdm_ubus_set_log_level(log_level);
	bbfdm_ubus_load_data_model(tDynamicObj);

	if (dm_type > 0) {
		int res = bbfdm_print_data_model_schema(&bbfdm_ctx, dm_type);
		exit(res);
	}

	openlog(SERVICE_NAME, LOG_CONS | LOG_PID | LOG_NDELAY, LOG_LOCAL1);

	/* check mapcontroller support to allow easymesh configuration */
	easymesh_enable = get_mapcontroller_enabled();

	if (easymesh_enable) {
		// Rename unnamed AP sections in mapcontroller
		rename_map_ap_sections();
		// Restore restricted options to default values
		restore_default_options();
		// sync mld to cleanup unused mld section (could be after FW upgrade)
		remove_unused_mld();
	}

	/* Store a mapping of radio device names and their band */
	if (0 != init_radio_devices_info())
		goto end;

	/* Radio should have mlo capability and device should have mapcontroller
	 * enabled to allow configuring MLDUnit
	 */
	if (easymesh_enable)
		mld_capable = device_supports_mlo;

	if (bbfdm_ubus_register_init(&bbfdm_ctx))
		goto out;

	ubus_register_event_handler(bbfdm_ctx.ubus_ctx, &ev_wifi_data_elements, "wifi.dataelements.*");

	BBF_DEBUG("mapcontroller: %d, mld: %d", easymesh_enable, mld_capable);

	uloop_run();

	ubus_unregister_event_handler(bbfdm_ctx.ubus_ctx, &ev_wifi_data_elements);

out:
	bbfdm_ubus_register_free(&bbfdm_ctx);

end:
	closelog();
	free_radio_devices_info();
	return 0;
}
