/*
 * Copyright (C) 2024-2025 IOPSYS Software Solutions AB. All rights reserved.
 *
 *
 * See LICENSE file for license related information.
 *
 * Author: Suvendhu Hansa <suvendhu.hansa@iopsys.eu>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "common.h"

char *get_default_key(void)
{
	char *def_key = NULL;

	json_object *json_obj = json_object_from_file(BOARD_JSON_FILE);
	if (!json_obj) {
		return dmstrdup("1234567890");
	}

	char *key = dmjson_get_value(json_obj, 5, "wlan", "defaults", "ssids", "all", "key");
	if (DM_STRLEN(key)) {
		def_key = dmstrdup(key);
	} else {
		def_key = dmstrdup("1234567890");
	}

	json_object_put(json_obj);

	return def_key;
}

char *get_default_encryption(void)
{
	char *def_encryption = NULL;

	json_object *json_obj = json_object_from_file(BOARD_JSON_FILE);
	if (!json_obj) {
		return dmstrdup("sae-mixed");
	}

	char *encryption = dmjson_get_value(json_obj, 5, "wlan", "defaults", "ssids", "all", "encryption");
	if (DM_STRLEN(encryption)) {
		def_encryption = dmstrdup(encryption);
	} else {
		def_encryption = dmstrdup("sae-mixed");
	}

	json_object_put(json_obj);

	return def_encryption;
}

char *get_device_from_band(const char *band)
{
	if (DM_STRLEN(band) == 0)
		return NULL;

	struct radio_info *node = NULL, *tmp = NULL;
	list_for_each_entry_safe(node, tmp, &dev_list, list) {
		if (DM_STRNCMP(node->band, band, strlen(band)) == 0)
			return node->radio_name;
	}

	return NULL;
}

void get_band_from_device(const char *device, char *band, int size)
{
	if (DM_STRLEN(device) == 0 || band == NULL)
		return;

	memset(band, 0, size);

	struct radio_info *node = NULL, *tmp = NULL;
	list_for_each_entry_safe(node, tmp, &dev_list, list) {
		if (DM_STRCMP(node->radio_name, device) != 0)
			continue;

		snprintf(band, size, "%c", node->band[0]);
		break;
	}

	return;
}

void remove_unused_mld_sections(void)
{
	char *cur_mld = NULL;
	struct uci_section *ssid_s = NULL, *s = NULL, *stmp = NULL;

	if (mld_capable == false)
		return;

	uci_foreach_sections_safe("mapcontroller", "mld", s, stmp) {
		dmuci_get_value_by_section_string(s, "id", &cur_mld);

		ssid_s = get_dup_section_in_dmmap_opt("WiFi", "SSID", "mld_id", cur_mld);
		if (ssid_s != NULL)
			continue;

		// No SSID has assigned this mld unit so delete
		dmuci_delete_by_section(s, NULL, NULL);
	}
}

bool get_mapcontroller_enabled(void)
{
	char map_enable[10] = {0};

	BBFDM_UCI_GET("mapcontroller", "controller", "enabled", "0", map_enable, sizeof(map_enable));
	return dmuci_string_to_boolean(map_enable);
}
