/*
 * Copyright (C) 2023-2024 IOPSYS Software Solutions AB. All rights reserved.
 *
 *
 * See LICENSE file for license related information.
 *
 * Author: Amin Ben Romdhane <amin.benromdhane@iopsys.eu>
 * Author: Saurabh Verma <saurabh.verma@iopsys.eu>
 *
 */

#include <math.h>

#include "wifi.h"
#include "common.h"

#define MAX_POWER_INDEX 64
#define UBUS_OBJ_LEN 32

struct power_percent_map
{
	int sup_percent;
	int neg_power;
};

static char *MFP_Config[] = {"Disabled", "Optional", "Required", NULL};
static struct power_percent_map txpower_map[] = { {-1,-1}, {25,6}, {50,3}, {75,1}, {100,0} };

struct radio_obj
{
	bool scan_done;
	char obj_name[15];
};

struct radio_scan_args
{
	struct radio_obj *radio;
	uint32_t obj_count;
	struct blob_attr *msg;
};

/**************************************************************************
* INIT
***************************************************************************/
static void remove_mld_unit(char *old_mld)
{
	if (DM_STRLEN(old_mld) == 0)
		return;

	struct uci_section *ss = get_dup_section_in_dmmap_opt("dmmap_wireless", "ssid", "mld_id", old_mld);
	if (ss != NULL)
		return;

	// No SSID has assigned this mld unit so delete
	struct uci_section *map_mld = get_dup_section_in_config_opt("mapcontroller", "mld", "id", old_mld);
	if (map_mld)
		dmuci_delete_by_section(map_mld, NULL, NULL);
}

static bool check_backhaul_mld(char *mld_unit)
{
	struct uci_section *map_sec = NULL;
	char *type = NULL;

	if (mld_unit == NULL)
		return false;

	map_sec = get_dup_section_in_config_opt("mapcontroller", "mld", "id", mld_unit);
	if (map_sec) {
		dmuci_get_value_by_section_string(map_sec, "type", &type);
		if (DM_STRCMP(type, "backhaul") == 0)
			return true;
	}

	return false;
}

static void create_mld_unit_mapcontroller(char *mld_id, char *ssid, char *key)
{
	struct uci_section *stmp = NULL;

	if (mld_id == NULL || ssid == NULL || key == NULL)
		return;

	// Need to create new mld section in mapcontroller
	dmuci_add_section("mapcontroller", "mld", &stmp);
	dmuci_set_value_by_section(stmp, "id", mld_id);
	dmuci_set_value_by_section(stmp, "ssid", ssid);
	dmuci_set_value_by_section(stmp, "key", key);
	dmuci_set_value_by_section(stmp, "type", "fronthaul");
}

static void configure_mld_section(const char *mld, const char *option, char *value)
{
	if (DM_STRLEN(mld) == 0 || option == NULL || value == NULL)
		return;

	struct uci_section *map_sec = get_dup_section_in_config_opt("mapcontroller", "mld", "id", mld);
	if (map_sec)
		dmuci_set_value_by_section(map_sec, option, value);
}

static char *get_encryption_mode(char *encryption)
{
	char *pch = NULL, *spch = NULL;
	char encryption_list[256] = {0};
	char buf[16] = {0};
	unsigned int pos = 0;

	if (DM_STRLEN(encryption) == 0)
		return dmstrdup("");

	snprintf(encryption_list, sizeof(encryption_list), "%s", encryption);

	for (pch = strtok_r(encryption_list, ",", &spch);
			pch != NULL;
			pch = strtok_r(NULL, ",", &spch)) {

		if (DM_STRCMP(pch, "CCMP") == 0 ||
			DM_STRCMP(pch, "CCMP256") == 0 ||
			DM_STRCMP(pch, "GCMP") == 0 ||
			DM_STRCMP(pch, "GCMP256") == 0 ||
			DM_STRCMP(pch, "CMAC") == 0 ||
			DM_STRCMP(pch, "CMAC256") == 0 ||
			DM_STRCMP(pch, "GMAC") == 0 ||
			DM_STRCMP(pch, "GMAC256") == 0)
			pos += snprintf(&buf[pos], sizeof(buf) - pos, "%s,", "AES");
		else if (DM_STRCMP(pch, "TKIP") == 0)
			pos += snprintf(&buf[pos], sizeof(buf) - pos, "%s,", "TKIP");
	}

	if (pos > 0) {
		buf[pos - 1] = 0; // Remove trailing comma
		return dmstrdup(buf);
	}

	return dmstrdup("");
}

/*************************************************************
* COMMON FUNCTIONS
**************************************************************/
static char *get_radio_option_nocache(const char *device_name, const char *option)
{
	json_object *res = NULL;
	char object[UBUS_OBJ_LEN];

	snprintf(object, sizeof(object), "wifi.radio.%s", device_name);
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);

	return (res) ? dmjson_get_value(res, 1, option) : "";
}

static char *get_data_model_mode(const char *ubus_mode)
{
	if (DM_LSTRCMP(ubus_mode, "WEP64") == 0)
		return "WEP-64";
	else if (DM_LSTRCMP(ubus_mode, "WEP128") == 0)
		return "WEP-128";
	else if (DM_LSTRCMP(ubus_mode, "WPAPSK") == 0)
		return "WPA-Personal";
	else if (DM_LSTRCMP(ubus_mode, "WPA2PSK") == 0)
		return "WPA2-Personal";
	else if (DM_LSTRCMP(ubus_mode, "WPA3PSK") == 0)
		return "WPA3-Personal";
	else if (DM_LSTRCMP(ubus_mode, "WPAPSK+WPA2PSK") == 0)
		return "WPA-WPA2-Personal";
	else if (DM_LSTRCMP(ubus_mode, "WPA2PSK+WPA3PSK") == 0)
		return "WPA3-Personal-Transition";
	else if (DM_LSTRCMP(ubus_mode, "WPA") == 0)
		return "WPA-Enterprise";
	else if (DM_LSTRCMP(ubus_mode, "WPA2") == 0)
		return "WPA2-Enterprise";
	else if (DM_LSTRCMP(ubus_mode, "WPA3") == 0)
		return "WPA3-Enterprise";
	else if (DM_LSTRCMP(ubus_mode, "WPA+WPA2") == 0)
		return "WPA-WPA2-Enterprise";
	else
		return "None";
}

static char *get_data_model_standard(char *standard)
{
	char *p = DM_LSTRSTR(standard, "802.11");
	if (p) {
		char *res = dmstrdup(p + strlen("802.11"));
		replace_char(res, '/', ',');
		return res;
	}

	return standard;
}

static int get_sub_band(const char *cc)
{
	int sub_band = 0;

	if (cc == NULL)
		return 0;

	switch (DM_STRTOL(cc)) {
		case 31:
		case 95:
		case 159:
			sub_band = 1;
			break;
		case 63:
		case 127:
		case 191:
			sub_band = 2;
			break;
		default:
			break;
	}

	return sub_band;
}

// Use 20MHz as default value, in case of null or 0
static char *get_data_model_band(const char *bandwidth, const char *ccfs0, const char *ccfs1)
{
	char *band = NULL;
	int sub_band = 0;
	const char *tmp = "20";

	if (DM_STRLEN(bandwidth) == 0) {
		tmp = "20";
	} else {
		if (DM_STRCMP(bandwidth, "8080") == 0) {
			tmp = "80+80";
		} else if (DM_LSTRCMP(bandwidth, "320") != 0) {
			tmp = bandwidth;
		} else {
			sub_band = get_sub_band(ccfs0);
			if (!sub_band) {
				sub_band = get_sub_band(ccfs1);
			}

			if (!sub_band) { // failed to get sub-band
				tmp = bandwidth;
			}
		}
	}

	if (sub_band) {
		dmasprintf(&band, "320MHz-%d", sub_band);
	} else {
		dmasprintf(&band, "%sMHz", tmp);
	}

	return band;
}

static int get_supported_modes(const char *ubus_method, const char *ifname, char **value)
{
	const char *dm_default_modes_supported = "None,WEP-64,WEP-128,WPA-Personal,WPA2-Personal,WPA3-Personal,WPA-WPA2-Personal,WPA3-Personal-Transition,WPA-Enterprise,WPA2-Enterprise,WPA3-Enterprise,WPA-WPA2-Enterprise";
	const char *dm_wifi_driver_modes_supported = "NONE,WEP64,WEP128,WPAPSK,WPA2PSK,WPA3PSK,WPAPSK+WPA2PSK,WPA2PSK+WPA3PSK,WPA,WPA2,WPA3,WPA+WPA2";
	json_object *res = NULL, *supported_modes = NULL;
	char list_modes[256], object[32], *mode = NULL;
	unsigned pos = 0, idx = 0;

	snprintf(object, sizeof(object), "%s.%s", ubus_method, ifname);
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);
	DM_ASSERT(res, *value = dmstrdup(dm_default_modes_supported));

	list_modes[0] = 0;
	dmjson_foreach_value_in_array(res, supported_modes, mode, idx, 1, "supp_security") {
		if (!DM_STRSTR(dm_wifi_driver_modes_supported, mode))
			continue;

		pos += snprintf(&list_modes[pos], sizeof(list_modes) - pos, "%s,", get_data_model_mode(mode));
	}

	/* cut tailing ',' */
	if (pos)
		list_modes[pos - 1] = 0;

	*value = (*list_modes != '\0') ? dmstrdup(list_modes) : dmstrdup(dm_default_modes_supported);

	return 0;
}

static char *get_security_mode(struct uci_section *section)
{
	char *encryption = NULL;
	char *ptrch = NULL;

	dmuci_get_value_by_section_string(section, "encryption", &encryption);
	if (!encryption || *encryption == '\0')
		return "None";

	/*Here the encryption type and the cipher are seperated*/
	ptrch  = DM_STRCHR(encryption, '+');
	if (ptrch)
		*ptrch = '\0';

	if (DM_LSTRSTR(encryption, "wep")) {
		char *key_index = NULL, *key = NULL;

		dmuci_get_value_by_section_string(section, "key", &key_index);
		if (key_index && (*key_index) > '0' && (*key_index) < '5' && *(key_index+1) == '\0') {
			char buf[16];

			snprintf(buf, sizeof(buf), "key%s", key_index);
			dmuci_get_value_by_section_string(section, buf, &key);
		}
		return (key && DM_STRLEN(key) == 10) ? "WEP-64" : "WEP-128";
	}
	else if (DM_LSTRNCMP(encryption, "psk-mixed", 9) == 0)
		return "WPA-WPA2-Personal";
	else if (DM_LSTRNCMP(encryption, "psk2", 4) == 0)
		return "WPA2-Personal";
	else if (DM_LSTRNCMP(encryption, "psk", 3) == 0)
		return "WPA-Personal";
	else if (DM_LSTRCMP(encryption, "sae") == 0)
		return "WPA3-Personal";
	else if (DM_LSTRCMP(encryption, "sae-mixed") == 0)
		return "WPA3-Personal-Transition";
	else if (DM_LSTRNCMP(encryption, "wpa-mixed", 9) == 0)
		return "WPA-WPA2-Enterprise";
	else if (DM_LSTRNCMP(encryption, "wpa2", 4) == 0)
		return "WPA2-Enterprise";
	else if (DM_LSTRNCMP(encryption, "wpa3", 4) == 0)
		return "WPA3-Enterprise";
	else if (DM_LSTRNCMP(encryption, "wpa", 3) == 0)
		return "WPA-Enterprise";
	else
		return "None";
}

static int get_bandwidth_from_config(struct uci_section *config_sec)
{
	char *htmode = NULL;
	int bw = 20; // default

	dmuci_get_value_by_section_string(config_sec, "htmode", &htmode);
	if (DM_STRLEN(htmode)) {
		int parsed_bw = 0;

		if (sscanf(htmode, "%*[^0-9]%d", &parsed_bw) == 1 && parsed_bw > 0)
			bw = parsed_bw;
	}

	return bw;
}

static void configure_map_sections(const char *mld_id, const char *option, const char *value, bool sync_dmmap)
{
	struct uci_section *s = NULL, *s_map = NULL;
	char *ap_sec_name = NULL;

	if (DM_STRLEN(mld_id) == 0)
		return;

	uci_path_foreach_option_eq(bbfdm, "dmmap_wireless", "ssid", "mld_id", mld_id, s){
		if (sync_dmmap == true)
			dmuci_set_value_by_section(s, option, value);

		dmuci_get_value_by_section_string(s, "ap_section_name", &ap_sec_name);
		s_map = get_origin_section_from_config("mapcontroller", "ap", ap_sec_name);
		if (s_map != NULL) {
			dmuci_set_value_by_section(s_map, option, value);
		}
	}
}

static bool multi_ap_type_backhaul(struct uci_section *config_section)
{
	bool ret = false;
	char *multi_ap = NULL;
	char *type = NULL;

	 /* multi_ap              type
	  * 1                     Backhaul
	  * 2                     Fronthaul
	  * 3                     Combined
	  * Others                None
	  */
	dmuci_get_value_by_section_string(config_section, "multi_ap", &multi_ap);
	dmuci_get_value_by_section_string(config_section, "type", &type);
	if (DM_STRCMP(multi_ap, "1") == 0 || DM_STRCMP(type, "backhaul") == 0)
		ret = true;

	return ret;
}

static bool check_slo_ap(struct uci_section *sec)
{
	char *m_ap = NULL;
	char *type = NULL;

	dmuci_get_value_by_section_string(sec, "type", &type);
	dmuci_get_value_by_section_string(sec, "multi_ap", &m_ap);

	if (DM_STRLEN(type) == 0 &&
	    (DM_STRLEN(m_ap) == 0 || DM_STRCMP(m_ap, "0") == 0))
		return true;

	return false;
}

static bool duplicate_ssid_exists(const char *ssid, const char *device)
{
	if (DM_STRLEN(ssid) == 0 || DM_STRLEN(device) == 0)
		return false;

	struct uci_section *s = NULL;
	uci_path_foreach_option_eq(bbfdm, "dmmap_wireless", "ssid", "ssid", ssid, s) {
		char *dev = NULL;

		dmuci_get_value_by_section_string(s, "device", &dev);
		if (DM_STRCMP(device, dev) == 0) {
			return true;
		}
	}

	return false;
}

static bool backhaul_ssid_exist(const char *ssid)
{
	struct uci_section *s = NULL;
	if (DM_STRLEN(ssid) == 0)
		return false;

	if (easymesh_enable == true)
		s = get_dup_section_in_config_opt("mapcontroller", "ap", "ssid", ssid);
	else
		s = get_dup_section_in_config_opt("wireless", "wifi-iface", "ssid", ssid);

	if (s != NULL && multi_ap_type_backhaul(s))
		return true;

	return false;
}

static bool easymesh_ssid_exist(const char *ssid)
{
	if (DM_STRLEN(ssid) == 0)
		return false;

	if (easymesh_enable == true) {
		struct uci_section *s = get_dup_section_in_config_opt("mapcontroller", "ap", "ssid", ssid);
		if (s != NULL)
			return true;
	} else {
		struct uci_section *s = get_dup_section_in_config_opt("wireless", "wifi-iface", "ssid", ssid);
		if (s != NULL && !check_slo_ap(s))
			return true;
	}

	return false;
}

static bool slo_ssid_exist(const char *ssid)
{
	if (DM_STRLEN(ssid) == 0)
		return false;

	struct uci_section *s = NULL;
	uci_foreach_option_eq("wireless", "wifi-iface", "ssid", ssid, s) {
		char *multi_ap = NULL;

		dmuci_get_value_by_section_string(s, "multi_ap", &multi_ap);
		if (DM_STRLEN(multi_ap) == 0 || DM_STRCMP(multi_ap, "0") == 0) {
			return true;
		}
	}

	return false;
}

static int set_wireless_uci_ap_section(void *data, const char *option, const char *value)
{
	struct uci_section *s = NULL, *config_s = NULL, *dmmap_s = NULL;
	char *wireless_sec_name = NULL;
	char *device = NULL;
	char *ssid = NULL;
	char *band = NULL;

	config_s = ((struct dm_data *)data)->config_section;
	dmmap_s = ((struct dm_data *)data)->dmmap_section;

	if (config_s == NULL)
		return -1;

	if (check_slo_ap(config_s)) {
		dmuci_set_value_by_section(config_s, option, value);
		return 0;
	}

	dmuci_get_value_by_section_string(dmmap_s, "wireless_sec_name", &wireless_sec_name);
	s = get_origin_section_from_config("wireless", "wifi-iface", wireless_sec_name);
	if (s) {
		dmuci_set_value_by_section(s, option, value);
		return 0;
	}

	dmuci_get_value_by_section_string(config_s, "ssid", &ssid);
	dmuci_get_value_by_section_string(config_s, "band", &band);
	device = get_device_from_band(band);

	if (DM_STRLEN(ssid) == 0 || DM_STRLEN(device) == 0)
		return -1;

	uci_foreach_sections("wireless", "wifi-iface", s) {
		char *curr_ssid = NULL;
		char *curr_dev = NULL;

		dmuci_get_value_by_section_string(s, "ssid", &curr_ssid);
		dmuci_get_value_by_section_string(s, "device", &curr_dev);

		if (DM_STRCMP(curr_ssid, ssid) == 0 && DM_STRCMP(curr_dev, device) == 0) {
			dmuci_set_value_by_section(s, option, value);
			return 0;
		}
	}

	return -1;
}

static void set_wireless_uci_ap_section_fallback(void *data, const char *option, const char *value)
{
	if (set_wireless_uci_ap_section(data, option, value) == 0)
		return;

	dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, option, value);
	return;
}

static char *get_wireless_uci_ap_section_fallback_def(void *data, const char *option, const char *def_val)
{
	struct uci_section *s = NULL, *config_s = NULL, *dmmap_s = NULL;
	char *wireless_sec_name = NULL;
	char *device = NULL;
	char *ssid = NULL;
	char *band = NULL;
	char *value = NULL;

	config_s = ((struct dm_data *)data)->config_section;
	dmmap_s = ((struct dm_data *)data)->dmmap_section;

	if (config_s == NULL)
		return dmstrdup(def_val);

	if (check_slo_ap(config_s)) {
		value = dmuci_get_value_by_section_fallback_def(config_s, option, def_val);
		return value;
	}

	dmuci_get_value_by_section_string(dmmap_s, "wireless_sec_name", &wireless_sec_name);
	s = get_origin_section_from_config("wireless", "wifi-iface", wireless_sec_name);
	if (s) {
		value = dmuci_get_value_by_section_fallback_def(s, option, def_val);
		dmuci_set_value_by_section(dmmap_s, option, "");
		return value;
	}

	dmuci_get_value_by_section_string(config_s, "ssid", &ssid);
	dmuci_get_value_by_section_string(config_s, "band", &band);
	device = get_device_from_band(band);

	if (DM_STRLEN(ssid) == 0 || DM_STRLEN(device) == 0) {
		value = dmuci_get_value_by_section_fallback_def(dmmap_s, option, def_val);
		return value;
	}

	uci_foreach_sections("wireless", "wifi-iface", s) {
		char *curr_ssid = NULL;
		char *curr_dev = NULL;

		dmuci_get_value_by_section_string(s, "ssid", &curr_ssid);
		dmuci_get_value_by_section_string(s, "device", &curr_dev);

		if (DM_STRCMP(curr_ssid, ssid) == 0 && DM_STRCMP(curr_dev, device) == 0) {
			value = dmuci_get_value_by_section_fallback_def(s, option, def_val);
			dmuci_set_value_by_section(dmmap_s, option, "");
			return value;
		}
	}

	value = dmuci_get_value_by_section_fallback_def(dmmap_s, option, def_val);
	return value;
}

static struct uci_section *find_wireless_section(const char *ssid, const char *device)
{
	struct uci_section *s = NULL;

	if (DM_STRLEN(ssid) == 0 || DM_STRLEN(device) == 0)
		return NULL;

	uci_foreach_option_eq("wireless", "wifi-iface", "ssid", ssid, s) {
		char *curr_device = NULL;

		dmuci_get_value_by_section_string(s, "device", &curr_device);
		if (DM_STRCMP(curr_device, device) == 0)
			return s;
	}

	return NULL;
}

static void get_ap_info(struct dm_data *data, char **ifname, json_object **res)
{
	struct uci_section *ap_s = data->config_section;
	char *ap_enable = NULL, *dev_enable = NULL, *ap_device = NULL;
	char object[32] = {0};

	if (!ap_s || !ifname || !res)
		return;

	*res = NULL;
	*ifname = NULL;

	if (check_slo_ap(ap_s) == true || easymesh_enable == false) {
		dmuci_get_value_by_section_string(ap_s, "device", &ap_device);

		dmuci_get_value_by_section_string(ap_s, "ifname", ifname);

		// Get 'disabled' option (1 means disabled), and convert to "enabled" logic
		dmuci_get_value_by_section_string(ap_s, "disabled", &ap_enable);
		ap_enable = (ap_enable && ap_enable[0] == '1') ? "0" : "1";
	} else {
		struct uci_section *wireless_s = NULL;
		char *ssid = NULL, *band = NULL;

		dmuci_get_value_by_section_string(ap_s, "ssid", &ssid);
		dmuci_get_value_by_section_string(ap_s, "band", &band);
		ap_device = get_device_from_band(band);

		// Map SSID and device to find wireless interface section
		wireless_s = find_wireless_section(ssid, ap_device);
		if (wireless_s) {
			dmuci_set_value_by_section(data->dmmap_section, "wireless_sec_name", section_name(wireless_s));
			dmuci_get_value_by_section_string(wireless_s, "ifname", ifname);
		}


		// Check if AP is enabled
		ap_enable = dmuci_get_value_by_section_fallback_def(ap_s, "enabled", "1");
	}

	// Skip if:
	//  - Interface name is empty
	//  - AP is disabled
	//  - AP Device is empty
	if (DM_STRLEN(*ifname) == 0 || DM_STRCMP(ap_enable, "0") == 0 || DM_STRLEN(ap_device) == 0)
		return;

	// Fetch wifi-device (radio) section from UCI confi
	struct uci_section *dev_s = get_origin_section_from_config("wireless", "wifi-device", ap_device);
	if (dev_s == NULL)
		return;

	// Check if the device (radio) is enabled
	dmuci_get_value_by_section_string(dev_s, "disabled", &dev_enable);
	dev_enable = (dev_enable && dev_enable[0] == '1') ? "0" : "1";
	if (DM_STRCMP(dev_enable, "0") == 0)
		return;

	snprintf(object, sizeof(object), "wifi.ap.%s", *ifname);
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, res);
}

static bool is_different_group(const char *mode1, const char *mode2)
{
	int i, g1 = 0, g2 = 0;
	char *security_modes[] = {
		"None",
		"WEP-64, WEP-128",
		"WPA-Personal, WPA2-Personal, WPA3-Personal, WPA-WPA2-Personal, WPA3-Personal-Transition",
		"WPA-Enterprise, WPA2-Enterprise, WPA3-Enterprise, WPA-WPA2-Enterprise"
	};

	for (i = 0; i < ARRAY_SIZE(security_modes); i++) {
		if (DM_STRSTR(security_modes[i], mode1)) {
			g1 = i;
		}

		if (DM_STRSTR(security_modes[i], mode2)) {
			g2 = i;
		}
	}

	return (g1 != g2);
}

static void reset_ap_security_config(void *data)
{
	// Available in both wireless and mapcontroller configs
	dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "key", "");

	// Available only in wireless config
	set_wireless_uci_ap_section_fallback(data, "wpa_group_rekey", "");
	set_wireless_uci_ap_section_fallback(data, "key1", "");
	set_wireless_uci_ap_section_fallback(data, "key2", "");
	set_wireless_uci_ap_section_fallback(data, "key3", "");
	set_wireless_uci_ap_section_fallback(data, "key4", "");
	set_wireless_uci_ap_section_fallback(data, "auth_server", "");
	set_wireless_uci_ap_section_fallback(data, "auth_port", "");
	set_wireless_uci_ap_section_fallback(data, "auth_secret", "");
}

static void set_security_mode(void *data, const char *required_mode)
{
	struct uci_section *config_s = ((struct dm_data *)data)->config_section;
	char *curr_mode = NULL;

	curr_mode = get_security_mode(config_s);

	// Only reset the security options if its belongs to different group
	if (is_different_group(required_mode, curr_mode))
		reset_ap_security_config(data);

	if (DM_LSTRCMP(required_mode, "None") == 0) { // None
		dmuci_set_value_by_section(config_s, "encryption", "none");
	}

	if (DM_LSTRSTR(required_mode, "Enterprise")) { // WPA-Enterprise, WPA2-Enterprise, WPA3-Enterprise, WPA-WPA2-Enterprise
		if (DM_LSTRNCMP(required_mode, "WPA3", 4) == 0) {
			dmuci_set_value_by_section(config_s, "encryption", "wpa3");
			set_wireless_uci_ap_section_fallback(data, "ieee80211w", "2");
		} else if (DM_LSTRNCMP(required_mode, "WPA2", 4) == 0) {
			dmuci_set_value_by_section(config_s, "encryption", "wpa2");
			set_wireless_uci_ap_section_fallback(data, "ieee80211w", "1");
		} else if (DM_LSTRNCMP(required_mode, "WPA-WPA2", 8) == 0) {
			dmuci_set_value_by_section(config_s, "encryption", "wpa-mixed");
		} else {
			dmuci_set_value_by_section(config_s, "encryption", "wpa");
		}
	}

	if (DM_LSTRSTR(required_mode, "Personal")) { // WPA-Personal, WPA2-Personal, WPA-WPA2-Personal, WPA3-Personal, WPA3-Personal-Transition
		char *key = NULL;

		dmuci_get_value_by_section_string(config_s, "key", &key);
		if (DM_STRLEN(key) == 0) {
			struct uci_section *dmmap_s = ((struct dm_data *)data)->dmmap_section;

			dmuci_get_value_by_section_string(dmmap_s, "key", &key);
			if (DM_STRLEN(key) == 0)
				key = get_default_key();

			dmuci_set_value_by_section(config_s, "key", key);
		}

		if (DM_LSTRCMP(required_mode, "WPA-Personal") == 0) {
			dmuci_set_value_by_section(config_s, "encryption", "psk");
		} else if (DM_LSTRCMP(required_mode, "WPA2-Personal") == 0) {
			dmuci_set_value_by_section(config_s, "encryption", "psk2");
			set_wireless_uci_ap_section_fallback(data, "ieee80211w", "1");
		} else if (DM_LSTRCMP(required_mode, "WPA-WPA2-Personal") == 0) {
			dmuci_set_value_by_section(config_s, "encryption", "psk-mixed");
		} else if (DM_LSTRCMP(required_mode, "WPA3-Personal") == 0) {
			dmuci_set_value_by_section(config_s, "encryption", "sae");
			set_wireless_uci_ap_section_fallback(data, "ieee80211w", "2");
		} else if (DM_LSTRCMP(required_mode, "WPA3-Personal-Transition") == 0) {
			dmuci_set_value_by_section(config_s, "encryption", "sae-mixed");
			set_wireless_uci_ap_section_fallback(data, "ieee80211w", "1");
		}
	}

	if (DM_LSTRNCMP(required_mode, "WEP", 3) == 0) { // WEP-64, WEP-128
		// No longer supported
	}
}

/*************************************************************
* ADD DEL OBJ
**************************************************************/
static int add_wifi_ssid(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct uci_section *dmmap = NULL;
	char ssid[32] = {0};

	snprintf(ssid, sizeof(ssid), "ssid_%s", *instance);

	dmuci_add_section_bbfdm("dmmap_wireless", "ssid", &dmmap);
	dmuci_set_value_by_section(dmmap, "enabled", "0");
	dmuci_set_value_by_section(dmmap, "ssid", ssid);
	dmuci_set_value_by_section(dmmap, "name", ssid);
	dmuci_set_value_by_section(dmmap, "added_by_user", "1");
	dmuci_set_value_by_section(dmmap, "ssid_instance", *instance);
	return 0;
}

static int delete_wifi_ssid(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	if (((struct dm_data *)data)->config_section) {
		/* If mapcontroller not enabled then do not allow to delete easymesh nodes */
		if (easymesh_enable == false && check_slo_ap(((struct dm_data *)data)->config_section) == false) {
			bbfdm_set_fault_message(ctx, "Easy-mesh node deletion not allowed on this device.");
			return FAULT_9001;
		}

		dmuci_delete_by_section(((struct dm_data *)data)->config_section, "device", NULL);
		dmuci_delete_by_section(((struct dm_data *)data)->config_section, "band", NULL);
		dmuci_delete_by_section(((struct dm_data *)data)->config_section, "mld_id", NULL);
		dmuci_delete_by_section(((struct dm_data *)data)->config_section, "ssid", NULL);

		/* sync associated AP section in dmmap file */
		struct uci_section *dmmap_s = get_dup_section_in_dmmap("dmmap_wireless", "wifi-iface", section_name(((struct dm_data *)data)->config_section));
		if (dmmap_s) {
			dmuci_set_value_by_section(dmmap_s, "added_by_user", "1");
			dmuci_delete_by_section(dmmap_s, "LowerLayers", NULL);
		}
	}

	/* If SSID has mld_id and if it is last instance having this mld_id then remove the mld section */
	char *mld = NULL;
	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "mld_id", &mld);
	dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, "mld_id", NULL);
	remove_mld_unit(mld);

	dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);

	return 0;
}

static int add_wifi_accesspoint(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct uci_section *s = NULL, *dmmap_wifi = NULL;
	char s_name[32] = {0};

	snprintf(s_name, sizeof(s_name), "wlan_ap_%s", *instance);

	if (easymesh_enable == false) {
		/* If easymesh not enabled then add AP as SLO */
		dmuci_add_section("wireless", "wifi-iface", &s);
		dmuci_rename_section_by_section(s, s_name);
		dmuci_set_value_by_section(s, "disabled", "1");
		dmuci_set_value_by_section(s, "network", "lan");
		dmuci_set_value_by_section(s, "mode", "ap");
		dmuci_set_value_by_section(s, "wps_pushbutton", "1");
	} else {
		/* Add AP as easymesh network node */
		dmuci_add_section("mapcontroller", "ap", &s);
		dmuci_rename_section_by_section(s, s_name);
		dmuci_set_value_by_section(s, "enabled", "0");
		dmuci_set_value_by_section(s, "type", "fronthaul");
	}

	dmuci_add_section_bbfdm("dmmap_wireless", "wifi-iface", &dmmap_wifi);
	dmuci_set_value_by_section(dmmap_wifi, "section_name", s_name);
	dmuci_set_value_by_section(dmmap_wifi, "added_by_user", "1");
	dmuci_set_value_by_section(dmmap_wifi, "ap_instance", *instance);
	dmuci_set_value_by_section(dmmap_wifi, "__is_new__", "1");
	return 0;
}

static int delete_wifi_accesspoint(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	struct uci_section *ss = NULL;

	/* If mapcontroller not enabled then do not allow to delete Easy-mesh AP */
	if (easymesh_enable == false && check_slo_ap(((struct dm_data *)data)->config_section) == false) {
		bbfdm_set_fault_message(ctx, "Easy-mesh AP node can not be deleted on this device");
		return FAULT_9001;
	}

	// If any SSID is refenced to this AP, remove the reference
	char *conf_s = section_name(((struct dm_data *)data)->config_section);
	if (DM_STRLEN(conf_s) != 0) {
		ss = get_dup_section_in_dmmap_opt("dmmap_wireless", "ssid", "ap_section_name", conf_s);
		if (ss) {
			dmuci_delete_by_section(ss, "ap_section_name", NULL);
		}
	}
	// Remove dmmap section
	dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);

	// Remove config section
	dmuci_delete_by_section(((struct dm_data *)data)->config_section, NULL, NULL);
	return 0;
}

static int addObjWiFiEndPoint(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct uci_section *endpoint_sec = NULL, *dmmap_sec = NULL;

	dmuci_add_section("wireless", "wifi-iface", &endpoint_sec);
	dmuci_set_value_by_section(endpoint_sec, "disabled", "1");
	dmuci_set_value_by_section(endpoint_sec, "device", "wl2"); // Should be removed after fixing Device.WiFi.EndPoint.{i}. object
	dmuci_set_value_by_section(endpoint_sec, "mode", "sta");
	dmuci_set_value_by_section(endpoint_sec, "network", "lan");

	dmuci_add_section_bbfdm("dmmap_wireless", "wifi-iface", &dmmap_sec);
	dmuci_set_value_by_section(dmmap_sec, "mode", "sta");
	dmuci_set_value_by_section(dmmap_sec, "section_name", section_name(endpoint_sec));
	dmuci_set_value_by_section(dmmap_sec, "endpointinstance", *instance);
	return 0;
}

static int delObjWiFiEndPoint(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "endpointinstance", "");
	dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "mode", "");
	return 0;
}

/*************************************************************
* ENTRY METHOD
**************************************************************/
/*#Device.WiFi.Radio.{i}.!UCI:wireless/wifi-device/dmmap_wireless*/
static int browseWifiRadioInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct dm_data *curr_data = NULL;
	LIST_HEAD(dup_list);
	char *inst = NULL;

	synchronize_specific_config_sections_with_dmmap("wireless", "wifi-device", "dmmap_wireless", &dup_list);
	list_for_each_entry(curr_data, &dup_list, list) {

		inst = handle_instance(dmctx, parent_node, curr_data->dmmap_section, "radioinstance", "radioalias");

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)curr_data, inst) == DM_STOP)
			break;
	}
	free_dmmap_config_dup_list(&dup_list);
	return 0;
}

static void dmmap_synchronizeWiFiSSID(void)
{
	struct uci_section *s = NULL, *ss = NULL, *stmp = NULL;
	char *user_s = NULL, *ap_sec_name = NULL;

	uci_path_foreach_sections_safe(bbfdm, "dmmap_wireless", "ssid", stmp, s) {
		dmuci_get_value_by_section_string(s, "ap_section_name", &ap_sec_name);
		if (DM_STRLEN(ap_sec_name)) {
			ss = get_origin_section_from_config("wireless", "wifi-iface", ap_sec_name);
			if (ss == NULL && easymesh_enable == true)
				ss = get_origin_section_from_config("mapcontroller", "ap", ap_sec_name);
		}

		// section added by user ==> skip it
		dmuci_get_value_by_section_string(s, "added_by_user", &user_s);
		if (DM_LSTRCMP(user_s, "1") == 0)
			continue;

		// if not exist in uci then delete from dmmap
		if (!ss) {
			dmuci_delete_by_section(s, NULL, NULL);
		}
	}

	uci_foreach_sections("wireless", "wifi-iface", s) {
		char *disabled = NULL, *ssid = NULL, *device = NULL;

		/* filter out easymesh types, only if mapcontroller is enabled. */
		if (!check_slo_ap(s) && easymesh_enable == true) {
			struct uci_section *dmmap_s = get_dup_section_in_dmmap_opt("dmmap_wireless",
						"ssid", "ap_section_name", section_name(s));
			if (dmmap_s) {
				dmuci_delete_by_section(dmmap_s, NULL, NULL);
			}

			continue;
		}

		dmuci_get_value_by_section_string(s, "disabled", &disabled);
		dmuci_get_value_by_section_string(s, "ssid", &ssid);
		dmuci_get_value_by_section_string(s, "device", &device);

		if (DM_STRLEN(ssid) == 0)
			continue;

		// if dmmap section exits sync it
		ss = get_dup_section_in_dmmap_opt("dmmap_wireless", "ssid", "ap_section_name", section_name(s));
		if (ss) {
			char *dm_ssid = NULL;

			dmuci_get_value_by_section_string(ss, "ssid", &dm_ssid);
			if (DM_STRCMP(dm_ssid, ssid)) { // if ssid does not match, sync it
				dmuci_set_value_by_section(ss, "enabled", DM_STRLEN(disabled) ? ((*disabled == '1') ? "0" : "1") : "1");
				dmuci_set_value_by_section(ss, "ssid", ssid);
				dmuci_set_value_by_section(ss, "device", device);
			}
			continue;
		}

		// section added by user ==> skip it
		ss = get_dup_section_in_dmmap("dmmap_wireless", "wifi-iface", section_name(s));
		dmuci_get_value_by_section_string(ss, "added_by_user", &user_s);
		if (DM_LSTRCMP(user_s, "1") == 0)
			continue;

		dmuci_add_section_bbfdm("dmmap_wireless", "ssid", &ss);
		dmuci_set_value_by_section(ss, "ap_section_name", section_name(s));
		dmuci_set_value_by_section(ss, "enabled", DM_STRLEN(disabled) ? ((*disabled == '1') ? "0" : "1") : "1");
		dmuci_set_value_by_section(ss, "ssid", ssid);
		dmuci_set_value_by_section(ss, "device", device);
		dmuci_set_value_by_section(ss, "name", section_name(s));
	}

	if (easymesh_enable == false) {
		// No need to sync with mapcontroller
		return;
	}

	uci_foreach_sections("mapcontroller", "ap", s) {
		char *enabled = NULL, *ssid = NULL, *device = NULL, *band = NULL, *mld = NULL;

		dmuci_get_value_by_section_string(s, "enabled", &enabled);
		dmuci_get_value_by_section_string(s, "ssid", &ssid);
		dmuci_get_value_by_section_string(s, "band", &band);
		if (mld_capable) {
			dmuci_get_value_by_section_string(s, "mld_id", &mld);
		}

		device = get_device_from_band(band);

		if (DM_STRLEN(ssid) == 0)
			continue;

		// if dmmap section exits sync it
		ss = get_dup_section_in_dmmap_opt("dmmap_wireless", "ssid", "ap_section_name", section_name(s));
		if (ss) {
			char *dm_ssid = NULL;

			dmuci_get_value_by_section_string(ss, "ssid", &dm_ssid);
			if (DM_STRCMP(dm_ssid, ssid)) { // if ssid does not match, sync it
				dmuci_set_value_by_section(ss, "enabled", DM_STRLEN(enabled) ? ((*enabled == '1') ? "1" : "0") : "1");
				dmuci_set_value_by_section(ss, "ssid", ssid);
				dmuci_set_value_by_section(ss, "mld_id", mld ? mld : "");
				dmuci_set_value_by_section(ss, "device", device ? device : "");
			}
			continue;
		}

		// section added by user ==> skip it
		ss = get_dup_section_in_dmmap("dmmap_wireless", "wifi-iface", section_name(s));
		dmuci_get_value_by_section_string(ss, "added_by_user", &user_s);
		if (DM_LSTRCMP(user_s, "1") == 0)
			continue;

		dmuci_add_section_bbfdm("dmmap_wireless", "ssid", &ss);
		dmuci_set_value_by_section(ss, "ap_section_name", section_name(s));
		dmuci_set_value_by_section(ss, "enabled", DM_STRLEN(enabled) ? ((*enabled == '1') ? "1" : "0") : "1");
		dmuci_set_value_by_section(ss, "ssid", ssid);
		dmuci_set_value_by_section(ss, "device", device ? device : "");
		dmuci_set_value_by_section(ss, "name", section_name(s));
		dmuci_set_value_by_section(ss, "mld_id", mld ? mld : "");
	}
}

/*#Device.WiFi.SSID.{i}.!UCI:wireless/wifi-iface/dmmap_wireless*/
static int browseWifiSsidInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct dm_data curr_data = {0};
	struct uci_section *dmmap_s = NULL;
	char *inst = NULL;

	dmmap_synchronizeWiFiSSID();

	uci_path_foreach_sections(bbfdm, "dmmap_wireless", "ssid", dmmap_s) {
		struct uci_section *config_s = NULL;
		json_object *ubus_res = NULL;
		char *ap_sec_name = NULL, *ifname = NULL;

		dmuci_get_value_by_section_string(dmmap_s, "ap_section_name", &ap_sec_name);
		if (DM_STRLEN(ap_sec_name)) {
			config_s = get_origin_section_from_config("wireless", "wifi-iface", ap_sec_name);
			if (config_s == NULL && easymesh_enable == true) {
				config_s = get_origin_section_from_config("mapcontroller", "ap", ap_sec_name);
			}

			/* filter out backhaul types. */
			if (multi_ap_type_backhaul(config_s))
				continue;

			curr_data.config_section = config_s;
			curr_data.dmmap_section = dmmap_s;
			get_ap_info(&curr_data, &ifname, &ubus_res);
		}

		curr_data.config_section = config_s;
		curr_data.dmmap_section = dmmap_s;
		curr_data.json_object = ubus_res;
		curr_data.additional_data = (void *)ifname;

		inst = handle_instance(dmctx, parent_node, dmmap_s, "ssid_instance", "ssid_alias");

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static void synchronize_ap_config_sections_with_dmmap(struct list_head *dup_list)
{
	struct uci_section *s = NULL, *stmp = NULL, *dmmap_sect = NULL;
	char *v = NULL, *mode = NULL;

	if (easymesh_enable == true) {
		uci_foreach_sections("mapcontroller", "ap", s) {
			/*
			 * create/update corresponding dmmap section that have same config_section link and using param_value_array
			 */
			if ((dmmap_sect = get_dup_section_in_dmmap("dmmap_wireless", "wifi-iface", section_name(s))) == NULL) {
				dmuci_add_section_bbfdm("dmmap_wireless", "wifi-iface", &dmmap_sect);
				dmuci_set_value_by_section_bbfdm(dmmap_sect, "section_name", section_name(s));
			}

			/*
			 * Add system and dmmap sections to the list
			 */
			add_dmmap_config_dup_list(dup_list, s, dmmap_sect);
		}
	}

	uci_foreach_sections("wireless", "wifi-iface", s) {
		if (!check_slo_ap(s) && easymesh_enable == true) {
			continue;
		}

		char *mode = NULL;
		dmuci_get_value_by_section_string(s, "mode", &mode);
		if (DM_LSTRCMP(mode, "ap") != 0)
			continue;

		/*
		 * create/update corresponding dmmap section that have same config_section link and using param_value_array
		 */
		if ((dmmap_sect = get_dup_section_in_dmmap("dmmap_wireless", "wifi-iface", section_name(s))) == NULL) {
			dmuci_add_section_bbfdm("dmmap_wireless", "wifi-iface", &dmmap_sect);
			dmuci_set_value_by_section_bbfdm(dmmap_sect, "section_name", section_name(s));
		}

		/*
		 * Add system and dmmap sections to the list
		 */
		add_dmmap_config_dup_list(dup_list, s, dmmap_sect);
	}

	/*
	 * Delete unused dmmap sections
	 */
	uci_path_foreach_sections_safe(bbfdm, "dmmap_wireless", "wifi-iface", stmp, s) {
		dmuci_get_value_by_section_string(s, "mode", &mode);
		if (DM_LSTRCMP(mode, "sta") == 0)
			continue;

		dmuci_get_value_by_section_string(s, "section_name", &v);
		if (easymesh_enable == true) {
			if (get_origin_section_from_config("mapcontroller", "ap", v) != NULL) {
				continue;
			}

			struct uci_section *wl_ap_sec = get_origin_section_from_config("wireless", "wifi-iface", v);
			/* if this is not a standalone AP or not found in wireless uci then remove the dmmap section */
			if (wl_ap_sec == NULL || !check_slo_ap(wl_ap_sec)) {
				dmuci_delete_by_section(s, NULL, NULL);
			}
		} else {
			if (get_origin_section_from_config("wireless", "wifi-iface", v) == NULL)
				dmuci_delete_by_section(s, NULL, NULL);
		}
	}
}

/*#Device.WiFi.AccessPoint.{i}.!UCI:wireless/wifi-iface/dmmap_wireless*/
static int browseWifiAccessPointInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct dm_data *curr_data = NULL;
	LIST_HEAD(dup_list);
	char *inst = NULL;

	synchronize_ap_config_sections_with_dmmap(&dup_list);
	list_for_each_entry(curr_data, &dup_list, list) {
		json_object *res = NULL;
		char *ifname = NULL;

		/* filter out backhaul types. */
		if (multi_ap_type_backhaul(curr_data->config_section))
			continue;

		get_ap_info(curr_data, &ifname, &res);

		curr_data->json_object = res;
		curr_data->additional_data = (void *)ifname;

		inst = handle_instance(dmctx, parent_node, curr_data->dmmap_section, "ap_instance", "ap_alias");

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)curr_data, inst) == DM_STOP)
			break;
	}
	free_dmmap_config_dup_list(&dup_list);
	return 0;
}

void synchronize_endpoint_config_sections_with_dmmap(const char *package, const char *section_type, const char *dmmap_package, struct list_head *dup_list)
{
	struct uci_section *s = NULL, *stmp = NULL, *dmmap_sect = NULL;
	char *v = NULL, *mode = NULL;

	uci_foreach_sections(package, section_type, s) {
		dmuci_get_value_by_section_string(s, "mode", &mode);
		if (DM_LSTRCMP(mode, "sta") != 0)
			continue;

		/*
		 * create/update corresponding dmmap section that have same config_section link and using param_value_array
		 */
		if ((dmmap_sect = get_dup_section_in_dmmap(dmmap_package, section_type, section_name(s))) == NULL) {
			dmuci_add_section_bbfdm(dmmap_package, section_type, &dmmap_sect);
			dmuci_set_value_by_section_bbfdm(dmmap_sect, "section_name", section_name(s));
			dmuci_set_value_by_section_bbfdm(dmmap_sect, "mode", "sta");
		}

		/*
		 * Add system and dmmap sections to the list
		 */
		add_dmmap_config_dup_list(dup_list, s, dmmap_sect);
	}

	/*
	 * Delete unused dmmap sections
	 */
	uci_path_foreach_sections_safe(bbfdm, dmmap_package, section_type, stmp, s) {
		dmuci_get_value_by_section_string(s, "mode", &mode);
		if (DM_LSTRCMP(mode, "sta") != 0)
			continue;

		dmuci_get_value_by_section_string(s, "section_name", &v);
		if (get_origin_section_from_config(package, section_type, v) == NULL)
			dmuci_delete_by_section(s, NULL, NULL);
	}
}

/*#Device.WiFi.EndPoint.{i}.!UCI:wireless/wifi-iface/dmmap_wireless*/
static int browseWiFiEndPointInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct dm_data *curr_data = NULL;
	LIST_HEAD(dup_list);
	char *inst = NULL;

	synchronize_endpoint_config_sections_with_dmmap("wireless", "wifi-iface", "dmmap_wireless", &dup_list);
	list_for_each_entry(curr_data, &dup_list, list) {
		char *ifname = NULL;

		dmuci_get_value_by_section_string(curr_data->config_section, "ifname", &ifname);

		curr_data->additional_data = (void *)ifname;

		inst = handle_instance(dmctx, parent_node, curr_data->dmmap_section, "endpointinstance", "endpointalias");

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)curr_data, inst) == DM_STOP)
			break;
	}
	free_dmmap_config_dup_list(&dup_list);
	return 0;
}

static int browseWiFiEndPointProfileInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	char *ep_instance = NULL;

	dmuci_get_value_by_section_string(((struct dm_data *)prev_data)->dmmap_section, "endpointinstance", &ep_instance);

	struct uci_section *s = is_dmmap_section_exist_eq("dmmap_wireless", "ep_profile", "ep_key", ep_instance);
	if (!s) dmuci_add_section_bbfdm("dmmap_wireless", "ep_profile", &s);
	dmuci_set_value_by_section_bbfdm(s, "ep_key", ep_instance);

	DM_LINK_INST_OBJ(dmctx, parent_node, prev_data, "1");
	return 0;
}

/*#Device.WiFi.NeighboringWiFiDiagnostic.Result.{i}.!UBUS:wifi.radio.@Name/scanresults//accesspoints*/
static int browseWifiNeighboringWiFiDiagnosticResultInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct dm_data curr_data = {0};
	struct uci_section *s = NULL;
	char *inst = NULL;

	uci_path_foreach_sections(bbfdm, "dmmap_wifi_neighboring", "result", s) {

		curr_data.config_section = s;

		inst = handle_instance(dmctx, parent_node, s, "wifineighbor_instance", "wifineighbor_alias");

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browse_wifi_associated_device(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *res = NULL, *stations = NULL, *arrobj = NULL;
	char object[32], *inst = NULL;
	struct dm_data curr_data = {0};
	int id = 0, i = 0;

	char *bssid = dmjson_get_value(((struct dm_data *)prev_data)->json_object, 1, "bssid");
	char *mld_ifname = dmjson_get_value(((struct dm_data *)prev_data)->json_object, 1, "mld_ifname");
	char *ifname = ((struct dm_data *)prev_data)->additional_data;

	if (DM_STRLEN(bssid) == 0 || (DM_STRLEN(mld_ifname) == 0 && DM_STRLEN(ifname) == 0))
		return 0;

	if (DM_STRLEN(mld_ifname) == 0) {
		snprintf(object, sizeof(object), "wifi.ap.%s", (char *)((struct dm_data *)prev_data)->additional_data);
	} else {
		snprintf(object, sizeof(object), "wifi.apmld.%s", mld_ifname);
	}

	dmubus_call(object, "stations", UBUS_ARGS{0}, 0, &res);

	dmjson_foreach_obj_in_array(res, arrobj, stations, i, 1, "stations") {

		char *cur_bssid = dmjson_get_value(stations, 1, "bssid");
		if (DM_STRCMP(cur_bssid, bssid) != 0)
			continue;

		// Only standard (non-MLO) STAs appear under AccessPoint{i}.AssociatedDevice.{i}.
		// MLO STAs use a separate data model and are listed under DataElements.Network.Device.{i}.APMLD.{i}.STAMLD.{i}.
		char *mlo_capable = dmjson_get_value(stations, 1, "mlo_capable");
		if (DM_STRLEN(mlo_capable) && DM_STRCMP(mlo_capable, "true") == 0)
			continue;

		curr_data.json_object = stations;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

/**************************************************************************
* SET & GET VALUE
***************************************************************************/
/*#Device.WiFi.RadioNumberOfEntries!UCI:wireless/wifi-device/*/
static int get_WiFi_RadioNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseWifiRadioInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

/*#Device.WiFi.SSIDNumberOfEntries!UCI:wireless/wifi-iface/*/
static int get_WiFi_SSIDNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseWifiSsidInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

/*#Device.WiFi.AccessPointNumberOfEntries!UCI:wireless/wifi-iface/*/
static int get_WiFi_AccessPointNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseWifiAccessPointInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

/*#Device.WiFi.EndPointNumberOfEntries!UCI:wireless/wifi-iface/*/
static int get_WiFi_EndPointNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseWiFiEndPointInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_WiFiEndPoint_ProfileNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseWiFiEndPointProfileInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_wifi_ssid_enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "enabled", value);
	return 0;
}

static int set_wifi_ssid_enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;

			if (((struct dm_data *)data)->config_section) {
				/* If mapcontroller not enabled then do not allow config change on easymesh nodes */
				if (easymesh_enable == false && check_slo_ap(((struct dm_data *)data)->config_section) == false) {
					bbfdm_set_fault_message(ctx, "Config change on Easy-mesh node not allowed on this device");
					return FAULT_9001;
				}
			}

			return 0;
		case VALUESET:
			string_to_bool(value, &b);

			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "enabled", b ? "1" : "0");

			if (((struct dm_data *)data)->config_section == NULL) {
				return 0;
			}

			if (check_slo_ap(((struct dm_data *)data)->config_section)) {
				// wireless config: Update disabled option
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "disabled", b ? "0" : "1");
			} else {
				// mapcontroller config: Update enabled option
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "enabled", b ? "1" : "0");
			}

			return 0;
	}
	return 0;
}

static int get_wifi_status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *enable = NULL;
	get_wifi_ssid_enable(refparam, ctx, data, instance, &enable);
	if (DM_LSTRCMP(enable, "0") == 0) {
		*value = dmstrdup("Down");
		return 0;
	}

	char *if_name = ((struct dm_data *)data)->additional_data;
	if (DM_STRLEN(if_name) == 0) {
		*value = dmstrdup("Error");
		return 0;
	}

	json_object *ubus_res = ((struct dm_data *)data)->json_object;

	if (ubus_res == NULL) {
		 *value = dmstrdup("Unknown");
	} else {
		char *status = dmjson_get_value(ubus_res, 1, "status");
		if (DM_LSTRCMP(status, "running") == 0)
			*value = dmstrdup("Up");
		else
			*value = dmstrdup("Unknown");
	}

	return 0;
}

/*#Device.WiFi.SSID.{i}.SSID!UCI:wireless/wifi-iface,@i-1/ssid*/
static int get_wlan_ssid(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "ssid", value);
	return 0;
}

static int set_wlan_ssid(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *device = NULL, *mld = NULL, *id = NULL, *key = NULL;
	struct uci_section *mld_sec = NULL, *ssid_sec = NULL;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, 1, 32, NULL, NULL))
				return FAULT_9007;

			if (((struct dm_data *)data)->config_section) {
				/* If mapcontroller not enabled then do not allow config change on easymesh nodes */
				if (easymesh_enable == false && check_slo_ap(((struct dm_data *)data)->config_section) == false) {
					bbfdm_set_fault_message(ctx, "Config change on Easy-mesh node not allowed on this device");
					return FAULT_9001;
				}
			}

			/* There should not be any backhaul AP assigned this SSID value */
			if (backhaul_ssid_exist(value)) {
				bbfdm_set_fault_message(ctx, "An instance with ssid (%s) is configured as backhaul AP.", value);
				return FAULT_9001;
			}

			/* Check if any instance with same SSID and Radio is already exist
			 * then reject the request to avoid multiple SSID instance with
			 * same SSID and Radio */
			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "device", &device);
			if (duplicate_ssid_exists(value, device)) {
				char *ll = NULL;
				dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "LowerLayers", &ll);
				bbfdm_set_fault_message(ctx, "An instance with ssid (%s) and LowerLayer (%s) already present.", value, ll);
				return FAULT_9001;
			}

			if (easymesh_enable && mld_capable) {
				dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "mld_id", &mld);

				/* if current SSID has mld unit configured and if SSID instance with new SSID is present,
				 * then the new SSID should not have other MLD id, otherwise multiple MLD sections can
				 * have same SSID */
				ssid_sec = get_dup_section_in_dmmap_opt("dmmap_wireless", "ssid", "ssid", value);
				if (ssid_sec != NULL && DM_STRLEN(mld) != 0) {
					mld_sec = get_dup_section_in_config_opt("mapcontroller", "mld", "ssid", value);
					if (mld_sec != NULL) {
						dmuci_get_value_by_section_string(mld_sec, "id", &id);
					}

					if (DM_STRCMP(id, mld) != 0) {
						bbfdm_set_fault_message(ctx, "SSID (%s) already configured with MLD (%s)."
									"Can not be configured with MLD (%s)", value, id ? id : "-1", mld);
						return FAULT_9001;
					}
				}
			}

			if (((struct dm_data *)data)->config_section == NULL)
				return 0;

			/* If AP is assigned then check:
			 * 1. If SLO AP then there should not be any easymesh node with this SSID value
			 * 2. If easymesh AP then there should not be any SLO AP with this SSID value
			 */
			if (check_slo_ap(((struct dm_data *)data)->config_section)) {
				if (easymesh_ssid_exist(value)) {
					bbfdm_set_fault_message(ctx, "Can't assign because an easymesh node is present with SSID (%s)", value);
					return FAULT_9001;
				}
			} else {
				if (slo_ssid_exist(value)) {
					bbfdm_set_fault_message(ctx, "Can't assign because an standalone AP is present with SSID (%s)", value);
					return FAULT_9001;
				}
			}

			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "ssid", value);
			if (easymesh_enable && mld_capable) {
				dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "mld_id", &mld);
				if (DM_STRLEN(mld) != 0) {
					// if mld unit is configured then update ssid in MLD section
					configure_mld_section(mld, "ssid", value);
					configure_map_sections(mld, "ssid", value, true);
				} else {
					// if new ssid has assigned mld then change MLDunit of this instance
					mld_sec = get_dup_section_in_config_opt("mapcontroller", "mld", "ssid", value);
					if (mld_sec != NULL) {
						dmuci_get_value_by_section_string(mld_sec, "id", &id);
						dmuci_get_value_by_section_string(mld_sec, "key", &key);
					}

					if (DM_STRLEN(id) != 0)
						dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "mld_id", id);
				}
			}

			if (((struct dm_data *)data)->config_section == NULL)
				return 0;

			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "ssid", value);
			if (easymesh_enable && mld_capable) {
				if (DM_STRLEN(id) != 0)
					dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "mld_id", id);
				if (DM_STRLEN(key) != 0)
					dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "key", key);
			}

			return 0;
	}
	return 0;
}

static int get_wlan_ssid_mld(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	if (mld_capable == false) {
		*value = dmstrdup("-1");
		return 0;
	}

	char *mld = NULL;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "mld_id", &mld);
	if (DM_STRLEN(mld) == 0) {
		*value = dmstrdup("-1");
	} else {
		*value = dmstrdup(mld);
	}

	return 0;
}

static int set_wlan_ssid_mld(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *old_mld = NULL, *ssid = NULL, *device = NULL, *key = NULL;
	struct uci_section *s = NULL;

	/* Remove unused MLD sections if any */
	remove_unused_mld_sections();

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_int(ctx, value, RANGE_ARGS{{"-1", "24"}}, 1))
				return FAULT_9007;

			if (DM_STRCMP(value, "-1") == 0)
				break;

			/* Reject if device has no support of MLD */
			if (mld_capable == false || easymesh_enable == false) {
				bbfdm_set_fault_message(ctx, "Device can not support MLD configuration");
				return FAULT_9001;
			}

			if (DM_STRCMP(value, "0") == 0) {
				bbfdm_set_fault_message(ctx, "MLDUnit 0 is not supported");
				return FAULT_9007;
			}

			/* Reject if MLD is requested for an SLO AP */
			if (((struct dm_data *)data)->config_section != NULL && check_slo_ap(((struct dm_data *)data)->config_section)) {
				bbfdm_set_fault_message(ctx, "SSID is referred by standalone AP. MLDUnit is only allowed to easymesh nodes.");
				return FAULT_9001;
			}

			/* Reject if MLD is configured as backhaul */
			if (check_backhaul_mld(value)) {
				bbfdm_set_fault_message(ctx, "MLDUnit %s is of backhaul type and can't be configured", value);
				return FAULT_9001;
			}

			/* Check if any instance with mld unit's SSID and current radio is already exist then reject the request
			 * to avoid multiple SSID instance with same SSID and Radio */
			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "device", &device);
			struct uci_section *map_mld = get_dup_section_in_config_opt("mapcontroller", "mld", "id", value);
			dmuci_get_value_by_section_string(map_mld, "ssid", &ssid);

			if (duplicate_ssid_exists(ssid, device)) {
				bbfdm_set_fault_message(ctx, "Another SSID instance with same radio and MLDUnit is present");
				return FAULT_9001;
			}

			break;
		case VALUESET:
			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "mld_id", &old_mld);
			if (DM_STRCMP(value, "-1") == 0) {
				/* MLDUnit removal scenario */
				if (DM_STRLEN(old_mld) == 0) {
					break;
				}

				/* remove mld from all ssid which are allocated with same mld_id as the current ssid instance */
				uci_foreach_option_eq("mapcontroller", "ap", "mld_id", old_mld, s) {
					dmuci_set_value_by_section(s, "mld_id", "");
				}

				uci_path_foreach_option_eq(bbfdm, "dmmap_wireless", "ssid", "mld_id", old_mld, s){
					dmuci_set_value_by_section(s, "mld_id", "");
				}

				/* Remove the MLD section from mapcontroller */
				remove_mld_unit(old_mld);
				break;
			}

			/* MLDUnit assignment scenarion, either MLD section with this ID can exist or need to be created newly */
			struct uci_section *mld_sec = get_dup_section_in_config_opt("mapcontroller", "mld", "id", value);
			if (mld_sec == NULL) {
				/* MLD section for requested id not exists */
				dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "ssid", &ssid);
				dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "key", &key);

				if (DM_STRLEN(old_mld) == 0) {
					/* MLD changing from -1 to N, create new MLD section */ 
					create_mld_unit_mapcontroller(value, ssid, key);
				} else {
					/* MLD changing from M to N, update existing MLD section with mld id N */
					configure_mld_section(old_mld, "id", value);
				}

				/* reset the MLDUnit with new value to all instances having same ssid and old_mld */
				uci_foreach_option_eq("mapcontroller", "ap", "mld_id", old_mld, s) {
					char *cur_ssid = NULL;

					dmuci_get_value_by_section_string(s, "ssid", &cur_ssid);
					if (DM_STRCMP(ssid, cur_ssid) != 0) {
						continue;
					}

					dmuci_set_value_by_section(s, "mld_id", value);
				}

				uci_path_foreach_option_eq(bbfdm, "dmmap_wireless", "ssid", "mld_id", old_mld, s){
					char *cur_ssid = NULL;

					dmuci_get_value_by_section_string(s, "ssid", &cur_ssid);
					if (DM_STRCMP(ssid, cur_ssid) != 0) {
						continue;
					}

					dmuci_set_value_by_section(s, "mld_id", value);
				}
			} else {
				// MLD section exist, change the SSID with MLD section's SSID and set mld_id, key */
				dmuci_get_value_by_section_string(mld_sec, "ssid", &ssid);
				dmuci_get_value_by_section_string(mld_sec, "key", &key);
				dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "ssid", ssid);
				dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "mld_id", value);

				if (((struct dm_data *)data)->config_section != NULL) {
					dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "ssid", ssid);
					dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "key", key);
					dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "mld_id", value);
				}
			}

			break;
	}
	return 0;
}

static int get_wlan_name(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "name", value);
	return 0;
}

static int get_WiFiSSID_MACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *ubus_res = ((struct dm_data *)data)->json_object;

	*value = (ubus_res) ? dmjson_get_value(ubus_res, 1, "bssid") : "";
	return 0;
}

static int get_radio_enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "disabled", value);
	*value = ((*value)[0] == '1') ? "0" : "1";
	return 0;
}

static int set_radio_enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "disabled", b ? "0" : "1");
			return 0;
	}
	return 0;
}

static int get_radio_status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;
	char radio_obj[32] = {0};

	snprintf(radio_obj, sizeof(radio_obj), "wifi.radio.%s", section_name(((struct dm_data *)data)->config_section));
	dmubus_call(radio_obj, "status", UBUS_ARGS{0}, 0, &res);
	DM_ASSERT(res, *value = dmstrdup("Unknown"));

	char *isup = dmjson_get_value(res, 1, "isup");

	if (DM_STRCMP(isup, "false") == 0) {
		*value = dmstrdup("Down");
	} else if (DM_STRCMP(isup, "true") == 0) {
		*value = dmstrdup("Up");
	} else {
		*value = dmstrdup("Unknown");
	}

	return 0;
}

static int get_WiFiRadio_LowerLayers(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("");
	return 0;
}

static int set_WiFiRadio_LowerLayers(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string_list(ctx, value, -1, -1, 1024, -1, -1, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			break;
	}
	return 0;
}

static int get_WiFiRadio_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmasprintf(value, "%s", section_name(((struct dm_data *)data)->config_section));
	return 0;
}

static int get_WiFiRadio_AutoChannelSupported(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("true");
	return 0;
}

/*#Device.WiFi.Radio.{i}.AutoChannelRefreshPeriod!UCI:wireless/wifi-device,@i-1/acs_refresh_period*/
static int get_WiFiRadio_AutoChannelRefreshPeriod(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "acs_refresh_period", "0");
	return 0;
}

static int set_WiFiRadio_AutoChannelRefreshPeriod(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "acs_refresh_period", value);
			break;
	}
	return 0;
}

/*#Device.WiFi.Radio.{i}.MaxSupportedAssociations!UCI:wireless/wifi-device,@i-1/maxassoc*/
static int get_WiFiRadio_MaxSupportedAssociations(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "maxassoc", "32");
	return 0;
}

/*#Device.WiFi.Radio.{i}.FragmentationThreshold!UCI:wireless/wifi-device,@i-1/frag_threshold*/
static int get_WiFiRadio_FragmentationThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "frag_threshold", "2346");
	return 0;
}

static int set_WiFiRadio_FragmentationThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "frag_threshold", value);
			break;
	}
	return 0;
}

/*#Device.WiFi.Radio.{i}.RTSThreshold!UCI:wireless/wifi-device,@i-1/rts_threshold*/
static int get_WiFiRadio_RTSThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "rts_threshold", "2347");
	return 0;
}

static int set_WiFiRadio_RTSThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "rts_threshold", value);
			break;
	}
	return 0;
}

/*#Device.WiFi.Radio.{i}.BeaconPeriod!UCI:wireless/wifi-device,@i-1/beacon_int*/
static int get_WiFiRadio_BeaconPeriod(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "beacon_int", "100");
	return 0;
}

static int set_WiFiRadio_BeaconPeriod(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "beacon_int", value);
			break;
	}
	return 0;
}

/*#Device.WiFi.Radio.{i}.DTIMPeriod!UCI:wireless/wifi-device,@i-1/dtim_period*/
static int get_WiFiRadio_DTIMPeriod(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "dtim_period", "2");
	return 0;
}

static int set_WiFiRadio_DTIMPeriod(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct uci_section *s = NULL;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;
			break;
		case VALUESET:
			//Here need to loop the iface and match the section name then add the dtim_period on each section
			//as this param is read by mac80211.sh script and is overwriten as 2 if not written in all section
			//of radio
			uci_foreach_option_eq("wireless", "wifi-iface", "device", section_name(((struct dm_data *)data)->config_section), s) {
				char *mode;

				dmuci_get_value_by_section_string(s, "mode", &mode);

				if (DM_LSTRCMP(mode, "ap") != 0)
					continue;

				dmuci_set_value_by_section(s, "dtim_period", value);
			}

			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "dtim_period", value);
			break;
	}
	return 0;
}

/*#Device.WiFi.Radio.{i}.OperatingChannelBandwidth!UCI:wireless/wifi-device,@i-1/htmode*/
static int get_WiFiRadio_OperatingChannelBandwidth(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int bw = get_bandwidth_from_config(((struct dm_data *)data)->config_section);
	dmasprintf(value, "%dMHz", bw);
	return 0;
}

/*#Device.WiFi.Radio.{i}.SupportedOperatingChannelBandwidths!UBUS:wifi.radio.@Name/status//supp_channels[0].bandwidth*/
static int get_WiFiRadio_SupportedOperatingChannelBandwidths(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;
	char object[32];
	char result[128] = {0};

	snprintf(object, sizeof(object), "wifi.radio.%s", section_name(((struct dm_data *)data)->config_section));
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);
	DM_ASSERT(res, *value = dmstrdup("Auto"));

	char *supp_bw = dmjson_get_value_array_all(res, ",", 1, "supp_bw");
	replace_str(supp_bw, "320MHz", "320MHz-1,320MHz-2", result, sizeof(result));

	*value = dmstrdup(result);

	return 0;
}

static int set_WiFiRadio_OperatingChannelBandwidth(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *Supported_Operating_Channel_Bandwidth[] = {"20MHz", "40MHz", "80MHz", "160MHz", "320MHz", "80+80MHz", "Auto", NULL};
	char *supported_bandwidths = NULL;
	char *curr_htmode = NULL;
	char htmode[32];
	int freq;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, Supported_Operating_Channel_Bandwidth, NULL))
				return FAULT_9007;

			// Get the list of all supported operating channel bandwidths
			get_WiFiRadio_SupportedOperatingChannelBandwidths(refparam, ctx, data, instance, &supported_bandwidths);

			// Check if the input value is a valid channel bandwidth value
			if (!value_exits_in_str_list(supported_bandwidths, ",", value)) {
				bbfdm_set_fault_message(ctx, "'%s' bandwidth is not supported by this radio. Possible bandwidths are [%s].", value, supported_bandwidths);
				return FAULT_9007;
			}

			break;
		case VALUESET:
			sscanf(value, "%d", &freq);

			dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "htmode", &curr_htmode);

			if (DM_LSTRNCMP(curr_htmode, "EHT", 3) == 0)
				snprintf(htmode, sizeof(htmode), "EHT%d", freq);
			else if (DM_LSTRNCMP(curr_htmode, "VHT", 3) == 0)
				snprintf(htmode, sizeof(htmode), "VHT%d", freq);
			else if (DM_LSTRNCMP(curr_htmode, "HT", 2) == 0)
				snprintf(htmode, sizeof(htmode), "HT%d", freq);
			else
				snprintf(htmode, sizeof(htmode), "HE%d", freq);

			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "htmode", htmode);
			break;
	}
	return 0;
}

/*#Device.WiFi.Radio.{i}.PreambleType!UCI:wireless/wifi-device,@i-1/short_preamble*/
static int get_WiFiRadio_PreambleType(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "short_preamble", value);
	*value = ((*value)[0] == '1') ? "short" : "long";
	return 0;
}

static int set_WiFiRadio_PreambleType(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *Preamble_Type[] = {"short", "long", "auto", NULL};

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, Preamble_Type, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "short_preamble", (DM_LSTRCMP(value, "short") == 0) ? "1" : "0");
			break;
	}
	return 0;
}

/*#Device.WiFi.Radio.{i}.IEEE80211hSupported!UBUS:wifi.radio.@Name/dot11h_capable*/
static int get_WiFiRadio_IEEE80211hSupported(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;
	char object[32];

	snprintf(object, sizeof(object), "wifi.radio.%s", section_name(((struct dm_data *)data)->config_section));
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);
	if (!res)
		return -1;

	*value = dmjson_get_value(res, 1, "dot11h_capable");
	return 0;
}

/*#Device.WiFi.Radio.{i}.IEEE80211hEnabled!UCI:wireless/wifi-device,@i-1/doth*/
static int get_WiFiRadio_IEEE80211hEnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "doth", "0");
	return 0;
}

static int set_WiFiRadio_IEEE80211hEnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "doth", b ? "1" : "0");
			break;
	}
	return 0;
}

static int get_WiFiRadio_TransmitPowerSupported(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char power_list[65] = {0};
	int i = 0, map_size = 0, index = 0;

	map_size = sizeof(txpower_map)/ sizeof(struct power_percent_map);
	for (i = 0; i < map_size; i++) {
		index += snprintf(power_list+index, sizeof(power_list) - strlen(power_list), "%d,", txpower_map[i].sup_percent);
	}

	map_size = strlen(power_list);
	if (power_list[map_size - 1] == ',')
		power_list[map_size - 1] = '\0';

	*value = dmstrdup(power_list);
	return 0;
}

static int get_WiFiRadio_TransmitPower(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *config = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "txpower", "-1");
	if (DM_STRCMP(config, "-1") == 0) {
		*value = dmstrdup("-1");
		return 0;
	}

	char *device = section_name(((struct dm_data *)data)->config_section);
	if (DM_STRLEN(device) == 0) {
		*value = dmstrdup("-1");
		return 0;
	}

	*value = get_radio_option_nocache(device, "txpower");

	return 0;
}

static int set_WiFiRadio_TransmitPower(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	int i = 0, map_size = 0;
	bool found = false;
	json_object *arrobj = NULL, *supp_ch = NULL, *res = NULL;
	int fault = FAULT_9002;
	char ubus_obj[32] = {0};

	int req_val = (int)strtod(value, NULL);

	switch (action)	{
		case VALUECHECK:
			/* check if requested value is present in supported list */
			map_size = sizeof(txpower_map)/ sizeof(struct power_percent_map);
			for (i = 0; i < map_size; i++) {
				if (req_val == txpower_map[i].sup_percent) {
					found = true;
					break;
				}
			}

			fault = (found == true) ? 0 : FAULT_9007;
			break;
		case VALUESET:
			if (req_val == -1) {
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "txpower", "");
				fault = 0;
				break;
			}

			char *device = section_name(((struct dm_data *)data)->config_section);
			if (DM_STRLEN(device) == 0) {
				bbfdm_set_fault_message(ctx, "Invalid radio device name");
				break;
			}

			snprintf(ubus_obj, sizeof(ubus_obj), "wifi.radio.%s", device);

			dmubus_call(ubus_obj, "status", UBUS_ARGS{0}, 0, &res);
			if (res == NULL) {
				bbfdm_set_fault_message(ctx, "Failed to get wifi radio status report");
				break;
			}

			char *opclass = dmjson_get_value(res, 1, "opclass");
			if (DM_STRLEN(opclass) == 0) {
				bbfdm_set_fault_message(ctx, "Failed to identify current operating class");
				break;
			}

			int i = 0;
			dmjson_foreach_obj_in_array(res, arrobj, supp_ch, i, 1, "supp_channels") {
				char *ch_opclass = dmjson_get_value(supp_ch, 1, "opclass");
				if (DM_STRCMP(opclass, ch_opclass) != 0)
					continue;

				char *max_txpower = dmjson_get_value(supp_ch, 1, "txpower");
				if (DM_STRLEN(max_txpower) == 0) {
					bbfdm_set_fault_message(ctx, "Failed to identify maximum transmit power");
					break;
				}

				int j = 0;
				int max_power = (int)strtod(max_txpower, NULL);

				map_size = sizeof(txpower_map)/ sizeof(struct power_percent_map);
				for (j = 0; j < map_size; j++) {
					if (req_val == txpower_map[j].sup_percent) {
						break;
					}
				}

				if (j == map_size) {
					bbfdm_set_fault_message(ctx, "Failed to calculate transmit power");
					break;
				}

				int set_power = max_power - txpower_map[j].neg_power;

				char str_val[10] = {0};
				snprintf(str_val, sizeof(str_val), "%d", set_power);

				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "txpower", str_val);

				fault = 0;
				break;
			}

			break;
	}

	return fault;
}

/*#Device.WiFi.Radio.{i}.RegulatoryDomain!UCI:wireless/wifi-device,@i-1/country*/
static int get_WiFiRadio_RegulatoryDomain(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *conf_country, *dmmap_contry = NULL;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "country", &conf_country);
	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "country", &dmmap_contry);

	dmasprintf(value, "%s%c", conf_country, (dmmap_contry && *dmmap_contry) ? dmmap_contry[2] : ' ');
	return 0;
}

static int set_WiFiRadio_RegulatoryDomain(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *Regulatory_Domain[] = {"^[A-Z][A-Z]$", "^[A-Z][A-Z][ OI]$", NULL};

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, 2, 3, NULL, Regulatory_Domain))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "country", value);

			// uci only support country code, so strip I/O from value before setting
			if (DM_STRLEN(value) == 3)
				value[2] = '\0';

			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "country", value);
			break;
	}
	return 0;
}

static int get_radio_ccfs0(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;
	char object[UBUS_OBJ_LEN] = {0};

	snprintf(object, sizeof(object), "wifi.radio.%s", section_name(((struct dm_data *)data)->config_section));
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);
	if (!res)
		return -1;

	*value = dmjson_get_value(res, 1, "ccfs0");
	return 0;
}

static int get_radio_ccfs1(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;
	char object[UBUS_OBJ_LEN] = {0};

	snprintf(object, sizeof(object), "wifi.radio.%s", section_name(((struct dm_data *)data)->config_section));
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);
	if (!res)
		return -1;

	*value = dmjson_get_value(res, 1, "ccfs1");
	return 0;
}

/*#Device.WiFi.Radio.{i}.PossibleChannels!UBUS:wifi.radio.@Name/status//supp_channels[0].channels*/
static int get_radio_possible_channels(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;
	char object[UBUS_OBJ_LEN], *bandwidth = NULL;

	snprintf(object, sizeof(object), "wifi.radio.%s", section_name(((struct dm_data *)data)->config_section));
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);
	if (!res)
		return -1;

	bandwidth = dmjson_get_value(res, 1, "bandwidth");

	if (DM_STRTOL(bandwidth) == 0) {
		dmubus_call(object, "channels", UBUS_ARGS{0}, 0, &res);
	} else {
		dmubus_call(object, "channels", UBUS_ARGS{{"bandwidth", bandwidth, Integer}}, 1, &res);
	}

	if (!res)
		return -1;

	*value = dmjson_get_value_array_all(res, ",", 1, "channels");
	return 0;
}

static int get_radio_channels_in_use(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = get_radio_option_nocache(section_name(((struct dm_data *)data)->config_section), "channel");
	return 0;
}

static int get_radio_channel(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *channel = NULL;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "channel", &channel);

	if (DM_LSTRCMP(channel, "auto") == 0 || DM_STRLEN(channel) == 0)
		channel = get_radio_option_nocache(section_name(((struct dm_data *)data)->config_section), "channel");

	*value = channel;
	return 0;
}

static int set_radio_channel(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *supported_channels = NULL;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"1","255"}}, 1))
				return FAULT_9007;

			// Get the list of all supported channels
			get_radio_possible_channels(refparam, ctx, data, instance, &supported_channels);

			// Check if the input value is a valid channel
			if (!value_exits_in_str_list(supported_channels, ",", value))
				return FAULT_9007;

			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "channel", value);
			return 0;
	}
	return 0;
}

static int get_radio_auto_channel_enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "channel", value);
	if (DM_LSTRCMP(*value, "auto") == 0 || (*value)[0] == '\0')
		*value = dmstrdup("1");
	else
		*value = dmstrdup("0");
	return 0;
}

static int set_radio_auto_channel_enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	const char *channel = NULL;
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			if (b)
				channel = "auto";
			else
				channel = get_radio_option_nocache(section_name(((struct dm_data *)data)->config_section), "channel");

			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "channel", channel);
			return 0;
	}
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.SSIDAdvertisementEnabled!UCI:wireless/wifi-iface,@i-1/hidden*/
static int get_wlan_ap_advertisement_enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "hidden", value);
	*value = ((*value)[0] == '1') ? "0" : "1";
	return 0;
}

static int set_wlan_ap_advertisement_enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;

			/* Only allowed for standalone AP */
			if (!check_slo_ap(((struct dm_data *)data)->config_section)) {
				bbfdm_set_fault_message(ctx, "SSIDAdvertisementEnabled not configurable for EasyMesh network node");
				return FAULT_9001;
			}

			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "hidden", b ? "0" : "1");
			return 0;

	}
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.WMMEnable!UCI:wireless/wifi-device,@i-1/wmm*/
static int get_wmm_enabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "wmm", "1");
	return 0;
}

static int set_wmm_enabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;

			/* Only allowed for standalone AP */
			if (!check_slo_ap(((struct dm_data *)data)->config_section)) {
				bbfdm_set_fault_message(ctx, "WMMEnable not configurable for EasyMesh network node");
				return FAULT_9001;
			}

			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "wmm", value);
			return 0;
	}
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.MACAddressControlEnabled!UCI:wireless/wifi-iface,@i-1/macfilter*/
static int get_access_point_control_enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *macfilter;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "macfilter", &macfilter);
	if (macfilter[0] == 0 || DM_LSTRCMP(macfilter, "deny") == 0 || DM_LSTRCMP(macfilter, "disable") == 0)
		*value = dmstrdup("false");
	else
		*value = dmstrdup("true");
	return 0;
}

static int get_WiFiAccessPoint_WMMCapability(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("true");
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.MaxAllowedAssociations!UCI:wireless/wifi-iface,@i-1/maxassoc*/
static int get_WiFiAccessPoint_MaxAllowedAssociations(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "maxassoc", "32");
	return 0;
}

static int set_WiFiAccessPoint_MaxAllowedAssociations(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;

			/* Only allowed for standalone AP */
			if (!check_slo_ap(((struct dm_data *)data)->config_section)) {
				bbfdm_set_fault_message(ctx, "MaxAllowedAssociations not configurable for EasyMesh network node");
				return FAULT_9001;
			}

			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "maxassoc", value);
			break;
	}
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.IsolationEnable!UCI:wireless/wifi-iface,@i-1/isolate*/
static int get_WiFiAccessPoint_IsolationEnable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "isolate", "0");
	return 0;
}

static int set_WiFiAccessPoint_IsolationEnable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;

			/* Only allowed for standalone AP */
			if (!check_slo_ap(((struct dm_data *)data)->config_section)) {
				bbfdm_set_fault_message(ctx, "IsolationEnable not configurable for EasyMesh network node");
				return FAULT_9001;
			}

			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "isolate", b ? "1" : "0");
			break;
	}
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.AllowedMACAddress!UCI:wireless/wifi-iface,@i-1/maclist*/
static int get_WiFiAccessPoint_AllowedMACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_list *val = NULL;
	dmuci_get_value_by_section_list(((struct dm_data *)data)->config_section, "maclist", &val);
	*value = dmuci_list_to_string(val, ",");
	return 0;
}

static int set_WiFiAccessPoint_AllowedMACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	size_t length;
	int i;
	char **arr;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string_list(ctx, value, -1, -1, -1, -1, 17, NULL, MACAddress))
				return FAULT_9007;

			/* Only allowed for standalone AP */
			if (!check_slo_ap(((struct dm_data *)data)->config_section)) {
				bbfdm_set_fault_message(ctx, "AllowedMACAddress not configurable for EasyMesh network node");
				return FAULT_9001;
			}

			break;
		case VALUESET:
			arr = strsplit(value, ",", &length);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "maclist", "");
			for (i = 0; i < length; i++)
				dmuci_add_list_value_by_section(((struct dm_data *)data)->config_section, "maclist", arr[i]);
			break;
	}
	return 0;
}

static int get_WiFiAccessPoint_MultiAPMode(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = check_slo_ap(((struct dm_data *)data)->config_section) ? dmstrdup("0") : dmstrdup("2");
	return 0;
}

static int set_WiFiAccessPoint_MultiAPMode(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct uci_section *uci_s = NULL;
	char sec_name[64] = {0};
	char *is_new_obj = NULL;

	switch (action)	{
		case VALUECHECK:
			/*
			 * Multi-AP mode can only be set at the time of Access Point creation.
			 * If the object is already created (__is_new__ != "1"), disallow changes.
			 */
			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "__is_new__", &is_new_obj);
			if (DM_STRCMP(is_new_obj, "1") != 0) {
				bbfdm_set_fault_message(ctx,
						"Multi-AP mode cannot be modified after the Access Point instance is created and applied");
				return FAULT_9007;
			}

			/* Only value "0" is currently supported for Multi-AP mode */
			if (DM_STRCMP(value, "0") != 0) {
				bbfdm_set_fault_message(ctx, "Only '0' is a supported Multi-AP mode value");
				return FAULT_9007;
			}

			break;
		case VALUESET:
			/* Create new section in wireless config */
			{
				char *ssid = NULL, *band = NULL, *encr = NULL, *key = NULL, *enabled = NULL, *device = NULL;
				char *ieee80211w = NULL;
				bool enable;

				dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "ssid", &ssid);
				dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "band", &band);
				dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "encryption", &encr);
				dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "key", &key);
				dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "enabled", &enabled);
				dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "ieee80211w", &ieee80211w);

				device = get_device_from_band(band);

				if (DM_STRLEN(enabled) != 0)
					enable = dmuci_string_to_boolean(enabled);
				else
					enable = true;

				snprintf(sec_name, sizeof(sec_name), "wifi_%s", section_name(((struct dm_data *)data)->config_section));

				dmuci_add_section("wireless", "wifi-iface", &uci_s);
				dmuci_rename_section_by_section(uci_s, sec_name);

				dmuci_set_value_by_section(uci_s, "disabled", enable ? "0" : "1");
				dmuci_set_value_by_section(uci_s, "network", "lan");
				dmuci_set_value_by_section(uci_s, "mode", "ap");
				dmuci_set_value_by_section(uci_s, "ssid", ssid);
				dmuci_set_value_by_section(uci_s, "device", device ? device : "");
				dmuci_set_value_by_section(uci_s, "encryption", encr);
				dmuci_set_value_by_section(uci_s, "key", key);
				dmuci_set_value_by_section(uci_s, "wps_pushbutton", "1");

				if (DM_STRLEN(ieee80211w) != 0) { // MFPConfig is automatically defined by ModeEnabled
					dmuci_set_value_by_section(uci_s, "ieee80211w", ieee80211w);
					dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "ieee80211w", "");
				}
			}

			/* Update the AP's dmmap section to keep the instance information intact */
			{
				dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "section_name", sec_name);
			}

			/* Update SSID dmmap section if AP was referenced to an SSID, to keep the references intact */
			{
				uci_s = get_dup_section_in_dmmap_opt("dmmap_wireless", "ssid", "ap_section_name", section_name(((struct dm_data *)data)->config_section));
				if (uci_s != NULL) {
					dmuci_set_value_by_section(uci_s, "ap_section_name", sec_name);
					dmuci_set_value_by_section(uci_s, "mld_id", "");
				}
			}

			/* Remove the AP section from mapcontroller */
			{
				dmuci_delete_by_section(((struct dm_data *)data)->config_section, NULL, NULL);
			}

			break;
	}
	return 0;
}

static int get_WiFiAccessPoint_UAPSDCapability(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("true");
	return 0;
}

static int set_access_point_control_enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;

			/* Only allowed for standalone AP */
			if (!check_slo_ap(((struct dm_data *)data)->config_section)) {
				bbfdm_set_fault_message(ctx, "MACAddressControlEnabled not configurable for EasyMesh network node");
				return FAULT_9001;
			}

			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "macfilter", b ? "allow" : "disable");
			return 0;
	}
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.UAPSDEnable!UCI:wireless/wifi-iface,@i-1/wmm_apsd*/
static int get_WiFiAccessPoint_UAPSDEnable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "wmm_apsd", "1");
	return 0;
}

static int set_WiFiAccessPoint_UAPSDEnable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;

			/* Only allowed for standalone AP */
			if (!check_slo_ap(((struct dm_data *)data)->config_section)) {
				bbfdm_set_fault_message(ctx, "UAPSDEnable not configurable for EasyMesh network node");
				return FAULT_9001;
			}

			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "wmm_apsd", b ? "1" : "0");
			break;
	}
	return 0;
}

static int get_access_point_security_supported_modes(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	const char *easymesh_security_modes = "None,WPA-Personal,WPA2-Personal,WPA-WPA2-Personal,WPA3-Personal,WPA3-Personal-Transition,WPA-Enterprise,WPA2-Enterprise,WPA-WPA2-Enterprise";
	const char *slo_security_modes = "None,WEP-64,WEP-128,WPA-Personal,WPA2-Personal,WPA3-Personal,WPA-WPA2-Personal,WPA3-Personal-Transition,WPA-Enterprise,WPA2-Enterprise,WPA3-Enterprise,WPA-WPA2-Enterprise";
	json_object *ubus_res = ((struct dm_data *)data)->json_object;

	if (ubus_res == NULL) {
		if (check_slo_ap(((struct dm_data *)data)->config_section))
			*value = dmstrdup(slo_security_modes);
		else
			*value = dmstrdup(easymesh_security_modes);
	} else {
		json_object *supported_modes = NULL;
		char list_modes[512] = {0};
		char *mode = NULL;
		unsigned pos = 0, idx = 0;

		pos += snprintf(&list_modes[pos], sizeof(list_modes) - pos, "None,");

		dmjson_foreach_value_in_array(ubus_res, supported_modes, mode, idx, 1, "supp_security") {
			char *dm_mode = get_data_model_mode(mode);
			if (DM_STRCMP(dm_mode, "None") == 0)
				continue;

			if (check_slo_ap(((struct dm_data *)data)->config_section)) {
				if (!DM_STRSTR(slo_security_modes, dm_mode))
					continue;
			} else {
				if (!DM_STRSTR(easymesh_security_modes, dm_mode))
					continue;
			}

			pos += snprintf(&list_modes[pos], sizeof(list_modes) - pos, "%s,", dm_mode);
		}

		/* cut tailing ',' */
		if (pos)
			list_modes[pos - 1] = 0;

		*value = dmstrdup(list_modes);
	}

	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.Security.ModeEnabled!UCI:wireless/wifi-iface,@i-1/encryption&UCI:wireless/wifi-iface,@i-1/encryption*/
static int get_access_point_security_modes(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = get_security_mode(((struct dm_data *)data)->config_section);
	return 0;
}

static int set_access_point_security_modes(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *supported_modes = NULL;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, NULL, NULL))
				return FAULT_9007;

			/* If mapcontroller not enabled then do not allow config change on easymesh nodes */
			if (easymesh_enable == false && check_slo_ap(((struct dm_data *)data)->config_section) == false) {
				bbfdm_set_fault_message(ctx, "Config change on Easy-mesh node is not allowed on this device.");
				return FAULT_9001;
			}

			// Get the list of all supported security modes
			get_access_point_security_supported_modes(refparam, ctx, data, instance, &supported_modes);

			// Check if the input value is a valid security mode
			if (!value_exits_in_str_list(supported_modes, ",", value))
				return FAULT_9007;

			return 0;
		case VALUESET:
			set_security_mode(data, value);
			return 0;
	}
	return 0;
}

static int get_access_point_encryption_mode(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *ubus_res = ((struct dm_data *)data)->json_object;

	if (ubus_res == NULL) {
		*value = dmstrdup("");
	} else {
		char *encryption = dmjson_get_value(ubus_res, 1, "encryption");
		*value = get_encryption_mode(encryption);
	}

	return 0;
}

static int get_access_point_security_wepkey(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	/* WEP is deprecated and not supported in current Wi-Fi configurations */
	return 0;
}

static int set_access_point_security_wepkey(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			bbfdm_set_fault_message(ctx, "WEP security mode is deprecated and no longer supported for EasyMesh or Standalone AP");
			return FAULT_9007;
		case VALUESET:
			return 0;
	}
	return 0;
}

static int get_access_point_security_shared_key(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "key", value);
	return 0;
}

static int set_access_point_security_shared_key(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_hexBinary(ctx, value, RANGE_ARGS{{NULL,"32"}}, 1))
				return FAULT_9007;

			/* Only allowed for standalone AP */
			if (!check_slo_ap(((struct dm_data *)data)->config_section)) {
				bbfdm_set_fault_message(ctx, "PreSharedKey not configurable for EasyMesh network node");
				return FAULT_9001;
			}

			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "key", value);
			return 0;
	}
	return 0;
}

static int get_access_point_security_passphrase(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "key", value);
	return 0;
}

static int set_access_point_security_passphrase(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct uci_section *s_map = NULL, *conf_sec = NULL;
	char *mld_id = NULL;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, 8, 63, NULL, NULL))
				return FAULT_9007;

			/* If mapcontroller not enabled then do not allow to change config on easymesh nodes */
			if (easymesh_enable == false && check_slo_ap(((struct dm_data *)data)->config_section) == false) {
				bbfdm_set_fault_message(ctx, "Config change on Easy-mesh node is not allowed on this device.");
				return FAULT_9001;
			}

			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "key", value);

			// Store a copy of key in dmmap. It may be required later when setting the ModeEnabled parameter
			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "key", value);

			if (check_slo_ap(((struct dm_data *)data)->config_section)) {
				/* This is a SLO AP so no need to check MLD section config */
				return 0;
			}

			if (mld_capable) {
				/* This is an easymesh AP node, check if AP is referenced to a mld configured SSID */
				conf_sec = ((struct dm_data *)data)->config_section;
				s_map = get_dup_section_in_dmmap_opt("dmmap_wireless", "ssid", "ap_section_name", section_name(conf_sec));
				dmuci_get_value_by_section_string(s_map, "mld_id", &mld_id);

				if (DM_STRLEN(mld_id) == 0)
					return 0;

				/* MLDUnit configured so need to change the key in MLD section */
				configure_mld_section(mld_id, "key", value);
				configure_map_sections(mld_id, "key", value, false);
			}
			return 0;
	}
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.Security.RekeyingInterval!UCI:wireless/wifi-iface,@i-1/wpa_group_rekey*/
static int get_access_point_security_rekey_interval(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = get_wireless_uci_ap_section_fallback_def(data, "wpa_group_rekey", "0");
	return 0;
}

static int set_access_point_security_rekey_interval(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;

			/* If mapcontroller not enabled then do not allow to change config on easymesh nodes */
			if (easymesh_enable == false && check_slo_ap(((struct dm_data *)data)->config_section) == false) {
				bbfdm_set_fault_message(ctx, "Config change on Easy-mesh node is not allowed on this device.");
				return FAULT_9001;
			}

			return 0;
		case VALUESET:
			set_wireless_uci_ap_section_fallback(data, "wpa_group_rekey", value);
			return 0;
	}
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.Security.SAEPassphrase!UCI:wireless/wifi-iface,@i-1/key*/
static int get_WiFiAccessPointSecurity_SAEPassphrase(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "key", value);
	return 0;
}

static int set_WiFiAccessPointSecurity_SAEPassphrase(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, 8, 63, NULL, NULL))
				return FAULT_9007;

			/* If mapcontroller not enabled then do not allow to change config on easymesh nodes */
			if (easymesh_enable == false && check_slo_ap(((struct dm_data *)data)->config_section) == false) {
				bbfdm_set_fault_message(ctx, "Config change on Easy-mesh node is not allowed on this device.");
				return FAULT_9001;
			}

			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "key", value);

			// Store a copy of key in dmmap. It may be required later when setting the ModeEnabled parameter
			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "key", value);

			if (check_slo_ap(((struct dm_data *)data)->config_section)) {
				/* This is a SLO AP so no need to check MLD section config */
				return 0;
			}

			if (mld_capable) {
				/* This is an easymesh AP node, check if AP is referenced to a mld configured SSID */
				struct uci_section *ssid_s = NULL;
				char *mld_id = NULL;

				ssid_s = get_dup_section_in_dmmap_opt("dmmap_wireless", "ssid", "ap_section_name", section_name(((struct dm_data *)data)->config_section));
				dmuci_get_value_by_section_string(ssid_s, "mld_id", &mld_id);

				if (DM_STRLEN(mld_id) == 0)
					return 0;

				/* MLDUnit configured so need to change the key in MLD section */
				configure_mld_section(mld_id, "key", value);
				configure_map_sections(mld_id, "key", value, false);
			}
			return 0;
	}
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.Security.RadiusServerIPAddr!UCI:wireless/wifi-iface,@i-1/auth_server*/
static int get_access_point_security_radius_ip_address(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = get_wireless_uci_ap_section_fallback_def(data, "auth_server", "");
	return 0;
}

static int set_access_point_security_radius_ip_address(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 45, NULL, IPAddress))
				return FAULT_9007;

			/* If mapcontroller not enabled then do not allow to change config on easymesh nodes */
			if (easymesh_enable == false && check_slo_ap(((struct dm_data *)data)->config_section) == false) {
				bbfdm_set_fault_message(ctx, "Config change on Easy-mesh node is not allowed on this device.");
				return FAULT_9001;
			}

			return 0;
		case VALUESET:
			set_wireless_uci_ap_section_fallback(data, "auth_server", value);
			return 0;
	}
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.Security.RadiusServerPort!UCI:wireless/wifi-iface,@i-1/auth_port*/
static int get_access_point_security_radius_server_port(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = get_wireless_uci_ap_section_fallback_def(data, "auth_port", "1812");
	return 0;
}

static int set_access_point_security_radius_server_port(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;

			/* If mapcontroller not enabled then do not allow to change config on easymesh nodes */
			if (easymesh_enable == false && check_slo_ap(((struct dm_data *)data)->config_section) == false) {
				bbfdm_set_fault_message(ctx, "Config change on Easy-mesh node is not allowed on this device.");
				return FAULT_9001;
			}

			return 0;
		case VALUESET:
			set_wireless_uci_ap_section_fallback(data, "auth_port", value);
			return 0;
	}
	return 0;
}

static int get_access_point_security_radius_secret(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = get_wireless_uci_ap_section_fallback_def(data, "auth_secret", "");
	return 0;
}

static int set_access_point_security_radius_secret(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, NULL, NULL))
				return FAULT_9007;

			/* If mapcontroller not enabled then do not allow to change config on easymesh nodes */
			if (easymesh_enable == false && check_slo_ap(((struct dm_data *)data)->config_section) == false) {
				bbfdm_set_fault_message(ctx, "Config change on Easy-mesh node is not allowed on this device.");
				return FAULT_9001;
			}

			return 0;
		case VALUESET:
			set_wireless_uci_ap_section_fallback(data, "auth_secret", value);
			return 0;
	}
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.Security.MFPConfig!UCI:wireless/wifi-iface,@i-1/ieee80211w*/
static int get_WiFiAccessPointSecurity_MFPConfig(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = get_wireless_uci_ap_section_fallback_def(data, "ieee80211w", "0");

	if (*value[0] == '1')
		*value = dmstrdup("Optional");
	else if (*value[0] == '2')
		*value = dmstrdup("Required");
	else
		*value = dmstrdup("Disabled");
	return 0;
}

static int set_WiFiAccessPointSecurity_MFPConfig(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char buf[2];

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, MFP_Config, NULL))
				return FAULT_9007;

			/* If mapcontroller not enabled then do not allow to change config on easymesh nodes */
			if (easymesh_enable == false && check_slo_ap(((struct dm_data *)data)->config_section) == false) {
				bbfdm_set_fault_message(ctx, "Config change on Easy-mesh node is not allowed on this device.");
				return FAULT_9001;
			}

			break;
		case VALUESET:
			if (DM_LSTRCMP(value, "Disabled") == 0)
				buf[0] = '0';
			else if (DM_LSTRCMP(value, "Optional") == 0)
				buf[0] = '1';
			else if (DM_LSTRCMP(value, "Required") == 0)
				buf[0] = '2';
			buf[1] = 0;
			set_wireless_uci_ap_section_fallback(data, "ieee80211w", buf);
			break;
	}
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.WPS.Enable!UCI:wireless/wifi-iface,@i-1/wps_pushbutton*/
static int get_WiFiAccessPointWPS_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	/* This is a restricted parameter for easymesh node and set to enabled always */
	if (!check_slo_ap(((struct dm_data *)data)->config_section)) {
		*value = dmstrdup("1");
		return 0;
	}

	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "wps_pushbutton", "0");
	return 0;
}

static int set_WiFiAccessPointWPS_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;

			/* This is a restricted parameter for easymesh node and set to enabled always */
			if (!check_slo_ap(((struct dm_data *)data)->config_section)) {
				bbfdm_set_fault_message(ctx, "WPS Enable not configurable for EasyMesh network node");
				return FAULT_9007;
			}

			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "wps_pushbutton", b ? "1" : "0");
			break;
	}
	return 0;
}

static int get_WiFiAccessPointWPS_ConfigMethodsSupported(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("PushButton");
	return 0;
}

static int get_WiFiAccessPointWPS_ConfigMethodsEnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("PushButton");
	return 0;
}

static int set_WiFiAccessPointWPS_ConfigMethodsEnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string_list(ctx, value, -1, -1, -1, -1, -1, NULL, NULL))
				return FAULT_9007;

			if (DM_STRCMP(value, "PushButton") != 0)
				return FAULT_9007;

			break;
		case VALUESET:
			// Don't do anything since we only support 'PushButton'
			break;
	}
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.WPS.Status!UCI:wireless/wifi-iface,@i-1/wps_pushbutton*/
static int get_WiFiAccessPointWPS_Status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *wps_status;

	/* This is a restricted parameter for easymesh node and set to enabled always */
	if (!check_slo_ap(((struct dm_data *)data)->config_section)) {
		*value = dmstrdup("Configured");
		return 0;
	}

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "wps_pushbutton", &wps_status);
	*value = (wps_status[0] == '1') ? dmstrdup("Configured") : dmstrdup("Disabled");
	return 0;
}

static operation_args WiFiAccessPointWPSInitiateWPSPBC_args = {
	.out = (const char *[]) {
		"Status",
		NULL
	}
};

static int get_operate_args_WiFiAccessPointWPS_InitiateWPSPBC(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char *)&WiFiAccessPointWPSInitiateWPSPBC_args;
	return 0;
}

static int operate_WiFiAccessPointWPS_InitiateWPSPBC(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *ifname = (char *)((struct dm_data *)data)->additional_data;
	char object[256], status[32];
	int ubus_ret = 0;

	if (DM_STRLEN(ifname) == 0)
		return USP_FAULT_COMMAND_FAILURE;

	snprintf(status, sizeof(status), "%s", "Success");

	snprintf(object, sizeof(object), "hostapd.%s", ifname);
	ubus_ret = dmubus_call_set(object, "wps_start", UBUS_ARGS{0}, 0);
	if (ubus_ret != 0)
		snprintf(status, sizeof(status), "%s", "Error_Not_Ready");

	fill_blob_param(&ctx->bb, "Status", status, DMT_TYPE[DMT_STRING], 0);
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.Accounting.ServerIPAddr!UCI:wireless/wifi-iface,@i-1/acct_server*/
static int get_WiFiAccessPointAccounting_ServerIPAddr(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "acct_server", value);
	return 0;
}

static int set_WiFiAccessPointAccounting_ServerIPAddr(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 45, NULL, IPAddress))
				return FAULT_9007;

			/* Only allowed for standalone AP */
			if (!check_slo_ap(((struct dm_data *)data)->config_section)) {
				bbfdm_set_fault_message(ctx, "Accounting ServerIPAddr not configurable for EasyMesh network node");
				return FAULT_9001;
			}

			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "acct_server", value);
			break;
	}
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.Accounting.ServerPort!UCI:wireless/wifi-iface,@i-1/acct_port*/
static int get_WiFiAccessPointAccounting_ServerPort(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "acct_port", "1813");
	return 0;
}

static int set_WiFiAccessPointAccounting_ServerPort(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;

			/* Only allowed for standalone AP */
			if (!check_slo_ap(((struct dm_data *)data)->config_section)) {
				bbfdm_set_fault_message(ctx, "Accounting ServerPort not configurable for EasyMesh network node");
				return FAULT_9001;
			}

			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "acct_port", value);
			break;
	}
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.Accounting.Secret!UCI:wireless/wifi-iface,@i-1/acct_secret*/
static int get_WiFiAccessPointAccounting_Secret(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "acct_secret", value);
	return 0;
}

static int set_WiFiAccessPointAccounting_Secret(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, NULL, NULL))
				return FAULT_9007;

			/* Only allowed for standalone AP */
			if (!check_slo_ap(((struct dm_data *)data)->config_section)) {
				bbfdm_set_fault_message(ctx, "Accounting Secret not configurable for EasyMesh network node");
				return FAULT_9001;
			}

			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "acct_secret", value);
			break;
	}
	return 0;
}

/*#Device.WiFi.EndPoint.{i}.Enable!UCI:wireless/wifi-iface,@i-1/disabled*/
static int get_WiFiEndPoint_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "disabled", value);
	*value = ((*value)[0] == '1') ? "0" : "1";
	return 0;
}

static int set_WiFiEndPoint_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "disabled", b ? "0" : "1");
			break;
	}
	return 0;
}

static int get_WiFiEndPoint_Status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return get_net_device_status((char *)((struct dm_data *)data)->additional_data, value);
}

/*#Device.WiFi.EndPoint.{i}.Alias!UCI:dmmap_wireless/wifi-iface,@i-1/endpointalias*/
static int get_WiFiEndPoint_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "endpointalias", instance, value);
}

static int set_WiFiEndPoint_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "endpointalias", instance, value);
}

static int get_WiFiEndPoint_SSIDReference(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *iface_s = get_dup_section_in_config_opt("wireless", "wifi-iface", "ifname", (char *)((struct dm_data *)data)->additional_data);

	_bbfdm_get_references(ctx, "Device.WiFi.SSID.", "Name", section_name(iface_s), value);
	return 0;
}

static int get_WiFiEndPointStats_LastDataDownlinkRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;
	char object[32];

	snprintf(object, sizeof(object), "wifi.bsta.%s", (char *)((struct dm_data *)data)->additional_data);
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);
	if (!res)
		return -1;

	*value = dmjson_get_value(res, 3, "stats", "rx_rate_latest", "rate");
	return 0;
}

static int get_WiFiEndPointStats_LastDataUplinkRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;
	char object[32];

	snprintf(object, sizeof(object), "wifi.bsta.%s", (char *)((struct dm_data *)data)->additional_data);
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);
	if (!res)
		return -1;

	*value = dmjson_get_value(res, 3, "stats", "tx_rate_latest", "rate");
	return 0;
}

static int get_WiFiEndPointStats_SignalStrength(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;
	char object[32];

	snprintf(object, sizeof(object), "wifi.bsta.%s", (char *)((struct dm_data *)data)->additional_data);
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);
	if (!res)
		return -1;

	*value = dmjson_get_value(res, 1, "rssi");
	return 0;
}

static int get_WiFiEndPointStats_Retransmissions(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;
	char object[32];

	snprintf(object, sizeof(object), "wifi.bsta.%s", (char *)((struct dm_data *)data)->additional_data);
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);
	if (!res)
		return -1;

	*value = dmjson_get_value(res, 2, "stats", "tx_pkts_retries");
	return 0;
}

static int get_WiFiEndPointSecurity_ModesSupported(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return get_supported_modes("wifi.bsta", (char *)((struct dm_data *)data)->additional_data, value);
}

static int get_WiFiEndPointProfile_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("1");
	return 0;
}

static int set_WiFiEndPointProfile_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			break;
	}
	return 0;
}

static int get_WiFiEndPointProfile_Status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("Active");
	return 0;
}

static int get_dmmap_wireless_section(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value, const char *section_name)
{
	struct uci_section *dmmap_section = NULL, *dm = NULL;
	char *epinst = NULL;

	get_dmmap_section_of_config_section("dmmap_wireless", "wifi-iface", section_name(((struct dm_data *)data)->config_section), &dmmap_section);
	dmuci_get_value_by_section_string(dmmap_section, "endpointinstance", &epinst);
	get_dmmap_section_of_config_section_eq("dmmap_wireless", "ep_profile", "ep_key", epinst, &dm);
	dmuci_get_value_by_section_string(dm, section_name, value);
	return 0;
}

static int set_dmmap_wireless_section(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action, const char *section_name)
{
	struct uci_section *dmmap_section = NULL, *dm = NULL;
	char *epinst = NULL;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 64, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			get_dmmap_section_of_config_section("dmmap_wireless", "wifi-iface", section_name(((struct dm_data *)data)->config_section), &dmmap_section);
			dmuci_get_value_by_section_string(dmmap_section, "endpointinstance", &epinst);
			get_dmmap_section_of_config_section_eq("dmmap_wireless", "ep_profile", "ep_key", epinst, &dm);
			dmuci_set_value_by_section_bbfdm(dm, section_name, value);
			break;
	}
	return 0;
}

static int get_WiFiEndPointProfile_Location(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return get_dmmap_wireless_section(refparam, ctx, data, instance, value, "ep_location");
}

static int set_WiFiEndPointProfile_Location(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return set_dmmap_wireless_section(refparam, ctx, data, instance, value, action, "ep_location");
}

static int get_WiFiEndPointProfile_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	get_dmmap_wireless_section(refparam, ctx, data, instance, value, "ep_profile_alias");
	if ((*value)[0] == '\0')
		dmasprintf(value, "cpe-%s", instance);
	return 0;
}

static int set_WiFiEndPointProfile_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return set_dmmap_wireless_section(refparam, ctx, data, instance, value, action, "ep_profile_alias");
}

/*#Device.WiFi.EndPoint.{i}.Profile.{i}.SSID!UCI:wireless/wifi-iface,@i-1/ssid*/
static int get_WiFiEndPointProfile_SSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "ssid", value);
	return 0;
}

static int set_WiFiEndPointProfile_SSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 32, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "ssid", value);
			break;
	}
	return 0;
}

/*#Device.WiFi.EndPoint.{i}.Profile.{i}.Security.ModeEnabled!UCI:wireless/wifi-iface,@i-1/encryption*/
static int get_WiFiEndPointProfileSecurity_ModeEnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = get_security_mode(((struct dm_data *)data)->config_section);
	return 0;
}

static int set_WiFiEndPointProfileSecurity_ModeEnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *supported_modes = NULL;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, NULL, NULL))
				return FAULT_9007;

			// Get the list of all supported security modes
			get_WiFiEndPointSecurity_ModesSupported(refparam, ctx, data, instance, &supported_modes);

			// Check if the input value is a valid security mode
			if (!value_exits_in_str_list(supported_modes, ",", value))
				return FAULT_9007;

			return 0;
		case VALUESET:
			set_security_mode(data, value);
			return 0;
	}
	return 0;
}

static int get_WiFiEndPointProfileSecurity_WEPKey(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	/* WEP is deprecated and not supported in current Wi-Fi configurations */
	return 0;
}

static int set_WiFiEndPointProfileSecurity_WEPKey(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			bbfdm_set_fault_message(ctx, "WEP security mode is deprecated and no longer supported");
			return FAULT_9007;
		case VALUESET:
			return 0;
	}
	return 0;
}

static int get_WiFiEndPointProfileSecurity_PreSharedKey(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "key", value);
	return 0;
}

static int set_WiFiEndPointProfileSecurity_PreSharedKey(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_hexBinary(ctx, value, RANGE_ARGS{{NULL,"32"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "key", value);
			return 0;
	}
	return 0;
}

static int get_WiFiEndPointProfileSecurity_KeyPassphrase(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "key", value);
	return 0;
}

static int set_WiFiEndPointProfileSecurity_KeyPassphrase(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, 8, 63, NULL, NULL))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "key", value);
			return 0;
	}
	return 0;
}

/*#Device.WiFi.EndPoint.{i}.Profile.{i}.Security.SAEPassphrase!UCI:wireless/wifi-iface,@i-1/key*/
static int get_WiFiEndPointProfileSecurity_SAEPassphrase(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "key", value);
	return 0;
}

static int set_WiFiEndPointProfileSecurity_SAEPassphrase(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, 8, 63, NULL, NULL))
				return FAULT_9007;

			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "key", value);
			break;
	}
	return 0;
}

/*#Device.WiFi.EndPoint.{i}.Profile.{i}.Security.MFPConfig!UCI:wireless/wifi-iface,@i-1/ieee80211w*/
static int get_WiFiEndPointProfileSecurity_MFPConfig(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "ieee80211w", value);
	if(*value[0] == 0 || *value[0] == '0')
		*value = dmstrdup("Disabled");
	else if (DM_LSTRCMP(*value, "1") == 0)
		*value = dmstrdup("Optional");
	else if (DM_LSTRCMP(*value, "2") == 0)
		*value = dmstrdup("Required");
	return 0;
}

static int set_WiFiEndPointProfileSecurity_MFPConfig(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, MFP_Config, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			if (DM_LSTRCMP(value, "Disabled") == 0)
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "ieee80211w", "0");
			else if (DM_LSTRCMP(value, "Optional") == 0)
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "ieee80211w", "1");
			else if (DM_LSTRCMP(value, "Required") == 0)
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "ieee80211w", "2");
			break;
	}
	return 0;
}

/*#Device.WiFi.EndPoint.{i}.WPS.Enable!UCI:wireless/wifi-iface,@i-1/wps_pushbutton*/
static int get_WiFiEndPointWPS_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "wps_pushbutton", "0");
	return 0;
}

static int set_WiFiEndPointWPS_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "wps_pushbutton", b ? "1" : "0");
			break;
	}
	return 0;
}

static int get_WiFiEndPointWPS_ConfigMethodsSupported(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("PushButton");
	return 0;
}

static int get_WiFiEndPointWPS_ConfigMethodsEnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("PushButton");
	return 0;
}

static int set_WiFiEndPointWPS_ConfigMethodsEnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string_list(ctx, value, -1, -1, -1, -1, -1, NULL, NULL))
				return FAULT_9007;

			if (DM_STRCMP(value, "PushButton") != 0)
				return FAULT_9007;

			break;
		case VALUESET:
			// Don't do anything since we only support 'PushButton'
			break;
	}
	return 0;
}

/*#Device.WiFi.EndPoint.{i}.WPS.Status!UCI:wireless/wifi-iface,@i-1/wps_pushbutton*/
static int get_WiFiEndPointWPS_Status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *wps_status;
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "wps_pushbutton", &wps_status);
	*value = (wps_status[0] == '1') ? "Configured" : "Disabled";
	return 0;
}

/**************************************************************************
* SET AND GET ALIAS
***************************************************************************/
/*#Device.WiFi.Radio.{i}.Alias!UCI:dmmap_wireless/wifi-device,@i-1/radioalias*/
static int get_radio_alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "radioalias", instance, value);
}

static int set_radio_alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "radioalias", instance, value);
}

/*#Device.WiFi.SSID.{i}.Alias!UCI:dmmap_wireless/wifi-iface,@i-1/ssidalias*/
static int get_ssid_alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "ssid_alias", instance, value);
}

static int set_ssid_alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "ssid_alias", instance, value);
}

/*#Device.WiFi.AccessPoint.{i}.Alias!UCI:dmmap_wireless/wifi-iface,@i-1/ap_alias*/
static int get_access_point_alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "ap_alias", instance, value);
}

static int set_access_point_alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "ssid_alias", instance, value);
}

static int get_ssid_lower_layer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "LowerLayers", value);

	if ((*value)[0] == '\0') {
		char *device = NULL;

		dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "device", &device);

		_bbfdm_get_references(ctx, "Device.WiFi.Radio.", "Name", device, value);

		// Store LowerLayers value
		dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "LowerLayers", *value);
	} else {
		if (!adm_entry_object_exists(ctx, *value))
			*value = dmstrdup("");
	}

	return 0;
}

static int set_ssid_lower_layer(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_objects[] = {"Device.WiFi.Radio.", NULL};
	struct dm_reference reference = {0};
	char *ssid = NULL;

	bbfdm_get_reference_linker(ctx, value, &reference);

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string_list(ctx, reference.path, -1, -1, 1024, -1, -1, NULL, NULL))
				return FAULT_9007;

			if (dm_validate_allowed_objects(ctx, &reference, allowed_objects))
				return FAULT_9007;

			if (((struct dm_data *)data)->config_section) {
				/* If mapcontroller not enabled then do not allow config change on easymesh nodes */
				if (easymesh_enable == false && check_slo_ap(((struct dm_data *)data)->config_section) == false) {
					bbfdm_set_fault_message(ctx, "Config change on Easy-mesh node not allowed on this device");
					return FAULT_9001;
				}
			}

			/* Check if any instance with same SSID and Radio is already exist
			 * then reject the request to avoid multiple SSID instance with
			 * same SSID and Radio */
			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "ssid", &ssid);
			if (duplicate_ssid_exists(ssid, reference.value)) {
				bbfdm_set_fault_message(ctx, "An instance with ssid (%s) and LowerLayer (%s) already present.", ssid, value);
				return FAULT_9001;
			}

			return 0;
		case VALUESET:
			// Store LowerLayers value under dmmap section
			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "LowerLayers", reference.path);
			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "device", reference.value);

			if (((struct dm_data *)data)->config_section == NULL)
				return 0;

			if (check_slo_ap(((struct dm_data *)data)->config_section)) {
				/* wireless config: change device */
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "device", reference.value);
			} else {
				/* mapcontroller config: change band */
				char band[2] = {0};
				get_band_from_device(reference.value, band, sizeof(band));
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "band", band);
			}

			return 0;
	}
	return 0;
}

static int get_ap_ssid_ref(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *name = NULL;
	struct uci_section *ss = NULL;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "LowerLayers", value);

	if ((*value)[0] == '\0') {
		ss = get_dup_section_in_dmmap_opt("dmmap_wireless", "ssid", "ap_section_name",
						section_name(((struct dm_data *)data)->config_section));
		if (ss == NULL) {
			dmmap_synchronizeWiFiSSID();
			ss = get_dup_section_in_dmmap_opt("dmmap_wireless", "ssid", "ap_section_name",
						section_name(((struct dm_data *)data)->config_section));
		}

		dmuci_get_value_by_section_string(ss, "name", &name);
		if (DM_STRLEN(name) == 0) {
			*value = dmstrdup("");
			return 0;
		}

		_bbfdm_get_references(ctx, "Device.WiFi.SSID.", "Name", name, value);

		// Store LowerLayers value
		dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "LowerLayers", *value);
	} else {
		if (!adm_entry_object_exists(ctx, *value))
			*value = dmstrdup("");
	}

	return 0;
}

static int set_ap_ssid_ref(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_objects[] = {"Device.WiFi.SSID.", NULL};
	struct uci_section *ss = NULL;
	char *device = NULL, *ssid = NULL, *enabled = NULL, *ap_sec = NULL, *mld = NULL;
	struct dm_reference reference = {0};

	bbfdm_get_reference_linker(ctx, value, &reference);

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, reference.path, -1, 256, NULL, NULL))
				return FAULT_9007;

			if (dm_validate_allowed_objects(ctx, &reference, allowed_objects))
				return FAULT_9007;

			/* If mapcontroller not enabled then do not allow to change config on easymesh nodes */
			if (easymesh_enable == false && check_slo_ap(((struct dm_data *)data)->config_section) == false) {
				bbfdm_set_fault_message(ctx, "Config change on Easy-mesh node is not allowed on this device.");
				return FAULT_9001;
			}

			if (DM_STRLEN(reference.path) == 0)
				return 0;

			/* If this SSID instance is already referred by other AP then reject */
			ss = get_dup_section_in_dmmap_opt("dmmap_wireless", "ssid", "name", reference.value);
			dmuci_get_value_by_section_string(ss, "ap_section_name", &ap_sec);
			if (DM_STRLEN(ap_sec) != 0) {
				bbfdm_set_fault_message(ctx, "SSID (%s) already referenced by other AP", reference.path);
				return FAULT_9001;
			}

			if (check_slo_ap(((struct dm_data *)data)->config_section)) {
				/* This is a SLO ap so SSID instance should not be assigned MLDUnit */
				if (mld_capable) {
					dmuci_get_value_by_section_string(ss, "mld_id", &mld);
					if (DM_STRLEN(mld) != 0) {
						bbfdm_set_fault_message(ctx, "SSID has MLDUnit (%s), so can not be "
									"referenced by standalone AP", mld);
						return FAULT_9001;
					}
				}

				/* Easymesh network should not have AP with same SSID */
				dmuci_get_value_by_section_string(ss, "ssid", &ssid);
				if (easymesh_ssid_exist(ssid)) {
					bbfdm_set_fault_message(ctx, "Easymesh AP is configured with SSID (%s),"
								" so can not be referenced by standalone AP", ssid);
					return FAULT_9001;
				}
			} else {
				/* SSID should not be part of SLO AP */
				dmuci_get_value_by_section_string(ss, "ssid", &ssid);
				if (slo_ssid_exist(ssid)) {
					bbfdm_set_fault_message(ctx, "Standalone AP is configured with SSID (%s),"
								" so can not be referenced by easymesh node", ssid);
					return FAULT_9001;
				}
			}

			break;
		case VALUESET:
			// Store LowerLayers value under dmmap_wireless section
			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "LowerLayers", reference.path);

			if (DM_STRLEN(reference.value)) {
				/* If AP is already referenced to a SSID instance then remove the references from existing SSID */
				ss = get_dup_section_in_dmmap_opt("dmmap_wireless", "ssid", "ap_section_name", section_name(((struct dm_data *)data)->config_section));
				if (ss != NULL) {
					dmuci_set_value_by_section(ss, "ap_section_name", "");
				}

				/* Find the new SSID section and add the configurations */
				ss = get_dup_section_in_dmmap_opt("dmmap_wireless", "ssid", "name", reference.value);

				/* Add reference to new SSID */
				dmuci_set_value_by_section(ss, "ap_section_name", section_name(((struct dm_data *)data)->config_section));

				/* Read ssid, device, mld, enabled from SSID section to set in AP section */
				dmuci_get_value_by_section_string(ss, "device", &device);
				dmuci_get_value_by_section_string(ss, "ssid", &ssid);
				dmuci_get_value_by_section_string(ss, "enabled", &enabled);
				dmuci_get_value_by_section_string(ss, "mld_id", &mld);

				bool b = dmuci_string_to_boolean(enabled);

				if (check_slo_ap(((struct dm_data *)data)->config_section)) {
					/* wireless config */
					dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "ssid", ssid);
					dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "device", device);
					dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "disabled", b ? "0" : "1");
					dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "ifname", "");
				} else {
					/* mapcontroller config */
					char band[2] = {0};

					get_band_from_device(device, band, sizeof(band));
					dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "ssid", ssid);
					dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "band", band);
					dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "enabled", b ? "1" : "0");

					if (!mld_capable || DM_STRLEN(mld) == 0) {
						break;
					}

					/* Set the MLDUnit in the AP section */
					dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "mld_id", mld);

					/* Sync the key of MLD section and AP section */
					char *ap_key = NULL, *mld_key = NULL;

					dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "key", &ap_key);

					struct uci_section *mld_sec = get_dup_section_in_config_opt("mapcontroller", "mld", "id", mld);
					if (mld_sec == NULL) {
						break;
					}

					dmuci_get_value_by_section_string(mld_sec, "key", &mld_key);

					if (DM_STRLEN(mld_key) == 0 && DM_STRLEN(ap_key) != 0) {
						configure_mld_section(mld, "key", ap_key);
						configure_map_sections(mld, "key", ap_key, false);
					} else if (DM_STRCMP(mld_key, ap_key) != 0) {
						dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "key", mld_key);
						dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "SAEPassphrase", "");
						dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "KeyPassphrase", "");
						dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "Key", "");
					}
				}
			} else {
				/* Case of removing SSIDReference from AP */
				ss = get_dup_section_in_dmmap_opt("dmmap_wireless", "ssid", "ap_section_name", section_name(((struct dm_data *)data)->config_section));
				if (ss != NULL) {
					dmuci_set_value_by_section(ss, "ap_section_name", "");
				}

				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "ssid", "");
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "ifname", "");
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "device", "");
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "band", "");
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "mld_id", "");
			}
			break;
	}
	return 0;
}

static int set_neighboring_wifi_diagnostics_diagnostics_state(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct uci_section *s;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, DiagnosticsState, NULL))
				return FAULT_9007;
			return 0;
		case VALUESET:
			if (file_exists("/etc/bbfdm/dmmap/dmmap_wifi_neighboring"))
				remove("/etc/bbfdm/dmmap/dmmap_wifi_neighboring");

			dmuci_add_section_bbfdm("dmmap_wifi_neighboring", "diagnostic_status", &s);
			dmuci_set_value_by_section(s, "DiagnosticsState", value);
			return 0;
	}
	return 0;
}

/*#Device.WiFi.SSID.{i}.BSSID!UBUS:wifi.ap.@Name/status//bssid*/
static int get_wlan_bssid(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *ubus_res = ((struct dm_data *)data)->json_object;

	*value = (ubus_res) ? dmjson_get_value(ubus_res, 1, "bssid") : "";
	return 0;
}

static int ssid_read_ubus(const struct dm_data *args, const char *name, char **value)
{
	json_object *res = NULL;
	char object[32] = {0};

	char *if_name = (char *)args->additional_data;
	if (DM_STRLEN(if_name) == 0)
		return 0;

	// Check if the AP/radio is enabled, if json_object is null means AP/radio is not up
	if (args->json_object == NULL)
		return 0;

	snprintf(object, sizeof(object), "wifi.ap.%s", if_name);
	dmubus_call(object, "stats", UBUS_ARGS{0}, 0, &res);

	if (res == NULL)
		return -1;

	*value = dmjson_get_value(res, 1, name);
	return 0;
}

static int radio_read_ubus(const struct dm_data *args, const char *name, char **value)
{
	json_object *res = NULL;
	char object[UBUS_OBJ_LEN];

	snprintf(object, sizeof(object), "wifi.radio.%s", section_name(args->config_section));
	dmubus_call(object, "stats", UBUS_ARGS{0}, 0, &res);
	if (!res)
		return -1;

	*value = dmjson_get_value(res, 1, name);
	return 0;
}

/*#Device.WiFi.Radio.{i}.Stats.Noise!UBUS:wifi.radio.@Name/status//noise*/
static int get_WiFiRadioStats_Noise(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;
	char object[UBUS_OBJ_LEN];

	snprintf(object, sizeof(object), "wifi.radio.%s", section_name(((struct dm_data *)data)->config_section));
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);
	DM_ASSERT(res, *value = dmstrdup("255"));
	*value = dmjson_get_value(res, 1, "noise");
	return 0;
}

/*#Device.WiFi.Radio.{i}.Stats.BytesSent!UBUS:wifi.radio.@Name/stats//tx_bytes*/
static int get_WiFiRadioStats_BytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return radio_read_ubus(data, "tx_bytes", value);
}

/*#Device.WiFi.Radio.{i}.Stats.BytesReceived!UBUS:wifi.radio.@Name/stats//rx_bytes*/
static int get_WiFiRadioStats_BytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return radio_read_ubus(data, "rx_bytes", value);
}

/*#Device.WiFi.Radio.{i}.Stats.PacketsSent!UBUS:wifi.radio.@Name/stats//tx_packets*/
static int get_WiFiRadioStats_PacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return radio_read_ubus(data, "tx_packets", value);
}

/*#Device.WiFi.Radio.{i}.Stats.PacketsReceived!UBUS:wifi.radio.@Name/stats//rx_packets*/
static int get_WiFiRadioStats_PacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return radio_read_ubus(data, "rx_packets", value);
}

/*#Device.WiFi.Radio.{i}.Stats.ErrorsSent!UBUS:wifi.radio.@Name/stats//tx_error_packets*/
static int get_WiFiRadioStats_ErrorsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return radio_read_ubus(data, "tx_error_packets", value);
}

/*#Device.WiFi.Radio.{i}.Stats.ErrorsReceived!UBUS:wifi.radio.@Name/stats//rx_error_packets*/
static int get_WiFiRadioStats_ErrorsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return radio_read_ubus(data, "rx_error_packets", value);
}

/*#Device.WiFi.Radio.{i}.Stats.DiscardPacketsSent!UBUS:wifi.radio.@Name/stats//tx_dropped_packets*/
static int get_WiFiRadioStats_DiscardPacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return radio_read_ubus(data, "tx_dropped_packets", value);
}

/*#Device.WiFi.Radio.{i}.Stats.DiscardPacketsReceived!UBUS:wifi.radio.@Name/stats//rx_dropped_packets*/
static int get_WiFiRadioStats_DiscardPacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return radio_read_ubus(data, "rx_dropped_packets", value);
}

/*#Device.WiFi.Radio.{i}.Stats.FCSErrorCount!UBUS:wifi.radio.@Name/stats//rx_fcs_error_packets*/
static int get_WiFiRadioStats_FCSErrorCount(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return radio_read_ubus(data, "rx_fcs_error_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.BytesSent!UBUS:wifi.ap.@Name/stats//tx_bytes*/
static int get_WiFiSSIDStats_BytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "tx_bytes", value);
}

/*#Device.WiFi.SSID.{i}.Stats.BytesReceived!UBUS:wifi.ap.@Name/stats//rx_bytes*/
static int get_WiFiSSIDStats_BytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "rx_bytes", value);
}

/*#Device.WiFi.SSID.{i}.Stats.PacketsSent!UBUS:wifi.ap.@Name/stats//tx_packets*/
static int get_WiFiSSIDStats_PacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "tx_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.PacketsReceived!UBUS:wifi.ap.@Name/stats//rx_packets*/
static int get_WiFiSSIDStats_PacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "rx_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.ErrorsSent!UBUS:wifi.ap.@Name/stats//tx_error_packets*/
static int get_WiFiSSIDStats_ErrorsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "tx_error_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.ErrorsReceived!UBUS:wifi.ap.@Name/stats//rx_error_packets*/
static int get_WiFiSSIDStats_ErrorsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "rx_error_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.DiscardPacketsSent!UBUS:wifi.ap.@Name/stats//tx_dropped_packets*/
static int get_WiFiSSIDStats_DiscardPacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "tx_dropped_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.DiscardPacketsReceived!UBUS:wifi.ap.@Name/stats//rx_dropped_packets*/
static int get_WiFiSSIDStats_DiscardPacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "rx_dropped_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.UnicastPacketsSent!UBUS:wifi.ap.@Name/stats//tx_unicast_packets*/
static int get_WiFiSSIDStats_UnicastPacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "tx_unicast_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.UnicastPacketsReceived!UBUS:wifi.ap.@Name/stats//rx_unicast_packets*/
static int get_WiFiSSIDStats_UnicastPacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "rx_unicast_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.MulticastPacketsSent!UBUS:wifi.ap.@Name/stats//tx_multicast_packets*/
static int get_WiFiSSIDStats_MulticastPacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "tx_multicast_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.MulticastPacketsReceived!UBUS:wifi.ap.@Name/stats//rx_multicast_packets*/
static int get_WiFiSSIDStats_MulticastPacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "rx_multicast_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.BroadcastPacketsSent!UBUS:wifi.ap.@Name/stats//tx_broadcast_packets*/
static int get_WiFiSSIDStats_BroadcastPacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "tx_broadcast_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.BroadcastPacketsReceived!UBUS:wifi.ap.@Name/stats//rx_broadcast_packets*/
static int get_WiFiSSIDStats_BroadcastPacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "rx_broadcast_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.RetransCount!UBUS:wifi.ap.@Name/stats//tx_retrans_packets*/
static int get_WiFiSSIDStats_RetransCount(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "tx_retrans_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.FailedRetransCount!UBUS:wifi.ap.@Name/stats//tx_retrans_fail_packets*/
static int get_WiFiSSIDStats_FailedRetransCount(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "tx_retrans_fail_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.RetryCount!UBUS:wifi.ap.@Name/stats//tx_retry_packets*/
static int get_WiFiSSIDStats_RetryCount(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "tx_retry_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.MultipleRetryCount!UBUS:wifi.ap.@Name/stats//tx_multi_retry_packets*/
static int get_WiFiSSIDStats_MultipleRetryCount(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "tx_multi_retry_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.ACKFailureCount!UBUS:wifi.ap.@Name/stats//ack_fail_packets*/
static int get_WiFiSSIDStats_ACKFailureCount(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "ack_fail_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.AggregatedPacketCount!UBUS:wifi.ap.@Name/stats//aggregate_packets*/
static int get_WiFiSSIDStats_AggregatedPacketCount(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "aggregate_packets", value);
}

/*#Device.WiFi.SSID.{i}.Stats.UnknownProtoPacketsReceived!UBUS:wifi.ap.@Name/stats//rx_unknown_packets*/
static int get_WiFiSSIDStats_UnknownProtoPacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ssid_read_ubus(data, "rx_unknown_packets", value);
}

static int get_WiFiAccessPointAssociatedDevice_Active(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("true");
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.AssociatedDevice.{i}.Noise!UBUS:wifi.ap.@Name/stations//stations[i-1].noise*/
static int get_WiFiAccessPointAssociatedDevice_Noise(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "noise");
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.AssociatedDevice.{i}.MACAddress!UBUS:wifi.ap.@Name/stations//stations[i-1].macaddr*/
static int get_WiFiAccessPointAssociatedDevice_MACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "macaddr");
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.AssociatedDevice.{i}.LastDataDownlinkRate!UBUS:wifi.ap.@Name/stations//stations[i-1].stats.rx_rate_latest.rate*/
static int get_WiFiAccessPointAssociatedDevice_LastDataDownlinkRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *rate_mbps = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "stats", "rx_rate_latest", "rate");
	float rate_kbps = (rate_mbps && *rate_mbps != '\0') ? (float)strtod(rate_mbps, NULL) * 1000 : 1000;

	dmasprintf(value, "%u", (unsigned int)rate_kbps);
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.AssociatedDevice.{i}.LastDataUplinkRate!UBUS:wifi.ap.@Name/stations//stations[i-1].stats.tx_rate_latest.rate*/
static int get_WiFiAccessPointAssociatedDevice_LastDataUplinkRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *rate_mbps = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "stats", "tx_rate_latest", "rate");
	float rate_kbps = (rate_mbps && *rate_mbps != '\0') ? (float)strtod(rate_mbps, NULL) * 1000 : 1000;

	dmasprintf(value, "%u", (unsigned int)rate_kbps);
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.AssociatedDevice.{i}.SignalStrength!UBUS:wifi.ap.@Name/stations//stations[i-1].rssi*/
static int get_WiFiAccessPointAssociatedDevice_SignalStrength(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *link_id = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "link_id");

	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, DM_STRLEN(link_id) ? "rssi_avg" : "rssi");
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.AssociatedDevice.{i}.AssociationTime!UBUS:wifi.ap.@Name/stations//stations[i-1].in_network*/
static int get_WiFiAccessPointAssociatedDevice_AssociationTime(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *in_network = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "in_network");
	if (in_network && *in_network != '\0' && DM_STRTOL(in_network) > 0) {
		time_t t_time = time(NULL) - DM_STRTOL(in_network);
		if (gmtime(&t_time) == NULL)
			return -1;

		char utc_time[32] = {0};

		if (strftime(utc_time, sizeof(utc_time), "%Y-%m-%dT%H:%M:%SZ", gmtime(&t_time)) == 0)
			return -1;

		*value = dmstrdup(utc_time);
	} else {
		*value = dmstrdup("0001-01-01T00:00:00Z");
	}
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.AssociatedDevice.{i}.Stats.BytesSent!UBUS:wifi.ap.@Name/stations//stations[i-1].stats.tx_total_bytes*/
static int get_access_point_associative_device_statistics_tx_bytes(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "stats", "tx_total_bytes");
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.AssociatedDevice.{i}.Stats.BytesReceived!UBUS:wifi.ap.@Name/stations//stations[i-1].stats.rx_data_bytes*/
static int get_access_point_associative_device_statistics_rx_bytes(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "stats", "rx_data_bytes");
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.AssociatedDevice.{i}.Stats.PacketsSent!UBUS:wifi.ap.@Name/stations//stations[i-1].stats.tx_total_pkts*/
static int get_access_point_associative_device_statistics_tx_packets(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "stats", "tx_total_pkts");
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.AssociatedDevice.{i}.Stats.PacketsReceived!UBUS:wifi.ap.@Name/stations//stations[i-1].stats.rx_data_pkts*/
static int get_access_point_associative_device_statistics_rx_packets(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "stats", "rx_data_pkts");
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.AssociatedDevice.{i}.Stats.ErrorsSent!UBUS:wifi.ap.@Name/stations//stations[i-1].stats.tx_failures*/
static int get_access_point_associative_device_statistics_tx_errors(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "stats", "tx_failures");
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.AssociatedDevice.{i}.Stats.RetransCount!UBUS:wifi.ap.@Name/stations//stations[i-1].stats.tx_pkts_retries*/
static int get_access_point_associative_device_statistics_retrans_count(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "stats", "tx_pkts_retries");
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.Enable!UCI:wireless/wifi-iface,@i-1/disabled*/
static int get_access_point_enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	if (check_slo_ap(((struct dm_data *)data)->config_section)) {
		dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "disabled", value);
		*value = ((*value)[0] == '1') ? dmstrdup("0") : dmstrdup("1");
	} else {
		*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enabled", "1");
	}

	return 0;
}

static int set_access_point_enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;

			/* If mapcontroller not enabled then do not allow to change config on easymesh nodes */
			if (easymesh_enable == false && check_slo_ap(((struct dm_data *)data)->config_section) == false) {
				bbfdm_set_fault_message(ctx, "Config change on Easy-mesh node is not allowed on this device.");
				return FAULT_9001;
			}

			return 0;
		case VALUESET:
			string_to_bool(value, &b);

			if (check_slo_ap(((struct dm_data *)data)->config_section)) {
				// wireless config: Update disabled option
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "disabled", b ? "0" : "1");
			} else {
				// mapcontroller config: Update enabled option
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "enabled", b ? "1" : "0");
			}

			struct uci_section *ss = get_dup_section_in_dmmap_opt("dmmap_wireless", "ssid", "ap_section_name",
							section_name(((struct dm_data *)data)->config_section));
			if (ss != NULL) {
				dmuci_set_value_by_section(ss, "enabled", b ? "1" : "0");
			}

			return 0;
	}
	return 0;
}

/*#Device.WiFi.AccessPoint.{i}.Status!UBUS:wifi.ap.@Name/status//status*/
static int get_wifi_access_point_status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *enable = NULL;

	get_access_point_enable(refparam, ctx, data, instance, &enable);
	if (DM_LSTRCMP(enable, "0") == 0) {
		*value = dmstrdup("Disabled");
		return 0;
	}

	char *if_name = ((struct dm_data *)data)->additional_data;
	if (DM_STRLEN(if_name) == 0) {
		*value = dmstrdup("Error");
		return 0;
	}

	json_object *ubus_res = ((struct dm_data *)data)->json_object;

	if (ubus_res == NULL) {
		*value = dmstrdup("Error");
	} else {
		char *status = dmjson_get_value(ubus_res, 1, "status");
		if (DM_LSTRCMP(status, "running") == 0 || DM_LSTRCMP(status, "up") == 0)
			*value = dmstrdup("Enabled");
		else
			*value = dmstrdup("Error");
	}

	return 0;
}

/*#Device.WiFi.Radio.{i}.MaxBitRate!UBUS:wifi.radio.@Name/status//maxrate*/
static int get_radio_max_bit_rate (char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;
	char object[UBUS_OBJ_LEN];

	snprintf(object, sizeof(object), "wifi.radio.%s", section_name(((struct dm_data *)data)->config_section));
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);
	if (!res)
		return -1;

	*value = dmjson_get_value(res, 1, "maxrate");
	return 0;
}

/*#Device.WiFi.Radio.{i}.SupportedFrequencyBands!UBUS:wifi.radio.@Name/status//supp_bands*/
static int get_radio_supported_frequency_bands(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;
	char object[UBUS_OBJ_LEN];

	snprintf(object, sizeof(object), "wifi.radio.%s", section_name(((struct dm_data *)data)->config_section));
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);
	if (!res)
		return -1;

	*value = dmjson_get_value_array_all(res, ",", 1, "supp_bands");
	return 0;
}

/*#Device.WiFi.Radio.{i}.OperatingFrequencyBand!UBUS:wifi.radio.@Name/status//band*/
static int get_radio_frequency(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;
	char object[UBUS_OBJ_LEN];

	snprintf(object, sizeof(object), "wifi.radio.%s", section_name(((struct dm_data *)data)->config_section));
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);
	if (!res)
		return -1;

	*value = dmjson_get_value(res, 1, "band");
	return 0;
}

static int set_radio_frequency(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *Supported_Frequency_Bands[] = {"2.4GHz", "5GHz", "6GHz", NULL};
	char *supported_frequency_bands = NULL;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, Supported_Frequency_Bands, NULL))
				return FAULT_9007;

			// Get the list of all supported frequency bands
			get_radio_supported_frequency_bands(refparam, ctx, data, instance, &supported_frequency_bands);

			// Check if the input value is a supported band value
			if (!value_exits_in_str_list(supported_frequency_bands, ",", value)) {
				bbfdm_set_fault_message(ctx, "'%s' band is not supported by this radio. Possible bands are [%s].", value, supported_frequency_bands);
				return FAULT_9007;
			}

			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "band", (value[0] == '6') ? "6g" : (value[0] == '5') ? "5g" : "2g");
			break;
	}
	return 0;
}

static int get_neighboring_wifi_diagnostics_diagnostics_state(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_option_value_string_bbfdm("dmmap_wifi_neighboring", "@diagnostic_status[0]", "DiagnosticsState", value);
	if ((*value)[0] == '\0')
		*value = dmstrdup("None");
	return 0;
}

static int get_neighboring_wifi_diagnostics_result_number_entries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseWifiNeighboringWiFiDiagnosticResultInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

/*#Device.WiFi.NeighboringWiFiDiagnostic.Result.{i}.SSID!UBUS:wifi.radio.@Name/scanresults//accesspoints[@i-1].ssid*/
static int get_neighboring_wifi_diagnostics_result_ssid(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "ssid", value);
	return 0;
}

/*#Device.WiFi.NeighboringWiFiDiagnostic.Result.{i}.BSSID!UBUS:wifi.radio.@Name/scanresults//accesspoints[@i-1].bssid*/
static int get_neighboring_wifi_diagnostics_result_bssid(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "bssid", value);
	return 0;
}

/*#Device.WiFi.NeighboringWiFiDiagnostic.Result.{i}.Channel!UBUS:wifi.radio.@Name/scanresults//accesspoints[@i-1].channel*/
static int get_neighboring_wifi_diagnostics_result_channel(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "channel", value);
	return 0;
}

/*#Device.WiFi.NeighboringWiFiDiagnostic.Result.{i}.SignalStrength!UBUS:wifi.radio.@Name/scanresults//accesspoints[@i-1].rssi*/
static int get_neighboring_wifi_diagnostics_result_signal_strength(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "rssi", value);
	return 0;
}

/*#Device.WiFi.NeighboringWiFiDiagnostic.Result.{i}.OperatingFrequencyBand!UBUS:wifi.radio.@Name/scanresults//accesspoints[@i-1].band*/
static int get_neighboring_wifi_diagnostics_result_operating_frequency_band(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "band", value);
	return 0;
}

/*#Device.WiFi.NeighboringWiFiDiagnostic.Result.{i}.Noise!UBUS:wifi.radio.@Name/scanresults//accesspoints[@i-1].noise*/
/*static int get_neighboring_wifi_diagnostics_result_noise(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "noise", value);
	return 0;
}*/

static int get_neighboring_wifi_diagnostics_result_radio(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "radio", value);
	return 0;
}

static int get_neighboring_wifi_diagnostics_result_operating_ch_band(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "bandwidth", value);
	return 0;
}

static int get_neighboring_wifi_diagnostics_result_op_std(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "standard", value);
	return 0;
}

static int get_neighboring_wifi_diagnostics_result_sec_mode(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "encryption", value);
	return 0;
}

static int get_neighboring_wifi_diagnostics_result_enc_mode(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "ciphers", value);
	return 0;
}

/*#Device.WiFi.Radio.{i}.CurrentOperatingChannelBandwidth!UBUS:wifi.radio.@Name/status//bandwidth*/
static int get_WiFiRadio_CurrentOperatingChannelBandwidth(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;
	char object[32];
	char *ccfs0 = NULL, *ccfs1 = NULL, *band = NULL;

	snprintf(object, sizeof(object), "wifi.radio.%s", section_name(((struct dm_data *)data)->config_section));
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);
	DM_ASSERT(res, *value = dmstrdup("20MHz"));

	band = dmjson_get_value(res, 1, "bandwidth");
	ccfs0 = dmjson_get_value(res, 1, "ccfs0");
	ccfs1 = dmjson_get_value(res, 1, "ccfs1");

	*value = get_data_model_band(band, ccfs0, ccfs1);
	return 0;
}

/*#Device.WiFi.Radio.{i}.SupportedStandards!UBUS:wifi.radio.@Name/status//supp_std*/
static int get_radio_supported_standard(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL, *supp_std_arr = NULL;
	char list_supp_std[32], object[UBUS_OBJ_LEN];
	char *supp_std = NULL;
	unsigned pos = 0, idx = 0;

	snprintf(object, sizeof(object), "wifi.radio.%s", section_name(((struct dm_data *)data)->config_section));
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);
	DM_ASSERT(res, *value = dmstrdup("n,ax"));

	list_supp_std[0] = 0;
	dmjson_foreach_value_in_array(res, supp_std_arr, supp_std, idx, 1, "supp_std") { // supp_std has as value 11xx
		pos += snprintf(&list_supp_std[pos], sizeof(list_supp_std) - pos, "%s,", supp_std + 2);
	}

	if (pos)
		list_supp_std[pos - 1] = 0;

	*value = dmstrdup(list_supp_std);
	return 0;
}

/*#Device.WiFi.Radio.{i}.OperatingStandards!UBUS:wifi.radio.@Name/status//standard*/
static int get_radio_operating_standard(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	json_object *res = NULL;
	char object[UBUS_OBJ_LEN] = {0};

	snprintf(object, sizeof(object), "wifi.radio.%s", section_name(((struct dm_data *)data)->config_section));
	dmubus_call(object, "status", UBUS_ARGS{0}, 0, &res);
	DM_ASSERT(res, *value = dmstrdup("n,ax"));
	*value = get_data_model_standard(dmjson_get_value(res, 1, "standard"));
	return 0;
}

static int set_radio_operating_standard(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *Supported_Standards[] = {"a", "b", "g", "n", "ac", "ax", "be", NULL};
	char *supported_standards = NULL;
	char buf[16], htmode[8];
	char *pch, *spch;
	int bandwidth;

	switch (action) {
			case VALUECHECK:
				if (bbfdm_validate_string_list(ctx, value, -1, -1, -1, -1, -1, Supported_Standards, NULL))
					return FAULT_9007;

				// Get the list of all supported standards
				get_radio_supported_standard(refparam, ctx, data, instance, &supported_standards);

				// Check if the input value is a valid standard value
				snprintf(buf, sizeof(buf), "%s", value);
				for (pch = strtok_r(buf, ",", &spch); pch != NULL; pch = strtok_r(NULL, ",", &spch)) {
					if (!value_exits_in_str_list(supported_standards, ",", pch)) {
						bbfdm_set_fault_message(ctx, "'%s' standard is not supported by this radio. Possible standards are [%s].", pch, supported_standards);
						return FAULT_9007;
					}
				}

				break;
			case VALUESET:
				bandwidth = get_bandwidth_from_config(((struct dm_data *)data)->config_section);

				if (DM_LSTRSTR(value, "be"))
					snprintf(htmode, sizeof(htmode), "EHT%d", bandwidth);
				else if (DM_LSTRSTR(value, "ax"))
					snprintf(htmode, sizeof(htmode), "HE%d", bandwidth);
				else if (DM_LSTRSTR(value, "ac"))
					snprintf(htmode, sizeof(htmode), "VHT%d", bandwidth);
				else if (DM_LSTRSTR(value, "n"))
					snprintf(htmode, sizeof(htmode), "HT%d", bandwidth);
				else
					snprintf(htmode, sizeof(htmode), "NOHT");

				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "htmode", htmode);
				break;
		}
		return 0;
}

static int get_access_point_total_associations(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browse_wifi_associated_device);
	dmasprintf(value, "%d", cnt);
	return 0;
}


/*************************************************************
 * OPERATE COMMANDS
 *************************************************************/
static int operate_WiFi_Reset(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	if (easymesh_enable) {
		int err = dmubus_call_set("map.agent", "reset", UBUS_ARGS{0}, 0);
		if (err)
			return err;

		err = remove("/etc/bbfdm/dmmap/dmmap_wireless");
		if (err)
			return err;

		return !dmubus_call_set("wifi.dataelements", "refresh", UBUS_ARGS{0}, 0) ? 0 : USP_FAULT_COMMAND_FAILURE;
	} else {
		return !dmcmd_no_wait("/sbin/wifi", 1, "reload") ? 0 : USP_FAULT_COMMAND_FAILURE;
	}
}

static operation_args neighboring_wifi_diagnostic_args = {
	.out = (const char *[]) {
		"Status",
		"Result.{i}.Radio",
		"Result.{i}.SSID",
		"Result.{i}.BSSID",
		"Result.{i}.Channel",
		"Result.{i}.SignalStrength",
		"Result.{i}.SecurityModeEnabled",
		"Result.{i}.EncryptionMode",
		"Result.{i}.OperatingFrequencyBand",
		"Result.{i}.OperatingStandards",
		"Result.{i}.OperatingChannelBandwidth",
		//"Result.{i}.Noise",
		NULL
	}
};

static int get_operate_args_WiFi_NeighboringWiFiDiagnostic(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char *)&neighboring_wifi_diagnostic_args;
	return 0;
}

static void dmubus_receive_wifi_radio(struct ubus_context *ctx, struct ubus_event_handler *ev,
				const char *type, struct blob_attr *msg)
{
	struct dmubus_event_data *data;

	if (!msg || !ev)
		return;

	data = container_of(ev, struct dmubus_event_data, ev);
	if (data == NULL)
		return;

	struct radio_scan_args *ev_data = data->ev_data;
	if (ev_data == NULL)
		return;

	struct blob_attr *res = (struct blob_attr *)ev_data->msg;
	if (res == NULL)
		return;

	struct blob_attr *event_list = NULL;
	struct blob_attr *cur = NULL;
	int rem = 0;

	struct blob_attr *tb_event[1] = {0};
	const struct blobmsg_policy p_event_table[1] = {
		{ "event_data", BLOBMSG_TYPE_ARRAY }
	};

	blobmsg_parse(p_event_table, 1, tb_event, blobmsg_data(res), blobmsg_len(res));
	if (tb_event[0] == NULL)
		return;

	event_list = tb_event[0];

	const struct blobmsg_policy p_event[2] = {
		{ "ifname", BLOBMSG_TYPE_STRING },
		{ "event", BLOBMSG_TYPE_STRING }
	};

	struct blob_attr *tb1[2] = { 0, 0 };
	blobmsg_parse(p_event, 2, tb1, blobmsg_data(msg), blobmsg_len(msg));
	if (!tb1[0] || !tb1[1])
		return;

	char *ifname = blobmsg_get_string(tb1[0]);
	char *action = blobmsg_get_string(tb1[1]);

	blobmsg_for_each_attr(cur, event_list, rem) {
		struct blob_attr *tb2[2] = { 0, 0 };

		blobmsg_parse(p_event, 2, tb2, blobmsg_data(cur), blobmsg_len(cur));
		if (!tb2[0] || !tb2[1])
			continue;

		if (DM_STRCMP(ifname, blobmsg_get_string(tb2[0])) != 0 || DM_STRCMP(action, blobmsg_get_string(tb2[1])) != 0)
			continue;

		for (uint32_t i = 0; i < ev_data->obj_count; i++) {
			if (DM_STRCMP(ifname, ev_data->radio[i].obj_name) == 0) {
				ev_data->radio[i].scan_done = true;
				break;
			}
		}

		break;
	}

	// If scan finished for all radios then end the uloop
	bool scan_finished = true;
	for (uint32_t i = 0; i < ev_data->obj_count; i++) {
		if (!ev_data->radio[i].scan_done) {
			scan_finished = false;
			break;
		}
	}

	if (scan_finished)
		uloop_end();

	return;
}

static void start_wifi_radio_scan(struct uloop_timeout *timeout)
{
	struct blob_attr *object_list = NULL;
	struct blob_attr *cur = NULL;
	int rem = 0;

	struct dmubus_ev_subtask *data = container_of(timeout, struct dmubus_ev_subtask, sub_tm);

	if (data == NULL)
		return;

	struct blob_attr *task_data = (struct blob_attr *)(data->subtask_data);

	struct blob_attr *tb_data[1] = {0};
	const struct blobmsg_policy p_task_table[1] = {
		{ "task_data", BLOBMSG_TYPE_ARRAY }
	};

	blobmsg_parse(p_task_table, 1, tb_data, blobmsg_data(task_data), blobmsg_len(task_data));
	if (tb_data[0] == NULL)
		return;

	object_list = tb_data[0];

	const struct blobmsg_policy p_object[1] = {
		{ "object", BLOBMSG_TYPE_STRING }
	};

	blobmsg_for_each_attr(cur, object_list, rem) {
		struct blob_attr *tb[1] = {0};

		blobmsg_parse(p_object, 1, tb, blobmsg_data(cur), blobmsg_len(cur));
		if (!tb[0])
			continue;

		char *radio_object = blobmsg_get_string(tb[0]);

		if (DM_STRLEN(radio_object) == 0)
			continue;

		dmubus_call_set(radio_object, "scan", UBUS_ARGS{0}, 0);
	}
}

static int operate_WiFi_NeighboringWiFiDiagnostic(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	json_object *res = NULL;

	dmubus_call("wifi", "status", UBUS_ARGS{0}, 0, &res);
	if (!res) {
		fill_blob_param(&ctx->bb, "Status", "Error", DMT_TYPE[DMT_STRING], 0);
		goto end;
	}

	json_object *radios = NULL, *arrobj = NULL;
	int i = 0, j = 0;
	uint32_t index = 1;
	uint32_t radio_count = 0;
	struct blob_buf ev_data;
	struct blob_buf task_data;
	struct radio_scan_args events;

	json_object *radio_list = dmjson_get_obj(res, 1, "radios");
	if (!radio_list || json_object_get_type(radio_list) != json_type_array) {
		fill_blob_param(&ctx->bb, "Status", "Error", DMT_TYPE[DMT_STRING], 0);
		goto end;
	}

	radio_count = json_object_array_length(radio_list);
	if (!radio_count) {
		fill_blob_param(&ctx->bb, "Status", "Error", DMT_TYPE[DMT_STRING], 0);
		goto end;
	}

	events.radio = (struct radio_obj *)malloc(sizeof(struct radio_obj) * radio_count);
	if (!events.radio) {
		fill_blob_param(&ctx->bb, "Status", "Error", DMT_TYPE[DMT_STRING], 0);
		goto end;
	}

	memset(&ev_data, 0, sizeof(struct blob_buf));
	memset(&task_data, 0, sizeof(struct blob_buf));
	memset(events.radio, 0, sizeof(struct radio_obj) * radio_count);

	blob_buf_init(&ev_data, 0);
	blob_buf_init(&task_data, 0);

	void *ev_arr = blobmsg_open_array(&ev_data, "event_data");
	void *task_arr = blobmsg_open_array(&task_data, "task_data");

	dmjson_foreach_obj_in_array(res, arrobj, radios, i, 1, "radios") {

		char *radio_name = dmjson_get_value(radios, 1, "name");
		if (!DM_STRLEN(radio_name))
			continue;

		char *isup = dmjson_get_value(radios, 1, "isup");
		if (!DM_STRLEN(isup))
			continue;

		bool ifup = false;
		string_to_bool(isup, &ifup);
		if (!ifup)
			continue;

		if (j < radio_count) {
			char object[UBUS_OBJ_LEN] = {0};

			snprintf(object, sizeof(object), "wifi.radio.%s", radio_name);
			snprintf(events.radio[j].obj_name, sizeof(events.radio[j].obj_name), "%s", radio_name);

			void *ev_table = blobmsg_open_table(&ev_data, "");
			blobmsg_add_string(&ev_data, "ifname", radio_name);
			blobmsg_add_string(&ev_data, "event", "scan_finished");
			blobmsg_close_table(&ev_data, ev_table);

			void *task_table = blobmsg_open_table(&task_data, "");
			blobmsg_add_string(&task_data, "object", object);
			blobmsg_close_table(&task_data, task_table);

			j++;
		}
	}

	blobmsg_close_array(&ev_data, ev_arr);
	blobmsg_close_array(&task_data, task_arr);

	if (j) {
		events.obj_count = j;
		events.msg = ev_data.head;

		struct dmubus_ev_subtask subtask = {
			.sub_tm.cb = start_wifi_radio_scan,
			.subtask_data = task_data.head,
			.timeout = 1
		};

		dmubus_wait_for_event("wifi.radio", 30, &events, dmubus_receive_wifi_radio, &subtask);
	}

	blob_buf_free(&ev_data);
	blob_buf_free(&task_data);

	fill_blob_param(&ctx->bb, "Status", "Complete", DMT_TYPE[DMT_STRING], 0);

	for (i = 0; i < j; i++) {
		json_object *scan_res = NULL, *obj = NULL;
		char object[UBUS_OBJ_LEN] = {0};
		char buf[256] = {0};
		char *ssid = NULL;
		char *bssid = NULL;
		//char *noise = NULL;
		char *channel = NULL;
		char *frequency = NULL;
		char *signal_strength = NULL;
		char *standard = NULL;
		char *bandwidth = NULL;
		char *radio = NULL;
		char *encryption = NULL;
		char *ciphers = NULL;

		snprintf(object, sizeof(object), "wifi.radio.%s", events.radio[i].obj_name);
		adm_entry_get_reference_param(ctx, "Device.WiFi.Radio.*.Name", events.radio[i].obj_name, &radio);

		dmubus_call(object, "scanresults", UBUS_ARGS{0}, 0, &scan_res);

		if (!scan_res)
			continue;

		if (!json_object_object_get_ex(scan_res,"accesspoints", &obj))
			continue;

		uint32_t len = obj ? json_object_array_length(obj) : 0;

		for (uint32_t k = 0; k < len; k++ ) {
			if (index == 0)
				break;

			json_object *array_obj = json_object_array_get_idx(obj, k);
			ssid = dmjson_get_value(array_obj, 1, "ssid");
			bssid = dmjson_get_value(array_obj, 1, "bssid");
			channel = dmjson_get_value(array_obj, 1, "channel");
			frequency = dmjson_get_value(array_obj, 1, "band");
			signal_strength = dmjson_get_value(array_obj, 1, "rssi");
			//noise = dmjson_get_value(array_obj, 1, "noise");
			char *band = dmjson_get_value(array_obj, 1, "bandwidth");
			char *ccfs0 = dmjson_get_value(array_obj, 1, "ccfs0");
			char *ccfs1 = dmjson_get_value(array_obj, 1, "ccfs1");
			bandwidth = get_data_model_band(band, ccfs0, ccfs1);
			standard = get_data_model_standard(dmjson_get_value(array_obj, 1, "standard"));
			encryption = get_data_model_mode(dmjson_get_value(array_obj, 1, "encryption"));
			ciphers = get_encryption_mode(dmjson_get_value(array_obj, 1, "ciphers"));

			if (ctx->dm_type != BBFDM_USP) {
				struct uci_section *dmmap_s = NULL;
				dmuci_add_section_bbfdm("dmmap_wifi_neighboring", "result", &dmmap_s);
				dmuci_set_value_by_section(dmmap_s, "radio", radio);
				dmuci_set_value_by_section(dmmap_s, "ssid", ssid);
				dmuci_set_value_by_section(dmmap_s, "bssid", bssid);
				dmuci_set_value_by_section(dmmap_s, "channel", channel);
				dmuci_set_value_by_section(dmmap_s, "rssi", signal_strength);
				dmuci_set_value_by_section(dmmap_s, "band", frequency);
				//dmuci_set_value_by_section(dmmap_s, "noise", noise);
				dmuci_set_value_by_section(dmmap_s, "bandwidth", bandwidth);
				dmuci_set_value_by_section(dmmap_s, "standard", standard);
				dmuci_set_value_by_section(dmmap_s, "encryption", encryption);
				dmuci_set_value_by_section(dmmap_s, "ciphers", ciphers);
			}

			snprintf(buf, sizeof(buf), "Result.%u.Radio", index);
			fill_blob_param(&ctx->bb, buf, radio, DMT_TYPE[DMT_STRING], 0);

			snprintf(buf, sizeof(buf), "Result.%u.SSID", index);
			fill_blob_param(&ctx->bb, buf, ssid, DMT_TYPE[DMT_STRING], 0);

			snprintf(buf, sizeof(buf), "Result.%u.BSSID", index);
			fill_blob_param(&ctx->bb, buf, bssid, DMT_TYPE[DMT_STRING], 0);

			snprintf(buf, sizeof(buf), "Result.%u.Channel", index);
			fill_blob_param(&ctx->bb, buf, channel, DMT_TYPE[DMT_UNINT], 0);

			snprintf(buf, sizeof(buf), "Result.%u.OperatingFrequencyBand", index);
			fill_blob_param(&ctx->bb, buf, frequency, DMT_TYPE[DMT_STRING], 0);

			snprintf(buf, sizeof(buf), "Result.%u.SignalStrength", index);
			fill_blob_param(&ctx->bb, buf, signal_strength, DMT_TYPE[DMT_INT], 0);

			//snprintf(buf, sizeof(buf), "Result.%u.Noise", index);
			//fill_blob_param(&ctx->bb, buf, noise, DMT_TYPE[DMT_INT], 0);

			snprintf(buf, sizeof(buf), "Result.%u.OperatingChannelBandwidth", index);
			fill_blob_param(&ctx->bb, buf, bandwidth, DMT_TYPE[DMT_STRING], 0);

			snprintf(buf, sizeof(buf), "Result.%u.OperatingStandards", index);
			fill_blob_param(&ctx->bb, buf, standard, DMT_TYPE[DMT_STRING], 0);

			snprintf(buf, sizeof(buf), "Result.%u.SecurityModeEnabled", index);
			fill_blob_param(&ctx->bb, buf, encryption, DMT_TYPE[DMT_STRING], 0);

			snprintf(buf, sizeof(buf), "Result.%u.EncryptionMode", index);
			fill_blob_param(&ctx->bb, buf, ciphers, DMT_TYPE[DMT_STRING], 0);

			index++;
		}
	}

	FREE(events.radio);
end:
	if (ctx->dm_type != BBFDM_USP) {
		dmuci_set_value_bbfdm("dmmap_wifi_neighboring", "@diagnostic_status[0]", "DiagnosticsState", "Complete");
		dmuci_commit_package_bbfdm("dmmap_wifi_neighboring");
	}

	return 0;
}

static int operate_WiFiAccessPointSecurity_Reset(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	/* If mapcontroller not enabled then do not allow to change config on easymesh nodes */
	if (easymesh_enable == false && check_slo_ap(((struct dm_data *)data)->config_section) == false) {
		bbfdm_set_fault_message(ctx, "Config change on Easy-mesh node is not allowed on this device.");
		return USP_FAULT_REQUEST_DENIED;
	}

	dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "encryption", get_default_encryption());
	dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "key", get_default_key());

	if (check_slo_ap(((struct dm_data *)data)->config_section)) {
		/* WPS configurations are only allowed for standalone AP */
		dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "wps_pushbutton", "1");
	}

	return 0;
}

/**********************************************************************************************************************************
*                                            OBJ & LEAF DEFINITION
***********************************************************************************************************************************/
/* *** Device. *** */
DMOBJ tDeviceWiFiObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"WiFi", &DMREAD, NULL, NULL, "dir:/sys/class/ieee80211", NULL, NULL, NULL, tWiFiObj, tWiFiParams, NULL, BBFDM_BOTH, NULL},
{0}
};

/* *** Device.WiFi. *** */
DMOBJ tWiFiObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Radio", &DMREAD, NULL, NULL, NULL, browseWifiRadioInst, NULL, NULL, tWiFiRadioObj, tWiFiRadioParams, NULL, BBFDM_BOTH, NULL},
{"SSID", &DMWRITE, add_wifi_ssid, delete_wifi_ssid, NULL, browseWifiSsidInst, NULL, NULL, tWiFiSSIDObj, tWiFiSSIDParams, NULL, BBFDM_BOTH, NULL},
{"AccessPoint", &DMWRITE, add_wifi_accesspoint, delete_wifi_accesspoint, NULL, browseWifiAccessPointInst, NULL, NULL, tWiFiAccessPointObj, tWiFiAccessPointParams, NULL, BBFDM_BOTH, NULL},
{"NeighboringWiFiDiagnostic", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiNeighboringWiFiDiagnosticObj, tWiFiNeighboringWiFiDiagnosticParams, NULL, BBFDM_BOTH, NULL},
{"EndPoint", &DMWRITE, addObjWiFiEndPoint, delObjWiFiEndPoint, NULL, browseWiFiEndPointInst, NULL, NULL, tWiFiEndPointObj, tWiFiEndPointParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"RadioNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFi_RadioNumberOfEntries, NULL, BBFDM_BOTH},
{"SSIDNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFi_SSIDNumberOfEntries, NULL, BBFDM_BOTH},
{"AccessPointNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFi_AccessPointNumberOfEntries, NULL, BBFDM_BOTH},
{"EndPointNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFi_EndPointNumberOfEntries, NULL, BBFDM_BOTH},
{"Reset()", &DMSYNC, DMT_COMMAND, NULL, operate_WiFi_Reset, BBFDM_USP},
{"NeighboringWiFiDiagnostic()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFi_NeighboringWiFiDiagnostic, operate_WiFi_NeighboringWiFiDiagnostic, BBFDM_USP},
{0}
};

/* *** Device.WiFi.Radio.{i}. *** */
DMOBJ tWiFiRadioObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Stats", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiRadioStatsParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiRadioParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Alias", &DMWRITE, DMT_STRING, get_radio_alias, set_radio_alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Enable", &DMWRITE, DMT_BOOL, get_radio_enable, set_radio_enable, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_radio_status, NULL, BBFDM_BOTH},
{"LowerLayers", &DMWRITE, DMT_STRING, get_WiFiRadio_LowerLayers, set_WiFiRadio_LowerLayers, BBFDM_BOTH},
{"Name", &DMREAD, DMT_STRING, get_WiFiRadio_Name, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE|DM_FLAG_LINKER},
{"MaxBitRate", &DMREAD, DMT_UNINT, get_radio_max_bit_rate, NULL, BBFDM_BOTH},
{"SupportedFrequencyBands", &DMREAD, DMT_STRING, get_radio_supported_frequency_bands, NULL, BBFDM_BOTH},
{"OperatingFrequencyBand", &DMWRITE, DMT_STRING, get_radio_frequency, set_radio_frequency, BBFDM_BOTH},
{"SupportedStandards", &DMREAD, DMT_STRING, get_radio_supported_standard, NULL, BBFDM_BOTH},
{"OperatingStandards", &DMWRITE, DMT_STRING, get_radio_operating_standard, set_radio_operating_standard, BBFDM_BOTH},
{"ChannelsInUse", &DMREAD, DMT_STRING, get_radio_channels_in_use, NULL, BBFDM_BOTH},
{"Channel", &DMWRITE, DMT_UNINT, get_radio_channel, set_radio_channel, BBFDM_BOTH},
{"AutoChannelEnable", &DMWRITE, DMT_BOOL, get_radio_auto_channel_enable, set_radio_auto_channel_enable, BBFDM_BOTH},
{"PossibleChannels", &DMREAD, DMT_STRING, get_radio_possible_channels, NULL, BBFDM_BOTH},
{"AutoChannelSupported", &DMREAD, DMT_BOOL, get_WiFiRadio_AutoChannelSupported, NULL, BBFDM_BOTH},
{"AutoChannelRefreshPeriod", &DMWRITE, DMT_UNINT, get_WiFiRadio_AutoChannelRefreshPeriod, set_WiFiRadio_AutoChannelRefreshPeriod, BBFDM_BOTH},
{"MaxSupportedAssociations", &DMREAD, DMT_UNINT, get_WiFiRadio_MaxSupportedAssociations, NULL, BBFDM_BOTH},
{"FragmentationThreshold", &DMWRITE, DMT_UNINT, get_WiFiRadio_FragmentationThreshold, set_WiFiRadio_FragmentationThreshold, BBFDM_BOTH},
{"RTSThreshold", &DMWRITE, DMT_UNINT, get_WiFiRadio_RTSThreshold, set_WiFiRadio_RTSThreshold, BBFDM_BOTH},
{"BeaconPeriod", &DMWRITE, DMT_UNINT, get_WiFiRadio_BeaconPeriod, set_WiFiRadio_BeaconPeriod, BBFDM_BOTH},
{"DTIMPeriod", &DMWRITE, DMT_UNINT, get_WiFiRadio_DTIMPeriod, set_WiFiRadio_DTIMPeriod, BBFDM_BOTH},
{"SupportedOperatingChannelBandwidths", &DMREAD, DMT_STRING, get_WiFiRadio_SupportedOperatingChannelBandwidths, NULL, BBFDM_BOTH},
{"OperatingChannelBandwidth", &DMWRITE, DMT_STRING, get_WiFiRadio_OperatingChannelBandwidth, set_WiFiRadio_OperatingChannelBandwidth, BBFDM_BOTH},
{"CurrentOperatingChannelBandwidth", &DMREAD, DMT_STRING, get_WiFiRadio_CurrentOperatingChannelBandwidth, NULL, BBFDM_BOTH},
{"PreambleType", &DMWRITE, DMT_STRING, get_WiFiRadio_PreambleType, set_WiFiRadio_PreambleType, BBFDM_BOTH},
{"IEEE80211hSupported", &DMREAD, DMT_BOOL, get_WiFiRadio_IEEE80211hSupported, NULL, BBFDM_BOTH},
{"IEEE80211hEnabled", &DMWRITE, DMT_BOOL, get_WiFiRadio_IEEE80211hEnabled, set_WiFiRadio_IEEE80211hEnabled, BBFDM_BOTH},
{"TransmitPowerSupported", &DMREAD, DMT_STRING, get_WiFiRadio_TransmitPowerSupported, NULL, BBFDM_BOTH},
{"TransmitPower", &DMWRITE, DMT_INT, get_WiFiRadio_TransmitPower, set_WiFiRadio_TransmitPower, BBFDM_BOTH},
{"RegulatoryDomain", &DMWRITE, DMT_STRING, get_WiFiRadio_RegulatoryDomain, set_WiFiRadio_RegulatoryDomain, BBFDM_BOTH},
{"CenterFrequencySegment0", &DMREAD, DMT_UNINT, get_radio_ccfs0, NULL, BBFDM_BOTH},
{"CenterFrequencySegment1", &DMREAD, DMT_UNINT, get_radio_ccfs1, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.Radio.{i}.Stats. *** */
DMLEAF tWiFiRadioStatsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Noise", &DMREAD, DMT_INT, get_WiFiRadioStats_Noise, NULL, BBFDM_BOTH},
{"BytesSent", &DMREAD, DMT_UNLONG, get_WiFiRadioStats_BytesSent, NULL, BBFDM_BOTH},
{"BytesReceived", &DMREAD, DMT_UNLONG, get_WiFiRadioStats_BytesReceived, NULL, BBFDM_BOTH},
{"PacketsSent", &DMREAD, DMT_UNLONG, get_WiFiRadioStats_PacketsSent, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNLONG, get_WiFiRadioStats_PacketsReceived, NULL, BBFDM_BOTH},
{"ErrorsSent", &DMREAD, DMT_UNINT, get_WiFiRadioStats_ErrorsSent, NULL, BBFDM_BOTH},
{"ErrorsReceived", &DMREAD, DMT_UNINT, get_WiFiRadioStats_ErrorsReceived, NULL, BBFDM_BOTH},
{"DiscardPacketsSent", &DMREAD, DMT_UNINT, get_WiFiRadioStats_DiscardPacketsSent, NULL, BBFDM_BOTH},
{"DiscardPacketsReceived", &DMREAD, DMT_UNINT, get_WiFiRadioStats_DiscardPacketsReceived, NULL, BBFDM_BOTH},
{"FCSErrorCount", &DMREAD, DMT_UNINT, get_WiFiRadioStats_FCSErrorCount, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.NeighboringWiFiDiagnostic. *** */
DMOBJ tWiFiNeighboringWiFiDiagnosticObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Result", &DMREAD, NULL, NULL, NULL, browseWifiNeighboringWiFiDiagnosticResultInst, NULL, NULL, NULL, tWiFiNeighboringWiFiDiagnosticResultParams, NULL, BBFDM_CWMP, NULL},
{0}
};

DMLEAF tWiFiNeighboringWiFiDiagnosticParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"DiagnosticsState", &DMWRITE, DMT_STRING, get_neighboring_wifi_diagnostics_diagnostics_state, set_neighboring_wifi_diagnostics_diagnostics_state, BBFDM_CWMP},
{"ResultNumberOfEntries", &DMREAD, DMT_UNINT, get_neighboring_wifi_diagnostics_result_number_entries, NULL, BBFDM_CWMP},
{0}
};

/* *** Device.WiFi.NeighboringWiFiDiagnostic.Result.{i}. *** */
DMLEAF tWiFiNeighboringWiFiDiagnosticResultParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Radio", &DMREAD, DMT_STRING, get_neighboring_wifi_diagnostics_result_radio, NULL, BBFDM_CWMP},
{"SSID", &DMREAD, DMT_STRING, get_neighboring_wifi_diagnostics_result_ssid, NULL, BBFDM_CWMP},
{"BSSID", &DMREAD, DMT_STRING, get_neighboring_wifi_diagnostics_result_bssid, NULL, BBFDM_CWMP},
{"Channel", &DMREAD, DMT_UNINT, get_neighboring_wifi_diagnostics_result_channel, NULL, BBFDM_CWMP},
{"SignalStrength", &DMREAD, DMT_INT, get_neighboring_wifi_diagnostics_result_signal_strength, NULL, BBFDM_CWMP},
{"OperatingFrequencyBand", &DMREAD, DMT_STRING, get_neighboring_wifi_diagnostics_result_operating_frequency_band, NULL, BBFDM_CWMP},
//{"Noise", &DMREAD, DMT_INT, get_neighboring_wifi_diagnostics_result_noise, NULL, BBFDM_CWMP},
{"OperatingChannelBandwidth", &DMREAD, DMT_STRING, get_neighboring_wifi_diagnostics_result_operating_ch_band, NULL, BBFDM_CWMP},
{"OperatingStandards", &DMREAD, DMT_STRING, get_neighboring_wifi_diagnostics_result_op_std, NULL, BBFDM_CWMP},
{"SecurityModeEnabled", &DMREAD, DMT_STRING, get_neighboring_wifi_diagnostics_result_sec_mode, NULL, BBFDM_CWMP},
{"EncryptionMode", &DMREAD, DMT_STRING, get_neighboring_wifi_diagnostics_result_enc_mode, NULL, BBFDM_CWMP},
{0}
};

/* *** Device.WiFi.SSID.{i}. *** */
DMOBJ tWiFiSSIDObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Stats", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiSSIDStatsParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiSSIDParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Alias", &DMWRITE, DMT_STRING, get_ssid_alias, set_ssid_alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Enable", &DMWRITE, DMT_BOOL, get_wifi_ssid_enable, set_wifi_ssid_enable, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_wifi_status, NULL, BBFDM_BOTH},
{"SSID", &DMWRITE, DMT_STRING, get_wlan_ssid, set_wlan_ssid, BBFDM_BOTH},
{"MLDUnit", &DMWRITE, DMT_INT, get_wlan_ssid_mld, set_wlan_ssid_mld, BBFDM_BOTH},
{"Name", &DMREAD, DMT_STRING,  get_wlan_name, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE|DM_FLAG_LINKER},
{"LowerLayers", &DMWRITE, DMT_STRING, get_ssid_lower_layer, set_ssid_lower_layer, BBFDM_BOTH, DM_FLAG_REFERENCE},
{"BSSID", &DMREAD, DMT_STRING, get_wlan_bssid, NULL, BBFDM_BOTH},
{"MACAddress", &DMREAD, DMT_STRING, get_WiFiSSID_MACAddress, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.SSID.{i}.Stats. *** */
DMLEAF tWiFiSSIDStatsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"BytesSent", &DMREAD, DMT_UNLONG, get_WiFiSSIDStats_BytesSent, NULL, BBFDM_BOTH},
{"BytesReceived", &DMREAD, DMT_UNLONG, get_WiFiSSIDStats_BytesReceived, NULL, BBFDM_BOTH},
{"PacketsSent", &DMREAD, DMT_UNLONG, get_WiFiSSIDStats_PacketsSent, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNLONG, get_WiFiSSIDStats_PacketsReceived, NULL, BBFDM_BOTH},
{"ErrorsSent", &DMREAD, DMT_UNINT, get_WiFiSSIDStats_ErrorsSent, NULL, BBFDM_BOTH},
{"RetransCount", &DMREAD, DMT_UNINT, get_WiFiSSIDStats_RetransCount, NULL, BBFDM_BOTH},
{"FailedRetransCount", &DMREAD, DMT_UNINT, get_WiFiSSIDStats_FailedRetransCount, NULL, BBFDM_BOTH},
{"RetryCount", &DMREAD, DMT_UNINT, get_WiFiSSIDStats_RetryCount, NULL, BBFDM_BOTH},
{"MultipleRetryCount", &DMREAD, DMT_UNINT, get_WiFiSSIDStats_MultipleRetryCount, NULL, BBFDM_BOTH},
{"ACKFailureCount", &DMREAD, DMT_UNINT, get_WiFiSSIDStats_ACKFailureCount, NULL, BBFDM_BOTH},
{"AggregatedPacketCount", &DMREAD, DMT_UNINT, get_WiFiSSIDStats_AggregatedPacketCount, NULL, BBFDM_BOTH},
{"ErrorsReceived", &DMREAD, DMT_UNINT, get_WiFiSSIDStats_ErrorsReceived, NULL, BBFDM_BOTH},
{"UnicastPacketsSent", &DMREAD, DMT_UNLONG, get_WiFiSSIDStats_UnicastPacketsSent, NULL, BBFDM_BOTH},
{"UnicastPacketsReceived", &DMREAD, DMT_UNLONG, get_WiFiSSIDStats_UnicastPacketsReceived, NULL, BBFDM_BOTH},
{"DiscardPacketsSent", &DMREAD, DMT_UNINT, get_WiFiSSIDStats_DiscardPacketsSent, NULL, BBFDM_BOTH},
{"DiscardPacketsReceived", &DMREAD, DMT_UNINT, get_WiFiSSIDStats_DiscardPacketsReceived, NULL, BBFDM_BOTH},
{"MulticastPacketsSent", &DMREAD, DMT_UNLONG, get_WiFiSSIDStats_MulticastPacketsSent, NULL, BBFDM_BOTH},
{"MulticastPacketsReceived", &DMREAD, DMT_UNLONG, get_WiFiSSIDStats_MulticastPacketsReceived, NULL, BBFDM_BOTH},
{"BroadcastPacketsSent", &DMREAD, DMT_UNLONG, get_WiFiSSIDStats_BroadcastPacketsSent, NULL, BBFDM_BOTH},
{"BroadcastPacketsReceived", &DMREAD, DMT_UNLONG, get_WiFiSSIDStats_BroadcastPacketsReceived, NULL, BBFDM_BOTH},
{"UnknownProtoPacketsReceived", &DMREAD, DMT_UNINT, get_WiFiSSIDStats_UnknownProtoPacketsReceived, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.AccessPoint.{i}. *** */
DMOBJ tWiFiAccessPointObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Security", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiAccessPointSecurityParams, NULL, BBFDM_BOTH, NULL},
{"AssociatedDevice", &DMREAD, NULL, NULL, NULL, browse_wifi_associated_device, NULL, NULL, tWiFiAccessPointAssociatedDeviceObj, tWiFiAccessPointAssociatedDeviceParams, NULL, BBFDM_BOTH, NULL},
{"WPS", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiAccessPointWPSParams, NULL, BBFDM_BOTH, NULL},
{"Accounting", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiAccessPointAccountingParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiAccessPointParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Alias", &DMWRITE, DMT_STRING, get_access_point_alias, set_access_point_alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Enable", &DMWRITE, DMT_BOOL,  get_access_point_enable, set_access_point_enable, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_wifi_access_point_status, NULL, BBFDM_BOTH},
{"SSIDReference", &DMWRITE, DMT_STRING, get_ap_ssid_ref, set_ap_ssid_ref, BBFDM_BOTH, DM_FLAG_UNIQUE|DM_FLAG_REFERENCE},
{"SSIDAdvertisementEnabled", &DMWRITE, DMT_BOOL, get_wlan_ap_advertisement_enable, set_wlan_ap_advertisement_enable, BBFDM_BOTH},
{"WMMEnable", &DMWRITE, DMT_BOOL, get_wmm_enabled, set_wmm_enabled, BBFDM_BOTH},
{"UAPSDEnable", &DMWRITE, DMT_BOOL, get_WiFiAccessPoint_UAPSDEnable, set_WiFiAccessPoint_UAPSDEnable, BBFDM_BOTH},
{"AssociatedDeviceNumberOfEntries", &DMREAD, DMT_UNINT, get_access_point_total_associations, NULL, BBFDM_BOTH},
{"MACAddressControlEnabled", &DMWRITE, DMT_BOOL, get_access_point_control_enable, set_access_point_control_enable, BBFDM_BOTH},
{"UAPSDCapability", &DMREAD, DMT_BOOL, get_WiFiAccessPoint_UAPSDCapability, NULL, BBFDM_BOTH},
{"WMMCapability", &DMREAD, DMT_BOOL, get_WiFiAccessPoint_WMMCapability, NULL, BBFDM_BOTH},
{"MaxAllowedAssociations", &DMWRITE, DMT_UNINT, get_WiFiAccessPoint_MaxAllowedAssociations, set_WiFiAccessPoint_MaxAllowedAssociations, BBFDM_BOTH},
{"IsolationEnable", &DMWRITE, DMT_BOOL, get_WiFiAccessPoint_IsolationEnable, set_WiFiAccessPoint_IsolationEnable, BBFDM_BOTH},
{"AllowedMACAddress", &DMWRITE, DMT_STRING, get_WiFiAccessPoint_AllowedMACAddress, set_WiFiAccessPoint_AllowedMACAddress, BBFDM_BOTH},
{BBF_VENDOR_PREFIX"MultiAPMode", &DMWRITE, DMT_UNINT, get_WiFiAccessPoint_MultiAPMode, set_WiFiAccessPoint_MultiAPMode, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.AccessPoint.{i}.Security. *** */
DMLEAF tWiFiAccessPointSecurityParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"ModesSupported", &DMREAD, DMT_STRING, get_access_point_security_supported_modes, NULL, BBFDM_BOTH},
{"ModeEnabled", &DMWRITE, DMT_STRING, get_access_point_security_modes, set_access_point_security_modes, BBFDM_BOTH},
{"EncryptionMode", &DMREAD, DMT_STRING, get_access_point_encryption_mode, NULL, BBFDM_BOTH},
{"WEPKey", &DMWRITE, DMT_HEXBIN, get_access_point_security_wepkey, set_access_point_security_wepkey, BBFDM_BOTH, DM_FLAG_SECURE},
{"PreSharedKey", &DMWRITE, DMT_HEXBIN, get_access_point_security_shared_key, set_access_point_security_shared_key, BBFDM_BOTH, DM_FLAG_SECURE},
{"KeyPassphrase", &DMWRITE, DMT_STRING, get_access_point_security_passphrase, set_access_point_security_passphrase, BBFDM_BOTH, DM_FLAG_SECURE},
{"RekeyingInterval", &DMWRITE, DMT_UNINT, get_access_point_security_rekey_interval, set_access_point_security_rekey_interval, BBFDM_BOTH},
{"SAEPassphrase", &DMWRITE, DMT_STRING, get_WiFiAccessPointSecurity_SAEPassphrase, set_WiFiAccessPointSecurity_SAEPassphrase, BBFDM_BOTH, DM_FLAG_SECURE},
{"RadiusServerIPAddr", &DMWRITE, DMT_STRING, get_access_point_security_radius_ip_address, set_access_point_security_radius_ip_address, BBFDM_BOTH},
{"RadiusServerPort", &DMWRITE, DMT_UNINT, get_access_point_security_radius_server_port, set_access_point_security_radius_server_port, BBFDM_BOTH},
{"RadiusSecret", &DMWRITE, DMT_STRING, get_access_point_security_radius_secret, set_access_point_security_radius_secret, BBFDM_BOTH, DM_FLAG_SECURE},
{"MFPConfig", &DMWRITE, DMT_STRING, get_WiFiAccessPointSecurity_MFPConfig, set_WiFiAccessPointSecurity_MFPConfig, BBFDM_BOTH},
{"Reset()", &DMSYNC, DMT_COMMAND, NULL, operate_WiFiAccessPointSecurity_Reset, BBFDM_USP},
{0}
};

/* *** Device.WiFi.AccessPoint.{i}.WPS. *** */
DMLEAF tWiFiAccessPointWPSParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_WiFiAccessPointWPS_Enable, set_WiFiAccessPointWPS_Enable, BBFDM_BOTH},
{"ConfigMethodsSupported", &DMREAD, DMT_STRING, get_WiFiAccessPointWPS_ConfigMethodsSupported, NULL, BBFDM_BOTH},
{"ConfigMethodsEnabled", &DMWRITE, DMT_STRING, get_WiFiAccessPointWPS_ConfigMethodsEnabled, set_WiFiAccessPointWPS_ConfigMethodsEnabled, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_WiFiAccessPointWPS_Status, NULL, BBFDM_BOTH},
//{"Version", &DMREAD, DMT_STRING, get_WiFiAccessPointWPS_Version, NULL, BBFDM_BOTH},
//{"PIN", &DMWRITE, DMT_STRING, get_WiFiAccessPointWPS_PIN, set_WiFiAccessPointWPS_PIN, BBFDM_BOTH, DM_FLAG_SECURE},
{"InitiateWPSPBC()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiAccessPointWPS_InitiateWPSPBC, operate_WiFiAccessPointWPS_InitiateWPSPBC, BBFDM_USP},
{0}
};

/* *** Device.WiFi.AccessPoint.{i}.AssociatedDevice.{i}. *** */
DMOBJ tWiFiAccessPointAssociatedDeviceObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Stats", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiAccessPointAssociatedDeviceStatsParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiAccessPointAssociatedDeviceParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Active", &DMREAD, DMT_BOOL, get_WiFiAccessPointAssociatedDevice_Active, NULL, BBFDM_BOTH},
{"Noise", &DMREAD, DMT_INT, get_WiFiAccessPointAssociatedDevice_Noise, NULL, BBFDM_BOTH},
{"MACAddress", &DMREAD, DMT_STRING, get_WiFiAccessPointAssociatedDevice_MACAddress, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE|DM_FLAG_LINKER},
{"LastDataDownlinkRate", &DMREAD, DMT_UNINT, get_WiFiAccessPointAssociatedDevice_LastDataDownlinkRate, NULL, BBFDM_BOTH},
{"LastDataUplinkRate", &DMREAD, DMT_UNINT, get_WiFiAccessPointAssociatedDevice_LastDataUplinkRate, NULL, BBFDM_BOTH},
{"SignalStrength", &DMREAD, DMT_INT, get_WiFiAccessPointAssociatedDevice_SignalStrength, NULL, BBFDM_BOTH},
//{"Retransmissions", &DMREAD, DMT_UNINT, get_WiFiAccessPointAssociatedDevice_Retransmissions, NULL, BBFDM_BOTH},
{"AssociationTime", &DMREAD, DMT_TIME, get_WiFiAccessPointAssociatedDevice_AssociationTime, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.AccessPoint.{i}.AssociatedDevice.{i}.Stats. *** */
DMLEAF tWiFiAccessPointAssociatedDeviceStatsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"BytesSent", &DMREAD, DMT_UNLONG, get_access_point_associative_device_statistics_tx_bytes, NULL, BBFDM_BOTH},
{"BytesReceived", &DMREAD, DMT_UNLONG, get_access_point_associative_device_statistics_rx_bytes, NULL, BBFDM_BOTH},
{"PacketsSent", &DMREAD, DMT_UNLONG, get_access_point_associative_device_statistics_tx_packets, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNLONG, get_access_point_associative_device_statistics_rx_packets, NULL, BBFDM_BOTH},
{"ErrorsSent", &DMREAD, DMT_UNINT, get_access_point_associative_device_statistics_tx_errors, NULL, BBFDM_BOTH},
{"RetransCount", &DMREAD, DMT_UNINT, get_access_point_associative_device_statistics_retrans_count, NULL, BBFDM_BOTH},
//{"FailedRetransCount", &DMREAD, DMT_UNINT, get_access_point_associative_device_statistics_failed_retrans_count, NULL, BBFDM_BOTH},
//{"RetryCount", &DMREAD, DMT_UNINT, get_access_point_associative_device_statistics_retry_count, NULL, BBFDM_BOTH},
//{"MultipleRetryCount", &DMREAD, DMT_UNINT, get_access_point_associative_device_statistics_multiple_retry_count, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.AccessPoint.{i}.Accounting. *** */
DMLEAF tWiFiAccessPointAccountingParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
//{"Enable", &DMWRITE, DMT_BOOL, get_WiFiAccessPointAccounting_Enable, set_WiFiAccessPointAccounting_Enable, BBFDM_BOTH},
{"ServerIPAddr", &DMWRITE, DMT_STRING, get_WiFiAccessPointAccounting_ServerIPAddr, set_WiFiAccessPointAccounting_ServerIPAddr, BBFDM_BOTH},
//{"SecondaryServerIPAddr", &DMWRITE, DMT_STRING, get_WiFiAccessPointAccounting_SecondaryServerIPAddr, set_WiFiAccessPointAccounting_SecondaryServerIPAddr, BBFDM_BOTH},
{"ServerPort", &DMWRITE, DMT_UNINT, get_WiFiAccessPointAccounting_ServerPort, set_WiFiAccessPointAccounting_ServerPort, BBFDM_BOTH},
//{"SecondaryServerPort", &DMWRITE, DMT_UNINT, get_WiFiAccessPointAccounting_SecondaryServerPort, set_WiFiAccessPointAccounting_SecondaryServerPort, BBFDM_BOTH},
{"Secret", &DMWRITE, DMT_STRING, get_WiFiAccessPointAccounting_Secret, set_WiFiAccessPointAccounting_Secret, BBFDM_BOTH, DM_FLAG_SECURE},
//{"SecondarySecret", &DMWRITE, DMT_STRING, get_WiFiAccessPointAccounting_SecondarySecret, set_WiFiAccessPointAccounting_SecondarySecret, BBFDM_BOTH},
//{"InterimInterval", &DMWRITE, DMT_UNINT, get_WiFiAccessPointAccounting_InterimInterval, set_WiFiAccessPointAccounting_InterimInterval, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.EndPoint.{i}. *** */
DMOBJ tWiFiEndPointObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Stats", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiEndPointStatsParams, NULL, BBFDM_BOTH, NULL},
{"Security", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiEndPointSecurityParams, NULL, BBFDM_BOTH, NULL},
{"Profile", &DMREAD, NULL, NULL, NULL, browseWiFiEndPointProfileInst, NULL, NULL, tWiFiEndPointProfileObj, tWiFiEndPointProfileParams, NULL, BBFDM_BOTH, NULL},
{"WPS", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiEndPointWPSParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiEndPointParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_WiFiEndPoint_Enable, set_WiFiEndPoint_Enable, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_WiFiEndPoint_Status, NULL, BBFDM_BOTH},
{"Alias", &DMWRITE, DMT_STRING, get_WiFiEndPoint_Alias, set_WiFiEndPoint_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
//{"ProfileReference", &DMWRITE, DMT_STRING, get_WiFiEndPoint_ProfileReference, set_WiFiEndPoint_ProfileReference, BBFDM_BOTH},
{"SSIDReference", &DMREAD, DMT_STRING, get_WiFiEndPoint_SSIDReference, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE|DM_FLAG_REFERENCE},
{"ProfileNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiEndPoint_ProfileNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.EndPoint.{i}.Stats. *** */
DMLEAF tWiFiEndPointStatsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"LastDataDownlinkRate", &DMREAD, DMT_UNINT, get_WiFiEndPointStats_LastDataDownlinkRate, NULL, BBFDM_BOTH},
{"LastDataUplinkRate", &DMREAD, DMT_UNINT, get_WiFiEndPointStats_LastDataUplinkRate, NULL, BBFDM_BOTH},
{"SignalStrength", &DMREAD, DMT_INT, get_WiFiEndPointStats_SignalStrength, NULL, BBFDM_BOTH},
{"Retransmissions", &DMREAD, DMT_UNINT, get_WiFiEndPointStats_Retransmissions, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.EndPoint.{i}.Security. *** */
DMLEAF tWiFiEndPointSecurityParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"ModesSupported", &DMREAD, DMT_STRING, get_WiFiEndPointSecurity_ModesSupported, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.EndPoint.{i}.Profile.{i}. *** */
DMOBJ tWiFiEndPointProfileObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Security", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiEndPointProfileSecurityParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiEndPointProfileParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_WiFiEndPointProfile_Enable, set_WiFiEndPointProfile_Enable, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_WiFiEndPointProfile_Status, NULL, BBFDM_BOTH},
{"Alias", &DMWRITE, DMT_STRING, get_WiFiEndPointProfile_Alias, set_WiFiEndPointProfile_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"SSID", &DMWRITE, DMT_STRING, get_WiFiEndPointProfile_SSID, set_WiFiEndPointProfile_SSID, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Location", &DMWRITE, DMT_STRING, get_WiFiEndPointProfile_Location, set_WiFiEndPointProfile_Location, BBFDM_BOTH, DM_FLAG_UNIQUE},
//{"Priority", &DMWRITE, DMT_UNINT, get_WiFiEndPointProfile_Priority, set_WiFiEndPointProfile_Priority, BBFDM_BOTH, DM_FLAG_UNIQUE},
{0}
};

/* *** Device.WiFi.EndPoint.{i}.Profile.{i}.Security. *** */
DMLEAF tWiFiEndPointProfileSecurityParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"ModeEnabled", &DMWRITE, DMT_STRING, get_WiFiEndPointProfileSecurity_ModeEnabled, set_WiFiEndPointProfileSecurity_ModeEnabled, BBFDM_BOTH},
{"WEPKey", &DMWRITE, DMT_HEXBIN, get_WiFiEndPointProfileSecurity_WEPKey, set_WiFiEndPointProfileSecurity_WEPKey, BBFDM_BOTH, DM_FLAG_SECURE},
{"PreSharedKey", &DMWRITE, DMT_HEXBIN, get_WiFiEndPointProfileSecurity_PreSharedKey, set_WiFiEndPointProfileSecurity_PreSharedKey, BBFDM_BOTH, DM_FLAG_SECURE},
{"KeyPassphrase", &DMWRITE, DMT_STRING, get_WiFiEndPointProfileSecurity_KeyPassphrase, set_WiFiEndPointProfileSecurity_KeyPassphrase, BBFDM_BOTH, DM_FLAG_SECURE},
{"SAEPassphrase", &DMWRITE, DMT_STRING, get_WiFiEndPointProfileSecurity_SAEPassphrase, set_WiFiEndPointProfileSecurity_SAEPassphrase, BBFDM_BOTH, DM_FLAG_SECURE},
{"MFPConfig", &DMWRITE, DMT_STRING, get_WiFiEndPointProfileSecurity_MFPConfig, set_WiFiEndPointProfileSecurity_MFPConfig, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.EndPoint.{i}.WPS. *** */
DMLEAF tWiFiEndPointWPSParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_WiFiEndPointWPS_Enable, set_WiFiEndPointWPS_Enable, BBFDM_BOTH},
{"ConfigMethodsSupported", &DMREAD, DMT_STRING, get_WiFiEndPointWPS_ConfigMethodsSupported, NULL, BBFDM_BOTH},
{"ConfigMethodsEnabled", &DMWRITE, DMT_STRING, get_WiFiEndPointWPS_ConfigMethodsEnabled, set_WiFiEndPointWPS_ConfigMethodsEnabled, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_WiFiEndPointWPS_Status, NULL, BBFDM_BOTH},
//{"Version", &DMREAD, DMT_STRING, get_WiFiEndPointWPS_Version, NULL, BBFDM_BOTH},
//{"PIN", &DMWRITE, DMT_UNINT, get_WiFiEndPointWPS_PIN, set_WiFiEndPointWPS_PIN, BBFDM_BOTH},
{0}
};
