/*
 * Copyright (C) 2023-2024 IOPSYS Software Solutions AB. All rights reserved.
 *
 *
 * See LICENSE file for license related information.
 *
 * Author: Amin Ben Romdhane <amin.benromdhane@iopsys.eu>
 *
 */

#include <libbbfdm-api/bbfdm_api.h>
#include <libbbfdm-ubus/bbfdm-ubus.h>

#include "dataelements.h"
#include "common.h"

#define ZEROTOUCH_KEYFILE "/etc/multiap/.zt-keys.json"

enum set_ssid_operations {
	SET_SSID_ADD,
	SET_SSID_REMOVE,
	SET_SSID_CHANGE,
	SET_SSID_INVALID
};

typedef struct set_ssid_operate_param {
	char *ssid;
	char *new_ssid;
	char *enab;
	char *key;
	char *band;
	char *akm;
	char *adv;
	char *mfp;
	char *haul;
	char *network_type;
	char *mld_enab;
	char *ts;
} set_ssid_param;

typedef struct validate_ssid_param {
	int op;
	int status;
	char *ssid;
} validate_ssid;

static json_object *wifi_de_dump_output = NULL;

static bool validate_wifi_de_dump_output(json_object *dump_output)
{
	if (!dump_output)
		return false;

	json_object *data_arr = NULL;
	if (!json_object_object_get_ex(dump_output, "data", &data_arr) ||
		json_object_get_type(data_arr) != json_type_array)
		return false;

	json_object *data_obj = json_object_array_get_idx(data_arr, 0);
	if (!data_obj)
		return false;

	json_object *da_network_obj = NULL;
	if (!json_object_object_get_ex(data_obj, "wfa-dataelements:Network", &da_network_obj) ||
		json_object_get_type(da_network_obj) != json_type_object)
		return false;

	json_object *dev_num_obj = NULL;
	if (!json_object_object_get_ex(da_network_obj, "DeviceNumberOfEntries", &dev_num_obj) ||
		json_object_get_type(dev_num_obj) != json_type_int)
		return false;

	return json_object_get_uint64(dev_num_obj) > 0;
}

static void wifi_dataelements_dump_callback(struct ubus_request *req, int type, struct blob_attr *msg)
{
	if (!msg) {
		BBF_ERR("Error: Received NULL message in (%s)", __func__);
		return;
	}

	char *str = blobmsg_format_json_indent(msg, true, -1);
	if (!str) {
		BBF_ERR("Error: Failed to format JSON from blobmsg in (%s)", __func__);
		return;
	}

	json_object *dump_output = json_tokener_parse(str);
	free((char *)str);

	if (!dump_output) {
		BBF_ERR("Error: Failed to parse JSON string in (%s)", __func__);
		return;
	}

	if (!validate_wifi_de_dump_output(dump_output)) {
		BBF_ERR("Error: WiFi DataElements Dump Output is not valid in (%s)", __func__);
		json_object_put(dump_output);
		return;
	}

	if (wifi_de_dump_output != NULL) {
		BBF_DEBUG("Warning: WiFi DataElements Dump Output is not empty. Freeing previous output before reusing");
		json_object_put(wifi_de_dump_output);
	}

	wifi_de_dump_output = dump_output;

	// Refresh References Data Base
	bbfdm_refresh_references(BBFDM_BOTH, "bbfdm."SERVICE_NAME);
	BBF_INFO("WiFi DataElements Dump Output has been successfully updated");
}

/*************************************************************
* GLOBAL FUNCTIONS
**************************************************************/
void init_wifi_dataelements_dump(void)
{
	struct blob_buf bb = {0};

	BBF_DEBUG("Initializing/Refreshing WiFi DataElements Dump Output");

	memset(&bb, 0, sizeof(struct blob_buf));
	blob_buf_init(&bb, 0);

	BBFDM_UBUS_INVOKE_SYNC("wifi.dataelements", "dump", bb.head, 5000, wifi_dataelements_dump_callback, NULL);

	blob_buf_free(&bb);
}

void clean_wifi_dataelements_dump(void)
{
	BBF_DEBUG("Cleaning WiFi DataElements Dump Output");

	if (wifi_de_dump_output != NULL) {
		json_object_put(wifi_de_dump_output);
		wifi_de_dump_output = NULL;
	}
}

/*************************************************************
* COMMON FUNCTIONS
**************************************************************/
static void reload_mapcontroller_service(unsigned int proto, bool is_commit, bool reload_required)
{
	struct blob_buf bb = {0};

	memset(&bb, 0, sizeof(struct blob_buf));

	blob_buf_init(&bb, 0);

	void *array = blobmsg_open_array(&bb, "services");
	blobmsg_add_string(&bb, NULL, "mapcontroller");
	blobmsg_close_array(&bb, array);

	blobmsg_add_string(&bb, "proto", (proto == BBFDM_USP) ? "usp" : "both");
	blobmsg_add_u8(&bb, "reload", reload_required);

	dmubus_call_blob_msg_timeout("bbf.config", is_commit ? "commit" : "revert", &bb, 10000);

	blob_buf_free(&bb);
}

static char *get_AKMs_uci_name(const char *value)
{
	if (DM_LSTRCMP(value, "psk") == 0) {
		return "psk2";
	} else if(DM_LSTRCMP(value, "dpp") == 0) {
		return "dpp";
	} else if(DM_LSTRCMP(value, "sae") == 0) {
		return "sae";
	} else if(DM_LSTRCMP(value, "psk+sae") == 0) {
		return "sae-mixed";
	} else if(DM_LSTRCMP(value, "dpp+sae") == 0) {
		return "dpp+sae";
	} else if(DM_LSTRCMP(value, "dpp+psk+sae") == 0) {
		return "dpp+sae-mixed";
	}

	return "sae-mixed";
}

static char *get_AKMs_dm_name(const char *value)
{
	if (DM_LSTRCMP(value, "psk2") == 0) {
		return dmstrdup("psk");
	} else if(DM_LSTRCMP(value, "dpp") == 0) {
		return dmstrdup("dpp");
	} else if(DM_LSTRCMP(value, "sae") == 0) {
		return dmstrdup("sae");
	} else if(DM_LSTRCMP(value, "sae-mixed") == 0) {
		return dmstrdup("psk+sae");
	} else if(DM_LSTRCMP(value, "dpp+sae") == 0) {
		return dmstrdup("dpp+sae");
	} else if(DM_LSTRCMP(value, "dpp+sae-mixed") == 0) {
		return dmstrdup("dpp+psk+sae");
	}

	return dmstrdup("");
}

static char *get_haultype_uci_value(const char *value)
{
	if (DM_LSTRCMP(value, "Backhaul") == 0) {
		return "backhaul";
	}

	return "fronthaul";
}

static int get_requested_operation(const char *req)
{
	if (DM_LSTRCMP(req, "Add") == 0)
		return SET_SSID_ADD;

	if (DM_LSTRCMP(req, "Remove") == 0)
		return SET_SSID_REMOVE;

	if (DM_LSTRCMP(req, "Change") == 0)
		return SET_SSID_CHANGE;

	return SET_SSID_INVALID;
}

static int validate_band_value(struct dmctx *ctx, const char *band)
{
	char *band_list[] = {"2.4", "5", "6", "All", NULL};

	if (DM_STRLEN(band) == 0)
		return 0;

	if (bbfdm_validate_string_list(ctx, band, -1, -1, -1, -1, -1, band_list, NULL))
		return -1;

	/* if "All" is present then no other values are allowed in list */
	if (DM_STRSTR(band, "All") != NULL && DM_STRLEN(band) > 3)
		return -1;

	if (DM_STRSTR(band, "All") == NULL) {
		size_t band_arr_length = 0, i;
		char **band_arr = strsplit(band, ",", &band_arr_length);

		/* Only allow supported bands */
		for (i = 0; i < band_arr_length; i++) {
			bool valid = false;
			struct radio_info *node = NULL;

			list_for_each_entry(node, &dev_list, list) {
				if (node->band[0] == band_arr[i][0]) {
					valid = true;
					break;
				}
			}

			if (valid == false) {
				bbfdm_set_fault_message(ctx, "Device not supports band (%s).", band_arr[i]);
				return -1;
			}
		}
	}

	return 0;
}

static int validate_akms_value(struct dmctx *ctx, const char *akms)
{
	char *akms_list[] = {"psk", "dpp", "sae", "psk+sae", "dpp+sae", "dpp+psk+sae", NULL};

	if (DM_STRLEN(akms) == 0)
		return 0;

	if (bbfdm_validate_string_list(ctx, akms, -1, -1, -1, -1, -1, akms_list, NULL)) {
		bbfdm_set_fault_message(ctx, "Only supports psk,dpp,sae,psk+sae,dpp+sae,dpp+psk+sae.");
		return -1;
	}

	return 0;
}

static int validate_haultype_value(struct dmctx *ctx, const char *haul)
{
	char *haul_list[] = {"Fronthaul", "Backhaul", NULL};

	if (DM_STRLEN(haul) == 0)
		return 0;

	if (bbfdm_validate_string(ctx, haul, -1, -1, haul_list, NULL)) {
		bbfdm_set_fault_message(ctx, "Invalid haultype (%s).", haul);
		return -1;
	}

	return 0;
}

/*************************************************************
* ADD DEL OBJ
**************************************************************/
static int addObjWiFiDataElementsNetworkProvisionedDPP(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct uci_section *dmmap = NULL;
	char s_name[16] = {0};

	snprintf(s_name, sizeof(s_name), "dpp_%s", *instance);

	dmuci_add_section_bbfdm("dmmap_wireless", "dpp", &dmmap);
	dmuci_rename_section_by_section(dmmap, s_name);

	dmuci_set_value_by_section(dmmap, "dpp_instance", *instance);
	return 0;
}

static int delObjWiFiDataElementsNetworkProvisionedDPP(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	char *uri, *key;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "uri", &uri);
	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "key", &key);

	if (DM_STRLEN(uri) || DM_STRLEN(key)) {
		bbfdm_set_fault_message(ctx,
			"DPP instance contains a defined URI or Passphrase. Instances with active credentials cannot be deleted.");
		return FAULT_9001;
	}

	dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);
	return 0;
}

/*************************************************************
* ENTRY METHOD
**************************************************************/
static int browseWiFiDataElementsNetworkSSIDInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct uci_section *s = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	uci_foreach_sections("mapcontroller", "ap", s) {
		char *ssid = NULL, *band = NULL;

		dmuci_get_value_by_section_string(s, "ssid", &ssid);
		dmuci_get_value_by_section_string(s, "band", &band);

		if (DM_STRLEN(ssid) == 0 || DM_STRLEN(band) == 0)
			continue;

		memset(&curr_data, 0, sizeof(struct dm_data));
		curr_data.config_section = s;
		curr_data.dmmap_section = get_dup_section_in_dmmap_opt("dmmap_wireless", "ssid", "ap_section_name", section_name(s));

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}

	return 0;
}

static int browseWiFiDataElementsNetworkProvisionedDPPInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *res = NULL, *arr = NULL;
	struct uci_section *dpp_s = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL, *uri = NULL;
	int i = 0, rv, len;

	dmubus_call("map.controller", "dpp_enrollee_uri", UBUS_ARGS{0}, 0, &res);
	if (res) {
		dmjson_foreach_value_in_array(res, arr, uri, i, 1, "uris") {
			dpp_s = get_dup_section_in_dmmap_opt("dmmap_wireless", "dpp", "uri", uri);
			if (dpp_s)
				continue;

			dmuci_add_section_bbfdm("dmmap_wireless", "dpp", &dpp_s);
			dmuci_set_value_by_section(dpp_s, "uri", uri);
		}		
	}

	res = json_object_from_file(ZEROTOUCH_KEYFILE);
	if (res) {
		rv = json_object_object_get_ex(res, "keys", &arr);
		if (!rv || !arr)
			goto out_put_json;

		if (!json_object_is_type(arr, json_type_array))
			goto out_put_json;

		len = json_object_array_length(arr);
		for (i = 0; i < len; i++) {
			struct json_object *t = json_object_array_get_idx(arr, i);
			const char *key = json_object_get_string(t);

			dpp_s = get_dup_section_in_dmmap_opt("dmmap_wireless", "dpp", "key", key);
			if (dpp_s)
				continue;

			dmuci_add_section_bbfdm("dmmap_wireless", "dpp", &dpp_s);
			dmuci_set_value_by_section(dpp_s, "key", key);
		}

		out_put_json:
			json_object_put(res);
	}

	uci_path_foreach_sections(bbfdm, "dmmap_wireless", "dpp", dpp_s) {
		curr_data.dmmap_section = dpp_s;
		inst = handle_instance(dmctx, parent_node, dpp_s, "dpp_instance", "dpp_alias");
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static struct uci_section *find_device_uci_section(const char *unique_key)
{
	struct uci_section *s = NULL;

	uci_foreach_sections("mapcontroller", "node", s) {
		char *agent_id = NULL;

		dmuci_get_value_by_section_string(s, "agent_id", &agent_id);

		// Device found ==> return the current device uci section
		if (DM_STRCMP(agent_id, unique_key) == 0)
			return s;
	}

	return NULL;
}

static struct uci_section *find_radio_uci_section(const char *agent_id, const char *unique_key)
{
	struct uci_section *s = NULL;

	uci_foreach_option_eq("mapcontroller", "radio", "agent_id", agent_id, s) {
		char *macaddr = NULL;

		dmuci_get_value_by_section_string(s, "macaddr", &macaddr);

		// Radio found ==> return the current radio uci section
		if (DM_STRCMP(macaddr, unique_key) == 0)
			return s;
	}

	return NULL;
}

static int browseWiFiDataElementsNetworkDeviceInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *data_arr = NULL, *data_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int i = 0, id = 0;

	if (wifi_de_dump_output == NULL)
		return -1;

	dmjson_foreach_obj_in_array(wifi_de_dump_output, data_arr, data_obj, i, 1, "data") {
		json_object *dev_arr = NULL, *dev_obj = NULL;
		int j = 0;

		dmjson_foreach_obj_in_array(data_obj, dev_arr, dev_obj, j, 2, "wfa-dataelements:Network", "DeviceList") {

			char *key = dmjson_get_value(dev_obj, 1, "ID");
			if (!key || *key == '\0')
				continue;

			curr_data.json_object = dev_obj;
			curr_data.config_section = find_device_uci_section(key);

			inst = handle_instance_without_section(dmctx, parent_node, ++id);

			if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
				return 0;
		}
	}

	return 0;
}

static int browseWiFiDataElementsNetworkDeviceDefault8021QInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	DM_LINK_INST_OBJ(dmctx, parent_node, prev_data, "1");
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceSSIDtoVIDMappingInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *ssid_to_vid_arr = NULL, *ssid_to_vid_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, ssid_to_vid_arr, ssid_to_vid_obj, i, 1, "SSIDtoVIDMapping") {

		curr_data.json_object = ssid_to_vid_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceCACStatusInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *cac_status_arr = NULL, *cac_status_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, cac_status_arr, cac_status_obj, i, 1, "CACStatus") {

		curr_data.json_object = cac_status_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceIEEE1905SecurityInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *ieee1905_security_arr = NULL, *ieee1905_security_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, ieee1905_security_arr, ieee1905_security_obj, i, 1, "IEEE1905Security") {

		curr_data.json_object = ieee1905_security_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}


static int browseWiFiDataElementsNetworkDeviceRadioInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct json_object *radio_arr = NULL, *radio_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int i = 0, id = 0;

	char *agent_id = dmjson_get_value(((struct dm_data *)prev_data)->json_object, 1, "ID");

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, radio_arr, radio_obj, i, 1, "RadioList") {

		char mac[32] = {0};
		char *radio_id = dmjson_get_value(radio_obj, 1, "ID");
		char *str = dm_base64_decode(radio_id);

		/* Cant use strlen on byte array that might genuinely include 0x00 */
		/* but to get 6 bytes, need 8 input BASE64 chars - check for that */
		if ((str != NULL) && (DM_STRLEN(radio_id) == 8))
			string_to_mac(str, 6, mac, sizeof(mac));

		if (DM_STRLEN(mac) == 0)
			continue;

		curr_data.json_object = radio_obj;
		curr_data.config_section = find_radio_uci_section(agent_id, mac);

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}

	return 0;
}

static int browseWiFiDataElementsNetworkDeviceAPMLDInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct json_object *apmld_arr = NULL, *apmld_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int i = 0, id = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, apmld_arr, apmld_obj, i, 1, "APMLDList") {

		curr_data.json_object = apmld_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}

	return 0;
}

static int browseWiFiDataElementsNetworkDeviceCACStatusCACAvailableChannelInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *cac_available_channel_arr = NULL, *cac_available_channel_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, cac_available_channel_arr, cac_available_channel_obj, i, 1, "CACAvailableChannel") {

		curr_data.json_object = cac_available_channel_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannelInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *cac_non_occupancy_channel_arr = NULL, *cac_non_occupancy_channel_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, cac_non_occupancy_channel_arr, cac_non_occupancy_channel_obj, i, 1, "CACNonOccupancyChannel") {

		curr_data.json_object = cac_non_occupancy_channel_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceCACStatusCACActiveChannelInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *cac_active_channel_arr = NULL, *cac_active_channel_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, cac_active_channel_arr, cac_active_channel_obj, i, 1, "CACActiveChannel") {

		curr_data.json_object = cac_active_channel_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfileInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *opclass_arr = NULL, *opclass_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, opclass_arr, opclass_obj, i, 1, "CurrentOperatingClassProfile") {

		curr_data.json_object = opclass_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioBSSInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *bss_arr = NULL, *bss_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, bss_arr, bss_obj, i, 1, "BSSList") {

		curr_data.json_object = bss_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioScanResultInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *scanres_arr = NULL, *scanres_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, scanres_arr, scanres_obj, i, 1, "ScanResult") {

		curr_data.json_object = scanres_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

/*static int browseWiFiDataElementsNetworkDeviceRadioDisAllowedOpClassChannelsInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	//TODO
	return 0;
}
*/

static int browseWiFiDataElementsNetworkDeviceRadioUnassociatedSTAInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *unassoc_arr = NULL, *unassoc_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, unassoc_arr, unassoc_obj, i, 1, "UnassociatedSTA") {

		curr_data.json_object = unassoc_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLMRFreqSeparationInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *freq_sep_arr = NULL, *freq_sep_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, freq_sep_arr, freq_sep_obj, i, 3, "Capabilities", "WiFi7APRole", "EMLMRFreqSeparation") {

		curr_data.json_object = freq_sep_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLSRFreqSeparationInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *freq_sep_arr = NULL, *freq_sep_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, freq_sep_arr, freq_sep_obj, i, 3, "Capabilities", "WiFi7APRole", "EMLSRFreqSeparation") {

		curr_data.json_object = freq_sep_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleSTRFreqSeparationInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *freq_sep_arr = NULL, *freq_sep_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, freq_sep_arr, freq_sep_obj, i, 3, "Capabilities", "WiFi7APRole", "STRFreqSeparation") {

		curr_data.json_object = freq_sep_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

#if 0
static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleNSTRFreqSeparationInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	//TODO
	return 0;
}
#endif

static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLMRFreqSeparationInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *freq_sep_arr = NULL, *freq_sep_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, freq_sep_arr, freq_sep_obj, i, 3, "Capabilities", "WiFi7bSTARole", "EMLMRFreqSeparation") {

		curr_data.json_object = freq_sep_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLSRFreqSeparationInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *freq_sep_arr = NULL, *freq_sep_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, freq_sep_arr, freq_sep_obj, i, 3, "Capabilities", "WiFi7bSTARole", "EMLSRFreqSeparation") {

		curr_data.json_object = freq_sep_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleSTRFreqSeparationInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *freq_sep_arr = NULL, *freq_sep_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, freq_sep_arr, freq_sep_obj, i, 3, "Capabilities", "WiFi7bSTARole", "STRFreqSeparation") {

		curr_data.json_object = freq_sep_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

#if 0
static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleNSTRFreqSeparationInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	//TODO
	return 0;
}
#endif

static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfileInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *opclass_arr = NULL, *opclass_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, opclass_arr, opclass_obj, i, 2, "Capabilities", "CapableOperatingClassProfile") {

		curr_data.json_object = opclass_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistoryInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *res = NULL, *sta_obj = NULL, *sta_arr = NULL, *inst_arr = NULL, *inst_obj = NULL;
	struct dm_data curr_data = {0};
	char *sta = NULL, *inst = NULL;
	int sta_id = 0, inst_id = 0, ob = 0;

	sta = dmjson_get_value(((struct dm_data *)prev_data)->json_object, 1, "MACAddress");
	if (!DM_STRLEN(sta))
		return 0;

	dmubus_call("map.controller", "dump_steer_history", UBUS_ARGS{0}, 0, &res);

	dmjson_foreach_obj_in_array(res, sta_arr, sta_obj, sta_id, 1, "sta") {
		char *mac = dmjson_get_value(sta_obj, 1, "macaddr");

		if (DM_STRCMP(mac, sta) != 0)
			continue;

		dmjson_foreach_obj_in_array(sta_obj, inst_arr, inst_obj, inst_id, 1, "history") {

			curr_data.json_object = inst_obj;

			inst = handle_instance_without_section(dmctx, parent_node, ++ob);

			if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
				break;
		}
		break;
	}

	return 0;
}

/*
static int browseWiFiDataElementsNetworkDeviceRadioBSSQMDescriptorInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *qmdescriptor_arr = NULL, *qmdescriptor_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, qmdescriptor_arr, qmdescriptor_obj, i, 1, "QMDescriptor") {

		curr_data.json_object = qmdescriptor_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}
*/

static int browseWiFiDataElementsNetworkDeviceRadioBSSSTAInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *sta_arr = NULL, *sta_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, sta_arr, sta_obj, i, 1, "STAList") {

		curr_data.json_object = sta_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *opclass_scan_arr = NULL, *opclass_scan_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, opclass_scan_arr, opclass_scan_obj, i, 1, "OpClassScan") {

		curr_data.json_object = opclass_scan_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *chscan_arr = NULL, *chscan_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, chscan_arr, chscan_obj, i, 1, "ChannelScan") {

		curr_data.json_object = chscan_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSSInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *neigh_arr = NULL, *neigh_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, neigh_arr, neigh_obj, i, 1, "NeighborBSS") {

		curr_data.json_object = neigh_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioScanCapabilityOpClassChannelsInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *opclass_arr = NULL, *opclass_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, opclass_arr, opclass_obj, i, 2, "ScanCapability", "OpClassChannels") {

		curr_data.json_object = opclass_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *cacmethod_arr = NULL, *cacmethod_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, cacmethod_arr, cacmethod_obj, i, 2, "CACCapability", "CACMethod") {

		curr_data.json_object = cacmethod_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodOpClassChannelsInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *op_class_arr = NULL, *op_class_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, op_class_arr, op_class_obj, i, 1, "OpClassChannels") {

		curr_data.json_object = op_class_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

/*
static int browseWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfileInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *curropclass_arr = NULL, *curropclass_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, curropclass_arr, curropclass_obj, i, 2, "MultiAPDevice", "Backhaul_CurrentOperatingClassProfile") {

		curr_data.json_object = curropclass_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}
*/

static int browseWiFiDataElementsNetworkDeviceAPMLDTIDLinkMapInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *tidlinkmap_arr = NULL, *tidlinkmap_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, tidlinkmap_arr, tidlinkmap_obj, i, 1, "TIDLinkMapList") {

		curr_data.json_object = tidlinkmap_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceAPMLDAffiliatedAPInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *affiliatedap_arr = NULL, *affiliatedap_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, affiliatedap_arr, affiliatedap_obj, i, 1, "AffiliatedAPList") {

		curr_data.json_object = affiliatedap_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceAPMLDSTAMLDInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *stamld_arr = NULL, *stamld_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, stamld_arr, stamld_obj, i, 1, "STAMLDList") {

		curr_data.json_object = stamld_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceAPMLDSTAMLDSTATIDLinkMapInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *statidlinkmap_arr = NULL, *statidlinkmap_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, statidlinkmap_arr, statidlinkmap_obj, i, 1, "STATIDLinkMapList") {

		curr_data.json_object = statidlinkmap_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTAInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *affiliatedsta_arr = NULL, *affiliatedsta_obj = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	dmjson_foreach_obj_in_array(((struct dm_data *)prev_data)->json_object, affiliatedsta_arr, affiliatedsta_obj, i, 1, "AffiliatedSTAList") {

		curr_data.json_object = affiliatedsta_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

/**************************************************************************
* SET AND GET ALIAS
***************************************************************************/
static int get_WiFiDataElementsNetwork_option(const char *option, bool is_array, char **value)
{
	json_object *jobj = NULL;

	if (wifi_de_dump_output == NULL)
		return -1;

	jobj = dmjson_select_obj_in_array_idx(wifi_de_dump_output, 0, 1, "data");
	if (!jobj)
		return -1;

	if (is_array)
		*value = dmjson_get_value_array_all(jobj, ",", 2, "wfa-dataelements:Network", option);
	else
		*value = dmjson_get_value(jobj, 2, "wfa-dataelements:Network", option);
	return 0;
}

/*#Device.WiFi.DataElements.Network.ID!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.ID*/
static int get_WiFiDataElementsNetwork_ID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return get_WiFiDataElementsNetwork_option("ID", false, value);
}

/*#Device.WiFi.DataElements.Network.TimeStamp!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.TimeStamp*/
static int get_WiFiDataElementsNetwork_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return get_WiFiDataElementsNetwork_option("TimeStamp", false, value);
}

/*#Device.WiFi.DataElements.Network.ControllerID!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.ControllerID*/
static int get_WiFiDataElementsNetwork_ControllerID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return get_WiFiDataElementsNetwork_option("ControllerID", false, value);
}

static int get_WiFiDataElementsNetwork_DeviceNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseWiFiDataElementsNetworkDeviceInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_WiFiDataElementsNetwork_MSCSDisallowedStaList(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return get_WiFiDataElementsNetwork_option("MSCSDisallowedStaList", true, value);
}

static int get_WiFiDataElementsNetwork_SCSDisallowedStaList(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return get_WiFiDataElementsNetwork_option("SCSDisallowedStaList", true, value);
}

static int get_WiFiDataElementsNetwork_ColocatedAgentID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return get_WiFiDataElementsNetwork_option("ColocatedAgentID", false, value);
}

static int get_WiFiDataElementsNetwork_SSIDNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseWiFiDataElementsNetworkSSIDInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_WiFiDataElementsNetwork_ProvisionedDPPNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseWiFiDataElementsNetworkProvisionedDPPInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_WiFiDataElementsNetworkSSID_SSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "ssid", value);
	return 0;
}

static int get_WiFiDataElementsNetworkSSID_Band(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *tmp;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "band", &tmp);

	if (DM_STRCMP(tmp, "2") == 0) {
		*value = dmstrdup("2.4");
		return 0;
	}

	*value = dmstrdup(tmp);
	return 0;
}

static int get_WiFiDataElementsNetworkSSID_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enabled", "1");
	return 0;
}

static int get_WiFiDataElementsNetworkSSID_AKMs(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *res = NULL;
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "encryption", &res);

	*value = get_AKMs_dm_name(res);
	return 0;
}

/*
static int get_WiFiDataElementsNetworkSSID_AdvEnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = get_adv_enabled_by_section(((struct dm_data *)data)->config_section);
	return 0;
}

static int get_WiFiDataElementsNetworkSSID_MFP(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = get_mfp_dm_value_by_section(((struct dm_data *)data)->config_section);
	return 0;
}
*/
static int get_WiFiDataElementsNetworkSSID_Haul(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *tmp;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "type", &tmp);

	if (DM_STRCMP(tmp, "backhaul") == 0) {
		*value = dmstrdup("Backhaul");
	} else {
		*value = dmstrdup("Fronthaul");
	}

	return 0;
}

static int get_WiFiDataElementsNetworkSSID_Type(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "network_type", value);
	return 0;
}

static int get_WiFiDataElementsNetworkProvisionedDPP_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "dpp_alias", instance, value);
}

static int set_WiFiDataElementsNetworkProvisionedDPP_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "dpp_alias", instance, value);
}

static int get_WiFiDataElementsNetworkProvisionedDPP_DPPURI(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "uri", value);
	return 0;
}

static int set_WiFiDataElementsNetworkProvisionedDPP_DPPURI(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	json_object *res = NULL;
	char *uri = NULL;
	char *key = NULL;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 2048, NULL, NULL))
				return FAULT_9007;

			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "uri", &uri);
			if (DM_STRLEN(uri)) {
				bbfdm_set_fault_message(ctx,
					"DPP URI is already configured as '%s'. Once defined, the URI value cannot be changed.", uri);
				return FAULT_9001;
			}

			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "key", &key);
			if (DM_STRLEN(key)) {
				bbfdm_set_fault_message(ctx,
					"A Passphrase is already defined for this instance. Only one of 'URI' or 'Passphrase' can be set.");
				return FAULT_9001;
			}

			break;
		case VALUESET:
			dmubus_call("map.controller", "dpp_enrollee_uri", UBUS_ARGS{{"uri", value, String}}, 1, &res);
			if (!res) {
				bbfdm_set_fault_message(ctx,
					"Failed to apply DPP URI: No response received from 'map.controller'. Requested URI: '%s'", value);
				return FAULT_9001;
			} else {
				char *status = dmjson_get_value(res, 1, "status");
				if (DM_STRCMP(status, "ok") != 0) {
					bbfdm_set_fault_message(ctx,
						"Failed to apply DPP URI: The controller rejected the URI value '%s'.", value);
					return FAULT_9001;
				}
			}

			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "uri", value);
			break;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkProvisionedDPP_Passphrase(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "key", value);
	return 0;
}

static int set_WiFiDataElementsNetworkProvisionedDPP_Passphrase(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	json_object *res = NULL;
	char *key = NULL;
	char *uri = NULL;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, 8, 63, NULL, NULL))
				return FAULT_9007;

			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "key", &key);
			if (DM_STRLEN(key)) {
				bbfdm_set_fault_message(ctx, "Passphrase is already defined and cannot be changed.");
				return FAULT_9001;
			}

			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "uri", &uri);
			if (DM_STRLEN(uri)) {
				bbfdm_set_fault_message(ctx,
					"A DPP URI is already defined for this instance. Only one of 'Passphrase' or 'URI' can be configured.");
				return FAULT_9001;
			}

			break;
		case VALUESET:
			dmubus_call("map.controller", "zerotouch_set_key", UBUS_ARGS{{"key", value, String}}, 1, &res);
			if (!res) {
				bbfdm_set_fault_message(ctx,
					"Failed to apply Passphrase: No response from 'map.controller'. Requested Passphrase value: '%s'", value);
				return FAULT_9001;
			} else {
				char *status = dmjson_get_value(res, 1, "status");
				if (DM_STRCMP(status, "ok") != 0) {
					bbfdm_set_fault_message(ctx,
						"Failed to apply Passphrase: The controller returned error status for value '%s'.", value);
					return FAULT_9001;
				}
			}

			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "key", value);
			break;
	}
	return 0;
}

static int ubus_get_multiap_steering_summary_stats(const char *option, char **value)
{
	json_object *jobj = NULL;

	if (wifi_de_dump_output == NULL)
		return -1;

	jobj = dmjson_select_obj_in_array_idx(wifi_de_dump_output, 0, 1, "data");
	if (!jobj)
		return -1;

	*value = dmjson_get_value(jobj, 3, "wfa-dataelements:Network", "MultiAPSteeringSummaryStats", option);
	return 0;
}

static int get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_NoCandidateAPFailures(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ubus_get_multiap_steering_summary_stats("NoCandidateAPFailures", value);
}

static int get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BlacklistAttempts(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ubus_get_multiap_steering_summary_stats("BlacklistAttempts", value);
}

static int get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BlacklistSuccesses(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ubus_get_multiap_steering_summary_stats("BlacklistSuccesses", value);
}

static int get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BlacklistFailures(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ubus_get_multiap_steering_summary_stats("BlacklistFailures", value);
}

static int get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BTMAttempts(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ubus_get_multiap_steering_summary_stats("BTMAttempts", value);
}

static int get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BTMSuccesses(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ubus_get_multiap_steering_summary_stats("BTMSuccesses", value);
}

static int get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BTMFailures(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ubus_get_multiap_steering_summary_stats("BTMFailures", value);
}

static int get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BTMQueryResponses(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return ubus_get_multiap_steering_summary_stats("BTMQueryResponses", value);
}

/*#Device.WiFi.DataElements.Network.Device.{i}.ID!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].ID*/
static int get_WiFiDataElementsNetworkDevice_ID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ID");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.CollectionInterval!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].MultiAPCapabilities*/
static int get_WiFiDataElementsNetworkDevice_MultiAPCapabilities(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MultiAPCapabilities");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.CollectionInterval!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].CollectionInterval*/
static int get_WiFiDataElementsNetworkDevice_CollectionInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "CollectionInterval");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.ReportUnsuccessfulAssociations!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].ReportUnsuccessfulAssociations*/
static int get_WiFiDataElementsNetworkDevice_ReportUnsuccessfulAssociations(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ReportUnsuccessfulAssociations");
	return 0;
}

static int set_WiFiDataElementsNetworkDevice_ReportUnsuccessfulAssociations(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "report_sta_assocfails", b ? "1" : "0");
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_MaxReportingRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MaxReportingRate");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_APMetricsReportingInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "report_metric_periodic", value);
	return 0;
}

static int set_WiFiDataElementsNetworkDevice_APMetricsReportingInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,"255"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "report_metric_periodic", value);
			return 0;
	}
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Manufacturer!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].Manufacturer*/
static int get_WiFiDataElementsNetworkDevice_Manufacturer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Manufacturer");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.SerialNumber!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].SerialNumber*/
static int get_WiFiDataElementsNetworkDevice_SerialNumber(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "SerialNumber");
	return 0;
}

/*
static int get_WiFiDataElementsNetworkDevice_ManufacturerModel(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ManufacturerModel");
	return 0;
}
*/

/*#Device.WiFi.DataElements.Network.Device.{i}.SoftwareVersion!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].SoftwareVersion*/
static int get_WiFiDataElementsNetworkDevice_SoftwareVersion(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "SoftwareVersion");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.ExecutionEnv!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].ExecutionEnv*/
static int get_WiFiDataElementsNetworkDevice_ExecutionEnv(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ExecutionEnv");
	return 0;
}

/*
static int get_WiFiDataElementsNetworkDevice_DSCPMap(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "DSCPMap");
	return 0;
}
*/

/*#Device.WiFi.DataElements.Network.Device.{i}.MaxPrioritizationRules!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].MaxPrioritizationRules*/
static int get_WiFiDataElementsNetworkDevice_MaxPrioritizationRules(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MaxPrioritizationRules");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.PrioritizationSupport!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].PrioritizationSupport*/
static int get_WiFiDataElementsNetworkDevice_PrioritizationSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "PrioritizationSupport");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.MaxVIDs!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].MaxVIDs*/
static int get_WiFiDataElementsNetworkDevice_MaxVIDs(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MaxVIDs");
	return 0;
}

/*
static int get_WiFiDataElementsNetworkDevice_APMetricsWiFi6(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "APMetricsWiFi6");
	return 0;
}
*/

/*#Device.WiFi.DataElements.Network.Device.{i}.CountryCode!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].CountryCode*/
static int get_WiFiDataElementsNetworkDevice_CountryCode(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "CountryCode");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_LocalSteeringDisallowedSTAList(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_list *uci_list = NULL;
	dmuci_get_value_by_section_list(((struct dm_data *)data)->config_section, "steer_exclude", &uci_list);
	*value = dmuci_list_to_string(uci_list, ",");
	return 0;
}

static int set_WiFiDataElementsNetworkDevice_LocalSteeringDisallowedSTAList(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	size_t length = 0;
	char **arr = NULL;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string_list(ctx, value, -1, -1, -1, -1, 17, NULL, MACAddress))
				return FAULT_9007;
			return 0;
		case VALUESET:
			arr = strsplit(value, ",", &length);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "steer_exclude", "");
			for (int i = 0; i < length; i++)
				dmuci_add_list_value_by_section(((struct dm_data *)data)->config_section, "steer_exclude", arr[i]);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_BTMSteeringDisallowedSTAList(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_list *uci_list = NULL;
	dmuci_get_value_by_section_list(((struct dm_data *)data)->config_section, "steer_exclude_btm", &uci_list);
	*value = dmuci_list_to_string(uci_list, ",");
	return 0;
}

static int set_WiFiDataElementsNetworkDevice_BTMSteeringDisallowedSTAList(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	size_t length = 0;
	char **arr = NULL;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string_list(ctx, value, -1, -1, -1, -1, 17, NULL, MACAddress))
				return FAULT_9007;
			return 0;
		case VALUESET:
			arr = strsplit(value, ",", &length);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "steer_exclude_btm", "");
			for (int i = 0; i < length; i++)
				dmuci_add_list_value_by_section(((struct dm_data *)data)->config_section, "steer_exclude_btm", arr[i]);
			return 0;
	}
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.DFSEnable!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].DFSEnable*/
static int get_WiFiDataElementsNetworkDevice_DFSEnable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "DFSEnable");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.ReportIndependentScans!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].ReportIndependentScans*/
static int get_WiFiDataElementsNetworkDevice_ReportIndependentScans(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ReportIndependentScans");
	return 0;
}

static int set_WiFiDataElementsNetworkDevice_ReportIndependentScans(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "report_scan", b ? "1" : "0");
			return 0;
	}
	return 0;
}

/*
static int get_WiFiDataElementsNetworkDevice_AssociatedSTAinAPMetricsWiFi6(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "AssociatedSTAinAPMetricsWiFi6");
	return 0;
}

static int set_WiFiDataElementsNetworkDevice_AssociatedSTAinAPMetricsWiFi6(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			//TODO
			break;
	}
	return 0;
}
*/

static int get_WiFiDataElementsNetworkDevice_MaxUnsuccessfulAssociationReportingRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MaxUnsuccessfulAssociationReportingRate");
	return 0;
}

static int set_WiFiDataElementsNetworkDevice_MaxUnsuccessfulAssociationReportingRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "report_sta_assocfails_rate", value);
			return 0;
	}
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.STASteeringState!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].STASteeringState*/
static int get_WiFiDataElementsNetworkDevice_STASteeringState(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "STASteeringState");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_CoordinatedCACAllowed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "CoordinatedCACAllowed");
	return 0;
}

static int set_WiFiDataElementsNetworkDevice_CoordinatedCACAllowed(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "coordinated_cac", b ? "1" : "0");
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_TrafficSeparationAllowed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_option_value_fallback_def("mapcontroller", "controller", "enable_ts", "0");
	return 0;
}

/*static int get_WiFiDataElementsNetworkDevice_ServicePrioritizationAllowed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ServicePrioritizationAllowed");
	return 0;
}
*/

static int get_WiFiDataElementsNetworkDevice_ControllerOperationMode(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "MultiAPDevice", "EasyMeshControllerOperationMode");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_BackhaulMACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BackhaulMACAddress");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_BackhaulALID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BackhaulALID");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_BackhaulDownMACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BackhaulDownMACAddress");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_BackhaulMediaType(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BackhaulMediaType");
	if (DM_LSTRCMP(*value, "None") == 0) {
		*value = dmstrdup("");
	}

	return 0;
}

static int get_WiFiDataElementsNetworkDevice_BackhaulPHYRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BackhaulPHYRate");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_TrafficSeparationCapability(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TrafficSeparationCapability");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_EasyConnectCapability(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "EasyConnectCapability");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.RadioNumberOfEntries!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].NumberOfRadios*/
static int get_WiFiDataElementsNetworkDevice_RadioNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "RadioNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_Default8021QNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Default8021QNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_SSIDtoVIDMappingNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "SSIDtoVIDMappingNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_CACStatusNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "CACStatusNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_IEEE1905SecurityNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "IEEE1905SecurityNumberOfEntries");
	return 0;
}

#if 0
static int get_WiFiDataElementsNetworkDevice_SPRuleNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "SPRuleNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_AnticipatedChannelsNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "AnticipatedChannelsNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_AnticipatedChannelUsageNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "AnticipatedChannelUsageNumberOfEntries");
	return 0;
}
#endif

static int get_WiFiDataElementsNetworkDevice_MaxNumMLDs(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MaxNumMLDs");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_APMLDMaxLinks(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "APMLDMaxLinks");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_bSTAMLDMaxLinks(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "bSTAMLDMaxLinks");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_TIDLinkMapCapability(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TIDLinkMapCapability");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_APMLDNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "APMLDNumberOfEntries");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ID!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].ID*/
static int get_WiFiDataElementsNetworkDeviceRadio_ID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ID");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Enabled!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].Enabled*/
static int get_WiFiDataElementsNetworkDeviceRadio_Enabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Enabled");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Noise!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].Noise*/
static int get_WiFiDataElementsNetworkDeviceRadio_Noise(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Noise");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Utilization!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].Utilization*/
static int get_WiFiDataElementsNetworkDeviceRadio_Utilization(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Utilization");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Transmit!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].Transmit*/
static int get_WiFiDataElementsNetworkDeviceRadio_Transmit(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Transmit");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ReceiveSelf!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].ReceiveSelf*/
static int get_WiFiDataElementsNetworkDeviceRadio_ReceiveSelf(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ReceiveSelf");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ReceiveOther!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].RecieveOther*/
static int get_WiFiDataElementsNetworkDeviceRadio_ReceiveOther(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ReceiveOther");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.TrafficSeparationCombinedFronthaul!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].TrafficSeparationCombinedFronthaul*/
static int get_WiFiDataElementsNetworkDeviceRadio_TrafficSeparationCombinedFronthaul(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TrafficSeparationCombinedFronthaul");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.TrafficSeparationCombinedBackhaul!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].TrafficSeparationCombinedBackhaul*/
static int get_WiFiDataElementsNetworkDeviceRadio_TrafficSeparationCombinedBackhaul(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TrafficSeparationCombinedBackhaul");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_SteeringPolicy(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "steer_policy", value);
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadio_SteeringPolicy(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,"2"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "steer_policy", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_ChannelUtilizationThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "util_threshold", value);
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadio_ChannelUtilizationThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,"255"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "util_threshold", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_RCPISteeringThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "rcpi_threshold", value);
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadio_RCPISteeringThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,"220"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "rcpi_threshold", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_STAReportingRCPIThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "report_rcpi_threshold", value);
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadio_STAReportingRCPIThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,"220"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "report_rcpi_threshold", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_STAReportingRCPIHysteresisMarginOverride(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "report_rcpi_hysteresis_margin", value);
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadio_STAReportingRCPIHysteresisMarginOverride(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "report_rcpi_hysteresis_margin", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_ChannelUtilizationReportingThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "report_util_threshold", value);
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadio_ChannelUtilizationReportingThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "report_util_threshold", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_AssociatedSTATrafficStatsInclusionPolicy(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "include_sta_stats", "1");
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadio_AssociatedSTATrafficStatsInclusionPolicy(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "include_sta_stats", b ? "1" : "0");
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_AssociatedSTALinkMetricsInclusionPolicy(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "include_sta_metric", "1");
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadio_AssociatedSTALinkMetricsInclusionPolicy(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "include_sta_metric", b ? "1" : "0");
			return 0;
	}
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ChipsetVendor!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].ChipsetVendor*/
static int get_WiFiDataElementsNetworkDeviceRadio_ChipsetVendor(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ChipsetVendor");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_APMetricsWiFi6(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "APMetricsWiFi6");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.CurrentOperatingClassProfileNumberOfEntries!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].NumberOfCurrOpClass*/
static int get_WiFiDataElementsNetworkDeviceRadio_CurrentOperatingClassProfileNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "CurrentOperatingClassProfileNumberOfEntries");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.UnassociatedSTANumberOfEntries!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].UnassociatedSTANumberOfEntries*/
static int get_WiFiDataElementsNetworkDeviceRadio_UnassociatedSTANumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "UnassociatedSTANumberOfEntries");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSSNumberOfEntries!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].NumberOfBSS*/
static int get_WiFiDataElementsNetworkDeviceRadio_BSSNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BSSNumberOfEntries");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResultNumberOfEntries!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].NumberOfScanRes*/
static int get_WiFiDataElementsNetworkDeviceRadio_ScanResultNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ScanResultNumberOfEntries");
	return 0;
}

#if 0
static int get_WiFiDataElementsNetworkDeviceRadio_DisAllowedOpClassChannelsNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "DisAllowedOpClassChannelsNumberOfEntries");
	return 0;
}
#endif

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BackhaulSta.MACAddress!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BackhaulSta.MACAddress*/
static int get_WiFiDataElementsNetworkDeviceRadioBackhaulSta_MACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "BackhaulSta", "MACAddress");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.HTCapabilities!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].Capabilites.HTCapabilities*/
static int get_WiFiDataElementsNetworkDeviceRadioCapabilities_HTCapabilities(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "Capabilities", "HTCapabilities");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.VHTCapabilities!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].Capabilites.VHTCapabilities*/
static int get_WiFiDataElementsNetworkDeviceRadioCapabilities_VHTCapabilities(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *cap = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "Capabilities", "VHTCapabilities");
	*value = (DM_STRLEN(cap)) ? cap : "AAA=";
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.CapableOperatingClassProfileNumberOfEntries!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].Capabilites.NumberOfOpClass*/
static int get_WiFiDataElementsNetworkDeviceRadioCapabilities_CapableOperatingClassProfileNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "Capabilities", "CapableOperatingClassProfileNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_HE160(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "HE160");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_HE8080(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "HE8080");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MCSNSS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "MCSNSS");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_SUBeamformer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "SUBeamformer");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_SUBeamformee(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "SUBeamformee");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MUBeamformer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "MUBeamformer");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_Beamformee80orLess(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "Beamformee80orLess");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_BeamformeeAbove80(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "BeamformeeAbove80");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_ULMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "ULMUMIMO");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_ULOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "ULOFDMA");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MaxDLMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "MaxDLMUMIMO");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MaxULMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "MaxULMUMIMO");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MaxDLOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "MaxDLOFDMA");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MaxULOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "MaxULOFDMA");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_RTS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "RTS");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MURTS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "MURTS");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MultiBSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "MultiBSSID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MUEDCA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "MUEDCA");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_TWTRequestor(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "TWTRequestor");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_TWTResponder(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "TWTResponder");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_SpatialReuse(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "SpatialReuse");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_AnticipatedChannelUsage(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6APRole", "AnticipatedChannelUsage");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_HE160(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "HE160");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_HE8080(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "HE8080");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MCSNSS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "MCSNSS");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_SUBeamformer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "SUBeamformer");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_SUBeamformee(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "SUBeamformee");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MUBeamformer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "MUBeamformer");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_Beamformee80orLess(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "Beamformee80orLess");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_BeamformeeAbove80(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "BeamformeeAbove80");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_ULMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "ULMUMIMO");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_ULOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "ULOFDMA");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MaxDLMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "MaxDLMUMIMO");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MaxULMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "MaxULMUMIMO");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MaxDLOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "MaxDLOFDMA");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MaxULOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "MaxULOFDMA");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_RTS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "RTS");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MURTS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "MURTS");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MultiBSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "MultiBSSID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MUEDCA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "MUEDCA");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_TWTRequestor(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "TWTRequestor");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_TWTResponder(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "TWTResponder");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_SpatialReuse(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "SpatialReuse");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_AnticipatedChannelUsage(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi6bSTARole", "AnticipatedChannelUsage");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_EMLMRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7APRole", "EMLMRSupport");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_EMLSRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7APRole", "EMLSRSupport");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_STRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7APRole", "STRSupport");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_NSTRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7APRole", "NSTRSupport");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_TIDLinkMapNegotiation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7APRole", "TIDLinkMapNegotiation");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_EMLMRFreqSeparationNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7APRole", "EMLMRFreqSeparationNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_EMLSRFreqSeparationNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7APRole", "EMLSRFreqSeparationNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_STRFreqSeparationNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7APRole", "STRFreqSeparationNumberOfEntries");
	return 0;
}

#if 0
static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_NSTRFreqSeparationNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7APRole", "NSTRFreqSeparationNumberOfEntries");
	return 0;
}
#endif

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLMRFreqSeparation_RUID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "RUID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLMRFreqSeparation_FreqSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "FreqSeparation");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLSRFreqSeparation_RUID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "RUID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLSRFreqSeparation_FreqSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "FreqSeparation");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleSTRFreqSeparation_RUID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "RUID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleSTRFreqSeparation_FreqSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "FreqSeparation");
	return 0;
}

#if 0
static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleNSTRFreqSeparation_RUID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "RUID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleNSTRFreqSeparation_FreqSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "FreqSeparation");
	return 0;
}
#endif

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_EMLMRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7bSTARole", "EMLMRSupport");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_EMLSRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7bSTARole", "EMLSRSupport");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_STRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7bSTARole", "STRSupport");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_NSTRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7bSTARole", "NSTRSupport");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_TIDLinkMapNegotiation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7bSTARole", "TIDLinkMapNegotiation");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_EMLMRFreqSeparationNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7bSTARole", "EMLMRFreqSeparationNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_EMLSRFreqSeparationNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7bSTARole", "EMLSRFreqSeparationNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_STRFreqSeparationNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7bSTARole", "STRFreqSeparationNumberOfEntries");
	return 0;
}

#if 0
static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_NSTRFreqSeparationNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "Capabilities", "WiFi7bSTARole", "NSTRFreqSeparationNumberOfEntries");
	return 0;
}
#endif

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLMRFreqSeparation_RUID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "RUID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLMRFreqSeparation_FreqSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "FreqSeparation");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLSRFreqSeparation_RUID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "RUID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLSRFreqSeparation_FreqSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "FreqSeparation");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleSTRFreqSeparation_RUID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "RUID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleSTRFreqSeparation_FreqSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "FreqSeparation");
	return 0;
}

#if 0
static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleNSTRFreqSeparation_RUID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "RUID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleNSTRFreqSeparation_FreqSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "FreqSeparation");
	return 0;
}
#endif

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.CapableOperatingClassProfile.{i}.Class!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].Capabilites.OperatingClasses[@i-1].Class*/
static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfile_Class(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Class");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.CapableOperatingClassProfile.{i}.MaxTxPower!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].Capabilites.OperatingClasses[@i-1].MaxTxPower*/
static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfile_MaxTxPower(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MaxTxPower");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.CapableOperatingClassProfile.{i}.NonOperable!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].Capabilites.OperatingClasses[@i-1].NonOperable*/
static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfile_NonOperable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value_array_all(((struct dm_data *)data)->json_object, ",", 1, "NonOperable");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.CapableOperatingClassProfile.{i}.NumberOfNonOperChan!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].Capabilites.OperatingClasses[@i-1].NumberOfNonOperChan*/
static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfile_NumberOfNonOperChan(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "NumberOfNonOperChan");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.CurrentOperatingClassProfile.{i}.Class!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].CurrentOperatingClasses[@i-1].Class*/
static int get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_Class(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Class");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.CurrentOperatingClassProfile.{i}.Channel!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].CurrentOperatingClasses[@i-1].Channel*/
static int get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_Channel(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Channel");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.CurrentOperatingClassProfile.{i}.TxPower!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].CurrentOperatingClasses[@i-1].TxPower*/
static int get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_TxPower(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TxPower");
	return 0;
}
/*
static int get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_TransmitPowerLimit(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TransmitPowerLimit");
	return 0;
}
*/
/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.CurrentOperatingClassProfile.{i}.TimeStamp!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].CurrentOperatingClasses[@i-1].TimeStamp*/
static int get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TimeStamp");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.BSSID!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].BSSID*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_BSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BSSID");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.SSID!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].SSID*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_SSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "SSID");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.Enabled!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].Enabled*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_Enabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Enabled");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.LastChange!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].LastChange*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_LastChange(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "LastChange");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.TimeStamp!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].TimeStamp*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TimeStamp");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.UnicastBytesSent!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].UnicastBytesSent*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_UnicastBytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "UnicastBytesSent");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.UnicastBytesReceived!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].UnicastBytesReceived*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_UnicastBytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "UnicastBytesReceived");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.MulticastBytesSent!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].MulticastBytesSent*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_MulticastBytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MulticastBytesSent");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.MulticastBytesReceived!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].MulticastBytesReceived*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_MulticastBytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MulticastBytesReceived");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.BroadcastBytesSent!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].BroadcastBytesSent*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_BroadcastBytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BroadcastBytesSent");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.BroadcastBytesReceived!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].BroadcastBytesReceived*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_BroadcastBytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BroadcastBytesReceived");
	return 0;
}

/*
static int get_WiFiDataElementsNetworkDeviceRadioBSS_ByteCounterUnits(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ByteCounterUnits");
	return 0;
}
*/

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.Profile1bSTAsDisallowed!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].Profile1bSTAsDisallowed*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_Profile1bSTAsDisallowed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Profile1bSTAsDisallowed");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.Profile2bSTAsDisallowed!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].Profile2bSTAsDisallowed*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_Profile2bSTAsDisallowed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Profile2bSTAsDisallowed");
	return 0;
}

/*
static int get_WiFiDataElementsNetworkDeviceRadioBSS_AssociationAllowanceStatus(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "AssociationAllowanceStatus");
	return 0;
}
*/

static int get_WiFiDataElementsNetworkDeviceRadioBSS_EstServiceParametersBE(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "EstServiceParametersBE");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_EstServiceParametersBK(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "EstServiceParametersBK");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_EstServiceParametersVI(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "EstServiceParametersVI");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_EstServiceParametersVO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "EstServiceParametersVO");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.Profile2bSTAsDisallowed!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].Profile2bSTAsDisallowed*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_BackhaulUse(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BackhaulUse");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.FronthaulUse!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].Profile2bSTAsDisallowed*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_FronthaulUse(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "FronthaulUse");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.R1disallowed!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].FronthaulUse*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_R1disallowed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "R1disallowed");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.R2disallowed!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].R2disallowed*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_R2disallowed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "R2disallowed");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.MultiBSSID!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].MultiBSSID*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_MultiBSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MultiBSSID");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.TransmittedBSSID!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].TransmittedBSSID*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_TransmittedBSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TransmittedBSSID");
	return 0;
}

/*
static int get_WiFiDataElementsNetworkDeviceRadioBSS_FronthaulAKMsAllowed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value_array_all(((struct dm_data *)data)->json_object, ",", 1, "FronthaulAKMsAllowed");
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadioBSS_FronthaulAKMsAllowed(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string_list(ctx, value, -1, -1, -1, -1, -1, AKMsAllowed, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			//TODO
			break;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_BackhaulAKMsAllowed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value_array_all(((struct dm_data *)data)->json_object, ",", 1, "BackhaulAKMsAllowed");
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadioBSS_BackhaulAKMsAllowed(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string_list(ctx, value, -1, -1, -1, -1, -1, AKMsAllowed, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			//TODO
			break;
	}
	return 0;
}
*/
/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STANumberOfEntries!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].NumberofSTA*/
static int get_WiFiDataElementsNetworkDeviceRadioBSS_STANumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "STANumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTA_SteeringHistoryNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistoryInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

/*
static int get_WiFiDataElementsNetworkDeviceRadioBSS_QMDescriptorNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "QMDescriptorNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSQMDescriptor_ClientMAC(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ClientMAC");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSQMDescriptor_DescriptorElement(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "DescriptorElement");
	return 0;
}
*/

static int get_WiFiDataElementsNetworkDeviceRadioBSSMultiAPSteering_BlacklistAttempts(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "MultiAPSteering", "BlacklistAttempts");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSMultiAPSteering_BTMAttempts(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "MultiAPSteering", "BTMAttempts");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSMultiAPSteering_BTMQueryResponses(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "MultiAPSteering", "BTMQueryResponses");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.MACAddress!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].MACAddress*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_MACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MACAddress");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.TimeStamp!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].TimeStamp*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TimeStamp");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.HTCapabilities!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].HTCapabilities*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_HTCapabilities(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "HTCapabilities");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.VHTCapabilities!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].VHTCapabilities*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_VHTCapabilities(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "VHTCapabilities");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.LastDataDownlinkRate!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].LastDataDownlinkRate*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_LastDataDownlinkRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "LastDataDownlinkRate");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.LastDataUplinkRate!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].LastDataUplinkRate*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_LastDataUplinkRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "LastDataUplinkRate");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.UtilizationReceive!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].UtilizationReceive*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_UtilizationReceive(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "UtilizationReceive");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.UtilizationTransmit!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].UtilizationTransmit*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_UtilizationTransmit(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "UtilizationTransmit");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.EstMACDataRateDownlink!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].EstMACDataRateDownlink*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_EstMACDataRateDownlink(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "EstMACDataRateDownlink");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.EstMACDataRateUplink!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].EstMACDataRateUplink*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_EstMACDataRateUplink(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "EstMACDataRateUplink");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.SignalStrength!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].SignalStrength*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_SignalStrength(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "SignalStrength");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.LastConnectTime!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].LastConnectTime*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_LastConnectTime(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "LastConnectTime");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.BytesSent!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].BytesSent*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_BytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BytesSent");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.BytesReceived!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].BytesReceived*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_BytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BytesReceived");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.PacketsSent!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].PacketsSent*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_PacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "PacketsSent");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.PacketsReceived!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].PacketsReceived*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_PacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "PacketsReceived");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.ErrorsSent!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].ErrorsSent*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_ErrorsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ErrorsSent");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.ErrorsReceived!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].ErrorsReceived*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_ErrorsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ErrorsReceived");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_RetransCount(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "RetransCount");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.MeasurementReport!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].Measurementreport*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_MeasurementReport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value_array_all(((struct dm_data *)data)->json_object, ",", 1, "MeasurementReport");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.NumberOfMeasureReports!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].NumberOfMeasureReports*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_NumberOfMeasureReports(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "NumberOfMeasureReports");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_IPV4Address(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "IPV4Address");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_IPV6Address(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "IPV6Address");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.Hostname!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].Hostname*/
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_Hostname(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Hostname");
	return 0;
}
/*
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_CellularDataPreference(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "CellularDataPreference");
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadioBSSSTA_CellularDataPreference(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, CellularDataPreference, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			//TODO
			break;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_ReAssociationDelay(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ReAssociationDelay");
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadioBSSSTA_ReAssociationDelay(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,"65535"}}, 1))
				return FAULT_9007;
			break;
		case VALUESET:
			//TODO
			break;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_TIDQueueSizesNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TIDQueueSizesNumberOfEntries");
	return 0;
}
*/

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_NoCandidateAPFailures(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "MultiAPSTA", "SteeringSummaryStats", "NoCandidateAPFailures");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BlacklistAttempts(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "MultiAPSTA", "SteeringSummaryStats", "BlacklistAttempts");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BlacklistSuccesses(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "MultiAPSTA", "SteeringSummaryStats", "BlacklistSuccesses");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BlacklistFailures(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "MultiAPSTA", "SteeringSummaryStats", "BlacklistFailures");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BTMAttempts(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "MultiAPSTA", "SteeringSummaryStats", "BTMAttempts");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BTMSuccesses(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "MultiAPSTA", "SteeringSummaryStats", "BTMSuccesses");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BTMFailures(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "MultiAPSTA", "SteeringSummaryStats", "BTMFailures");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BTMQueryResponses(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "MultiAPSTA", "SteeringSummaryStats", "BTMQueryResponses");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_LastSteerTime(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "MultiAPSTA", "SteeringSummaryStats", "LastSteerTime");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_Time(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "time");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_APOrigin(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ap");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_TriggerEvent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *trigger = NULL;

	trigger = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "trigger");

	if (DM_STRCMP(trigger, "link_quality") == 0) {
		dmasprintf(value, "%s", "Wi-Fi Link Quality");
	} else if (DM_STRCMP(trigger, "channel_util") == 0) {
		dmasprintf(value, "%s", "Wi-Fi Channel Utilization");
	} else if (DM_STRCMP(trigger, "bk_link_util") == 0) {
		dmasprintf(value, "%s", "Backhaul Link Utilization");
	} else {
		dmasprintf(value, "%s", "Unknown");
	}

	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_SteeringApproach(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *method = NULL;

	method = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "method");

	if (DM_STRCMP(method, "btm") == 0) {
		dmasprintf(value, "%s", "BTM Request");
	} else if (DM_STRCMP(method, "assoc_ctl") == 0) {
		dmasprintf(value, "%s", "Blacklist");
	} else if (DM_STRCMP(method, "async_btm") == 0) {
		dmasprintf(value, "%s", "Async BTM Query");
	} else {
		dmasprintf(value, "%s", method);
	}

	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_APDestination(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "target_ap");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_SteeringDuration(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "duration");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.HE160!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.HE160 */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_HE160(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "HE160");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.HE8080!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.HE8080 */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_HE8080(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "HE8080");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.MCSNSS!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.MCSNSS */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MCSNSS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "MCSNSS");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.SUBeamformer!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.SUBeamformer */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_SUBeamformer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "SUBeamformer");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.SUBeamformee!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.SUBeamformee */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_SUBeamformee(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "SUBeamformee");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.MUBeamformer!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.MUBeamformer */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MUBeamformer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "MUBeamformer");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.Beamformee80orLess!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.Beamformee80orLess */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_Beamformee80orLess(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "Beamformee80orLess");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.BeamformeeAbove80!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.BeamformeeAbove80 */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_BeamformeeAbove80(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "BeamformeeAbove80");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.ULMUMIMO!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.ULMUMIMO */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_ULMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "ULMUMIMO");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.ULOFDMA!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.ULOFDMA */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_ULOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "ULOFDMA");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.MaxDLMUMIMO!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.MaxDLMUMIMO */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MaxDLMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "MaxDLMUMIMO");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.MaxULMUMIMO!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.MaxULMUMIMO */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MaxULMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "MaxULMUMIMO");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.MaxDLOFDMA!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.MaxDLOFDMA */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MaxDLOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "MaxDLOFDMA");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.MaxULOFDMA!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.MaxULOFDMA */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MaxULOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "MaxULOFDMA");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.RTS!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.RTS */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_RTS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "RTS");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.MURTS!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.MURTS */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MURTS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "MURTS");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.MultiBSSID!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.MultiBSSID */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MultiBSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "MultiBSSID");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.MUEDCA!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.MUEDCA */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MUEDCA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "MUEDCA");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.TWTRequestor!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.TWTRequestor */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_TWTRequestor(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "TWTRequestor");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.TWTResponder!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.TWTResponder */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_TWTResponder(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "TWTResponder");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.SpatialReuse!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.SpatialReuse */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_SpatialReuse(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "SpatialReuse");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities.AnticipatedChannelUsage!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].STAList[@i-1].WiFi6Capabilities.AnticipatedChannelUsage */
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_AnticipatedChannelUsage(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi6Capabilities", "AnticipatedChannelUsage");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.TimeStamp!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].ScanResultList[@i-1].TimeStamp*/
static int get_WiFiDataElementsNetworkDeviceRadioScanResult_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TimeStamp");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScanNumberOfEntries!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].ScanResultList[@i-1].NumberOfOpClassScans*/
static int get_WiFiDataElementsNetworkDeviceRadioScanResult_OpClassScanNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "OpClassScanNumberOfEntries");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}.OperatingClass!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].ScanResultList[@i-1].OpClassScanList[@i-1].OperatingClass*/
static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScan_OperatingClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "OperatingClass");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}.ChannelScanNumberOfEntries!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].ScanResultList[@i-1].OpClassScanList[@i-1].NumberOfChannelScans*/
static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScan_ChannelScanNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ChannelScanNumberOfEntries");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}.ChannelScan.{i}.Channel!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].ScanResultList[@i-1].OpClassScanList[@i-1].ChannelScanList[@i-1].Channel*/
static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_Channel(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Channel");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}.ChannelScan.{i}.TimeStamp!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].ScanResultList[@i-1].OpClassScanList[@i-1].ChannelScanList[@i-1].TimeStamp*/
static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TimeStamp");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}.ChannelScan.{i}.Utilization!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].ScanResultList[@i-1].OpClassScanList[@i-1].ChannelScanList[@i-1].Utilization*/
static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_Utilization(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Utilization");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}.ChannelScan.{i}.Noise!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].ScanResultList[@i-1].OpClassScanList[@i-1].ChannelScanList[@i-1].Noise*/
static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_Noise(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Noise");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_ScanStatus(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ScanStatus");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}.ChannelScan.{i}.NeighborBSSNumberOfEntries!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].ScanResultList[@i-1].OpClassScanList[@i-1].ChannelScanList[@i-1].NumberofNeighbors*/
static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_NeighborBSSNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "NeighborBSSNumberOfEntries");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}.ChannelScan.{i}.NeighborBSS.{i}.BSSID!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].ScanResultList[@i-1].OpClassScanList[@i-1].ChannelScanList[@i-1].NeighborList[@i-1].BSSID*/
static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_BSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BSSID");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}.ChannelScan.{i}.NeighborBSS.{i}.SSID!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].ScanResultList[@i-1].OpClassScanList[@i-1].ChannelScanList[@i-1].NeighborList[@i-1].SSID*/
static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_SSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "SSID");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}.ChannelScan.{i}.NeighborBSS.{i}.SignalStrength!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].ScanResultList[@i-1].OpClassScanList[@i-1].ChannelScanList[@i-1].NeighborList[@i-1].SignalStrengh*/
static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_SignalStrength(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "SignalStrength");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}.ChannelScan.{i}.NeighborBSS.{i}.ChannelBandwidth!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].ScanResultList[@i-1].OpClassScanList[@i-1].ChannelScanList[@i-1].NeighborList[@i-1].ChannelBandwidth*/
static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_ChannelBandwidth(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ChannelBandwidth");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}.ChannelScan.{i}.NeighborBSS.{i}.ChannelUtilization!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].ScanResultList[@i-1].OpClassScanList[@i-1].ChannelScanList[@i-1].NeighborList[@i-1].ChannelUtilization*/
static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_ChannelUtilization(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ChannelUtilization");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}.ChannelScan.{i}.NeighborBSS.{i}.StationCount!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].BSSList[@i-1].ScanResultList[@i-1].OpClassScanList[@i-1].ChannelScanList[@i-1].NeighborList[@i-1].StationCount*/
static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_StationCount(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "StationCount");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_MLDMACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MLDMACAddress");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_ReportingBSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ReportingBSSID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_MultiBSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MultiBSSID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_BSSLoadElementPresent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BSSLoadElementPresent");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_BSSColor(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BSSColor");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanCapability_OnBootOnly(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "ScanCapability", "OnBootOnly");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanCapability_Impact(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "ScanCapability", "Impact");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanCapability_MinimumInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "ScanCapability", "MinimumInterval");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanCapability_OpClassChannelsNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "ScanCapability", "OpClassChannelsNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanCapabilityOpClassChannels_OpClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "OpClass");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanCapabilityOpClassChannels_ChannelList(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value_array_all(((struct dm_data *)data)->json_object, ",", 1, "ChannelList");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCACCapability_CACMethodNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "CACCapability", "CACMethodNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethod_Method(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Method");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethod_NumberOfSeconds(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "NumberOfSeconds");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethod_OpClassChannelsNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "OpClassChannelsNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodOpClassChannels_OpClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "OpClass");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodOpClassChannels_ChannelList(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value_array_all(((struct dm_data *)data)->json_object, ",", 1, "ChannelList");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.UnassociatedSTA.{i}.MACAddress!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].UnassociatedStaList[@i-1].MACAddress*/
static int get_WiFiDataElementsNetworkDeviceRadioUnassociatedSTA_MACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MACAddress");
	return 0;
}

/*#Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.UnassociatedSTA.{i}.SignalStrength!UBUS:wifi.dataelements/dump//data[0].wfa-dataelements:Network.DeviceList[@i-1].RadioList[@i-1].UnassociatedStaList[@i-1].SignalStrength*/
static int get_WiFiDataElementsNetworkDeviceRadioUnassociatedSTA_SignalStrength(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "SignalStrength");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDevice_LastContactTime(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "MultiAPDevice", "LastContactTime");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDevice_AssocIEEE1905DeviceRef(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *device_id = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ID");
	_bbfdm_get_references(ctx, "Device.IEEE1905.AL.NetworkTopology.IEEE1905Device.", "IEEE1905Id", device_id, value);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDevice_EasyMeshAgentOperationMode(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "MultiAPDevice", "EasyMeshAgentOperationMode");
	return 0;
}

/*
static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaul_CurrentOperatingClassProfileNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "MultiAPDevice", "Backhaul", "CurrentOperatingClassProfileNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfile_Class(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Class");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfile_Channel(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Channel");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfile_TxPower(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TxPower");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfile_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TimeStamp");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_BytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 4, "MultiAPDevice", "Backhaul", "Stats", "BytesSent");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_BytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 4, "MultiAPDevice", "Backhaul", "Stats", "BytesReceived");
	return 0;
}
*/

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_PacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 4, "MultiAPDevice", "Backhaul", "Stats", "PacketsSent");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_PacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 4, "MultiAPDevice", "Backhaul", "Stats", "PacketsReceived");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_ErrorsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 4, "MultiAPDevice", "Backhaul", "Stats", "ErrorsSent");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_ErrorsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 4, "MultiAPDevice", "Backhaul", "Stats", "ErrorsReceived");
	return 0;
}

/*static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_LinkUtilization(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 4, "MultiAPDevice", "Backhaul", "Stats", "LinkUtilization");
	return 0;
}*/

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_SignalStrength(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 4, "MultiAPDevice", "Backhaul", "Stats", "SignalStrength");
	return 0;
}

/*static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_LastDataDownlinkRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 4, "MultiAPDevice", "Backhaul", "Stats", "LastDataDownlinkRate");
	return 0;
}*/

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_LastDataUplinkRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 4, "MultiAPDevice", "Backhaul", "Stats", "LastDataUplinkRate");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 4, "MultiAPDevice", "Backhaul", "Stats", "TimeStamp");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceDefault8021Q_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("1");
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceDefault8021Q_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			//TODO
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceDefault8021Q_PrimaryVID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_option_value_string("mapcontroller", "controller", "primary_vid", value);
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceDefault8021Q_PrimaryVID(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,"4095"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value("mapcontroller", "controller", "primary_vid", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceDefault8021Q_DefaultPCP(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_option_value_string("mapcontroller", "controller", "primary_pcp", value);
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceDefault8021Q_DefaultPCP(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,"7"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value("mapcontroller", "controller", "primary_pcp", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceSSIDtoVIDMapping_SSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "SSID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceSSIDtoVIDMapping_VID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "VID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatus_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TimeStamp");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatus_CACAvailableChannelNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "CACAvailableChannelNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatus_CACNonOccupancyChannelNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "CACNonOccupancyChannelNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatus_CACActiveChannelNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "CACActiveChannelNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACAvailableChannel_OpClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "OpClass");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACAvailableChannel_Channel(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Channel");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACAvailableChannel_Minutes(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Minutes");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannel_OpClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "OpClass");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannel_Channel(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Channel");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannel_Seconds(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Seconds");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACActiveChannel_OpClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "OpClass");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACActiveChannel_Channel(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Channel");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACActiveChannel_Countdown(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Countdown");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceIEEE1905Security_OnboardingProtocol(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "OnboardingProtocol");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceIEEE1905Security_IntegrityAlgorithm(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "IntegrityAlgorithm");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceIEEE1905Security_EncryptionAlgorithm(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "EncryptionAlgorithm");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLD_MLDMACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MLDMACAddress");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLD_TIDLinkMapNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TIDLinkMapNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLD_AffiliatedAPNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "AffiliatedAPNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLD_STAMLDNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "STAMLDNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_EMLMREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "APMLDConfig", "EMLMREnabled");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_EMLSREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "APMLDConfig", "EMLSREnabled");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_STREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "APMLDConfig", "STREnabled");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_NSTREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "APMLDConfig", "NSTREnabled");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_TIDLinkMapNegotiation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "APMLDConfig", "TIDLinkMapNegotiation");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDTIDLinkMap_Direction(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Direction");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDTIDLinkMap_TID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TID");
	return 0;
}

#if 0
static int get_WiFiDataElementsNetworkDeviceAPMLDTIDLinkMap_BSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	//TODO
	return 0;
}
#endif

static int get_WiFiDataElementsNetworkDeviceAPMLDTIDLinkMap_LinkID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "LinkID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_BSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BSSID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_LinkID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "LinkID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_RUID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "RUID");
	return 0;
}

#if 0
static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_DisabledSubChannels(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	//TODO
	return 0;
}
#endif

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_PacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "PacketsSent");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_PacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "PacketsReceived");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_UnicastBytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "UnicastBytesSent");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_UnicastBytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "UnicastBytesReceived");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_ErrorsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ErrorsSent");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_MulticastBytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MulticastBytesSent");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_MulticastBytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MulticastBytesReceived");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_BroadcastBytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BroadcastBytesSent");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_BroadcastBytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BroadcastBytesReceived");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_EstServiceParametersBE(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "EstServiceParametersBE");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_EstServiceParametersBK(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "EstServiceParametersBK");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_EstServiceParametersVI(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "EstServiceParametersVI");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_EstServiceParametersVO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "EstServiceParametersVO");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_MLDMACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MLDMACAddress");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_Hostname(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Hostname");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_IPv4Address(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "IPV4Address");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_IPv6Address(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "IPV6Address");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_IsbSTA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "IsbSTA");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_LastConnectTime(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "LastConnectTime");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_BytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BytesSent");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_BytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BytesReceived");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_PacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "PacketsSent");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_PacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "PacketsReceived");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_ErrorsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ErrorsSent");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_ErrorsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ErrorsReceived");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_RetransCount(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "RetransCount");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_STATIDLinkMapNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "STATIDLinkMapNumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_AffiliatedSTANumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "AffiliatedSTANumberOfEntries");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_EMLMRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi7Capabilities", "EMLMRSupport");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_EMLSRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi7Capabilities", "EMLSRSupport");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_STRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi7Capabilities", "STRSupport");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_NSTRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi7Capabilities", "NSTRSupport");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_TIDLinkMapNegotiation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "WiFi7Capabilities", "TIDLinkMapNegotiation");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_EMLMREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "STAMLDConfig", "EMLMREnabled");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_EMLSREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "STAMLDConfig", "EMLSREnabled");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_STREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "STAMLDConfig", "STREnabled");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_NSTREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "STAMLDConfig", "NSTREnabled");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_TIDLinkMapNegotiation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "STAMLDConfig", "TIDLinkMapNegotiation");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTATIDLinkMap_Direction(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "Direction");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTATIDLinkMap_TID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "TID");
	return 0;
}

#if 0
static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTATIDLinkMap_BSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	//TODO
	return 0;
}
#endif

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTATIDLinkMap_LinkID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "LinkID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_MACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MACAddress");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_BSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BSSID");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_BytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BytesSent");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_BytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "BytesReceived");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_PacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "PacketsSent");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_PacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "PacketsReceived");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_ErrorsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ErrorsSent");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_SignalStrength(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "SignalStrength");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_EstMACDataRateDownlink(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "EstMACDataRateDownlink");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_EstMACDataRateUplink(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "EstMACDataRateUplink");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_LastDataDownlinkRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "LastDataDownlinkRate");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_LastDataUplinkRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "LastDataUplinkRate");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_UtilizationReceive(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "UtilizationReceive");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_UtilizationTransmit(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "UtilizationTransmit");
	return 0;
}

static int get_WiFiDataElementsNetworkDevicebSTAMLD_MLDMACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "bSTAMLD", "MLDMACAddress");
	return 0;
}

static int get_WiFiDataElementsNetworkDevicebSTAMLD_BSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "bSTAMLD", "BSSID");
	return 0;
}

static int get_WiFiDataElementsNetworkDevicebSTAMLD_AffiliatedbSTAList(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value_array_all(((struct dm_data *)data)->json_object, ",", 2, "bSTAMLD", "AffiliatedbSTAList");
	return 0;
}

static int get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_EMLMREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "bSTAMLD", "bSTAMLDConfig", "EMLMREnabled");
	return 0;
}

static int get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_EMLSREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "bSTAMLD", "bSTAMLDConfig", "EMLSREnabled");
	return 0;
}

static int get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_STREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "bSTAMLD", "bSTAMLDConfig", "STREnabled");
	return 0;
}

static int get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_NSTREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "bSTAMLD", "bSTAMLDConfig", "NSTREnabled");
	return 0;
}

static int get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_TIDLinkMapNegotiation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 3, "bSTAMLD", "bSTAMLDConfig", "TIDLinkMapNegotiation");
	return 0;
}

/*************************************************************
 * OPERATE COMMANDS
 *************************************************************/
static operation_args WiFiDataElementsNetwork_SetTrafficSeparation_args = {
	.in = (const char *[]) {
		"Enable",
		"SSIDtoVIDMapping.{i}.SSID",
		"SSIDtoVIDMapping.{i}.VID",
		NULL
	},
	.out = (const char *[]) {
		"Status",
		NULL
	}
};

struct wifi_operate_args
{
	char *arg1;
	char *arg2;
};

static int get_operate_args_WiFiDataElementsNetwork_SetTrafficSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char *)&WiFiDataElementsNetwork_SetTrafficSeparation_args;
	return 0;
}

static int operate_WiFiDataElementsNetwork_SetTrafficSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
#define MAX_ARGS 16

	struct wifi_operate_args operate_args[MAX_ARGS] = {0};
	char status[32];
	bool b = false;

	snprintf(status, sizeof(status), "%s", "Success");

	char *enable = dmjson_get_value((json_object *)value, 1, "Enable");
	if (!enable || *enable == '\0' || bbfdm_validate_boolean(ctx, enable)) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	for (int i = 0; i < MAX_ARGS; i++) {

		char ssid[32] = {0};
		char vid[32] = {0};

		snprintf(ssid, sizeof(ssid), "SSIDtoVIDMapping.%d.SSID", i+1);
		snprintf(vid, sizeof(vid), "SSIDtoVIDMapping.%d.VID", i+1);

		operate_args[i].arg1 = dmjson_get_value((json_object *)value, 1, ssid);
		operate_args[i].arg2 = dmjson_get_value((json_object *)value, 1, vid);

		// Instance number must be assigned sequentially without gaps, if one is empty then break the loop
		if (*(operate_args[i].arg1) == '\0')
			break;

		struct uci_section *s = NULL;

		uci_foreach_option_eq("mapcontroller", "ap", "ssid", operate_args[i].arg1, s) {

			// If VID is not empty then update it
			if (*(operate_args[i].arg2) != '\0')
				dmuci_set_value_by_section(s, "vid", operate_args[i].arg2);
		}
	}

	string_to_bool(enable, &b);
	dmuci_set_value("mapcontroller", "controller", "enable_ts", b ? "1" : "0");

	reload_mapcontroller_service(ctx->dm_type, true, true);

end:
	fill_blob_param(&ctx->bb, "Status", status, DMT_TYPE[DMT_STRING], 0);
	return 0;

#undef MAX_ARGS
}

#if 0
static operation_args WiFiDataElementsNetwork_SetPreferredBackhauls_args = {
	.in = (const char *[]) {
		"PreferredBackhauls.{i}.BackhaulMACAddress",
		"PreferredBackhauls.{i}.bSTAMACAddress",
		NULL
	},
	.out = (const char *[]) {
		"Status",
		NULL
	}
};

static int get_operate_args_WiFiDataElementsNetwork_SetPreferredBackhauls(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char *)&WiFiDataElementsNetwork_SetPreferredBackhauls_args;
	return 0;
}

static int operate_WiFiDataElementsNetwork_SetPreferredBackhauls(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	//TODO
	return 0;
}
#endif

static operation_args WiFiDataElementsNetwork_SetSSID_args = {
	.in = (const char *[]) {
		"SSID",
		BBF_VENDOR_PREFIX"NewSSID",
		"Enable",
		"AddRemoveChange",
		"PassPhrase",
		"Band",
		"AKMsAllowed",
		"HaulType",
		"Type",
		BBF_VENDOR_PREFIX"EnableMLD",
		BBF_VENDOR_PREFIX"TrafficSeparation",
		NULL
	},
	.out = (const char *[]) {
		"Status",
		NULL
	}
};

static int get_operate_args_WiFiDataElementsNetwork_SetSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char *)&WiFiDataElementsNetwork_SetSSID_args;
	return 0;
}

static bool duplicate_band_exist(char **band_arr, int band_arr_length)
{
	if (band_arr == NULL)
		return false;

	for (int i = 0; i < (band_arr_length - 1); i++) {
		for (int j = i + 1; j < band_arr_length; j++) {
			if (DM_STRCMP(band_arr[i], band_arr[j]) == 0)
				return true;
		}
	}

	return false;
}

static bool validate_requested_ssid(const char *ssid, unsigned *idx)
{
	struct uci_section *s = NULL;
	char *curr_ssid = NULL;
	unsigned count = 1;

	if (ssid == NULL || idx == NULL)
		return false;

	/* Check if ssid is already added */
	uci_path_foreach_sections(bbfdm, "dmmap_wireless", "ssid", s) {
		dmuci_get_value_by_section_string(s, "ssid", &curr_ssid);

		if (DM_STRCMP(curr_ssid, ssid) == 0) {
			return false;
		}

		count = count + 1;
	}

	*idx = count;
	return true;
}

static int find_free_vlan_id(void)
{
	struct uci_section *ap_sec = NULL;
	char vid_str[8] = {0};
	char *primary_vid;
	int vid;

	primary_vid = dmuci_get_option_value_fallback_def("mapcontroller", "controller", "primary_vid", "1");

	for (vid = 4094; vid >= 2; vid--) {
		snprintf(vid_str, sizeof(vid_str), "%d", vid);
		ap_sec = get_dup_section_in_config_opt("mapcontroller", "ap", "vid", vid_str);
		if (ap_sec == NULL && DM_STRCMP(vid_str, primary_vid) != 0)
			return vid;
	}

	return -1; // no free VLAN found
}

static char *process_set_ssid_add_req(struct dmctx *ctx, set_ssid_param *op_param)
{
	struct uci_section *s = NULL;
	char curr_band[90] = {0};
	char passwd[256] = {0};
	char **band_arr = NULL;
	size_t band_arr_length = 0;
	char **enc_arr = NULL;
	size_t enc_arr_length = 0;
	char *haul_type = NULL;
	char vid_str[8] = {0};
	char mld_id[3] = {0};
	unsigned idx = 1;
	bool enable = true;
	bool mld_enable = false;
	bool ts_enable = false;

	if (op_param == NULL) {
		bbfdm_set_fault_message(ctx, "Internal Error");
		return "Error_Other";
	}

	if (DM_STRLEN(op_param->enab) != 0) {
		string_to_bool(op_param->enab, &enable);
	}

	/* If not mld capabled device ignore mld enable config */
	if (mld_capable && DM_STRLEN(op_param->mld_enab) != 0) {
		string_to_bool(op_param->mld_enab, &mld_enable);
	}

	if (DM_STRLEN(op_param->ts) != 0) {
		string_to_bool(op_param->ts, &ts_enable);
	}

	if (DM_STRLEN(op_param->band) == 0 || DM_LSTRCMP(op_param->band, "All") == 0) {
		struct radio_info *node = NULL;
		int len = 0;

		list_for_each_entry(node, &dev_list, list) {
			if (node->band[0] == '2') {
				len = DM_STRLEN(curr_band);
				snprintf(curr_band+len, sizeof(curr_band)-len, "%s,", "2.4");
			} else {
				len = DM_STRLEN(curr_band);
				snprintf(curr_band+len, sizeof(curr_band)-len, "%c,", node->band[0]);
			}
		}

		len = DM_STRLEN(curr_band);
		if (len) {
			curr_band[len-1] = '\0';
		}
	} else {
		snprintf(curr_band, sizeof(curr_band), "%s", op_param->band);
	}

	band_arr = strsplit(curr_band, ",", &band_arr_length);

	/* Band should not have duplicate value to avoid multiple instance with
	 * same SSID and Band */
	if (duplicate_band_exist(band_arr, band_arr_length)) {
		bbfdm_set_fault_message(ctx, "Band should not contain duplicate value");
		return "Error_Invalid_Input";
	}

	if (DM_STRLEN(op_param->akm) != 0) {
		enc_arr = strsplit(op_param->akm, ",", &enc_arr_length);
	}

	if (DM_STRLEN(op_param->key) == 0) {
		char *def_key = get_default_key();
		if (DM_STRLEN(def_key) == 0)
			return "Error_Other";

		snprintf(passwd, sizeof(passwd), "%s", def_key);
	} else {
		snprintf(passwd, sizeof(passwd), "%s", op_param->key);
	}

	if (enc_arr_length > band_arr_length) {
		return "Error_Invalid_Input";
	}

	/* Reject if requested SSID can not be configured */
	if (!validate_requested_ssid(op_param->ssid, &idx)) {
		bbfdm_set_fault_message(ctx, "SSID (%s) already exist", op_param->ssid);
		return "Error_Invalid_Input";
	}

	haul_type = get_haultype_uci_value(op_param->haul);

	if (ts_enable && DM_STRCMP(haul_type, "fronthaul") == 0) {
		int vid;

		vid = find_free_vlan_id();
		if (vid < 0) {
			bbfdm_set_fault_message(ctx, "No free VLAN ID in range 2–4094");
			return "Error_Other";
		}

		snprintf(vid_str, sizeof(vid_str), "%d", vid);
	}

	/* If MLD config enabled then find an unused mld id */
	if (mld_enable == true) {
		/* First remove unused MLD section if any */
		remove_unused_mld_sections();

		for (int id = 1; id <= 24; id++) {
			char mld_str[3] = {0};
			struct uci_section *mld_sec = NULL;

			snprintf(mld_str, sizeof(mld_str), "%d", id);
			mld_sec = get_dup_section_in_config_opt("mapcontroller", "mld", "id", mld_str);
			if (mld_sec == NULL) {
				snprintf(mld_id, sizeof(mld_id), "%s", mld_str);
				break;
			}
		}

		if (DM_STRLEN(mld_id) == 0) {
			bbfdm_set_fault_message(ctx, "No free MLD unit in range of 1 to 24");
			return "Error_Other";
		} else {

			dmuci_add_section("mapcontroller", "mld", &s);
			dmuci_set_value_by_section(s, "ssid", op_param->ssid);
			dmuci_set_value_by_section(s, "key", passwd);
			dmuci_set_value_by_section(s, "type", haul_type);
			dmuci_set_value_by_section(s, "id", mld_id);
		}
	}

	for (int i = 0; i < band_arr_length; i++) {
		char sec_name[64];
		char *encryp = NULL;
		char *device = NULL;

		snprintf(sec_name, sizeof(sec_name), "ap_%s_%c_%u", op_param->ssid, band_arr[i][0], idx+i);
		replace_special_char(sec_name, '_');

		encryp = (enc_arr != NULL && i < enc_arr_length) ? get_AKMs_uci_name(enc_arr[i]) : "sae-mixed";

		/* Add section in mapcontroller */
		dmuci_add_section("mapcontroller", "ap", &s);
		dmuci_rename_section_by_section(s, sec_name);

		dmuci_set_value_by_section(s, "ssid", op_param->ssid);
		dmuci_set_value_by_section(s, "key", passwd);
		dmuci_set_value_by_section(s, "encryption", encryp);
		dmuci_set_value_by_section(s, "type", haul_type);
		dmuci_set_value_by_section(s, "enabled", (enable == true) ? "1" : "0");
		dmuci_set_value_by_section(s, "mld_id", (mld_enable == true) ? mld_id : "");
		dmuci_set_value_by_section(s, "vid", vid_str);
		dmuci_set_value_by_section(s, "network_type", DM_STRLEN(op_param->network_type) ? op_param->network_type : "");

		if (DM_LSTRCMP(band_arr[i], "2.4") == 0) {
			dmuci_set_value_by_section(s, "band", "2");
			device = get_device_from_band("2");
		} else {
			dmuci_set_value_by_section(s, "band", band_arr[i]);
			device = get_device_from_band(band_arr[i]);
		}

		/* Add section in dmmap */
		char name[64];
		struct uci_section *dmmap_s = NULL;

		snprintf(name, sizeof(name), "%s_%c_%u", op_param->ssid, band_arr[i][0], idx+i);

		dmuci_add_section_bbfdm("dmmap_wireless", "ssid", &dmmap_s);
		dmuci_set_value_by_section(dmmap_s, "enabled", (enable == true) ? "1" : "0");
		dmuci_set_value_by_section(dmmap_s, "ssid", op_param->ssid);
		dmuci_set_value_by_section(dmmap_s, "name", name);
		dmuci_set_value_by_section(dmmap_s, "ap_section_name", sec_name);
		dmuci_set_value_by_section(dmmap_s, "device", device ? device : "");
		dmuci_set_value_by_section(dmmap_s, "mld_id", (mld_enable == true) ? mld_id : "");
	}

	return "Success";
}

static char *process_set_ssid_remove_req(struct dmctx *ctx, set_ssid_param *op_param)
{
	struct uci_section *s = NULL, *tmp = NULL, *dmmap = NULL;
	char *curr_ssid = NULL, *mld = NULL;
	bool ap_deleted = false;

	if (op_param == NULL) {
		bbfdm_set_fault_message(ctx, "Internal Error");
		return "Error_Other";
	}

	/* Remove from mapcontroller */
	uci_foreach_sections_safe("mapcontroller", "ap", tmp, s) {
		dmuci_get_value_by_section_string(s, "ssid", &curr_ssid);
		if (DM_STRCMP(op_param->ssid, curr_ssid) != 0) {
			continue;
		}

		dmuci_get_value_by_section_string(s, "mld_id", &mld);

		// delete this section
		dmuci_delete_by_section(s, NULL, NULL);

		dmmap = get_dup_section_in_dmmap_opt("dmmap_wireless", "wifi-iface", "section_name", section_name(s));
		if (dmmap) {
			dmuci_delete_by_section(dmmap, NULL, NULL);
		}

		ap_deleted = true;
	}

	if (ap_deleted == false) {
		bbfdm_set_fault_message(ctx, "No such SSID exists");
		return "Error_Invalid_Input";
	}

	/* Remove from dmmap */
	tmp = NULL;
	s = NULL;
	uci_path_foreach_sections_safe(bbfdm, "dmmap_wireless", "ssid", tmp, s) {
		dmuci_get_value_by_section_string(s, "ssid", &curr_ssid);
		if (DM_STRCMP(op_param->ssid, curr_ssid) != 0) {
			continue;
		}

		// delete this section
		dmuci_delete_by_section(s, NULL, NULL);
	}

	/* If mld was configured then after deleting all SSIDs we should remove MLD section */
	if (DM_STRLEN(mld) != 0) {
		s = get_dup_section_in_config_opt("mapcontroller", "mld", "id", mld);
		if (s) {
			dmuci_delete_by_section(s, NULL, NULL);
		}
	}

	return "Success";
}

static char *process_set_ssid_change_req(struct dmctx *ctx, set_ssid_param *op_param)
{
	struct uci_section *s = NULL;
	char *curr_ssid = NULL;
	unsigned ap_count = 0;

	if (op_param == NULL) {
		bbfdm_set_fault_message(ctx, "Internal Error");
		return "Error_Other";
	}

	/* Check if ssid is present */
	uci_foreach_sections("mapcontroller", "ap", s) {
		dmuci_get_value_by_section_string(s, "ssid", &curr_ssid);

		if (DM_STRCMP(curr_ssid, op_param->ssid) == 0) {
			ap_count++;
		}
	}

	if (ap_count == 0) {
		bbfdm_set_fault_message(ctx, "No such SSID exists");
		return "Error_Invalid_Input";
	}

	if (DM_STRLEN(op_param->band) != 0) {
		bbfdm_set_fault_message(ctx, "Once assigned, band can not be modified");
		return "Error_Invalid_Input";
	}

	if (DM_STRLEN(op_param->akm) != 0) {
		bbfdm_set_fault_message(ctx, "Once assigned, AKMsAllowed can not be modified");
		return "Error_Invalid_Input";
	}

	if (DM_STRLEN(op_param->mld_enab) != 0) {
		bbfdm_set_fault_message(ctx, "Once assigned, MLD setting can not be modified");
		return "Error_Invalid_Input";
	}

	if (DM_STRLEN(op_param->new_ssid) != 0) {
		/* There should not exist any ssid with new value */
		unsigned idx = 0;
		if (!validate_requested_ssid(op_param->new_ssid, &idx)) {
			bbfdm_set_fault_message(ctx, "SSID can't be changed to (%s), since it's already exist", op_param->new_ssid);
			return "Error_Invalid_Input";
		}
	}

	uci_foreach_sections("mapcontroller", "ap", s) {
		dmuci_get_value_by_section_string(s, "ssid", &curr_ssid);
		if (DM_STRCMP(op_param->ssid, curr_ssid) != 0) {
			continue;
		}

		// modify this section
		bool enable = true;
		if (DM_STRLEN(op_param->enab) != 0) {
			string_to_bool(op_param->enab, &enable);
			dmuci_set_value_by_section(s, "enabled", (enable == true) ? "1" : "0");
		}

		if (DM_STRLEN(op_param->key) != 0) {
			dmuci_set_value_by_section(s, "key", op_param->key);
		}

		if (DM_STRLEN(op_param->haul) != 0) {
			dmuci_set_value_by_section(s, "type", get_haultype_uci_value(op_param->haul));
		}

		if (DM_STRLEN(op_param->new_ssid) != 0) {
			dmuci_set_value_by_section(s, "ssid", op_param->new_ssid);
		}

		if (DM_STRLEN(op_param->network_type) != 0) {
			dmuci_set_value_by_section(s, "network_type", op_param->network_type);
		}

	}

	/* Modify in dmmap */
	uci_path_foreach_sections(bbfdm, "dmmap_wireless", "ssid", s) {
		dmuci_get_value_by_section_string(s, "ssid", &curr_ssid);
		if (DM_STRCMP(op_param->ssid, curr_ssid) != 0) {
			continue;
		}

		// modify this section
		bool enable = true;
		if (DM_STRLEN(op_param->enab) != 0) {
			string_to_bool(op_param->enab, &enable);
			dmuci_set_value_by_section(s, "enabled", (enable == true) ? "1" : "0");
		}

		if (DM_STRLEN(op_param->new_ssid) != 0) {
			dmuci_set_value_by_section(s, "ssid", op_param->new_ssid);
		}
	}

	if (mld_capable == false) {
		return "Success";
	}

	/* if there exist a MLD section with key ssid then MLD section should be updated */
	struct uci_section *mld_sec = get_dup_section_in_config_opt("mapcontroller", "mld", "ssid", op_param->ssid);
	if (mld_sec != NULL) {
		if (DM_STRLEN(op_param->key) != 0) {
			dmuci_set_value_by_section(mld_sec, "key", op_param->key);
		}

		if (DM_STRLEN(op_param->haul) != 0) {
			dmuci_set_value_by_section(mld_sec, "type", get_haultype_uci_value(op_param->haul));
		}

		if (DM_STRLEN(op_param->new_ssid) != 0) {
			dmuci_set_value_by_section(mld_sec, "ssid", op_param->new_ssid);
		}
	}

	return "Success";
}

static void wifi_validate_easymesh_ssid(struct ubus_request *req, int type, struct blob_attr *msg)
{
	validate_ssid *param = (validate_ssid *)req->priv;
	if (param == NULL)
		return;

	param->status = -1;

	if (!msg) {
		BBF_ERR("Error: Received NULL message in (%s)", __func__);
		return;
	}

	char *str = blobmsg_format_json_indent(msg, true, -1);
	if (!str) {
		BBF_ERR("Error: Failed to format JSON from blobmsg in (%s)", __func__);
		return;
	}

	json_object *dump_output = json_tokener_parse(str);
	free((char *)str);

	if (!dump_output) {
		BBF_ERR("Error: Failed to parse JSON string in (%s)", __func__);
		return;
	}

	if (!validate_wifi_de_dump_output(dump_output)) {
		BBF_ERR("Error: WiFi DataElements Dump Output is not valid in (%s)", __func__);
		json_object_put(dump_output);
		return;
	}

	json_object *data_arr = NULL;
	if (!json_object_object_get_ex(dump_output, "data", &data_arr) ||
	    json_object_get_type(data_arr) != json_type_array) {
		json_object_put(dump_output);
		return;
	}

	json_object *data_obj = json_object_array_get_idx(data_arr, 0);
	if (!data_obj) {
		json_object_put(dump_output);
		return;
	}

	json_object *dev_arr = NULL, *dev_obj = NULL;
	int j = 0;

	dmjson_foreach_obj_in_array(data_obj, dev_arr, dev_obj, j, 2, "wfa-dataelements:Network", "DeviceList") {
		bool found = false;
		struct json_object *radio_arr = NULL, *radio_obj = NULL;
		int k = 0;

		char *key = dmjson_get_value(dev_obj, 1, "ID");
		if (!key || *key == '\0')
			continue;

		dmjson_foreach_obj_in_array(dev_obj, radio_arr, radio_obj, k, 1, "RadioList") {
			json_object *bss_arr = NULL, *bss_obj = NULL;
			int l = 0;
			char mac[32] = {0};

			char *radio_id = dmjson_get_value(radio_obj, 1, "ID");
			char *str = dm_base64_decode(radio_id);

			if ((str != NULL) && (DM_STRLEN(radio_id) == 8))
				string_to_mac(str, 6, mac, sizeof(mac));

			if (DM_STRLEN(mac) == 0)
				continue;

			dmjson_foreach_obj_in_array(radio_obj, bss_arr, bss_obj, l, 1, "BSSList") {
				char *id = dmjson_get_value(bss_obj, 1, "SSID");
				if (DM_STRCMP(id, param->ssid) == 0) {
					found = true;
					break;
				}
			}

			if (param->op == SET_SSID_REMOVE && found == true) {
				json_object_put(dump_output);
				return;
			}

			if (param->op != SET_SSID_REMOVE && found == true) {
				break;
			}
		}

		if (param->op != SET_SSID_REMOVE && found == false) {
			json_object_put(dump_output);
			return;
		}
	}

	param->status = 0;
	// Free json output
	json_object_put(dump_output);
}

static void check_easymesh_agent_ssid(validate_ssid *param)
{
	int i = 0;
	struct blob_buf bb = {0};

	if (param == NULL)
		return;

	struct uci_section *ap = get_dup_section_in_config_opt("mapcontroller", "ap", "ssid", param->ssid);
	if ((ap != NULL) && (param->op == SET_SSID_ADD || param->op == SET_SSID_CHANGE)) {
		bool enable = true;
		char *val = dmuci_get_value_by_section_fallback_def(ap, "enabled", "1");
		string_to_bool(val, &enable);

		if (enable == false) {
			// No need to wait for network
			param->status = 0;
			return;
		}
	}

	memset(&bb, 0, sizeof(struct blob_buf));
	blob_buf_init(&bb, 0);

	for (i = 0; i < 7; i++) {
		sleep(10);
		BBFDM_UBUS_INVOKE_SYNC("wifi.dataelements", "dump", bb.head, 5000, wifi_validate_easymesh_ssid, param);
		if (param->status == 0) {
			break;
		}
	}

	blob_buf_free(&bb);
}

static int operate_WiFiDataElementsNetwork_SetSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	set_ssid_param op_param;
	char status[32] = {0};

	if (easymesh_enable == false) {
		bbfdm_set_fault_message(ctx, "Easymesh not enabled/supported on device");
		snprintf(status, sizeof(status), "%s", "Error_Other");
		goto end;
	}

	char *add_remove_change = dmjson_get_value((json_object *)value, 1, "AddRemoveChange");
	int op = get_requested_operation(add_remove_change);

	if (op == SET_SSID_INVALID) {
		bbfdm_set_fault_message(ctx, "Invalid AddRemoveChange value: (%s)", add_remove_change);
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	memset(&op_param, 0, sizeof(set_ssid_param));

	op_param.ssid = dmjson_get_value((json_object *)value, 1, "SSID");
	op_param.new_ssid = dmjson_get_value((json_object *)value, 1, BBF_VENDOR_PREFIX"NewSSID");
	op_param.enab = dmjson_get_value((json_object *)value, 1, "Enable");
	op_param.key = dmjson_get_value((json_object *)value, 1, "PassPhrase");
	op_param.band = dmjson_get_value((json_object *)value, 1, "Band");
	op_param.akm = dmjson_get_value((json_object *)value, 1, "AKMsAllowed");
	op_param.haul = dmjson_get_value((json_object *)value, 1, "HaulType");
	op_param.network_type = dmjson_get_value((json_object *)value, 1, "Type");
	op_param.mld_enab = dmjson_get_value((json_object *)value, 1, BBF_VENDOR_PREFIX"EnableMLD");
	op_param.ts = dmjson_get_value((json_object *)value, 1, BBF_VENDOR_PREFIX"TrafficSeparation");

	if (bbfdm_validate_string(ctx, op_param.ssid, 1, 32, NULL, NULL) ||
			bbfdm_validate_string(ctx, op_param.key, -1, 64, NULL, NULL)||
			validate_band_value(ctx, op_param.band) != 0 ||
			validate_akms_value(ctx, op_param.akm) != 0 ||
			validate_haultype_value(ctx, op_param.haul) != 0) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	if (bbfdm_validate_string(ctx, op_param.new_ssid, 1, 32, NULL, NULL) &&
			op == SET_SSID_CHANGE ) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	if (DM_STRLEN(op_param.enab) != 0 && bbfdm_validate_boolean(ctx, op_param.enab) != 0) {
		bbfdm_set_fault_message(ctx, "Invalid Enable value: (%s)", op_param.enab);
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	if (DM_STRLEN(op_param.mld_enab) != 0 && bbfdm_validate_boolean(ctx, op_param.mld_enab) != 0) {
		bbfdm_set_fault_message(ctx, "Invalid Enable value: (%s)", op_param.mld_enab);
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	if (DM_STRLEN(op_param.ts) != 0 && bbfdm_validate_boolean(ctx, op_param.ts) != 0) {
		bbfdm_set_fault_message(ctx, "Invalid Enable value: (%s)", op_param.ts);
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	switch (op) {
	case SET_SSID_ADD:
		snprintf(status, sizeof(status), "%s", process_set_ssid_add_req(ctx, &op_param));
		break;
	case SET_SSID_REMOVE:
		snprintf(status, sizeof(status), "%s", process_set_ssid_remove_req(ctx, &op_param));
		break;
	case SET_SSID_CHANGE:
		snprintf(status, sizeof(status), "%s", process_set_ssid_change_req(ctx, &op_param));
		break;
	}

	if (DM_LSTRCMP(status, "Success") != 0) {
		// Revert mapcontroller config changes
		reload_mapcontroller_service(ctx->dm_type, false, false);
	} else {
		// Commit mapcontroller config changes
		reload_mapcontroller_service(ctx->dm_type, true, true);

		validate_ssid v_param;
		v_param.op = op;
		v_param.status = -1;
		v_param.ssid = op_param.ssid;

		if (op == SET_SSID_CHANGE && DM_STRLEN(op_param.new_ssid) != 0) {
			/* SSID would be changed with new_ssid */
			v_param.ssid = op_param.new_ssid;
		}

		check_easymesh_agent_ssid(&v_param);
		if (v_param.status != 0) {
			snprintf(status, sizeof(status), "%s", "Error_Other");
		}
		dmubus_call_set("wifi.dataelements", "refresh", UBUS_ARGS{0}, 0);
	}

end:
	fill_blob_param(&ctx->bb, "Status", status, DMT_TYPE[DMT_STRING], 0);
	return 0;
}

static operation_args WiFiDataElementsNetworkDevice_SetSTASteeringState_args = {
	.in = (const char *[]) {
		"Disallowed",
		NULL
	},
	.out = (const char *[]) {
		"Status",
		NULL
	}
};

static int get_operate_args_WiFiDataElementsNetworkDevice_SetSTASteeringState(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char *)&WiFiDataElementsNetworkDevice_SetSTASteeringState_args;
	return 0;
}

static int operate_WiFiDataElementsNetworkDevice_SetSTASteeringState(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char status[32];
	bool b = false;

	snprintf(status, sizeof(status), "%s", "Success");

	char *disallowed = dmjson_get_value((json_object *)value, 1, "Disallowed");
	if (!disallowed || *disallowed == '\0' || bbfdm_validate_boolean(ctx, disallowed)) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	string_to_bool(disallowed, &b);
	dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "steer_disallow", b ? "1" : "0");

	// Commit mapcontroller config changes
	reload_mapcontroller_service(ctx->dm_type, true, true);

end:
	fill_blob_param(&ctx->bb, "Status", status, DMT_TYPE[DMT_STRING], 0);
	return 0;
}

static operation_args WiFiDataElementsNetworkDevice_InitiateWPSPBC_args = {
	.out = (const char *[]) {
		"Status",
		NULL
	}
};

static int get_operate_args_WiFiDataElementsNetworkDevice_InitiateWPSPBC(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char *)&WiFiDataElementsNetworkDevice_InitiateWPSPBC_args;
	return 0;
}

static int operate_WiFiDataElementsNetworkDevice_InitiateWPSPBC(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	/* This is a temporary solution as pushback operation to network devices is not supported
	 * right now. This should be done later through mapcontroller. So as of now it is being
	 * supported on controller GW device only */
	if (DM_LSTRCMP(instance, "1") != 0) {
		bbfdm_set_fault_message(ctx, "InitiateWPSPBC() is only supported on controller gateway device");
		return USP_FAULT_REQUEST_DENIED;
	}

	dmubus_call_set("map.agent", "WPSPBC", UBUS_ARGS{0}, 0);
	fill_blob_param(&ctx->bb, "Status", "Success", DMT_TYPE[DMT_STRING], 0);
	return 0;
}

/*
static operation_args WiFiDataElementsNetworkDevice_SetDFSState_args = {
	.in = (const char *[]) {
		"DFSEnable",
		NULL
	},
	.out = (const char *[]) {
		"Status",
		NULL
	}
};

static int get_operate_args_WiFiDataElementsNetworkDevice_SetDFSState(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char *)&WiFiDataElementsNetworkDevice_SetDFSState_args;
	return 0;
}

static int operate_WiFiDataElementsNetworkDevice_SetDFSState(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	//TODO
	return 0;
}*/

static operation_args wifidataelementsnetworkdeviceradio_channelscanrequest_args = {
    .in = (const char *[]) {
        "OpClass",
        "ChannelList",
        NULL
    },
    .out = (const char *[]) {
        "Status",
        NULL
    }
};

static int get_operate_args_WiFiDataElementsNetworkDeviceRadio_ChannelScanRequest(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
    *value = (char *)&wifidataelementsnetworkdeviceradio_channelscanrequest_args;
    return 0;
}

static int operate_WiFiDataElementsNetworkDeviceRadio_ChannelScanRequest(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char status[32] = {0};
	char *agent_id = NULL;
	char *macaddr = NULL;
	char *pch = NULL;
	char *spch = NULL;

	snprintf(status, sizeof(status), "%s", "Success");

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "agent_id", &agent_id);
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "macaddr", &macaddr);

	if ((bbfdm_validate_string(ctx, agent_id, -1, 17, NULL, MACAddress)) ||
		(bbfdm_validate_string(ctx, macaddr, -1, 17, NULL, MACAddress))) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	char *channel_list = dmjson_get_value((json_object *)value, 1, "ChannelList");

	if (bbfdm_validate_unsignedInt_list(ctx, channel_list, -1, -1, -1, RANGE_ARGS{{NULL,"255"}}, 1)) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	struct json_object *in_args = json_object_new_object();
	json_object_object_add(in_args, "agent", json_object_new_string(agent_id));
	json_object_object_add(in_args, "radio", json_object_new_string(macaddr));

	if (DM_STRLEN(channel_list) != 0) {
		struct json_object *channel_array = json_object_new_array();

		for (pch = strtok_r(channel_list, ",", &spch); pch != NULL; pch = strtok_r(NULL, ",", &spch)) {
			struct json_object *val_array = json_object_new_array();
			json_object_array_add(val_array, json_object_new_int(DM_STRTOL(pch)));
			json_object_array_add(channel_array, val_array);
		}

		json_object_object_add(in_args, "channel", channel_array);
	}

	int res = dmubus_call_blob_set("map.controller", "scan", in_args);
	if (res) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
	}

	json_object_put(in_args);

end:
	fill_blob_param(&ctx->bb, "Status", status, DMT_TYPE[DMT_STRING], 0);

	// Update WiFi DataElements Dump Output
	dmubus_call_set("wifi.dataelements", "refresh", UBUS_ARGS{0}, 0);

    return 0;
}

/*static operation_args wifidataelementsnetworkdeviceradio_wifirestart_args = {
    .out = (const char *[]) {
        "Status",
        NULL
    }
};

static int get_operate_args_WiFiDataElementsNetworkDeviceRadio_WiFiRestart(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
    *value = (char *)&wifidataelementsnetworkdeviceradio_wifirestart_args;
    return 0;
}

static int operate_WiFiDataElementsNetworkDeviceRadio_WiFiRestart(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
    //TODO
    return 0;
}*/

static char *find_radio_band(const char *agent_id, const char *radio_mac)
{
	struct uci_section *sec = NULL;
	char *band = NULL;

	uci_foreach_option_eq("mapcontroller", "radio", "agent_id", agent_id, sec) {
		char *macaddr = NULL;

		dmuci_get_value_by_section_string(sec, "macaddr", &macaddr);

		if (DM_STRCMP(macaddr, radio_mac) == 0) {
			dmuci_get_value_by_section_string(sec, "band", &band);
			break;
		}
	}

	return band;
}

static char *find_radio_mac(const char *agent_id, const char *radio_band)
{
	struct uci_section *sec = NULL;
	char *macaddr = NULL;

	uci_foreach_option_eq("mapcontroller", "radio", "agent_id", agent_id, sec) {
		char *band = NULL;

		dmuci_get_value_by_section_string(sec, "band", &band);

		if (DM_STRCMP(band, radio_band) == 0) {
			dmuci_get_value_by_section_string(sec, "macaddr", &macaddr);
			break;
		}
	}

	return macaddr;
}

static char *get_backhaul_sta_from_radio(json_object *dev_obj, const char *radio_mac)
{
	struct json_object *radio_arr = NULL, *radio_obj = NULL;
	char *bsta = NULL;
	int i = 0;

	if (!dev_obj || !radio_mac)
		return NULL;

	dmjson_foreach_obj_in_array(dev_obj, radio_arr, radio_obj, i, 1, "RadioList") {
		char mac[32] = {0};

		char *radio_id = dmjson_get_value(radio_obj, 1, "ID");
		char *str = dm_base64_decode(radio_id);

		/* Cant use strlen on byte array that might genuinely include 0x00 */
		/* but to get 6 bytes, need 8 input BASE64 chars - check for that */
		if ((str != NULL) && (DM_STRLEN(radio_id) == 8))
			string_to_mac(str, 6, mac, sizeof(mac));

		if (DM_STRLEN(mac) == 0)
			continue;

		if (DM_STRCASECMP(mac, radio_mac) == 0) {
			bsta = dmjson_get_value(radio_obj, 2, "BackhaulSta", "MACAddress");
			break;
		}
	}

	return bsta;
}

static int validate_bssid(const char *bss, char **bss_band)
{
	json_object *data_arr = NULL, *data_obj = NULL;
	int i = 0;

	if (wifi_de_dump_output == NULL || !bss || !bss_band)
		return -1;

	dmjson_foreach_obj_in_array(wifi_de_dump_output, data_arr, data_obj, i, 1, "data") {
		json_object *dev_arr = NULL, *dev_obj = NULL;
		int j = 0;

		dmjson_foreach_obj_in_array(data_obj, dev_arr, dev_obj, j, 2, "wfa-dataelements:Network", "DeviceList") {
			struct json_object *radio_arr = NULL, *radio_obj = NULL;
			int k = 0;

			char *key = dmjson_get_value(dev_obj, 1, "ID");
			if (!key || *key == '\0')
				continue;

			dmjson_foreach_obj_in_array(dev_obj, radio_arr, radio_obj, k, 1, "RadioList") {
				json_object *bss_arr = NULL, *bss_obj = NULL;
				int l = 0;
				char mac[32] = {0};

				char *radio_id = dmjson_get_value(radio_obj, 1, "ID");
				char *str = dm_base64_decode(radio_id);

				/* Cant use strlen on byte array that might genuinely include 0x00 */
				/* but to get 6 bytes, need 8 input BASE64 chars - check for that */
				if ((str != NULL) && (DM_STRLEN(radio_id) == 8))
					string_to_mac(str, 6, mac, sizeof(mac));

				if (DM_STRLEN(mac) == 0)
					continue;

				dmjson_foreach_obj_in_array(radio_obj, bss_arr, bss_obj, l, 1, "BSSList") {
					char *id = dmjson_get_value(bss_obj, 1, "BSSID");
					char *is_backhaul = dmjson_get_value(bss_obj, 1, "BackhaulUse");
					if (DM_STRCASECMP(id, bss) == 0 && DM_STRCMP(is_backhaul, "true") == 0) {
						*bss_band = find_radio_band(key, mac);
						return 0;
					}
				}
			}
		}
	}

	return -1;
}

static operation_args wifidataelementsnetworkdevicemultiapdevicebackhaul_steerwifibackhaul_args = {
    .in = (const char *[]) {
        "TargetBSS",
        "TimeOut",
        NULL
    },
    .out = (const char *[]) {
        "Status",
        NULL
    }
};

static int get_operate_args_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaul_SteerWiFiBackhaul(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
    *value = (char *)&wifidataelementsnetworkdevicemultiapdevicebackhaul_steerwifibackhaul_args;
    return 0;
}

static int operate_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaul_SteerWiFiBackhaul(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	json_object *res = NULL;
	char status[32];
	char *bss_band = NULL;

	snprintf(status, sizeof(status), "%s", "Success");

	char *target_bss = dmjson_get_value((json_object *)value, 1, "TargetBSS");
	char *time_out = dmjson_get_value((json_object *)value, 1, "TimeOut");
	if (DM_STRLEN(target_bss) == 0 || DM_STRLEN(time_out) == 0) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	if (validate_bssid(target_bss, &bss_band) != 0 || DM_STRLEN(bss_band) == 0) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	char *agent = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ID");

	char *radio_mac = find_radio_mac(agent, bss_band);

	char *bsta = get_backhaul_sta_from_radio(((struct dm_data *)data)->json_object, radio_mac);
	if (DM_STRLEN(agent) == 0 || DM_STRLEN(bsta) == 0) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	dmubus_call_timeout("map.controller", "steer_backhaul", UBUS_ARGS{{"agent", agent, String},
															{"bsta", bsta, String},
															{"target_bssid", target_bss, String},
															{"timeout", time_out, Integer}}, 4, (int)DM_STRTOUL(time_out), &res);

	if (res == NULL) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	char *res_status = dmjson_get_value((json_object *)res, 1, "status");
	if (DM_STRCMP(res_status, "ok") != 0)
		snprintf(status, sizeof(status), "%s", "Error_Other");

end:
	fill_blob_param(&ctx->bb, "Status", status, DMT_TYPE[DMT_STRING], 0);

	if (res != NULL) {
		json_object_put(res);
		res = NULL;
	}

	// Update WiFi DataElements Dump Output
	dmubus_call_set("wifi.dataelements", "refresh", UBUS_ARGS{0}, 0);

	return 0;
}

static operation_args wifidataelementsnetworkdeviceradiobsssta_clientsteer_args = {
	.in = (const char *[]) {
		"BTMAbridged",
		"TargetBSSID",
		"BTMDisassociationTimer",
		NULL
	},
	.out = (const char *[]) {
		"Status",
		NULL
	}
};

static int get_operate_args_WiFiDataElementsNetworkDeviceRadioBSSSTA_ClientSteer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char *)&wifidataelementsnetworkdeviceradiobsssta_clientsteer_args;
	return 0;
}

static int operate_WiFiDataElementsNetworkDeviceRadioBSSSTA_ClientSteer(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	json_object *res = NULL;
	char buf[1024] = {0}, status[32];
	char *agent = NULL;
	bool abridged = false;
	char *endval = NULL;
	char *bss_band = NULL;

	snprintf(status, sizeof(status), "%s", "Success");

	char *target_bss = dmjson_get_value((json_object *)value, 1, "TargetBSSID");
	char *diass_timer = dmjson_get_value((json_object *)value, 1, "BTMDisassociationTimer");
	char *btm_abridged = dmjson_get_value((json_object *)value, 1, "BTMAbridged");
	if (DM_STRLEN(target_bss) == 0 || DM_STRLEN(diass_timer) == 0) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	if (validate_bssid(target_bss, &bss_band) != 0 || DM_STRLEN(bss_band) == 0) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	errno = 0;
	strtoul(diass_timer, &endval, 10);
	if ((*endval != 0) || (errno != 0)) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	if (DM_STRLEN(btm_abridged) != 0 && string_to_bool(btm_abridged, &abridged) != 0) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	if (abridged == true) {
		btm_abridged = "1";
	} else {
		btm_abridged = "0";
	}

	DM_STRNCPY(buf, refparam, sizeof(buf));
	char *p = DM_STRSTR(buf, "Radio");
	if (p) *p = 0;

	adm_entry_get_reference_value(ctx, buf, &agent);
	if (DM_STRLEN(agent) == 0) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	char *sta = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "MACAddress");
	if (DM_STRLEN(sta) == 0) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	dmubus_call("map.controller", "steer", UBUS_ARGS{{"agent", agent, String},
							{"sta", sta, String},
							{"target_bssid", target_bss, String},
							{"disassoc_tmo", diass_timer, Integer},
							{"abridged", btm_abridged, Integer}},
							5, &res);

	if (res == NULL) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	char *res_status = dmjson_get_value((json_object *)res, 1, "status");
	if (DM_STRCMP(res_status, "ok") != 0)
		snprintf(status, sizeof(status), "%s", "Error_Other");

end:
	fill_blob_param(&ctx->bb, "Status", status, DMT_TYPE[DMT_STRING], 0);

	// Update WiFi DataElements Dump Output
	dmubus_call_set("wifi.dataelements", "refresh", UBUS_ARGS{0}, 0);

	return 0;
}

/*************************************************************
 * EVENTS
 *************************************************************/
static event_args wifidataelementsassociationevent_associated_args = {
    .name = "wifi.dataelements.Associated",
    .param = (const char *[]) {
        "TimeStamp",
        "BSSID",
        "MACAddress",
        "StatusCode",
        "HTCapabilities",
        "VHTCapabilities",
        NULL
    }
};

static int get_event_args_WiFiDataElementsAssociationEvent_Associated(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
    *value = (char *)&wifidataelementsassociationevent_associated_args;
    return 0;
}

static int event_WiFiDataElementsAssociationEvent_Associated(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case EVENT_CHECK:
			// Nothing to check
			break;
		case EVENT_RUN:
		{
			char *event_time = dmjson_get_value((json_object *)value, 1, "eventTime");
			char *bssid = dmjson_get_value((json_object *)value, 2, "wfa-dataelements:AssociationEvent", "BSSID");
			char *mac_addr = dmjson_get_value((json_object *)value, 2, "wfa-dataelements:AssociationEvent", "MACAddress");
			char *status_code = dmjson_get_value((json_object *)value, 2, "wfa-dataelements:AssociationEvent", "StatusCode");
			char *ht_cap = dmjson_get_value((json_object *)value, 2, "wfa-dataelements:AssociationEvent", "HTCapabilities");
			char *vht_cap = dmjson_get_value((json_object *)value, 2, "wfa-dataelements:AssociationEvent", "VHTCapabilities");

			fill_blob_param(&ctx->bb, "TimeStamp", event_time, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "BSSID", bssid, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "MACAddress", mac_addr, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "StatusCode", status_code, DMT_TYPE[DMT_UNINT], 0);
			fill_blob_param(&ctx->bb, "HTCapabilities", ht_cap, DMT_TYPE[DMT_BASE64], 0);
			fill_blob_param(&ctx->bb, "VHTCapabilities", vht_cap, DMT_TYPE[DMT_BASE64], 0);
			break;
		}
	}

	return 0;
}

static event_args wifidataelementsdisassociationevent_disassociated_args = {
    .name = "wifi.dataelements.Disassociated",
    .param = (const char *[]) {
        "BSSID",
        "MACAddress",
        "ReasonCode",
        "BytesSent",
        "BytesReceived",
        "PacketsSent",
        "PacketsReceived",
        "ErrorsSent",
        "ErrorsReceived",
        "RetransCount",
        "TimeStamp",
        "LastDataDownlinkRate",
        "LastDataUplinkRate",
        "UtilizationReceive",
        "UtilizationTransmit",
        "EstMACDataRateDownlink",
        "EstMACDataRateUplink",
        "SignalStrength",
        "LastConnectTime",
        "Noise",
        NULL
    }
};

static int get_event_args_WiFiDataElementsDisassociationEvent_Disassociated(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
    *value = (char *)&wifidataelementsdisassociationevent_disassociated_args;
    return 0;
}

static int event_WiFiDataElementsDisassociationEvent_Disassociated(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case EVENT_CHECK:
			// Nothing to check
			break;
		case EVENT_RUN:
		{
			char *tmp = NULL;
			json_object *jobj = NULL;

			tmp = dmjson_get_value((json_object *)value, 1, "eventTime");
			fill_blob_param(&ctx->bb, "TimeStamp", tmp, DMT_TYPE[DMT_STRING], 0);

			jobj = dmjson_get_obj((json_object *)value, 1, "wfa-dataelements:DisassociationEvent.Disassociated");

			tmp = dmjson_get_value(jobj, 1, "BSSID");
			fill_blob_param(&ctx->bb, "BSSID", tmp, DMT_TYPE[DMT_STRING], 0);

			tmp = dmjson_get_value(jobj, 1, "MACAddress");
			fill_blob_param(&ctx->bb, "MACAddress", tmp, DMT_TYPE[DMT_STRING], 0);

			tmp = dmjson_get_value(jobj, 1, "ReasonCode");
			fill_blob_param(&ctx->bb, "ReasonCode", tmp, DMT_TYPE[DMT_UNINT], 0);

			tmp = dmjson_get_value(jobj, 1, "BytesSent");
			fill_blob_param(&ctx->bb, "BytesSent", tmp, DMT_TYPE[DMT_UNLONG], 0);

			tmp = dmjson_get_value(jobj, 1, "BytesReceived");
			fill_blob_param(&ctx->bb, "BytesReceived", tmp, DMT_TYPE[DMT_UNLONG], 0);

			tmp = dmjson_get_value(jobj, 1, "PacketsSent");
			fill_blob_param(&ctx->bb, "PacketsSent", tmp, DMT_TYPE[DMT_UNLONG], 0);

			tmp = dmjson_get_value(jobj, 1, "PacketsReceived");
			fill_blob_param(&ctx->bb, "PacketsReceived", tmp, DMT_TYPE[DMT_UNLONG], 0);

			tmp = dmjson_get_value(jobj, 1, "ErrorsSent");
			fill_blob_param(&ctx->bb, "ErrorsSent", tmp, DMT_TYPE[DMT_UNINT], 0);

			tmp = dmjson_get_value(jobj, 1, "ErrorsReceived");
			fill_blob_param(&ctx->bb, "ErrorsReceived", tmp, DMT_TYPE[DMT_UNINT], 0);

			tmp = dmjson_get_value(jobj, 1, "RetransCount");
			fill_blob_param(&ctx->bb, "RetransCount", tmp, DMT_TYPE[DMT_UNINT], 0);

			tmp = dmjson_get_value(jobj, 1, "LastDataDownlinkRate");
			fill_blob_param(&ctx->bb, "LastDataDownlinkRate", tmp, DMT_TYPE[DMT_UNINT], 0);

			tmp = dmjson_get_value(jobj, 1, "LastDataUplinkRate");
			fill_blob_param(&ctx->bb, "LastDataUplinkRate", tmp, DMT_TYPE[DMT_UNINT], 0);

			tmp = dmjson_get_value(jobj, 1, "UtilizationReceive");
			fill_blob_param(&ctx->bb, "UtilizationReceive", tmp, DMT_TYPE[DMT_UNLONG], 0);

			tmp = dmjson_get_value(jobj, 1, "UtilizationTransmit");
			fill_blob_param(&ctx->bb, "UtilizationTransmit", tmp, DMT_TYPE[DMT_UNLONG], 0);

			tmp = dmjson_get_value(jobj, 1, "EstMACDataRateDownlink");
			fill_blob_param(&ctx->bb, "EstMACDataRateDownlink", tmp, DMT_TYPE[DMT_UNINT], 0);

			tmp = dmjson_get_value(jobj, 1, "EstMACDataRateUplink");
			fill_blob_param(&ctx->bb, "EstMACDataRateUplink", tmp, DMT_TYPE[DMT_UNINT], 0);

			tmp = dmjson_get_value(jobj, 1, "SignalStrength");
			fill_blob_param(&ctx->bb, "SignalStrength", tmp, DMT_TYPE[DMT_UNINT], 0);

			tmp = dmjson_get_value(jobj, 1, "LastConnectTime");
			fill_blob_param(&ctx->bb, "LastConnectTime", tmp, DMT_TYPE[DMT_UNINT], 0);

			tmp = dmjson_get_value(jobj, 1, "Noise");
			fill_blob_param(&ctx->bb, "Noise", tmp, DMT_TYPE[DMT_UNINT], 0);

			break;
		}
	}

    return 0;
}

/**********************************************************************************************************************************
*                                            OBJ & LEAF DEFINITION
***********************************************************************************************************************************/
/* *** Device.WiFi. *** */
DMOBJ tDeviceWiFiDataElementsObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"DataElements", &DMREAD, NULL, NULL, "file:/etc/init.d/decollector", NULL, NULL, NULL, tWiFiDataElementsObj, NULL, NULL, BBFDM_BOTH, NULL},
{0}
};

/* *** Device.WiFi.DataElements. *** */
DMOBJ tWiFiDataElementsObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Network", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkObj, tWiFiDataElementsNetworkParams, NULL, BBFDM_BOTH},
{"AssociationEvent", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsAssociationEventParams, NULL, BBFDM_BOTH, NULL},
{"DisassociationEvent", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsDisassociationEventParams, NULL, BBFDM_BOTH, NULL},
//{"FailedConnectionEvent", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsFailedConnectionEventParams, NULL, BBFDM_BOTH, NULL},
{0}
};

/* *** Device.WiFi.DataElements.Network. *** */
DMOBJ tWiFiDataElementsNetworkObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"SSID", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkSSIDInst, NULL, NULL, NULL, tWiFiDataElementsNetworkSSIDParams, NULL, BBFDM_BOTH, NULL},
{"ProvisionedDPP", &DMWRITE, addObjWiFiDataElementsNetworkProvisionedDPP, delObjWiFiDataElementsNetworkProvisionedDPP, NULL, browseWiFiDataElementsNetworkProvisionedDPPInst, NULL, NULL, NULL, tWiFiDataElementsNetworkProvisionedDPPParams, NULL, BBFDM_BOTH, NULL},
{"MultiAPSteeringSummaryStats", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkMultiAPSteeringSummaryStatsParams, NULL, BBFDM_BOTH, NULL},
{"Device", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceInst, NULL, NULL, tWiFiDataElementsNetworkDeviceObj, tWiFiDataElementsNetworkDeviceParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"ID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetwork_ID, NULL, BBFDM_BOTH},
{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetwork_TimeStamp, NULL, BBFDM_BOTH},
{"ControllerID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetwork_ControllerID, NULL, BBFDM_BOTH},
{"DeviceNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetwork_DeviceNumberOfEntries, NULL, BBFDM_BOTH},
{"MSCSDisallowedStaList", &DMREAD, DMT_STRING, get_WiFiDataElementsNetwork_MSCSDisallowedStaList, NULL, BBFDM_BOTH},
{"SCSDisallowedStaList", &DMREAD, DMT_STRING, get_WiFiDataElementsNetwork_SCSDisallowedStaList, NULL, BBFDM_BOTH},
{"ColocatedAgentID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetwork_ColocatedAgentID, NULL, BBFDM_BOTH},
{"SSIDNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetwork_SSIDNumberOfEntries, NULL, BBFDM_BOTH},
{"ProvisionedDPPNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetwork_ProvisionedDPPNumberOfEntries, NULL, BBFDM_BOTH},
{"SetTrafficSeparation()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetwork_SetTrafficSeparation, operate_WiFiDataElementsNetwork_SetTrafficSeparation, BBFDM_USP},
//{"SetServicePrioritization()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetwork_SetServicePrioritization, operate_WiFiDataElementsNetwork_SetServicePrioritization, BBFDM_USP},
//{"SetPreferredBackhauls()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetwork_SetPreferredBackhauls, operate_WiFiDataElementsNetwork_SetPreferredBackhauls, BBFDM_USP},
{"SetSSID()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetwork_SetSSID, operate_WiFiDataElementsNetwork_SetSSID, BBFDM_USP},
//{"SetMSCSDisallowed()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetwork_SetMSCSDisallowed, operate_WiFiDataElementsNetwork_SetMSCSDisallowed, BBFDM_USP},
//{"SetSCSDisallowed()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetwork_SetSCSDisallowed, operate_WiFiDataElementsNetwork_SetSCSDisallowed, BBFDM_USP},
{0}
};

/* *** Device.WiFi.DataElements.Network.SSID.{i}. *** */
DMLEAF tWiFiDataElementsNetworkSSIDParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"SSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkSSID_SSID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Band", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkSSID_Band, NULL, BBFDM_BOTH},
{"Enable", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkSSID_Enable, NULL, BBFDM_BOTH},
{"AKMsAllowed", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkSSID_AKMs, NULL, BBFDM_BOTH},
//{"AdvertisementEnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkSSID_AdvEnabled, NULL, BBFDM_BOTH},
//{"MFPConfig", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkSSID_MFP, NULL, BBFDM_BOTH},
{"HaulType", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkSSID_Haul, NULL, BBFDM_BOTH},
{"Type", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkSSID_Type, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.ProvisionedDPP.{i}. *** */
DMLEAF tWiFiDataElementsNetworkProvisionedDPPParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"Alias", &DMWRITE, DMT_STRING, get_WiFiDataElementsNetworkProvisionedDPP_Alias, set_WiFiDataElementsNetworkProvisionedDPP_Alias, BBFDM_BOTH},
{"DPPURI", &DMWRITE, DMT_STRING, get_WiFiDataElementsNetworkProvisionedDPP_DPPURI, set_WiFiDataElementsNetworkProvisionedDPP_DPPURI, BBFDM_BOTH},
{BBF_VENDOR_PREFIX"Passphrase", &DMWRITE, DMT_STRING, get_WiFiDataElementsNetworkProvisionedDPP_Passphrase, set_WiFiDataElementsNetworkProvisionedDPP_Passphrase, BBFDM_BOTH, DM_FLAG_SECURE},
{0}
};

/* *** Device.WiFi.DataElements.Network.MultiAPSteeringSummaryStats. *** */
DMLEAF tWiFiDataElementsNetworkMultiAPSteeringSummaryStatsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"NoCandidateAPFailures", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_NoCandidateAPFailures, NULL, BBFDM_BOTH},
{"BlacklistAttempts", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BlacklistAttempts, NULL, BBFDM_BOTH},
{"BlacklistSuccesses", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BlacklistSuccesses, NULL, BBFDM_BOTH},
{"BlacklistFailures", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BlacklistFailures, NULL, BBFDM_BOTH},
{"BTMAttempts", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BTMAttempts, NULL, BBFDM_BOTH},
{"BTMSuccesses", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BTMSuccesses, NULL, BBFDM_BOTH},
{"BTMFailures", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BTMFailures, NULL, BBFDM_BOTH},
{"BTMQueryResponses", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BTMQueryResponses, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Default8021Q", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceDefault8021QInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceDefault8021QParams, NULL, BBFDM_BOTH, NULL},
{"SSIDtoVIDMapping", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceSSIDtoVIDMappingInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceSSIDtoVIDMappingParams, NULL, BBFDM_BOTH, NULL},
{"CACStatus", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceCACStatusInst, NULL, NULL, tWiFiDataElementsNetworkDeviceCACStatusObj, tWiFiDataElementsNetworkDeviceCACStatusParams, NULL, BBFDM_BOTH, NULL},
//{"SPRule", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceSPRuleInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceSPRuleParams, NULL, BBFDM_BOTH, NULL},
{"IEEE1905Security", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceIEEE1905SecurityInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceIEEE1905SecurityParams, NULL, BBFDM_BOTH, NULL},
//{"AnticipatedChannels", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceAnticipatedChannelsInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceAnticipatedChannelsParams, NULL, BBFDM_BOTH, NULL},
//{"AnticipatedChannelUsage", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceAnticipatedChannelUsageInst, NULL, NULL, tWiFiDataElementsNetworkDeviceAnticipatedChannelUsageObj, tWiFiDataElementsNetworkDeviceAnticipatedChannelUsageParams, NULL, BBFDM_BOTH, NULL},
{"MultiAPDevice", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceMultiAPDeviceObj, tWiFiDataElementsNetworkDeviceMultiAPDeviceParams, NULL, BBFDM_BOTH, NULL},
{"Radio", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioInst, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioObj, tWiFiDataElementsNetworkDeviceRadioParams, NULL, BBFDM_BOTH, NULL},
{"APMLD", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceAPMLDInst, NULL, NULL, tWiFiDataElementsNetworkDeviceAPMLDObj, tWiFiDataElementsNetworkDeviceAPMLDParams, NULL, BBFDM_BOTH, NULL},
{"bSTAMLD", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDevicebSTAMLDObj, tWiFiDataElementsNetworkDevicebSTAMLDParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"ID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_ID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE|DM_FLAG_LINKER},
{"MultiAPCapabilities", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDevice_MultiAPCapabilities, NULL, BBFDM_BOTH},
{"CollectionInterval", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_CollectionInterval, NULL, BBFDM_BOTH},
{"ReportUnsuccessfulAssociations", &DMWRITE, DMT_BOOL, get_WiFiDataElementsNetworkDevice_ReportUnsuccessfulAssociations, set_WiFiDataElementsNetworkDevice_ReportUnsuccessfulAssociations, BBFDM_BOTH},
{"MaxReportingRate", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_MaxReportingRate, NULL, BBFDM_BOTH},
{"APMetricsReportingInterval", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDevice_APMetricsReportingInterval, set_WiFiDataElementsNetworkDevice_APMetricsReportingInterval, BBFDM_BOTH},
{"Manufacturer", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_Manufacturer, NULL, BBFDM_BOTH},
{"SerialNumber", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_SerialNumber, NULL, BBFDM_BOTH},
// {"ManufacturerModel", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_ManufacturerModel, NULL, BBFDM_BOTH},
{"SoftwareVersion", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_SoftwareVersion, NULL, BBFDM_BOTH},
{"ExecutionEnv", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_ExecutionEnv, NULL, BBFDM_BOTH},
// {"DSCPMap", &DMREAD, DMT_HEXBIN, get_WiFiDataElementsNetworkDevice_DSCPMap, NULL, BBFDM_BOTH},
{"MaxPrioritizationRules", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_MaxPrioritizationRules, NULL, BBFDM_BOTH},
{"PrioritizationSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevice_PrioritizationSupport, NULL, BBFDM_BOTH},
{"MaxVIDs", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_MaxVIDs, NULL, BBFDM_BOTH},
// {"APMetricsWiFi6", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevice_APMetricsWiFi6, NULL, BBFDM_BOTH},
{"CountryCode", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_CountryCode, NULL, BBFDM_BOTH},
{"LocalSteeringDisallowedSTAList", &DMWRITE, DMT_STRING, get_WiFiDataElementsNetworkDevice_LocalSteeringDisallowedSTAList, set_WiFiDataElementsNetworkDevice_LocalSteeringDisallowedSTAList, BBFDM_BOTH},
{"BTMSteeringDisallowedSTAList", &DMWRITE, DMT_STRING, get_WiFiDataElementsNetworkDevice_BTMSteeringDisallowedSTAList, set_WiFiDataElementsNetworkDevice_BTMSteeringDisallowedSTAList, BBFDM_BOTH},
{"DFSEnable", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevice_DFSEnable, NULL, BBFDM_BOTH},
{"ReportIndependentScans", &DMWRITE, DMT_BOOL, get_WiFiDataElementsNetworkDevice_ReportIndependentScans, set_WiFiDataElementsNetworkDevice_ReportIndependentScans, BBFDM_BOTH},
// {"AssociatedSTAinAPMetricsWiFi6", &DMWRITE, DMT_BOOL, get_WiFiDataElementsNetworkDevice_AssociatedSTAinAPMetricsWiFi6, set_WiFiDataElementsNetworkDevice_AssociatedSTAinAPMetricsWiFi6, BBFDM_BOTH},
{"MaxUnsuccessfulAssociationReportingRate", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDevice_MaxUnsuccessfulAssociationReportingRate, set_WiFiDataElementsNetworkDevice_MaxUnsuccessfulAssociationReportingRate, BBFDM_BOTH},
{"STASteeringState", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevice_STASteeringState, NULL, BBFDM_BOTH},
{"CoordinatedCACAllowed", &DMWRITE, DMT_BOOL, get_WiFiDataElementsNetworkDevice_CoordinatedCACAllowed, set_WiFiDataElementsNetworkDevice_CoordinatedCACAllowed, BBFDM_BOTH},
{"TrafficSeparationAllowed", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevice_TrafficSeparationAllowed, NULL, BBFDM_BOTH},
// {"ServicePrioritizationAllowed", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevice_ServicePrioritizationAllowed, NULL, BBFDM_BOTH},
{"ControllerOperationMode", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_ControllerOperationMode, NULL, BBFDM_BOTH},
{"BackhaulMACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_BackhaulMACAddress, NULL, BBFDM_BOTH},
{"BackhaulALID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_BackhaulALID, NULL, BBFDM_BOTH},
{"BackhaulDownMACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_BackhaulDownMACAddress, NULL, BBFDM_BOTH},
{"BackhaulMediaType", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_BackhaulMediaType, NULL, BBFDM_BOTH},
{"BackhaulPHYRate", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_BackhaulPHYRate, NULL, BBFDM_BOTH},
{"TrafficSeparationCapability", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevice_TrafficSeparationCapability, NULL, BBFDM_BOTH},
{"EasyConnectCapability", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevice_EasyConnectCapability, NULL, BBFDM_BOTH},
{"RadioNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_RadioNumberOfEntries, NULL, BBFDM_BOTH},
{"Default8021QNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_Default8021QNumberOfEntries, NULL, BBFDM_BOTH},
{"SSIDtoVIDMappingNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_SSIDtoVIDMappingNumberOfEntries, NULL, BBFDM_BOTH},
{"CACStatusNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_CACStatusNumberOfEntries, NULL, BBFDM_BOTH},
{"IEEE1905SecurityNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_IEEE1905SecurityNumberOfEntries, NULL, BBFDM_BOTH},
//{"SPRuleNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_SPRuleNumberOfEntries, NULL, BBFDM_BOTH},
//{"AnticipatedChannelsNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_AnticipatedChannelsNumberOfEntries, NULL, BBFDM_BOTH},
//{"AnticipatedChannelUsageNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_AnticipatedChannelUsageNumberOfEntries, NULL, BBFDM_BOTH},
{"MaxNumMLDs", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_MaxNumMLDs, NULL, BBFDM_BOTH},
{"APMLDMaxLinks", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_APMLDMaxLinks, NULL, BBFDM_BOTH},
{"bSTAMLDMaxLinks", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_bSTAMLDMaxLinks, NULL, BBFDM_BOTH},
{"TIDLinkMapCapability", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_TIDLinkMapCapability, NULL, BBFDM_BOTH},
{"APMLDNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_APMLDNumberOfEntries, NULL, BBFDM_BOTH},
{"SetSTASteeringState()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDevice_SetSTASteeringState, operate_WiFiDataElementsNetworkDevice_SetSTASteeringState, BBFDM_USP},
{"InitiateWPSPBC()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDevice_InitiateWPSPBC, operate_WiFiDataElementsNetworkDevice_InitiateWPSPBC, BBFDM_USP},
//{"SetDFSState()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDevice_SetDFSState, operate_WiFiDataElementsNetworkDevice_SetDFSState, BBFDM_USP},
//{"SetAnticipatedChannelPreference()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDevice_SetAnticipatedChannelPreference, operate_WiFiDataElementsNetworkDevice_SetAnticipatedChannelPreference, BBFDM_USP},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Default8021Q.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceDefault8021QParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_WiFiDataElementsNetworkDeviceDefault8021Q_Enable, set_WiFiDataElementsNetworkDeviceDefault8021Q_Enable, BBFDM_BOTH},
{"PrimaryVID", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceDefault8021Q_PrimaryVID, set_WiFiDataElementsNetworkDeviceDefault8021Q_PrimaryVID, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"DefaultPCP", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceDefault8021Q_DefaultPCP, set_WiFiDataElementsNetworkDeviceDefault8021Q_DefaultPCP, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.SSIDtoVIDMapping.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceSSIDtoVIDMappingParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"SSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceSSIDtoVIDMapping_SSID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"VID", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceSSIDtoVIDMapping_VID, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.CACStatus.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceCACStatusObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"CACAvailableChannel", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceCACStatusCACAvailableChannelInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceCACStatusCACAvailableChannelParams, NULL, BBFDM_BOTH, NULL},
{"CACNonOccupancyChannel", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannelInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannelParams, NULL, BBFDM_BOTH, NULL},
{"CACActiveChannel", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceCACStatusCACActiveChannelInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceCACStatusCACActiveChannelParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceCACStatusParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceCACStatus_TimeStamp, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"CACAvailableChannelNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatus_CACAvailableChannelNumberOfEntries, NULL, BBFDM_BOTH},
{"CACNonOccupancyChannelNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatus_CACNonOccupancyChannelNumberOfEntries, NULL, BBFDM_BOTH},
{"CACActiveChannelNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatus_CACActiveChannelNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.CACStatus.{i}.CACAvailableChannel.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceCACStatusCACAvailableChannelParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"OpClass", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACAvailableChannel_OpClass, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Channel", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACAvailableChannel_Channel, NULL, BBFDM_BOTH},
{"Minutes", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACAvailableChannel_Minutes, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.CACStatus.{i}.CACNonOccupancyChannel.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannelParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"OpClass", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannel_OpClass, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Channel", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannel_Channel, NULL, BBFDM_BOTH},
{"Seconds", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannel_Seconds, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.CACStatus.{i}.CACActiveChannel.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceCACStatusCACActiveChannelParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"OpClass", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACActiveChannel_OpClass, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Channel", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACActiveChannel_Channel, NULL, BBFDM_BOTH},
{"Countdown", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACActiveChannel_Countdown, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.SPRule.{i}. *** */
//DMLEAF tWiFiDataElementsNetworkDeviceSPRuleParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
//{"ID", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceSPRule_ID, NULL, BBFDM_BOTH},
//{"Precedence", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceSPRule_Precedence, NULL, BBFDM_BOTH},
//{"Output", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceSPRule_Output, NULL, BBFDM_BOTH},
//{"AlwaysMatch", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceSPRule_AlwaysMatch, NULL, BBFDM_BOTH},
//{0}
//};

/* *** Device.WiFi.DataElements.Network.Device.{i}.IEEE1905Security.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceIEEE1905SecurityParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"OnboardingProtocol", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceIEEE1905Security_OnboardingProtocol, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"IntegrityAlgorithm", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceIEEE1905Security_IntegrityAlgorithm, NULL, BBFDM_BOTH},
{"EncryptionAlgorithm", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceIEEE1905Security_EncryptionAlgorithm, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.AnticipatedChannels.{i}. *** */
//DMLEAF tWiFiDataElementsNetworkDeviceAnticipatedChannelsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
//{"OpClass", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAnticipatedChannels_OpClass, NULL, BBFDM_BOTH},
//{"ChannelList", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAnticipatedChannels_ChannelList, NULL, BBFDM_BOTH},
//{0}
//};

/* *** Device.WiFi.DataElements.Network.Device.{i}.AnticipatedChannelUsage.{i}. *** */
//DMOBJ tWiFiDataElementsNetworkDeviceAnticipatedChannelUsageObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
//{"Entry", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceAnticipatedChannelUsageEntryInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceAnticipatedChannelUsageEntryParams, NULL, BBFDM_BOTH, NULL},
//{0}
//};

//DMLEAF tWiFiDataElementsNetworkDeviceAnticipatedChannelUsageParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
//{"OpClass", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAnticipatedChannelUsage_OpClass, NULL, BBFDM_BOTH},
//{"Channel", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAnticipatedChannelUsage_Channel, NULL, BBFDM_BOTH},
//{"ReferenceBSSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAnticipatedChannelUsage_ReferenceBSSID, NULL, BBFDM_BOTH},
//{"EntryNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAnticipatedChannelUsage_EntryNumberOfEntries, NULL, BBFDM_BOTH},
//{0}
//};

/* *** Device.WiFi.DataElements.Network.Device.{i}.AnticipatedChannelUsage.{i}.Entry.{i}. *** */
//DMLEAF tWiFiDataElementsNetworkDeviceAnticipatedChannelUsageEntryParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
//{"BurstStartTime", &DMREAD, DMT_HEXBIN, get_WiFiDataElementsNetworkDeviceAnticipatedChannelUsageEntry_BurstStartTime, NULL, BBFDM_BOTH},
//{"BurstLength", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAnticipatedChannelUsageEntry_BurstLength, NULL, BBFDM_BOTH},
//{"Repetitions", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAnticipatedChannelUsageEntry_Repetitions, NULL, BBFDM_BOTH},
//{"BurstInterval", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAnticipatedChannelUsageEntry_BurstInterval, NULL, BBFDM_BOTH},
//{"RUBitmask", &DMREAD, DMT_HEXBIN, get_WiFiDataElementsNetworkDeviceAnticipatedChannelUsageEntry_RUBitmask, NULL, BBFDM_BOTH},
//{"TransmitterIdentifier", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAnticipatedChannelUsageEntry_TransmitterIdentifier, NULL, BBFDM_BOTH},
//{"PowerLevel", &DMREAD, DMT_INT, get_WiFiDataElementsNetworkDeviceAnticipatedChannelUsageEntry_PowerLevel, NULL, BBFDM_BOTH},
//{"ChannelUsageReason", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAnticipatedChannelUsageEntry_ChannelUsageReason, NULL, BBFDM_BOTH},
//{0}
//};

/* *** Device.WiFi.DataElements.Network.Device.{i}.MultiAPDevice. *** */
DMOBJ tWiFiDataElementsNetworkDeviceMultiAPDeviceObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Backhaul", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulObj, tWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceMultiAPDeviceParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"LastContactTime", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceMultiAPDevice_LastContactTime, NULL, BBFDM_BOTH},
{"AssocIEEE1905DeviceRef", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceMultiAPDevice_AssocIEEE1905DeviceRef, NULL, BBFDM_BOTH, DM_FLAG_REFERENCE},
{"EasyMeshAgentOperationMode", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceMultiAPDevice_EasyMeshAgentOperationMode, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.MultiAPDevice.Backhaul. *** */
DMOBJ tWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
//{"CurrentOperatingClassProfile", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfileInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfileParams, NULL, BBFDM_BOTH, NULL},
{"Stats", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStatsParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
//{"CurrentOperatingClassProfileNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaul_CurrentOperatingClassProfileNumberOfEntries, NULL, BBFDM_BOTH},
{"SteerWiFiBackhaul()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaul_SteerWiFiBackhaul, operate_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaul_SteerWiFiBackhaul, BBFDM_USP},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.MultiAPDevice.Backhaul.CurrentOperatingClassProfile.{i}. *** */
//DMLEAF tWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfileParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
//{"Class", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfile_Class, NULL, BBFDM_BOTH},
//{"Channel", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfile_Channel, NULL, BBFDM_BOTH},
//{"TxPower", &DMREAD, DMT_INT, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfile_TxPower, NULL, BBFDM_BOTH},
//{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfile_TimeStamp, NULL, BBFDM_BOTH},
//{0}
//};

/* *** Device.WiFi.DataElements.Network.Device.{i}.MultiAPDevice.Backhaul.Stats. *** */
DMLEAF tWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStatsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
//{"BytesSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_BytesSent, NULL, BBFDM_BOTH},
//{"BytesReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_BytesReceived, NULL, BBFDM_BOTH},
{"PacketsSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_PacketsSent, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_PacketsReceived, NULL, BBFDM_BOTH},
{"ErrorsSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_ErrorsSent, NULL, BBFDM_BOTH},
{"ErrorsReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_ErrorsReceived, NULL, BBFDM_BOTH},
//{"LinkUtilization", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_LinkUtilization, NULL, BBFDM_BOTH},
{"SignalStrength", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_SignalStrength, NULL, BBFDM_BOTH},
//{"LastDataDownlinkRate", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_LastDataDownlinkRate, NULL, BBFDM_BOTH},
{"LastDataUplinkRate", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_LastDataUplinkRate, NULL, BBFDM_BOTH},
{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_TimeStamp, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"ScanResult", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioScanResultInst, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioScanResultObj, tWiFiDataElementsNetworkDeviceRadioScanResultParams, NULL, BBFDM_BOTH, NULL},
{"BackhaulSta", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBackhaulStaParams, NULL, BBFDM_BOTH, NULL},
{"ScanCapability", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioScanCapabilityObj, tWiFiDataElementsNetworkDeviceRadioScanCapabilityParams, NULL, BBFDM_BOTH, NULL},
{"CACCapability", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCACCapabilityObj, tWiFiDataElementsNetworkDeviceRadioCACCapabilityParams, NULL, BBFDM_BOTH, NULL},
{"Capabilities", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesObj, tWiFiDataElementsNetworkDeviceRadioCapabilitiesParams, NULL, BBFDM_BOTH, NULL},
{"CurrentOperatingClassProfile", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfileInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfileParams, NULL, BBFDM_BOTH, NULL},
//{"DisAllowedOpClassChannels", &DMWRITE, addObjWiFiDataElementsNetworkDeviceRadioDisAllowedOpClassChannels, delObjWiFiDataElementsNetworkDeviceRadioDisAllowedOpClassChannels, NULL, browseWiFiDataElementsNetworkDeviceRadioDisAllowedOpClassChannelsInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioDisAllowedOpClassChannelsParams, NULL, BBFDM_BOTH, NULL},
//{"SpatialReuse", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioSpatialReuseParams, NULL, BBFDM_BOTH, NULL},
{"BSS", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioBSSInst, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSObj, tWiFiDataElementsNetworkDeviceRadioBSSParams, NULL, BBFDM_BOTH, NULL},
{"UnassociatedSTA", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioUnassociatedSTAInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioUnassociatedSTAParams, NULL, BBFDM_BOTH, NULL},
//{"MultiAPRadio", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioMultiAPRadioParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"ID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadio_ID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Enabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadio_Enabled, NULL, BBFDM_BOTH},
{"Noise", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_Noise, NULL, BBFDM_BOTH},
{"Utilization", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_Utilization, NULL, BBFDM_BOTH},
{"Transmit", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_Transmit, NULL, BBFDM_BOTH},
{"ReceiveSelf", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_ReceiveSelf, NULL, BBFDM_BOTH},
{"ReceiveOther", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_ReceiveOther, NULL, BBFDM_BOTH},
{"TrafficSeparationCombinedFronthaul", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadio_TrafficSeparationCombinedFronthaul, NULL, BBFDM_BOTH},
{"TrafficSeparationCombinedBackhaul", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadio_TrafficSeparationCombinedBackhaul, NULL, BBFDM_BOTH},
{"SteeringPolicy", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_SteeringPolicy, set_WiFiDataElementsNetworkDeviceRadio_SteeringPolicy, BBFDM_BOTH},
{"ChannelUtilizationThreshold", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_ChannelUtilizationThreshold, set_WiFiDataElementsNetworkDeviceRadio_ChannelUtilizationThreshold, BBFDM_BOTH},
{"RCPISteeringThreshold", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_RCPISteeringThreshold, set_WiFiDataElementsNetworkDeviceRadio_RCPISteeringThreshold, BBFDM_BOTH},
{"STAReportingRCPIThreshold", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_STAReportingRCPIThreshold, set_WiFiDataElementsNetworkDeviceRadio_STAReportingRCPIThreshold, BBFDM_BOTH},
{"STAReportingRCPIHysteresisMarginOverride", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_STAReportingRCPIHysteresisMarginOverride, set_WiFiDataElementsNetworkDeviceRadio_STAReportingRCPIHysteresisMarginOverride, BBFDM_BOTH},
{"ChannelUtilizationReportingThreshold", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_ChannelUtilizationReportingThreshold, set_WiFiDataElementsNetworkDeviceRadio_ChannelUtilizationReportingThreshold, BBFDM_BOTH},
{"AssociatedSTATrafficStatsInclusionPolicy", &DMWRITE, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadio_AssociatedSTATrafficStatsInclusionPolicy, set_WiFiDataElementsNetworkDeviceRadio_AssociatedSTATrafficStatsInclusionPolicy, BBFDM_BOTH},
{"AssociatedSTALinkMetricsInclusionPolicy", &DMWRITE, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadio_AssociatedSTALinkMetricsInclusionPolicy, set_WiFiDataElementsNetworkDeviceRadio_AssociatedSTALinkMetricsInclusionPolicy, BBFDM_BOTH},
{"ChipsetVendor", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadio_ChipsetVendor, NULL, BBFDM_BOTH},
{"APMetricsWiFi6", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadio_APMetricsWiFi6, NULL, BBFDM_BOTH},
{"CurrentOperatingClassProfileNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_CurrentOperatingClassProfileNumberOfEntries, NULL, BBFDM_BOTH},
{"UnassociatedSTANumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_UnassociatedSTANumberOfEntries, NULL, BBFDM_BOTH},
{"BSSNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_BSSNumberOfEntries, NULL, BBFDM_BOTH},
{"ScanResultNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_ScanResultNumberOfEntries, NULL, BBFDM_BOTH},
//{"DisAllowedOpClassChannelsNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_DisAllowedOpClassChannelsNumberOfEntries, NULL, BBFDM_BOTH},
{"ChannelScanRequest()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDeviceRadio_ChannelScanRequest, operate_WiFiDataElementsNetworkDeviceRadio_ChannelScanRequest, BBFDM_USP},
//{"RadioEnable()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDeviceRadio_RadioEnable, operate_WiFiDataElementsNetworkDeviceRadio_RadioEnable, BBFDM_USP},
//{"SetTxPowerLimit()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDeviceRadio_SetTxPowerLimit, operate_WiFiDataElementsNetworkDeviceRadio_SetTxPowerLimit, BBFDM_USP},
//{"SetSpatialReuse()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDeviceRadio_SetSpatialReuse, operate_WiFiDataElementsNetworkDeviceRadio_SetSpatialReuse, BBFDM_USP},
//{"WiFiRestart()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDeviceRadio_WiFiRestart, operate_WiFiDataElementsNetworkDeviceRadio_WiFiRestart, BBFDM_USP},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioScanResultObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"OpClassScan", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanInst, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanObj, tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanParams, NULL, BBFDM_BOTH},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioScanResultParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceRadioScanResult_TimeStamp, NULL, BBFDM_BOTH},
{"OpClassScanNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResult_OpClassScanNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"ChannelScan", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanInst, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanObj, tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"OperatingClass", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScan_OperatingClass, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"ChannelScanNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScan_ChannelScanNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}.ChannelScan.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"NeighborBSS", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSSInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSSParams, NULL, BBFDM_BOTH},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Channel", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_Channel, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_TimeStamp, NULL, BBFDM_BOTH},
{"Utilization", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_Utilization, NULL, BBFDM_BOTH},
{"Noise", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_Noise, NULL, BBFDM_BOTH},
{"ScanStatus", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_ScanStatus, NULL, BBFDM_BOTH},
{"NeighborBSSNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_NeighborBSSNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}.ChannelScan.{i}.NeighborBSS.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSSParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"BSSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_BSSID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"SSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_SSID, NULL, BBFDM_BOTH},
{"SignalStrength", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_SignalStrength, NULL, BBFDM_BOTH},
{"ChannelBandwidth", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_ChannelBandwidth, NULL, BBFDM_BOTH},
{"ChannelUtilization", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_ChannelUtilization, NULL, BBFDM_BOTH},
{"StationCount", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_StationCount, NULL, BBFDM_BOTH},
{"MLDMACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_MLDMACAddress, NULL, BBFDM_BOTH},
{"ReportingBSSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_ReportingBSSID, NULL, BBFDM_BOTH},
{"MultiBSSID", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_MultiBSSID, NULL, BBFDM_BOTH},
{"BSSLoadElementPresent", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_BSSLoadElementPresent, NULL, BBFDM_BOTH},
{"BSSColor", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_BSSColor, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BackhaulSta. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioBackhaulStaParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"MACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBackhaulSta_MACAddress, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanCapability. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioScanCapabilityObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"OpClassChannels", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioScanCapabilityOpClassChannelsInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioScanCapabilityOpClassChannelsParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioScanCapabilityParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"OnBootOnly", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioScanCapability_OnBootOnly, NULL, BBFDM_BOTH},
{"Impact", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanCapability_Impact, NULL, BBFDM_BOTH},
{"MinimumInterval", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanCapability_MinimumInterval, NULL, BBFDM_BOTH},
{"OpClassChannelsNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanCapability_OpClassChannelsNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanCapability.OpClassChannels.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioScanCapabilityOpClassChannelsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"OpClass", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanCapabilityOpClassChannels_OpClass, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"ChannelList", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioScanCapabilityOpClassChannels_ChannelList, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.CACCapability. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioCACCapabilityObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"CACMethod", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodInst, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodObj, tWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioCACCapabilityParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"CACMethodNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCACCapability_CACMethodNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.CACCapability.CACMethod.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"OpClassChannels", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodOpClassChannelsInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodOpClassChannelsParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Method", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethod_Method, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"NumberOfSeconds", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethod_NumberOfSeconds, NULL, BBFDM_BOTH},
{"OpClassChannelsNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethod_OpClassChannelsNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.CACCapability.CACMethod.{i}.OpClassChannels.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodOpClassChannelsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"OpClass", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodOpClassChannels_OpClass, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"ChannelList", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodOpClassChannels_ChannelList, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioCapabilitiesObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"WiFi6APRole", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRoleParams, NULL, BBFDM_BOTH, NULL},
{"WiFi6bSTARole", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARoleParams, NULL, BBFDM_BOTH, NULL},
{"WiFi7APRole", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleObj, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleParams, NULL, BBFDM_BOTH, NULL},
{"WiFi7bSTARole", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleObj, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleParams, NULL, BBFDM_BOTH, NULL},
//{"AKMFrontHaul", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesAKMFrontHaulInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesAKMFrontHaulParams, NULL, BBFDM_BOTH, NULL},
//{"AKMBackhaul", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesAKMBackhaulInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesAKMBackhaulParams, NULL, BBFDM_BOTH, NULL},
{"CapableOperatingClassProfile", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfileInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfileParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"HTCapabilities", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilities_HTCapabilities, NULL, BBFDM_BOTH},
{"VHTCapabilities", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilities_VHTCapabilities, NULL, BBFDM_BOTH},
{"CapableOperatingClassProfileNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilities_CapableOperatingClassProfileNumberOfEntries, NULL, BBFDM_BOTH},
//{"AKMFrontHaulNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilities_AKMFrontHaulNumberOfEntries, NULL, BBFDM_BOTH},
//{"AKMBackhaulNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilities_AKMBackhaulNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi6APRole. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRoleParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"HE160", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_HE160, NULL, BBFDM_BOTH},
{"HE8080", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_HE8080, NULL, BBFDM_BOTH},
{"MCSNSS", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MCSNSS, NULL, BBFDM_BOTH},
{"SUBeamformer", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_SUBeamformer, NULL, BBFDM_BOTH},
{"SUBeamformee", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_SUBeamformee, NULL, BBFDM_BOTH},
{"MUBeamformer", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MUBeamformer, NULL, BBFDM_BOTH},
{"Beamformee80orLess", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_Beamformee80orLess, NULL, BBFDM_BOTH},
{"BeamformeeAbove80", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_BeamformeeAbove80, NULL, BBFDM_BOTH},
{"ULMUMIMO", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_ULMUMIMO, NULL, BBFDM_BOTH},
{"ULOFDMA", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_ULOFDMA, NULL, BBFDM_BOTH},
{"MaxDLMUMIMO", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MaxDLMUMIMO, NULL, BBFDM_BOTH},
{"MaxULMUMIMO", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MaxULMUMIMO, NULL, BBFDM_BOTH},
{"MaxDLOFDMA", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MaxDLOFDMA, NULL, BBFDM_BOTH},
{"MaxULOFDMA", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MaxULOFDMA, NULL, BBFDM_BOTH},
{"RTS", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_RTS, NULL, BBFDM_BOTH},
{"MURTS", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MURTS, NULL, BBFDM_BOTH},
{"MultiBSSID", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MultiBSSID, NULL, BBFDM_BOTH},
{"MUEDCA", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MUEDCA, NULL, BBFDM_BOTH},
{"TWTRequestor", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_TWTRequestor, NULL, BBFDM_BOTH},
{"TWTResponder", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_TWTResponder, NULL, BBFDM_BOTH},
{"SpatialReuse", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_SpatialReuse, NULL, BBFDM_BOTH},
{"AnticipatedChannelUsage", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_AnticipatedChannelUsage, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi6bSTARole. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARoleParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"HE160", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_HE160, NULL, BBFDM_BOTH},
{"HE8080", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_HE8080, NULL, BBFDM_BOTH},
{"MCSNSS", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MCSNSS, NULL, BBFDM_BOTH},
{"SUBeamformer", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_SUBeamformer, NULL, BBFDM_BOTH},
{"SUBeamformee", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_SUBeamformee, NULL, BBFDM_BOTH},
{"MUBeamformer", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MUBeamformer, NULL, BBFDM_BOTH},
{"Beamformee80orLess", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_Beamformee80orLess, NULL, BBFDM_BOTH},
{"BeamformeeAbove80", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_BeamformeeAbove80, NULL, BBFDM_BOTH},
{"ULMUMIMO", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_ULMUMIMO, NULL, BBFDM_BOTH},
{"ULOFDMA", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_ULOFDMA, NULL, BBFDM_BOTH},
{"MaxDLMUMIMO", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MaxDLMUMIMO, NULL, BBFDM_BOTH},
{"MaxULMUMIMO", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MaxULMUMIMO, NULL, BBFDM_BOTH},
{"MaxDLOFDMA", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MaxDLOFDMA, NULL, BBFDM_BOTH},
{"MaxULOFDMA", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MaxULOFDMA, NULL, BBFDM_BOTH},
{"RTS", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_RTS, NULL, BBFDM_BOTH},
{"MURTS", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MURTS, NULL, BBFDM_BOTH},
{"MultiBSSID", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MultiBSSID, NULL, BBFDM_BOTH},
{"MUEDCA", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MUEDCA, NULL, BBFDM_BOTH},
{"TWTRequestor", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_TWTRequestor, NULL, BBFDM_BOTH},
{"TWTResponder", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_TWTResponder, NULL, BBFDM_BOTH},
{"SpatialReuse", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_SpatialReuse, NULL, BBFDM_BOTH},
{"AnticipatedChannelUsage", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_AnticipatedChannelUsage, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7APRole. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys */
{"EMLMRFreqSeparation", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLMRFreqSeparationInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLMRFreqSeparationParams, NULL, BBFDM_BOTH, NULL},
{"EMLSRFreqSeparation", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLSRFreqSeparationInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLSRFreqSeparationParams, NULL, BBFDM_BOTH, NULL},
{"STRFreqSeparation", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleSTRFreqSeparationInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleSTRFreqSeparationParams, NULL, BBFDM_BOTH, NULL},
//{"NSTRFreqSeparation", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleNSTRFreqSeparationInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleNSTRFreqSeparationParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"EMLMRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_EMLMRSupport, NULL, BBFDM_BOTH},
{"EMLSRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_EMLSRSupport, NULL, BBFDM_BOTH},
{"STRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_STRSupport, NULL, BBFDM_BOTH},
{"NSTRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_NSTRSupport, NULL, BBFDM_BOTH},
{"TIDLinkMapNegotiation", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_TIDLinkMapNegotiation, NULL, BBFDM_BOTH},
{"EMLMRFreqSeparationNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_EMLMRFreqSeparationNumberOfEntries, NULL, BBFDM_BOTH},
{"EMLSRFreqSeparationNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_EMLSRFreqSeparationNumberOfEntries, NULL, BBFDM_BOTH},
{"STRFreqSeparationNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_STRFreqSeparationNumberOfEntries, NULL, BBFDM_BOTH},
//{"NSTRFreqSeparationNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_NSTRFreqSeparationNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7APRole.EMLMRFreqSeparation.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLMRFreqSeparationParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLMRFreqSeparation_RUID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"FreqSeparation", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLMRFreqSeparation_FreqSeparation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7APRole.EMLSRFreqSeparation.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLSRFreqSeparationParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLSRFreqSeparation_RUID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"FreqSeparation", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLSRFreqSeparation_FreqSeparation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7APRole.STRFreqSeparation.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleSTRFreqSeparationParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleSTRFreqSeparation_RUID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"FreqSeparation", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleSTRFreqSeparation_FreqSeparation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7APRole.NSTRFreqSeparation.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleNSTRFreqSeparationParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
//{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleNSTRFreqSeparation_RUID, NULL, BBFDM_BOTH},
//{"FreqSeparation", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleNSTRFreqSeparation_FreqSeparation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7bSTARole. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys */
{"EMLMRFreqSeparation", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLMRFreqSeparationInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLMRFreqSeparationParams, NULL, BBFDM_BOTH, NULL},
{"EMLSRFreqSeparation", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLSRFreqSeparationInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLSRFreqSeparationParams, NULL, BBFDM_BOTH, NULL},
{"STRFreqSeparation", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleSTRFreqSeparationInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleSTRFreqSeparationParams, NULL, BBFDM_BOTH, NULL},
//{"NSTRFreqSeparation", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleNSTRFreqSeparationInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleNSTRFreqSeparationParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"EMLMRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_EMLMRSupport, NULL, BBFDM_BOTH},
{"EMLSRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_EMLSRSupport, NULL, BBFDM_BOTH},
{"STRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_STRSupport, NULL, BBFDM_BOTH},
{"NSTRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_NSTRSupport, NULL, BBFDM_BOTH},
{"TIDLinkMapNegotiation", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_TIDLinkMapNegotiation, NULL, BBFDM_BOTH},
{"EMLMRFreqSeparationNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_EMLMRFreqSeparationNumberOfEntries, NULL, BBFDM_BOTH},
{"EMLSRFreqSeparationNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_EMLSRFreqSeparationNumberOfEntries, NULL, BBFDM_BOTH},
{"STRFreqSeparationNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_STRFreqSeparationNumberOfEntries, NULL, BBFDM_BOTH},
//{"NSTRFreqSeparationNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_NSTRFreqSeparationNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7bSTARole.EMLMRFreqSeparation.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLMRFreqSeparationParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLMRFreqSeparation_RUID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"FreqSeparation", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLMRFreqSeparation_FreqSeparation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7bSTARole.EMLSRFreqSeparation.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLSRFreqSeparationParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLSRFreqSeparation_RUID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"FreqSeparation", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLSRFreqSeparation_FreqSeparation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7bSTARole.STRFreqSeparation.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleSTRFreqSeparationParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleSTRFreqSeparation_RUID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"FreqSeparation", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleSTRFreqSeparation_FreqSeparation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7bSTARole.NSTRFreqSeparation.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleNSTRFreqSeparationParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
//{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleNSTRFreqSeparation_RUID, NULL, BBFDM_BOTH},
//{"FreqSeparation", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleNSTRFreqSeparation_FreqSeparation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.AKMFrontHaul.{i}. *** */
//DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesAKMFrontHaulParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
//{"OUI", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesAKMFrontHaul_OUI, NULL, BBFDM_BOTH},
//{"Type", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesAKMFrontHaul_Type, NULL, BBFDM_BOTH},
//{0}
//};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.AKMBackhaul.{i}. *** */
//DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesAKMBackhaulParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
//{"OUI", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesAKMBackhaul_OUI, NULL, BBFDM_BOTH},
//{"Type", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesAKMBackhaul_Type, NULL, BBFDM_BOTH},
//{0}
//};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.CapableOperatingClassProfile.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfileParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Class", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfile_Class, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"MaxTxPower", &DMREAD, DMT_INT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfile_MaxTxPower, NULL, BBFDM_BOTH},
{"NonOperable", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfile_NonOperable, NULL, BBFDM_BOTH},
{"NumberOfNonOperChan", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfile_NumberOfNonOperChan, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.CurrentOperatingClassProfile.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfileParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Class", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_Class, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Channel", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_Channel, NULL, BBFDM_BOTH},
{"TxPower", &DMREAD, DMT_INT, get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_TxPower, NULL, BBFDM_BOTH},
// {"TransmitPowerLimit", &DMREAD, DMT_INT, get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_TransmitPowerLimit, NULL, BBFDM_BOTH},
{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_TimeStamp, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.DisAllowedOpClassChannels.{i}. *** */
//DMLEAF tWiFiDataElementsNetworkDeviceRadioDisAllowedOpClassChannelsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
//{"Enable", &DMWRITE, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioDisAllowedOpClassChannels_Enable, set_WiFiDataElementsNetworkDeviceRadioDisAllowedOpClassChannels_Enable, BBFDM_BOTH},
//{"OpClass", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioDisAllowedOpClassChannels_OpClass, set_WiFiDataElementsNetworkDeviceRadioDisAllowedOpClassChannels_OpClass, BBFDM_BOTH},
//{"ChannelList", &DMWRITE, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioDisAllowedOpClassChannels_ChannelList, set_WiFiDataElementsNetworkDeviceRadioDisAllowedOpClassChannels_ChannelList, BBFDM_BOTH},
//{0}
//};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.SpatialReuse. *** */
//DMLEAF tWiFiDataElementsNetworkDeviceRadioSpatialReuseParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
//{"PartialBSSColor", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioSpatialReuse_PartialBSSColor, NULL, BBFDM_BOTH},
//{"BSSColor", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioSpatialReuse_BSSColor, NULL, BBFDM_BOTH},
//{"HESIGASpatialReuseValue15Allowed", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioSpatialReuse_HESIGASpatialReuseValue15Allowed, NULL, BBFDM_BOTH},
//{"SRGInformationValid", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioSpatialReuse_SRGInformationValid, NULL, BBFDM_BOTH},
//{"NonSRGOffsetValid", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioSpatialReuse_NonSRGOffsetValid, NULL, BBFDM_BOTH},
//{"PSRDisallowed", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioSpatialReuse_PSRDisallowed, NULL, BBFDM_BOTH},
//{"NonSRGOBSSPDMaxOffset", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioSpatialReuse_NonSRGOBSSPDMaxOffset, NULL, BBFDM_BOTH},
//{"SRGOBSSPDMinOffset", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioSpatialReuse_SRGOBSSPDMinOffset, NULL, BBFDM_BOTH},
//{"SRGOBSSPDMaxOffset", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioSpatialReuse_SRGOBSSPDMaxOffset, NULL, BBFDM_BOTH},
//{"SRGBSSColorBitmap", &DMREAD, DMT_HEXBIN, get_WiFiDataElementsNetworkDeviceRadioSpatialReuse_SRGBSSColorBitmap, NULL, BBFDM_BOTH},
//{"SRGPartialBSSIDBitmap", &DMREAD, DMT_HEXBIN, get_WiFiDataElementsNetworkDeviceRadioSpatialReuse_SRGPartialBSSIDBitmap, NULL, BBFDM_BOTH},
//{"NeighborBSSColorInUseBitmap", &DMREAD, DMT_HEXBIN, get_WiFiDataElementsNetworkDeviceRadioSpatialReuse_NeighborBSSColorInUseBitmap, NULL, BBFDM_BOTH},
//{0}
//};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioBSSObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
// {"QMDescriptor", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioBSSQMDescriptorInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSQMDescriptorParams, NULL, BBFDM_BOTH, NULL},
{"MultiAPSteering", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSMultiAPSteeringParams, NULL, BBFDM_BOTH, NULL},
{"STA", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioBSSSTAInst, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSSTAObj, tWiFiDataElementsNetworkDeviceRadioBSSSTAParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioBSSParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"BSSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSS_BSSID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"SSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSS_SSID, NULL, BBFDM_BOTH},
{"Enabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_Enabled, NULL, BBFDM_BOTH},
{"LastChange", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSS_LastChange, NULL, BBFDM_BOTH},
{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceRadioBSS_TimeStamp, NULL, BBFDM_BOTH},
{"UnicastBytesSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSS_UnicastBytesSent, NULL, BBFDM_BOTH},
{"UnicastBytesReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSS_UnicastBytesReceived, NULL, BBFDM_BOTH},
{"MulticastBytesSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSS_MulticastBytesSent, NULL, BBFDM_BOTH},
{"MulticastBytesReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSS_MulticastBytesReceived, NULL, BBFDM_BOTH},
{"BroadcastBytesSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSS_BroadcastBytesSent, NULL, BBFDM_BOTH},
{"BroadcastBytesReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSS_BroadcastBytesReceived, NULL, BBFDM_BOTH},
// {"ByteCounterUnits", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSS_ByteCounterUnits, NULL, BBFDM_BOTH},
{"Profile1bSTAsDisallowed", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_Profile1bSTAsDisallowed, NULL, BBFDM_BOTH},
{"Profile2bSTAsDisallowed", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_Profile2bSTAsDisallowed, NULL, BBFDM_BOTH},
// {"AssociationAllowanceStatus", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSS_AssociationAllowanceStatus, NULL, BBFDM_BOTH},
{"EstServiceParametersBE", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioBSS_EstServiceParametersBE, NULL, BBFDM_BOTH},
{"EstServiceParametersBK", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioBSS_EstServiceParametersBK, NULL, BBFDM_BOTH},
{"EstServiceParametersVI", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioBSS_EstServiceParametersVI, NULL, BBFDM_BOTH},
{"EstServiceParametersVO", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioBSS_EstServiceParametersVO, NULL, BBFDM_BOTH},
{"BackhaulUse", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_BackhaulUse, NULL, BBFDM_BOTH},
{"FronthaulUse", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_FronthaulUse, NULL, BBFDM_BOTH},
{"R1disallowed", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_R1disallowed, NULL, BBFDM_BOTH},
{"R2disallowed", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_R2disallowed, NULL, BBFDM_BOTH},
{"MultiBSSID", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_MultiBSSID, NULL, BBFDM_BOTH},
{"TransmittedBSSID", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_TransmittedBSSID, NULL, BBFDM_BOTH},
// {"FronthaulAKMsAllowed", &DMWRITE, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSS_FronthaulAKMsAllowed, set_WiFiDataElementsNetworkDeviceRadioBSS_FronthaulAKMsAllowed, BBFDM_BOTH},
// {"BackhaulAKMsAllowed", &DMWRITE, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSS_BackhaulAKMsAllowed, set_WiFiDataElementsNetworkDeviceRadioBSS_BackhaulAKMsAllowed, BBFDM_BOTH},
{"STANumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSS_STANumberOfEntries, NULL, BBFDM_BOTH},
// {"QMDescriptorNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSS_QMDescriptorNumberOfEntries, NULL, BBFDM_BOTH},
//{"SetQMDescriptors()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDeviceRadioBSS_SetQMDescriptors, operate_WiFiDataElementsNetworkDeviceRadioBSS_SetQMDescriptors, BBFDM_USP},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.QMDescriptor.{i}. *** */
// DMLEAF tWiFiDataElementsNetworkDeviceRadioBSSQMDescriptorParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
// {"ClientMAC", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSQMDescriptor_ClientMAC, NULL, BBFDM_BOTH},
// {"DescriptorElement", &DMREAD, DMT_HEXBIN, get_WiFiDataElementsNetworkDeviceRadioBSSQMDescriptor_DescriptorElement, NULL, BBFDM_BOTH},
// {0}
//};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.MultiAPSteering. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioBSSMultiAPSteeringParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"BlacklistAttempts", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSMultiAPSteering_BlacklistAttempts, NULL, BBFDM_BOTH},
{"BTMAttempts", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSMultiAPSteering_BTMAttempts, NULL, BBFDM_BOTH},
{"BTMQueryResponses", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSMultiAPSteering_BTMQueryResponses, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioBSSSTAObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"MultiAPSTA", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTAObj, tWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTAParams, NULL, BBFDM_BOTH, NULL},
{"WiFi6Capabilities", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6CapabilitiesParams, NULL, BBFDM_BOTH, NULL},
//{"TIDQueueSizes", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioBSSSTATIDQueueSizesInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSSTATIDQueueSizesParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioBSSSTAParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"MACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_MACAddress, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE|DM_FLAG_LINKER},
{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_TimeStamp, NULL, BBFDM_BOTH},
{"HTCapabilities", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_HTCapabilities, NULL, BBFDM_BOTH},
{"VHTCapabilities", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_VHTCapabilities, NULL, BBFDM_BOTH},
//{"ClientCapabilities", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_ClientCapabilities, NULL, BBFDM_BOTH},
{"LastDataDownlinkRate", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_LastDataDownlinkRate, NULL, BBFDM_BOTH},
{"LastDataUplinkRate", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_LastDataUplinkRate, NULL, BBFDM_BOTH},
{"UtilizationReceive", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_UtilizationReceive, NULL, BBFDM_BOTH},
{"UtilizationTransmit", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_UtilizationTransmit, NULL, BBFDM_BOTH},
{"EstMACDataRateDownlink", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_EstMACDataRateDownlink, NULL, BBFDM_BOTH},
{"EstMACDataRateUplink", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_EstMACDataRateUplink, NULL, BBFDM_BOTH},
{"SignalStrength", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_SignalStrength, NULL, BBFDM_BOTH},
{"LastConnectTime", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_LastConnectTime, NULL, BBFDM_BOTH},
{"BytesSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_BytesSent, NULL, BBFDM_BOTH},
{"BytesReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_BytesReceived, NULL, BBFDM_BOTH},
{"PacketsSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_PacketsSent, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_PacketsReceived, NULL, BBFDM_BOTH},
{"ErrorsSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_ErrorsSent, NULL, BBFDM_BOTH},
{"ErrorsReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_ErrorsReceived, NULL, BBFDM_BOTH},
{"RetransCount", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_RetransCount, NULL, BBFDM_BOTH},
{"MeasurementReport", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_MeasurementReport, NULL, BBFDM_BOTH},
{"NumberOfMeasureReports", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_NumberOfMeasureReports, NULL, BBFDM_BOTH},
{"IPV4Address", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_IPV4Address, NULL, BBFDM_BOTH},
{"IPV6Address", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_IPV6Address, NULL, BBFDM_BOTH},
{"Hostname", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_Hostname, NULL, BBFDM_BOTH},
{"ClientSteer()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDeviceRadioBSSSTA_ClientSteer, operate_WiFiDataElementsNetworkDeviceRadioBSSSTA_ClientSteer, BBFDM_USP},
// {"CellularDataPreference", &DMWRITE, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_CellularDataPreference, set_WiFiDataElementsNetworkDeviceRadioBSSSTA_CellularDataPreference, BBFDM_BOTH},
// {"ReAssociationDelay", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_ReAssociationDelay, set_WiFiDataElementsNetworkDeviceRadioBSSSTA_ReAssociationDelay, BBFDM_BOTH},
//{"TIDQueueSizesNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_TIDQueueSizesNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.MultiAPSTA. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTAObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"SteeringSummaryStats", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStatsParams, NULL, BBFDM_BOTH, NULL},
{"SteeringHistory", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistoryInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistoryParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTAParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
//{"AssociationTime", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTA_AssociationTime, NULL, BBFDM_BOTH},
//{"Noise", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTA_Noise, NULL, BBFDM_BOTH},
{"SteeringHistoryNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTA_SteeringHistoryNumberOfEntries, NULL, BBFDM_BOTH},
//{"Disassociate()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTA_Disassociate, operate_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTA_Disassociate, BBFDM_USP},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.MultiAPSTA.SteeringSummaryStats. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStatsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"NoCandidateAPFailures", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_NoCandidateAPFailures, NULL, BBFDM_BOTH},
{"BlacklistAttempts", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BlacklistAttempts, NULL, BBFDM_BOTH},
{"BlacklistSuccesses", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BlacklistSuccesses, NULL, BBFDM_BOTH},
{"BlacklistFailures", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BlacklistFailures, NULL, BBFDM_BOTH},
{"BTMAttempts", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BTMAttempts, NULL, BBFDM_BOTH},
{"BTMSuccesses", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BTMSuccesses, NULL, BBFDM_BOTH},
{"BTMFailures", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BTMFailures, NULL, BBFDM_BOTH},
{"BTMQueryResponses", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BTMQueryResponses, NULL, BBFDM_BOTH},
{"LastSteerTime", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_LastSteerTime, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.MultiAPSTA.SteeringHistory.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistoryParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Time", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_Time, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"APOrigin", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_APOrigin, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"TriggerEvent", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_TriggerEvent, NULL, BBFDM_BOTH},
{"SteeringApproach", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_SteeringApproach, NULL, BBFDM_BOTH},
{"APDestination", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_APDestination, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"SteeringDuration", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_SteeringDuration, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6CapabilitiesParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"HE160", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_HE160, NULL, BBFDM_BOTH},
{"HE8080", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_HE8080, NULL, BBFDM_BOTH},
{"MCSNSS", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MCSNSS, NULL, BBFDM_BOTH},
{"SUBeamformer", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_SUBeamformer, NULL, BBFDM_BOTH},
{"SUBeamformee", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_SUBeamformee, NULL, BBFDM_BOTH},
{"MUBeamformer", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MUBeamformer, NULL, BBFDM_BOTH},
{"Beamformee80orLess", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_Beamformee80orLess, NULL, BBFDM_BOTH},
{"BeamformeeAbove80", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_BeamformeeAbove80, NULL, BBFDM_BOTH},
{"ULMUMIMO", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_ULMUMIMO, NULL, BBFDM_BOTH},
{"ULOFDMA", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_ULOFDMA, NULL, BBFDM_BOTH},
{"MaxDLMUMIMO", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MaxDLMUMIMO, NULL, BBFDM_BOTH},
{"MaxULMUMIMO", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MaxULMUMIMO, NULL, BBFDM_BOTH},
{"MaxDLOFDMA", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MaxDLOFDMA, NULL, BBFDM_BOTH},
{"MaxULOFDMA", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MaxULOFDMA, NULL, BBFDM_BOTH},
{"RTS", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_RTS, NULL, BBFDM_BOTH},
{"MURTS", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MURTS, NULL, BBFDM_BOTH},
{"MultiBSSID", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MultiBSSID, NULL, BBFDM_BOTH},
{"MUEDCA", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MUEDCA, NULL, BBFDM_BOTH},
{"TWTRequestor", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_TWTRequestor, NULL, BBFDM_BOTH},
{"TWTResponder", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_TWTResponder, NULL, BBFDM_BOTH},
{"SpatialReuse", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_SpatialReuse, NULL, BBFDM_BOTH},
{"AnticipatedChannelUsage", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_AnticipatedChannelUsage, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.TIDQueueSizes.{i}. *** */
//DMLEAF tWiFiDataElementsNetworkDeviceRadioBSSSTATIDQueueSizesParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
//{"TID", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTATIDQueueSizes_TID, NULL, BBFDM_BOTH},
//{"Size", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTATIDQueueSizes_Size, NULL, BBFDM_BOTH},
//{0}
//};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.UnassociatedSTA.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioUnassociatedSTAParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"MACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioUnassociatedSTA_MACAddress, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"SignalStrength", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioUnassociatedSTA_SignalStrength, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.MultiAPRadio. *** */
//DMLEAF tWiFiDataElementsNetworkDeviceRadioMultiAPRadioParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
//{"RadarDetections", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioMultiAPRadio_RadarDetections, NULL, BBFDM_BOTH},
//{"FullScan()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDeviceRadioMultiAPRadio_FullScan, operate_WiFiDataElementsNetworkDeviceRadioMultiAPRadio_FullScan, BBFDM_USP},
//{"ChannelScan()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDeviceRadioMultiAPRadio_ChannelScan, operate_WiFiDataElementsNetworkDeviceRadioMultiAPRadio_ChannelScan, BBFDM_USP},
//{0}
//};

/* *** Device.WiFi.DataElements.Network.Device.{i}.APMLD.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceAPMLDObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys */
{"APMLDConfig", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceAPMLDAPMLDConfigParams, NULL, BBFDM_BOTH, NULL},
{"TIDLinkMap", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceAPMLDTIDLinkMapInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceAPMLDTIDLinkMapParams, NULL, BBFDM_BOTH, NULL},
{"AffiliatedAP", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceAPMLDAffiliatedAPInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceAPMLDAffiliatedAPParams, NULL, BBFDM_BOTH, NULL},
{"STAMLD", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceAPMLDSTAMLDInst, NULL, NULL, tWiFiDataElementsNetworkDeviceAPMLDSTAMLDObj, tWiFiDataElementsNetworkDeviceAPMLDSTAMLDParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceAPMLDParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"MLDMACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLD_MLDMACAddress, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"TIDLinkMapNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLD_TIDLinkMapNumberOfEntries, NULL, BBFDM_BOTH},
{"AffiliatedAPNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLD_AffiliatedAPNumberOfEntries, NULL, BBFDM_BOTH},
{"STAMLDNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLD_STAMLDNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.APMLD.{i}.APMLDConfig. *** */
DMLEAF tWiFiDataElementsNetworkDeviceAPMLDAPMLDConfigParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"EMLMREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_EMLMREnabled, NULL, BBFDM_BOTH},
{"EMLSREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_EMLSREnabled, NULL, BBFDM_BOTH},
{"STREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_STREnabled, NULL, BBFDM_BOTH},
{"NSTREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_NSTREnabled, NULL, BBFDM_BOTH},
{"TIDLinkMapNegotiation", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_TIDLinkMapNegotiation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.APMLD.{i}.TIDLinkMap.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceAPMLDTIDLinkMapParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"Direction", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDTIDLinkMap_Direction, NULL, BBFDM_BOTH},
{"TID", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDTIDLinkMap_TID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
//{"BSSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDTIDLinkMap_BSSID, NULL, BBFDM_BOTH},
{"LinkID", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDTIDLinkMap_LinkID, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.APMLD.{i}.AffiliatedAP.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceAPMLDAffiliatedAPParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"BSSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_BSSID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"LinkID", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_LinkID, NULL, BBFDM_BOTH},
{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_RUID, NULL, BBFDM_BOTH},
//{"DisabledSubChannels", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_DisabledSubChannels, NULL, BBFDM_BOTH},
{"PacketsSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_PacketsSent, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_PacketsReceived, NULL, BBFDM_BOTH},
{"UnicastBytesSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_UnicastBytesSent, NULL, BBFDM_BOTH},
{"UnicastBytesReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_UnicastBytesReceived, NULL, BBFDM_BOTH},
{"ErrorsSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_ErrorsSent, NULL, BBFDM_BOTH},
{"MulticastBytesSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_MulticastBytesSent, NULL, BBFDM_BOTH},
{"MulticastBytesReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_MulticastBytesReceived, NULL, BBFDM_BOTH},
{"BroadcastBytesSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_BroadcastBytesSent, NULL, BBFDM_BOTH},
{"BroadcastBytesReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_BroadcastBytesReceived, NULL, BBFDM_BOTH},
{"EstServiceParametersBE", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_EstServiceParametersBE, NULL, BBFDM_BOTH},
{"EstServiceParametersBK", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_EstServiceParametersBK, NULL, BBFDM_BOTH},
{"EstServiceParametersVI", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_EstServiceParametersVI, NULL, BBFDM_BOTH},
{"EstServiceParametersVO", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_EstServiceParametersVO, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.APMLD.{i}.STAMLD.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceAPMLDSTAMLDObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys */
{"WiFi7Capabilities", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7CapabilitiesParams, NULL, BBFDM_BOTH, NULL},
{"STAMLDConfig", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfigParams, NULL, BBFDM_BOTH, NULL},
{"STATIDLinkMap", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceAPMLDSTAMLDSTATIDLinkMapInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceAPMLDSTAMLDSTATIDLinkMapParams, NULL, BBFDM_BOTH, NULL},
{"AffiliatedSTA", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTAInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTAParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceAPMLDSTAMLDParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"MLDMACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_MLDMACAddress, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE|DM_FLAG_LINKER},
{"Hostname", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_Hostname, NULL, BBFDM_BOTH},
{"IPv4Address", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_IPv4Address, NULL, BBFDM_BOTH},
{"IPv6Address", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_IPv6Address, NULL, BBFDM_BOTH},
{"IsbSTA", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_IsbSTA, NULL, BBFDM_BOTH},
{"LastConnectTime", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_LastConnectTime, NULL, BBFDM_BOTH},
{"BytesSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_BytesSent, NULL, BBFDM_BOTH},
{"BytesReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_BytesReceived, NULL, BBFDM_BOTH},
{"PacketsSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_PacketsSent, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_PacketsReceived, NULL, BBFDM_BOTH},
{"ErrorsSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_ErrorsSent, NULL, BBFDM_BOTH},
{"ErrorsReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_ErrorsReceived, NULL, BBFDM_BOTH},
{"RetransCount", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_RetransCount, NULL, BBFDM_BOTH},
{"STATIDLinkMapNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_STATIDLinkMapNumberOfEntries, NULL, BBFDM_BOTH},
{"AffiliatedSTANumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_AffiliatedSTANumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.APMLD.{i}.STAMLD.{i}.WiFi7Capabilities. *** */
DMLEAF tWiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7CapabilitiesParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"EMLMRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_EMLMRSupport, NULL, BBFDM_BOTH},
{"EMLSRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_EMLSRSupport, NULL, BBFDM_BOTH},
{"STRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_STRSupport, NULL, BBFDM_BOTH},
{"NSTRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_NSTRSupport, NULL, BBFDM_BOTH},
{"TIDLinkMapNegotiation", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_TIDLinkMapNegotiation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.APMLD.{i}.STAMLD.{i}.STAMLDConfig. *** */
DMLEAF tWiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfigParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"EMLMREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_EMLMREnabled, NULL, BBFDM_BOTH},
{"EMLSREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_EMLSREnabled, NULL, BBFDM_BOTH},
{"STREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_STREnabled, NULL, BBFDM_BOTH},
{"NSTREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_NSTREnabled, NULL, BBFDM_BOTH},
{"TIDLinkMapNegotiation", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_TIDLinkMapNegotiation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.APMLD.{i}.STAMLD.{i}.STATIDLinkMap.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceAPMLDSTAMLDSTATIDLinkMapParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"Direction", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTATIDLinkMap_Direction, NULL, BBFDM_BOTH},
{"TID", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTATIDLinkMap_TID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
//{"BSSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTATIDLinkMap_BSSID, NULL, BBFDM_BOTH},
{"LinkID", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTATIDLinkMap_LinkID, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.APMLD.{i}.STAMLD.{i}.AffiliatedSTA.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTAParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"MACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_MACAddress, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"BSSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_BSSID, NULL, BBFDM_BOTH},
{"BytesSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_BytesSent, NULL, BBFDM_BOTH},
{"BytesReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_BytesReceived, NULL, BBFDM_BOTH},
{"PacketsSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_PacketsSent, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_PacketsReceived, NULL, BBFDM_BOTH},
{"ErrorsSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_ErrorsSent, NULL, BBFDM_BOTH},
{"SignalStrength", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_SignalStrength, NULL, BBFDM_BOTH},
{"EstMACDataRateDownlink", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_EstMACDataRateDownlink, NULL, BBFDM_BOTH},
{"EstMACDataRateUplink", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_EstMACDataRateUplink, NULL, BBFDM_BOTH},
{"LastDataDownlinkRate", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_LastDataDownlinkRate, NULL, BBFDM_BOTH},
{"LastDataUplinkRate", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_LastDataUplinkRate, NULL, BBFDM_BOTH},
{"UtilizationReceive", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_UtilizationReceive, NULL, BBFDM_BOTH},
{"UtilizationTransmit", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_UtilizationTransmit, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.bSTAMLD. *** */
DMOBJ tWiFiDataElementsNetworkDevicebSTAMLDObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys */
{"bSTAMLDConfig", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfigParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDevicebSTAMLDParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"MLDMACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevicebSTAMLD_MLDMACAddress, NULL, BBFDM_BOTH},
{"BSSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevicebSTAMLD_BSSID, NULL, BBFDM_BOTH},
{"AffiliatedbSTAList", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevicebSTAMLD_AffiliatedbSTAList, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.bSTAMLD.bSTAMLDConfig. *** */
DMLEAF tWiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfigParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"EMLMREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_EMLMREnabled, NULL, BBFDM_BOTH},
{"EMLSREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_EMLSREnabled, NULL, BBFDM_BOTH},
{"STREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_STREnabled, NULL, BBFDM_BOTH},
{"NSTREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_NSTREnabled, NULL, BBFDM_BOTH},
{"TIDLinkMapNegotiation", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_TIDLinkMapNegotiation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.AssociationEvent. *** */
DMLEAF tWiFiDataElementsAssociationEventParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Associated!", &DMREAD, DMT_EVENT, get_event_args_WiFiDataElementsAssociationEvent_Associated, event_WiFiDataElementsAssociationEvent_Associated, BBFDM_USP},
{0}
};

/* *** Device.WiFi.DataElements.DisassociationEvent. *** */
DMLEAF tWiFiDataElementsDisassociationEventParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Disassociated!", &DMREAD, DMT_EVENT, get_event_args_WiFiDataElementsDisassociationEvent_Disassociated, event_WiFiDataElementsDisassociationEvent_Disassociated, BBFDM_USP},
{0}
};

/* *** Device.WiFi.DataElements.FailedConnectionEvent. *** */
//DMLEAF tWiFiDataElementsFailedConnectionEventParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
//{"FailedConnection!", &DMREAD, DMT_EVENT, get_event_args_WiFiDataElementsFailedConnectionEvent_FailedConnection, NULL, BBFDM_USP},
//{0}
//};
