import { defaultMissingErrorMessage } from "./errors.js";
const isPresent = (item) => item !== undefined;
const isBool = (item) => typeof item === "boolean";
const isNumber = (item) => typeof item === "number";
const isString = (item) => typeof item === "string";
const isArray = (item) => Array.isArray(item);
const isTupleArray = (item) => Array.isArray(item) &&
item.length > 0 &&
item.every((it) => isArray(it) && it.length === 2);
const checkGetOptions = ({ options }) => {
if (!isPresent(options))
return "VerifyResult.NoError" /* NoError */;
if (isPresent(options.raw) && !isBool(options.raw))
return "VerifyResult.RawOptionNotBool" /* RawOptionNotBool */;
if (isPresent(options.retainPath) && !isBool(options.retainPath))
return "VerifyResult.RetainPathOptionNotBool" /* RetainPathOptionNotBool */;
if (isPresent(options.max_depth)) {
if (!isNumber(options.max_depth))
return "VerifyResult.MaxDepthOptionNotNumber" /* MaxDepthOptionNotNumber */;
if (options.max_depth < 0)
return "VerifyResult.MaxDepthOptionIsNegative" /* MaxDepthOptionIsNegative */;
}
return "VerifyResult.NoError" /* NoError */;
};
const checkSetValue = ({ path, value }) => {
if (isTupleArray(path))
return "VerifyResult.NoError" /* NoError */;
if (isArray(path)) {
if (!isPresent(value) && !isTupleArray(path))
return "VerifyResult.ValueIsNotTupleArray" /* ValueIsNotTupleArray */;
if (!isArray(value))
return "VerifyResult.ValueMustBeArray" /* ValueMustBeArray */;
if (path.length !== value.length)
return "VerifyResult.ValueAndPathArraysMustBeSameLength" /* ValueAndPathArraysMustBeSameLength */;
}
return value !== undefined && value !== null
? "VerifyResult.NoError" /* NoError */
: "VerifyResult.ValueIsMissing" /* ValueIsMissing */;
};
const checkAddValue = ({ path, value }) => {
if (isTupleArray(path))
return "VerifyResult.NoError" /* NoError */;
if (isArray(path)) {
if (!isPresent(value)) {
if (!isTupleArray(path))
return "VerifyResult.ValueIsNotTupleArray" /* ValueIsNotTupleArray */;
else
return "VerifyResult.NoError" /* NoError */;
}
if (!isArray(value))
return "VerifyResult.ValueMustBeArray" /* ValueMustBeArray */;
if (path.length !== value.length)
return "VerifyResult.ValueAndPathArraysMustBeSameLength" /* ValueAndPathArraysMustBeSameLength */;
}
return "VerifyResult.NoError" /* NoError */;
};
const checkSetOptions = ({ options }) => {
if (!isPresent(options))
return "VerifyResult.NoError" /* NoError */;
if (isPresent(options.raw) && !isBool(options.raw))
return "VerifyResult.RawOptionNotBool" /* RawOptionNotBool */;
return "VerifyResult.NoError" /* NoError */;
};
const checkDelOptions = ({ allowPartial }) => {
if (!isPresent(allowPartial))
return "VerifyResult.NoError" /* NoError */;
if (!isBool(allowPartial))
return "VerifyResult.AllowPartialOptionNotBool" /* AllowPartialOptionNotBool */;
return "VerifyResult.NoError" /* NoError */;
};
const checkSupportedDMOptions = ({ opts }) => {
if (!isPresent(opts))
return "VerifyResult.NoError" /* NoError */;
const allOptsAreBool = Object.values(opts).every(isBool);
return allOptsAreBool ? "VerifyResult.NoError" /* NoError */ : "VerifyResult.OptionsMustBeBool" /* OptionsMustBeBool */;
};
const checkInstancesOptions = ({ opts }) => {
if (!isPresent(opts))
return "VerifyResult.NoError" /* NoError */;
if (isPresent(opts.firstLevelOnly) && !isBool(opts.firstLevelOnly))
return "VerifyResult.FirstLevelOnlyNotBool" /* FirstLevelOnlyNotBool */;
return "VerifyResult.NoError" /* NoError */;
};
const checkProto = ({ proto }) => {
if (isPresent(proto) && !isString(proto))
return "VerifyResult.ProtoNotString" /* ProtoNotString */;
return "VerifyResult.NoError" /* NoError */;
};
const checkSetPaths = ({ path, value }) => {
if (!isPresent(path))
return "VerifyResult.PathsMissing" /* PathsMissing */;
if (!isPresent(value) && isArray(path) && !isTupleArray(path))
return "VerifyResult.ValueIsNotTupleArray" /* ValueIsNotTupleArray */;
return isString(path) ||
(isArray(path) &&
path.length > 0 &&
path.every((path) => typeof path === "string" || isArray(path)))
? "VerifyResult.NoError" /* NoError */
: "VerifyResult.PathsNotStringOrStringArray" /* PathsNotStringOrStringArray */;
};
const checkPaths = ({ paths: initalPaths, path: initialPath, }) => {
const paths = initalPaths || initialPath;
return !isPresent(paths)
? "VerifyResult.PathsMissing" /* PathsMissing */
: isString(paths) ||
(isArray(paths) &&
paths.length > 0 &&
paths.every((path) => typeof path === "string"))
? "VerifyResult.NoError" /* NoError */
: "VerifyResult.PathsNotStringOrStringArray" /* PathsNotStringOrStringArray */;
};
const argsVerifiers = {
GET: [checkPaths, checkGetOptions],
SET: [checkSetPaths, checkSetValue, checkSetOptions],
ADD: [checkSetPaths, checkAddValue, checkSetOptions],
DELETE: [checkPaths, checkDelOptions],
OPERATE: [checkPaths],
GET_SUPPORTED_DM: [checkPaths, checkSupportedDMOptions],
GET_INSTANCES: [checkPaths, checkInstancesOptions],
GET_SUPPORTED_PROTO: [checkProto],
};
const translateError = (errors, verifyResult) => new Error(errors?.[verifyResult] || defaultMissingErrorMessage);
export const makeVerify = (errors) => (command, args) => {
const verifiers = argsVerifiers[command];
if (!verifiers)
return null;
const results = verifiers.map((verifier) => verifier(args));
const finalResult = results.find((value) => value !== "VerifyResult.NoError" /* NoError */) ||
"VerifyResult.NoError" /* NoError */;
return finalResult === "VerifyResult.NoError" /* NoError */
? null
: translateError(errors, finalResult);
};
export default makeVerify;
