import { makeEncode, makeRecipes, readMsg, decode, decodeWithOptions, makeSession, makeEmptySessionMessage as makeInitSessionMessage, decodeRecord, readRecord, } from "../commands/index.js";
import { knownUSPVersions, makeCallbackRouter, makeRouter, parseCallbackOptions, } from "../util.js";
import makeVerify from "../verify.js";
import defaultErrors from "../errors.js";
const defaultPublishEndpoint = "/usp/endpoint";
const defaultSubscribeEndpoint = "/usp/controller";
const defaultIdEndpoint = "obuspa/EndpointID";
const defaultFromId = "proto::interop-usp-controller";
const defaultConnectionTimeout = 5000;
const defaultIdResolveTimeout = 5000;
const defaultSessionInitTimeout = 5000;
const fixId = (s) => s.split("+").join("%2B");
const wait = (ms) => {
let waitId;
let waitPromise = new Promise((_, reject) => {
waitId = setTimeout(() => {
reject(`timeout of ${ms}ms reached`);
}, ms);
});
return [waitId, waitPromise];
};
const timeoutWrapper = (promise, ms) => {
if (typeof ms === "number" && ms >= 0) {
const [waitId, waitPromise] = wait(ms);
return Promise.race([
promise().then((res) => {
clearTimeout(waitId);
return res;
}),
waitPromise,
]);
}
else
return promise();
};
const applyCommandChanges = (opts, cmdName, cmd, args) => {
if (cmdName === "get" && opts?.[cmdName])
return cmd(args[0], {
...opts[cmdName],
...(args.length === 2 ? args[1] : {}),
});
return cmd(...args);
};
const wrap = (opts, cmdName, cmd) => {
return ((...args) => {
// Promise.resolve added to handle non-promise commands
const cmdCall = () => Promise.resolve(applyCommandChanges(opts, cmdName, cmd, args));
const finalCall = () => timeoutWrapper(cmdCall, opts.timeout)
.then((res) => {
opts.postCall && opts.postCall(cmdName, args, res);
return res;
})
.catch((err) => {
opts.postCall && opts.postCall(cmdName, args, err);
throw err;
});
if (opts.preCall)
opts.preCall(cmdName, args);
return finalCall();
});
};
const addOptions = (usp, opts) => {
const mainUSP = Object.entries(usp).reduce((acc, [k, fn]) => ({
...acc,
[k]: wrap(opts, k, fn),
}), {});
mainUSP.options = (opts) => addOptions(mainUSP, opts);
return mainUSP;
};
const correctOptions = (options) => ({
...options,
reconnectPeriod: options.closeOnDisconnect ? 0 : 1000,
});
const isSessionInitMsg = (msg) => typeof msg === "object" && "sessionContext" in msg;
const makeDeferred = () => {
let resolve, reject;
const promise = new Promise((res, rej) => {
resolve = res;
reject = rej;
});
return { resolve, reject, promise };
};
const initializeSession = ({ client, proto, bufferOptions, sessionOptions, version, publishEndpoint, subscribeEndpoint, }) => {
const deferred = makeDeferred();
let sessionInitTimeoutId = 0;
let sessionId = 1;
client.on("message", (_topic, data) => {
const decodedMsg = decodeRecord(proto, data);
if (isSessionInitMsg(decodedMsg)) {
clearTimeout(sessionInitTimeoutId);
sessionId = Number(decodedMsg.sessionContext.sessionId);
sessionOptions.sessionId = sessionId;
deferred.resolve();
}
});
const sessionMessage = makeInitSessionMessage(proto, bufferOptions, sessionOptions, version, true);
client.subscribe(subscribeEndpoint);
client.publish(publishEndpoint, sessionMessage);
sessionInitTimeoutId = setTimeout(() => {
deferred.reject();
throw `Session initalization timed out(${defaultSessionInitTimeout}ms)`;
}, defaultSessionInitTimeout);
return deferred.promise;
};
const buildConnect = ({ connectClient, decodeID, loadProtobuf, errors = defaultErrors }) => async (options, events) => {
const subscribeEndpoint = options.subscribeEndpoint || defaultSubscribeEndpoint;
const publishEndpoint = options.publishEndpoint || defaultPublishEndpoint;
const idEndpoint = options.idEndpoint || defaultIdEndpoint;
const connectTimeout = options.connectTimeout || defaultConnectionTimeout;
const correctedOptions = correctOptions(options);
const client = await timeoutWrapper(() => connectClient(correctedOptions), connectTimeout);
const readers = await Promise.all(knownUSPVersions.map(async (version) => ({
version,
proto: await loadProtobuf(version),
})));
// default to latest available version
let currentReader = readers[readers.length - 1];
const router = makeRouter();
const callbackRouter = makeCallbackRouter();
const handleError = (err) => events?.onError?.(err);
const handleOffline = () => {
events && events.onOffline && events.onOffline();
client.end();
};
const handleReconnect = () => events && events.onReconnect && events.onReconnect();
const handleClose = () => events && events.onClose && events.onClose();
callbackRouter.add("error", handleError);
const handleInit = () => new Promise((resolve, reject) => {
const id = setTimeout(() => reject({
errMsg: `toId was not received within timeout(${defaultIdResolveTimeout}ms)`,
}), defaultIdResolveTimeout);
client.on("message", (topic, data) => {
clearTimeout(id);
client.unsubscribe(idEndpoint);
resolve(decodeID(data || topic));
});
client.subscribe(idEndpoint);
});
const toId = fixId(options.toId || (await handleInit()));
const fromId = options.fromId || defaultFromId;
const bufferOptions = { fromId, toId };
const useSession = options.useSession || typeof options.sessionId === "number";
const sessionOptions = makeSession(options.sessionId || null);
let encode = makeEncode(bufferOptions, useSession);
if (useSession && !options.sessionId)
await initializeSession({
client,
bufferOptions,
sessionOptions,
publishEndpoint,
subscribeEndpoint,
...currentReader,
});
client.on("message", (_topic, data) => {
const [decodedRecord, convertedDecodedRecord] = readRecord(currentReader.proto, data || _topic);
if (convertedDecodedRecord.version !== currentReader.version) {
currentReader =
readers.find((reader) => reader.version === convertedDecodedRecord.version) || currentReader;
}
const [parsedMsg, parseErr, isDisconnect] = readMsg(currentReader.proto, decodedRecord, convertedDecodedRecord);
if (isDisconnect)
throw parseErr;
if (parseErr) {
handleError(JSON.stringify(parseErr, null, 2));
return;
}
const [id, message, err, cmdType] = decode(parsedMsg, currentReader.version);
if (typeof id !== "string") {
handleError(`Could not locate id for message:\n${JSON.stringify(parsedMsg, null, 2)}`);
}
else {
const call = router.get(id, cmdType || "NOTIFY");
if (call && call.resolve && call.reject) {
if (err)
call.reject(err);
else if (call.options) {
const messageAfterOptions = decodeWithOptions(parsedMsg, cmdType, call.options, currentReader.version);
call.resolve(messageAfterOptions);
}
else
call.resolve(message);
}
const cbs = callbackRouter.get(id);
const callbackOptions = parseCallbackOptions(parsedMsg);
cbs.forEach((cb) => {
if (callbackOptions)
cb(message || err || "", parsedMsg, callbackOptions);
else
cb(message || err || "", parsedMsg);
});
}
});
client.on("error", (err) => {
callbackRouter.get("error").forEach((cb) => cb(err));
handleError(JSON.stringify(err, null, 2));
});
client.on("offline", handleOffline);
client.on("reconnect", () => {
handleReconnect();
});
client.on("close", () => {
handleClose();
});
const on = (ident, callback) => {
callbackRouter.add(ident, callback);
return () => {
callbackRouter.del(ident);
};
};
const verify = makeVerify(errors);
const call = (command, args, callOpts) => new Promise((resolve, reject) => {
const error = verify(command, args);
if (error) {
reject(error);
return;
}
// todo make looky nice
sessionOptions.sequenceId++;
sessionOptions.expectedId = sessionOptions.sequenceId;
const [id, msg, err] = encode(currentReader.proto, command, currentReader.version, args, sessionOptions);
if (err)
reject(err);
else {
router.add(id, callOpts?.responseMsgType || command, {
resolve,
reject,
options: args.options,
});
client.publish(publishEndpoint, msg);
}
});
await client.subscribe(subscribeEndpoint);
const baseUSP = {
get: (paths, options) => call("GET", { paths, options }),
getNested: (paths, options) => call("GET", { paths, options: { ...options, retainPath: true } }),
set: (path, value, options) => call("SET", { path, value, options }),
add: (path, value, options) => call("ADD", { path, value, options }),
del: (paths, allowPartial) => call("DELETE", { paths, allowPartial }),
instances: (paths, opts) => call("GET_INSTANCES", { paths, opts }),
supportedDM: (paths, opts) => call("GET_SUPPORTED_DM", { paths, opts }),
supportedProto: (proto) => call("GET_SUPPORTED_PROTO", { proto }),
_operate: (path, id, resp, input, commandKey) => call("OPERATE", { path, input, id, resp, commandKey }),
_notifyResp: (subscriptionId, msgId) => call("NOTIFY_RESP", { subscriptionId, msgId }),
on,
...makeRecipes(call, on),
disconnect: () => client.end(),
getUSPVersion: () => currentReader.version,
};
return {
...baseUSP,
options: (opts) => addOptions(baseUSP, opts),
};
};
export default buildConnect;
