/**
 * Makes a router for storing resolve/reject for a message
 */
export const makeRouter = () => {
    const routes = new Map();
    return {
        get: (id, cmd) => {
            const res = { ...routes.get(`${id}--${cmd}`) };
            routes.delete(id);
            return res;
        },
        add: (id, cmd, data) => {
            routes.set(`${id}--${cmd}`, { ...data });
        },
    };
};
const toId = (id) => id.toString();
const isRegExp = (v) => typeof v === "object";
const satisfies = (id, matches) => isRegExp(id) ? matches.match(id) !== null : id === matches;
export const makeCallbackRouter = () => {
    const routes = new Map();
    return {
        get: (id) => {
            return Array.from(routes.values())
                .filter(({ ident }) => satisfies(ident, id))
                .map((v) => v.callback);
        },
        add: (ident, callback) => {
            routes.set(toId(ident), { callback, ident });
        },
        del: (id) => routes.delete(toId(id)),
    };
};
export const parseCallbackOptions = (msg) => {
    // Fast-path: `sendResp` is only expected on request bodies (NOTIFY / OPERATE).
    // Avoid deep recursive search (very expensive for large GET responses).
    const sendResp = msg?.body?.request?.notify?.sendResp ??
        msg?.body?.request?.operate?.sendResp;
    if (typeof sendResp === "boolean")
        return { sendResp };
    return null;
};
export const knownUSPVersions = ["1.0", "1.1", "1.2"];
export const parseSetArgs = (value, initialPath) => {
    if (Array.isArray(initialPath) &&
        initialPath.every((v) => Array.isArray(v) && v.length === 2) &&
        value === undefined) {
        return [initialPath.map((v) => v[0]), initialPath.map((v) => v[1])];
    }
    const paths = Array.isArray(initialPath) ? initialPath : [initialPath];
    const values = value ? (Array.isArray(value) ? value : [value]) : [];
    return [paths, values];
};
