import { extractCommand, makeBuffer, search, searchParent } from "./util.js";
import resolve from "./recipes/resolve.js";
import operateRecipe from "./recipes/operate.js";
import subscribe from "./recipes/subscribe.js";
import v10 from "./1.0,1.1/index.js";
import v12 from "./1.2/index.js";
import common from "./common/index.js";
// Note: version files are empty for now, they used to be necessary but were later made obsolete. I still keep them around in case version specific changes crop up again.
const versionSpecificCommands = {
"1.0": v10,
"1.1": v10,
"1.2": v12,
};
const getCommand = (version, commandType) => version in versionSpecificCommands &&
commandType in versionSpecificCommands[version]
? versionSpecificCommands[version][commandType]
: common[commandType] || null;
const recipes = [resolve, operateRecipe, subscribe];
export const makeRecipes = (call, on) => recipes.reduce((acc, { make, name }) => ({ ...acc, [name]: make(call, on) }), {});
export const decodeId = (data) => String(data);
const unkownErr = (msg) => ["error", "", msg];
// NOTE: This code runs in multiple runtimes (Node, browser bundle, QuickJS).
// Avoid JSON.stringify/parse conversions for protobuf messages – it is extremely
// expensive for large responses (e.g. big GET results), and it also base64-encodes
// large payload byte arrays. Instead, use protobufjs' toObject conversion.
const protobufToObjectOptions = {
longs: String,
enums: String,
bytes: String,
};
export const decodeRecord = (proto, data) => {
const record = proto.rootRecord.lookupType("usp_record.Record");
const decodedRecord = record.decode("binaryData" in data ? data.binaryData : data);
return record.toObject(decodedRecord, protobufToObjectOptions);
};
const readableError = (err) => Object.getOwnPropertyNames(err).reduce((acc, key) => {
acc[key] = err[key];
return acc;
}, {});
export const readRecord = (proto, data) => {
const record = proto.rootRecord.lookupType("usp_record.Record");
const decodedRecord = record.decode("binaryData" in data ? data.binaryData : data);
// IMPORTANT: do NOT convert the full record to JSON here.
// The record contains the entire USP Msg payload as bytes; converting it causes
// a huge base64 string allocation for large responses.
const converted = {
version: decodedRecord.version,
toId: decodedRecord.toId,
fromId: decodedRecord.fromId,
};
if (decodedRecord.disconnect !== undefined && decodedRecord.disconnect !== null)
converted.disconnect = decodedRecord.disconnect;
return [decodedRecord, converted];
};
export const readMsg = (proto, decodedRecord, convertedDecodedRecord) => {
if ("disconnect" in convertedDecodedRecord)
return [{}, convertedDecodedRecord.disconnect, true];
try {
const msg = proto.rootMsg.lookupType("usp.Msg");
const decodedMsg = msg.decode(decodedRecord.noSessionContext?.payload ||
decodedRecord.sessionContext?.payload[0]);
const convertedMsg = msg.toObject(decodedMsg, protobufToObjectOptions);
return [convertedMsg, null, false];
}
catch (err) {
console.log(JSON.stringify(err, null, 2));
return [{}, readableError(err), false];
}
};
export const decode = (parsedMsg, version) => {
const err = searchParent(parsedMsg, "errMsg") || null;
const command = extractCommand(parsedMsg);
const foundId = search(parsedMsg, "msgId");
// if id is formatted by me (command@) then use, otherwise check for sub id
const id = foundId.includes("@")
? foundId
: search(parsedMsg, "subscriptionId") || null;
// if command is unkown
if (!command)
return unkownErr(parsedMsg);
if (err)
return [id, null, err, command];
const cmd = getCommand(version, command);
if (!cmd)
return unkownErr(parsedMsg);
const [decodedData, decodedId, decodedErr] = cmd.decode(parsedMsg);
return [decodedId || id, decodedData, decodedErr || err, command];
};
export const decodeWithOptions = (parsedMsg, cmdType, options, version) => {
const cmd = getCommand(version, cmdType);
if (!cmd)
return unkownErr(parsedMsg);
if (options.raw)
return parsedMsg;
const [decodedData] = cmd.decode(parsedMsg, options);
return decodedData;
};
export const makeEmptySessionMessage = (proto, bufferOptions, sessionOptions, version, useSession) => {
const converted = _convert(proto, {});
if (isError(converted))
return null;
const encoded = proto.rootMsg
.lookupType("usp.Msg")
.encode(converted)
.finish();
const buffer = makeBuffer(proto.rootRecord, encoded, version, bufferOptions, sessionOptions, useSession);
return buffer;
};
export const makeEncode = (bufferOptions, useSession) => (proto, command, version, args, sessionOptions) => {
const cmd = getCommand(version, command);
if (!cmd)
return ["error", null, `Uknown command: ${command}`];
const msg = cmd.encode(args);
const id = msg.header.msgId;
msg.header.msgType = proto.header.MsgType[msg.header.msgType];
const converted = _convert(proto, msg);
if (isError(converted))
return [id, null, converted];
const encoded = proto.rootMsg
.lookupType("usp.Msg")
.encode(converted)
.finish();
const buffer = makeBuffer(proto.rootRecord, encoded, version, bufferOptions, sessionOptions, useSession);
return [id, buffer, null];
};
export const makeSession = (sessionId) => ({
sessionId: sessionId || 1,
sequenceId: 1,
expectedId: 1,
});
export const hasSessionDisconnectError = (msg) => msg?.disconnect?.reasonCode === 7105;
export const msgLacksPayload = (msg) => msg?.sessionContext?.payload[0] === "";
const isError = (o) => typeof o == "string";
const internalKeys = ["lookup"];
const isInternal = (key) => internalKeys.includes(key);
const makePayload = (items, isArr) => items
.filter(([k]) => !isInternal(k))
.reduce((acc, [k, v]) => isArr
? [...acc, v]
: {
...acc,
[k]: v,
}, isArr ? [] : {});
const isStringArray = (obj) => Array.isArray(obj) && obj.every((v) => typeof v === "string");
const needsConversion = (v) => typeof v === "object" && !isStringArray(v);
const _convert = (proto, value) => {
const skip = value.lookup === undefined;
const lookup = "usp." + value.lookup;
const item = skip ? null : proto.rootMsg.lookupType(lookup);
const simpleValues = Object.entries(value).filter(([, v]) => !needsConversion(v));
const toConvert = Object.entries(value).filter(([, v]) => needsConversion(v));
const converted = toConvert.map(([k, v]) => [
k,
_convert(proto, v),
]);
const err = converted.find(([, v]) => isError(v));
if (err)
return err[1];
const total = converted.concat(simpleValues);
const payload = makePayload(total, Array.isArray(value));
const payloadErr = item?.verify(payload);
if (payloadErr)
return payloadErr;
return item ? item.create(payload) : payload;
};
