/*
 * datamodel.c: USB datamodel handler
 *
 * Copyright (C) 2024 IOPSYS Software Solutions AB. All rights reserved.
 *
 *
 * See LICENSE file for license related information.
 */

#include "libbbfdm-api/dmcommon.h"

// Forward declarations
DMOBJ tUSBObj[];
DMLEAF tUSBParams[];
DMOBJ tUSBInterfaceObj[];
DMLEAF tUSBInterfaceParams[];
DMLEAF tUSBInterfaceStatsParams[];
DMLEAF tUSBPortParams[];
DMOBJ tUSBUSBHostsObj[];
DMLEAF tUSBUSBHostsParams[];
DMOBJ tUSBUSBHostsHostObj[];
DMLEAF tUSBUSBHostsHostParams[];
DMOBJ tUSBUSBHostsHostDeviceObj[];
DMLEAF tUSBUSBHostsHostDeviceParams[];
DMOBJ tUSBUSBHostsHostDeviceConfigurationObj[];
DMLEAF tUSBUSBHostsHostDeviceConfigurationParams[];
DMLEAF tUSBUSBHostsHostDeviceConfigurationInterfaceParams[];

#define SYSFS_USB_DEVICES_PATH "/sys/bus/usb/devices"

struct sysfs_dmsection {
	struct list_head list;
	char *sysfs_folder_path;
	char *sysfs_folder_name;
	struct uci_section *dmmap_section;
};

struct usb_port
{
	char *folder_name;
	char *folder_path;
};

struct usb_interface
{
	char *iface_name;
	char *iface_path;
	char *statistics_path;
	char *portlink;
};


/*************************************************************
* INIT
*************************************************************/
static void init_usb_port(const char *folder_name, const char *folder_path, struct usb_port *port)
{
	port->folder_name = dmstrdup(folder_name);
	port->folder_path = dmstrdup(folder_path);
}

static void init_usb_interface(const char *iface_name, const char *iface_path, const char *statistics_path, const char *portlink, struct usb_interface *iface)
{
	iface->iface_name = dmstrdup(iface_name);
	iface->iface_path = dmstrdup(iface_path);
	iface->portlink = dmstrdup(portlink);
	iface->statistics_path = dmstrdup(statistics_path);
}

/*************************************************************
* ENTRY METHOD
*************************************************************/
static void add_sysfs_sections_list(struct list_head *dup_list, struct uci_section *dmmap_section, char *file_name, char *file_path)
{
	struct sysfs_dmsection *dmmap_sysfs;

	dmmap_sysfs = dmcalloc(1, sizeof(struct sysfs_dmsection));
	list_add_tail(&dmmap_sysfs->list, dup_list);
	dmmap_sysfs->dmmap_section = dmmap_section;
	dmmap_sysfs->sysfs_folder_name = dmstrdup(file_name);
	dmmap_sysfs->sysfs_folder_path = dmstrdup(file_path);
}

void free_sysfs_sections_list(struct list_head *dup_list)
{
	struct sysfs_dmsection *dmmap_sysfs = NULL, *tmp = NULL;

	list_for_each_entry_safe(dmmap_sysfs, tmp, dup_list, list) {
		dmfree(dmmap_sysfs->sysfs_folder_name);
		dmfree(dmmap_sysfs->sysfs_folder_path);
		list_del(&dmmap_sysfs->list);
		dmfree(dmmap_sysfs);
	}
}

static int synchronize_system_folders_with_dmmap(const char *sysfsrep, const char *dmmap_package, const char *dmmap_section,
		const char *opt_name, struct list_head *dup_list)
{
	struct uci_section *s = NULL, *stmp = NULL, *dmmap_sect = NULL;
	char sysfs_rep_path[512];
	DIR *dir;
	struct dirent *ent;

	sysfs_foreach_file(sysfsrep, dir, ent) {
		if (strcmp(ent->d_name, ".") == 0 || strcmp(ent->d_name, "..") == 0)
			continue;

		/*
		 * create/update corresponding dmmap section that have same config_section link and using param_value_array
		 */
		snprintf(sysfs_rep_path, sizeof(sysfs_rep_path), "%s/%s", sysfsrep, ent->d_name);
		if ((dmmap_sect = get_dup_section_in_dmmap_opt(dmmap_package, dmmap_section, opt_name, sysfs_rep_path)) == NULL) {
			dmuci_add_section_bbfdm(dmmap_package, dmmap_section, &dmmap_sect);
			dmuci_set_value_by_section_bbfdm(dmmap_sect, opt_name, sysfs_rep_path);
		}

		/*
		 * Add system and dmmap sections to the list
		 */
		add_sysfs_sections_list(dup_list, dmmap_sect, ent->d_name, sysfs_rep_path);
	}
	if (dir)
		closedir(dir);

	/*
	 * Delete unused dmmap sections
	 */
	uci_path_foreach_sections_safe(bbfdm, dmmap_package, dmmap_section, stmp, s) {
		char *opt_val = NULL;

		dmuci_get_value_by_section_string(s, opt_name, &opt_val);
		if (!folder_exists(opt_val))
			dmuci_delete_by_section(s, NULL, NULL);
	}
	return 0;
}

static int read_sysfs_file(const char *file, char **value)
{
	char buf[128];
	int rc;

	rc =  dm_read_sysfs_file(file, buf, sizeof(buf));
	*value = dmstrdup(buf);

	return rc;
}

static int read_sysfs(const char *path, const char *name, char **value)
{
	char file[256];

	snprintf(file, sizeof(file), "%s/%s", path, name);
	return read_sysfs_file(file, value);
}

static int __read_sysfs(const char *path, const char *name, char *dst, unsigned len)
{
	char file[256];

	snprintf(file, sizeof(file), "%s/%s", path, name);
	return dm_read_sysfs_file(file, dst, len);
}

static int read_sysfs_usb_port(const struct usb_port *port, const char *name, char **value)
{
	char *val = NULL;
	int ret = 0;

	ret = read_sysfs(port->folder_path, name, &val);
	if (ret == 0 && DM_STRLEN(val)) {
		/* skip leading whitespaces */
		while (*val == ' ') {
			++val;
		}

		if (DM_STRLEN(val)) {
			*value = dmstrdup(val);
		}

		return 0;
	} else {
		return ret;
	}
}

static int read_sysfs_usb_iface(const struct usb_interface *iface, const char *name, char **value)
{
	return read_sysfs(iface->iface_path, name, value);
}

static int read_sysfs_usb_net_iface(const struct usb_interface *iface, const char *name, char **value)
{
	return get_net_device_sysfs(iface->iface_name, name, value);
}

static int __read_sysfs_usb_port(const struct usb_port *port, const char *name, char *dst, unsigned len)
{
	return __read_sysfs(port->folder_path, name, dst, len);
}

static int __read_sysfs_usb_iface(const struct usb_interface *iface, const char *name, char *dst, unsigned len)
{
	return __read_sysfs(iface->iface_path, name, dst, len);
}

static void writeFileContent(const char *filepath, const char *data)
{
	// cppcheck-suppress cert-MSC24-C
	FILE *fp = fopen(filepath, "ab");

	if (fp != NULL) {
		fputs(data, fp);
		fclose(fp);
	}
}

/*************************************************************
* ENTRY METHOD
**************************************************************/
static int browseUSBInterfaceInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	DIR *dir = NULL;
	struct dirent *ent = NULL;
	char *inst = NULL;
	size_t length;
	char **foldersplit;
	struct usb_interface iface = {0};
	LIST_HEAD(dup_list);
	struct sysfs_dmsection *p = NULL;
	struct dm_data data = {0};

	synchronize_system_folders_with_dmmap(SYSFS_USB_DEVICES_PATH, "dmmap_usb", "dmmap_interface", "usb_iface_link", &dup_list);
	list_for_each_entry(p, &dup_list, list) {
		char statistics_path[652] = {0};
		char iface_path[620] = {0};
		char netfolderpath[256] = {0};
		char iface_name[260] = {0};
		char port_link[128] = {0};

		snprintf(netfolderpath, sizeof(netfolderpath), "%s/%s/net", SYSFS_USB_DEVICES_PATH, p->sysfs_folder_name);
		if (!folder_exists(netfolderpath))
			continue;

		if (p->dmmap_section) {
			foldersplit= strsplit(p->sysfs_folder_name, ":", &length);
			snprintf(port_link, sizeof(port_link), "%s", foldersplit[0]);
		}
		sysfs_foreach_file(netfolderpath, dir, ent) {
			if(DM_LSTRCMP(ent->d_name, ".")==0 || DM_LSTRCMP(ent->d_name, "..")==0)
				continue;

			snprintf(iface_name, sizeof(iface_name), "%s", ent->d_name);
			break;
		}
		if (dir)
			closedir(dir);

		snprintf(iface_path, sizeof(iface_path), "%s/%s", netfolderpath, iface_name);
		if (p->dmmap_section)
			dmuci_set_value_by_section_bbfdm(p->dmmap_section, "usb_iface_path", iface_path);

		snprintf(statistics_path, sizeof(statistics_path), "%s/statistics", iface_path);
		init_usb_interface(iface_name, iface_path, statistics_path, port_link, &iface);

		inst = handle_instance(dmctx, parent_node, p->dmmap_section, "usb_iface_instance", "usb_iface_alias");

		memset(&data, 0, sizeof(struct dm_data));
		data.dmmap_section = p->dmmap_section;
		data.additional_data = (void *)&iface;

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&data, inst) == DM_STOP)
			break;
	}
	free_sysfs_sections_list(&dup_list);
	return 0;
}

static int browseUSBPortInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	char *inst = NULL;
	struct usb_port port = {0};
	struct sysfs_dmsection *p = NULL;
	LIST_HEAD(dup_list);
	regex_t regex1 = {0};
	regex_t regex2 = {0};
	struct dm_data data = {0};

	regcomp(&regex1, "^[0-9][0-9]*-[0-9]*[0-9]$", 0);
	regcomp(&regex2, "^[0-9][0-9]*-[0-9]*[0-9]\\.[0-9]*[0-9]$", 0);

	synchronize_system_folders_with_dmmap(SYSFS_USB_DEVICES_PATH, "dmmap_usb", "dmmap_port", "port_link", &dup_list);
	list_for_each_entry(p, &dup_list, list) {

		if (regexec(&regex1, p->sysfs_folder_name, 0, NULL, 0) != 0 &&
			regexec(&regex2, p->sysfs_folder_name, 0, NULL, 0) !=0 &&
			DM_LSTRSTR(p->sysfs_folder_name, "usb") != p->sysfs_folder_name)
			continue;

		init_usb_port(p->sysfs_folder_name, p->sysfs_folder_path, &port);

		inst = handle_instance(dmctx, parent_node, p->dmmap_section, "usb_port_instance", "usb_port_alias");

		memset(&data, 0, sizeof(struct dm_data));
		data.dmmap_section = p->dmmap_section;
		data.additional_data = (void *)&port;

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&data, inst) == DM_STOP)
			break;
	}
	free_sysfs_sections_list(&dup_list);
	regfree(&regex1);
	regfree(&regex2);
	return 0;
}

static int browseUSBUSBHostsHostInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct sysfs_dmsection *p = NULL;
	char *inst = NULL;
	struct usb_port port = {0};
	LIST_HEAD(dup_list);
	struct dm_data data = {0};

	synchronize_system_folders_with_dmmap(SYSFS_USB_DEVICES_PATH, "dmmap_usb", "dmmap_host", "port_link", &dup_list);

	list_for_each_entry(p, &dup_list, list) {

		if(!DM_LSTRSTR(p->sysfs_folder_name, "usb"))
			continue;

		init_usb_port(p->sysfs_folder_name, p->sysfs_folder_path, &port);

		inst = handle_instance(dmctx, parent_node, p->dmmap_section, "usb_host_instance", "usb_host_alias");

		memset(&data, 0, sizeof(struct dm_data));
		data.dmmap_section = p->dmmap_section;
		data.additional_data = (void *)&port;

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&data, inst) == DM_STOP)
			break;
	}
	free_sysfs_sections_list(&dup_list);
	return 0;
}

static int synchronize_usb_devices_with_dmmap_opt_recursively(const char *sysfsrep, const char *dmmap_package, const char *dmmap_section,
		const char *opt_name, const char *inst_opt, int is_root, struct list_head *dup_list)
{
	struct uci_section *s = NULL, *stmp = NULL, *dmmap_sect = NULL;
	DIR *dir = NULL;
	struct dirent *ent = NULL;
	char *v, *sysfs_repo_path, *instance = NULL;
	struct sysfs_dmsection *p = NULL;
	regex_t regex1 = {}, regex2 = {};

	regcomp(&regex1, "^[0-9][0-9]*-[0-9]*[0-9]$", 0);
	regcomp(&regex2, "^[0-9][0-9]*-[0-9]*[0-9]\\.[0-9]*[0-9]$", 0);

	LIST_HEAD(dup_list_no_inst);

	sysfs_foreach_file(sysfsrep, dir, ent) {
		if (DM_LSTRCMP(ent->d_name, ".") == 0 || DM_LSTRCMP(ent->d_name, "..") == 0)
			continue;

		if (regexec(&regex1, ent->d_name, 0, NULL, 0) == 0 || regexec(&regex2, ent->d_name, 0, NULL, 0) ==0) {
			char deviceClassFile[270];
			char deviceClass[16];

			snprintf(deviceClassFile, sizeof(deviceClassFile), "%s/%s/bDeviceClass", sysfsrep, ent->d_name);
			dm_read_sysfs_file(deviceClassFile, deviceClass, sizeof(deviceClass));

			if (DM_LSTRNCMP(deviceClass, "09", 2) == 0) {
				char hubpath[270];

				snprintf(hubpath, sizeof(hubpath), "%s/%s", sysfsrep, ent->d_name);
				synchronize_usb_devices_with_dmmap_opt_recursively(hubpath, dmmap_package, dmmap_section, opt_name, inst_opt, 0, dup_list);
			}
			/*
			 * create/update corresponding dmmap section that have same config_section link and using param_value_array
			 */
			dmasprintf(&sysfs_repo_path, "%s/%s", sysfsrep, ent->d_name);
			if ((dmmap_sect = get_dup_section_in_dmmap_opt(dmmap_package, dmmap_section, opt_name, sysfs_repo_path)) == NULL) {
				dmuci_add_section_bbfdm(dmmap_package, dmmap_section, &dmmap_sect);
				dmuci_set_value_by_section_bbfdm(dmmap_sect, opt_name, sysfs_repo_path);
			}
			dmuci_get_value_by_section_string(dmmap_sect, inst_opt, &instance);
			/*
			 * Add system and dmmap sections to the list
			 */
			if (instance == NULL || *instance == '\0')
				add_sysfs_sections_list(&dup_list_no_inst, dmmap_sect, ent->d_name, sysfs_repo_path);
			else
				add_sysfs_sections_list(dup_list, dmmap_sect, ent->d_name, sysfs_repo_path);
		}
	}
	if (dir)
		closedir(dir);
	regfree(&regex1);
	regfree(&regex2);
	/*
	 * fusion two lists
	 */
	list_for_each_entry(p, &dup_list_no_inst, list) {
		add_sysfs_sections_list(dup_list, p->dmmap_section, p->sysfs_folder_name, p->sysfs_folder_path);
	}
	/*
	 * Delete unused dmmap sections
	 */
	if (is_root) {
		uci_path_foreach_sections_safe(bbfdm, dmmap_package, dmmap_section, stmp, s) {
			dmuci_get_value_by_section_string(s, opt_name, &v);
			if (!folder_exists(v)) {
				dmuci_delete_by_section(s, NULL, NULL);
			}
		}
	}
	return 0;
}

static int browseUSBUSBHostsHostDeviceInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct sysfs_dmsection *p = NULL;
	char *inst = NULL, *parent_host_instance = NULL;
	struct dm_data *prev_node = (struct dm_data *)prev_data;
	struct usb_port *prev_port = (struct usb_port *)(prev_node->additional_data);
	LIST_HEAD(dup_list);
	struct dm_data data = {0};


	synchronize_usb_devices_with_dmmap_opt_recursively(prev_port->folder_path,
		"dmmap_usb", "dmmap_host_device", "port_link", "usb_host_device_instance", 1, &dup_list);

	list_for_each_entry(p, &dup_list, list) {
		struct usb_port port= {};

		init_usb_port(p->sysfs_folder_name, p->sysfs_folder_path, &port);

		if (p->dmmap_section && prev_node->dmmap_section ) {
			dmuci_get_value_by_section_string(prev_node->dmmap_section, "usb_host_instance", &parent_host_instance);
			dmuci_set_value_by_section_bbfdm(p->dmmap_section, "usb_host_device_parent_host_instance", parent_host_instance);
		}

		inst = handle_instance(dmctx, parent_node, p->dmmap_section, "usb_host_device_instance", "usb_host_device_alias");

		memset(&data, 0, sizeof(struct dm_data));
		data.dmmap_section = p->dmmap_section;
		data.additional_data = (void *)&port;

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&data, inst) == DM_STOP)
			break;
	}
	free_dmmap_config_dup_list(&dup_list);
	return 0;
}

static int browseUSBUSBHostsHostDeviceConfigurationInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct dm_data *prev_node = (struct dm_data *)prev_data;
	const struct usb_port *usb_dev = (struct usb_port *)(prev_node->additional_data);
	struct usb_port port = {};
	struct uci_section *s = NULL;
	char nbre[16];
	struct dm_data data = {0};

	__read_sysfs_usb_port(usb_dev, "bNumConfigurations", nbre, sizeof(nbre));
	if(nbre[0] == '0')
		return 0;

	s = is_dmmap_section_exist("dmmap_usb", "usb_device_conf");
	if (!s)
		dmuci_add_section_bbfdm("dmmap_usb", "usb_device_conf", &s);
	dmuci_set_value_by_section_bbfdm(s, "usb_parent_device", usb_dev->folder_path);

	init_usb_port(usb_dev->folder_name, usb_dev->folder_path, &port);

	handle_instance(dmctx, parent_node, s, "usb_device_conf_instance", "usb_device_conf_alias");

	memset(&data, 0, sizeof(struct dm_data));
	data.dmmap_section = s;
	data.additional_data = (void *)&port;

	DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&data, "1");
	return 0;
}

static int browseUSBUSBHostsHostDeviceConfigurationInterfaceInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	DIR *dir = NULL;
	struct dirent *ent = NULL;
	struct dm_data *prev_node = (struct dm_data *)prev_data;
	struct usb_port *usb_dev = (struct usb_port*)(prev_node->additional_data);
	struct usb_port port = {0};
	char *sysfs_rep_path, *inst = NULL;
	struct uci_section *dmmap_sect = NULL;
	regex_t regex1 = {};
	regex_t regex2 = {};
	struct dm_data data = {0};

	regcomp(&regex1, "^[0-9][0-9]*-[0-9]*[0-9]:[0-9][0-9]*\\.[0-9]*[0-9]$", 0);
	regcomp(&regex2, "^[0-9][0-9]*-[0-9]*[0-9]\\.[0-9]*[0-9]:[0-9][0-9]*\\.[0-9]*[0-9]$", 0);

	sysfs_foreach_file(usb_dev->folder_path, dir, ent) {

		if (DM_LSTRCMP(ent->d_name, ".") == 0 || DM_LSTRCMP(ent->d_name, "..") == 0)
			continue;

		if (regexec(&regex1, ent->d_name, 0, NULL, 0) == 0 || regexec(&regex2, ent->d_name, 0, NULL, 0) == 0) {
			dmasprintf(&sysfs_rep_path, "%s/%s", usb_dev->folder_path, ent->d_name);
			if ((dmmap_sect = get_dup_section_in_dmmap_opt("dmmap_usb", "usb_device_conf_interface", "port_link", sysfs_rep_path)) == NULL) {
				dmuci_add_section_bbfdm("dmmap_usb", "usb_device_conf_interface", &dmmap_sect);
				dmuci_set_value_by_section_bbfdm(dmmap_sect, "port_link", sysfs_rep_path);
			}

			init_usb_port(ent->d_name, sysfs_rep_path, &port);

			inst = handle_instance(dmctx, parent_node, dmmap_sect, "usb_device_conf_iface_instance", "usb_device_conf_iface_alias");

			memset(&data, 0, sizeof(struct dm_data));
			data.dmmap_section = dmmap_sect;
			data.additional_data = (void *)&port;

			if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&data, inst) == DM_STOP)
				break;
		}
	}
	if (dir)
		closedir(dir);
	regfree(&regex1);
	regfree(&regex2);
	return 0;
}

/*************************************************************
* GET & SET PARAM
**************************************************************/
static int get_USB_InterfaceNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseUSBInterfaceInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_USB_PortNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseUSBPortInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_USBInterface_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char carrier[8];
	struct dm_data *p = (struct dm_data *)data;

	__read_sysfs_usb_iface(p->additional_data, "carrier", carrier, sizeof(carrier));

	if (carrier[0] == '1')
		*value = dmstrdup("1");
	else
		*value = dmstrdup("0");
	return 0;
}

static int set_USBInterface_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			break;
	}
	return 0;
}

static int get_USBInterface_Status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char carrier[8];
	struct dm_data *p = (struct dm_data *)data;

	__read_sysfs_usb_iface(p->additional_data, "carrier", carrier, sizeof(carrier));

	if (carrier[0] == '1')
		*value = dmstrdup("Up");
	else
		*value = dmstrdup("Down");
	return 0;
}

static int get_USBInterface_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return bbf_get_alias(ctx, p->dmmap_section, "usb_iface_alias", instance, value);
}

static int set_USBInterface_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct dm_data *p = (struct dm_data *)data;
	return bbf_set_alias(ctx, p->dmmap_section, "usb_iface_alias", instance, value);
}

static int get_USBInterface_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	struct usb_interface *usbiface= (struct usb_interface *)(p->additional_data);

	dmasprintf(value, "%s", usbiface->iface_name);
	return 0;
}

static int get_USBInterface_LastChange(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("0");
	return 0;
}

static int get_USBInterface_LowerLayers(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	struct usb_interface *usbiface= (struct usb_interface *)(p->additional_data);

	_bbfdm_get_references(ctx, "Device.Ethernet.Interface.", "Name", usbiface->iface_name, value);
	return 0;
}

static int get_USBInterface_Upstream(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("0");
	return 0;
}

static int set_USBInterface_LowerLayers(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string_list(ctx, value, -1, -1, 1024, -1, -1, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			break;
	}
	return 0;
}

static int get_USBInterface_MACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_iface(p->additional_data, "address", value);
}

static int get_USBInterface_MaxBitRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_iface(p->additional_data, "queues/tx-0/tx_maxrate", value);
}

static int get_USBInterface_Port(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	struct usb_interface *iface = (struct usb_interface *)p->additional_data;

	_bbfdm_get_references(ctx, "Device.USB.Port.", "Name", iface->portlink, value);
	return 0;
}

static int get_USBInterfaceStats_BytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_net_iface(p->additional_data, "statistics/tx_bytes", value);
}

static int get_USBInterfaceStats_BytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_net_iface(p->additional_data, "statistics/rx_bytes", value);
}

static int get_USBInterfaceStats_PacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_net_iface(p->additional_data, "statistics/tx_packets", value);
}

static int get_USBInterfaceStats_PacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_net_iface(p->additional_data, "statistics/rx_packets", value);
}

static int get_USBInterfaceStats_ErrorsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_net_iface(p->additional_data, "statistics/tx_errors", value);
}

static int get_USBInterfaceStats_ErrorsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_net_iface(p->additional_data, "statistics/rx_errors", value);
}

static int get_USBInterfaceStats_UnicastPacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("0");
	return 0;
}

static int get_USBInterfaceStats_UnicastPacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("0");
	return 0;
}

static int get_USBInterfaceStats_DiscardPacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_net_iface(p->additional_data, "statistics/tx_dropped", value);
}

static int get_USBInterfaceStats_DiscardPacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_net_iface(p->additional_data, "statistics/rx_dropped", value);
}

static int get_USBInterfaceStats_MulticastPacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("0");
	return 0;
}

static int get_USBInterfaceStats_MulticastPacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_net_iface(p->additional_data, "statistics/multicast", value);
}

static int get_USBInterfaceStats_BroadcastPacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("0");
	return 0;
}

static int get_USBInterfaceStats_BroadcastPacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("0");
	return 0;
}

static int get_USBInterfaceStats_UnknownProtoPacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_net_iface(p->additional_data, "statistics/rx_unknown_packets", value);
}

static int get_USBPort_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return bbf_get_alias(ctx, p->dmmap_section, "usb_port_alias", instance, value);
}

static int set_USBPort_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct dm_data *p = (struct dm_data *)data;
	return bbf_set_alias(ctx, p->dmmap_section, "usb_port_alias", instance, value);
}

static int get_USBPort_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	const struct usb_port *port = (struct usb_port *)(p->additional_data);
	*value = dmstrdup(port->folder_name);
	return 0;
}

static int get_USBPort_Standard(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char buf[16];
	struct dm_data *p = (struct dm_data *)data;

	__read_sysfs_usb_port(p->additional_data, "bcdDevice", buf, sizeof(buf));
	dmasprintf(value, "%c.%c", buf[0], buf[0]);
	return 0;
}

static int get_USBPort_Type(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	const struct usb_port *port = (struct usb_port *)(p->additional_data);
	char deviceclass[32];

	__read_sysfs_usb_port(port, "bDeviceClass", deviceclass, sizeof(deviceclass));

	if(DM_LSTRSTR(port->folder_name, "usb") == port->folder_name)
		*value = dmstrdup("Host");
	else if (DM_LSTRCMP(deviceclass, "09") == 0)
		*value = dmstrdup("Hub");
	else
		*value = dmstrdup("Device");
	return 0;
}

static int get_USBPort_Receptacle(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("Standard-A");
	return 0;
}

static int get_USBPort_Rate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char speed[16];
	struct dm_data *p = (struct dm_data *)data;

	__read_sysfs_usb_port(p->additional_data, "speed", speed, sizeof(speed));

	if(DM_LSTRCMP(speed, "1.5") == 0)
		*value = dmstrdup("Low");
	else if(DM_LSTRCMP(speed, "12") == 0)
		*value = dmstrdup("Full");
	else if(DM_LSTRCMP(speed, "480") == 0)
		*value = dmstrdup("High");
	else
		*value = dmstrdup("Super");
	return 0;
}

static int get_USBPort_Power(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char pwrctl[16];
	struct dm_data *p = (struct dm_data *)data;

	__read_sysfs_usb_port(p->additional_data, "power/control", pwrctl, sizeof(pwrctl));

	if (pwrctl[0] == 0)
		*value = dmstrdup("Unknown");
	else if (!DM_LSTRCMP(pwrctl, "auto"))
		*value = dmstrdup("Self");
	else
		*value = dmstrdup("Bus");

	return 0;
}

static int get_USBUSBHosts_HostNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseUSBUSBHostsHostInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_USBUSBHostsHost_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return bbf_get_alias(ctx, p->dmmap_section, "usb_host_alias", instance, value);
}

static int set_USBUSBHostsHost_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct dm_data *p = (struct dm_data *)data;
	return bbf_set_alias(ctx, p->dmmap_section, "usb_host_alias", instance, value);
}

static int get_USBUSBHostsHost_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char up[32];
	struct dm_data *p = (struct dm_data *)data;

	__read_sysfs_usb_port(p->additional_data, "power/wakeup", up, sizeof(up));
	*value = DM_LSTRCMP(up, "enabled") == 0 ? dmstrdup("1") : dmstrdup("0");
	return 0;
}

static int set_USBUSBHostsHost_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct dm_data *p = (struct dm_data *)data;
	struct usb_port *usbhost= (struct usb_port *)(p->additional_data);
	bool b;
	char *filepath;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmasprintf(&filepath, "%s/power/wakeup", usbhost->folder_path);
			if(b)
				writeFileContent(filepath, "enabled");
			else
				writeFileContent(filepath, "disabled");
			break;
	}
	return 0;
}

static int get_USBUSBHostsHost_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	struct usb_port* port = (struct usb_port *)(p->additional_data);
	dmasprintf(value, "%s", port->folder_name);
	return 0;
}

static int get_USBUSBHostsHost_Type(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	char serial[64];

	__read_sysfs_usb_port(p->additional_data, "serial", serial, sizeof(serial));

	if(strcasestr(serial, "ohci")!=NULL)
		*value = dmstrdup("OHCI");
	else if(strcasestr(serial, "ehci")!=NULL)
		*value = dmstrdup("EHCI");
	else if(strcasestr(serial, "uhci")!=NULL)
		*value = dmstrdup("UHCI");
	else
		*value = dmstrdup("xHCI");
	return 0;
}

static int get_USBUSBHostsHost_Reset(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("0");
	return 0;
}

static int operate_USBUSBHostsHost_Reset(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct dm_data *p = (struct dm_data *)data;
	struct usb_port *usbhost= (struct usb_port *)(p->additional_data);
	char *filepath;

	dmasprintf(&filepath, "%s/authorized", usbhost->folder_path);
	writeFileContent(filepath, "0");
	writeFileContent(filepath, "1");

	return 0;
}

static int set_USBUSBHostsHost_Reset(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			if (b) {
				 operate_USBUSBHostsHost_Reset(refparam, ctx, data, instance, value, action);
			}

			break;
	}
	return 0;
}

static int get_USBUSBHostsHost_PowerManagementEnable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	char power[64] = {0};

	__read_sysfs_usb_port(p->additional_data, "power/level", power, sizeof(power));

	if(power[0] == 0 || DM_LSTRCMP(power, "suspend") == 0)
		*value = dmstrdup("false");
	else
		*value = dmstrdup("true");

	return 0;
}

static int set_USBUSBHostsHost_PowerManagementEnable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct dm_data *p = (struct dm_data *)data;
	struct usb_port *host= (struct usb_port *)(p->additional_data);
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			char *filepath;
			dmasprintf(&filepath, "%s/power/level", host->folder_path);
			if (!file_exists(filepath))
				break;
			writeFileContent(filepath, b?"on":"suspend");
			break;
	}
	return 0;
}

static int get_USBUSBHostsHost_USBVersion(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	const struct usb_port *port = (struct usb_port *)(p->additional_data);
	char file[256];
	char buf[16] = { 0, 0 };

	snprintf(file, sizeof(file), "%s/bcdDevice", port->folder_path);
	dm_read_sysfs_file(file, buf, sizeof(buf));

	dmasprintf(value, "%c.%c", buf[1], buf[2]);
	return 0;
}

static int get_USBUSBHostsHost_DeviceNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseUSBUSBHostsHostDeviceInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_USBUSBHostsHostDevice_DeviceNumber(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	struct usb_port *usbdev = (struct usb_port *)(p->additional_data);
	size_t length;
	char **filename = strsplit(usbdev->folder_name, "-", &length);
	char **port = strsplit(filename[1], ".", &length);
	dmasprintf(value ,"%s", port[0]);
	return 0;
}

static int get_USBUSBHostsHostDevice_USBVersion(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_port(p->additional_data, "version", value);
}

static int get_USBUSBHostsHostDevice_DeviceClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_port(p->additional_data, "bDeviceClass", value);
}

static int get_USBUSBHostsHostDevice_DeviceSubClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_port(p->additional_data, "bDeviceSubClass", value);
}

static int get_USBUSBHostsHostDevice_DeviceVersion(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_port(p->additional_data, "bcdDevice", value);
}

static int get_USBUSBHostsHostDevice_DeviceProtocol(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_port(p->additional_data, "bDeviceProtocol", value);
}

static int get_USBUSBHostsHostDevice_ProductID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *idproduct = NULL;
	unsigned int ui_idproduct;
	struct dm_data *p = (struct dm_data *)data;

	*value = dmstrdup("0");
	int rc = read_sysfs_usb_port(p->additional_data, "idProduct", &idproduct);

	if (rc != -1 && idproduct != NULL) {
		sscanf(idproduct, "%x", &ui_idproduct);
		dmasprintf(value, "%u", ui_idproduct);
	}
	return rc;
}

static int get_USBUSBHostsHostDevice_VendorID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *idvendor = NULL;
	unsigned int ui_idvendor;
	struct dm_data *p = (struct dm_data *)data;

	*value = dmstrdup("0");
	int rc = read_sysfs_usb_port(p->additional_data, "idVendor", &idvendor);

	if (rc != -1 && idvendor != NULL) {
		sscanf(idvendor, "%x", &ui_idvendor);
		dmasprintf(value, "%u", ui_idvendor);
	}
	return rc;
}

static int get_USBUSBHostsHostDevice_Manufacturer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_port(p->additional_data, "manufacturer", value);
}

static int get_USBUSBHostsHostDevice_ProductClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_port(p->additional_data, "product", value);
}

static int get_USBUSBHostsHostDevice_SerialNumber(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_port(p->additional_data, "urbnum", value);
}

static int get_USBUSBHostsHostDevice_Port(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	struct usb_port *port= (struct usb_port *)(p->additional_data);
	size_t length;
	char **busname, **portname;
	regex_t regex1 = {};
	regex_t regex2 = {};

	regcomp(&regex1, "^[0-9][0-9]*-[0-9]*[0-9]$", 0);
	regcomp(&regex2, "^[0-9][0-9]*-[0-9]*[0-9]\\.[0-9]*[0-9]$", 0);
	if (regexec(&regex1, port->folder_name, 0, NULL, 0) == 0 || regexec(&regex2, port->folder_name, 0, NULL, 0) == 0) {
		busname = strsplit(port->folder_name, "-", &length);
		portname = strsplit(busname[1], ".", &length);
		*value = dmstrdup(portname[0]);
		goto out;
	}
	*value = dmstrdup("0");
out:
	regfree(&regex1);
	regfree(&regex2);
	return 0;
}

static int get_USBUSBHostsHostDevice_USBPort(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	_bbfdm_get_references(ctx, "Device.USB.Port.", "Name", ((struct usb_port *)(p->additional_data))->folder_name, value);
	return 0;
}

static int get_USBUSBHostsHostDevice_Rate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return get_USBPort_Rate(refparam, ctx, data, instance, value);
}

static int get_USBUSBHostsHostDevice_Parent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	struct usb_port *usbdev = (struct usb_port*)(p->additional_data);
	char *host_inst, usb_host_path[128] = {0};
	regex_t regex1 = {0};
	size_t length;

	regcomp(&regex1, "^[0-9][0-9]*-[0-9]*[0-9]\\.[0-9]*[0-9]$", 0);
	if (regexec(&regex1, usbdev->folder_name, 0, NULL, 0) != 0 || p->dmmap_section == NULL) {
		*value = dmstrdup("");
		goto out;
	}

	char **filename = strsplit(usbdev->folder_name, "-", &length);
	char **port = strsplit(filename[1], ".", &length);
	if (!port || port[0] == NULL) {
		*value = dmstrdup("");
		goto out;
	}

	dmuci_get_value_by_section_string(p->dmmap_section, "usb_host_instance", &host_inst);
	snprintf(usb_host_path, sizeof(usb_host_path), "Device.USB.USBHosts.Host.%s.Device.", host_inst);

	_bbfdm_get_references(ctx, usb_host_path, "DeviceNumber", port[0], value);

out:
	regfree(&regex1);
	return 0;
}

static int get_USBUSBHostsHostDevice_MaxChildren(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_port(p->additional_data, "maxchild", value);
}

static int get_USBUSBHostsHostDevice_IsSuspended(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char status[16] = {0};
	struct dm_data *p = (struct dm_data *)data;

	__read_sysfs_usb_port(p->additional_data, "power/runtime_status", status, sizeof(status));
	if(DM_LSTRNCMP(status, "suspended", 9) == 0)
		*value = dmstrdup("1");
	else
		*value = dmstrdup("0");
	return 0;
}

static int get_USBUSBHostsHostDevice_IsSelfPowered(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *attr = NULL;
	struct dm_data *p = (struct dm_data *)data;
	read_sysfs_usb_port(p->additional_data, "bmAttribute", &attr);

	*value = dmstrdup("0");
	if (DM_STRLEN(attr)) {
		unsigned long val = 0;
		val = strtoul(attr, NULL, 16);

		/* now extract the value at 6th bit(from 0-7) */
		unsigned long bits = (1 << 6);

		val = val & bits;

		*value = val ? dmstrdup("1") : dmstrdup("0");
	}
	return 0;
}

static int get_USBUSBHostsHostDevice_ConfigurationNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_port(p->additional_data, "bNumConfigurations", value);
}

static int get_USBUSBHostsHostDeviceConfiguration_ConfigurationNumber(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_port(p->additional_data, "bConfigurationValue", value);
}

static int get_USBUSBHostsHostDeviceConfiguration_InterfaceNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_port(p->additional_data, "bNumInterfaces", value);
}

static int get_USBUSBHostsHostDeviceConfigurationInterface_InterfaceNumber(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_port(p->additional_data, "bInterfaceNumber", value);
}

static int get_USBUSBHostsHostDeviceConfigurationInterface_InterfaceClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_port(p->additional_data, "bInterfaceClass", value);
}

static int get_USBUSBHostsHostDeviceConfigurationInterface_InterfaceSubClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_port(p->additional_data, "bInterfaceSubClass", value);
}

static int get_USBUSBHostsHostDeviceConfigurationInterface_InterfaceProtocol(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct dm_data *p = (struct dm_data *)data;
	return read_sysfs_usb_port(p->additional_data, "bInterfaceProtocol", value);
}

/**********************************************************************************************************************************
*                                            OBJ & PARAM DEFINITION
***********************************************************************************************************************************/
/* *** Device.USB. *** */
DMOBJ tUSBObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Interface", &DMREAD, NULL, NULL, NULL, browseUSBInterfaceInst, NULL, NULL, tUSBInterfaceObj, tUSBInterfaceParams, NULL, BBFDM_BOTH, NULL},
{"Port", &DMREAD, NULL, NULL, NULL, browseUSBPortInst, NULL, NULL, NULL, tUSBPortParams, NULL, BBFDM_BOTH, NULL},
{"USBHosts", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tUSBUSBHostsObj, tUSBUSBHostsParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tUSBParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"InterfaceNumberOfEntries", &DMREAD, DMT_UNINT, get_USB_InterfaceNumberOfEntries, NULL, BBFDM_BOTH},
{"PortNumberOfEntries", &DMREAD, DMT_UNINT, get_USB_PortNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.USB.Interface.{i}. *** */
DMOBJ tUSBInterfaceObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Stats", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tUSBInterfaceStatsParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tUSBInterfaceParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_USBInterface_Enable, set_USBInterface_Enable, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_USBInterface_Status, NULL, BBFDM_BOTH},
{"Alias", &DMWRITE, DMT_STRING, get_USBInterface_Alias, set_USBInterface_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Name", &DMREAD, DMT_STRING, get_USBInterface_Name, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"LastChange", &DMREAD, DMT_UNINT, get_USBInterface_LastChange, NULL, BBFDM_BOTH},
{"LowerLayers", &DMWRITE, DMT_STRING, get_USBInterface_LowerLayers, set_USBInterface_LowerLayers, BBFDM_BOTH, DM_FLAG_REFERENCE},
{"Upstream", &DMREAD, DMT_BOOL, get_USBInterface_Upstream, NULL, BBFDM_BOTH},
{"MACAddress", &DMREAD, DMT_STRING, get_USBInterface_MACAddress, NULL, BBFDM_BOTH},
{"MaxBitRate", &DMREAD, DMT_UNINT, get_USBInterface_MaxBitRate, NULL, BBFDM_BOTH},
{"Port", &DMREAD, DMT_STRING, get_USBInterface_Port, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.USB.Interface.{i}.Stats. *** */
DMLEAF tUSBInterfaceStatsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"BytesSent", &DMREAD, DMT_UNLONG, get_USBInterfaceStats_BytesSent, NULL, BBFDM_BOTH},
{"BytesReceived", &DMREAD, DMT_UNLONG, get_USBInterfaceStats_BytesReceived, NULL, BBFDM_BOTH},
{"PacketsSent", &DMREAD, DMT_UNLONG, get_USBInterfaceStats_PacketsSent, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNLONG, get_USBInterfaceStats_PacketsReceived, NULL, BBFDM_BOTH},
{"ErrorsSent", &DMREAD, DMT_UNINT, get_USBInterfaceStats_ErrorsSent, NULL, BBFDM_BOTH},
{"ErrorsReceived", &DMREAD, DMT_UNINT, get_USBInterfaceStats_ErrorsReceived, NULL, BBFDM_BOTH},
{"UnicastPacketsSent", &DMREAD, DMT_UNLONG, get_USBInterfaceStats_UnicastPacketsSent, NULL, BBFDM_BOTH},
{"UnicastPacketsReceived", &DMREAD, DMT_UNLONG, get_USBInterfaceStats_UnicastPacketsReceived, NULL, BBFDM_BOTH},
{"DiscardPacketsSent", &DMREAD, DMT_UNINT, get_USBInterfaceStats_DiscardPacketsSent, NULL, BBFDM_BOTH},
{"DiscardPacketsReceived", &DMREAD, DMT_UNINT, get_USBInterfaceStats_DiscardPacketsReceived, NULL, BBFDM_BOTH},
{"MulticastPacketsSent", &DMREAD, DMT_UNLONG, get_USBInterfaceStats_MulticastPacketsSent, NULL, BBFDM_BOTH},
{"MulticastPacketsReceived", &DMREAD, DMT_UNLONG, get_USBInterfaceStats_MulticastPacketsReceived, NULL, BBFDM_BOTH},
{"BroadcastPacketsSent", &DMREAD, DMT_UNLONG, get_USBInterfaceStats_BroadcastPacketsSent, NULL, BBFDM_BOTH},
{"BroadcastPacketsReceived", &DMREAD, DMT_UNLONG, get_USBInterfaceStats_BroadcastPacketsReceived, NULL, BBFDM_BOTH},
{"UnknownProtoPacketsReceived", &DMREAD, DMT_UNINT, get_USBInterfaceStats_UnknownProtoPacketsReceived, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.USB.Port.{i}. *** */
DMLEAF tUSBPortParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Alias", &DMWRITE, DMT_STRING, get_USBPort_Alias, set_USBPort_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Name", &DMREAD, DMT_STRING, get_USBPort_Name, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Standard", &DMREAD, DMT_STRING, get_USBPort_Standard, NULL, BBFDM_BOTH},
{"Type", &DMREAD, DMT_STRING, get_USBPort_Type, NULL, BBFDM_BOTH},
{"Receptacle", &DMREAD, DMT_STRING, get_USBPort_Receptacle, NULL, BBFDM_BOTH},
{"Rate", &DMREAD, DMT_STRING, get_USBPort_Rate, NULL, BBFDM_BOTH},
{"Power", &DMREAD, DMT_STRING, get_USBPort_Power, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.USB.USBHosts. *** */
DMOBJ tUSBUSBHostsObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Host", &DMREAD, NULL, NULL, NULL, browseUSBUSBHostsHostInst, NULL, NULL, tUSBUSBHostsHostObj, tUSBUSBHostsHostParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tUSBUSBHostsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"HostNumberOfEntries", &DMREAD, DMT_UNINT, get_USBUSBHosts_HostNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.USB.USBHosts.Host.{i}. *** */
DMOBJ tUSBUSBHostsHostObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Device", &DMREAD, NULL, NULL, NULL, browseUSBUSBHostsHostDeviceInst, NULL, NULL, tUSBUSBHostsHostDeviceObj, tUSBUSBHostsHostDeviceParams, NULL, BBFDM_BOTH},
{0}
};

DMLEAF tUSBUSBHostsHostParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Alias", &DMWRITE, DMT_STRING, get_USBUSBHostsHost_Alias, set_USBUSBHostsHost_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Enable", &DMWRITE, DMT_BOOL, get_USBUSBHostsHost_Enable, set_USBUSBHostsHost_Enable, BBFDM_BOTH},
{"Name", &DMREAD, DMT_STRING, get_USBUSBHostsHost_Name, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Type", &DMREAD, DMT_STRING, get_USBUSBHostsHost_Type, NULL, BBFDM_BOTH},
{"Reset", &DMWRITE, DMT_BOOL, get_USBUSBHostsHost_Reset, set_USBUSBHostsHost_Reset, BBFDM_CWMP},
{"Reset()", &DMSYNC, DMT_COMMAND, NULL, operate_USBUSBHostsHost_Reset, BBFDM_USP},
{"PowerManagementEnable", &DMWRITE, DMT_BOOL, get_USBUSBHostsHost_PowerManagementEnable, set_USBUSBHostsHost_PowerManagementEnable, BBFDM_BOTH},
{"USBVersion", &DMREAD, DMT_STRING, get_USBUSBHostsHost_USBVersion, NULL, BBFDM_BOTH},
{"DeviceNumberOfEntries", &DMREAD, DMT_UNINT, get_USBUSBHostsHost_DeviceNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.USB.USBHosts.Host.{i}.Device.{i}. *** */
DMOBJ tUSBUSBHostsHostDeviceObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Configuration", &DMREAD, NULL, NULL, NULL, browseUSBUSBHostsHostDeviceConfigurationInst, NULL, NULL, tUSBUSBHostsHostDeviceConfigurationObj, tUSBUSBHostsHostDeviceConfigurationParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tUSBUSBHostsHostDeviceParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"DeviceNumber", &DMREAD, DMT_UNINT, get_USBUSBHostsHostDevice_DeviceNumber, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"USBVersion", &DMREAD, DMT_STRING, get_USBUSBHostsHostDevice_USBVersion, NULL, BBFDM_BOTH},
{"DeviceClass", &DMREAD, DMT_HEXBIN, get_USBUSBHostsHostDevice_DeviceClass, NULL, BBFDM_BOTH},
{"DeviceSubClass", &DMREAD, DMT_HEXBIN, get_USBUSBHostsHostDevice_DeviceSubClass, NULL, BBFDM_BOTH},
{"DeviceVersion", &DMREAD, DMT_UNINT, get_USBUSBHostsHostDevice_DeviceVersion, NULL, BBFDM_BOTH},
{"DeviceProtocol", &DMREAD, DMT_HEXBIN, get_USBUSBHostsHostDevice_DeviceProtocol, NULL, BBFDM_BOTH},
{"ProductID", &DMREAD, DMT_UNINT, get_USBUSBHostsHostDevice_ProductID, NULL, BBFDM_BOTH},
{"VendorID", &DMREAD, DMT_UNINT, get_USBUSBHostsHostDevice_VendorID, NULL, BBFDM_BOTH},
{"Manufacturer", &DMREAD, DMT_STRING, get_USBUSBHostsHostDevice_Manufacturer, NULL, BBFDM_BOTH},
{"ProductClass", &DMREAD, DMT_STRING, get_USBUSBHostsHostDevice_ProductClass, NULL, BBFDM_BOTH},
{"SerialNumber", &DMREAD, DMT_STRING, get_USBUSBHostsHostDevice_SerialNumber, NULL, BBFDM_BOTH},
{"Port", &DMREAD, DMT_UNINT, get_USBUSBHostsHostDevice_Port, NULL, BBFDM_BOTH},
{"USBPort", &DMREAD, DMT_STRING, get_USBUSBHostsHostDevice_USBPort, NULL, BBFDM_BOTH, DM_FLAG_REFERENCE},
{"Rate", &DMREAD, DMT_STRING, get_USBUSBHostsHostDevice_Rate, NULL, BBFDM_BOTH},
{"Parent", &DMREAD, DMT_STRING, get_USBUSBHostsHostDevice_Parent, NULL, BBFDM_BOTH, DM_FLAG_REFERENCE},
{"MaxChildren", &DMREAD, DMT_UNINT, get_USBUSBHostsHostDevice_MaxChildren, NULL, BBFDM_BOTH},
{"IsSuspended", &DMREAD, DMT_BOOL, get_USBUSBHostsHostDevice_IsSuspended, NULL, BBFDM_BOTH},
{"IsSelfPowered", &DMREAD, DMT_BOOL, get_USBUSBHostsHostDevice_IsSelfPowered, NULL, BBFDM_BOTH},
{"ConfigurationNumberOfEntries", &DMREAD, DMT_UNINT, get_USBUSBHostsHostDevice_ConfigurationNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.USB.USBHosts.Host.{i}.Device.{i}.Configuration.{i}. *** */
DMOBJ tUSBUSBHostsHostDeviceConfigurationObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Interface", &DMREAD, NULL, NULL, NULL, browseUSBUSBHostsHostDeviceConfigurationInterfaceInst, NULL, NULL, NULL, tUSBUSBHostsHostDeviceConfigurationInterfaceParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tUSBUSBHostsHostDeviceConfigurationParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"ConfigurationNumber", &DMREAD, DMT_UNINT, get_USBUSBHostsHostDeviceConfiguration_ConfigurationNumber, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"InterfaceNumberOfEntries", &DMREAD, DMT_UNINT, get_USBUSBHostsHostDeviceConfiguration_InterfaceNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.USB.USBHosts.Host.{i}.Device.{i}.Configuration.{i}.Interface.{i}. *** */
DMLEAF tUSBUSBHostsHostDeviceConfigurationInterfaceParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"InterfaceNumber", &DMREAD, DMT_UNINT, get_USBUSBHostsHostDeviceConfigurationInterface_InterfaceNumber, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"InterfaceClass", &DMREAD, DMT_HEXBIN, get_USBUSBHostsHostDeviceConfigurationInterface_InterfaceClass, NULL, BBFDM_BOTH},
{"InterfaceSubClass", &DMREAD, DMT_HEXBIN, get_USBUSBHostsHostDeviceConfigurationInterface_InterfaceSubClass, NULL, BBFDM_BOTH},
{"InterfaceProtocol", &DMREAD, DMT_HEXBIN, get_USBUSBHostsHostDeviceConfigurationInterface_InterfaceProtocol, NULL, BBFDM_BOTH},
{0}
};


DMOBJ tDeviceUSBObj[] = {
{"USB", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tUSBObj, tUSBParams, NULL, BBFDM_BOTH, NULL},
{0}
};

/* ********** DynamicObj ********** */
DM_MAP_OBJ tDynamicObj[] = {
/* parentobj, nextobject, parameter */
{"Device.", tDeviceUSBObj, NULL},
{0}
};
