/*
 * twamp_light.c: TWAMP Light functionality
 *
 * Copyright (C) 2022, IOPSYS Software Solutions AB.
 *
 * Author: amin.benramdhane@pivasoftware.com
 *         vivek.dutta@iopsys.eu
 *         suvendhu.hansa@iopsys.eu
 *
 * See LICENSE file for license related information.
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <pthread.h>
#include <time.h>
#include <arpa/inet.h>
#include <libubox/uloop.h>
#include <libubus.h>

#include "common.h"

#define TWAMP_MODE_TESTING	3
#define TWAMP_UNAUTH_PKT_SIZE	1472       // MTU 1514

void *twamp_connect_light(void *arg);

typedef struct active_session {
	int socket;
	uint16_t server_oct;
	uint32_t sid_addr;
	twamp_ntptime_t sid_time;
	uint32_t sid_rand;
	uint32_t seq_nb;
	uint32_t snd_nb;
	uint32_t fw_msg;
	uint32_t fw_lst_msg;
} twamp_active_sess_t;

typedef struct client_info {
	int status;
	int socket;
	struct sockaddr_in addr;
	struct sockaddr_in6 addr6;
	int sess_no;
	struct timeval shutdown_time;
	twamp_active_sess_t sessions;
} twamp_client_info_t;

typedef struct reflector_unauth_params {
    uint32_t seq_num;
    twamp_ntptime_t ts;
    uint16_t err_count;
    twamp_ntptime_t recv_time;
    uint32_t sender_pack_seq;
    twamp_ntptime_t send_time;
    uint16_t sender_err_count;
    uint8_t sender_ttl;
    uint8_t sender_tos;
} ref_unauth_param_t;

static void print_fw_metrics(char *client_addr, uint16_t send_port, uint8_t fw_tos,
			     const ref_unauth_param_t *ref_param)
{
	uint64_t sender_usec = convert_to_usec(&ref_param->send_time);
	uint64_t receiver_usec = convert_to_usec(&ref_param->recv_time);
	uint64_t reflsender_usec = convert_to_usec(&ref_param->ts);
	int64_t fwd_del = receiver_usec - sender_usec;
	int64_t int_del = reflsender_usec - receiver_usec;
	uint32_t snd_seq = ntohl(ref_param->sender_pack_seq);
	uint32_t rcv_seq = ntohl(ref_param->seq_num);

	INFO("Sender Add: %s", client_addr);
	INFO("Sent Time: %.0f", (double)sender_usec * 1e-3);
	INFO("Sender Seq: %d", snd_seq);
	INFO("Receiver Seq: %d", rcv_seq);
	INFO("Sender Port: %d", send_port);
	INFO("Sync: %c", fwd_del < 0 ? 'N' : 'Y');
	INFO("TTL: %d", ref_param->sender_ttl);
	INFO("Fw TOS: %d", fw_tos);
	INFO("Int Delay: %.3f", (double)int_del * 1e-3);
}

static int twamp_global_init(twamp_ref_cfg_t **ref_cfg)
{
	twamp_glob_cfg_t glb_cfg;
	memset(&glb_cfg, 0, sizeof(glb_cfg));

	int ret = twamp_get_uci_config(ref_cfg, &glb_cfg);

	if (ret == -1)
		glb_cfg.log_level = SINFO;

	twamp_log_init(glb_cfg.log_level);

	DEBUG("TWAMP Reflector Log Level:%d", glb_cfg.log_level);

	return ret;
}

static int twamp_spawn(void)
{
	twamp_ref_cfg_t *ref_cfg = NULL;

	int ref_num = twamp_global_init(&ref_cfg);
	INFO("START TWAMP Reflector");

	if (ref_num == 0) {
		NOTICE("No enabled twamp reflector found");
		return 0;
	}

	if (ref_cfg == NULL) {
		NOTICE("Twamp reflector cfg could not read.");
		return 0;
	}

	int i;
	for (i = 0; i < ref_num; i++) {
		twamp_ref_cfg_t *tc = &ref_cfg[i];
		DEBUG("TWAMP Reflector Interface: %s", tc->ref_intf);
		DEBUG("TWAMP Reflector Device: %s", tc->ref_dev);
		DEBUG("TWAMP Reflector Port: %d", tc->ref_port);
		DEBUG("TWAMP Reflector MaximumTTL: %d", tc->ref_ttl);
		DEBUG("TWAMP Reflector IPAllowedList: %s", tc->ref_ip_allow);
		DEBUG("TWAMP Reflector PortAllowedList: %s", tc->ref_port_allow);

		pthread_t ptid = 0;
		int ret = pthread_create(&ptid, NULL, twamp_connect_light, tc);
		if (ret != 0) {
			CRIT("Failed to create light thread %d", ret);
			break;
		}
	}

	FREE(ref_cfg);
	return 0;
}

int check_port_allowed(const char *allowed_port_list, int port)
{
	char *pch, *spch, *min, *max, *str;
	char *port_list;
	int ret = 0;

	if (allowed_port_list == NULL) {
		DEBUG("Empty port_list to validate");
		return 1;
	}

	port_list = strdup(allowed_port_list);
	for (pch = strtok_r(port_list, ",", &spch); pch != NULL; pch = strtok_r(NULL, ",", &spch))
	{
		if(strstr(pch, "-")) {
			min = strtok_r(pch, "-", &str);
			max = strtok_r(NULL, "", &str);
			if(port >= strtol(min, NULL, 10) && port <= strtol(max, NULL, 10)) {
				ret = 1;
				break;
			}
		}

		if (port == strtol(pch, NULL, 10)) {
			ret = 1;
			break;
		}
	}

	FREE(port_list);
	return ret;
}

static char *check_ipv6_address_active(char *ip)
{
	unsigned char buf[sizeof(struct in6_addr)];
	char str[INET6_ADDRSTRLEN] = {0};
	int s;

	s = inet_pton(AF_INET6, ip, buf);
	if (s <= 0) {
		ERR("inet_pton failed with return=%d", s);
		goto exit;
	}

	if (inet_ntop(AF_INET6, buf, str, INET6_ADDRSTRLEN) == NULL) {
		ERR("inet_ntop failed");
		goto exit;
	}

	return strdup(str);
exit:
	return strdup("");
}

static int check_ipv6_address(char *ip, char *maskstr, char *address)
{
	struct sockaddr_in6 sa = {0};
	unsigned long netaddress, maxaddress;
	unsigned long mask = ~((1 << (128 - strtol(maskstr, 0, 10))) - 1);

	inet_pton(AF_INET6, address, &(sa.sin6_addr));
	netaddress = (ntohl(*((uint32_t *)sa.sin6_addr.s6_addr)) & mask);
	inet_pton(AF_INET6, ip, &(sa.sin6_addr));
	maxaddress = (ntohl(*((uint32_t*)sa.sin6_addr.s6_addr)) & mask);
	if (maxaddress == netaddress)
		return 1;

	return 0;
}

static int check_ipv4_address(char *ip, char *maskstr, char *address)
{
	struct sockaddr_in sa = {0};
	unsigned long netaddress, maxaddress;
	unsigned long mask = ~((1 << (32 - strtol(maskstr, NULL, 10))) - 1);

	inet_pton(AF_INET, address, &(sa.sin_addr));
	netaddress = (ntohl(sa.sin_addr.s_addr) & mask);
	sa.sin_addr.s_addr = 0;
	inet_pton(AF_INET, ip, &(sa.sin_addr));
	maxaddress = (ntohl(sa.sin_addr.s_addr) & mask);
	if (maxaddress == netaddress)
		return 1;

	return 0;
}

static int check_ip_address_allowed(const char *allowed_ip_list, char *address)
{
	char *pch, *spch, *ip, *mask, *str, *addr;
	char *ip_list = NULL;
	int ret = 0;

	if (allowed_ip_list == NULL) {
		DEBUG("Empty ip_list to validate");
		return 1;
	}

	if (address == NULL) {
		CRIT("Empty ip address to validate");
		return 0;
	}

	ip_list = strdup(allowed_ip_list);
	for (pch = strtok_r(ip_list, ",", &spch); pch != NULL; pch = strtok_r(NULL, ",", &spch))
	{
		if(strstr(pch, ".")) {
			if(strstr(pch, "/")) {
				ip = strtok_r(pch, "/", &str);
				mask = strtok_r(NULL, "", &str);
				if(check_ipv4_address(ip, mask, address)) {
					ret = 1;
					break;
				}
			}

			if (strcmp(pch, address) == 0) {
				ret = 1;
				break;
			}
		} else {
			addr = check_ipv6_address_active(address);
			if(strstr(pch, "/")) {
				ip = strtok_r(pch, "/", &str);
				mask = strtok_r(NULL, "", &str);
				ip = check_ipv6_address_active(ip);
				if(check_ipv6_address(ip, mask, addr)) {
					ret = 1;
					break;
				}

				FREE(ip);
			}

			pch = check_ipv6_address_active(pch);
			if (strcmp(pch, addr) == 0) {
				ret = 1;
				break;
			}

			FREE(pch);
			FREE(addr);
		}
	}

	FREE(ip_list);
	return ret;
}

/* This function will receive a TWAMP-Test packet and will send a response. In TWAMP the Session-Sender (in our case the Control-Client, meaning the
 * TWAMP-Client) is always sending TWAMP-Test packets and the Session-Reflector (Server) is receiving TWAMP-Test packets.
 */
static int receive_test_message(twamp_ref_cfg_t *tc, twamp_client_info_t *client)
{
	struct sockaddr_in addr;
	struct sockaddr_in6 addr6;
	socklen_t len = sizeof(addr);
	char str_client[INET6_ADDRSTRLEN];
	int rv;
	int socket_family = AF_INET;

	if (tc->ip_ver == 6)
		socket_family = AF_INET6;

	memset(&addr, 0, sizeof(struct sockaddr_in));

	inet_ntop(socket_family, (socket_family == AF_INET6)? (void*) &(client->addr6.sin6_addr) : (void*) &(client->addr.sin_addr), str_client, sizeof(str_client));

	ref_unauth_param_t refl_parm;
	memset(&refl_parm, 0, sizeof(refl_parm));

	uint8_t pack[TWAMP_UNAUTH_PKT_SIZE];
	memset(pack, 0, sizeof(pack));

	/* New for recvmsg */
	struct msghdr message;
	struct cmsghdr *c_msg;
	char control_buffer[TWAMP_UNAUTH_PKT_SIZE];
	uint16_t control_length = TWAMP_UNAUTH_PKT_SIZE;
	struct iovec vec;

	memset(&message, 0, sizeof(struct msghdr));
	memset(&control_buffer, 0, TWAMP_UNAUTH_PKT_SIZE);
	memset(&vec, 0, sizeof(struct iovec));

	message.msg_name = (socket_family == AF_INET6)? (void*)&addr6: (void*)&addr;
	message.msg_namelen = len;

	vec.iov_base = &pack;
	vec.iov_len = TWAMP_UNAUTH_PKT_SIZE;
	message.msg_iov = &vec;
	message.msg_iovlen = 1;
	message.msg_control = &control_buffer;
	message.msg_controllen = control_length;

	rv = recvmsg(client->sessions.socket, &message, 0);
	if (rv <= 0) {
		ERR("Failed to receive TWAMP-Test packet");
		return rv;
	} else if (rv < 14) {
		WARNING("Short TWAMP-Test packet");
		return rv;
	}

	if(tc->ref_ip_allow[0] != '\0') {
		if(!check_ip_address_allowed(tc->ref_ip_allow, inet_ntoa(addr.sin_addr))) {
			INFO("IP Address %s is not allowed", str_client);
			return -1;
		}
	}

	if(tc->ref_port_allow[0] != '\0') {
		if(!check_port_allowed(tc->ref_port_allow, ntohs(addr.sin_port))) {
			INFO("Port %d is not in allowed port list", ntohs(addr.sin_port));
			return -1;
		}
	}

	refl_parm.recv_time = get_current_timestamp();

	char str_server[INET6_ADDRSTRLEN];

	inet_ntop(socket_family, (socket_family == AF_INET6)? (void*) &(addr6.sin6_addr) : (void*) &(addr.sin_addr), str_server, sizeof(str_server));
	/* Get TTL/TOS values from IP header */
	uint8_t fw_ttl = 0;
	uint8_t fw_tos = 0;

	for (c_msg = CMSG_FIRSTHDR(&message); c_msg; c_msg = (CMSG_NXTHDR(&message, c_msg))) {
		if ((c_msg->cmsg_level == IPPROTO_IP && c_msg->cmsg_type == IP_TTL) || (c_msg->cmsg_level == IPPROTO_IPV6 && c_msg->cmsg_type == IPV6_HOPLIMIT)) {
			fw_ttl = *(int *)CMSG_DATA(c_msg);
		} else if (c_msg->cmsg_level == IPPROTO_IP && c_msg->cmsg_type == IP_TOS) {
			fw_tos = *(int *)CMSG_DATA(c_msg);
		} else {
			INFO("Warning, unexpected data of level %i and type %i", c_msg->cmsg_level, c_msg->cmsg_type);
		}
	}

	INFO("Received TWAMP-Test message from %s", inet_ntoa(addr.sin_addr));
	refl_parm.seq_num = htonl(client->sessions.seq_nb++);
	refl_parm.err_count = htons(0x8001);    // Sync = 1, Multiplier = 1
	READ_N_BYTES(&refl_parm.sender_pack_seq, pack, 4);
	READ_N_BYTES(&refl_parm.send_time, &pack[4], 8);
	READ_N_BYTES(&refl_parm.sender_err_count, &pack[12], 2);
	refl_parm.sender_ttl = fw_ttl;   // Copy from the IP header packet from Sender

	/* FW Loss Calculation */
	if (client->sessions.fw_msg == 0) {
		client->sessions.fw_msg = 1;
		/* Response packet for TOS with ECN */
		if ((fw_tos & 0x03) > 0) {
#ifdef IP_TOS
			uint8_t ip_tos = (fw_tos & 0x03) - (((fw_tos & 0x2) >> 1) & (fw_tos & 0x1));
			int socket = client->sessions.socket;
			if (0 != setsockopt(socket, IPPROTO_IP, IP_TOS, &ip_tos, sizeof(ip_tos))) {
				DEBUG("Failed to set the TOS value.\n");
			}
#else
			DEBUG("Not configured to set the TOS value for leaving packets.\n");
#endif
		}
	} else {
		client->sessions.fw_msg = client->sessions.fw_msg + ntohl(refl_parm.sender_pack_seq) - client->sessions.snd_nb;
		client->sessions.fw_lst_msg = client->sessions.fw_lst_msg + ntohl(refl_parm.sender_pack_seq) - client->sessions.snd_nb - 1;
	}

	client->sessions.snd_nb = ntohl(refl_parm.sender_pack_seq);

	refl_parm.ts = get_current_timestamp();

	/* Response packet */
	uint8_t send_pack[TWAMP_UNAUTH_PKT_SIZE];
	memset(send_pack, 0, sizeof(send_pack));

	WRITE_N_BYTES(send_pack, &refl_parm.seq_num, 4);
	WRITE_N_BYTES(&send_pack[4], &refl_parm.ts, 8);
	WRITE_N_BYTES(&send_pack[12], &refl_parm.err_count, 2);
	WRITE_N_BYTES(&send_pack[16], &refl_parm.recv_time, 8);
	WRITE_N_BYTES(&send_pack[24], &refl_parm.sender_pack_seq, 4);
	WRITE_N_BYTES(&send_pack[28], &refl_parm.send_time, 8);
	WRITE_N_BYTES(&send_pack[36], &refl_parm.sender_err_count, 2);
	WRITE_N_BYTES(&send_pack[40], &refl_parm.sender_ttl, 1);
	WRITE_N_BYTES(&send_pack[41], &refl_parm.sender_tos, 1);

	if(socket_family == AF_INET6) {
		if (rv < 41) {
			rv = sendto(client->sessions.socket, send_pack, 41, 0, (struct sockaddr *)&addr6, sizeof(addr6));
		} else {
			rv = sendto(client->sessions.socket, send_pack, rv, 0, (struct sockaddr *)&addr6, sizeof(addr6));
		}
	} else {
		if (rv < 41) {
			rv = sendto(client->sessions.socket, send_pack, 41, 0, (struct sockaddr *)&addr, sizeof(addr));
		} else {
			rv = sendto(client->sessions.socket, send_pack, rv, 0, (struct sockaddr *)&addr, sizeof(addr));
		}
	}

	if (rv <= 0) {
		NOTICE("Failed to send TWAMP-Test packet");
	}

	/* Print the FW metrics */
	print_fw_metrics(str_client, socket_family == AF_INET6 ? ntohs(addr6.sin6_port): ntohs(addr.sin_port), fw_tos, &refl_parm);

	if ((client->sessions.fw_msg % 10) == 0) {
		INFO("FW Lost packets: %u/%u", client->sessions.fw_lst_msg, client->sessions.fw_msg);
		INFO("FW Loss Ratio: %3.2f%%", (float)100 * client->sessions.fw_lst_msg / client->sessions.fw_msg);
	}

	return rv;
}

static void configure_client(twamp_client_info_t *client, int sockfd,
			     struct sockaddr_in serv_addr, struct sockaddr_in6 serv6_addr)
{
	client->socket = sockfd;
	client->sessions.socket = sockfd;
	client->addr = serv_addr;
	client->addr6 = serv6_addr;
}

void *twamp_connect_light(void *arg)
{
	int sockfd;
	int ret=0;
	int family = AF_INET;
        struct sockaddr_in6 serv6_addr;
	struct sockaddr_in serv_addr;
	twamp_ref_cfg_t *tc = (twamp_ref_cfg_t *)arg;
	twamp_ref_cfg_t config;
	twamp_client_info_t client;

	memset(&serv_addr, 0, sizeof(serv_addr));
	memset(&serv6_addr, 0, sizeof(serv6_addr));
	memset(&config, 0, sizeof(twamp_ref_cfg_t));
	memset(&client, 0, sizeof(twamp_client_info_t));

	if (tc == NULL) {
		return NULL;
	}

	config.ref_enable = tc->ref_enable;
	config.ip_ver= tc->ip_ver;
	config.ref_port = tc->ref_port;
	config.ref_ttl = tc->ref_ttl;

	strncpy(config.ref_intf, tc->ref_intf, 16);
	strncpy(config.ref_dev, tc->ref_dev, 16);
	strncpy(config.ref_ip_allow, tc->ref_ip_allow, 1024);
	strncpy(config.ref_port_allow, tc->ref_port_allow, 1024);

	DEBUG("Twamp light thread started on %d tid::%d", config.ref_port, pthread_self());

	if (config.ip_ver == 6)
		family = AF_INET6;

	sockfd = socket(family, SOCK_DGRAM, 0);
	if (sockfd < 0) {
		CRIT("Error opening UDP socket");
		goto end;
	}

	if (strlen(config.ref_dev) != 0) {
		ret = setsockopt(sockfd, SOL_SOCKET, SO_BINDTODEVICE, config.ref_dev, strlen(config.ref_dev)+1);
		if(ret) {
			CRIT("Error on setsockopt with ret %d", ret);
			goto end;
		}
	}

	if(config.ip_ver == 4) {
		// Set Server address and bind on the TWAMP port
		serv_addr.sin_family = AF_INET;
		serv_addr.sin_addr.s_addr = htonl(INADDR_ANY);
		serv_addr.sin_port = htons(config.ref_port);

		if (bind(sockfd, (struct sockaddr *)&serv_addr, sizeof(struct sockaddr)) < 0) {
			CRIT("Error on binding");
			goto end;
		}
	} else {
        	// Set Server address for IPv6 and bind on the TWAMP port
		serv6_addr.sin6_family = AF_INET6;
		serv6_addr.sin6_addr = in6addr_any;
		serv6_addr.sin6_port = htons(config.ref_port);

		if (bind(sockfd, (struct sockaddr *)&serv_addr, sizeof(struct sockaddr)) < 0) {
			CRIT("Error on binding");
			goto end;
		}
	}

	while (1) {
		configure_client(&client, sockfd, serv_addr, serv6_addr);
		ret = receive_test_message(&config, &client);
		if(ret>14)
			INFO("Reply to test packet sent");
	}

end:
	DEBUG("Closing tid::%d", pthread_self());

	return NULL;
}

int main(void)
{
	uloop_init();

	twamp_spawn();

	uloop_run();
	uloop_done();

	closelog();
	INFO("EXIT TWAMP Reflector");

	return 0;
}
