/*
 * datamodel.c: TWAMP datamodel plugin
 *
 * Copyright (C) 2022, IOPSYS Software Solutions AB.
 *
 * Author: amin.benramdhane@pivasoftware.com
 *
 * See LICENSE file for license related information.
 *
 */

#include <libbbfdm-api/dmcommon.h>
#include <libbbfdm-api/dmuci.h>
#include <libbbfdm_api.h>

#include "datamodel.h"

static int get_enabled_reflector_count(void)
{
	int count = 0;
	struct uci_section *s = NULL;

	uci_foreach_sections("twamp", "twamp_reflector", s) {
		char *enable = bbf_uci_get_value_by_section_fallback_def(s, "enable", "1");
		if (enable[0] == '1') {
			count = count + 1;
		}
	}

	return count;
}

/*************************************************************
* Init & Clean Module
**************************************************************/
static int init_twamp_module(void *data)
{
	struct dmctx bbf_ctx = {0};
	struct uci_section *dmmap_iface_s = NULL;

	bbf_ctx_init(&bbf_ctx, NULL);

	// Device.IP.Interface.{i}.
	uci_path_foreach_sections(bbfdm, "IP", "Interface", dmmap_iface_s) {
		struct uci_section *config_iface_s = NULL;
		char *iface_sec_name = NULL;
		char *instance = NULL;

		dmuci_get_value_by_section_string(dmmap_iface_s, "__section_name__", &iface_sec_name);
		config_iface_s = get_config_section_from_dmmap_section_name(iface_sec_name);
		if (config_iface_s == NULL)
			continue;

		// This call is necessary since TWAMPReflector is in the second level
		create_dmmap_obj(&bbf_ctx, 0, "IP", "Interface", config_iface_s, &instance);

		// Device.IP.Interface.{i}.TWAMPReflector.{i}.
		struct uci_section *twamp_s = NULL;
		uci_foreach_option_eq("twamp", "twamp_reflector", "interface", section_name(config_iface_s), twamp_s) {
			create_dmmap_obj(&bbf_ctx, 1, "IP", "TWAMPReflector", twamp_s, &instance);
		}
	}

	dmuci_commit_package_bbfdm("IP");

	bbf_ctx_clean(&bbf_ctx);

	return 0;
}

/*************************************************************
* ADD & DEL OBJ
**************************************************************/
static int addObjIPInterfaceTWAMPReflector(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct dm_data *curr_data = (struct dm_data *)ctx->addobj_instance;
	char sec_name[64] = {0};

	snprintf(sec_name, sizeof(sec_name), "%s_twamp_%s", section_name(((struct dm_data *)data)->config_section), *instance);

	bbf_uci_add_section("twamp", "twamp_reflector", &curr_data->config_section);
	bbf_uci_rename_section(curr_data->config_section, sec_name);

	bbf_uci_set_value_by_section(curr_data->config_section, "enable", "0");
	bbf_uci_set_value_by_section(curr_data->config_section, "interface", section_name(((struct dm_data *)data)->config_section));
	bbf_uci_set_value_by_section(curr_data->config_section, "port", "862");
	bbf_uci_set_value_by_section(curr_data->config_section, "max_ttl", "1");
	bbf_uci_set_value_by_section(curr_data->config_section, "port_list", "");
	bbf_uci_set_value_by_section(curr_data->config_section, "ip_list", "");

	if (get_enabled_reflector_count() == 0) {
		bbf_uci_set_value("twamp", "twamp", "enable", "0");
	} else {
		bbf_uci_set_value("twamp", "twamp", "enable", "1");
	}

	return 0;
}

static int delObjIPInterfaceTWAMPReflector(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	bbf_uci_delete_section_by_section(((struct dm_data *)data)->config_section, NULL, NULL);
	bbf_uci_delete_section_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);

	if (get_enabled_reflector_count() == 0) {
		bbf_uci_set_value("twamp", "twamp", "enable", "0");
	}

	return 0;
}

/*************************************************************
* GET & SET PARAM
**************************************************************/
static int get_IPInterface_TWAMPReflectorNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = bbf_get_number_of_entries(ctx, data, instance, NULL);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_IPInterfaceTWAMPReflector_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = bbf_uci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enable", "1");
	return 0;
}

static int configure_layers(struct dm_data *data)
{
	char *interface = NULL;

	bbf_uci_get_value_by_section(data->config_section, "interface", &interface);
	if (interface != NULL && interface[0] == '\0') {
		return -1;
	}

	bbf_uci_set_value_by_section(data->config_section, "device", get_device(interface));
	return 0;
}

static int set_IPInterfaceTWAMPReflector_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			bbf_convert_string_to_bool(value, &b);
			if(b) {
				if (configure_layers((struct dm_data *) data) == 0) {
					bbf_uci_set_value("twamp", "twamp", "enable", "1");
					bbf_uci_set_value_by_section(((struct dm_data *)data)->config_section, "enable", "1");
				} else {
					return FAULT_9002;
				}
			} else {
				bbf_uci_set_value_by_section(((struct dm_data *)data)->config_section, "enable", "0");

				if (get_enabled_reflector_count() == 0) {
					bbf_uci_set_value("twamp", "twamp", "enable", "0");
				}
			}
			break;
	}

	return 0;
}

static int get_IPInterfaceTWAMPReflector_Status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *enable;

	bbf_uci_get_value_by_section(((struct dm_data *)data)->config_section, "enable", &enable);
	if (strcmp(enable, "1") == 0)
		*value = dmstrdup("Active");
	else
		*value = dmstrdup("Disabled");

	return 0;
}

static int get_IPInterfaceTWAMPReflector_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "Alias", instance, value);
}

static int set_IPInterfaceTWAMPReflector_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "Alias", instance, value);
}

static int get_IPInterfaceTWAMPReflector_Port(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = bbf_uci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "port", "862");
	return 0;
}

static int set_IPInterfaceTWAMPReflector_Port(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,"65535"}}, 1))
				return FAULT_9007;
			break;
		case VALUESET:
			bbf_uci_set_value_by_section(((struct dm_data *)data)->config_section, "port", value);
			break;
	}
	return 0;
}

static int get_IPInterfaceTWAMPReflector_MaximumTTL(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = bbf_uci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "max_ttl", "1");
	return 0;
}

static int set_IPInterfaceTWAMPReflector_MaximumTTL(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"1","255"}}, 1))
				return FAULT_9007;
			break;
		case VALUESET:
			bbf_uci_set_value_by_section(((struct dm_data *)data)->config_section, "max_ttl", value);
			break;
	}
	return 0;
}

static int get_IPInterfaceTWAMPReflector_IPAllowedList(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	bbf_uci_get_value_by_section(((struct dm_data *)data)->config_section, "ip_list", value);
	return 0;
}

static int set_IPInterfaceTWAMPReflector_IPAllowedList(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string_list(ctx, value, -1, -1, 255, -1, -1, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			bbf_uci_set_value_by_section(((struct dm_data *)data)->config_section, "ip_list", value);
			break;
	}
	return 0;
}

static int get_IPInterfaceTWAMPReflector_PortAllowedList(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	bbf_uci_get_value_by_section(((struct dm_data *)data)->config_section, "port_list", value);
	return 0;
}

static int set_IPInterfaceTWAMPReflector_PortAllowedList(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string_list(ctx, value, -1, -1, 255, -1, -1, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			bbf_uci_set_value_by_section(((struct dm_data *)data)->config_section, "port_list", value);
			break;
	}
	return 0;
}

/**********************************************************************************************************************************
*                                            OBJ & LEAF DEFINITION
***********************************************************************************************************************************/
/* ********** DynamicObj ********** */
DM_MAP_OBJ tDynamicObj[] = {
/* parentobj, nextobject, parameter */
{"Device.IP.Interface.{i}.", tDeviceTWAMPReflectorObj, tDeviceTWAMPReflectorParams, init_twamp_module, NULL},
{0}
};

/* *** Device.IP.Interface.{i}. *** */
DMOBJ tDeviceTWAMPReflectorObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys*/
{"TWAMPReflector", &DMWRITE, addObjIPInterfaceTWAMPReflector, delObjIPInterfaceTWAMPReflector, NULL, generic_browse, NULL, NULL, NULL, tIPInterfaceTWAMPReflectorParams, NULL, BBFDM_BOTH},
{0}
};

DMLEAF tDeviceTWAMPReflectorParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"TWAMPReflectorNumberOfEntries", &DMREAD, DMT_UNINT, get_IPInterface_TWAMPReflectorNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.IP.Interface.{i}.TWAMPReflector.{i}. *** */
DMLEAF tIPInterfaceTWAMPReflectorParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"Enable", &DMWRITE, DMT_BOOL, get_IPInterfaceTWAMPReflector_Enable, set_IPInterfaceTWAMPReflector_Enable, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_IPInterfaceTWAMPReflector_Status, NULL, BBFDM_BOTH},
{"Alias", &DMWRITE, DMT_STRING, get_IPInterfaceTWAMPReflector_Alias, set_IPInterfaceTWAMPReflector_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Port", &DMWRITE, DMT_UNINT, get_IPInterfaceTWAMPReflector_Port, set_IPInterfaceTWAMPReflector_Port, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"MaximumTTL", &DMWRITE, DMT_UNINT, get_IPInterfaceTWAMPReflector_MaximumTTL, set_IPInterfaceTWAMPReflector_MaximumTTL, BBFDM_BOTH},
{"IPAllowedList", &DMWRITE, DMT_STRING, get_IPInterfaceTWAMPReflector_IPAllowedList, set_IPInterfaceTWAMPReflector_IPAllowedList, BBFDM_BOTH},
{"PortAllowedList", &DMWRITE, DMT_STRING, get_IPInterfaceTWAMPReflector_PortAllowedList, set_IPInterfaceTWAMPReflector_PortAllowedList, BBFDM_BOTH},
{0}
};
