/*
 * common.c: TWAMP utility functions
 *
 * Copyright (C) 2022, IOPSYS Software Solutions AB.
 *
 * Author: amin.benramdhane@pivasoftware.com
 *         suvendhu.hansa@iopsys.eu
 *
 * See LICENSE file for license related information.
 *
 */

#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <arpa/inet.h>
#include <sys/time.h>

#include "common.h"

#define NTP_SECS_OFFSET	2208988800uL

static int g_log_level;

typedef enum {
	E_TWAMP_GLOB_ENABLE,
	E_TWAMP_GLOB_LOG_LEVEL,
	E_TWAMP_GLOB_MAX
} twamp_glob_opt_t;

const struct uci_parse_option global_opts[] = {
	{ .name = "enable", .type = UCI_TYPE_STRING },
	{ .name = "log_level", .type = UCI_TYPE_STRING },
};

typedef enum {
	E_TWAMP_REFL_ENABLE,
	E_TWAMP_REFL_INTF,
	E_TWAMP_REFL_PORT,
	E_TWAMP_REFL_TTL,
	E_TWAMP_REFL_DEV,
	E_TWAMP_REFL_IP_VER,
	E_TWAMP_REFL_IP_LIST,
	E_TWAMP_REFL_PORT_LIST,
	E_TWAMP_REFL_MAX,
} twamp_refl_opt_t;

const struct uci_parse_option refl_opts[] = {
	{ .name = "enable", .type = UCI_TYPE_STRING },
	{ .name = "interface", .type = UCI_TYPE_STRING },
	{ .name = "port", .type = UCI_TYPE_STRING },
	{ .name = "max_ttl", .type = UCI_TYPE_STRING },
	{ .name = "device", .type = UCI_TYPE_STRING },
	{ .name = "ip_version", .type = UCI_TYPE_STRING },
	{ .name = "ip_list", .type = UCI_TYPE_STRING },
	{ .name = "port_list", .type = UCI_TYPE_STRING },
};

static bool get_boolean_string(const char *value)
{
	if (!value)
		return true;

	if (strncasecmp(value, "true", 4) == 0 ||
			value[0] == '1' ||
			strncasecmp(value, "on", 2) == 0 ||
			strncasecmp(value, "yes", 3) == 0 ||
			strncasecmp(value, "enabled", 7) == 0)
		return true;

	return false;
}

void twamp_log_init(int loglevel)
{
	openlog("twamp", LOG_CONS | LOG_PID | LOG_NDELAY, LOG_LOCAL1);
	g_log_level = loglevel;
}

void print_crit(const char *format, ...)
{
	if (g_log_level >= SCRIT) {
		va_list arglist;
		va_start(arglist, format);
		vsyslog(LOG_CRIT, format, arglist);
		va_end(arglist);
	}
}

void print_error(const char *format, ...)
{
	if (g_log_level >= SERROR) {
		va_list arglist;
		va_start(arglist, format);
		vsyslog(LOG_ERR, format, arglist);
		va_end(arglist);
	}
}

void print_warning(const char *format, ...)
{
	if (g_log_level >= SWARNING) {
		va_list arglist;
		va_start(arglist, format);
		vsyslog(LOG_WARNING, format, arglist);
		va_end(arglist);
	}
}

void print_notice(const char *format, ...)
{
	if (g_log_level >= SNOTICE) {
		va_list arglist;
		va_start(arglist, format);
		vsyslog(LOG_NOTICE, format, arglist);
		va_end(arglist);
	}
}

void print_info(const char *format, ...)
{
	if (g_log_level >= SINFO) {
		va_list arglist;
		va_start(arglist, format);
		vsyslog(LOG_INFO, format, arglist);
		va_end(arglist);
	}
}

void print_debug(const char *format, ...)
{
	if (g_log_level >= SDEBUG) {
		va_list arglist;
		va_start(arglist, format);
		vsyslog(LOG_DEBUG, format, arglist);
		va_end(arglist);
	}
}

int twamp_get_uci_config(twamp_ref_cfg_t **ref_cfg, twamp_glob_cfg_t *glb_cfg)
{
	int ret = 0;

	if (ref_cfg == NULL || glb_cfg == NULL)
		return -1;

	struct uci_context *ctx;
	struct uci_package *pkg;
	struct uci_element *e;
	twamp_ref_cfg_t *tmp = NULL;

	ctx = uci_alloc_context();
	if (!ctx) {
		return -1;
	}

	if (uci_load(ctx, "twamp", &pkg)) {
		uci_free_context(ctx);
		return -1;
	}

	uci_foreach_element(&pkg->sections, e) {
		struct uci_section *s = uci_to_section(e);

		if (strcmp(s->type, "twamp") == 0 && strcmp(e->name, "twamp") == 0) {
			struct uci_option *tb[E_TWAMP_GLOB_MAX];
			uci_parse_section(s, global_opts, E_TWAMP_GLOB_MAX, tb);

			if (tb[E_TWAMP_GLOB_ENABLE]) {
				const char *val = tb[E_TWAMP_GLOB_ENABLE]->v.string;
				glb_cfg->twamp_enable = get_boolean_string(val);
			}

			if (tb[E_TWAMP_GLOB_LOG_LEVEL]) {
				const char *val = tb[E_TWAMP_GLOB_LOG_LEVEL]->v.string;
				glb_cfg->log_level = strtol(val, NULL, 10) > SDEBUG ? SDEBUG : strtol(val, NULL, 10);
			} else {
				glb_cfg->log_level = SINFO;
			}

			continue;
		}

		if (strcmp(s->type, "twamp_reflector") == 0) {
			struct uci_option *tb[E_TWAMP_REFL_MAX];
			uci_parse_section(s, refl_opts, E_TWAMP_REFL_MAX, tb);

			bool refl_enable = false;
			if (tb[E_TWAMP_REFL_ENABLE]) {
				const char *val = tb[E_TWAMP_REFL_ENABLE]->v.string;
				refl_enable = get_boolean_string(val);
			}

			if (refl_enable == false)
				continue;

			tmp = (twamp_ref_cfg_t *)realloc(*ref_cfg, sizeof(twamp_ref_cfg_t) * (ret + 1));
			if (tmp == NULL) {
				FREE(*ref_cfg);
				return -1;
			}

			memset(&tmp[ret], 0, sizeof(twamp_ref_cfg_t));
			tmp[ret].ref_enable = true;

			if (tb[E_TWAMP_REFL_INTF]) {
				strncpy(tmp[ret].ref_intf, tb[E_TWAMP_REFL_INTF]->v.string, 16);
			}

			if (tb[E_TWAMP_REFL_PORT]) {
				const char *val = tb[E_TWAMP_REFL_PORT]->v.string;
				tmp[ret].ref_port = strtol(val, NULL, 10);
			}

			if (tb[E_TWAMP_REFL_TTL]) {
				const char *val = tb[E_TWAMP_REFL_TTL]->v.string;
				tmp[ret].ref_ttl = strtol(val, NULL, 10);
			}

			if (tb[E_TWAMP_REFL_DEV]) {
				strncpy(tmp[ret].ref_dev, tb[E_TWAMP_REFL_DEV]->v.string, 16);
			}

			if (tb[E_TWAMP_REFL_IP_VER]) {
				const char *val = tb[E_TWAMP_REFL_IP_VER]->v.string;
				tmp[ret].ip_ver = strtol(val, NULL, 10) == 0 ? 4 : strtol(val, NULL, 10);
			} else {
				tmp[ret].ip_ver = 4;
			}

			if (tb[E_TWAMP_REFL_IP_LIST]) {
				strncpy(tmp[ret].ref_ip_allow, tb[E_TWAMP_REFL_IP_LIST]->v.string, 1024);
			}

			if (tb[E_TWAMP_REFL_PORT_LIST]) {
				strncpy(tmp[ret].ref_port_allow, tb[E_TWAMP_REFL_PORT_LIST]->v.string, 1024);
			}

			*ref_cfg = tmp;
			ret = ret + 1;
		}
	}

	return ret;
}

static void unix_to_ntptime(const struct timeval *tv, twamp_ntptime_t *ts)
{
	if (!tv || !ts)
		return;

	/* Convert Unix time to NTP Time format in network byte order */
	ts->int_val = htonl(tv->tv_sec + NTP_SECS_OFFSET);

	uint64_t aux = tv->tv_usec;
	aux <<= 32;
	aux /= 1000000;
	ts->frac_val = htonl((uint32_t) aux);
}

static void ntp_to_unixtime(const twamp_ntptime_t *ts, struct timeval *tv)
{
	if (!ts || !tv)
		return;

	/* Convert NTP time to Unix time format in host byte order */
	tv->tv_sec = ntohl(ts->int_val) - NTP_SECS_OFFSET;

	uint64_t aux = ntohl(ts->frac_val);
	aux *= 1000000;
	aux >>= 32;
	tv->tv_usec = (uint32_t) aux;
}

twamp_ntptime_t get_current_timestamp(void)
{
	struct timeval tv;
	memset(&tv, 0, sizeof(tv));

	gettimeofday(&tv, NULL);

	twamp_ntptime_t ts;
	unix_to_ntptime(&tv, &ts);

	return ts;
}

uint64_t convert_to_usec(const twamp_ntptime_t *ts)
{
	struct timeval tv;
	memset(&tv, 0, sizeof(tv));

	ntp_to_unixtime(ts, &tv);

	return tv.tv_sec * 1000000 + tv.tv_usec;
}
