/*
 * Copyright (C) 2024 IOPSYS Software Solutions AB. All rights reserved.
 *
 * Author: Suvendhu Hansa <suvendhu.hansa@iopsys.eu>
 *
 * See LICENSE file for license related information.
 */


#include "libbbfdm-api/dmcommon.h"

#define IPLAYER_CAP_DIAGNOSTIC_PATH BBFDM_SCRIPTS_PATH"/iplayercap"

static char *Protocol_Version[] = {"Any", "IPv4", "IPv6", NULL};
static char *IPLayerCapacity_Role[] = {"Receiver", "Sender", NULL};
static char *UDP_Payload_Content[] = {"zeroes", "random", NULL};
static char *IPLayerCapacity_TestType[] = {"Search", "Fixed", NULL};
static char *RateAdj_Algorithm[] = {"B", "C", NULL};

static void stop_iplayercap_diagnostics()
{
	char output[256] = {0}, cmd[256] = {0};
	
	snprintf(cmd, sizeof(cmd), "sh %s '{\"proto\":\"both_proto\",\"cancel\":\"1\"}'", IPLAYER_CAP_DIAGNOSTIC_PATH);

	run_cmd(cmd, output, sizeof(output));
}

/*
 * *** Device.IP.Diagnostics.IPLayerCapacityMetrics. ***
 */
static int browseIPLayerCapacityModalResultInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct uci_section *s = NULL;
	char *inst = NULL;

	uci_path_foreach_sections(bbfdm, DMMAP_DIAGNOSTIGS, "modalresult", s) {
		inst = handle_instance(dmctx, parent_node, s, "modal_instance", "modal_alias");
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)s, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseIPLayerCapacityIncrementalResultInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct uci_section *s = NULL;
	char *inst = NULL;

	uci_path_foreach_sections(bbfdm, DMMAP_DIAGNOSTIGS, "incrementalresult", s) {
		inst = handle_instance(dmctx, parent_node, s, "incremental_instance", "incremental_alias");
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)s, inst) == DM_STOP)
			break;
	}
	return 0;
}


static int get_IPDiagnosticsIPLayerCapacity_DiagnosticsState(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *val = diagnostics_get_option_fallback_def("iplayercapacity", "DiagnosticState", "None");
	if (DM_STRSTR(val, "Requested") != NULL)
		*value = dmstrdup("Requested");
	else
		*value = dmstrdup(val);

	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_DiagnosticsState(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, DiagnosticsState, NULL))
				return FAULT_9007;
			return 0;
		case VALUESET:
			if (DM_LSTRCMP(value, "Requested") == 0) {
				diagnostics_set_option("iplayercapacity", "DiagnosticState", value);
			} else if (DM_LSTRCMP(value, "Canceled") == 0) {
				diagnostics_set_option("iplayercapacity", "DiagnosticState", "None");
				stop_iplayercap_diagnostics();
			}
			return 0;
	}
	return 0;
}

int get_IPDiagnosticsIPLayerCapacity_SoftwareVersion(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("7.5.1");
	return 0;
}

int get_IPDiagnosticsIPLayerCapacity_CapacitySupported(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("1");
	return 0;
}

int get_IPDiagnosticsIPLayerCapacity_MaxConnections(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("1");
	return 0;
}

int get_IPDiagnosticsIPLayerCapacity_MaxIncrementalResult(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("3600");
	return 0;
}

int get_IPDiagnosticsIPLayerCapacity_ControlProtocolVersion(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("9");
	return 0;
}

int get_IPDiagnosticsIPLayerCapacity_SupportedMetrics(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("IPLR,Sampled_RTT,IPDV,IPRR,RIPR");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_Interface(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *linker = diagnostics_get_option("iplayercapacity", "interface");
	_bbfdm_get_references(ctx, "Device.IP.Interface.", "Name", linker, value);
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_Interface(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_objects[] = {"Device.IP.Interface.", NULL};
	struct dm_reference reference = {0};

	bbfdm_get_reference_linker(ctx, value, &reference);

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, reference.path, -1, 256, NULL, NULL))
				return FAULT_9007;

			if (dm_validate_allowed_objects(ctx, &reference, allowed_objects))
				return FAULT_9007;

			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "interface", reference.value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_Role(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "Role");
	if (DM_STRLEN(*value) == 0) {
		*value = dmstrdup("Sender");
	}
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_Role(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, IPLayerCapacity_Role, NULL))
				return FAULT_9007;

			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "Role", value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_Host(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "Host");
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_Host(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 256, NULL, NULL))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "Host", value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_Port(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option_fallback_def("iplayercapacity", "Port", "25000");
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_Port(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"1","65535"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "Port", value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_JumboFramesPermitted(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "JumboFramesPermitted");
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_JumboFramesPermitted(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			string_to_bool(value, &b);
			diagnostics_set_option("iplayercapacity", "JumboFramesPermitted", b ? "1" : "0");
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_DSCP(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "DSCP");
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_DSCP(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"0","63"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "DSCP", value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_ProtocolVersion(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "ProtocolVersion");
	if (DM_STRLEN(*value) == 0) {
		*value = dmstrdup("Any");
	}
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_ProtocolVersion(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, Protocol_Version, NULL))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "ProtocolVersion", value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_UDPPayloadContent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "UDPPayloadContent");
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_UDPPayloadContent(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, UDP_Payload_Content, NULL))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "UDPPayloadContent", value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_TestType(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "TestType");
	if (DM_STRLEN(*value) == 0) {
		*value = dmstrdup("Search");
	}
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_TestType(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, IPLayerCapacity_TestType, NULL))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "TestType", value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_IPDVEnable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "IPDVEnable");
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_IPDVEnable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			string_to_bool(value, &b);
			diagnostics_set_option("iplayercapacity", "IPDVEnable", b ? "1" : "0");
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_StartSendingRateIndex(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "StartSendingRateIndex");
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_StartSendingRateIndex(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			/* As per TR max should be 11108 but udpst supports 1108 */
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"0","1108"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "StartSendingRateIndex", value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_NumberFirstModeTestSubIntervals(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "NumberFirstModeTestSubIntervals");
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_NumberFirstModeTestSubIntervals(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"0","100"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "NumberFirstModeTestSubIntervals", value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_NumberTestSubIntervals(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option_fallback_def("iplayercapacity", "NumberTestSubIntervals", "10");
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_NumberTestSubIntervals(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			/* As per TR max should be 100 and (NumberTestSubIntervals * TestSubInterval) <= 60 sec
			 * since supported min value of TestSubInterval by udpst is 1 sec, so can be supported
			 * upto 60 value
			 */
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"1","60"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "NumberTestSubIntervals", value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_TestSubInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option_fallback_def("iplayercapacity", "TestSubInterval", "1000");
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_TestSubInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			/* As per DM min should be 100 but udpst supports min 1000 */
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"1000","6000"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "TestSubInterval", value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_StatusFeedbackInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option_fallback_def("iplayercapacity", "StatusFeedbackInterval", "50");
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_StatusFeedbackInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"5","250"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "StatusFeedbackInterval", value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_SeqErrThresh(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "SeqErrThresh");
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_SeqErrThresh(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"0","100"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "SeqErrThresh", value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_ReordDupIgnoreEnable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "ReordDupIgnoreEnable");
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_ReordDupIgnoreEnable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			string_to_bool(value, &b);
			diagnostics_set_option("iplayercapacity", "ReordDupIgnoreEnable", b ? "1" : "0");
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_LowerThresh(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option_fallback_def("iplayercapacity", "LowerThresh", "30");
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_LowerThresh(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"5","250"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "LowerThresh", value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_UpperThresh(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option_fallback_def("iplayercapacity", "UpperThresh", "90");
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_UpperThresh(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"5","250"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "UpperThresh", value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_HighSpeedDelta(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option_fallback_def("iplayercapacity", "HighSpeedDelta", "10");
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_HighSpeedDelta(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"2", NULL}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "HighSpeedDelta", value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_RateAdjAlgorithm(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "RateAdjAlgorithm");
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_RateAdjAlgorithm(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, RateAdj_Algorithm, NULL))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "RateAdjAlgorithm", value);
			return 0;
	}
	return 0;
}

static int set_IPDiagnosticsIPLayerCapacity_SlowAdjThresh(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"2",NULL}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			diagnostics_reset_state("iplayercapacity");
			stop_iplayercap_diagnostics();
			diagnostics_set_option("iplayercapacity", "SlowAdjThresh", value);
			return 0;
	}
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_SlowAdjThresh(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option_fallback_def("iplayercapacity", "SlowAdjThresh", "3");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_BOMTime(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option_fallback_def("iplayercapacity", "BOMTime", "0001-01-01T00:00:00.000000Z");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_EOMTime(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option_fallback_def("iplayercapacity", "EOMTime", "0001-01-01T00:00:00.000000Z");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_TmaxUsed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "TmaxUsed");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_TestInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "TestInterval");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_MaxIPLayerCapacity(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "MaxIPLayerCapacity");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_TimeOfMax(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option_fallback_def("iplayercapacity", "TimeOfMax", "0001-01-01T00:00:00.000000Z");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_MaxETHCapacityNoFCS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "MaxETHCapacityNoFCS");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_MaxETHCapacityWithFCS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "MaxETHCapacityWithFCS");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_MaxETHCapacityWithFCSVLAN(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "MaxETHCapacityWithFCSVLAN");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_LossRatioAtMax(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "LossRatioAtMax");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_RTTRangeAtMax(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "RTTRangeAtMax");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_PDVRangeAtMax(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "PDVRangeAtMax");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_MinOnewayDelayAtMax(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "MinOnewayDelayAtMax");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_ReorderedRatioAtMax(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "ReorderedRatioAtMax");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_ReplicatedRatioAtMax(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "ReplicatedRatioAtMax");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_InterfaceEthMbpsAtMax(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "InterfaceEthMbpsAtMax");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_IPLayerCapacitySummary(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "IPLayerCapacitySummary");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_LossRatioSummary(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "LossRatioSummary");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_RTTRangeSummary(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "RTTRangeSummary");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_PDVRangeSummary(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "PDVRangeSummary");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_MinOnewayDelaySummary(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "MinOnewayDelaySummary");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_MinRTTSummary(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "MinRTTSummary");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_ReorderedRatioSummary(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "ReorderedRatioSummary");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_ReplicatedRatioSummary(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "ReplicatedRatioSummary");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_InterfaceEthMbpsSummary(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "InterfaceEthMbpsSummary");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_TmaxRTTUsed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "TmaxRTTUsed");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_TimestampResolutionUsed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = diagnostics_get_option("iplayercapacity", "TimestampResolutionUsed");
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_IncrementalResultNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseIPLayerCapacityIncrementalResultInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_IPDiagnosticsIPLayerCapacity_ModalResultNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseIPLayerCapacityModalResultInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_IPLayerCapacityModal_MaxIPLayerCapacity(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "MaxIPLayerCapacity", value);
	return 0;
}

static int get_IPLayerCapacityModal_TimeOfMax(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def((struct uci_section *)data, "TimeOfMax", "0001-01-01T00:00:00.000000Z");
	return 0;
}

static int get_IPLayerCapacityModal_MaxETHCapacityNoFCS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "MaxETHCapacityNoFCS", value);
	return 0;
}

static int get_IPLayerCapacityModal_MaxETHCapacityWithFCS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "MaxETHCapacityWithFCS", value);
	return 0;
}

static int get_IPLayerCapacityModal_MaxETHCapacityWithFCSVLAN(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "MaxETHCapacityWithFCSVLAN", value);
	return 0;
}

static int get_IPLayerCapacityModal_LossRatioAtMax(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "LossRatioAtMax", value);
	return 0;
}

static int get_IPLayerCapacityModal_RTTRangeAtMax(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "RTTRangeAtMax", value);
	return 0;
}

static int get_IPLayerCapacityModal_PDVRangeAtMax(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "PDVRangeAtMax", value);
	return 0;
}

static int get_IPLayerCapacityModal_MinOnewayDelayAtMax(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "MinOnewayDelayAtMax", value);
	return 0;
}

static int get_IPLayerCapacityModal_ReorderedRatioAtMax(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "ReorderedRatioAtMax", value);
	return 0;
}

static int get_IPLayerCapacityModal_ReplicatedRatioAtMax(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "ReplicatedRatioAtMax", value);
	return 0;
}

static int get_IPLayerCapacityModal_InterfaceEthMbpsAtMax(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "InterfaceEthMbpsAtMax", value);
	return 0;
}

static int get_IPLayerCapacityIncremental_IPLayerCapacity(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "IPLayerCapacity", value);
	return 0;
}

static int get_IPLayerCapacityIncremental_TimeOfSubInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def((struct uci_section *)data, "TimeOfSubInterval", "0001-01-01T00:00:00.000000Z");
	return 0;
}

static int get_IPLayerCapacityIncremental_LossRatio(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "LossRatio", value);
	return 0;
}

static int get_IPLayerCapacityIncremental_RTTRange(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "RTTRange", value);
	return 0;
}

static int get_IPLayerCapacityIncremental_PDVRange(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "PDVRange", value);
	return 0;
}

static int get_IPLayerCapacityIncremental_MinOnewayDelay(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "MinOnewayDelay", value);
	return 0;
}

static int get_IPLayerCapacityIncremental_ReorderedRatio(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "ReorderedRatio", value);
	return 0;
}

static int get_IPLayerCapacityIncremental_ReplicatedRatio(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "ReplicatedRatio", value);
	return 0;
}

static int get_IPLayerCapacityIncremental_InterfaceEthMbps(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string((struct uci_section *)data, "InterfaceEthMbps", value);
	return 0;
}

static operation_args ip_diagnostics_iplayercapacity_args = {
	.in = (const char *[]) {
		"Interface",
		"Role",
		"Host",
		"Port",
		"JumboFramesPermitted",
		"DSCP",
		"ProtocolVersion",
		"UDPPayloadContent",
		"TestType",
		"IPDVEnable",
		"StartSendingRateIndex",
		"NumberTestSubIntervals",
		"NumberFirstModeTestSubIntervals",
		"TestSubInterval",
		"StatusFeedbackInterval",
		"SeqErrThresh",
		"ReordDupIgnoreEnable",
		"LowerThresh",
		"UpperThresh",
		"HighSpeedDelta",
		"SlowAdjThresh",
		"RateAdjAlgorithm",
		NULL
	},
	.out = (const char *[]) {
		"Status",
		"BOMTime",
		"EOMTime",
		"TmaxUsed",
		"TestInterval",
		"MaxIPLayerCapacity",
		"TimeOfMax",
		"MaxETHCapacityNoFCS",
		"MaxETHCapacityWithFCS",
		"MaxETHCapacityWithFCSVLAN",
		"LossRatioAtMax",
		"RTTRangeAtMax",
		"PDVRangeAtMax",
		"MinOnewayDelayAtMax",
		"ReorderedRatioAtMax",
		"ReplicatedRatioAtMax",
		"InterfaceEthMbpsAtMax",
		"IPLayerCapacitySummary",
		"LossRatioSummary",
		"RTTRangeSummary",
		"PDVRangeSummary",
		"MinOnewayDelaySummary",
		"MinRTTSummary",
		"ReorderedRatioSummary",
		"ReplicatedRatioSummary",
		"InterfaceEthMbpsSummary",
		"TmaxRTTUsed",
		"TimestampResolutionUsed",
		"ModalResult.{i}.MaxIPLayerCapacity",
		"ModalResult.{i}.TimeOfMax",
		"ModalResult.{i}.MaxETHCapacityNoFCS",
		"ModalResult.{i}.MaxETHCapacityWithFCS",
		"ModalResult.{i}.MaxETHCapacityWithFCSVLAN",
		"ModalResult.{i}.LossRatioAtMax",
		"ModalResult.{i}.RTTRangeAtMax",
		"ModalResult.{i}.PDVRangeAtMax",
		"ModalResult.{i}.MinOnewayDelayAtMax",
		"ModalResult.{i}.ReorderedRatioAtMax",
		"ModalResult.{i}.ReplicatedRatioAtMax",
		"ModalResult.{i}.InterfaceEthMbpsAtMax",
		"IncrementalResult.{i}.IPLayerCapacity",
		"IncrementalResult.{i}.TimeOfSubInterval",
		"IncrementalResult.{i}.LossRatio",
		"IncrementalResult.{i}.RTTRange",
		"IncrementalResult.{i}.PDVRange",
		"IncrementalResult.{i}.MinOnewayDelay",
		"IncrementalResult.{i}.ReorderedRatio",
		"IncrementalResult.{i}.ReplicatedRatio",
		"IncrementalResult.{i}.InterfaceEthMbps",
		NULL
	}
};

int get_operate_args_IPDiagnostics_IPLayerCapacity(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char *)&ip_diagnostics_iplayercapacity_args;
	return 0;
}

int operate_IPDiagnostics_IPLayerCapacity(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char input[4096] = {0};
	char output[40000] = {0};
	char cmd[5120] = {0};
	unsigned int idx = 0;

	char *host = dmjson_get_value((json_object *)value, 1, "Host");
	if (host[0] == '\0') {
		bbfdm_set_fault_message(ctx, "IPLayerCapacity: 'Host' input should be defined");
		return USP_FAULT_INVALID_ARGUMENT;
	}

	char *ip_interface = dmjson_get_value((json_object *)value, 1, "Interface");
	char *interface = diagnostics_get_interface_name(ctx, ip_interface);
	char *role = dmjson_get_value((json_object *)value, 1, "Role");
	if (role[0] != '\0' && bbfdm_validate_string(ctx, role, -1, -1, IPLayerCapacity_Role, NULL))
		return USP_FAULT_INVALID_ARGUMENT;

	char *port = dmjson_get_value((json_object *)value, 1, "Port");
	if (port[0] != '\0' && bbfdm_validate_unsignedInt(ctx, port, RANGE_ARGS{{"1","65535"}}, 1))
		return USP_FAULT_INVALID_ARGUMENT;

	char *jumbo = dmjson_get_value((json_object *)value, 1, "JumboFramesPermitted");
	bool jumbo_en = false;
	string_to_bool(jumbo, &jumbo_en);

	char *dscp = dmjson_get_value((json_object *)value, 1, "DSCP");
	if (dscp[0] != '\0' && bbfdm_validate_unsignedInt(ctx, dscp, RANGE_ARGS{{"0","63"}}, 1))
		return USP_FAULT_INVALID_ARGUMENT;

	char *ip_proto = dmjson_get_value((json_object *)value, 1, "ProtocolVersion");
	if (ip_proto[0] != '\0' && bbfdm_validate_string(ctx, ip_proto, -1, -1, Protocol_Version, NULL))
		return USP_FAULT_INVALID_ARGUMENT;

	char *content = dmjson_get_value((json_object *)value, 1, "UDPPayloadContent");
	if (content[0] != '\0' && bbfdm_validate_string(ctx, content, -1, -1, UDP_Payload_Content, NULL))
		return USP_FAULT_INVALID_ARGUMENT;

	char *test_type = dmjson_get_value((json_object *)value, 1, "TestType");
	if (test_type[0] != '\0' && bbfdm_validate_string(ctx, test_type, -1, -1, IPLayerCapacity_TestType, NULL))
		return USP_FAULT_INVALID_ARGUMENT;

	char *ipdv = dmjson_get_value((json_object *)value, 1, "IPDVEnable");
	bool ipdv_en = false;
	string_to_bool(ipdv, &ipdv_en);

	char *start_rate = dmjson_get_value((json_object *)value, 1, "StartSendingRateIndex");
	if (start_rate[0] != '\0' && bbfdm_validate_unsignedInt(ctx, start_rate, RANGE_ARGS{{"0","1108"}}, 1))
		return USP_FAULT_INVALID_ARGUMENT;

	char *num_interval = dmjson_get_value((json_object *)value, 1, "NumberTestSubIntervals");
	if (num_interval[0] != '\0' && bbfdm_validate_unsignedInt(ctx, num_interval, RANGE_ARGS{{"1","60"}}, 1))
		return USP_FAULT_INVALID_ARGUMENT;

	char *mode_test = dmjson_get_value((json_object *)value, 1, "NumberFirstModeTestSubIntervals");
	if (mode_test[0] != '\0' && bbfdm_validate_unsignedInt(ctx, mode_test, RANGE_ARGS{{"0","100"}}, 1))
		return USP_FAULT_INVALID_ARGUMENT;

	char *sub_interval = dmjson_get_value((json_object *)value, 1, "TestSubInterval");
	if (sub_interval[0] != '\0' && bbfdm_validate_unsignedInt(ctx, sub_interval, RANGE_ARGS{{"1000","6000"}}, 1))
		return USP_FAULT_INVALID_ARGUMENT;

	char *feed_interval = dmjson_get_value((json_object *)value, 1, "StatusFeedbackInterval");
	if (feed_interval[0] != '\0' && bbfdm_validate_unsignedInt(ctx, feed_interval, RANGE_ARGS{{"5","250"}}, 1))
		return USP_FAULT_INVALID_ARGUMENT;

	char *seq_err = dmjson_get_value((json_object *)value, 1, "SeqErrThresh");
	if (seq_err[0] != '\0' && bbfdm_validate_unsignedInt(ctx, seq_err, RANGE_ARGS{{"0","100"}}, 1))
		return USP_FAULT_INVALID_ARGUMENT;

	char *dup_ignore = dmjson_get_value((json_object *)value, 1, "ReordDupIgnoreEnable");
	bool dup_ignore_en = false;
	string_to_bool(dup_ignore, &dup_ignore_en);

	char *low_thresh = dmjson_get_value((json_object *)value, 1, "LowerThresh");
	if (low_thresh[0] != '\0' && bbfdm_validate_unsignedInt(ctx, low_thresh, RANGE_ARGS{{"5","250"}}, 1))
		return USP_FAULT_INVALID_ARGUMENT;

	char *up_thresh = dmjson_get_value((json_object *)value, 1, "UpperThresh");
	if (up_thresh[0] != '\0' && bbfdm_validate_unsignedInt(ctx, up_thresh, RANGE_ARGS{{"5","250"}}, 1))
		return USP_FAULT_INVALID_ARGUMENT;

	char *speed_delta = dmjson_get_value((json_object *)value, 1, "HighSpeedDelta");
	if (speed_delta[0] != '\0' && bbfdm_validate_unsignedInt(ctx, speed_delta, RANGE_ARGS{{"2", NULL}}, 1))
		return USP_FAULT_INVALID_ARGUMENT;

	char *slow_adj = dmjson_get_value((json_object *)value, 1, "SlowAdjThresh");
	if (slow_adj[0] != '\0' && bbfdm_validate_unsignedInt(ctx, slow_adj, RANGE_ARGS{{"2", NULL}}, 1))
		return USP_FAULT_INVALID_ARGUMENT;

	char *rate_adj = dmjson_get_value((json_object *)value, 1, "RateAdjAlgorithm");
	if (rate_adj[0] != '\0' && bbfdm_validate_string(ctx, rate_adj, -1, -1, RateAdj_Algorithm, NULL))
		return USP_FAULT_INVALID_ARGUMENT;

	snprintf(input, sizeof(input), "'{\"host\": \"%s\",\"interface\":\"%s\",\"role\":\"%s\",\"port\":\"%s\",\"jumbo_frames\":\"%s\",\"proto_ver\":\"%s\",\"udp_content\":\"%s\",\"test_type\":\"%s\",\"ipdv_enable\":\"%s\",\"DSCP\":\"%s\",\"rate_index\":\"%s\",\"mode_subintervals\":\"%s\",\"test_subinterval\":\"%s\",\"feedback_interval\":\"%s\",\"seq_err_thresh\":\"%s\",\"dup_ignore\":\"%s\",\"lower_thresh\":\"%s\",\"upper_thresh\":\"%s\",\"high_speed_delta\":\"%s\",\"algorithm\":\"%s\",\"slow_adj_thresh\":\"%s\",\"num_interval\":\"%s\",\"proto\":\"%s\"}'",
					host, interface, role, port, DM_STRLEN(jumbo) > 0 ? (jumbo_en ? "1" : "0") : "\0",
					ip_proto, content, test_type, DM_STRLEN(ipdv) > 0 ? (ipdv_en ? "1" : "0") : "\0",
					dscp, start_rate, mode_test, sub_interval, feed_interval, seq_err,
					DM_STRLEN(dup_ignore) > 0 ? (dup_ignore_en ? "1" : "0") : "\0", low_thresh, up_thresh,
					speed_delta, rate_adj, slow_adj, num_interval, (ctx->dm_type == BBFDM_USP) ? "usp" : "both_proto");

	snprintf(cmd, sizeof(cmd), "sh %s %s", IPLAYER_CAP_DIAGNOSTIC_PATH, input);

	if (run_cmd(cmd, output, sizeof(output)) != 0) {
		bbfdm_set_fault_message(ctx, "IPLayerCapacity: 'sh %s {input}' command failed to run", IPLAYER_CAP_DIAGNOSTIC_PATH);
		return USP_FAULT_COMMAND_FAILURE;
	}

	json_object *res = (DM_STRLEN(output)) ? json_tokener_parse(output) : NULL;

	if (res == NULL) {
		bbfdm_set_fault_message(ctx, "IPLayerCapacity: there is no output from '%s' script", IPLAYER_CAP_DIAGNOSTIC_PATH);
		return USP_FAULT_COMMAND_FAILURE;
	}

	char *err = dmjson_get_value(res, 1, "ErrorStatus");

	fill_blob_param(&ctx->bb, "Status", (DM_STRCMP(err, "0") != 0) ? "Error_Internal" : "Complete", DMT_TYPE[DMT_STRING], 0);

	json_object *jobj = dmjson_get_obj(res, 1, "Output");
	if (jobj) {
		char *BOMTime = dmjson_get_value(jobj, 1, "BOMTime");
		char *EOMTime = dmjson_get_value(jobj, 1, "EOMTime");
		char *TmaxUsed = dmjson_get_value(jobj, 1, "TmaxUsed");
		char *TestInterval = dmjson_get_value(jobj, 1, "TestInterval");
		char *TmaxRTTUsed = dmjson_get_value(jobj, 1, "TmaxRTTUsed");
		char *TimestampResolutionUsed = dmjson_get_value(jobj, 1, "TimestampResolutionUsed");

		fill_blob_param(&ctx->bb, "BOMTime", BOMTime, DMT_TYPE[DMT_TIME], 0);
		fill_blob_param(&ctx->bb, "EOMTime", EOMTime, DMT_TYPE[DMT_TIME], 0);
		fill_blob_param(&ctx->bb, "TmaxUsed", TmaxUsed, DMT_TYPE[DMT_UNINT], 0);
		fill_blob_param(&ctx->bb, "TestInterval", TestInterval, DMT_TYPE[DMT_UNINT], 0);
		fill_blob_param(&ctx->bb, "TmaxRTTUsed", TmaxRTTUsed, DMT_TYPE[DMT_UNINT], 0);
		fill_blob_param(&ctx->bb, "TimestampResolutionUsed", TimestampResolutionUsed, DMT_TYPE[DMT_UNINT], 0);

		json_object *atmax = dmjson_get_obj(jobj, 1, "AtMax");
		if (atmax) {
			char *MaxIPLayerCapacity = dmjson_get_value(atmax, 1, "MaxIPLayerCapacity");
			char *TimeOfMax = dmjson_get_value(atmax, 1, "TimeOfMax");
			char *MaxETHCapacityNoFCS = dmjson_get_value(atmax, 1, "MaxETHCapacityNoFCS");
			char *MaxETHCapacityWithFCS = dmjson_get_value(atmax, 1, "MaxETHCapacityWithFCS");
			char *MaxETHCapacityWithFCSVLAN = dmjson_get_value(atmax, 1, "MaxETHCapacityWithFCSVLAN");
			char *LossRatioAtMax = dmjson_get_value(atmax, 1, "LossRatioAtMax");
			char *RTTRangeAtMax = dmjson_get_value(atmax, 1, "RTTRangeAtMax");
			char *PDVRangeAtMax = dmjson_get_value(atmax, 1, "PDVRangeAtMax");
			char *MinOnewayDelayAtMax = dmjson_get_value(atmax, 1, "MinOnewayDelayAtMax");
			char *ReorderedRatioAtMax = dmjson_get_value(atmax, 1, "ReorderedRatioAtMax");
			char *ReplicatedRatioAtMax = dmjson_get_value(atmax, 1, "ReplicatedRatioAtMax");
			char *InterfaceEthMbpsAtMax = dmjson_get_value(atmax, 1, "InterfaceEthMbps");

			fill_blob_param(&ctx->bb, "MaxIPLayerCapacity", MaxIPLayerCapacity, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "TimeOfMax", TimeOfMax, DMT_TYPE[DMT_TIME], 0);
			fill_blob_param(&ctx->bb, "MaxETHCapacityNoFCS", MaxETHCapacityNoFCS, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "MaxETHCapacityWithFCS", MaxETHCapacityWithFCS, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "MaxETHCapacityWithFCSVLAN", MaxETHCapacityWithFCSVLAN, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "LossRatioAtMax", LossRatioAtMax, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "RTTRangeAtMax", RTTRangeAtMax, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "PDVRangeAtMax", PDVRangeAtMax, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "MinOnewayDelayAtMax", MinOnewayDelayAtMax, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "ReorderedRatioAtMax", ReorderedRatioAtMax, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "ReplicatedRatioAtMax", ReplicatedRatioAtMax, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "InterfaceEthMbpsAtMax", InterfaceEthMbpsAtMax, DMT_TYPE[DMT_STRING], 0);
		}

		json_object *sum = dmjson_get_obj(jobj, 1, "Summary");
		if (sum) {
			char *IPLayerCapacitySummary = dmjson_get_value(sum, 1, "IPLayerCapacitySummary");
			char *LossRatioSummary = dmjson_get_value(sum, 1, "LossRatioSummary");
			char *RTTRangeSummary = dmjson_get_value(sum, 1, "RTTRangeSummary");
			char *PDVRangeSummary = dmjson_get_value(sum, 1, "PDVRangeSummary");
			char *MinOnewayDelaySummary = dmjson_get_value(sum, 1, "MinOnewayDelaySummary");
			char *MinRTTSummary = dmjson_get_value(sum, 1, "MinRTTSummary");
			char *ReorderedRatioSummary = dmjson_get_value(sum, 1, "ReorderedRatioSummary");
			char *ReplicatedRatioSummary = dmjson_get_value(sum, 1, "ReplicatedRatioSummary");
			char *InterfaceEthMbpsSummary = dmjson_get_value(sum, 1, "InterfaceEthMbps");

			fill_blob_param(&ctx->bb, "IPLayerCapacitySummary", IPLayerCapacitySummary, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "LossRatioSummary", LossRatioSummary, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "RTTRangeSummary", RTTRangeSummary, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "PDVRangeSummary", PDVRangeSummary, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "MinOnewayDelaySummary", MinOnewayDelaySummary, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "MinRTTSummary", MinRTTSummary, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "ReorderedRatioSummary", ReorderedRatioSummary, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "ReplicatedRatioSummary", ReplicatedRatioSummary, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "InterfaceEthMbpsSummary", InterfaceEthMbpsSummary, DMT_TYPE[DMT_STRING], 0);
		}

		json_object *arrobj = NULL, *modal = NULL;
		dmjson_foreach_obj_in_array(jobj, arrobj, modal, idx, 1, "ModalResult") {
			unsigned int i = idx + 1;
			char buf[256] = {0};

			char *MaxIPLayerCapacity = dmjson_get_value(modal, 1, "MaxIPLayerCapacity");
			snprintf(buf, sizeof(buf), "ModalResult.%u.MaxIPLayerCapacity", i);
			fill_blob_param(&ctx->bb, buf, MaxIPLayerCapacity, DMT_TYPE[DMT_STRING], 0);
			
			char *TimeOfMax = dmjson_get_value(modal, 1, "TimeOfMax");
			snprintf(buf, sizeof(buf), "ModalResult.%u.TimeOfMax", i);
			fill_blob_param(&ctx->bb, buf, TimeOfMax, DMT_TYPE[DMT_TIME], 0);

			char *MaxETHCapacityNoFCS = dmjson_get_value(modal, 1, "MaxETHCapacityNoFCS");
			snprintf(buf, sizeof(buf), "ModalResult.%u.MaxETHCapacityNoFCS", i);
			fill_blob_param(&ctx->bb, buf, MaxETHCapacityNoFCS, DMT_TYPE[DMT_STRING], 0);

			char *MaxETHCapacityWithFCS = dmjson_get_value(modal, 1, "MaxETHCapacityWithFCS");
			snprintf(buf, sizeof(buf), "ModalResult.%u.MaxETHCapacityWithFCS", i);
			fill_blob_param(&ctx->bb, buf, MaxETHCapacityWithFCS, DMT_TYPE[DMT_STRING], 0);

			char *MaxETHCapacityWithFCSVLAN = dmjson_get_value(modal, 1, "MaxETHCapacityWithFCSVLAN");
			snprintf(buf, sizeof(buf), "ModalResult.%u.MaxETHCapacityWithFCSVLAN", i);
			fill_blob_param(&ctx->bb, buf, MaxETHCapacityWithFCSVLAN, DMT_TYPE[DMT_STRING], 0);

			char *LossRatioAtMax = dmjson_get_value(modal, 1, "LossRatioAtMax");
			snprintf(buf, sizeof(buf), "ModalResult.%u.LossRatioAtMax", i);
			fill_blob_param(&ctx->bb, buf, LossRatioAtMax, DMT_TYPE[DMT_STRING], 0);

			char *RTTRangeAtMax = dmjson_get_value(modal, 1, "RTTRangeAtMax");
			snprintf(buf, sizeof(buf), "ModalResult.%u.RTTRangeAtMax", i);
			fill_blob_param(&ctx->bb, buf, RTTRangeAtMax, DMT_TYPE[DMT_STRING], 0);

			char *PDVRangeAtMax = dmjson_get_value(modal, 1, "PDVRangeAtMax");
			snprintf(buf, sizeof(buf), "ModalResult.%u.PDVRangeAtMax", i);
			fill_blob_param(&ctx->bb, buf, PDVRangeAtMax, DMT_TYPE[DMT_STRING], 0);

			char *MinOnewayDelayAtMax = dmjson_get_value(modal, 1, "MinOnewayDelayAtMax");
			snprintf(buf, sizeof(buf), "ModalResult.%u.MinOnewayDelayAtMax", i);
			fill_blob_param(&ctx->bb, buf, MinOnewayDelayAtMax, DMT_TYPE[DMT_STRING], 0);

			char *ReorderedRatioAtMax = dmjson_get_value(modal, 1, "ReorderedRatioAtMax");
			snprintf(buf, sizeof(buf), "ModalResult.%u.ReorderedRatioAtMax", i);
			fill_blob_param(&ctx->bb, buf, ReorderedRatioAtMax, DMT_TYPE[DMT_STRING], 0);

			char *ReplicatedRatioAtMax = dmjson_get_value(modal, 1, "ReplicatedRatioAtMax");
			snprintf(buf, sizeof(buf), "ModalResult.%u.ReplicatedRatioAtMax", i);
			fill_blob_param(&ctx->bb, buf, ReplicatedRatioAtMax, DMT_TYPE[DMT_STRING], 0);

			char *InterfaceEthMbpsAtMax = dmjson_get_value(modal, 1, "InterfaceEthMbps");
			snprintf(buf, sizeof(buf), "ModalResult.%u.InterfaceEthMbpsAtMax", i);
			fill_blob_param(&ctx->bb, buf, InterfaceEthMbpsAtMax, DMT_TYPE[DMT_STRING], 0);
		}

		idx = 0;
		json_object *arrob = NULL, *incremental = NULL;
		dmjson_foreach_obj_in_array(jobj, arrob, incremental, idx, 1, "IncrementalResult") {
			unsigned int i = idx + 1;
			char buf[256] = {0};

			char *IPLayerCapacity = dmjson_get_value(incremental, 1, "IPLayerCapacity");
			snprintf(buf, sizeof(buf), "IncrementalResult.%u.IPLayerCapacity", i);
			fill_blob_param(&ctx->bb, buf, IPLayerCapacity, DMT_TYPE[DMT_STRING], 0);

			char *TimeOfSubInterval = dmjson_get_value(incremental, 1, "TimeOfSubInterval");
			snprintf(buf, sizeof(buf), "IncrementalResult.%u.TimeOfSubInterval", i);
			fill_blob_param(&ctx->bb, buf, TimeOfSubInterval, DMT_TYPE[DMT_STRING], 0);

			char *LossRatio = dmjson_get_value(incremental, 1, "LossRatio");
			snprintf(buf, sizeof(buf), "IncrementalResult.%u.LossRatio", i);
			fill_blob_param(&ctx->bb, buf, LossRatio, DMT_TYPE[DMT_STRING], 0);

			char *RTTRange = dmjson_get_value(incremental, 1, "RTTRange");
			snprintf(buf, sizeof(buf), "IncrementalResult.%u.RTTRange", i);
			fill_blob_param(&ctx->bb, buf, RTTRange, DMT_TYPE[DMT_STRING], 0);

			char *PDVRange = dmjson_get_value(incremental, 1, "PDVRange");
			snprintf(buf, sizeof(buf), "IncrementalResult.%u.PDVRange", i);
			fill_blob_param(&ctx->bb, buf, PDVRange, DMT_TYPE[DMT_STRING], 0);

			char *MinOnewayDelay = dmjson_get_value(incremental, 1, "MinOnewayDelay");
			snprintf(buf, sizeof(buf), "IncrementalResult.%u.MinOnewayDelay", i);
			fill_blob_param(&ctx->bb, buf, MinOnewayDelay, DMT_TYPE[DMT_STRING], 0);

			char *ReorderedRatio = dmjson_get_value(incremental, 1, "ReorderedRatio");
			snprintf(buf, sizeof(buf), "IncrementalResult.%u.ReorderedRatio", i);
			fill_blob_param(&ctx->bb, buf, ReorderedRatio, DMT_TYPE[DMT_STRING], 0);

			char *ReplicatedRatio = dmjson_get_value(incremental, 1, "ReplicatedRatio");
			snprintf(buf, sizeof(buf), "IncrementalResult.%u.ReplicatedRatio", i);
			fill_blob_param(&ctx->bb, buf, ReplicatedRatio, DMT_TYPE[DMT_STRING], 0);

			char *InterfaceEthMbps = dmjson_get_value(incremental, 1, "InterfaceEthMbps");
			snprintf(buf, sizeof(buf), "IncrementalResult.%u.InterfaceEthMbps", i);
			fill_blob_param(&ctx->bb, buf, InterfaceEthMbps, DMT_TYPE[DMT_STRING], 0);
		}
	}

	if (res != NULL)
		json_object_put(res);

	return 0;
}

/**********************************************************************************************************************************
*                                            OBJ & LEAF DEFINITION
***********************************************************************************************************************************/

/* *** Device.IP.Diagnostics.IPLayerCapacityMetrics.{i}.ModalResult. *** */
DMLEAF tIPLayerCapacityModalResultParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"MaxIPLayerCapacity", &DMREAD, DMT_STRING, get_IPLayerCapacityModal_MaxIPLayerCapacity, NULL, BBFDM_CWMP},
{"TimeOfMax", &DMREAD, DMT_TIME, get_IPLayerCapacityModal_TimeOfMax, NULL, BBFDM_CWMP},
{"MaxETHCapacityNoFCS", &DMREAD, DMT_STRING, get_IPLayerCapacityModal_MaxETHCapacityNoFCS, NULL, BBFDM_CWMP},
{"MaxETHCapacityWithFCS", &DMREAD, DMT_STRING, get_IPLayerCapacityModal_MaxETHCapacityWithFCS, NULL, BBFDM_CWMP},
{"MaxETHCapacityWithFCSVLAN", &DMREAD, DMT_STRING, get_IPLayerCapacityModal_MaxETHCapacityWithFCSVLAN, NULL, BBFDM_CWMP},
{"LossRatioAtMax", &DMREAD, DMT_STRING, get_IPLayerCapacityModal_LossRatioAtMax, NULL, BBFDM_CWMP},
{"RTTRangeAtMax", &DMREAD, DMT_STRING, get_IPLayerCapacityModal_RTTRangeAtMax, NULL, BBFDM_CWMP},
{"PDVRangeAtMax", &DMREAD, DMT_STRING, get_IPLayerCapacityModal_PDVRangeAtMax, NULL, BBFDM_CWMP},
{"MinOnewayDelayAtMax", &DMREAD, DMT_STRING, get_IPLayerCapacityModal_MinOnewayDelayAtMax, NULL, BBFDM_CWMP},
{"ReorderedRatioAtMax", &DMREAD, DMT_STRING, get_IPLayerCapacityModal_ReorderedRatioAtMax, NULL, BBFDM_CWMP},
{"ReplicatedRatioAtMax", &DMREAD, DMT_STRING, get_IPLayerCapacityModal_ReplicatedRatioAtMax, NULL, BBFDM_CWMP},
{"InterfaceEthMbpsAtMax", &DMREAD, DMT_STRING, get_IPLayerCapacityModal_InterfaceEthMbpsAtMax, NULL, BBFDM_CWMP},
{0}
};

/* *** Device.IP.Diagnostics.IPLayerCapacityMetrics.{i}.IncrementalResult. *** */
DMLEAF tIPLayerCapacityIncrementalResultParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"IPLayerCapacity", &DMREAD, DMT_STRING, get_IPLayerCapacityIncremental_IPLayerCapacity, NULL, BBFDM_CWMP},
{"TimeOfSubInterval", &DMREAD, DMT_TIME, get_IPLayerCapacityIncremental_TimeOfSubInterval, NULL, BBFDM_CWMP},
{"LossRatio", &DMREAD, DMT_STRING, get_IPLayerCapacityIncremental_LossRatio, NULL, BBFDM_CWMP},
{"RTTRange", &DMREAD, DMT_STRING, get_IPLayerCapacityIncremental_RTTRange, NULL, BBFDM_CWMP},
{"PDVRange", &DMREAD, DMT_STRING, get_IPLayerCapacityIncremental_PDVRange, NULL, BBFDM_CWMP},
{"MinOnewayDelay", &DMREAD, DMT_STRING, get_IPLayerCapacityIncremental_MinOnewayDelay, NULL, BBFDM_CWMP},
{"ReorderedRatio", &DMREAD, DMT_STRING, get_IPLayerCapacityIncremental_ReorderedRatio, NULL, BBFDM_CWMP},
{"ReplicatedRatio", &DMREAD, DMT_STRING, get_IPLayerCapacityIncremental_ReplicatedRatio, NULL, BBFDM_CWMP},
{"InterfaceEthMbps", &DMREAD, DMT_STRING, get_IPLayerCapacityIncremental_InterfaceEthMbps, NULL, BBFDM_CWMP},
{0}
};

/* *** Device.IP.Diagnostics.IPLayerCapacityMetrics. *** */
DMOBJ tIPLayerCapacityObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys*/
{"ModalResult", &DMREAD, NULL, NULL, NULL, browseIPLayerCapacityModalResultInst, NULL, NULL, NULL, tIPLayerCapacityModalResultParams, NULL, BBFDM_CWMP},
{"IncrementalResult", &DMREAD, NULL, NULL, NULL, browseIPLayerCapacityIncrementalResultInst, NULL, NULL, NULL, tIPLayerCapacityIncrementalResultParams, NULL, BBFDM_CWMP},
{0}
};

DMLEAF tIPLayerCapacityParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"DiagnosticsState", &DMWRITE, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_DiagnosticsState, set_IPDiagnosticsIPLayerCapacity_DiagnosticsState, BBFDM_CWMP},
{"IPLayerMaxConnections", &DMREAD, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_MaxConnections, NULL, BBFDM_CWMP},
{"IPLayerMaxIncrementalResult", &DMREAD, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_MaxIncrementalResult, NULL, BBFDM_CWMP},
{"IPLayerCapSupportedSoftwareVersion", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_SoftwareVersion, NULL, BBFDM_CWMP},
{"IPLayerCapSupportedControlProtocolVersion", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_ControlProtocolVersion, NULL, BBFDM_CWMP},
{"IPLayerCapSupportedMetrics", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_SupportedMetrics, NULL, BBFDM_CWMP},
{"Interface", &DMWRITE, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_Interface, set_IPDiagnosticsIPLayerCapacity_Interface, BBFDM_CWMP},
{"Role", &DMWRITE, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_Role, set_IPDiagnosticsIPLayerCapacity_Role, BBFDM_CWMP},
{"Host", &DMWRITE, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_Host, set_IPDiagnosticsIPLayerCapacity_Host, BBFDM_CWMP},
{"Port", &DMWRITE, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_Port, set_IPDiagnosticsIPLayerCapacity_Port, BBFDM_CWMP},
{"JumboFramesPermitted", &DMWRITE, DMT_BOOL, get_IPDiagnosticsIPLayerCapacity_JumboFramesPermitted, set_IPDiagnosticsIPLayerCapacity_JumboFramesPermitted, BBFDM_CWMP},
{"DSCP", &DMWRITE, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_DSCP, set_IPDiagnosticsIPLayerCapacity_DSCP, BBFDM_CWMP},
{"ProtocolVersion", &DMWRITE, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_ProtocolVersion, set_IPDiagnosticsIPLayerCapacity_ProtocolVersion, BBFDM_CWMP},
{"UDPPayloadContent", &DMWRITE, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_UDPPayloadContent, set_IPDiagnosticsIPLayerCapacity_UDPPayloadContent, BBFDM_CWMP},
{"TestType", &DMWRITE, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_TestType, set_IPDiagnosticsIPLayerCapacity_TestType, BBFDM_CWMP},
{"IPDVEnable", &DMWRITE, DMT_BOOL, get_IPDiagnosticsIPLayerCapacity_IPDVEnable, set_IPDiagnosticsIPLayerCapacity_IPDVEnable, BBFDM_CWMP},
{"StartSendingRateIndex", &DMWRITE, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_StartSendingRateIndex, set_IPDiagnosticsIPLayerCapacity_StartSendingRateIndex, BBFDM_CWMP},
{"NumberTestSubIntervals", &DMWRITE, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_NumberTestSubIntervals, set_IPDiagnosticsIPLayerCapacity_NumberTestSubIntervals, BBFDM_CWMP},
{"NumberFirstModeTestSubIntervals", &DMWRITE, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_NumberFirstModeTestSubIntervals, set_IPDiagnosticsIPLayerCapacity_NumberFirstModeTestSubIntervals, BBFDM_CWMP},
{"TestSubInterval", &DMWRITE, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_TestSubInterval, set_IPDiagnosticsIPLayerCapacity_TestSubInterval, BBFDM_CWMP},
{"StatusFeedbackInterval", &DMWRITE, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_StatusFeedbackInterval, set_IPDiagnosticsIPLayerCapacity_StatusFeedbackInterval, BBFDM_CWMP},
{"SeqErrThresh", &DMWRITE, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_SeqErrThresh, set_IPDiagnosticsIPLayerCapacity_SeqErrThresh, BBFDM_CWMP},
{"ReordDupIgnoreEnable", &DMWRITE, DMT_BOOL, get_IPDiagnosticsIPLayerCapacity_ReordDupIgnoreEnable, set_IPDiagnosticsIPLayerCapacity_ReordDupIgnoreEnable, BBFDM_CWMP},
{"LowerThresh", &DMWRITE, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_LowerThresh, set_IPDiagnosticsIPLayerCapacity_LowerThresh, BBFDM_CWMP},
{"UpperThresh", &DMWRITE, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_UpperThresh, set_IPDiagnosticsIPLayerCapacity_UpperThresh, BBFDM_CWMP},
{"HighSpeedDelta", &DMWRITE, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_HighSpeedDelta, set_IPDiagnosticsIPLayerCapacity_HighSpeedDelta, BBFDM_CWMP},
{"RateAdjAlgorithm", &DMWRITE, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_RateAdjAlgorithm, set_IPDiagnosticsIPLayerCapacity_RateAdjAlgorithm, BBFDM_CWMP},
{"SlowAdjThresh", &DMWRITE, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_SlowAdjThresh, set_IPDiagnosticsIPLayerCapacity_SlowAdjThresh, BBFDM_CWMP},
{"BOMTime", &DMREAD, DMT_TIME, get_IPDiagnosticsIPLayerCapacity_BOMTime, NULL, BBFDM_CWMP},
{"EOMTime", &DMREAD, DMT_TIME, get_IPDiagnosticsIPLayerCapacity_EOMTime, NULL, BBFDM_CWMP},
{"TmaxUsed", &DMREAD, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_TmaxUsed, NULL, BBFDM_CWMP},
{"TestInterval", &DMREAD, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_TestInterval, NULL, BBFDM_CWMP},
{"MaxIPLayerCapacity", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_MaxIPLayerCapacity, NULL, BBFDM_CWMP},
{"TimeOfMax", &DMREAD, DMT_TIME, get_IPDiagnosticsIPLayerCapacity_TimeOfMax, NULL, BBFDM_CWMP},
{"MaxETHCapacityNoFCS", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_MaxETHCapacityNoFCS, NULL, BBFDM_CWMP},
{"MaxETHCapacityWithFCS", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_MaxETHCapacityWithFCS, NULL, BBFDM_CWMP},
{"MaxETHCapacityWithFCSVLAN", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_MaxETHCapacityWithFCSVLAN, NULL, BBFDM_CWMP},
{"LossRatioAtMax", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_LossRatioAtMax, NULL, BBFDM_CWMP},
{"RTTRangeAtMax", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_RTTRangeAtMax, NULL, BBFDM_CWMP},
{"PDVRangeAtMax", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_PDVRangeAtMax, NULL, BBFDM_CWMP},
{"MinOnewayDelayAtMax", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_MinOnewayDelayAtMax, NULL, BBFDM_CWMP},
{"ReorderedRatioAtMax", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_ReorderedRatioAtMax, NULL, BBFDM_CWMP},
{"ReplicatedRatioAtMax", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_ReplicatedRatioAtMax, NULL, BBFDM_CWMP},
{"InterfaceEthMbpsAtMax", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_InterfaceEthMbpsAtMax, NULL, BBFDM_CWMP},
{"IPLayerCapacitySummary", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_IPLayerCapacitySummary, NULL, BBFDM_CWMP},
{"LossRatioSummary", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_LossRatioSummary, NULL, BBFDM_CWMP},
{"RTTRangeSummary", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_RTTRangeSummary, NULL, BBFDM_CWMP},
{"PDVRangeSummary", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_PDVRangeSummary, NULL, BBFDM_CWMP},
{"MinOnewayDelaySummary", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_MinOnewayDelaySummary, NULL, BBFDM_CWMP},
{"MinRTTSummary", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_MinRTTSummary, NULL, BBFDM_CWMP},
{"ReorderedRatioSummary", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_ReorderedRatioSummary, NULL, BBFDM_CWMP},
{"ReplicatedRatioSummary", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_ReplicatedRatioSummary, NULL, BBFDM_CWMP},
{"InterfaceEthMbpsSummary", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_InterfaceEthMbpsSummary, NULL, BBFDM_CWMP},
{"IncrementalResultNumberOfEntries", &DMREAD, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_IncrementalResultNumberOfEntries, NULL, BBFDM_CWMP},
{"ModalResultNumberOfEntries", &DMREAD, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_ModalResultNumberOfEntries, NULL, BBFDM_CWMP},
{"TmaxRTTUsed", &DMREAD, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_TmaxRTTUsed, NULL, BBFDM_CWMP},
{"TimestampResolutionUsed", &DMREAD, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_TimestampResolutionUsed, NULL, BBFDM_CWMP},
{0}
};

/* *** Device.IP.Diagnostics. *** */
DMOBJ tDeviceIPDiagnosticsObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys*/
{"IPLayerCapacityMetrics", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tIPLayerCapacityObj, tIPLayerCapacityParams, NULL, BBFDM_CWMP},
{0}
};

DMLEAF tDeviceIPDiagnosticsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"IPLayerCapacitySupported", &DMREAD, DMT_BOOL, get_IPDiagnosticsIPLayerCapacity_CapacitySupported, NULL, BBFDM_BOTH},
{"IPLayerMaxConnections", &DMREAD, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_MaxConnections, NULL, BBFDM_USP},
{"IPLayerMaxIncrementalResult", &DMREAD, DMT_UNINT, get_IPDiagnosticsIPLayerCapacity_MaxIncrementalResult, NULL, BBFDM_USP},
{"IPLayerCapSupportedSoftwareVersion", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_SoftwareVersion, NULL, BBFDM_USP},
{"IPLayerCapSupportedControlProtocolVersion", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_ControlProtocolVersion, NULL, BBFDM_USP},
{"IPLayerCapSupportedMetrics", &DMREAD, DMT_STRING, get_IPDiagnosticsIPLayerCapacity_SupportedMetrics, NULL, BBFDM_USP},
{"IPLayerCapacity()", &DMASYNC, DMT_COMMAND, get_operate_args_IPDiagnostics_IPLayerCapacity, operate_IPDiagnostics_IPLayerCapacity, BBFDM_USP},
{0}
};

/* *** Device.IP. *** */
DMOBJ tDeviceIPObjs[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Diagnostics", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tDeviceIPDiagnosticsObj, tDeviceIPDiagnosticsParams, NULL, BBFDM_BOTH, NULL},
{0}
};

/* ********** DynamicObj ********** */
DM_MAP_OBJ tDynamicObj[] = {
/* parentobj, nextobject, parameter */
{"Device.IP.", tDeviceIPObjs, NULL},
{0}
};
