/*
 * Copyright (C) 2023 iopsys Software Solutions AB
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1
 * as published by the Free Software Foundation
 *
 *	  Author: Suvendhu Hansa <suvendhu.hansa@iopsys.eu>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <uci.h>
#include <libubox/uloop.h>
#include <libubus.h>
#include <libbbfdm-ubus/bbfdm-ubus.h>

#include "helper.h"

static int timemngr_uci_init(void);
static int timemngr_uci_fini(void);
static bool timemngr_uci_validate_section(const char *str);
static int timemngr_uci_init_ptr(struct uci_ptr *ptr, const char *package, const char *section, const char *option, const char *value);
static struct uci_section *timemngr_uci_walk_section(const char *package, const char *section_type, struct uci_section *prev_section);
static struct uci_element *timemngr_uci_lookup_list(struct uci_list *list, const char *name);
static int timemngr_uci_lookup_ptr_by_section(struct uci_ptr *ptr, struct uci_section *section, const char *option, const char *value);
static void timemngr_uci_get_value_by_section(struct uci_section *section, const char *option, char *value, uint32_t len);


#define timemngr_uci_foreach_section_safe(package, section_type, _tmp, section) \
	for(section = timemngr_uci_walk_section(package, section_type, NULL), \
		_tmp = (section) ? timemngr_uci_walk_section(package, section_type, section) : NULL; \
		section != NULL; \
		section = _tmp, _tmp = (section) ? timemngr_uci_walk_section(package, section_type, section) : NULL)

static bool get_timemngr_enable(void);
static bool get_timemngr_server_enable(void);
static bool get_timemngr_client_enable(const char *sec);
static bool ntpd_running(void);

static struct uci_context *uci_ctx_timemngr = NULL;

static int timemngr_uci_init(void)
{
	uci_ctx_timemngr = uci_alloc_context();
	if (!uci_ctx_timemngr)
		return -1;

	uci_set_confdir(uci_ctx_timemngr, "/etc/config/");

	return 0;
}

static int timemngr_uci_fini(void)
{
	/* Freed uci context  */
	if (uci_ctx_timemngr)
		uci_free_context(uci_ctx_timemngr);

	return 0;
}

static bool timemngr_uci_validate_section(const char *str)
{
	if (!*str)
		return false;

	for (; *str; str++) {
		unsigned char c = *str;

		if (isalnum(c) || c == '_')
			continue;

		return false;
	}

	return true;
}

static int timemngr_uci_init_ptr(struct uci_ptr *ptr, const char *package, const char *section, const char *option, const char *value)
{
	memset(ptr, 0, sizeof(struct uci_ptr));

	/* value */
	if (value)
		ptr->value = value;

	ptr->package = package;
	if (!ptr->package)
		goto error;

	ptr->section = section;
	if (!ptr->section) {
		ptr->target = UCI_TYPE_PACKAGE;
		goto lastval;
	}

	ptr->option = option;
	if (!ptr->option) {
		ptr->target = UCI_TYPE_SECTION;
		goto lastval;
	} else
		ptr->target = UCI_TYPE_OPTION;

lastval:
	if (ptr->section && !timemngr_uci_validate_section(ptr->section))
		ptr->flags |= UCI_LOOKUP_EXTENDED;

	return 0;

error:
	return -1;
}

static struct uci_section *timemngr_uci_walk_section(const char *package, const char *section_type, struct uci_section *prev_section)
{
	struct uci_ptr ptr;
	struct uci_element *e;
	struct uci_section *next_section;

	if (section_type == NULL) {
		if (prev_section) {
			e = &prev_section->e;
			if (e->list.next == &prev_section->package->sections)
				return NULL;
			e = container_of(e->list.next, struct uci_element, list);
			next_section = uci_to_section(e);
			return next_section;
		} else {
			if (timemngr_uci_init_ptr(&ptr, package, NULL, NULL, NULL))
				return NULL;

			if (uci_lookup_ptr(uci_ctx_timemngr, &ptr, NULL, true) != UCI_OK)
				return NULL;

			if (ptr.p->sections.next == &ptr.p->sections)
				return NULL;
			e = container_of(ptr.p->sections.next, struct uci_element, list);
			next_section = uci_to_section(e);

			return next_section;
		}
	} else {
		struct uci_list *ul = NULL, *shead = NULL;

		if (prev_section) {
			ul = &prev_section->e.list;
			shead = &prev_section->package->sections;
		} else {
			if (timemngr_uci_init_ptr(&ptr, package, NULL, NULL, NULL))
				return NULL;

			if (uci_lookup_ptr(uci_ctx_timemngr, &ptr, NULL, true) != UCI_OK)
				return NULL;

			ul = &ptr.p->sections;
			shead = &ptr.p->sections;
		}
		while (ul->next != shead) {
			e = container_of(ul->next, struct uci_element, list);
			next_section = uci_to_section(e);
			if (TIME_STRCMP(next_section->type, section_type) == 0)
				return next_section;
			ul = ul->next;
		}
		return NULL;
	}
	return NULL;
}

static struct uci_element *timemngr_uci_lookup_list(struct uci_list *list, const char *name)
{
	struct uci_element *e;

	uci_foreach_element(list, e) {
		if (!TIME_STRCMP(e->name, name))
			return e;
	}

	return NULL;
}

static int timemngr_uci_lookup_ptr_by_section(struct uci_ptr *ptr, struct uci_section *section, const char *option, const char *value)
{
	struct uci_element *e = NULL;

	memset(ptr, 0, sizeof(struct uci_ptr));

	ptr->package = section->package->e.name;
	ptr->section = section->e.name;
	ptr->option = option;
	ptr->value = value;
	ptr->flags |= UCI_LOOKUP_DONE;

	ptr->p = section->package;
	ptr->s = section;

	if (ptr->option) {
		e = timemngr_uci_lookup_list(&ptr->s->options, ptr->option);
		if (!e)
			return UCI_OK;
		ptr->o = uci_to_option(e);
		ptr->last = e;
		ptr->target = UCI_TYPE_OPTION;
	} else {
		ptr->last = &ptr->s->e;
		ptr->target = UCI_TYPE_SECTION;
	}

	ptr->flags |= UCI_LOOKUP_COMPLETE;

	return UCI_OK;
}

static void timemngr_uci_get_value_by_section(struct uci_section *section, const char *option, char *value, uint32_t len)
{
	struct uci_ptr ptr = {0};

	if (!value || !len)
		return;

	memset(value, 0, len);

	if (timemngr_uci_lookup_ptr_by_section(&ptr, section, option, NULL) != UCI_OK)
		return;

	if (!ptr.o)
		return;

	if (ptr.o->v.string) {
		snprintf(value, len, "%s", ptr.o->v.string);
	}

	return;
}

static bool get_timemngr_enable(void)
{
	char enable[10] = {0};
	if (timemngr_uci_init() != 0)
		return false;
	
	struct uci_section *s = NULL, *stmp = NULL;
	timemngr_uci_foreach_section_safe("time", "global", stmp, s) {
		timemngr_uci_get_value_by_section(s, "enable", enable, sizeof(enable));
		break;
	}

	timemngr_uci_fini();

	if (TIME_STRLEN(enable) == 0 || TIME_STRCMP(enable, "1") == 0)
		return true;
	
	return false;
}

static bool get_timemngr_server_enable(void)
{
	char enable[10] = {0};
	if (timemngr_uci_init() != 0)
		return false;
	
	struct uci_section *s = NULL, *stmp = NULL;
	timemngr_uci_foreach_section_safe("time", "server", stmp, s) {
		timemngr_uci_get_value_by_section(s, "enable", enable, sizeof(enable));
		break;
	}

	timemngr_uci_fini();

	if (TIME_STRLEN(enable) == 0 || TIME_STRCMP(enable, "1") == 0)
		return true;
	
	return false;
}

static bool get_timemngr_client_enable(const char *sec)
{
	bool enable = false;

	if (timemngr_uci_init() != 0)
		return enable;
	
	struct uci_section *s = NULL, *stmp = NULL;
	timemngr_uci_foreach_section_safe("time", "client", stmp, s) {
		char value[10] = {0};
		timemngr_uci_get_value_by_section(s, "enable", value, sizeof(value));
		if (TIME_STRLEN(value) == 0 || TIME_STRCMP(value, "1") == 0) {
			if (sec == NULL || TIME_STRCMP(sec, s->e.name) == 0) {
				enable = true;
				break;
			}
		}
	}

	timemngr_uci_fini();

	return enable;
}

static bool ntpd_running(void)
{
	bool running = false;
	char buff[256] = {0};
	fd_set fdset;
	struct timeval timeout;

	timeout.tv_sec = 2;
	timeout.tv_usec = 0;

	FILE *fp = popen("pgrep /sbin/ntpd", "r"); // flawfinder: ignore
	if (!fp) {
		PRINT_ERR("popen failed");
		return running;
	}

	int fd = fileno(fp);
	FD_ZERO(&fdset);
	FD_SET(fd, &fdset);

	int rc = select(fd+1, &fdset, 0, 0, &timeout);
	if (rc == 0) {
		PRINT_DEBUG("select timedout");
		pclose(fp);
		return running;
	}

	if ((NULL != fgets(buff, sizeof(buff), fp)) && (TIME_STRLEN(buff) != 0)) {
		running = true;
	}

	pclose(fp);

	return running;
}

bool ntpd_time_sync(void)
{
	bool unsync = true;
	char buff[256] = {0};
	fd_set fdset;
	struct timeval timeout;

	timeout.tv_sec = 2;
	timeout.tv_usec = 0;

	bool timemngr_enable = get_timemngr_enable();
	if (timemngr_enable == false) {
		PRINT_ERR("timemngr is disabled");
		return unsync;
	}

	FILE *fp = popen("ntpq -c \'rv 0 stratum\'", "r"); // flawfinder: ignore
	if (!fp) {
		PRINT_ERR("popen failed");
		return unsync;
	}

	int fd = fileno(fp);
	FD_ZERO(&fdset);
	FD_SET(fd, &fdset);

	int rc = select(fd+1, &fdset, 0, 0, &timeout);
	if (rc == 0) {
		PRINT_DEBUG("select timedout");
		pclose(fp);
		return unsync;
	}

	if (NULL != fgets(buff, sizeof(buff), fp)) {
		PRINT_DEBUG("%s", buff);
		if ((TIME_STRLEN(buff) != 0) && (TIME_STRSTR(buff, "stratum=") != NULL)) {
			char *tmp = strrchr(buff, '=');
			if (tmp && strlen(tmp+1) && (strtol(tmp+1, NULL, 10) < 16)) {
				unsync = false;
			}
		}
	}

	pclose(fp);
	return unsync;
}

static void update_ntpd_counters(struct time_args *time_stats)
{
	struct timeval timeout;
	char buff[256] = {0};
	fd_set fdset;

	timeout.tv_sec = 2;
	timeout.tv_usec = 0;

	FILE *fp = popen("ntpq -c iostats", "r"); // flawfinder: ignore
	if (!fp) {
		PRINT_ERR("popen failed");
		return;
	}

	int fd = fileno(fp);
	FD_ZERO(&fdset);
	FD_SET(fd, &fdset);

	int rc = select(fd+1, &fdset, 0, 0, &timeout);
	if (rc == 0) {
		PRINT_DEBUG("select timedout");
		pclose(fp);
		return;
	}

	while (NULL != fgets(buff, sizeof(buff), fp)) {
		if (TIME_STRSTR(buff, "packets sent:") != NULL) {
			sscanf(buff, "packets sent: %u", &time_stats->pack_sent);
		}

		if (TIME_STRSTR(buff, "packet send failures:") != NULL) {
			sscanf(buff, "packet send failures: %u", &time_stats->pack_sent_fail);
		}

		if (TIME_STRSTR(buff, "received packets:") != NULL) {
			sscanf(buff, "received packets: %u", &time_stats->pack_recv);
		}

		if (TIME_STRSTR(buff, "dropped packets:") != NULL) {
			sscanf(buff, "dropped packets: %u", &time_stats->pack_drop);
		}
	}

	pclose(fp);
}

int time_strcmp(const char *s1, const char *s2, const char *origin, int pos)
{
	if (s1 != NULL && s2 != NULL)
		return strcmp(s1, s2);
	else {
		PRINT_DEBUG("%s:%d NULL argument found", origin, pos);
		return -1;
	}
}

int time_strlen(const char *s1, const char *origin, int pos)
{
	if (s1 != NULL)
		return strlen(s1);
	else {
		PRINT_DEBUG("%s:%d NULL argument found", origin, pos);
		return 0;
	}
}

char *time_strstr(const char *s1, const char *s2, const char *origin, int pos)
{
	if (s1 != NULL && s2 != NULL)
		return strstr(s1, s2);
	else {
		PRINT_DEBUG("%s:%d NULL argument found", origin, pos);
		return NULL;
	}
}

char *time_strncpy(char *s1, const char *s2, size_t n, const char *origin, int pos)
{
	if (s1 != NULL && s2 != NULL) {
		strncpy(s1, s2, n);
		if (n > 0) {
			s1[n - 1] = '\0'; // Ensure null-termination at the end
		}
		return s1;
	} else {
		PRINT_DEBUG("%s:%d NULL argument found", origin, pos);
		return NULL;
	}
}

void fill_time_args(const char *obj, const char *sec, struct time_args *time_args)
{
	PRINT_DEBUG("timemngr: entry in %s", __func__);
	bool client_enable = false;

	if (!time_args)
		return;

	memset(time_args, 0, sizeof(struct time_args));

	if (!obj) {
		TIME_STRNCPY(time_args->status, "Error", sizeof(time_args->status));
		return;
	}

	bool timemngr_enable = get_timemngr_enable();
	PRINT_DEBUG("timemngr enable=%d", timemngr_enable);
	if (TIME_STRCMP(obj, "server") == 0) {
		if (false == timemngr_enable) {
			TIME_STRNCPY(time_args->status, "Down", sizeof(time_args->status));
			return;
		}

		PRINT_DEBUG("checking server enable");
		if (false == get_timemngr_server_enable()) {
			TIME_STRNCPY(time_args->status, "Down", sizeof(time_args->status));
			return;
		}

		PRINT_DEBUG("checking ntpd running");
		if (false == ntpd_running()) {
			TIME_STRNCPY(time_args->status, "Error", sizeof(time_args->status));
			return;
		}

		TIME_STRNCPY(time_args->status, "Up", sizeof(time_args->status));
		PRINT_DEBUG("server status=Up");

		PRINT_DEBUG("stats collected");
		update_ntpd_counters(time_args);
		return;
	} else if (TIME_STRCMP(obj, "global") == 0) {
		if (false == timemngr_enable) {
			TIME_STRNCPY(time_args->status, "Disabled", sizeof(time_args->status));
			return;
		}

		PRINT_DEBUG("checking for enabled client instance");
		// Check if any of the client instance is enabled
		if (true == get_timemngr_client_enable(NULL))
			client_enable = true;
	} else if (TIME_STRCMP(obj, "client") == 0) {
		if (sec == NULL || TIME_STRLEN(sec) == 0) {
			TIME_STRNCPY(time_args->status, "Error", sizeof(time_args->status));
			return;
		}

		if (false == timemngr_enable) {
			TIME_STRNCPY(time_args->status, "Disabled", sizeof(time_args->status));
			return;
		}

		PRINT_DEBUG("checking client section=%s enabled", sec);
		// Check the client instance is enabled
		if (true == get_timemngr_client_enable(sec))
			client_enable = true;
	} else {
		PRINT_ERR("Invalid object %s", obj);
		TIME_STRNCPY(time_args->status, "Error", sizeof(time_args->status));
		return;
	}

	if (true == client_enable) {
		PRINT_DEBUG("checking ntpd running");
		if (false == ntpd_running()) {
			TIME_STRNCPY(time_args->status, "Error", sizeof(time_args->status));
			return;
		}

		if (TIME_STRCMP(obj, "client") == 0) {
			PRINT_DEBUG("stats collected");
			update_ntpd_counters(time_args);
		}

		PRINT_DEBUG("checking sync status");
		if (false == ntpd_time_sync()) {
			TIME_STRNCPY(time_args->status, "Synchronized", sizeof(time_args->status));
			return;
		}

		TIME_STRNCPY(time_args->status, "Unsynchronized", sizeof(time_args->status));
		PRINT_DEBUG("client status=Unsync");
		return;
	}

	TIME_STRNCPY(time_args->status, "Disabled", sizeof(time_args->status));
	PRINT_DEBUG("status=Disabled");
	return;
}

void print_log(int severity, const char *format, ...)
{
	va_list arglist;

	va_start(arglist, format);
	vsyslog(severity, format, arglist);
	va_end(arglist);
}
