/*
 * Copyright (C) 2021-2025 iopsys Software Solutions AB
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1
 * as published by the Free Software Foundation
 *
 *	  Author Amin Ben Romdhane <amin.benromdhane@iopsys.eu>
 *
 */

#include "libbbfdm-api/dmcommon.h"

/*************************************************************
 * ENTRY METHODS
 *************************************************************/
static int browseTimeZones(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *zone_obj = NULL, *arrobj = NULL;
	char *inst = NULL, *file_path = NULL;
	struct dm_data curr_data = {0};
	int idx = 0, i = 0;

	dmuci_get_option_value_string("time", "global", "supported_zones_file", &file_path);
	if (DM_STRLEN(file_path) == 0)
		return -1;

	if (!bbfdm_file_exists(file_path))
		return -1;

	json_object *json_root = json_object_from_file(file_path);
	if (!json_root)
		return -1;

	dmjson_foreach_obj_in_array(json_root, arrobj, zone_obj, idx, 1, "supported_zones") {

		curr_data.json_object = zone_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++i);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}

	json_object_put(json_root);
	return 0;
}

static char *get_time_zone_from_zone_name(const char *zone_name)
{
	json_object *zone_obj = NULL, *arrobj = NULL;
	char *time_zone = NULL, *file_path = NULL;
	int idx = 0;

	if (!zone_name)
		return NULL;

	dmuci_get_option_value_string("time", "global", "supported_zones_file", &file_path);
	if (DM_STRLEN(file_path) == 0)
		return NULL;

	json_object *json_root = json_object_from_file(file_path);
	if (!json_root)
		return NULL;

	dmjson_foreach_obj_in_array(json_root, arrobj, zone_obj, idx, 1, "supported_zones") {
		char *token = NULL, *saveptr = NULL;

		char *zone_name_list = dmstrdup(dmjson_get_value(zone_obj, 1, "zone_name"));

		for (token = strtok_r(zone_name_list, ",", &saveptr); token != NULL; token = strtok_r(NULL, ",", &saveptr)) {
			if (DM_STRCMP(token, zone_name) == 0) {
				time_zone = dmjson_get_value(zone_obj, 1, "time_zone");
				json_object_put(json_root);
				return time_zone;
			}
		}
	}

	json_object_put(json_root);
	return NULL;
}

static int get_local_zone_name(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_option_value_string("system", "@system[0]", "zonename", value);
	return 0;
}

static int set_local_zone_name(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *time_zone = NULL;

	switch (action) {
		case VALUECHECK:
			if (!value || bbfdm_validate_string(ctx, value, -1, 256, NULL, NULL))
				return FAULT_9007;

			if (DM_STRLEN(value)) {
				time_zone = get_time_zone_from_zone_name(value);
				if (time_zone == NULL) {
					bbfdm_set_fault_message(ctx, "Zone Name '%s' is not supported by the device.", value);
					return FAULT_9007;
				}
			}

			break;
		case VALUESET:
			dmuci_set_value("system", "@system[0]", "zonename", value);

			if (DM_STRLEN(value)) {
				dmuci_set_value("system", "@system[0]", "timezone", get_time_zone_from_zone_name(value));
			}
			break;
	}
	return 0;
}

static int get_time_zone(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "time_zone");
	return 0;
}

static int get_zone_name(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "zone_name");
	return 0;
}

/**********************************************************************************************************************************
*                                            OBJ & PARAM DEFINITION
***********************************************************************************************************************************/
DMLEAF tIOPSYS_ZonesParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"TimeZone", &DMREAD, DMT_STRING, get_time_zone, NULL, BBFDM_BOTH},
{"ZoneName", &DMREAD, DMT_STRING, get_zone_name, NULL, BBFDM_BOTH},
{0}
};

/*** Device. ***/
DMOBJ tIOPSYS_TimeObjs[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type*/
{BBF_VENDOR_PREFIX"SupportedZones", &DMREAD, NULL, NULL, NULL, browseTimeZones, NULL, NULL, NULL, tIOPSYS_ZonesParams, NULL, BBFDM_BOTH},
{0}
};

DMLEAF tIOPSYS_TimeParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{BBF_VENDOR_PREFIX"LocalTimeZoneName", &DMWRITE, DMT_STRING, get_local_zone_name, set_local_zone_name, BBFDM_BOTH},
{0}
};

/* ********** DynamicObj ********** */
DM_MAP_OBJ tDynamicObj[] = {
/* parentobj, nextobject, parameter */
{"Device.Time.", tIOPSYS_TimeObjs, tIOPSYS_TimeParams},
{0}
};
