/*
 * Copyright (C) 2023 iopsys Software Solutions AB
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1
 * as published by the Free Software Foundation
 *
 *	  Author: Suvendhu Hansa <suvendhu.hansa@iopsys.eu>
 *
 */

#include <libbbfdm-api/dmcommon.h>

#include "helper.h"

#define POSIX_TZ "^[A-Z]{3,}[-+]?[0-9]{1,2}([A-Z]{3,}([-+]?[0-9]{1,2})?)?(,M[0-9]{1,2}\\.[0-9]\\.[0-9](/?[0-9]{1,2})?,M[0-9]{1,2}\\.[0-9]\\.[0-9](/?[0-9]{1,2})?)?$"

/*************************************************************
* ADD & DEL METHODS
**************************************************************/
static int addTimeClient(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct uci_section *s = NULL, *dmmap_s = NULL;
	char s_name[16] = {0};

	snprintf(s_name, sizeof(s_name), "client_%s", *instance);

	dmuci_add_section("time", "client", &s);
	dmuci_rename_section_by_section(s, s_name);

	dmuci_set_value_by_section(s, "enable", "0");
	dmuci_set_value_by_section(s, "dhcp_discovery", "1");
	dmuci_set_value_by_section(s, "iburst", "1");
	dmuci_set_value_by_section(s, "version", "4");
	dmuci_set_value_by_section(s, "peer", "0");
	dmuci_set_value_by_section(s, "minpoll", "6");
	dmuci_set_value_by_section(s, "maxpoll", "10");
	dmuci_set_value_by_section(s, "mode", "Unicast");

	dmuci_add_section_bbfdm("dmmap_time", "client", &dmmap_s);
	dmuci_set_value_by_section(dmmap_s, "section_name", s_name);
	dmuci_set_value_by_section(dmmap_s, "client_instance", *instance);
	return 0;
}

static int delTimeClient(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	dmuci_delete_by_section(((struct dm_data *)data)->config_section, NULL, NULL);
	dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);

	return 0;
}

static int addTimeServer(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct uci_section *s = NULL, *dmmap_s = NULL;
	char *sec = NULL;

	// Check if server section already exists
	dmuci_get_section_type("time", "server", &sec);
	if (DM_STRCMP(sec, "server") == 0) {
		bbfdm_set_fault_message(ctx, "The current implementation only supports one 'Server', so adding another 'Server' is not allowed.");
		return FAULT_9003;
	}

	dmuci_add_section("time", "server", &s);
	dmuci_rename_section_by_section(s, "server");

	dmuci_set_value_by_section(s, "enable", "0");
	dmuci_set_value_by_section(s, "mode", "Unicast");
	dmuci_set_value_by_section(s, "ttl", "255");

	dmuci_add_section_bbfdm("dmmap_time", "server", &dmmap_s);
	dmuci_set_value_by_section(dmmap_s, "section_name", "server");
	dmuci_set_value_by_section(dmmap_s, "server_instance", *instance);

	return 0;
}

static int delTimeServer(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	dmuci_delete_by_section(((struct dm_data *)data)->config_section, NULL, NULL);
	dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);

	return 0;
}

/*************************************************************
 * ENTRY METHODS
 *************************************************************/
static int browseTimeClient(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct time_args curr_time_args = {0};
	struct dm_data *p = NULL;
	char *inst = NULL;
	LIST_HEAD(dup_list);

	synchronize_specific_config_sections_with_dmmap("time", "client", "dmmap_time", &dup_list);
	list_for_each_entry(p, &dup_list, list) {

		fill_time_args("client", section_name(p->config_section), &curr_time_args);

		p->additional_data = (void *)&curr_time_args;

		inst = handle_instance(dmctx, parent_node, p->dmmap_section, "client_instance", "client_alias");

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)p, inst) == DM_STOP)
			break;
	}
	free_dmmap_config_dup_list(&dup_list);
	return 0;
}

static int browseTimeServer(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct time_args curr_time_args = {0};
	struct dm_data *p = NULL;
	char *inst = NULL;
	LIST_HEAD(dup_list);

	synchronize_specific_config_sections_with_dmmap("time", "server", "dmmap_time", &dup_list);
	list_for_each_entry(p, &dup_list, list) {

		fill_time_args("server", section_name(p->config_section), &curr_time_args);

		p->additional_data = (void *)&curr_time_args;

		inst = handle_instance(dmctx, parent_node, p->dmmap_section, "server_instance", "server_alias");

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)p, inst) == DM_STOP)
			break;
	}
	free_dmmap_config_dup_list(&dup_list);
	return 0;
}

/*************************************************************
* GET & SET PARAM
**************************************************************/
static int get_time_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_option_value_fallback_def("time", "global", "enable", "1");
	return 0;
}

static int set_time_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_boolean(ctx, value))
			ret = FAULT_9007;
		break;
	case VALUESET:
		string_to_bool(value, &b);
		dmuci_set_value("time", "global", "enable", b ? "1" : "0");
		break;
	}

	return ret;
}

static int get_time_Status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct time_args curr_time_args = {0};

	fill_time_args("global", NULL, &curr_time_args);

	*value = dmstrdup(curr_time_args.status);
	return 0;
}

static int get_time_CurrentLocalTime(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return dm_time_format(time(NULL), value);
}

static bool is_time_zone_supported(const char *time_zone)
{
	json_object *zone_obj = NULL, *arrobj = NULL;
	char *file_path = NULL;
	bool is_supported = false;
	int idx = 0;

	if (!time_zone)
		return false;

	dmuci_get_option_value_string("time", "global", "supported_zones_file", &file_path);
	if (DM_STRLEN(file_path) == 0)
		return true;

	json_object *json_root = json_object_from_file(file_path);
	if (!json_root)
		return false;

	dmjson_foreach_obj_in_array(json_root, arrobj, zone_obj, idx, 1, "supported_zones") {
		char *curr_time_zone = dmjson_get_value(zone_obj, 1, "time_zone");
		if (DM_STRCMP(curr_time_zone, time_zone) == 0) {
			is_supported = true;
			break;
		}
	}

	json_object_put(json_root);
	return is_supported;
}

static bool is_known_abbreviation(const char *abbr)
{
	const char *validTZ[] = {
		"EST", "EDT", "PST", "PDT", "CST", "CDT", "MST", "MDT",
		"UTC", "GMT", "WET", "WEST", "EET", "EEST", "MET", "CEST", "EAT",
		NULL
	};

	for (int i = 0; validTZ[i] != NULL; i++) {
		if (DM_STRCMP(abbr, validTZ[i]) == 0)
			return true;
	}

	return false;
}

static bool is_valid_posix_tz(const char *tz)
{
	regmatch_t pmatch[5];

	if (!match(tz, POSIX_TZ, 5, pmatch)) {
		BBF_ERR("TZ format does not match POSIX specification");
		return false;
	}

	// Parse first abbreviation (e.g., "EST")
	int i = 0;
	while (tz[i] >= 'A' && tz[i] <= 'Z')
		i++;

	if (i < 3 || i > 5) {
		BBF_ERR("First abbreviation must be 3 to 5 uppercase letters");
		return false;
	}

	char abbr1[6] = {0};
	DM_STRNCPY(abbr1, tz, i + 1);

	if (!is_known_abbreviation(abbr1)) {
		BBF_ERR("Unrecognized primary time zone abbreviation: %s", abbr1);
		return false;
	}

	// Skip numeric offset (e.g., "+5", "-10")
	int pos = i;
	while (tz[pos] == '+' || tz[pos] == '-' || (tz[pos] >= '0' && tz[pos] <= '9'))
		pos++;

	// Parse second abbreviation if present (e.g., "EDT")
	if (tz[pos] >= 'A' && tz[pos] <= 'Z') {
		int j = pos;

		while (tz[j] >= 'A' && tz[j] <= 'Z')
			j++;

		int len = j - pos;
		if (len < 3 || len > 5) {
			BBF_ERR("Second abbreviation must be 3 to 5 uppercase letters");
			return false;
		}

		char abbr2[6] = {0};
		DM_STRNCPY(abbr2, tz + pos, len + 1);

		if (!is_known_abbreviation(abbr2)) {
			BBF_ERR("Unrecognized DST abbreviation: %s", abbr2);
			return false;
		}
	}

	return true;
}

/*#Device.Time.LocalTimeZone!UCI:system/system,@system[0]/timezone*/
static int get_time_LocalTimeZone(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_option_value_string("system", "@system[0]", "timezone", value);
	return 0;
}

static int set_time_LocalTimeZone(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (!value || bbfdm_validate_string(ctx, value, -1, 256, NULL, NULL))
				return FAULT_9007;

			if (DM_STRLEN(value)) {
				if (is_time_zone_supported(value))
					return 0;

				if (is_valid_posix_tz(value))
					return 0;

				bbfdm_set_fault_message(ctx,
					"Time Zone '%s' is not supported by the device and is not a valid POSIX format",
					value);
				return FAULT_9007;
			}
			break;
		case VALUESET:
			dmuci_set_value("system", "@system[0]", "timezone", value);
			dmuci_set_value("system", "@system[0]", "zonename", "");
			setenv("TZ", value, 1);
			tzset();
			break;
	}
	return 0;
}

static int get_time_ClientNumber(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	unsigned int cnt = get_number_of_entries(ctx, data, instance, browseTimeClient);
	dmasprintf(value, "%u", cnt);
	return 0;
}

static int get_client_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enable", "1");
	return 0;
}

static int set_client_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_boolean(ctx, value))
			ret = FAULT_9007;
		break;
	case VALUESET:
		string_to_bool(value, &b);
		dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "enable", b ? "1" : "0");
		if (b) {
			/* Only one client instance can be enabled at a time */
			struct uci_section *s = NULL;
			uci_foreach_sections("time", "client", s) {
				char *config_section = section_name(((struct dm_data *)data)->config_section);
				char *uci_section = section_name(s);
				if (DM_STRCMP(uci_section, config_section) != 0)
					dmuci_set_value_by_section(s, "enable", "0");
			}
		}
		break;
	}

	return ret;
}

static int get_client_Status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct time_args *curr_time_args = (struct time_args *)((struct dm_data *)data)->additional_data;

	*value = dmstrdup(curr_time_args->status);
	return 0;
}

static int get_client_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "client_alias", instance, value);
}

static int set_client_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "client_alias", instance, value);
}

static int get_client_Mode(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "mode", value);
	return 0;
}

static int set_client_Mode(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	int ret = 0;
	char *allowed_params[] = {"Unicast", "Broadcast", "Multicast", "Manycast", NULL};

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_string(ctx, value, -1, -1, allowed_params, NULL))
			ret = FAULT_9007;
		break;
	case VALUESET:
		dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "mode", value);
		break;
	}

	return ret;
}

static int get_client_Port(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("123");
	return 0;
}

static int set_client_Port(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"1","65535"}}, 1))
			ret = FAULT_9007;

		if (DM_STRCMP(value, "123") != 0) {
			bbfdm_set_fault_message(ctx, "The current implementation supports only one port which is '123'. So the only allowed value is '123'.");
			ret = FAULT_9007;
		}

		break;
	case VALUESET:
		break;
	}

	return ret;
}

static int get_client_Version(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "version", value);
	return 0;
}

static int set_client_Version(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"1", "4"}}, 1))
			ret = FAULT_9007;
		break;
	case VALUESET:
		dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "version", value);
		break;
	}

	return ret;
}

static int get_client_Servers(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_list *val = NULL;
	dmuci_get_value_by_section_list(((struct dm_data *)data)->config_section, "server", &val);
	*value = dmuci_list_to_string(val, ",");
	return 0;
}

static int set_client_Servers(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	size_t length, i;
	int ret = 0;
	char **arr;

	switch (action) {
	case VALUECHECK:
		if (!value || bbfdm_validate_string(ctx, value, -1, -1, NULL, NULL))
			ret = FAULT_9007;
		break;
	case VALUESET:
		dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "server", "");

		arr = strsplit(value, ",", &length);
		if (length == 0) {
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "dhcp_discovery", "1");
			break;
		}

		dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "dhcp_discovery", "0");
		for (i = 0; i < length; i++) {
			dmuci_add_list_value_by_section(((struct dm_data *)data)->config_section, "server", arr[i]);
		}
		break;
	}

	return ret;
}

static int get_client_Peer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "peer", value);
	return 0;
}

static int set_client_Peer(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_boolean(ctx, value))
			ret = FAULT_9007;
		break;
	case VALUESET:
		string_to_bool(value, &b);
		dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "peer", b ? "1" : "0");
		break;
	}

	return ret;
}

static int get_client_MinPoll(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "minpoll", "6");
	return 0;
}

static int set_client_MinPoll(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL, NULL}}, 1))
			ret = FAULT_9007;
		break;
	case VALUESET:
		dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "minpoll", value);
		break;
	}

	return ret;
}

static int get_client_MaxPoll(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "maxpoll", "10");
	return 0;
}

static int set_client_MaxPoll(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL, NULL}}, 1))
			ret = FAULT_9007;
		break;
	case VALUESET:
		dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "maxpoll", value);
		break;
	}

	return ret;
}

static int get_client_IBurst(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "iburst", value);
	return 0;
}

static int set_client_IBurst(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_boolean(ctx, value))
			ret = FAULT_9007;
		break;
	case VALUESET:
		string_to_bool(value, &b);
		dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "iburst", b ? "1" : "0");
		break;
	}

	return ret;
}

static int get_client_Interface(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *linker = NULL;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "interface", &linker);

	return _bbfdm_get_references(ctx, "Device.IP.Interface.", "Name", linker, value);

}

static int set_client_Interface(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_objects[] = {"Device.IP.Interface.", NULL};
	struct dm_reference reference = {0};

	bbfdm_get_reference_linker(ctx, value, &reference);

	switch (action)	{
		case VALUECHECK:
			if (dm_validate_allowed_objects(ctx, &reference, allowed_objects))
				return FAULT_9007;

			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "interface", reference.value);
			break;
	}
	return 0;	
}

static int get_client_PacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct time_args *curr_time_args = (struct time_args *)((struct dm_data *)data)->additional_data;

	dmasprintf(value, "%u", curr_time_args->pack_sent);
	return 0;
}

static int get_client_PacketsSentFailed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct time_args *curr_time_args = (struct time_args *)((struct dm_data *)data)->additional_data;

	dmasprintf(value, "%u", curr_time_args->pack_sent_fail);
	return 0;
}

static int get_client_PacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct time_args *curr_time_args = (struct time_args *)((struct dm_data *)data)->additional_data;

	dmasprintf(value, "%u", curr_time_args->pack_recv);
	return 0;
}

static int get_client_PacketsDropped(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct time_args *curr_time_args = (struct time_args *)((struct dm_data *)data)->additional_data;

	dmasprintf(value, "%u", curr_time_args->pack_drop);
	return 0;
}

static int get_time_ServerNumber(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	unsigned int cnt = get_number_of_entries(ctx, data, instance, browseTimeServer);
	dmasprintf(value, "%u", cnt);
	return 0;
}

static int get_server_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_option_value_fallback_def("time", "server", "enable", "1");
	return 0;
}

static int set_server_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_boolean(ctx, value))
			ret = FAULT_9007;
		break;
	case VALUESET:
		string_to_bool(value, &b);
		dmuci_set_value("time", "server", "enable", b ? "1" : "0");
		break;
	}

	return ret;
}

static int get_server_Status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct time_args *curr_time_args = (struct time_args *)((struct dm_data *)data)->additional_data;

	*value = dmstrdup(curr_time_args->status);
	return 0;
}

static int get_server_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "server_alias", instance, value);
}

static int set_server_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "server_alias", instance, value);
}

static int get_server_Mode(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_option_value_string("time", "server", "mode", value);
	return 0;
}

static int set_server_Mode(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	int ret = 0;
	char *allowed_params[] = {"Unicast", "Broadcast", "Multicast", "Manycast", NULL};

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_string(ctx, value, -1, -1, allowed_params, NULL))
			ret = FAULT_9007;
		break;
	case VALUESET:
		dmuci_set_value("time", "server", "mode", value);
		break;
	}

	return ret;
}

static int get_server_Port(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("123");
	return 0;
}

static int set_server_Port(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"1","65535"}}, 1))
			ret = FAULT_9007;

		if (DM_STRCMP(value, "123") != 0) {
			bbfdm_set_fault_message(ctx, "The current implementation supports only one port which is '123'. So the only allowed value is '123'.");
			ret = FAULT_9007;
		}

		break;
	case VALUESET:
		break;
	}

	return ret;
}

static int get_server_Interface(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *linker = NULL;

	dmuci_get_option_value_string("time", "server", "interface", &linker);

	return _bbfdm_get_references(ctx, "Device.IP.Interface.", "Name", linker, value);
}

static int set_server_Interface(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_objects[] = {"Device.IP.Interface.", NULL};
	struct dm_reference reference = {0};

	bbfdm_get_reference_linker(ctx, value, &reference);

	switch (action)	{
		case VALUECHECK:
			if (dm_validate_allowed_objects(ctx, &reference, allowed_objects))
				return FAULT_9007;

			break;
		case VALUESET:
			dmuci_set_value("time", "server", "interface", reference.value);
			break;
	}
	return 0;	
}

static int get_server_TTL(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_option_value_fallback_def("time", "server", "ttl", "255");
	return 0;
}

static int set_server_TTL(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"1", "255"}}, 1))
			ret = FAULT_9007;
		break;
	case VALUESET:
		dmuci_set_value("time", "server", "ttl", value);
		break;
	}

	return ret;
}

static int get_server_PacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct time_args *curr_time_args = (struct time_args *)((struct dm_data *)data)->additional_data;

	dmasprintf(value, "%u", curr_time_args->pack_sent);
	return 0;
}

static int get_server_PacketsSentFailed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct time_args *curr_time_args = (struct time_args *)((struct dm_data *)data)->additional_data;

	dmasprintf(value, "%u", curr_time_args->pack_sent_fail);
	return 0;
}

static int get_server_PacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct time_args *curr_time_args = (struct time_args *)((struct dm_data *)data)->additional_data;

	dmasprintf(value, "%u", curr_time_args->pack_recv);
	return 0;
}

static int get_server_PacketsDropped(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct time_args *curr_time_args = (struct time_args *)((struct dm_data *)data)->additional_data;

	dmasprintf(value, "%u", curr_time_args->pack_drop);
	return 0;
}

/**********************************************************************************************************************************
*                                            OBJ & LEAF DEFINITION
***********************************************************************************************************************************/

/* *** Device.Time.Client.{i}.Stats. *** */
DMLEAF tTimeClientStatsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"PacketsSent", &DMREAD, DMT_UNINT, get_client_PacketsSent, NULL, BBFDM_BOTH},
{"PacketsSentFailed", &DMREAD, DMT_UNINT, get_client_PacketsSentFailed, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNINT, get_client_PacketsReceived, NULL, BBFDM_BOTH},
{"PacketsDropped", &DMREAD, DMT_UNINT, get_client_PacketsDropped, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.Time.Client.{i}. *** */
DMOBJ tTimeClientObjs[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys*/
{"Stats", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tTimeClientStatsParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tTimeClientParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"Enable", &DMWRITE, DMT_BOOL, get_client_Enable, set_client_Enable, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_client_Status, NULL, BBFDM_BOTH},
{"Alias", &DMWRITE, DMT_STRING, get_client_Alias, set_client_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Mode", &DMWRITE, DMT_STRING, get_client_Mode, set_client_Mode, BBFDM_BOTH},
{"Port", &DMWRITE, DMT_UNINT, get_client_Port, set_client_Port, BBFDM_BOTH},
{"Version", &DMWRITE, DMT_UNINT, get_client_Version, set_client_Version, BBFDM_BOTH},
{"Servers", &DMWRITE, DMT_STRING, get_client_Servers, set_client_Servers, BBFDM_BOTH},
{"Peer", &DMWRITE, DMT_BOOL, get_client_Peer, set_client_Peer, BBFDM_BOTH},
{"MinPoll", &DMWRITE, DMT_UNINT, get_client_MinPoll, set_client_MinPoll, BBFDM_BOTH},
{"MaxPoll", &DMWRITE, DMT_UNINT, get_client_MaxPoll, set_client_MaxPoll, BBFDM_BOTH},
{"IBurst", &DMWRITE, DMT_BOOL, get_client_IBurst, set_client_IBurst, BBFDM_BOTH},
{"Interface", &DMWRITE, DMT_STRING, get_client_Interface, set_client_Interface, BBFDM_BOTH, DM_FLAG_REFERENCE},
{0}
};

/* *** Device.Time.Server.{i}.Stats. *** */
DMLEAF tTimeServerStatsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"PacketsSent", &DMREAD, DMT_UNINT, get_server_PacketsSent, NULL, BBFDM_BOTH},
{"PacketsSentFailed", &DMREAD, DMT_UNINT, get_server_PacketsSentFailed, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNINT, get_server_PacketsReceived, NULL, BBFDM_BOTH},
{"PacketsDropped", &DMREAD, DMT_UNINT, get_server_PacketsDropped, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.Time.Server.{i}. *** */
DMOBJ tTimeServerObjs[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys*/
{"Stats", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tTimeServerStatsParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tTimeServerParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"Enable", &DMWRITE, DMT_BOOL, get_server_Enable, set_server_Enable, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_server_Status, NULL, BBFDM_BOTH},
{"Alias", &DMWRITE, DMT_STRING, get_server_Alias, set_server_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Mode", &DMWRITE, DMT_STRING, get_server_Mode, set_server_Mode, BBFDM_BOTH},
{"Port", &DMWRITE, DMT_UNINT, get_server_Port, set_server_Port, BBFDM_BOTH},
{"Interface", &DMWRITE, DMT_STRING, get_server_Interface, set_server_Interface, BBFDM_BOTH, DM_FLAG_REFERENCE},
{"TTL", &DMWRITE, DMT_UNINT, get_server_TTL, set_server_TTL, BBFDM_BOTH},
{0}
};

/* *** Device.Time. *** */
DMOBJ tTimeObjs[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys*/
{"Client", &DMWRITE, addTimeClient, delTimeClient, NULL, browseTimeClient, NULL, NULL, tTimeClientObjs, tTimeClientParams, NULL, BBFDM_BOTH, NULL},
{"Server", &DMWRITE, addTimeServer, delTimeServer, NULL, browseTimeServer, NULL, NULL, tTimeServerObjs, tTimeServerParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tTimeParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_time_Enable, set_time_Enable, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_time_Status, NULL, BBFDM_BOTH},
{"CurrentLocalTime", &DMREAD, DMT_TIME, get_time_CurrentLocalTime, NULL, BBFDM_BOTH},
{"LocalTimeZone", &DMWRITE, DMT_STRING, get_time_LocalTimeZone, set_time_LocalTimeZone, BBFDM_BOTH},
{"ClientNumberOfEntries", &DMREAD, DMT_UNINT, get_time_ClientNumber, NULL, BBFDM_BOTH},
{"ServerNumberOfEntries", &DMREAD, DMT_UNINT, get_time_ServerNumber, NULL, BBFDM_BOTH},
{0}
};

/*** Device. ***/
DMOBJ tDeviceObjs[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys*/
{"Time", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tTimeObjs, tTimeParams, NULL, BBFDM_BOTH, NULL},
{0}
};

/* ********** DynamicObj ********** */
DM_MAP_OBJ tDynamicObj[] = {
/* parentobj, nextobject, parameter */
{"Device.", tDeviceObjs, NULL},
{0}
};
