/*
 * Copyright (C) 2023-2024 iopsys Software Solutions AB
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1
 * as published by the Free Software Foundation
 *
 *	  Author: Suvendhu Hansa <suvendhu.hansa@iopsys.eu>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libubus.h>
#include <libubox/uloop.h>
#include <libbbfdm-ubus/bbfdm-ubus.h>

#include "helper.h"

extern DM_MAP_OBJ tDynamicObj[];

static struct bbfdm_context bbfdm_ctx = {0};

static void timemngr_start_sync_timer(int seconds);

static void timemngr_schedule_sync(struct uloop_timeout *timeout __attribute__((unused)))
{
	int time = 10;

	PRINT_DEBUG("Cheking current status of time synchronization");
	if (false == ntpd_time_sync()) {
		PRINT_DEBUG("Current time is synced");
		uint32_t id;
		int rc = -1;
		struct blob_buf b;

		memset(&b, 0, sizeof(struct blob_buf));
		blob_buf_init(&b, 0);

		void *a = blobmsg_open_array(&b, "env");
		blobmsg_add_string(&b, NULL, "ACTION=stratum");
		blobmsg_close_array(&b, a);

		PRINT_DEBUG("Executing ntp hotplug");
		if (!ubus_lookup_id(&bbfdm_ctx.ubus_ctx, "hotplug.ntp", &id)) {
			rc = ubus_invoke(&bbfdm_ctx.ubus_ctx, id, "call", b.head, NULL, NULL, 5000);
		}

		blob_buf_free(&b);

		if (rc == 0) {
			time = 660;
		}
	}

	PRINT_DEBUG("Scheduling synchronize timer after %d secs", time);
	timemngr_start_sync_timer(time);
}

static struct uloop_timeout timemngr_sync_timer = { .cb = timemngr_schedule_sync };

static void timemngr_start_sync_timer(int seconds)
{
	PRINT_DEBUG("Setting sync timer timeout for: %d secs", seconds);
	uloop_timeout_set(&timemngr_sync_timer, seconds * 1000);
}

static void timemngr_usage(char *n)
{
	fprintf(stdout,
		"\nUsage: %s: <options>\n"
		"Options:\n"
		"  -l <loglevel> log verbosity value as per standard syslog\n"
		"  -d <schema dm> Display the schema data model supported by micro-service\n"
		"  -h help\n\n"
		, n);
}

int main(int argc, char **argv)
{
	int log_level = 3; // Default is LOG_ERR
	int c = 0, dm_type = 0;

	while ((c = getopt(argc, argv, "dl:h")) != -1) {
		switch (c) {
		case 'h':
			timemngr_usage(argv[0]);
			return EXIT_SUCCESS;
		case 'd':
			dm_type++;
			break;
		case 'l':
			if (optarg) {
				log_level = (int)strtod(optarg, NULL);
				if (log_level < 0 || log_level > 7)
					log_level = 3;
			}

			break;
		default:
			timemngr_usage(argv[0]);
			return EXIT_FAILURE;
		}
	}

	memset(&bbfdm_ctx, 0, sizeof(struct bbfdm_context));

	bbfdm_ubus_set_service_name(&bbfdm_ctx, "timemngr");
	bbfdm_ubus_set_log_level(log_level);
	bbfdm_ubus_load_data_model(tDynamicObj);

	if (dm_type > 0) {
		int res = bbfdm_print_data_model_schema(&bbfdm_ctx, dm_type);
		exit(res);
	}

	openlog("timemngr", LOG_CONS | LOG_PID | LOG_NDELAY, LOG_LOCAL1);

	PRINT_DEBUG("Starting timemngr daemon");

	if (bbfdm_ubus_regiter_init(&bbfdm_ctx))
		goto out;

	timemngr_start_sync_timer(10);
	uloop_run();

out:
	PRINT_DEBUG("timemngr going to exit");
	bbfdm_ubus_regiter_free(&bbfdm_ctx);
	closelog();
	PRINT_DEBUG("timemngr exit successful");

	return 0;
}
