#!/bin/sh

# Helper utility to configure iopsys specific template for LXC container images.
#
# Copyright © 2021 IOPSYS Software Solutions AB
# Author: Vivek Dutta <vivek.dutta@iopsys.eu>
#

OPKG_LIB_PATH="/usr/lib/opkg"
CUR_DEP=""

log() {
    echo "${@}"|logger -t swmodd.template -p debug
}

check_installed()
{
    local root="${1}"
    local pkg="${2}"

    if opkg --offline-root "${root}" list "${pkg}" |grep -q "${pkg} "; then
        # log "Package [$pkg] already installed [$root]"
        return 0;
    fi

    return 1;
}

check_dependency()
{
    local root="${1}"
    local pkg="${2}"
    local dep_list=""
    local list=""

    if [ -z "${pkg}" ]; then
        return 0;
    fi

    # log "## Checking dependency for [${pkg}] root[$root]"
    list=$(opkg depends "${pkg}" |awk 'NR>1 {printf $1" "}')

    for f in ${list}; do
        local t=""

        # get which package provides this
        t=$(opkg whatprovides "${f}"| awk 'NR>1 {printf $1}')
        if ! check_installed "${root}" "${t}"; then
            # log "## Adding into dependency[$dep_list] [$t] [$f] ##"
            if ! echo "${dep_list}" |grep -q "${t}"; then
                dep_list="${dep_list} ${t}"
            fi
        fi
    done

    # log "PKG[${pkg}] has dependency[${dep_list}] list [$list]"
    CUR_DEP="${dep_list}"
}

copy_single_opkg()
{
    local root="${1}"
    local pkg="${2}"

    if [ -z "${pkg}" ]; then
        return 0;
    fi

    if ! check_installed / "${pkg}"; then
        log "Package [${pkg}] not available locally";
        return 1;
    fi

    if check_installed "${root}" "${pkg}" >/dev/null ; then
        # log "Package [${pkg}] already installed in target";
        return 0;
    fi

    for f in "${OPKG_LIB_PATH}"/info/"${pkg}".*; do
        cp "$f" "${root}/${OPKG_LIB_PATH}/info/"
    done
    opkg info "${pkg}" >> "${root}/${OPKG_LIB_PATH}/status"

    # copy the artifacts associated with the package
    while read -r line
    do
        if [ -f "${line}" ]; then
            local p

            p=$(dirname "${line}")
            mkdir -p "${root}"/"${p}"
            cp "${line}" "${root}"/"${p}"/
        fi
    done < "${OPKG_LIB_PATH}/info/${pkg}.list"

    return 0;
}

resolve_dependency()
{
    local deps="${1}"

    for d in ${deps}; do
        # log "Installing dependency [${d}] from list[${deps}]"
        install_package "${root}" "${d}"
    done
}

install_package()
{
    local root="${1}"
    local pkg="${2}"
    local dep=""
    local ret

    if ! check_installed "/" "${pkg}"; then
        echo "Package [${pkg}] not available locally";
        return 1;
    fi

    if check_installed "${root}" "${pkg}" >/dev/null; then
        # echo "Package [${pkg}] already installed in target";
        return 0;
    fi

    CUR_DEP=""
    check_dependency "${root}" "${pkg}"
    dep="${CUR_DEP}"
    # log "Processing [${pkg}] with [${dep}] dependencies"

    if [ -n "${dep}" ]; then
        resolve_dependency "${dep}"
        install_package "${root}" "${pkg}"
        ret=$?
    else
        log "Installing package ${pkg}"
        copy_single_opkg "${root}" "${pkg}"
        ret=$?
    fi
    return $ret;
}

if [ ! -d "${1}" ]; then
    echo "Root [${1}] does not exist"
    exit 0;
fi

if [ -z "${2}" ]; then
    echo "No package specified"
    exit 0;
fi

echo "# Creating container might take ~1 min #"
echo "Installing pkg [${2}] with dependencies"
if install_package "${1}" "${2}"; then
    echo " - Offline install of ${2} successful"
else
    echo " - Failed to install ${2} in offline mode"
fi
