/*
 * tools.h: SWMOD deamon
 *
 * Copyright (C) 2020-2023 IOPSYS Software Solutions AB. All rights reserved.
 *
 * Author: Amin Ben Ramdhane <amin.benramdhane@pivasoftware.com>
 *
 * See LICENSE file for license related information.
 */

#ifndef __TOOLS_H
#define __TOOLS_H

#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <libubox/list.h>
#include <syslog.h>

#include "swmod.h"

#define OPKG_INFO_PATH "/usr/lib/opkg/info/"
#define STD_UCI_PATH "/etc/config/"
#define SWMOD_UCI_FILE "swmodd"
#define SWMOD_OCI_DU_UCI "ocicontainer"
#define SWMOD_LXC_DU_UCI "lxccontainer"
#define SWMOD_INIT_PATH "/etc/init.d/"

#define SYSFS_FOREACH_FILE(path, dir, ent) \
	if ((dir = opendir(path)) == NULL) return 0; \
	while ((ent = readdir (dir)) != NULL) \

#ifndef FREE
#define FREE(x) do { if(x) {free(x); x = NULL;} } while (0)
#endif

enum swmod_action_enum {
	SWMOD_DEPLOYMENT_UNIT,
	SWMOD_EXECUTION_UNIT
};

enum {
	SUCCESS,
	FAILURE,
	LXC_NOT_SUPPORTED,
	NOT_FOUND,
	ALREADY_RUNNING,
	NOT_RUNNING,
	INVALID_REQUEST,
	CRUN_NOT_SUPPORTED,
	__ERROR_MAX
};

enum {
	CONT_START,
	CONT_STOP,
	CONT_PAUSE,
	CONT_RESUME,
	__CONT_STATE_MAX
};

struct buffer_list
{
	struct list_head list;
	char line[MAX_LEN_128];
};

bool file_exists(const char *path);
unsigned long get_file_size_kb(const char *path);
bool create_dir(const char *path);
int run_cmd(const char *cmd, char *output, size_t out_len);
int run_cmd_no_output(const char *cmd);
void remove_new_line(char *buf);
void create_file(const char *path);
void swmod_strncpy(char *dst, const char *src, size_t n);
char *generate_uuid(void);
char *generate_duid(bool sysnchronise, int number);
int wait_for_pid(pid_t pid);

void configure_debug_level(unsigned char level);
void print_error(const char *format, ...);
void print_warning(const char *format, ...);
void print_info(const char *format, ...);
void print_debug(const char *format, ...);
int get_pid_details(int pid, char *cmdline, int cmd_len, int *vsize);
int update_eu_from_blob(ExecEnv_t *env, struct blob_attr *msg, ExecUnit *node);
int ubus_call_service_state(char *service, bool state);
bool dir_exist(const char *dir);
bool get_autoboot_from_config_file(const char *env, const char *lxc_uci_path);
void set_autoboot_to_config_file(const char *env, bool state, const char *lxc_uci_path);
int get_timeout_from_config_file(const char *env, const char *lxc_uci_path);
void swmod_add_eu_in_list(struct list_head *eu_list, ExecUnit *eu);
void swmod_delete_eu_list(struct list_head *eu_list);
void swmod_add_ee_in_list(struct list_head *ee_list, ExecEnv_t *eu);
void swmod_delete_ee_list(struct list_head *ee_list);
void sync_eeid_with_uci(struct list_head *ee_list, ExecEnv_t *environments, const char *lxc_uci_path);
void get_swmod_config_params(ConfigParams *cfg);
const char* ee_state_failure_reason(int err);
int get_requested_container_state(char *state);
int get_env_type(const char *type);
bool valid_uuid(char *uuid, const char *lxc_uci_path, const char *oci_uci_path);
void get_opkg_description(ExecEnv_t *env, const char *pkg_name, char *desc, size_t desc_size);
void get_opkg_service_name(ExecEnv_t *env, const char *pkg_name, char *service, size_t serv_size);
void buffer_add_line(struct list_head *head, char *entry);
void create_buffer_list(struct list_head *opkg_list, char *buffer);
void delete_buffer_list(struct list_head *opkg_list);
time_t convert_str_to_time(const char *time);
int swmod_ee_uci_init(ExecEnv_t *env, char *uci_file, int len);
int set_du_alias_to_config(const char *ee_name, const char *uuid,
			   const char *alias, const char *bundle, const char *uci_file);
int set_eu_alias_to_config(const char *ee_name, const char *eu_name,
			   const char *alias, const char *bundle, const char *uci_file);
int set_eu_autostart_to_config(const char *ee_name, const char *eu_name,
				bool enable, const char *bundle, const char *uci_file);

#define PRINT_DEBUG(fmt, args...) \
	print_debug("[%s:%d]"fmt, __func__, __LINE__, ##args)

#define PRINT_INFO(fmt, args...) \
	print_info("[%s:%d] " fmt, __func__, __LINE__, ##args)

#define PRINT_ERR(fmt, args...) \
	print_error("[%s:%d] " fmt, __func__, __LINE__, ##args)

#define PRINT_WARNING(fmt, args...) \
	print_warning("[%s:%d] " fmt, __func__, __LINE__, ##args)

#endif //__TOOLS_H

