/*
 * ubus.c -- Utility functions to make ubus calls
 *
 * Copyright (C) 2022 iopsys Software Solutions AB. All rights reserved.
 *
 * Author: Omar Kallel <omar.kallel@pivasoftware.com>
 *
 * See LICENSE file for license related information.
 *
 */

#include <strings.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <json-c/json.h>
#include <libubox/blobmsg_json.h>
#include <libubus.h>

#include "common.h"

static struct ubus_context *ubus_ctx = NULL;

int subus_init(void)
{
	ubus_ctx = ubus_connect(NULL);

	if (ubus_ctx == NULL)
		return -1;

	return 0;
}

int subus_fini(void)
{
	if (ubus_ctx != NULL) {
		ubus_free(ubus_ctx);
		ubus_ctx = NULL;
	}

	return 0;
}

static struct blob_attr* get_params(struct blob_attr *msg)
{
	struct blob_attr *params = NULL;
	struct blob_attr *cur;
	size_t len;

	blobmsg_for_each_attr(cur, msg, len) {
		if (blobmsg_type(cur) == BLOBMSG_TYPE_ARRAY) {
			params = cur;
			break;
		}
	}

	return params;
}

static void read_result(struct ubus_request *req, int type, struct blob_attr *msg)
{
	(void)type;

	size_t len;
	struct blob_attr *cur;
	struct blob_attr *params = NULL;

	enum {
		E_PARAM,
		E_VALUE,
		E_MAX
	};

	const struct blobmsg_policy p[E_MAX] = {
		{ "path", BLOBMSG_TYPE_STRING },
		{ "data", BLOBMSG_TYPE_STRING }
	};

	if (msg == NULL || req == NULL)
		return;

	struct ubus_result *output = (struct ubus_result *) req->priv;

	if (output == NULL)
		return;

	params = get_params(msg);
	if (params == NULL)
		return;

	blobmsg_for_each_attr(cur, params, len) {
		struct blob_attr *tb[E_MAX] = {NULL, NULL};
		char *path, *val;

		blobmsg_parse(p, E_MAX, tb, blobmsg_data(cur), blobmsg_len(cur));
		if (tb[E_PARAM] == NULL || tb[E_VALUE] == NULL)
			continue;

		path = blobmsg_get_string(tb[E_PARAM]);
		val = blobmsg_get_string(tb[E_VALUE]);

		if (strcmp(path, output->path) == 0)
			snprintf(output->value, sizeof(output->value), "%s", val);
	}
}

int stunc_ubus_call(const char *path, const char *method, struct blob_attr *msg, struct ubus_result *output)
{
	int ret = -1;
	uint32_t id;

	if (msg == NULL)
		return ret;

	subus_init();
	if (ubus_ctx != NULL) {
		if (ubus_lookup_id(ubus_ctx, path, &id) != 0)
			goto end;

		ret = ubus_invoke(ubus_ctx, id, method, msg, read_result, output, UBUS_TIMEOUT);

		if (ret > 0)
			ret = -1;
	}

end:
	subus_fini();
	return ret;
}
