/*
 * main.c - STUNC protocol handling
 *
 * Copyright (C) 2022, IOPSYS Software Solutions AB.
 *
 * Author: suvendhu.hansa@iopsys.eu
 *
 * See LICENSE file for license related information.
 *
 */
#define _GNU_SOURCE

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <libubox/uloop.h>
#include <string.h>
#include <ifaddrs.h>
#include <assert.h>

#include <openssl/ssl.h>

#include "stunc.h"
#include "common.h"

int keepalive_time;
int retry_timeout;
static struct binding_request req;

struct msg_var {
	unsigned int ip;
	unsigned short port;
	int code;
};

static int stunc_init(void);
static int stunc_finish(void);
static int stunc_start(void);
static int stunc_open_socket(int port);
static void stunc_listen_message(struct uloop_fd *fd, unsigned int event);
static void stunc_send_bind_req(struct uloop_timeout *timeout);
static void change_response_status(void);
static int stunc_create_req(char *msg, int msglen);
static void change_next_retry_interval(void);
static int add_username(unsigned char **pos, int rem_len);
static int add_connection_req(unsigned char **pos, int rem_len);
static int add_binding_change(unsigned char **pos, int rem_len);
static int add_msg_integrity(unsigned char **pos, int rem_len, struct stun_header *hdr);
static int add_padding(char *act_val, unsigned int act_len, unsigned int pad_len,
		       char **pad_val, char pad_byte);
static void add_attribute_to_msg(unsigned char **pos, const char *attr, unsigned short attr_len,
				 unsigned short attr_type, int rem_len);
static void process_server_message(void);
static void stunc_packet_process(struct uloop_timeout *timeout);
static void stunc_send_cr_event(struct uloop_timeout *timeout);
static ssize_t stunc_recv_from_server(int fd, char *buf, int len, struct sockaddr_in *con, int timeout);
static void process_error_response(char *resp_msg);
static void process_connection_request(char *resp_msg);
static void process_binding_response(char *resp_msg);
static void stun_read_from_packet(char *resp_msg, int att_type, struct msg_var *var);
static void change_nat_existance(unsigned int ip);
static void change_mapped_address(unsigned int ip_addr, unsigned short port);

static struct uloop_fd stunc_fd = {
	.cb = stunc_listen_message,
};

struct stunc_timer_t {
	struct uloop_timeout stunc_timer;
	char *timer_type;
};

static struct stunc_timer_t request_timer = {
	.stunc_timer = {.cb = stunc_send_bind_req },
	.timer_type = "request timer"
};

static struct stunc_timer_t keepalive_timer = {
	.stunc_timer = {.cb = stunc_send_bind_req },
	.timer_type = "keepalive timer"
};

struct uloop_timeout packet_process_timer = {.cb = stunc_packet_process };
struct uloop_timeout send_crevent_timer = {.cb = stunc_send_cr_event };

static void get_hexstring(char *hash, int len, char *hexstr, int buflen)
{
	int i, j;

	if (hash == NULL || hexstr == NULL)
		return;

	if (buflen <= len * 2)
		return;

	memset(hexstr, 0, buflen);

	for (i = 0, j = 0; i < len; i++) {
		sprintf(hexstr + j, "%02X", hash[i]);
		j = j + 2;
	}
}

static int stunc_get_auth_code(char *data, int data_len, const char *key, char *hash, int hash_size)
{
	if (data == NULL || key == NULL || hash == NULL)
		return -1;

	unsigned char *digest;
	digest = HMAC(EVP_sha1(), key, STRLEN(key), (unsigned char *)data, data_len, NULL, NULL);
	memcpy(hash, digest, hash_size);

	return 0;
}

static int stunc_init(void)
{
	int len;
	char *val;

	stunc_uci_init();
	memset(&conf, 0, sizeof(struct stun_config));
	conf.loglevel = DEFAULT_LOGLEVEL;

	openlog("stunc", LOG_CONS | LOG_PID | LOG_NDELAY, LOG_LOCAL1);
	SLOG(INFO, "stunc init");

	val = stunc_uci_get_value("stunc", "stunc", "server_address");
	if (STRLEN(val) == 0) {
		char address[256] = {0};

		SLOG(INFO, "Server Address not present in the STUN config");
		FREE(val);

		/* Read from ACS address */
		get_acs_address(&val);
		if (val == NULL)
			goto error;

		char *tmp = val;
		if (strstr(val, "http://")) {
			val = val + 7;
		} else if (strstr(val, "https://")) {
			val = val + 8;
		} else {
			free(tmp);
			goto error;
		}

		char *end = strchr(val, ':');

		if (end != NULL)
			goto get_addr;

		end = strchr(val, '/');
		if (end != NULL)
			goto get_addr;

		end = val + STRLEN(val) - 1;

get_addr:
		len = end - val;

		if (len <= 0 || len >= 256) {
			free(tmp);
			goto error;
		}

		snprintf(address, len + 1, "%s", val);
		free(tmp);
		conf.server_address = strdup(address);
	} else {
		conf.server_address = strdup(val);
		FREE(val);
	}

	val = stunc_uci_get_value("stunc", "stunc", "username");
	if (STRLEN(val))
		conf.username = strdup(val);
	FREE(val);

	val = stunc_uci_get_value("stunc", "stunc", "password");
	if (STRLEN(val))
		conf.password = strdup(val);
	FREE(val);

	val = stunc_uci_get_value("stunc", "stunc", "server_port");
	if (STRLEN(val)) {
		conf.server_port = (int)strtol(val, NULL, 10);
	} else {
		conf.server_port = DEFAULT_SERVERPORT;
	}
	FREE(val);

	val = stunc_uci_get_value("stunc", "stunc", "log_level");
	if (STRLEN(val)) {
		conf.loglevel = (int)strtol(val, NULL, 10);
	} else {
		conf.loglevel = DEFAULT_LOGLEVEL;
	}
	FREE(val);

	val = stunc_uci_get_value("stunc", "stunc", "min_keepalive");
	if (STRLEN(val)) {
		conf.min_keepalive = (int)strtol(val, NULL, 10);
	} else {
		conf.min_keepalive = DEFAULT_MINKEEPALIVE;
	}
	FREE(val);

	val = stunc_uci_get_value("stunc", "stunc", "max_keepalive");
	if (STRLEN(val)) {
		conf.max_keepalive = (int)strtol(val, NULL, 10);
	} else {
		conf.max_keepalive = DEFAULT_MAXKEEPALIVE;
	}
	FREE(val);

	val = stunc_uci_get_value("stunc", "stunc", "client_port");
	if (STRLEN(val))
		conf.client_port = (int)strtol(val, NULL, 10);
	FREE(val);

	if (conf.max_keepalive <= 0)
		conf.max_keepalive = DEFAULT_MAXKEEPALIVE;

	if (conf.min_keepalive <= 0)
		conf.min_keepalive = DEFAULT_MINKEEPALIVE;

	if (conf.server_port <= 0)
		conf.server_port = DEFAULT_SERVERPORT;

	if (conf.loglevel >= __MAX_LEVEL || conf.loglevel < 0)
		conf.loglevel = DEFAULT_LOGLEVEL;

	SLOG(INFO, "STUN CONFIG - Server Address: [%s]", conf.server_address);
	SLOG(INFO, "STUN CONFIG - Username: [%s]", conf.username ? conf.username : "<not defined>");
	SLOG(INFO, "STUN CONFIG - Server port: %d", conf.server_port);
	SLOG(INFO, "STUN CONFIG - min keepalive: %d", conf.min_keepalive);
	SLOG(INFO, "STUN CONFIG - max keepalive: %d", conf.max_keepalive);
	SLOG(INFO, "STUN CONFIG - Client port: %d", (conf.client_port > 0) ? conf.client_port : -1);
	SLOG(INFO, "STUN CONFIG - LOG Level: %d (Critical=0, Warning=1, Notice=2, Info=3, Debug=4)", conf.loglevel);

	state_conf_init();

	stunc_uci_finish();
	keepalive_time = conf.min_keepalive;
	retry_timeout = DEFAULT_RETRYTIME;
	memset(&req, 0, sizeof(struct binding_request));
	req.binding_cr = 1;

	return 0;

error:
	stunc_uci_finish();
	stunc_finish();
	exit(1);
}

static int stunc_start(void)
{
	stunc_fd.fd = stunc_open_socket(conf.client_port);
	if (stunc_fd.fd < 0) {
		SLOG(INFO, "Failed to open socket for stun client");
		return -1;
	}

	SLOG(INFO, "stunc new socket: %d", stunc_fd.fd);
	uloop_fd_add(&stunc_fd, ULOOP_READ | ULOOP_EDGE_TRIGGER);
	uloop_timeout_set(&request_timer.stunc_timer, 100);

	return 0;
}

static int stunc_finish(void)
{
	FREE(conf.server_address);
	FREE(conf.password);
	FREE(conf.username);
	conf.server_address = NULL;
	conf.username = NULL;
	conf.password = NULL;
	SLOG(INFO, "stunc stopped");
	closelog();

	return 0;
}

static void stunc_send_cr_event(struct uloop_timeout *timeout __attribute__((unused)))
{
	int ret;
	struct blob_buf b;

	memset(&b, 0, sizeof(struct blob_buf));
	blob_buf_init(&b, 0);
	blobmsg_add_string(&b, "event", "6 CONNECTION REQUEST");

	ret = stunc_ubus_call("tr069", "inform", b.head, NULL);
	blob_buf_free(&b);

	if (ret < 0) {
		SLOG(INFO, "failed to send connection req event, retry after 1 sec");
		uloop_timeout_set(&send_crevent_timer, 1000);
	} else
		SLOG(INFO, "Successfully sent connection req event to tr069");
}

static void stunc_packet_process(struct uloop_timeout *timeout __attribute__((unused)))
{
	process_server_message();
}

static int stunc_open_socket(int port)
{
	int reuse_addr = 1;
	int reuse_port = 1;
	struct sockaddr_in bindadd = {0};
	int fd = socket(AF_INET, SOCK_DGRAM, 0);

	if (fd < 0)
		return fd;

	if (port > 0) {
		memset(&bindadd, 0, sizeof(bindadd));

		if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &reuse_addr, sizeof(int)) < 0)
			SLOG(WARNING, "setsockopt failed to configure REUSEADDR");

		if (setsockopt(fd, SOL_SOCKET, SO_REUSEPORT, &reuse_port, sizeof(int)) < 0)
			SLOG(WARNING, "setsockopt failed to configure REUSEPORT");

		bindadd.sin_family = AF_INET;
		bindadd.sin_addr.s_addr = htonl(INADDR_ANY);
		bindadd.sin_port = htons((unsigned short)port);

		if (bind(fd, (struct sockaddr *)&bindadd, sizeof(bindadd)) < 0) {
			close(fd);
			fd = -1;
			SLOG(CRIT, "socket binding failed on port %u", port);
			return fd;
		}

		SLOG(INFO, "socket binding done on port %u", port);
	}

	fcntl(fd, F_SETFL, fcntl(fd, F_GETFL) | O_NONBLOCK);
	fcntl(fd, F_SETFD, fcntl(fd, F_GETFD) | FD_CLOEXEC);
	return fd;
}

static ssize_t stunc_recv_from_server(int fd, char *buf, int len, struct sockaddr_in *con, int timeout)
{
	ssize_t ret = -1;
	struct timeval time;
	unsigned int size;

	if (buf == NULL || con == NULL)
		return ret;

	fd_set fds;

	FD_ZERO(&fds);
	FD_SET(fd, &fds);

	memset(&time, 0, sizeof(struct timeval));
	time.tv_sec = timeout;

	size = sizeof(struct sockaddr_in);

	if (select(fd + 1, &fds, NULL, NULL, &time) != 0)
		ret = recvfrom(fd, buf, len, 0, (struct sockaddr *)con, &size);

	return ret;
}

static void change_response_status(void)
{
	req.resp_success = (req.resp_success > 0) ? 0 : req.resp_success;
}

static void change_next_retry_interval(void)
{
	if (req.retry_interval == 0)
		req.retry_interval = retry_timeout;
	else
		req.retry_interval = 2 * req.retry_interval;

	if (req.retry_interval > 1500)
		req.retry_interval = 1500;
}

static void stunc_send_bind_req(struct uloop_timeout *timeout)
{
	int res = 0;
	char msg[2048] = {0};
	int msglen;
	struct stunc_timer_t *timer = container_of(timeout, struct stunc_timer_t, stunc_timer);
	struct hostent *host;
	struct sockaddr_in to = {0};
	struct stun_header *hdr;

	SLOG(INFO, "%s expired", timer->timer_type);
	change_response_status();

	if (stunc_create_req(msg, sizeof(msg) - 1)) {
		change_next_retry_interval();
		uloop_timeout_set(timeout, req.retry_interval * 1000);
		return;
	}

	host = gethostbyname(conf.server_address);
	if (host == NULL) {
		res = -1;
		goto exit;
	}

	memcpy(&(to.sin_addr), host->h_addr_list[0], host->h_length);
	to.sin_port = htons(conf.server_port);
	to.sin_family = AF_INET;

	hdr = (struct stun_header *)msg;
	msglen = ntohs(hdr->len) + sizeof(struct stun_header);
	res = sendto(stunc_fd.fd, msg, msglen, 0, (struct sockaddr *)&to, sizeof(to));
exit:
	if (res < 0) {
		if (stunc_fd.fd > 0) {
			uloop_fd_delete(&stunc_fd);
			close(stunc_fd.fd);
			stunc_fd.fd = -1;
			stunc_fd.fd = stunc_open_socket(conf.client_port);
			assert(stunc_fd.fd >= 0);
			uloop_fd_add(&stunc_fd, ULOOP_READ | ULOOP_EDGE_TRIGGER);
		}

		change_next_retry_interval();
		uloop_timeout_set(timeout, req.retry_interval * 1000);
		SLOG(INFO, "retry to send the request with new socket after %d sec", req.retry_interval);
	} else {
		SLOG(INFO, "request send successfully, start keepalive timer");
		uloop_timeout_set(&keepalive_timer.stunc_timer, keepalive_time * 1000);
		uloop_timeout_set(&packet_process_timer, 1);
		req.retry_interval = 0;
		req.br_resp_pending = true;
	}
}

static void stunc_listen_message(struct uloop_fd *fd, unsigned int event __attribute__((unused)))
{
	if (fd->fd < 0)
		return;

	process_server_message();
}

static void process_error_response(char *resp_msg)
{
	int interval;
	struct msg_var var;

	memset(&var, 0, sizeof(var));
	req.br_resp_pending = false;
	stun_read_from_packet(resp_msg, ATTR_ERROR_CODE, &var);
	if (var.code != 401) {
		SLOG(INFO, "Error code: %d not supported", var.code);
		return;
	}

	req.auth_fail++;
	req.msg_integrity = 1;
	SLOG(INFO, "cancel keep-alive timer if running");
	uloop_timeout_cancel(&keepalive_timer.stunc_timer);
	SLOG(INFO, "send new binding request");
	interval = (req.auth_fail < 3) ? 0 : ((req.auth_fail - 2) * 3000);
	uloop_timeout_set(&request_timer.stunc_timer, interval);
}

static void process_connection_request(char *resp_msg)
{
	char *temp;
	unsigned int cr_id, cr_ts;
	char cr_un[64] = {0}, cr_cn[64] = {0}, cr_sig[64] = {0};
	bool is_valid_req = true;
	char *username;
	char *password;

	if (resp_msg == NULL)
		return;

	if (strcasestr(resp_msg, "http") == NULL) {
		SLOG(INFO, "unsupported stun message");
		return;
	}

	SLOG(INFO, "got UDP connection request");

	temp = strstr(resp_msg, "ts=");
	if (temp == NULL) {
		SLOG(WARNING, "CR ts not found in message");
		return;
	}

	sscanf(temp, "ts=%u", &cr_ts);
	SLOG(INFO, "received CR ts is %u", cr_ts);

	temp = strstr(resp_msg, "id=");
	if (temp == NULL) {
		SLOG(WARNING, "CR id not found in message");
		return;
	}

	sscanf(temp, "id=%u", &cr_id);
	SLOG(INFO, "received CR id is %u", cr_id);

	if (cr_ts == 0 || cr_id == 0) {
		SLOG(INFO, "CR ts or id is 0");
		return;
	}

	if (cr_id == req.env.last_crid || cr_ts <= req.env.last_ts) {
		SLOG(INFO, "id had already been used or ts is older");
		return;
	}

	SLOG(INFO, "got new connection request");
	req.env.last_crid = cr_id;
	req.env.last_ts = cr_ts;

	temp = strstr(resp_msg, "un=");
	if (temp == NULL) {
		SLOG(WARNING, "CR un not found in message");
		return;
	}

	sscanf(temp, "un=%63[^?& \t\n\r]", cr_un);
	SLOG(INFO, "received CR un is %s", cr_un);

	temp = strstr(resp_msg, "cn=");
	if (temp == NULL) {
		SLOG(WARNING, "CR cn not found in message");
		return;
	}

	sscanf(temp, "cn=%63[^?& \t\n\r]", cr_cn);
	SLOG(INFO, "received CR cn is %s", cr_cn);

	temp = strstr(resp_msg, "sig=");
	if (temp == NULL) {
		SLOG(WARNING, "CR sig not found in message");
		return;
	}

	sscanf(temp, "sig=%63[^?& \t\n\r]", cr_sig);
	SLOG(INFO, "received CR cn is %s", cr_sig);

	stunc_uci_init();
	username = stunc_uci_get_value("cwmp", "cpe", "userid");
	password = stunc_uci_get_value("cwmp", "cpe", "password");
	stunc_uci_finish();

	if (STRLEN(username) != 0 && STRLEN(password) != 0) {
		if (strcmp(username, cr_un) != 0) {
			SLOG(INFO, "username mismatched");
			is_valid_req = false;
		} else {
			char hash[20] = {0}, data[256] = {0};

			snprintf(data, sizeof(data), "%u%u%s%s", cr_ts, cr_id, cr_un, cr_cn);
			stunc_get_auth_code(data, STRLEN(data), password, hash, sizeof(hash));

			char hexstr[64] = {0};

			get_hexstring(hash, sizeof(hash), hexstr, sizeof(hexstr));
			if (strcasecmp(hexstr, cr_sig) != 0) {
				SLOG(INFO, "signature mismatched");
				is_valid_req = false;
			}
		}
	}

	FREE(username);
	FREE(password);
	if (is_valid_req == true)
		uloop_timeout_set(&send_crevent_timer, 0);
}

static void process_binding_response(char *resp_msg)
{
	struct msg_var var;

	memset(&var, 0, sizeof(var));

	stun_read_from_packet(resp_msg, ATTR_MAPPED_ADDRESS, &var);
	req.br_resp_pending = false;
	req.resp_success = 1;
	req.msg_integrity = 0;
	req.auth_fail = 0;
	change_nat_existance(var.ip);
	change_mapped_address(var.ip, var.port);
}

static void stun_read_from_packet(char *resp_msg, int att_type, struct msg_var *var)
{
	struct stun_header *hdr;
	struct stun_attribute *attr;
	struct stun_address *map_addr;
	unsigned int value, class, num, *p;
	int attr_len;
	int covered_len;

	if (resp_msg == NULL || var == NULL)
		return;

	memset(var, 0, sizeof(struct msg_var));
	hdr = (struct stun_header *)resp_msg;
	attr = (struct stun_attribute *)hdr->stunmsg;
	attr_len = ntohs(hdr->len);
	covered_len = (char *)attr - (char *)hdr - sizeof(struct stun_header);

	while (covered_len < attr_len) {
		char *temp;

		if (ntohs(attr->type) == att_type) {
			switch (att_type) {
			case ATTR_MAPPED_ADDRESS:
				map_addr = (struct stun_address *)attr->value;
				var->port = map_addr->port;
				var->ip = map_addr->address;
				return;
			case ATTR_ERROR_CODE:
				p = (unsigned int *)attr->value;
				value = ntohl(*p);
				class = (value >> 8) & 0x7;
				num = value & 0xff;
				var->code = (int)(class * 100 + num);
				return;
			}
		}

		temp = (char *)attr;
		temp = temp + sizeof(struct stun_attribute) + ntohs(attr->len);
		attr = (struct stun_attribute *)temp;
		covered_len = temp - (char *)hdr - sizeof(struct stun_header);
	}
}

static void change_nat_existance(unsigned int ip)
{
	struct ifaddrs *iflist = NULL, *iter = NULL;
	bool local_ip = false;
	char *val, *p;

	getifaddrs(&iflist);

	for (iter = iflist; iter != NULL; iter = iter->ifa_next) {
		if (!iter->ifa_addr)
			continue;

		if (iter->ifa_addr->sa_family == AF_INET) {
			struct sockaddr_in *ifadd = (struct sockaddr_in *)iter->ifa_addr;
			unsigned int if_ip = ifadd->sin_addr.s_addr;

			if (if_ip == ip) {
				local_ip = true;
				break;
			}
		}
	}

	if (iflist)
		freeifaddrs(iflist);

	stunc_uci_init();
	val = stunc_uci_get_value_state("stun", "stunc", "nat_detected");

	if ((local_ip == true && STRLEN(val) == 0) || (local_ip == false && STRLEN(val) != 0)) {
		SLOG(INFO, "local_ip: %d & uci val: (%s)", local_ip, val ? val : "");
		FREE(val);
		stunc_uci_finish();
		return;
	}

	FREE(val);
	p = local_ip ? "" : "1";
	stunc_uci_set_value_state("stun", "stunc", "nat_detected", p);
	stunc_uci_finish();
	SLOG(INFO, "configured nat detected to %d", !local_ip);
}

static void change_mapped_address(unsigned int ip_addr, unsigned short port)
{
	char map_addr[64] = {0};
	struct in_addr ip;
	char *val;

	ip.s_addr = ip_addr;

	snprintf(map_addr, sizeof(map_addr), "%s:%d", inet_ntoa(ip), ntohs(port));
	SLOG(INFO, "received MAPPED-ADDRESS: %s", map_addr);

	stunc_uci_init();
	val = stunc_uci_get_value_state("stun", "stunc", "crudp_address");
	if (val && strcmp(map_addr, val) != 0) {
		req.binding_change = 1;
		uloop_timeout_set(&request_timer.stunc_timer, 0);
		SLOG(INFO, "save new connection request address to /var/state");
		stunc_uci_set_value_state("stun", "stunc", "crudp_address", map_addr);
	} else {
		req.binding_change = 0;
	}

	FREE(val);
	stunc_uci_finish();
}

static void process_server_message(void)
{
	char resp_msg[2048] = {0};
	struct sockaddr_in from = {0};
	int res;

	if (stunc_fd.fd < 0)
		return;

	res = stunc_recv_from_server(stunc_fd.fd, resp_msg, sizeof(resp_msg) - 1, &from, 1);
	if (res > 0) {
		SLOG(INFO, "received packet from server");
		struct stun_header *hdr = (struct stun_header *)resp_msg;
		unsigned int msg_type = ntohs(hdr->type);

		switch (msg_type) {
		case BINDING_ERROR:
			SLOG(INFO, "got BINDING_ERROR");
			if (memcmp(&(hdr->id), &(req.id), sizeof(hdr->id)) == 0)
				process_error_response(resp_msg);
			break;
		case BINDING_RESPONSE:
			SLOG(INFO, "got BINDING_RESPONSE");
			if (memcmp(&(hdr->id), &(req.id), sizeof(hdr->id)) == 0)
				process_binding_response(resp_msg);
			break;
		default:
			process_connection_request(resp_msg);
		}
	} else {
		if (req.resp_success == 1)
			return;

		SLOG(DEBUG, "Timed Out on listening to server");
		req.resp_success--;
		int fail_count = abs(req.resp_success);

		if ((fail_count % 9) == 0) {
			SLOG(INFO, "Retry with new socket");
			uloop_fd_delete(&stunc_fd);
			close(stunc_fd.fd);
			stunc_fd.fd = stunc_open_socket(conf.client_port);
			assert(stunc_fd.fd >= 0);
			uloop_fd_add(&stunc_fd, ULOOP_READ | ULOOP_EDGE_TRIGGER);
			uloop_timeout_set(&request_timer.stunc_timer, 0);
		} else if ((fail_count % 3) == 0) {
			SLOG(INFO, "send the message again");
			uloop_timeout_set(&request_timer.stunc_timer, 0);
		} else {
			uloop_timeout_set(&packet_process_timer, 1);
		}
	}
}

static int add_padding(char *act_val, unsigned int act_len, unsigned int pad_len, char **pad_val, char pad_byte)
{
	unsigned int total_len;

	if (act_val == NULL || pad_val == NULL)
		return -1;

	total_len = act_len + pad_len;
	*pad_val = (char *)malloc(sizeof(char) * total_len);
	if (*pad_val == NULL)
		return -1;

	memset(*pad_val, 0, sizeof(char) * total_len);
	memcpy(*pad_val, act_val, act_len);

	if (pad_byte != 0) {
		unsigned int i;

		for (i = act_len; i < total_len; i++)
			*(*pad_val + i) = pad_byte;
	}

	return 0;
}

static void add_attribute_to_msg(unsigned char **pos, const char *attr, unsigned short attr_len, unsigned short attr_type, int rem_len)
{
	int new_attr_len;
	struct stun_attribute *temp;

	if (pos == NULL || attr == NULL)
		return;

	new_attr_len = (int)(sizeof(struct stun_attribute) + attr_len);
	if (new_attr_len > rem_len)
		return;

	temp = (struct stun_attribute *)*pos;
	temp->type = htons(attr_type);
	temp->len = htons(attr_len);
	memcpy(temp->value, attr, attr_len);

	*pos = *pos + sizeof(struct stun_attribute) + attr_len;
}

static int add_username(unsigned char **pos, int rem_len)
{
	unsigned short username_len;

	if (conf.username == NULL)
		return 0;

	username_len = STRLEN(conf.username);
	if (username_len == 0)
		return 0;

	if ((username_len % 4) != 0) {
		char *padded_val;
		unsigned short padded_len = 4 - (username_len % 4);

		if (0 != add_padding(conf.username, username_len, padded_len, &padded_val, ' '))
			return -1;

		add_attribute_to_msg(pos, padded_val, username_len + padded_len, ATTR_USERNAME, rem_len);
		SLOG(INFO, "added attribute USERNAME=%.*s", username_len+padded_len, padded_val);
		free(padded_val);
	} else {
		add_attribute_to_msg(pos, conf.username, username_len, ATTR_USERNAME, rem_len);
		SLOG(INFO, "added attribute USERNAME=%.*s", username_len, conf.username);
	}

	return 0;
}

static int add_connection_req(unsigned char **pos, int rem_len)
{
	if (!req.binding_cr)
		return 0;

	add_attribute_to_msg(pos, "dslforum.org/TR-111 ", 20, ATTR_CONNECTION_REQUEST_BINDING, rem_len);
	SLOG(INFO, "added attribute CONNECTION-REQUEST-BINDING");

	return 0;
}

static int add_binding_change(unsigned char **pos, int rem_len)
{
	if (!req.binding_change)
		return 0;

	add_attribute_to_msg(pos, "", 0, ATTR_BINDING_CHANGE, rem_len);
	SLOG(INFO, "added attribute BINDING-CHANGE");

	return 0;
}

static int add_msg_integrity(unsigned char **pos, int rem_len, struct stun_header *hdr)
{
	if (!req.msg_integrity) {
		unsigned short len = (*pos - (unsigned char *)hdr) - sizeof(struct stun_header);
		hdr->len = htons(len);
		return 0;
	}

	if (conf.username) {
		bool padded = false;
		char hash[20] = {0};
		char *padded_msg;
		unsigned int msg_len;
		char hexstr[64] = {0};

		unsigned short len = ((*pos - (unsigned char *)hdr) - sizeof(struct stun_header)) + sizeof(struct stun_attribute) + 20;
		hdr->len = htons(len);

		msg_len = *pos - (unsigned char *)hdr;
		if (msg_len % 64 != 0) {
			unsigned int padded_len = 64 - (msg_len % 64);

			if (0 != add_padding((char *)hdr, msg_len, padded_len, &padded_msg, 0))
				return -1;
			msg_len = msg_len + padded_len;
			padded = true;
		} else {
			padded_msg = (char *)hdr;
		}

		if (conf.password)
			stunc_get_auth_code(padded_msg, msg_len, conf.password, hash, sizeof(hash));
		else
			stunc_get_auth_code(padded_msg, msg_len, "", hash, sizeof(hash));

		add_attribute_to_msg(pos, hash, sizeof(hash), ATTR_MESSAGE_INTEGRITY, rem_len);
		if (padded == true)
			free(padded_msg);

		get_hexstring(hash, sizeof(hash), hexstr, sizeof(hexstr));
		SLOG(INFO, "added attribute MESSAGE-INTEGRITY=%s", hexstr);

		return 0;
	} else {
		req.msg_integrity = 0;
		unsigned short len = (*pos - (unsigned char *)hdr) - sizeof(struct stun_header);
		hdr->len = htons(len);
		return -1;
	}
}

static int stunc_create_req(char *msg, int msglen)
{
	struct stun_header *hdr = (struct stun_header *)msg;
	unsigned char *attr;
	int remain_len;

	hdr->type = htons(BINDING_REQUEST);

	if (req.br_resp_pending == false) {
		int ind;

		srand(time(NULL));
		for (ind = 0; ind < 4; ind++) {
			// cppcheck-suppress cert-MSC30-c
			hdr->id.id[ind] = rand();
		}
		memcpy(&(req.id), &(hdr->id), sizeof(req.id));
	} else {
		memcpy(&(hdr->id), &(req.id), sizeof(hdr->id));
	}

	attr = hdr->stunmsg;
	remain_len = msglen - (attr - (unsigned char *)msg);
	if (0 != add_username(&attr, remain_len))
		return -1;

	remain_len = msglen - (attr - (unsigned char *)msg);
	if (0 != add_connection_req(&attr, remain_len))
		return -1;

	remain_len = msglen - (attr - (unsigned char *)msg);
	if (0 != add_binding_change(&attr, remain_len))
		return -1;

	remain_len = msglen - (attr - (unsigned char *)msg);
	if (0 != add_msg_integrity(&attr, remain_len, hdr))
		return -1;

	SLOG(INFO, "message length: %d", ntohs(hdr->len));

	return 0;
}

static bool g_usp_object_available = false;

static void lookup_event_cb(struct ubus_context *ctx __attribute__((unused)),
		struct ubus_event_handler *ev __attribute__((unused)),
		const char *type, struct blob_attr *msg)
{
	const struct blobmsg_policy policy = {
		"path", BLOBMSG_TYPE_STRING
	};
	struct blob_attr *attr;
	const char *path;

	if (strcmp(type, "ubus.object.add") != 0)
		return;

	blobmsg_parse(&policy, 1, &attr, blob_data(msg), blob_len(msg));
	if (!attr)
		return;

	path = blobmsg_data(attr);
	if (strcmp(path, DATAMODEL_OBJECT) == 0) {
		g_usp_object_available = true;
		uloop_end();
	}
}

static void lookup_timeout_cb(struct uloop_timeout *timeout __attribute__((unused)))
{
	uloop_end();
}

static int wait_for_usp_raw_object()
{
#define USP_RAW_WAIT_TIMEOUT 60

	struct ubus_context *uctx;
	int ret;
	uint32_t ubus_id;
	struct ubus_event_handler add_event;
	struct uloop_timeout u_timeout;

	g_usp_object_available = false;
	uctx = ubus_connect(NULL);
	if (uctx == NULL) {
		fprintf(stderr, "Can't create ubus context");
		return -1;
	}

	uloop_init();
	ubus_add_uloop(uctx);

	// register for add event
	memset(&add_event, 0, sizeof(struct ubus_event_handler));
	add_event.cb = lookup_event_cb;
	ubus_register_event_handler(uctx, &add_event, "ubus.object.add");

	// check if object already present
	ret = ubus_lookup_id(uctx, DATAMODEL_OBJECT, &ubus_id);
	if (ret == 0) {
		g_usp_object_available = true;
		goto end;
	}

	// Set timeout to expire lookup
	memset(&u_timeout, 0, sizeof(struct uloop_timeout));
	u_timeout.cb = lookup_timeout_cb;
	uloop_timeout_set(&u_timeout, USP_RAW_WAIT_TIMEOUT * 1000);

	uloop_run();
	uloop_done();

end:
	ubus_free(uctx);

	if (g_usp_object_available == false) {
		fprintf(stderr, "%s object not found", DATAMODEL_OBJECT);
		return -1;
	}

	return 0;
}

int main(void)
{
	int ret;

	stunc_init();
	uloop_init();

	ret = wait_for_usp_raw_object();
	if (ret) {
		fprintf(stderr, "Wait for %s results fault %d", DATAMODEL_OBJECT, ret);
		return ret;
	}

	if (0 != stunc_start()) {
		stunc_finish();
		return 0;
	}

	uloop_run();
	uloop_done();

	close(stunc_fd.fd);

	stunc_finish();
	return 0;
}
