/*
 * datamodel.c: File to expose stun related parameters
 *
 * Copyright (C) 2022 IOPSYS Software Solutions AB. All rights reserved.
 *
 * Author: Vivek Dutta <vivek.dutta@iopsys.eu>
 *
 * See LICENSE file for license related information.
 *
 */

#include <libbbfdm_api.h>
#include <libbbfdm-api/dmcommon.h>

/*************************************************************
* GET & SET PARAM
*************************************************************/
/*#Device.ManagementServer.STUNEnable!UCI:stunc/stunc,stunc/enabled*/
static int get_stun_enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_option_value_fallback_def("stunc", "stunc", "enabled", "1");
	return 0;
}

static int set_stun_enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_boolean(ctx, value))
			ret = FAULT_9007;
		break;
	case VALUESET:
		bbf_convert_string_to_bool(value, &b);
		bbf_uci_set_value("stunc", "stunc", "enabled", b ? "1" : "0");
		break;
	}

	return ret;
}

/*#Device.ManagementServer.STUNServerAddress!UCI:stunc/stunc,stunc/server_address*/
static int get_stun_server_address(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	bbf_uci_get_value("stunc", "stunc", "server_address", value);
	return 0;
}

static int set_stun_server_address(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_string(ctx, value, -1, 256, NULL, NULL))
			ret = FAULT_9007;
		break;
	case VALUESET:
		bbf_uci_set_value("stunc", "stunc", "server_address", value);
		break;
	}

	return ret;
}

/*#Device.ManagementServer.STUNServerPort!UCI:stunc/stunc,stunc/server_port*/
static int get_stun_server_port(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_option_value_fallback_def("stunc", "stunc", "server_port", "3478");
	return 0;
}

static int set_stun_server_port(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"0", "65535"}}, 1))
			ret = FAULT_9007;
		break;
	case VALUESET:
		bbf_uci_set_value("stunc", "stunc", "server_port", value);
		break;
	}

	return ret;
}

/*#Device.ManagementServer.STUNUsername!UCI:stunc/stunc,stunc/username*/
static int get_stun_username(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	bbf_uci_get_value("stunc", "stunc", "username", value);
	return 0;
}

static int set_stun_username(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_string(ctx, value, -1, 256, NULL, NULL))
			ret = FAULT_9007;
		break;
	case VALUESET:
		bbf_uci_set_value("stunc", "stunc", "username", value);
		break;
	}

	return ret;
}

/*#Device.ManagementServer.STUNPassword!UCI:stunc/stunc,stunc/password*/
static int get_stun_password(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return 0;
}

static int set_stun_password(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_string(ctx, value, -1, 256, NULL, NULL))
			ret = FAULT_9007;
		break;
	case VALUESET:
		bbf_uci_set_value("stunc", "stunc", "password", value);
		break;
	}

	return ret;
}

/*#Device.ManagementServer.STUNMaximumKeepAlivePeriod!UCI:stunc/stunc,stunc/max_keepalive*/
static int get_stun_maximum_keepalive_period(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_option_value_fallback_def("stunc", "stunc", "max_keepalive", "-1");
	return 0;
}

static int set_stun_maximum_keepalive_period(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_int(ctx, value, RANGE_ARGS{{"-1", NULL}}, 1))
			ret = FAULT_9007;
		break;
	case VALUESET:
		bbf_uci_set_value("stunc", "stunc", "max_keepalive", value);
		break;
	}

	return ret;
}

/*#Device.ManagementServer.STUNMinimumKeepAlivePeriod!UCI:stunc/stunc,stunc/min_keepalive*/
static int get_stun_minimum_keepalive_period(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_option_value_fallback_def("stunc", "stunc", "min_keepalive", "30");
	return 0;
}

static int set_stun_minimum_keepalive_period(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	int ret = 0;

	switch (action) {
	case VALUECHECK:
		if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL, NULL}}, 1))
			ret = FAULT_9007;
		break;
	case VALUESET:
		bbf_uci_set_value("stunc", "stunc", "min_keepalive", value);
		break;
	}

	return ret;
}

/*** ManagementServer. ***/
DMLEAF tManagementServerStunParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"STUNEnable", &DMWRITE, DMT_BOOL, get_stun_enable, set_stun_enable, BBFDM_CWMP},
{"STUNServerAddress", &DMWRITE, DMT_STRING, get_stun_server_address, set_stun_server_address, BBFDM_CWMP},
{"STUNServerPort", &DMWRITE, DMT_UNINT, get_stun_server_port, set_stun_server_port, BBFDM_CWMP},
{"STUNUsername", &DMWRITE, DMT_STRING, get_stun_username, set_stun_username, BBFDM_CWMP},
{"STUNPassword", &DMWRITE, DMT_STRING, get_stun_password, set_stun_password, BBFDM_CWMP},
{"STUNMaximumKeepAlivePeriod", &DMWRITE, DMT_INT, get_stun_maximum_keepalive_period, set_stun_maximum_keepalive_period, BBFDM_CWMP},
{"STUNMinimumKeepAlivePeriod", &DMWRITE, DMT_UNINT, get_stun_minimum_keepalive_period, set_stun_minimum_keepalive_period, BBFDM_CWMP},
{0}
};

DM_MAP_OBJ tDynamicObj[] = {
/* parentobj, nextobject, parameter */
{"Device.ManagementServer.", NULL, tManagementServerStunParams},
{0}
};
