/*
 * config.c -- Utility functions to access over uci config parameters of stunc
 *
 * Copyright (C) 2022 IOPSYS Software Solutions AB. All rights reserved.
 *
 * Author: Omar Kallel <omar.kallel@pivasoftware.com>
 *
 * See LICENSE file for license related information.
 *
 */

#include "common.h"
#include <strings.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/stat.h>
#include <libubox/blobmsg.h>

static struct uci_context *uci_ctx = NULL;
static struct uci_context *uci_ctx_state = NULL;
struct stun_config conf;

void get_acs_address(char **val)
{
	struct ubus_result res;
	struct blob_buf b;
	int ret;

	*val = NULL;

	memset(&res, 0, sizeof(struct ubus_result));
	snprintf(res.path, sizeof(res.path), "Device.ManagementServer.URL");

	memset(&b, 0, sizeof(struct blob_buf));
	blob_buf_init(&b, 0);
	blobmsg_add_string(&b, "path", "Device.ManagementServer.URL");

	void *table = blobmsg_open_table(&b, "optional");
	blobmsg_add_string(&b, "proto", "cwmp");
	blobmsg_add_string(&b, "format", "raw");
	blobmsg_close_table(&b, table);

	ret = stunc_ubus_call(DATAMODEL_OBJECT, "get", b.head, &res);
	blob_buf_free(&b);

	if (ret < 0 || STRLEN(res.value) == 0)
		return;

	*val = strdup(res.value);
}

bool file_exists(const char *path)
{
	struct stat statbuf;

	return stat(path, &statbuf) == 0;
}

void create_file(const char *path)
{
	if (!file_exists(path)) {
		// cppcheck-suppress cert-MSC24-C
		FILE *fp = fopen(path, "w"); // new empty file

		if (fp != NULL)
			fclose(fp);

		sync();
	}
}

int stunc_uci_init(void)
{
	uci_ctx = uci_alloc_context();
	if (uci_ctx == NULL)
		return -1;

	uci_ctx_state = uci_alloc_context();
	if (uci_ctx_state == NULL)
		return -1;

	uci_set_confdir(uci_ctx_state, VAR_STATE);
	return 0;
}

int stunc_uci_finish(void)
{
	if (uci_ctx != NULL) {
		uci_free_context(uci_ctx);
		uci_ctx = NULL;
	}

	if (uci_ctx_state != NULL) {
		uci_free_context(uci_ctx_state);
		uci_ctx_state = NULL;
	}

	return 0;
}

static bool suci_validate_section(const char *str)
{
	if (!*str)
		return false;

	for (; *str; str++) {
		unsigned char c = *str;

		if (isalnum(c) || c == '_')
			continue;

		return false;
	}

	return true;
}

static int suci_lookup_ptr(struct uci_context *ctx, struct uci_ptr *ptr, const char *package, const char *section, const char *option, const char *value)
{
	memset(ptr, 0, sizeof(struct uci_ptr));

	/* value */
	if (value)
		ptr->value = value;

	ptr->package = package;
	if (!ptr->package)
		goto error;

	ptr->section = section;
	if (!ptr->section) {
		ptr->target = UCI_TYPE_PACKAGE;
		goto lastval;
	}

	ptr->option = option;
	if (!ptr->option) {
		ptr->target = UCI_TYPE_SECTION;
		goto lastval;
	} else {
		ptr->target = UCI_TYPE_OPTION;
	}

lastval:
	if (ptr->section && !suci_validate_section(ptr->section))
		ptr->flags |= UCI_LOOKUP_EXTENDED;

	if (uci_lookup_ptr(ctx, ptr, NULL, false) == UCI_OK)
		return 0;
error:
	return -1;
}

static void suci_print_list(struct uci_list *uh, char **val, const char *delimiter)
{
	struct uci_element *e;
	char buffer[512] = {0};
	unsigned pos = 0;

	buffer[0] = 0;
	uci_foreach_element(uh, e) {
		if (e->name)
			pos += snprintf(&buffer[pos], sizeof(buffer) - pos, "%s%s", e->name, delimiter);
	}

	if (pos)
		buffer[pos - 1] = 0;

	*val = strdup(buffer);
}

void state_conf_init(void)
{
	struct uci_ptr ptr;
	struct uci_section *s = NULL;

	create_file("/var/state/stun");

	if (suci_lookup_ptr(uci_ctx_state, &ptr, "stun", NULL, NULL, NULL))
		return;

	s = uci_lookup_section(uci_ctx_state, ptr.p, "stunc");
	if (s != NULL) {
		SLOG(INFO, "stunc already present in var section");
		return;
	}

	if (uci_add_section(uci_ctx_state, ptr.p, "stunc", &s) != UCI_OK)
		return;

	ptr.package = s->package->e.name;
	ptr.p = s->package;

	ptr.section = s->e.name;
	ptr.s = s;

	ptr.value = "stunc";
	ptr.target = UCI_TYPE_SECTION;

	if (uci_lookup_ptr(uci_ctx_state, &ptr, NULL, true) == UCI_OK) {
		uci_rename(uci_ctx_state, &ptr);
		uci_commit(uci_ctx_state, &ptr.p, false);
	}
}

char *stunc_uci_get_value(const char *package, const char *section, const char *option)
{
	struct uci_ptr ptr;
	char *val = NULL;

	if (!section || !option)
		return strdup("");

	if (suci_lookup_ptr(uci_ctx, &ptr, package, section, option, NULL))
		return strdup("");

	if (!ptr.o)
		return strdup("");

	if (ptr.o->type == UCI_TYPE_LIST) {
		suci_print_list(&ptr.o->v.list, &val, " ");
		return val;
	}

	if (ptr.o->v.string)
		return strdup(ptr.o->v.string);
	else
		return strdup("");
}

char *stunc_uci_set_value(const char *package, const char *section, const char *option, const char *value, bool is_commit)
{
	struct uci_ptr ptr;
	int ret = UCI_OK;

	if (!section)
		return "";

	if (suci_lookup_ptr(uci_ctx, &ptr, package, section, option, value))
		return "";

	ret = uci_set(uci_ctx, &ptr);
	if (ret == UCI_OK) {
		ret = uci_save(uci_ctx, ptr.p);
		if (is_commit)
			uci_commit(uci_ctx, &ptr.p, false);
	}

	if (ptr.o && ptr.o->v.string)
		return ptr.o->v.string;

	return "";
}

/*************************---/var/state--***************************/

char *stunc_uci_get_value_state(const char *package, const char *section, const char *option)
{
	char *val;
	struct uci_context *save_uci_ctx = uci_ctx;

	uci_ctx = uci_ctx_state;
	val = stunc_uci_get_value(package, section, option);
	uci_ctx = save_uci_ctx;

	return val;
}

char *stunc_uci_set_value_state(const char *package, const char *section, const char *option, const char *value)
{
	char *val;
	struct uci_context *save_uci_ctx = uci_ctx;

	uci_ctx = uci_ctx_state;
	val = stunc_uci_set_value(package, section, option, value, true);
	uci_ctx = save_uci_ctx;

	return val;
}
