/*
 * Copyright (C) 2024 iopsys Software Solutions AB
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1
 * as published by the Free Software Foundation
 *
 *	  Author Suvendhu Hansa <suvendhu.hansa@iopsys.eu>
 *	  Author Mohd Husaam Mehdi <husaam.mehdi@iopsys.eu>
 *
 */

#include "sshmngr.h"

/* 
 * the name of config file depends on the backend
 * so we first determine which backend is currently running
 * on the device, and then use appropriate file names
 *
 * NOTE: this approach works mainly because currently only
 * the filenames need to be changed
 * the "syntax" of the files for both dropbear and openssh
 * is the same, if that changes, we will need to differentiate
 * per parameter logic or re-evaluate this approach
 */

#ifdef DROPBEAR_BACKEND
#define BACKEND_UCI "dropbear"
#define BACKEND_SECTION_TYPE "dropbear"
#else
#define BACKEND_UCI "sshd"
#define BACKEND_SECTION_TYPE "sshd"
#endif

/*************************************************************
 * UTILITY FUNCTIONS
 *************************************************************/
/* check if key exists in dmmap file */
static bool key_exists_in_dmmap(const char *key)
{
	struct uci_section *s = NULL, *stmp = NULL;

	uci_path_foreach_sections_safe(bbfdm, "SSH", "AuthorizedKey", stmp, s) {
		char *val = NULL;
		dmuci_get_value_by_section_string(s, "Key", &val);
		if (DM_STRCMP(val, key) == 0) {
			return true;
		}
	}

	return false;
}

static void dmmap_synchronize_SSH_Server(struct dmctx *dmctx)
{
	struct uci_section *config_s = NULL;
	char *instance = NULL;

	uci_foreach_sections(BACKEND_UCI, BACKEND_SECTION_TYPE, config_s) {
		create_dmmap_obj(dmctx, 0, "SSH", "Server", config_s, &instance);
	}
}

static int dmmap_synchronize_SSH_AuthorizedKey(struct dmctx *dmctx)
{
	char *instance = NULL;
	char *key = NULL;
	unsigned idx = 0;
	json_object *json_key_list = NULL, *arr_obj = NULL;

	dmubus_call_timeout("sshmngr", "list_keys", UBUS_ARGS{0}, 0, 5000, &json_key_list);
	DM_ASSERT(json_key_list, return 0);

	dmjson_foreach_value_in_array(json_key_list, arr_obj, key, idx, 1, "keys") {
		if (!DM_STRLEN(key))
			continue;

		if (!key_exists_in_dmmap(key)) {
			struct uci_section *dmmap_sect = create_dmmap_obj(dmctx, 0, "SSH", "AuthorizedKey", NULL, &instance);
			dmuci_set_value_by_section(dmmap_sect, "Key", key);
		}
	}

	return 0;
}

static void dmmap_synchronize_SSH(struct dmctx *dmctx)
{
	// Device.SSH.Server.{i}.
	dmmap_synchronize_SSH_Server(dmctx);

	// Device.SSH.AuthorizedKey.{i}.
	dmmap_synchronize_SSH_AuthorizedKey(dmctx);
}

/* replace/add key to backend key file */
static void add_pubkey(const char *cur, const char *new)
{
	struct json_object *res = NULL;

	dmubus_call("sshmngr", "add_pubkey", UBUS_ARGS{{"current_key", cur, String}, {"new_key", new, String}}, 2, &res);
}

/* remove key from backend key file */
static void remove_pubkey(const char *key)
{
	struct json_object *res = NULL;

	if (DM_STRLEN(key) == 0)
		return;

	dmubus_call("sshmngr", "remove_pubkey", UBUS_ARGS{{"key", key, String}}, 1, &res);
}

/* close all active sessions of a server */
static void close_active_sessions(struct uci_section *s)
{
	bool b;
	char *sec_name = NULL;

	char *enable = dmuci_get_value_by_section_fallback_def(s, "enable", "0");
	string_to_bool(enable, &b);
	if (!b)
		return;

	sec_name = section_name(s);
	if (DM_STRLEN(sec_name) == 0)
		return;

	json_object *kill_res = NULL;
	dmubus_call("sshmngr", "kill_session", UBUS_ARGS{{"server_name", sec_name, String}}, 1, &kill_res);
}

/*************************************************************
 * ADD & DEL OBJ
 *************************************************************/
static int addObjSSHServer(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct dm_data *curr_data = (struct dm_data *)ctx->addobj_instance;
	char s_name[16];

	snprintf(s_name, sizeof(s_name), "server_%s", *instance);

	/* set default values as mentioned in data model spec */
	dmuci_add_section(BACKEND_UCI, BACKEND_SECTION_TYPE, &curr_data->config_section);
	dmuci_rename_section_by_section(curr_data->config_section, s_name);
	dmuci_set_value_by_section(curr_data->config_section, "enable", "0");
	dmuci_set_value_by_section(curr_data->config_section, "Port", "22");
	dmuci_set_value_by_section(curr_data->config_section, "IdleTimeout", "180");
	dmuci_set_value_by_section(curr_data->config_section, "SSHKeepAlive", "300");
	dmuci_set_value_by_section(curr_data->config_section, "RootLogin", "0");
	dmuci_set_value_by_section(curr_data->config_section, "PasswordAuth", "0");
	dmuci_set_value_by_section(curr_data->config_section, "RootPasswordAuth", "0");
	dmuci_set_value_by_section(curr_data->config_section, "MaxAuthTries", "3");

	return 0;
}

static int delObjSSHServer(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	close_active_sessions(((struct dm_data *)data)->config_section);
	dmuci_delete_by_section(((struct dm_data *)data)->config_section, NULL, NULL);
	dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);
	return 0;
}

static int addObjSSHKey(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	/* only dmmap sections are present for AuthorizedKey, which will be added by bbfdm core automatically*/

	return 0;
}

static int delObjSSHKey(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	char *value = NULL;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "Key", &value);
	remove_pubkey(value);
	dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);
	return 0;
}

/*************************************************************
 * ENTRY METHODS
 *************************************************************/
static int browseSSHServerInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct dm_data curr_data = {0};
	char *inst = NULL;
	json_object *ssh_dump = NULL;

	/* to take advantage to bbfdm cache mechanism, we do a single ubus call that dumps all sessions */
	dmubus_call("sshmngr", "dump", UBUS_ARGS{0}, 0, &ssh_dump);

	uci_path_foreach_sections(bbfdm, "SSH", "Server", curr_data.dmmap_section) {
		char *config_sec_name = NULL;

		dmuci_get_value_by_section_string(curr_data.dmmap_section, "__section_name__", &config_sec_name);
		curr_data.config_section = get_config_section_from_dmmap_section_name(config_sec_name);

		if (curr_data.config_section) {
			bool b;
			char *enable = dmuci_get_value_by_section_fallback_def(curr_data.config_section, "enable", "0");

			string_to_bool(enable, &b);
			if (b) {
				char *act_date = dmuci_get_value_by_section_fallback_def(curr_data.dmmap_section, "ActivationDate", "");
				if (DM_STRLEN(act_date) == 0) {
					char *tm = NULL;
					dm_time_format(time(NULL), &tm);
					dmuci_set_value_by_section(curr_data.dmmap_section, "ActivationDate", tm);
				}

				if (ssh_dump) {
					json_object_object_foreach(ssh_dump, key, val) {
						if (DM_LSTRCMP(key, section_name(curr_data.config_section)) == 0) {
							curr_data.json_object = val;
						}
					}
				}
			}

			curr_data.additional_data = NULL;
		}

		inst = uci_handle_instance(dmctx, parent_node, &curr_data);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}

	return 0;
}

static int browseSSHServerSessionInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	char *inst = NULL;
	int id = 0;
	int idx = 0;
	json_object *json_session_list = NULL, *session_obj = NULL, *arr_obj = NULL;
	struct dm_data curr_data = {0};
	bool b;

	struct uci_section *s = ((struct dm_data *)prev_data)->config_section;
	char *value = dmuci_get_value_by_section_fallback_def(s, "enable", "0");
	string_to_bool(value, &b);
	if (!b) {
		return 0;
	}

	json_session_list = ((struct dm_data *)prev_data)->json_object;

	if (json_session_list == NULL) {
		return 0;
	}

	dmjson_foreach_obj_in_array(json_session_list, arr_obj, session_obj, idx, 1, "sessions") {
		curr_data.json_object = session_obj;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}

	return 0;
}

/*************************************************************
* GET & SET PARAM
**************************************************************/
static int get_ssh_server_num(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseSSHServerInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_ssh_key_num(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, NULL);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_ssh_status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("Enabled");
	return 0;
}

static int get_ssh_server_session_num(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseSSHServerSessionInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_ssh_server_enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enable", "0");
	return 0;
}

static int set_ssh_server_enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			char *cur = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enable", "0");
			bool cur_val;
			string_to_bool(cur, &cur_val);

			if (b == cur_val)
				break;

			if (b) {
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "enable", "1");
				char *tm = NULL;
				dm_time_format(time(NULL), &tm);
				dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "ActivationDate", tm);
			} else {
				/* should be called before setting enable to 0, because enable is checked in this function */
				close_active_sessions(((struct dm_data *)data)->config_section);
				dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "enable", "0");
				dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "ActivationDate", "");
			}
			break;
	}
	return 0;
}

static int get_ssh_server_alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "Alias", instance, value);
}

static int set_ssh_server_alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "Alias", instance, value);
}

static int get_ssh_server_interface(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *linker = NULL;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "Interface", &linker);
	_bbfdm_get_references(ctx, "Device.IP.Interface.", "Name", linker, value);
	return 0;
}

static int set_ssh_server_interface(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_objects[] = {"Device.IP.Interface.", NULL};
	struct dm_reference reference = {0};

	bbfdm_get_reference_linker(ctx, value, &reference);

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, reference.path, -1, 256, NULL, NULL))
				return FAULT_9007;

			if (dm_validate_allowed_objects(ctx, &reference, allowed_objects))
				return FAULT_9007;

			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "Interface", reference.value);
			break;
	}
	return 0;
}

static int get_ssh_server_port(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "Port", "22");
	return 0;
}

static int set_ssh_server_port(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"1","65535"}},1))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "Port", value);
			break;
	}
	return 0;
}

static int get_ssh_server_idle(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "IdleTimeout", "0");
	return 0;
}

static int set_ssh_server_idle(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}},1))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "IdleTimeout", value);
			break;
	}
	return 0;
}

static int get_ssh_server_keepalive(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "SSHKeepAlive", "300");
	return 0;
}

static int set_ssh_server_keepalive(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}},1))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "SSHKeepAlive", value);
			break;
	}
	return 0;
}

static int get_ssh_server_rootlogin(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "RootLogin", "1");
	return 0;
}

static int set_ssh_server_rootlogin(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "RootLogin", b ? "1" : "0");
			break;
	}
	return 0;
}

static int get_ssh_server_passwordlogin(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "PasswordAuth", "1");
	return 0;
}

static int set_ssh_server_passwordlogin(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "PasswordAuth", b ? "1" : "0");
			break;
	}
	return 0;
}

static int get_ssh_server_rootpasswordlogin(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "RootPasswordAuth", "1");
	return 0;
}

static int set_ssh_server_rootpasswordlogin(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "RootPasswordAuth", b ? "1" : "0");
			break;
	}
	return 0;
}

static int get_ssh_server_maxauthtries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "MaxAuthTries", "3");
	return 0;
}

static int set_ssh_server_maxauthtries(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}},1))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "MaxAuthTries", value);
			break;
	}
	return 0;
}

static int get_ssh_server_activationdate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "ActivationDate", value);
	return 0;
}

static int get_ssh_server_pid(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	bool b;
	json_object *res = NULL;
	struct uci_section *s = ((struct dm_data *)data)->config_section;
	char *en = dmuci_get_value_by_section_fallback_def(s, "enable", "0");

	*value = dmstrdup("0");
	string_to_bool(en, &b);
	if (!b)
		return 0;

	char *sec_name = section_name(s);
	if (DM_STRLEN(sec_name) == 0)
		return 0;

	res = ((struct dm_data *)data)->json_object;
        if (res == NULL) {
                return 0;
        }

	*value = dmjson_get_value(res, 1, "pid");
	return 0;
}

static int get_ssh_server_session_ip(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ip");
	return 0;
}

static int get_ssh_server_session_port(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "port");
	return 0;
}

static int get_ssh_key_alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "Alias", instance, value);
}

static int set_ssh_key_alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "Alias", instance, value);
}

static int get_ssh_key_pubkey(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "Key", value);
	return 0;
}

static int set_ssh_key_pubkey(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *cur_val = NULL;

	switch (action)	{
		case VALUECHECK:
			if (!DM_STRLEN(value))
				break;

			if (bbfdm_validate_string(ctx, value, -1, -1, NULL, NULL))
				return FAULT_9007;

			/* check if same as current key value */
			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "Key", &cur_val);
			if (DM_STRCMP(cur_val, value) == 0)
				break;

			if (key_exists_in_dmmap(value))
				return FAULT_9001;

			break;
		case VALUESET:
			/* check if same as current key value then nothing to do */
			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "Key", &cur_val);

			if (DM_STRCMP(cur_val, value) == 0)
				break;

			/* empty value received, since backend just returns if we set empty value, we have to explicitly remove it */
			if (!DM_STRLEN(value))
				remove_pubkey(cur_val);
			else
				add_pubkey(cur_val, value);

			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "Key", value);
			break;
	}
	return 0;
}

static int operate_session_delete(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *pid = NULL;

	pid = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "pid");

	if (DM_STRLEN(pid) != 0) {
		json_object *kill_res = NULL;

		// kill the session
		dmubus_call("sshmngr", "kill_session", UBUS_ARGS{{"session_pid", pid, String}}, 1, &kill_res);
	}

	return 0;
}

/*************************************************************
* Init & Clean Module
**************************************************************/
int init_ssh_module(void *data)
{
	struct dmctx bbf_ctx = {0};

	bbfdm_create_empty_file("/etc/bbfdm/dmmap/SSH");

	bbf_ctx_init(&bbf_ctx, NULL);

	dmmap_synchronize_SSH(&bbf_ctx);

	dmuci_commit_package_bbfdm("SSH");

	bbf_ctx_clean(&bbf_ctx);

	return 0;
}

/**********************************************************************************************************************************
*                                            OBJ & LEAF DEFINITION
***********************************************************************************************************************************/
/* *** Device.SSH. *** */
DMOBJ tDeviceSSHObj[] = {
{"SSH", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tSSHObj, tSSHParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMOBJ tSSHObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Server", &DMWRITE, addObjSSHServer, delObjSSHServer, NULL, browseSSHServerInst, NULL, NULL, tSSHServerObj, tSSHServerParams, NULL, BBFDM_BOTH, NULL},
{"AuthorizedKey", &DMWRITE, addObjSSHKey, delObjSSHKey, NULL, generic_browse, NULL, NULL, NULL, tSSHKeyParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMOBJ tSSHServerObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Session", &DMREAD, NULL, NULL, NULL, browseSSHServerSessionInst, NULL, NULL, NULL, tSSHServerSessionParams, NULL, BBFDM_BOTH, NULL},
{0}
};

/* *** Device.SSH. *** */
DMLEAF tSSHParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"ServerNumberOfEntries", &DMREAD, DMT_UNINT, get_ssh_server_num, NULL, BBFDM_BOTH},
{"AuthorizedKeyNumberOfEntries", &DMREAD, DMT_UNINT, get_ssh_key_num, NULL, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_ssh_status, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.SSH.Server. *** */
DMLEAF tSSHServerParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_ssh_server_enable, set_ssh_server_enable, BBFDM_BOTH},
{"Alias", &DMWRITE, DMT_STRING, get_ssh_server_alias, set_ssh_server_alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Interface", &DMWRITE, DMT_STRING, get_ssh_server_interface, set_ssh_server_interface, BBFDM_BOTH, DM_FLAG_REFERENCE},
{"Port", &DMWRITE, DMT_UNINT, get_ssh_server_port, set_ssh_server_port, BBFDM_BOTH},
{"IdleTimeout", &DMWRITE, DMT_UNINT, get_ssh_server_idle, set_ssh_server_idle, BBFDM_BOTH},
{"KeepAlive", &DMWRITE, DMT_UNINT, get_ssh_server_keepalive, set_ssh_server_keepalive, BBFDM_BOTH},
{"AllowRootLogin", &DMWRITE, DMT_BOOL, get_ssh_server_rootlogin, set_ssh_server_rootlogin, BBFDM_BOTH},
{"AllowPasswordLogin", &DMWRITE, DMT_BOOL, get_ssh_server_passwordlogin, set_ssh_server_passwordlogin, BBFDM_BOTH},
{"AllowRootPasswordLogin", &DMWRITE, DMT_BOOL, get_ssh_server_rootpasswordlogin, set_ssh_server_rootpasswordlogin, BBFDM_BOTH},
{"MaxAuthTries", &DMWRITE, DMT_UNINT, get_ssh_server_maxauthtries, set_ssh_server_maxauthtries, BBFDM_BOTH},
{"ActivationDate", &DMREAD, DMT_TIME, get_ssh_server_activationdate, NULL, BBFDM_BOTH},
{"PID", &DMREAD, DMT_UNINT, get_ssh_server_pid, NULL, BBFDM_BOTH},
{"SessionNumberOfEntries", &DMREAD, DMT_UNINT, get_ssh_server_session_num, NULL, BBFDM_BOTH},
{0}
};

DMLEAF tSSHServerSessionParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"IPAddress", &DMREAD, DMT_STRING, get_ssh_server_session_ip, NULL, BBFDM_BOTH},
{"Port", &DMREAD, DMT_UNINT, get_ssh_server_session_port, NULL, BBFDM_BOTH},
{"Delete()", &DMSYNC, DMT_COMMAND, NULL, operate_session_delete, BBFDM_USP},
{0}
};

DMLEAF tSSHKeyParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Alias", &DMWRITE, DMT_STRING, get_ssh_key_alias, set_ssh_key_alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Key", &DMWRITE, DMT_STRING, get_ssh_key_pubkey, set_ssh_key_pubkey, BBFDM_BOTH},
{0}
};

/* ********** DynamicObj ********** */
DM_MAP_OBJ tDynamicObj[] = {
/* parentobj, nextobject, parameter */
{"Device.", tDeviceSSHObj, NULL, init_ssh_module, NULL},
{0}
};
