/*
 * main.c - for periodicstats everything starts here.
 *
 * Copyright (C) 2022 iopsys Software Solutions AB. All rights reserved.
 *
 * Author: Shubham Sharma <shubham.sharma@iopsys.eu>
 * Author: Suvendhu Hansa <suvendhu.hansa@iopsys.eu>
 *
 * See LICENSE file for license related information.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <libbbfdm-ubus/bbfdm-ubus.h>

#include "periodicstats.h"

// Global periodic stats structure
extern DM_MAP_OBJ tDynamicObj[];

struct bbfdm_context bbfdm_ctx = {0};
static struct ubus_event_handler ev = {0};
static LIST_HEAD(ss_list);

void periodicstats_force_sample(const char *sample_set_name)
{
	smpl_set *sample_set = NULL;

	if (sample_set_name == NULL)
		return;

	if (strlen(sample_set_name) == 0)
		return;

	list_for_each_entry(sample_set, &ss_list, list) {
		if (strcmp(sample_set->name, sample_set_name) != 0)
			continue;

		if (sample_set->enable != 1)
			continue;

		if (list_empty(&sample_set->params))
			break;

		sample_set->ftimer.cb = pstats_force_sampling_cb;
		uloop_timeout_cancel(&sample_set->ftimer); // if one is already scheduled cancel and reschedule
		uloop_timeout_set(&sample_set->ftimer, 100);
	}

	return;
}

static void config_reload_cb(struct ubus_context *ctx __attribute__((unused)),
			struct ubus_event_handler *ev __attribute__((unused)),
			const char *type __attribute__((unused)),
			struct blob_attr *msg __attribute__((unused)))
{
	smpl_set *sample_set = NULL;

	BBF_INFO("Reloading upon receiving periodicstats.reload event\n");
	list_for_each_entry(sample_set, &ss_list, list) {
		// If any timer is alive, cancel it
		uloop_timeout_cancel(&sample_set->ftimer);
		uloop_timeout_cancel(&sample_set->utimer);
	}

	pstats_global_free(&ss_list);
	pstats_global_init(&ss_list);
	pstats_get_stats(&ss_list);
}

static int register_config_change(struct ubus_context *uctx)
{
	int ret;

	memset(&ev, 0, sizeof(struct ubus_event_handler));
	ev.cb = config_reload_cb;

	ret = ubus_register_event_handler(uctx, &ev, "periodicstats.reload");
	if (ret)
		return -1;

	return 0;
}

static void unregister_config_change(struct ubus_context *uctx)
{
	// unregister ubus handler if it exists
	if (uctx) {
		ubus_unregister_event_handler(uctx, &ev);
	}
}

/**
 *  This function prints the usage of the daemon
 *  @param prog input pointer to char, for program name
 */
static void pstat_usage(char *prog)
{
	fprintf(stderr, "Usage: %s [options]\n", prog);
	fprintf(stderr, "\n");
	fprintf(stderr, "options:\n");
	fprintf(stderr, "    -l <log level>  As per syslog, 0 to 7\n");
	fprintf(stderr, "    -d <schema dm>  Display the schema data model supported by micro-service\n");
	fprintf(stderr, "    -h <help>       To print this help menu\n");
	fprintf(stderr, "\n");
}

/**
 *  Main function for periodicstats, everything starts here
 *  @param argc input number of input arguments
 *  @param argv input double pointer array of optional command line arguments
 *  retrun integer value 0 on success and -1 on failure
 */
int main(int argc, char **argv)
{
	int ret = 0;
	int ch, dm_type = 0;
	int log_level = 3;

	while ((ch = getopt(argc, argv, "hdl:")) != -1) {
		switch (ch) {
		case 'l':
			log_level = (int)strtoul(optarg, NULL, 10);
			if (log_level < 0 || log_level > 7)
				log_level = 7;
			break;
		case 'd':
			dm_type++;
			break;
		case 'h':
			pstat_usage(argv[0]);
			exit(0);
		default:
			break;
		}
	}
	/* Logging to syslog */
	openlog(argv[0], LOG_PID|LOG_CONS|LOG_NDELAY, LOG_LOCAL1);

	memset(&bbfdm_ctx, 0, sizeof(struct bbfdm_context));
	bbfdm_ubus_set_service_name(&bbfdm_ctx, "periodicstats");
	bbfdm_ubus_set_log_level(log_level);
	bbfdm_ubus_load_data_model(tDynamicObj);

	if (dm_type > 0) {
		int res = bbfdm_print_data_model_schema(&bbfdm_ctx, dm_type);
		exit(res);
	}

	if (bbfdm_ubus_register_init(&bbfdm_ctx))
		goto out;

	if (register_config_change(bbfdm_ctx.ubus_ctx) != 0)
		goto out;

	// Initialize global periodic stats structure
	ret = pstats_global_init(&ss_list);
	if (ret) {
		BBF_ERR("Failed to initialize global periodic stats");
		goto out;
	}

	// stats collection invoked from here
	pstats_get_stats(&ss_list);

	/* Main loop of periodicstats */
	uloop_run();

out:
	unregister_config_change(bbfdm_ctx.ubus_ctx);
	bbfdm_ubus_register_free(&bbfdm_ctx);
 	pstats_global_free(&ss_list);
	closelog();

	return 0;
}
