/*
 * config.c - deals with uci configuration readup and maintain locally
 *
 * Copyright (C) 2022 iopsys Software Solutions AB. All rights reserved.
 *
 * Author: Shubham Sharma <shubham.sharma@iopsys.eu>
 *
 * See LICENSE file for license related information.
 *
 */

#include <stdio.h>
#include <stdlib.h>

#include "periodicstats.h"

/**
 *  pstats_add_parameter function to add local param to global list
 *  @param name input parameter pointer to char string containing parameter name
 *  @param local input parameter as param structure containing parameter data
 *  @param parameter output pointer to list_head parameter, specific to sampleset
 */
static void pstat_add_parameter(const char *name, struct list_head *parameter, param local)
{
	param *node = (param *)calloc(1, sizeof(param));
	if (!node) {
		BBF_ERR("Out of memory!");
		return;
	}

	strncpy(node->name, name, MAX_NAME_STR);

	node->enable = local.enable;
	node->calculation_mode = local.calculation_mode;
	strncpy(node->reference, local.reference, MAX_REF_STR);
	node->low_threshold = local.low_threshold;
	node->high_threshold = local.high_threshold;

	INIT_LIST_HEAD(&node->list);
	list_add_tail(&node->list, parameter);
}

/**
 *  pstats_add_sampleset function to add local SampleSet node to global list
 *  @param name input parameter pointer to char string containing uci option for sampleset
 *  @param ss input parameter smpl_set structure containing local uci sampleset data
 *  @param ss_list output pointer to list_head structure for global sampleset list
 */
static void pstats_add_sampleset(const char *name, smpl_set ss, struct list_head *ss_list)
{
	smpl_set *node = (smpl_set *)calloc(1, sizeof(smpl_set));
	if (!node) {
		BBF_ERR("Out of memory!");
		return;
	}

	strncpy(node->name, name, MAX_NAME_STR);

	node->enable = ss.enable;
	node->rprt_smpls = ss.rprt_smpls;
	node->smpl_intrvl = ss.smpl_intrvl;
	node->fetch_samples = ss.fetch_samples;
	node->time_ref = ss.time_ref;
	INIT_LIST_HEAD(&node->params);

	INIT_LIST_HEAD(&node->list);
	list_add_tail(&node->list, ss_list);
}

/**
 *  pstats_fill_local_params function to fill parameter info to local param structure
 *  @param uci_ctx input parameter pointer to uci_context structure
 *  @param uci_sec input parameter pointer to uci_section structure
 *  @param local output parameter pointer to param structure to be filled
 */
static void pstats_fill_local_params(struct uci_context *uci_ctx, struct uci_section *uci_sec, param *local)
{
	struct uci_option *opn;

	local->enable = 0;

	opn = uci_lookup_option(uci_ctx, uci_sec, "enable");
	if (opn)
		string_to_bool(opn->v.string, &local->enable);

	opn = uci_lookup_option(uci_ctx, uci_sec, "reference");
	if (opn) {
		strncpy(local->reference, opn->v.string, MAX_REF_STR);
	} else {
		local->reference[0] = '\0';
	}

	opn = uci_lookup_option(uci_ctx, uci_sec, "calculation_mode");
	if (opn == NULL) {
		local->calculation_mode = CAL_MODE_LATEST;
	} else if (strcmp(opn->v.string, "Average") == 0) {
		local->calculation_mode = CAL_MODE_AVERAGE;
	} else if (strcmp(opn->v.string, "Maximum") == 0) {
		local->calculation_mode = CAL_MODE_MAXIMUM;
	} else if (strcmp(opn->v.string, "Minimum") == 0) {
		local->calculation_mode = CAL_MODE_MINIMUM;
	} else {
		local->calculation_mode = CAL_MODE_LATEST;
	}

	opn = uci_lookup_option(uci_ctx, uci_sec, "low_threshold");
	local->low_threshold = opn ? (int)strtol(opn->v.string, NULL, 10) : 0;

	opn = uci_lookup_option(uci_ctx, uci_sec, "high_threshold");
	local->high_threshold = opn ? (int)strtol(opn->v.string, NULL, 10) : 0;
}

/**
 *  pstats_get_parameters function to add local SampleSet node to global list
 *  @param sampleset input parameter pointer to char string containing sampleset name
 *  @param parameter input parameter pointer to list_head from specific sampleset
 *  retrun integer value 0 on success and -1 on failure
 */
static int pstats_get_parameters(const char *sampleset, struct list_head *parameter)
{
	int ret = 0;
	struct uci_package *uci_pkg = NULL;
	struct uci_element *uci_elmnt = NULL;

	struct uci_context *uci_ctx = uci_alloc_context();

	uci_load(uci_ctx, "periodicstats", &uci_pkg);
	if (!uci_pkg) {
		BBF_ERR("Failed to load configuration\n");
		ret = -1;
		goto done;
	}

	uci_foreach_element(&uci_pkg->sections, uci_elmnt) {
		struct uci_section *uci_sec = uci_to_section(uci_elmnt);
		if (uci_sec && !strcmp(uci_sec->type, "parameter")) {
			struct uci_option *opn = uci_lookup_option(uci_ctx, uci_sec, "sample_set");
			if (opn && !strcmp(opn->v.string, sampleset)) {
				param local;

				// Get related Parameters to local param structure
				pstats_fill_local_params(uci_ctx, uci_sec, &local);
				pstat_add_parameter(uci_sec->e.name, parameter, local);

				// Reinitialize the parameters
				uci_set_option_value(uci_sec->e.name, "values", "");
				uci_set_option_value(uci_sec->e.name, "failures", "");
				uci_set_option_value(uci_sec->e.name, "sample_secs", "");
				uci_set_option_value(uci_sec->e.name, "suspect_data", "");
			}
		}
	}

	uci_unload(uci_ctx, uci_pkg);
done:
	uci_free_context(uci_ctx);
	return ret;
}

/**
 *  pstats_convert_time function to convert string UTC reference time to time_t usable time
 *  @param option input pointer to char string containing uci option value
 *  retrun time_t value, non-zero on success and 0 on failure
 */
static time_t pstats_convert_time(const char *time_reference)
{
	time_t cvt_time = 0;
	time_t local_time = 0;
	struct tm ref_time;
	struct tm *local_date;
	int year = 0, month = 0, day = 0, hour = 0, min = 0, sec = 0;

	// Leave date
	time(&local_time);
	local_date = localtime(&local_time);

	memset(&ref_time, 0, sizeof(struct tm));
	ref_time.tm_year = local_date->tm_year;
	ref_time.tm_mon = local_date->tm_mon;
	ref_time.tm_mday = local_date->tm_mday;
	ref_time.tm_isdst = -1;

	if (time_reference && sscanf(time_reference, "%4d-%2d-%2dT%2d:%2d:%2dZ", &year, &month, &day, &hour, &min, &sec) == 6) {
		ref_time.tm_hour = hour;
		ref_time.tm_min = min;
		ref_time.tm_sec = sec;
	}

	cvt_time = mktime(&ref_time);

	return cvt_time;
}

/**
 *  pstats_fill_local_sampleset function to fill SampleSet info to local smpl_set structure
 *  @param option input parameter pointer to char string containing uci option for sampleset
 *  @param value input parameter pointer to char string containing uci option value
 *  @param ss output parameter pointer to smpl_set structure to be filled
 */
static void pstats_fill_local_sampleset(const char *option, const char *value, smpl_set *ss)
{
	if (!strcmp(option, "enable")) {
		string_to_bool(value, &ss->enable);
	} else if (!strcmp(option, "report_sample")) {
		ss->rprt_smpls = (uint32_t)strtol(value, NULL, 10);
	} else if (!strcmp(option, "sample_interval")) {
		ss->smpl_intrvl = (uint32_t)strtol(value, NULL, 10);
	} else if (!strcmp(option, "time_reference")) {
		// Convert to time_t
		ss->time_ref = pstats_convert_time(value);
	} else if (!strcmp(option, "fetch_samples")) {
		ss->fetch_samples = (uint32_t)strtol(value, NULL, 10);
	}
}

/**
 *  pstats_get_parameters_list function to get all parameters per sampleset and initialize to global list
 *  @param ss_list output pointer to list_head structure for global sampleset list
 *  retrun integer value 0 on success and -1 on failure
 */
static int pstats_get_parameters_list(struct list_head *ss_list, bool is_param_sampleset)
{
	int ret = 0;

	struct uci_package *uci_pkg = NULL;
	struct uci_element *uci_elmnt = NULL;
	struct uci_context *uci_ctx = uci_alloc_context();

	uci_load(uci_ctx, "periodicstats", &uci_pkg);
	if (!uci_pkg) {
		BBF_ERR("Failed to load configuration\n");
		ret = -1;
		goto done;
	}

	uci_foreach_element(&uci_pkg->sections, uci_elmnt) {
		struct uci_section *uci_sec = uci_to_section(uci_elmnt);
		if (uci_sec && !strcmp(uci_sec->type, "sampleset")) {
			if (is_param_sampleset) {
				// Add all corresponding parameters to this sample set
				smpl_set *pos = NULL;
				list_for_each_entry(pos, ss_list, list) {
					if (!strcmp(pos->name, uci_sec->e.name)) {
						// Actual parameter collection done here
						pstats_get_parameters(uci_sec->e.name, &pos->params);
					}
				}
			} else {
				// Fill all the parameters of sampleset structure
				smpl_set local = {0};
				struct uci_element *e = NULL;

				uci_foreach_element(&uci_sec->options, e) {
					struct uci_option *uci_opn = uci_to_option(e);
					if (uci_opn) {
						// Get all SampleSets locally
						pstats_fill_local_sampleset(uci_opn->e.name, uci_opn->v.string, &local);
					}
				}

				// Reset sample set params
				uci_set_option_value(uci_sec->e.name, "report_starttime", "");
				uci_set_option_value(uci_sec->e.name, "report_endtime", "");
				uci_set_option_value(uci_sec->e.name, "status", "");
				uci_set_option_value(uci_sec->e.name, "sample_secs", "");

				// Add local SampleSet to global list
				pstats_add_sampleset(uci_sec->e.name, local, ss_list);
			}
		}
	}

	uci_unload(uci_ctx, uci_pkg);
done:
	uci_free_context(uci_ctx);
	return ret;
}

/**
 *  pstats_global_init function for initialization of global periodicstats
 *  @param ss_list output pointer to list_head structure for global sampleset list
 *  retrun integer value 0 on success and -1 on failure
 */
int pstats_global_init(struct list_head *ss_list)
{
	int ret = 0;

	// Get no. of SampleSet and initialize to global list
	ret = pstats_get_parameters_list(ss_list, 0);
	if (ret)
		return ret;

	// Get all Parameters per SampleSets and initialize to global list
	ret = pstats_get_parameters_list(ss_list, 1);

	return ret;
}

/**
 *  pstats_global_free function for freed of global periodicstats
 *  @param ss_list output pointer to list_head structure for global sampleset list
 */
void pstats_global_free(struct list_head *ss_list)
{
	smpl_set *node = NULL;
	smpl_set *_tmp = NULL;

	list_for_each_entry_safe(node, _tmp, ss_list, list) {
		param *node_param = NULL;
		param *__tmp = NULL;

		list_for_each_entry_safe(node_param, __tmp, &node->params, list) {
			int i;

			for (i = 0; i < INTERNAL_SAMPLE_RATE; i++) {
				FREE(node_param->values[i]);
			}

			list_del(&node_param->list);
			FREE(node_param);
		}

		list_del(&node->list);
		FREE(node);
	}
}

int pstats_get_sample_set_instance(const char *sample_set)
{
	int index = 0;

	if (sample_set == NULL)
		return index;

	if (strlen(sample_set) == 0)
		return index;

	/*
	 * dmmap format (USP/TR-181):
	 * - /etc/bbfdm/dmmap/PeriodicStatistics (package "PeriodicStatistics")
	 * - section type: "SampleSet"
	 * - option "__section_name__": "periodicstats.<uci_sampleset>"
	 * - option "__instance__": "<tr181_instance>"
	 *
	 * The Push! event uses the TR-181 instance index.
	 */

	struct uci_package *uci_pkg = NULL;
	struct uci_element *uci_elmnt = NULL;
	struct uci_context *uci_ctx = uci_alloc_context();

	if (uci_ctx == NULL)
		return index;

	uci_set_confdir(uci_ctx, "/etc/bbfdm/dmmap");

	/* Package "PeriodicStatistics" */
	uci_load(uci_ctx, "PeriodicStatistics", &uci_pkg);
	if (!uci_pkg) {
		uci_free_context(uci_ctx);
		return index;
	}

	char expected[256] = {0};
	snprintf(expected, sizeof(expected), "periodicstats.%s", sample_set);

	uci_foreach_element(&uci_pkg->sections, uci_elmnt) {
		struct uci_section *uci_sec = uci_to_section(uci_elmnt);
		if (uci_sec && !strcmp(uci_sec->type, "SampleSet")) {
			struct uci_element *e = NULL;
			const char *section_name = NULL;
			const char *instance = NULL;

			uci_foreach_element(&uci_sec->options, e) {
				struct uci_option *uci_opn = uci_to_option(e);
				if (!uci_opn || uci_opn->type != UCI_TYPE_STRING)
					continue;

				if (strcmp(uci_opn->e.name, "__section_name__") == 0) {
					section_name = uci_opn->v.string;
				} else if (strcmp(uci_opn->e.name, "__instance__") == 0) {
					instance = uci_opn->v.string;
				}
			}

			if (section_name && instance &&
			    (strcmp(section_name, expected) == 0 || strcmp(section_name, sample_set) == 0)) {
				index = (int)strtoul(instance, NULL, 10);
				break;
			}
		}
	}

	uci_unload(uci_ctx, uci_pkg);
	uci_free_context(uci_ctx);

	return index;
}
