/*
 * periodicstats.c - periodic stats method handeling code
 *
 * Copyright (C) 2022 iopsys Software Solutions AB. All rights reserved.
 *
 * Author: Shubham Sharma <shubham.sharma@iopsys.eu>
 *
 * See LICENSE file for license related information.
 *
 */

#include <time.h>
#include <libubus.h>

#include "periodicstats.h"
#include <libbbfdm-ubus/bbfdm-ubus.h>

#define UBUS_TIMEOUT 3000 // 3s

extern struct bbfdm_context bbfdm_ctx;

static void _ubus_handler_cb(struct ubus_request *req, int type __attribute__((unused)), struct blob_attr *msg)
{
	struct blob_buf *bb_out = (struct blob_buf *)req->priv;
	char *buffer = NULL;

	if (msg == NULL || bb_out == NULL)
		return;

	buffer = blobmsg_format_json(msg, true);
	if (buffer == NULL)
		return;

	blobmsg_add_json_from_string(bb_out, buffer);

	FREE(buffer);
}

static int bbfdm_ubus_call(const char *obj_name, const char *method, struct blob_buf *bb, struct blob_buf *bb_out)
{
	struct ubus_context *ubus_ctx = NULL;
	uint32_t id;
	int ret = 0;

	ubus_ctx = ubus_connect(NULL);
	if (ubus_ctx == NULL) {
		BBF_ERR("ubus connect failed");
		return -1;
	}

	ret = ubus_lookup_id(ubus_ctx, obj_name, &id);
	if (ret != 0) {
		ubus_free(ubus_ctx);
		BBF_ERR("ubus lookup failed");
		return -1;
	}

	ret = ubus_invoke(ubus_ctx, id, method, bb->head, _ubus_handler_cb, bb_out, UBUS_TIMEOUT);
	ubus_free(ubus_ctx);

	return ret;
}

static struct blob_attr *get_results_array(struct blob_attr *msg)
{
	struct blob_attr *tb[1] = {0};
	const struct blobmsg_policy p[1] = {
			{ "results", BLOBMSG_TYPE_ARRAY }
	};

	if (msg == NULL)
		return NULL;

	blobmsg_parse(p, 1, tb, blobmsg_data(msg), blobmsg_len(msg));

	return tb[0];
}

static void bbfdm_get_parameter_value(const char *path_name, char **value, char **type)
{
	struct blob_buf bb = {0};
	struct blob_buf bb_res = {0};

	if (value == NULL || type == NULL)
		return;

	memset(&bb, 0, sizeof(struct blob_buf));
	blob_buf_init(&bb, 0);

	memset(&bb_res, 0, sizeof(struct blob_buf));
	blob_buf_init(&bb_res, 0);

	blobmsg_add_string(&bb, "path", path_name);
	void *table = blobmsg_open_table(&bb, "optional");
	blobmsg_add_string(&bb, "format", "raw");
	blobmsg_close_table(&bb, table);

	int err = bbfdm_ubus_call("bbfdm", "get", &bb, &bb_res);

	blob_buf_free(&bb);

	if (err) {
		BBF_ERR("Could not read stats");
		blob_buf_free(&bb_res);
		return;
	}

	struct blob_attr *parameters = get_results_array(bb_res.head);

	if (parameters == NULL || blobmsg_len(parameters) == 0) {
		BBF_ERR("Could not read stats");
		blob_buf_free(&bb_res);
		return;
	}

	struct blob_attr *cur = NULL;
	int rem = 0;

	const struct blobmsg_policy p[2] = {
			{ "data", BLOBMSG_TYPE_STRING },
			{ "type", BLOBMSG_TYPE_STRING }
	};

	blobmsg_for_each_attr(cur, parameters, rem) {
		struct blob_attr *tb[2] = {0};

		blobmsg_parse(p, 2, tb, blobmsg_data(cur), blobmsg_len(cur));

		*value = strdup(tb[0] ? blobmsg_get_string(tb[0]) : "");
		*type = strdup(tb[1] ? blobmsg_get_string(tb[1]) : "");
	}

	blob_buf_free(&bb_res);
}

static char *uci_get_param_option_value(const char *sampleset, const char *reference, const char *option)
{
	struct uci_context *uci_ctx = uci_alloc_context();
	struct uci_package *uci_pkg = NULL;
	struct uci_element *uci_elmnt = NULL;
	char *value = NULL;

	if (sampleset == NULL || reference == NULL || option == NULL)
		return value;

	uci_load(uci_ctx, "periodicstats", &uci_pkg);
	if (!uci_pkg) {
		BBF_ERR("Failed to load configuration");
		uci_free_context(uci_ctx);
		return value;
	}

	uci_foreach_element(&uci_pkg->sections, uci_elmnt) {
		struct uci_section *uci_sec = uci_to_section(uci_elmnt);
		if (uci_sec && !strcmp(uci_sec->type, "parameter")) {
			struct uci_option *samp_set = uci_lookup_option(uci_ctx, uci_sec, "sample_set");
			struct uci_option *ref = uci_lookup_option(uci_ctx, uci_sec, "reference");
			if (samp_set && ref && !strcmp(samp_set->v.string, sampleset) && !strcmp(ref->v.string, reference)) {
				struct uci_option *u_option = uci_lookup_option(uci_ctx, uci_sec, option);
				value = u_option ? strdup(u_option->v.string) : NULL;
				break;
			}
		}
	}

	uci_unload(uci_ctx, uci_pkg);
	uci_free_context(uci_ctx);

	return value;
}

static char *uci_get_smplset_option_value(const char *sampleset, const char *option)
{
	struct uci_context *uci_ctx = uci_alloc_context();
	struct uci_package *uci_pkg = NULL;
	struct uci_element *uci_elmnt = NULL;
	char *value = NULL;

	if (sampleset == NULL || option == NULL)
		return value;

	uci_load(uci_ctx, "periodicstats", &uci_pkg);
	if (!uci_pkg) {
		BBF_ERR("Failed to load configuration");
		uci_free_context(uci_ctx);
		return value;
	}

	uci_foreach_element(&uci_pkg->sections, uci_elmnt) {
		struct uci_section *uci_sec = uci_to_section(uci_elmnt);
		if (uci_sec && !strcmp(uci_sec->type, "sampleset")) {
			if (strcmp(uci_sec->e.name, sampleset) == 0) {
				struct uci_option *u_option = uci_lookup_option(uci_ctx, uci_sec, option);
				value = u_option ? strdup(u_option->v.string) : NULL;
				break;
			}
		}
	}

	uci_unload(uci_ctx, uci_pkg);
	uci_free_context(uci_ctx);

	return value;
}


void uci_set_option_value(const char *sec_name, const char *option, const char *value)
{
	struct uci_context *uci_ctx = uci_alloc_context();
	struct uci_ptr ptr = {0};
	char uci_path[256] = {0};

	snprintf(uci_path, sizeof(uci_path), "periodicstats.%s.%s=%s", sec_name, option, value);

	if (uci_lookup_ptr(uci_ctx, &ptr, uci_path, true) != UCI_OK) {
		uci_free_context(uci_ctx);
		return;
	}

	if (uci_set(uci_ctx, &ptr) != UCI_OK) {
		uci_free_context(uci_ctx);
		return;
	}

	if (uci_save(uci_ctx, ptr.p) != UCI_OK) {
		uci_free_context(uci_ctx);
		return;
	}

	uci_commit(uci_ctx, &ptr.p, false);
	uci_free_context(uci_ctx);
}

static time_t get_next_sample_time(time_t time_ref, uint32_t periodic_interval)
{
	unsigned int next_period = 0;
	time_t curr_time;

	curr_time = time(NULL);

	if (curr_time > time_ref)
		next_period = (periodic_interval - ((curr_time - time_ref) % periodic_interval));
	else
		next_period = (time_ref - curr_time) % periodic_interval;

	// if less than (INTERNAL_SAMPLE_RATE) seconds is pending then
	// consider next time ref, since based on internal sampling rate
	// the timeout value will be further reduced. So by doing this
	// it ensures each sampling will have timeout of 1 sec at least.
	if (next_period < INTERNAL_SAMPLE_RATE)
		next_period = periodic_interval + next_period;

	return (curr_time + next_period);
}

static unsigned int pstats_get_nb_sample_stored(const char *list)
{
	unsigned int count = 0;

	if (!list)
		return count;

	if (*list)
		count = 1;

	while (*list) {
		if (*list++ == ',')
			count++;
	}

	return count;
}

static bool data_type_numeric(const char *type)
{
	bool ret = false;

	if (type == NULL)
		return ret;

	if (strcmp(type, "xsd:unsignedInt") == 0 ||
	    strcmp(type, "xsd:int") == 0 ||
	    strcmp(type, "xsd:unsignedLong") == 0 ||
	    strcmp(type, "xsd:long") == 0) {
		ret = true;
	}

	return ret;
}

static void reset_sampling_params(param *sample_param)
{
	int i;

	if (sample_param == NULL)
		return;

	sample_param->param_numeric = false;
	for (i = 0; i < INTERNAL_SAMPLE_RATE; i++) {
		FREE(sample_param->values[i]);
		sample_param->sample_seconds[i] = 0;
		sample_param->forced = false;
	}
}

static char *measure_threshold_failure(smpl_set *sample_set, param *sample_param, const char *value)
{
	long sample_val = 0;

	if (sample_set == NULL || sample_param == NULL || value == NULL)
		return NULL;

	if (sample_param->param_numeric == false)
		return NULL;

	if (sample_param->low_threshold == sample_param->high_threshold)
		return NULL;

	sample_val = strtol(value, NULL, 10);

	if (sample_val > sample_param->low_threshold && sample_val < sample_param->high_threshold)
		return NULL;

	char *failures = uci_get_param_option_value(sample_set->name, sample_param->reference, "failures");
	if (failures) {
		char failed[64] = {0};

		snprintf(failed, sizeof(failed), "%u", (unsigned int)strtoul(failures, NULL, 10) + 1);

		FREE(failures);
		return strdup(failed);
	}

	return strdup("1");
}

static char *measure_param_suspect_data(smpl_set *sample_set, param *sample_param)
{
	if (sample_set == NULL || sample_param == NULL)
		return NULL;

	char *sample_list = uci_get_param_option_value(sample_set->name, sample_param->reference, "suspect_data");

	if (sample_list && strlen(sample_list) != 0) {
		int buf_size = 0;
		unsigned int samples_stored = pstats_get_nb_sample_stored(sample_list);

		if (sample_param->forced == true) {
			if (samples_stored <= sample_set->rprt_smpls) {
				char *tmp = strrchr(sample_list, ',');
				if (tmp != NULL) {
					*tmp = '\0';
					buf_size = strlen(sample_list) + 3;
				}
			}
		} else {
			if (samples_stored < sample_set->rprt_smpls) {
				buf_size = strlen(sample_list) + 3;
			}
		}

		if (buf_size) {
			char value[buf_size];

			snprintf(value, buf_size, "%s,0", sample_list);
			FREE(sample_list);
			return strdup(value);
		}
	}

	FREE(sample_list);

	return strdup("0");
}

static char *measure_param_sample_secs(smpl_set *sample_set, param *sample_param, int diff_time)
{
	time_t total_time = 0;

	if (sample_set == NULL || sample_param == NULL)
		return NULL;

	if (diff_time >= 0) {
		// This is for forced sampling
		total_time = diff_time;
	} else {
		int i;

		for (i = 0; i < INTERNAL_SAMPLE_RATE; i++) {
			total_time += sample_param->sample_seconds[i];
		}
	}

	char *sample_list = uci_get_param_option_value(sample_set->name, sample_param->reference, "sample_secs");

	if (sample_list && strlen(sample_list) != 0) {
		unsigned int buf_size = 0;
		unsigned int samples_stored = pstats_get_nb_sample_stored(sample_list);

		if (sample_param->forced == true) {
			if (samples_stored <= sample_set->rprt_smpls) {
				char *tmp = strrchr(sample_list, ',');
				if (tmp != NULL) {
					*tmp = '\0';
					buf_size = strlen(sample_list) + 34;
				}
			}
		} else {
			if (samples_stored < sample_set->rprt_smpls) {
				buf_size = strlen(sample_list) + 34;
			}
		}

		if (buf_size) {
			char values[buf_size];

			snprintf(values, buf_size, "%s,%lu", sample_list, (unsigned long)total_time);
			FREE(sample_list);
			return strdup(values);
		}
	}

	FREE(sample_list);

	/* Either all report samples completed or the initial sample value */
	char values[34];
	snprintf(values, 34, "%lu", (unsigned long)total_time);

	return strdup(values);
}

static char *measure_parameter_values(smpl_set *sample_set, param *sample_param, const char *report_val)
{
	char *values = NULL;
	unsigned int buf_size = 0;

	if (sample_set == NULL || sample_param == NULL || report_val == NULL)
		return values;

	char *sample_list = uci_get_param_option_value(sample_set->name, sample_param->reference, "values");

	if (sample_list && strlen(sample_list) != 0) {
		unsigned int samples_stored = pstats_get_nb_sample_stored(sample_list);

		if (sample_param->forced == true) {
			if (samples_stored <= sample_set->rprt_smpls) {
				char *tmp = strrchr(sample_list, ',');
				if (tmp != NULL) {
					*tmp = '\0';
					buf_size = strlen(sample_list) + strlen(report_val) + 2;
				}
			}
		} else {
			if (samples_stored < sample_set->rprt_smpls) {
				buf_size = strlen(sample_list) + strlen(report_val) + 2;
			}
		}

		if (buf_size) {
			values = (char *)calloc(buf_size, sizeof(char));
			if (values != NULL) {
				snprintf(values, buf_size, "%s,%s", sample_list, report_val);
			}

			FREE(sample_list);
			return values;
		}

		FREE(sample_list);
	}

	/* Either all report samples completed or the initial sample value */
	buf_size = strlen(report_val) + 1;

	values = (char *)calloc(buf_size, sizeof(char));
	if (values != NULL) {
		snprintf(values, buf_size, "%s", report_val);
	}

	return values;
}

static void adjust_stats_sample_seconds(smpl_set *sample_set, int diff_time)
{
	time_t total_time = 0;

	if (sample_set == NULL)
		return;

	if (diff_time >= 0) {
		// This is for forced sampling
		total_time = diff_time;
	} else {
		int i;
		for (i = 0; i < INTERNAL_SAMPLE_RATE; i++) {
			total_time += sample_set->sample_seconds[i];
			sample_set->sample_seconds[i] = 0;
		}
	}

	char *sample_list = uci_get_smplset_option_value(sample_set->name, "sample_secs");

	if (sample_list && strlen(sample_list) != 0) {
		unsigned int buf_size = 0;
		unsigned int samples_stored = pstats_get_nb_sample_stored(sample_list);

		if (sample_set->forced == true) {
			if (samples_stored <= sample_set->rprt_smpls) {
				char *tmp = strrchr(sample_list, ',');
				if (tmp != NULL) {
					*tmp = '\0';
					buf_size = strlen(sample_list) + 34;
				}
			}
		} else {
			if (samples_stored < sample_set->rprt_smpls) {
				buf_size = strlen(sample_list) + 34;
			}
		}

		if (buf_size) {
			char values[buf_size];

			snprintf(values, buf_size, "%s,%lu", sample_list, (unsigned long)total_time);
			uci_set_option_value(sample_set->name, "sample_secs", values);
			FREE(sample_list);
			return;
		}
	}

	FREE(sample_list);

	/* Either all report samples completed or the initial sample value */
	char values[34];
	snprintf(values, 34, "%lu", (unsigned long)total_time);
	uci_set_option_value(sample_set->name, "sample_secs", values);

	return;
}

static void generate_report(smpl_set *sample_set)
{
	if (sample_set == NULL) {
		return;
	}
	
	if (sample_set->int_smpl_count != INTERNAL_SAMPLE_RATE - 1) {
		// All internal samples are not ready
		return;
	}

	param *sample_param = NULL;
	list_for_each_entry(sample_param, &sample_set->params, list) {
		int mode, i, avg_devider = 0;
		char *report_val = NULL;
		long total_val = 0, avg_val = 0;
		char str_val[64] = {0};

		if ((sample_param->enable != 1) || (sample_param->reference[0] == '\0'))
			continue;

		if (sample_param->param_numeric == false) {
			mode = CAL_MODE_LATEST;
		} else {
			mode = sample_param->calculation_mode;
		}

		switch (mode) {
			case CAL_MODE_AVERAGE:
				for (i = 0; i < INTERNAL_SAMPLE_RATE; i++) {
					if (sample_param->values[i] == NULL) {
						continue;
					}
					avg_devider++;
					total_val = total_val + strtol(sample_param->values[i], NULL, 10);
				}

				if (avg_devider == 0) {
					break;
				}

				avg_val = total_val/avg_devider;
				snprintf(str_val, sizeof(str_val), "%ld", avg_val);
				report_val = str_val;
				break;
			case CAL_MODE_MINIMUM:
				for (i = 0; i < INTERNAL_SAMPLE_RATE; i++) {
					if (sample_param->values[i] == NULL) {
						continue;
					}

					if (report_val == NULL) {
						report_val = sample_param->values[i];
					} else {
						if (strtol(report_val, NULL, 10) > strtol(sample_param->values[i], NULL, 10)) {
							report_val = sample_param->values[i];
						}
					}
				}
				break;
			case CAL_MODE_MAXIMUM:
				for (i = 0; i < INTERNAL_SAMPLE_RATE; i++) {
					if (sample_param->values[i] == NULL) {
						continue;
					}

					if (report_val == NULL) {
						report_val = sample_param->values[i];
					} else {
						if (strtol(report_val, NULL, 10) < strtol(sample_param->values[i], NULL, 10)) {
							report_val = sample_param->values[i];
						}
					}
				}
				break;
			case CAL_MODE_LATEST:
				i = INTERNAL_SAMPLE_RATE - 1;
				report_val = sample_param->values[i];
				break;
		}

		if (report_val != NULL) {
			char *values = measure_parameter_values(sample_set, sample_param, report_val);
			if (values != NULL) {
				uci_set_option_value(sample_param->name, "values", values);
				FREE(values);
			}

			// measure the threshold for failure count
			char *failure_count = measure_threshold_failure(sample_set, sample_param, report_val);
			if (failure_count != NULL) {
				uci_set_option_value(sample_param->name, "failures", failure_count);
				FREE(failure_count);
			}

			// measure the sample second
			char *sample_sec = measure_param_sample_secs(sample_set, sample_param, -1);
			if (sample_sec != NULL) {
				uci_set_option_value(sample_param->name, "sample_secs", sample_sec);
				FREE(sample_sec);
			}

			// measure the suspect data
			char *suspect_data = measure_param_suspect_data(sample_set, sample_param);
			if (suspect_data != NULL) {
				uci_set_option_value(sample_param->name, "suspect_data", suspect_data);
				FREE(suspect_data);
			}
		}

		reset_sampling_params(sample_param);
	}

	return;
}

static unsigned long get_next_measurement_interval(smpl_set *sample_set)
{
	unsigned long next_internal_time = 0;

	if (!sample_set)
		return next_internal_time;

	if (sample_set->int_smpl_count != INTERNAL_SAMPLE_RATE) {
		time_t cur_time = time(NULL);
		if (sample_set->next_sample_time <= cur_time) {
			next_internal_time = 100; // start immediate
		} else {
			unsigned long time_remaining;
			time_remaining = (sample_set->next_sample_time - cur_time) * 1000;
			next_internal_time = time_remaining / (INTERNAL_SAMPLE_RATE - sample_set->int_smpl_count);
		}
	} else {
		// All internal sampling within a sample time completed
		sample_set->next_sample_time = get_next_sample_time(sample_set->time_ref, sample_set->smpl_intrvl);
		next_internal_time = ((sample_set->next_sample_time - time(NULL)) * 1000) / INTERNAL_SAMPLE_RATE;
	}

	return next_internal_time;
}

static void send_sample_set_push_event(smpl_set *sample_set)
{
	param *sample_param = NULL;
	struct blob_buf b;
	int i = 0;

	if (sample_set == NULL)
		return;

	int index = pstats_get_sample_set_instance(sample_set->name);
	if (index == 0)
		return;

	char event_path[64] = {0};
	snprintf(event_path, sizeof(event_path), "Device.PeriodicStatistics.SampleSet.%d.Push!", index);

	memset(&b, 0, sizeof(struct blob_buf));
	blob_buf_init(&b, 0);

	blobmsg_add_string(&b, "name", event_path);
	void *res = blobmsg_open_array(&b, "input");
	list_for_each_entry(sample_param, &sample_set->params, list) {
		char path[64] = {0};
		char *values = NULL, *failures = NULL, *secs = NULL;
		void *item;

		if ((sample_param->enable != 1) || (sample_param->reference[0] == '\0'))
			continue;

		values = uci_get_param_option_value(sample_set->name, sample_param->reference, "values");
		failures = uci_get_param_option_value(sample_set->name, sample_param->reference, "failures");
		secs = uci_get_param_option_value(sample_set->name, sample_param->reference, "sample_secs");

		item = blobmsg_open_table(&b, "");
		snprintf(path, sizeof(path), "Parameter.%d.Reference", i+1);
		blobmsg_add_string(&b, "path", path);
		blobmsg_add_string(&b, "data", sample_param->reference);
		blobmsg_add_string(&b, "type", "xsd:string");
		blobmsg_close_table(&b, item);

		item = blobmsg_open_table(&b, "");
		snprintf(path, sizeof(path), "Parameter.%d.Values", i+1);
		blobmsg_add_string(&b, "path", path);
		blobmsg_add_string(&b, "data", values ? values : "");
		blobmsg_add_string(&b, "type", "xsd:string");
		blobmsg_close_table(&b, item);

		item = blobmsg_open_table(&b, "");
		snprintf(path, sizeof(path), "Parameter.%d.SampleSeconds", i+1);
		blobmsg_add_string(&b, "path", path);
		blobmsg_add_string(&b, "data", secs ? secs : "");
		blobmsg_add_string(&b, "type", "xsd:string");
		blobmsg_close_table(&b, item);

		item = blobmsg_open_table(&b, "");
		snprintf(path, sizeof(path), "Parameter.%d.Failures", i+1);
		blobmsg_add_string(&b, "path", path);
		blobmsg_add_string(&b, "data", failures ? failures : "");
		blobmsg_add_string(&b, "type", "xsd:string");
		blobmsg_close_table(&b, item);

		FREE(values);
		FREE(secs);
		FREE(failures);
		i++;
	}
	blobmsg_close_array(&b, res);

	if (i == 0) {
		blob_buf_free(&b);
		return;
	}

	ubus_send_event(bbfdm_ctx.ubus_ctx, "bbfdm.event", b.head);
	blob_buf_free(&b);
	return;
}

static void adjust_reporting_times_status(smpl_set *sample_set)
{
	time_t now;
	char time_str[32] = {0};

	if (sample_set == NULL)
		return;

	// All internal sampling done i.e completion of a sample interval
	sample_set->rep_smpl_count++;

	now = time(NULL);
	strftime(time_str, sizeof(time_str), "%Y-%m-%dT%H:%M:%SZ", gmtime(&now));

	// One sample complete so update the report end time for the last sample
	uci_set_option_value(sample_set->name, "report_endtime", time_str);

	// change status value
	if (sample_set->fetch_samples >= 1 && sample_set->fetch_samples <= sample_set->rprt_smpls &&
	    sample_set->rep_smpl_count == sample_set->fetch_samples) {
		uci_set_option_value(sample_set->name, "status", "Trigger");
	} else {
		uci_set_option_value(sample_set->name, "status", "Enabled");
	}

	// send push event ## fetch_samples value could be 0 as per TR069 but not in case of USP
	// So when this parameter is set to 0 treat it as 1 for USP
	unsigned int push_sample = (sample_set->fetch_samples == 0) ? 1 : sample_set->fetch_samples;
	if (push_sample <= sample_set->rprt_smpls && push_sample == sample_set->rep_smpl_count) {
		send_sample_set_push_event(sample_set);
	}

	if (sample_set->rep_smpl_count == sample_set->rprt_smpls) {
		// This is the last sample collected for a statistics report
		// so reset report sample counter
		sample_set->rep_smpl_count = 0;
		// update the report start time for next reporting interval
		uci_set_option_value(sample_set->name, "report_starttime", time_str);
	}
}

static void prepare_next_sampling(smpl_set *sample_set)
{
	unsigned long next_internal_time;

	if (sample_set == NULL)
		return;

	// update the time elapsed to collect the data for current internal interval
	time_t end_time = time(NULL);
	sample_set->sample_seconds[sample_set->int_smpl_count] = end_time - sample_set->start_time;

	sample_set->int_smpl_count++;

	next_internal_time = get_next_measurement_interval(sample_set);
	if (next_internal_time == 0) {
		BBF_ERR("Internal interval time invalid");
		return;
	}

	if (sample_set->int_smpl_count == INTERNAL_SAMPLE_RATE) {
		/* All internal sampling complete, capture reporting times, status and
		   reset internal sample counter for next sampling period */
		adjust_reporting_times_status(sample_set);
		adjust_stats_sample_seconds(sample_set, -1);
		sample_set->int_smpl_count = 0;
		sample_set->forced = false;
	}

	BBF_INFO("Start new collection session of sample set '%s' in %lu msec\n", sample_set->name, next_internal_time);
	uloop_timeout_set(&sample_set->utimer, next_internal_time);
}

static int pstats_handle_sampleset_param(smpl_set *sample_set, param *sample_param, bool forced)
{
	char *data = NULL, *type = NULL;
	int err = 1;

	if (sample_set == NULL || sample_param == NULL)
		return err;

	bbfdm_get_parameter_value(sample_param->reference, &data, &type);

	if (data == NULL || type == NULL) {
		BBF_ERR("Value and/or Type is NULL");
	} else {
		time_t end_time = time(NULL);

		if (forced == false) {
			sample_param->values[sample_set->int_smpl_count] = strdup(data ? data : "");
			sample_param->param_numeric = data_type_numeric(type);
			sample_param->sample_seconds[sample_set->int_smpl_count] = end_time - sample_set->start_time;
		} else {
			// generate forced report
			char *values = measure_parameter_values(sample_set, sample_param, data);
			if (values != NULL) {
				uci_set_option_value(sample_param->name, "values", values);
				free(values);
			}

			// measure the sample second
			time_t diff_time = end_time - sample_set->start_time;
			char *sample_sec = measure_param_sample_secs(sample_set, sample_param, diff_time);
			if (sample_sec != NULL) {
				uci_set_option_value(sample_param->name, "sample_secs", sample_sec);
				free(sample_sec);
			}

			// measure the suspect data
			char *suspect_data = measure_param_suspect_data(sample_set, sample_param);
			if (suspect_data != NULL) {
				uci_set_option_value(sample_param->name, "suspect_data", suspect_data);
				free(suspect_data);
			}
		}

		err = 0;
	}

	FREE(data);
	FREE(type);

	return err;
}

static void pstats_sampleset_cb(struct uloop_timeout *timeout)
{
	smpl_set *sample_set = NULL;
	param *sample_param = NULL;
	bool valid_param_exist = false;

	sample_set = container_of(timeout, smpl_set, utimer);
	if (!sample_set)
		return;

	if (list_empty(&sample_set->params)) {
		// No parameters exist in the sample set no need to do the measurement
		// and reschedule the next sampling
		return;
	}

	sample_set->start_time = time(NULL);

	list_for_each_entry(sample_param, &sample_set->params, list) {
		if ((sample_param->enable != 1) || (sample_param->reference[0] == '\0'))
			continue;

		valid_param_exist = true;
		pstats_handle_sampleset_param(sample_set, sample_param, false);
	}

	if (valid_param_exist == false) {
		// This sample set has no valid parameter to measure so do not
		// reschedule the next sampling
		return;
	}

	generate_report(sample_set);
	prepare_next_sampling(sample_set);
}

/**
 *  pstats_get_stats function for invoking periodic statistics
 *  @param global_list input pointer to global list_head structure
 */
void pstats_get_stats(struct list_head *global_list)
{
	smpl_set *sample_set = NULL;
	unsigned long next_internal_time;

	list_for_each_entry(sample_set, global_list, list) {
		if (sample_set->enable != 1)
			continue;

		// update initial iteration report start time
		char time_str[32] = {0};
		time_t now = time(NULL);

		strftime(time_str, sizeof(time_str), "%Y-%m-%dT%H:%M:%SZ", gmtime(&now));
		uci_set_option_value(sample_set->name, "report_starttime", time_str);

		sample_set->utimer.cb = pstats_sampleset_cb;
		sample_set->next_sample_time = get_next_sample_time(sample_set->time_ref, sample_set->smpl_intrvl);
		next_internal_time = ((sample_set->next_sample_time - time(NULL)) * 1000) / INTERNAL_SAMPLE_RATE;
		BBF_INFO("Start new collection session of sample set '%s' in %lu msec\n", sample_set->name, next_internal_time);
		uloop_timeout_set(&sample_set->utimer, next_internal_time);
	}
}

void pstats_force_sampling_cb(struct uloop_timeout *timeout)
{
	smpl_set *sample_set = NULL;
	param *sample_param = NULL;
	int ret = 1;

	sample_set = container_of(timeout, smpl_set, ftimer);
	if (!sample_set)
		return;

	time_t start_time = time(NULL);
	list_for_each_entry(sample_param, &sample_set->params, list) {
		if ((sample_param->enable != 1) || (sample_param->reference[0] == '\0'))
			continue;

		ret &= pstats_handle_sampleset_param(sample_set, sample_param, true);
	}

	if (ret == 0) {
		// update the time elapsed to collect the data
		time_t end_time = time(NULL);
		time_t diff_time = end_time - start_time;
		char time_str[32] = {0};

		strftime(time_str, sizeof(time_str), "%Y-%m-%dT%H:%M:%SZ", gmtime(&end_time));
		uci_set_option_value(sample_set->name, "report_endtime", time_str);
		adjust_stats_sample_seconds(sample_set, diff_time);
		sample_set->forced = true;
	}
}

void uci_get_option_value_dmmap(const char *sec_name, const char *option, char **value)
{
	struct uci_ptr ptr = {0};
	char uci_path[256] = {0};

	if (value == NULL)
		return;

	*value = NULL;
	if (sec_name == NULL || option == NULL)
		return;

	struct uci_context *uci_ctx = uci_alloc_context();
	if (uci_ctx == NULL)
		return;

	uci_set_confdir(uci_ctx, "/etc/bbfdm/dmmap");
	snprintf(uci_path, sizeof(uci_path), "dmmap_periodicstats.%s.%s", sec_name, option);

	if (uci_lookup_ptr(uci_ctx, &ptr, uci_path, true) != UCI_OK) {
		uci_free_context(uci_ctx);
		return;
	}

	if (ptr.flags & UCI_LOOKUP_COMPLETE) {
		if (ptr.o->type == UCI_TYPE_STRING) {
			*value = strdup(ptr.o->v.string);
		}
	}

	uci_free_context(uci_ctx);
}
