/*
 * Copyright (C) 2022 iopsys Software Solutions AB
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1
 * as published by the Free Software Foundation
 *
 *	Author: Shubham Sharma <shubham.sharma@iopsys.eu>
 *	Author: Suvendhu Hansa <suvendhu.hansa@iopsys.eu>
 */

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <libbbfdm_api.h>

#include "periodicstats.h"

static bool sample_name_exist(char *name)
{
	struct uci_section *s = NULL, *stmp = NULL;
	char *sample_enab = NULL;
	char *sample_name = NULL;
	bool enable = false;

	if (name == NULL)
		return false;

	uci_foreach_sections_safe("periodicstats", "sampleset", stmp, s) {
		dmuci_get_value_by_section_string(s, "enable", &sample_enab);
		dmuci_get_value_by_section_string(s, "name", &sample_name);

		bbf_convert_string_to_bool(sample_enab, &enable);
		if (enable && DM_STRCMP(sample_name, name) == 0)
			return true;
	}

	return false;
}

static bool reference_exist(char *ref, char *sample_name)
{
	struct uci_section *s = NULL, *stmp = NULL;
	char *param_enable = NULL;
	char *sample_set = NULL;
	char *param_ref = NULL;
	bool enable = false;

	if (ref == NULL || sample_name == NULL)
		return false;

	uci_foreach_sections_safe("periodicstats", "parameter", stmp, s) {
		dmuci_get_value_by_section_string(s, "sample_set", &sample_set);
		dmuci_get_value_by_section_string(s, "reference", &param_ref);
		dmuci_get_value_by_section_string(s, "enable", &param_enable);

		bbf_convert_string_to_bool(param_enable, &enable);
		if (enable && DM_STRCMP(sample_set, sample_name) == 0 && DM_STRCMP(param_ref, ref) == 0)
			return true;
	}

	return false;
}

/*************************************************************
* ENTRY METHOD
**************************************************************/
static void synchronize_sampleset_parameters(struct dmctx *dmctx, struct uci_section *sample_s, const char *sample_instance)
{
	struct uci_section *param_s = NULL;

	// Device.PeriodicStatistics.SampleSet.{i}.Parameter.{i}.
	uci_foreach_sections("periodicstats", "parameter", param_s) {
		char *parent_s = NULL;
		char *instance = NULL;

		dmuci_get_value_by_section_string(param_s, "sample_set", &parent_s);

		if (DM_STRCMP(parent_s, section_name(sample_s)) != 0)
			continue;

		create_dmmap_obj(dmctx, 1, "PeriodicStatistics", "Parameter", param_s, &instance);
	}
}

static void dmmap_synchronize_pstats(struct dmctx *dmctx)
{

	struct uci_section *sample_s = NULL;

	bbfdm_create_empty_file("/etc/bbfdm/dmmap/PeriodicStatistics");

	// Device.PeriodicStatistics.SampleSet.{i}.
	uci_foreach_sections("periodicstats", "sampleset", sample_s) {
		char *sample_instance = NULL;

		create_dmmap_obj(dmctx, 0, "PeriodicStatistics", "SampleSet", sample_s, &sample_instance);

		synchronize_sampleset_parameters(dmctx, sample_s, sample_instance);
	}

	dmuci_commit_package_bbfdm("PeriodicStatistics");
}

static int browsePeriodicStatisticsSampleSetInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	generic_browse(dmctx, parent_node, prev_data, prev_instance);
	return 0;
}

static int browsePeriodicStatisticsSampleSetParameterInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	generic_browse(dmctx, parent_node, prev_data, prev_instance);
	return 0;
}

/****************************************************************************
 * Init & Clean Module
 ***************************************************************************/
int init_pstat_module(void *data)
{
	struct dmctx bbf_ctx = {0};

	bbf_ctx_init(&bbf_ctx, NULL);
	dmmap_synchronize_pstats(&bbf_ctx);
	bbf_ctx_clean(&bbf_ctx);

	return 0;
}

/*************************************************************
* ADD & DEL OBJ
**************************************************************/
static int addObjPeriodicStatisticsSampleSet(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct dm_data *curr_data = (struct dm_data *)ctx->addobj_instance;
	char name[32];

	snprintf(name, sizeof(name), "set_%s", *instance);

	dmuci_add_section("periodicstats", "sampleset", &curr_data->config_section);
	dmuci_rename_section_by_section(curr_data->config_section, name);

	dmuci_set_value_by_section(curr_data->config_section, "enable", DEFAULT_ZERO);
	dmuci_set_value_by_section(curr_data->config_section, "report_sample", DEFAULT_REPORT_SAMPLE);
	dmuci_set_value_by_section(curr_data->config_section, "sample_interval", DEFAULT_SAMPLE_INTERVAL);
	dmuci_set_value_by_section(curr_data->config_section, "fetch_samples", DEFAULT_ZERO);
	dmuci_set_value_by_section(curr_data->config_section, "time_reference", DEFAULT_REF_TIME);

	return 0;
}

static int delObjPeriodicStatisticsSampleSet(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	struct uci_section *ss = NULL, *stmp = NULL;
	char *param_sec_name = NULL;

	// Loop over to delete all related sub-sections i.e., parameters
	uci_path_foreach_option_eq_safe(bbfdm, "PeriodicStatistics", "Parameter", "SampleSet", instance, stmp, ss) {
		dmuci_get_value_by_section_string(ss, "__section_name__", &param_sec_name);

		struct uci_section *param_s = get_config_section_from_dmmap_section_name(param_sec_name);

		dmuci_delete_by_section(param_s, NULL, NULL);
		dmuci_delete_by_section(ss, NULL, NULL);
	}

	dmuci_delete_by_section(((struct dm_data *)data)->config_section, NULL, NULL);
	dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);

	return 0;
}

static int addObjPeriodicStatisticsSampleSetParameter(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct dm_data *curr_data = (struct dm_data *)ctx->addobj_instance;
	char param_name[64] = {0};

	struct uci_section *sample_sec = ((struct dm_data *)data)->config_section;

	snprintf(param_name, sizeof(param_name), "param_%s_%s", *instance, section_name(sample_sec));

	dmuci_add_section("periodicstats", "parameter", &curr_data->config_section);
	dmuci_rename_section_by_section(curr_data->config_section, param_name);
	dmuci_set_value_by_section(curr_data->config_section, "sample_set", section_name(sample_sec));
	dmuci_set_value_by_section(curr_data->config_section, "enable", DEFAULT_ZERO);
	dmuci_set_value_by_section(curr_data->config_section, "reference", NULL);
	dmuci_set_value_by_section(curr_data->config_section, "calculation_mode", "Latest");
	dmuci_set_value_by_section(curr_data->config_section, "sample_mode", "Current");
	dmuci_set_value_by_section(curr_data->config_section, "low_threshold", DEFAULT_ZERO);
	dmuci_set_value_by_section(curr_data->config_section, "high_threshold", DEFAULT_ZERO);

	return 0;
}

static int delObjPeriodicStatisticsSampleSetParameter(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	struct uci_section *dmmap_section = ((struct dm_data *)data)->dmmap_section;
	dmuci_delete_by_section(dmmap_section, NULL, NULL);

	struct uci_section *section = ((struct dm_data *)data)->config_section;
	dmuci_delete_by_section(section, NULL, NULL);
	return 0;
}

/*************************************************************
* GET & SET PARAM
**************************************************************/
static int get_PeriodicStatistics_MinSampleInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup(MIN_SAMPLE_INTERVAL);
	return 0;
}

static int get_PeriodicStatistics_MaxReportSamples(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("0");
	return 0;
}

static int get_PeriodicStatistics_SampleSetNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, NULL);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_PeriodicStatisticsSampleSet_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *sample_sec = ((struct dm_data *)data)->dmmap_section;
	return bbf_get_alias(ctx, sample_sec, "Alias", instance, value);
}

static int set_PeriodicStatisticsSampleSet_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct uci_section *sample_sec = ((struct dm_data *)data)->dmmap_section;
	return bbf_set_alias(ctx, sample_sec, "Alias", instance, value);
}

static int get_PeriodicStatisticsSampleSet_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *sample_sec = ((struct dm_data *)data)->config_section;
	*value = dmuci_get_value_by_section_fallback_def(sample_sec, "enable", DEFAULT_ZERO);
	return 0;
}

static int set_PeriodicStatisticsSampleSet_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool new_val, cur_val;
	struct uci_section *sample_sec = NULL;
	char *cur_enable = NULL, *cur_name = NULL;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;

			bbf_convert_string_to_bool(value, &new_val);
			if (!new_val) {
				break;
			}

			sample_sec = ((struct dm_data *)data)->config_section;

			dmuci_get_value_by_section_string(sample_sec, "enable", &cur_enable);
			bbf_convert_string_to_bool(cur_enable, &cur_val);
			if (cur_val == new_val) {
				break;
			}

			dmuci_get_value_by_section_string(sample_sec, "name", &cur_name);
			if (DM_STRLEN(cur_name) == 0) {
				break;
			}

			/* Now check if another instance with same name is already enabled */
			if (sample_name_exist(cur_name)) {
				return FAULT_9007;
			}

			break;
		case VALUESET:
			sample_sec = ((struct dm_data *)data)->config_section;
			bbf_convert_string_to_bool(value, &new_val);

			dmuci_set_value_by_section(sample_sec, "enable", new_val ? "1" : DEFAULT_ZERO);

			break;
	}
	return 0;
}

static int get_PeriodicStatisticsSampleSet_Status(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	bool b = false;
	struct uci_section *sample_sec = ((struct dm_data *)data)->config_section;

	dmuci_get_value_by_section_string(sample_sec, "enable", value);
	bbf_convert_string_to_bool(*value, &b);
	if (b == false) {
		*value = dmstrdup("Disabled");
		return 0;
	}

	*value = dmuci_get_value_by_section_fallback_def(sample_sec, "status", "Enabled");

	return 0;
}

static int get_PeriodicStatisticsSampleSet_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *sample_sec = ((struct dm_data *)data)->config_section;
	dmuci_get_value_by_section_string(sample_sec, "name", value);
	return 0;
}

static int set_PeriodicStatisticsSampleSet_Name(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct uci_section *sample_sec = NULL;
	char *cur_name = NULL, *cur_enable = NULL;
	bool enable;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 128, NULL, NULL))
				return FAULT_9007;

			// Check if the value is empty
			if (*value == '\0') {
				break;
			}

			sample_sec = ((struct dm_data *)data)->config_section;

			dmuci_get_value_by_section_string(sample_sec, "name", &cur_name);
			if (DM_STRCMP(cur_name, value) == 0) {
				break;
			}

			dmuci_get_value_by_section_string(sample_sec, "enable", &cur_enable);
			bbf_convert_string_to_bool(cur_enable, &enable);
			if (!enable) {
				break;
			}

			// Check if other enabled instance with same name
			if (sample_name_exist(value)) {
				return FAULT_9007;
			}

			break;
		case VALUESET:
			sample_sec = ((struct dm_data *)data)->config_section;
			dmuci_set_value_by_section(sample_sec, "name", value);
			break;
	}
	return 0;
}

static int get_PeriodicStatisticsSampleSet_Controller(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("");
	return 0;
}

static int get_PeriodicStatisticsSampleSet_SampleInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *sample_sec = ((struct dm_data *)data)->config_section;
	*value = dmuci_get_value_by_section_fallback_def(sample_sec, "sample_interval", DEFAULT_SAMPLE_INTERVAL);
	return 0;
}

static int set_PeriodicStatisticsSampleSet_SampleInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct uci_section *sample_sec = NULL;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{MIN_SAMPLE_INTERVAL,NULL}}, 1))
				return FAULT_9007;
			break;
		case VALUESET:
			sample_sec = ((struct dm_data *)data)->config_section;
			dmuci_set_value_by_section(sample_sec, "sample_interval", value);
			break;
	}
	return 0;
}

static int get_PeriodicStatisticsSampleSet_ReportSamples(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *sample_sec = ((struct dm_data *)data)->config_section;
	*value = dmuci_get_value_by_section_fallback_def(sample_sec, "report_sample", DEFAULT_REPORT_SAMPLE);
	return 0;
}

static int set_PeriodicStatisticsSampleSet_ReportSamples(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct uci_section *sample_sec = NULL;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{"1",NULL}}, 1))
				return FAULT_9007;
			break;
		case VALUESET:
			sample_sec = ((struct dm_data *)data)->config_section;
			dmuci_set_value_by_section(sample_sec, "report_sample", value);
			break;
	}
	return 0;
}

static int get_PeriodicStatisticsSampleSet_TimeReference(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *sample_sec = ((struct dm_data *)data)->config_section;
	*value = dmuci_get_value_by_section_fallback_def(sample_sec, "time_reference", DEFAULT_REF_TIME);
	return 0;
}

static int set_PeriodicStatisticsSampleSet_TimeReference(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct uci_section *sample_sec = NULL;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_dateTime(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			sample_sec = ((struct dm_data *)data)->config_section;
			dmuci_set_value_by_section(sample_sec, "time_reference", value);
			break;
	}
	return 0;
}

static int get_PeriodicStatisticsSampleSet_FetchSamples(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *sample_sec = ((struct dm_data *)data)->config_section;
	*value = dmuci_get_value_by_section_fallback_def(sample_sec, "fetch_samples", DEFAULT_ZERO);

	if (DM_LSTRCMP(*value, DEFAULT_ZERO) == 0 && ctx->dm_type == BBFDM_USP) {
		*value = dmstrdup("1");
	}

	return 0;
}

static int set_PeriodicStatisticsSampleSet_FetchSamples(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct uci_section *sample_sec = NULL;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			sample_sec = ((struct dm_data *)data)->config_section;
			dmuci_set_value_by_section(sample_sec, "fetch_samples", value);
			return 0;
	}
	return 0;
}

static int get_PeriodicStatisticsSampleSet_ForceSample(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("0");
	return 0;
}

static int set_PeriodicStatisticsSampleSet_ForceSample(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct uci_section *sample_sec = NULL;
	char *sample_name = NULL;
	bool val = false;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &val);
			if (val == false)
				break;

			sample_sec = ((struct dm_data *)data)->config_section;
			sample_name = section_name(sample_sec);

			periodicstats_force_sample(sample_name);
			break;
	}
	return 0;
}

static int get_PeriodicStatisticsSampleSet_ReportStartTime(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *sample_sec = ((struct dm_data *)data)->config_section;
	*value = dmuci_get_value_by_section_fallback_def(sample_sec, "report_starttime", DEFAULT_REF_TIME);
	return 0;
}

static int get_PeriodicStatisticsSampleSet_ReportEndTime(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *sample_sec = ((struct dm_data *)data)->config_section;
	*value = dmuci_get_value_by_section_fallback_def(sample_sec, "report_endtime", DEFAULT_REF_TIME);
	return 0;
}

static int get_PeriodicStatisticsSampleSet_SampleSeconds(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *sample_sec = ((struct dm_data *)data)->config_section;
	dmuci_get_value_by_section_string(sample_sec, "sample_secs", value);
	return 0;
}

static int get_PeriodicStatisticsSampleSet_ParameterNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, NULL);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_PeriodicStatisticsSampleSetParameter_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *param_sec = ((struct dm_data *)data)->dmmap_section;
	return bbf_get_alias(ctx, param_sec, "Alias", instance, value);
}

static int set_PeriodicStatisticsSampleSetParameter_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct uci_section *param_sec = ((struct dm_data *)data)->dmmap_section;
	return bbf_set_alias(ctx, param_sec, "Alias", instance, value);
}

static int get_PeriodicStatisticsSampleSetParameter_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *param_sec = ((struct dm_data *)data)->config_section;
	*value = dmuci_get_value_by_section_fallback_def(param_sec, "enable", DEFAULT_ZERO);
	return 0;
}

static int set_PeriodicStatisticsSampleSetParameter_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool cur_val, new_val;
	char *cur_ref = NULL, *cur_enable = NULL, *sample_set = NULL;
	struct uci_section *param_sec = NULL;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;

			bbf_convert_string_to_bool(value, &new_val);
			if (!new_val) {
				break;
			}

			param_sec = ((struct dm_data *)data)->config_section;

			dmuci_get_value_by_section_string(param_sec, "enable", &cur_enable);
			bbf_convert_string_to_bool(cur_enable, &cur_val);
			if (cur_val == new_val) {
				break;
			}

			dmuci_get_value_by_section_string(param_sec, "reference", &cur_ref);
			if (DM_STRLEN(cur_ref) == 0) {
				break;
			}

			/* Now check if another instance with same reference is already enabled */
			dmuci_get_value_by_section_string(param_sec, "sample_set", &sample_set);
			if (reference_exist(cur_ref, sample_set)) {
				return FAULT_9007;
			}

			break;
		case VALUESET:
			param_sec = ((struct dm_data *)data)->config_section;
			bbf_convert_string_to_bool(value, &new_val);
			dmuci_set_value_by_section(param_sec, "enable", new_val ? "1" : DEFAULT_ZERO);

			break;
	}
	return 0;
}

static int get_PeriodicStatisticsSampleSetParameter_Reference(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *param_sec = ((struct dm_data *)data)->config_section;
	dmuci_get_value_by_section_string(param_sec, "reference", value);
	return 0;
}

static int set_PeriodicStatisticsSampleSetParameter_Reference(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct uci_section *param_sec = NULL;
	bool enable;
	char *uci_val = NULL, *uci_ref = NULL, *sample_set = NULL;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 256, NULL, NULL))
				return FAULT_9007;

			if (DM_STRLEN(value) == 0) {
				break;
			}

			if (DM_LSTRSTR(value, "Device.PeriodicStatistics.") != NULL) {
				bbfdm_set_fault_message(ctx, "PeriodicStatistics parameter reference not supported.");
				return FAULT_9007;
			}

			// Path should be a parameter and not an object, so it must not end with '.'
			int len = DM_STRLEN(value);
			if (len > 0 && value[len - 1] == '.' ) {
				bbfdm_set_fault_message(ctx, "An object path can't be referenced.");
				return FAULT_9007;
			}

			param_sec = ((struct dm_data *)data)->config_section;

			dmuci_get_value_by_section_string(param_sec, "reference", &uci_ref);
			if (DM_STRCMP(uci_ref, value) == 0) {
				break;
			}

			dmuci_get_value_by_section_string(param_sec, "enable", &uci_val);
			bbf_convert_string_to_bool(uci_val, &enable);
			if (!enable) {
				break;
			}

			// if same reference is enabled at other instance then error
			dmuci_get_value_by_section_string(param_sec, "sample_set", &sample_set);
			if (reference_exist(value, sample_set)) {
				return FAULT_9007;
			}

			break;
		case VALUESET:
			param_sec = ((struct dm_data *)data)->config_section;
			dmuci_set_value_by_section(param_sec, "reference", value);
			return 0;
	}
	return 0;
}

static int get_PeriodicStatisticsSampleSetParameter_CalculationMode(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *param_sec = ((struct dm_data *)data)->config_section;
	*value = dmuci_get_value_by_section_fallback_def(param_sec, "calculation_mode", "Latest");
	return 0;
}

static int set_PeriodicStatisticsSampleSetParameter_CalculationMode(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *CalculationMode[] = {"Latest", "Average", "Minimum", "Maximum", NULL};
	struct uci_section *param_sec = NULL;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, CalculationMode, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			param_sec = ((struct dm_data *)data)->config_section;
			dmuci_set_value_by_section(param_sec, "calculation_mode", value);
			break;
	}
	return 0;
}

static int get_PeriodicStatisticsSampleSetParameter_SampleMode(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *param_sec = ((struct dm_data *)data)->config_section;
	*value = dmuci_get_value_by_section_fallback_def(param_sec, "sample_mode", "Current");
	return 0;
}

static int set_PeriodicStatisticsSampleSetParameter_SampleMode(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *SampleMode[] = {"Current", NULL};
	struct uci_section *param_sec = NULL;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, SampleMode, NULL)) {
				bbfdm_set_fault_message(ctx, "Only Current mode is supported");
				return FAULT_9007;
			}
			break;
		case VALUESET:
			param_sec = ((struct dm_data *)data)->config_section;
			dmuci_set_value_by_section(param_sec, "sample_mode", value);
			break;
	}
	return 0;
}

static int get_PeriodicStatisticsSampleSetParameter_LowThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *param_sec = ((struct dm_data *)data)->config_section;
	*value = dmuci_get_value_by_section_fallback_def(param_sec, "low_threshold", DEFAULT_ZERO);
	return 0;
}

static int set_PeriodicStatisticsSampleSetParameter_LowThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct uci_section *param_sec = NULL;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_int(ctx, value, RANGE_ARGS{{"NULL",NULL}}, 1))
				return FAULT_9007;
			break;
		case VALUESET:
			param_sec = ((struct dm_data *)data)->config_section;
			dmuci_set_value_by_section(param_sec, "low_threshold", value);
			break;
	}
	return 0;
}

static int get_PeriodicStatisticsSampleSetParameter_HighThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *param_sec = ((struct dm_data *)data)->config_section;
	*value = dmuci_get_value_by_section_fallback_def(param_sec, "high_threshold", DEFAULT_ZERO);
	return 0;
}

static int set_PeriodicStatisticsSampleSetParameter_HighThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct uci_section *param_sec = NULL;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_int(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;
			break;
		case VALUESET:
			param_sec = ((struct dm_data *)data)->config_section;
			dmuci_set_value_by_section(param_sec, "high_threshold", value);
			break;
	}
	return 0;
}

static int get_PeriodicStatisticsSampleSetParameter_SampleSeconds(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *sample_sec = ((struct dm_data *)data)->config_section;
	dmuci_get_value_by_section_string(sample_sec, "sample_secs", value);
	return 0;
}

static int get_PeriodicStatisticsSampleSetParameter_SuspectData(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *sample_sec = ((struct dm_data *)data)->config_section;
	dmuci_get_value_by_section_string(sample_sec, "suspect_data", value);
	return 0;
}

static int get_PeriodicStatisticsSampleSetParameter_Values(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *param_sec = ((struct dm_data *)data)->config_section;
	dmuci_get_value_by_section_string(param_sec, "values", value);
	return 0;
}

static int get_PeriodicStatisticsSampleSetParameter_Failures(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_section *param_sec = ((struct dm_data *)data)->config_section;
	*value = dmuci_get_value_by_section_fallback_def(param_sec, "failures", DEFAULT_ZERO);
	return 0;
}

static event_args periodic_statistics_push_args = {
	.param = (const char *[]) {
		"Parameter.{i}.Reference",
		"Parameter.{i}.Values",
		"Parameter.{i}.SampleSeconds",
		"Parameter.{i}.Failures",
		NULL
	}
};

static int get_event_args_PeriodicStatisticsSampleSet_Push(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char*)&periodic_statistics_push_args;
	return 0;
}

static int operate_PeriodicStatisticsSampleSet_ForceSample(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct uci_section *sample_sec = ((struct dm_data *)data)->config_section;
	char *sample_name = section_name(sample_sec);

	periodicstats_force_sample(sample_name);

	return 0;
}

/**********************************************************************************************************************************
*                                            OBJ & PARAM DEFINITION
***********************************************************************************************************************************/
/* *** Device.PeriodicStatistics.SampleSet.{i}.Parameter.{i}. *** */
DMLEAF tPeriodicStatisticsSampleSetParameterParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"Alias", &DMWRITE, DMT_STRING, get_PeriodicStatisticsSampleSetParameter_Alias, set_PeriodicStatisticsSampleSetParameter_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Enable", &DMWRITE, DMT_BOOL, get_PeriodicStatisticsSampleSetParameter_Enable, set_PeriodicStatisticsSampleSetParameter_Enable, BBFDM_BOTH},
{"Reference", &DMWRITE, DMT_STRING, get_PeriodicStatisticsSampleSetParameter_Reference, set_PeriodicStatisticsSampleSetParameter_Reference, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"SampleMode", &DMWRITE, DMT_STRING, get_PeriodicStatisticsSampleSetParameter_SampleMode, set_PeriodicStatisticsSampleSetParameter_SampleMode, BBFDM_BOTH},
{"CalculationMode", &DMWRITE, DMT_STRING, get_PeriodicStatisticsSampleSetParameter_CalculationMode, set_PeriodicStatisticsSampleSetParameter_CalculationMode, BBFDM_BOTH},
{"LowThreshold", &DMWRITE, DMT_INT, get_PeriodicStatisticsSampleSetParameter_LowThreshold, set_PeriodicStatisticsSampleSetParameter_LowThreshold, BBFDM_BOTH},
{"HighThreshold", &DMWRITE, DMT_INT, get_PeriodicStatisticsSampleSetParameter_HighThreshold, set_PeriodicStatisticsSampleSetParameter_HighThreshold, BBFDM_BOTH},
{"SampleSeconds", &DMREAD, DMT_STRING, get_PeriodicStatisticsSampleSetParameter_SampleSeconds, NULL, BBFDM_BOTH},
{"SuspectData", &DMREAD, DMT_STRING, get_PeriodicStatisticsSampleSetParameter_SuspectData, NULL, BBFDM_BOTH},
{"Values", &DMREAD, DMT_STRING, get_PeriodicStatisticsSampleSetParameter_Values, NULL, BBFDM_BOTH},
{"Failures", &DMREAD, DMT_UNINT, get_PeriodicStatisticsSampleSetParameter_Failures, NULL, BBFDM_BOTH},
{0}
};

DMLEAF tPeriodicStatisticsSampleSetParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"Alias", &DMWRITE, DMT_STRING, get_PeriodicStatisticsSampleSet_Alias, set_PeriodicStatisticsSampleSet_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Enable", &DMWRITE, DMT_BOOL, get_PeriodicStatisticsSampleSet_Enable, set_PeriodicStatisticsSampleSet_Enable, BBFDM_BOTH},
{"Status", &DMREAD, DMT_STRING, get_PeriodicStatisticsSampleSet_Status, NULL, BBFDM_CWMP},
{"Name", &DMWRITE, DMT_STRING, get_PeriodicStatisticsSampleSet_Name, set_PeriodicStatisticsSampleSet_Name, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Controller", &DMWRITE, DMT_STRING, get_PeriodicStatisticsSampleSet_Controller, NULL, BBFDM_BOTH},
{"SampleInterval", &DMWRITE, DMT_UNINT, get_PeriodicStatisticsSampleSet_SampleInterval, set_PeriodicStatisticsSampleSet_SampleInterval, BBFDM_BOTH},
{"ReportSamples", &DMWRITE, DMT_UNINT, get_PeriodicStatisticsSampleSet_ReportSamples, set_PeriodicStatisticsSampleSet_ReportSamples, BBFDM_BOTH},
{"TimeReference", &DMWRITE, DMT_TIME, get_PeriodicStatisticsSampleSet_TimeReference, set_PeriodicStatisticsSampleSet_TimeReference, BBFDM_BOTH},
{"FetchSamples", &DMWRITE, DMT_UNINT, get_PeriodicStatisticsSampleSet_FetchSamples, set_PeriodicStatisticsSampleSet_FetchSamples, BBFDM_BOTH},
{"ForceSample", &DMWRITE, DMT_BOOL, get_PeriodicStatisticsSampleSet_ForceSample, set_PeriodicStatisticsSampleSet_ForceSample, BBFDM_CWMP},
{"ReportStartTime", &DMREAD, DMT_TIME, get_PeriodicStatisticsSampleSet_ReportStartTime, NULL, BBFDM_BOTH},
{"ReportEndTime", &DMREAD, DMT_TIME, get_PeriodicStatisticsSampleSet_ReportEndTime, NULL, BBFDM_BOTH},
{"SampleSeconds", &DMREAD, DMT_STRING, get_PeriodicStatisticsSampleSet_SampleSeconds, NULL, BBFDM_BOTH},
{"ParameterNumberOfEntries", &DMREAD, DMT_UNINT, get_PeriodicStatisticsSampleSet_ParameterNumberOfEntries, NULL, BBFDM_BOTH},
{"Push!", &DMREAD, DMT_EVENT, get_event_args_PeriodicStatisticsSampleSet_Push, NULL, BBFDM_USP},
{"ForceSample()", &DMSYNC, DMT_COMMAND, NULL, operate_PeriodicStatisticsSampleSet_ForceSample, BBFDM_USP},
{0}
};

/* *** Device.PeriodicStatistics.SampleSet.{i}. *** */
DMOBJ tPeriodicStatisticsSampleSetObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys*/
{"Parameter", &DMWRITE, addObjPeriodicStatisticsSampleSetParameter, delObjPeriodicStatisticsSampleSetParameter, NULL, browsePeriodicStatisticsSampleSetParameterInst, NULL, NULL, NULL, tPeriodicStatisticsSampleSetParameterParams, NULL, BBFDM_BOTH, NULL},
{0}
};

/* *** Device.PeriodicStatistics. *** */
DMOBJ tPeriodicStatisticsObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys*/
{"SampleSet", &DMWRITE, addObjPeriodicStatisticsSampleSet, delObjPeriodicStatisticsSampleSet, NULL, browsePeriodicStatisticsSampleSetInst, NULL, NULL, tPeriodicStatisticsSampleSetObj, tPeriodicStatisticsSampleSetParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tPeriodicStatisticsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type*/
{"MinSampleInterval", &DMREAD, DMT_UNINT, get_PeriodicStatistics_MinSampleInterval, NULL, BBFDM_BOTH},
{"MaxReportSamples", &DMREAD, DMT_UNINT, get_PeriodicStatistics_MaxReportSamples, NULL, BBFDM_BOTH},
{"SampleSetNumberOfEntries", &DMREAD, DMT_UNINT, get_PeriodicStatistics_SampleSetNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device. *** */
DMOBJ tDevicePeriodicStatisticsObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys*/
{"PeriodicStatistics", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tPeriodicStatisticsObj, tPeriodicStatisticsParams, NULL, BBFDM_BOTH},
{0}
};

/* ********** DynamicObj ********** */
DM_MAP_OBJ tDynamicObj[] = {
/* parentobj, nextobject, parameter */
{"Device.", tDevicePeriodicStatisticsObj, NULL, init_pstat_module, NULL},
{0}
};
