/*
 * periodicstats.h - definitions used in periodic statistics
 *
 * Copyright (C) 2022 iopsys Software Solutions AB. All rights reserved.
 *
 * Author: Shubham Sharma <shubham.sharma@iopsys.eu>
 *
 * See LICENSE file for license related information.
 *
 */

#ifndef PERIODICSTATS_H
#define PERIODICSTATS_H

#include <string.h>
#include <time.h>
#include <uci.h>
#include <sys/stat.h>
#include <sys/types.h>

/* To utalize kernel list implementation. */
#include <libubox/list.h>
#include <libubus.h>
#include <libubox/uloop.h>

#include <libbbfdm-api/dmcommon.h>

/* UCI definitions, used to parse the config files at their right location */
#define UCI_CONFIG_PSTATS   "periodicstats"

#define MAX_DATA_STR 4096
#define MAX_REF_STR 256
#define MAX_NAME_STR 128
#define MAX_ALIAS_STR 64
#define MAX_TIME_STR 25

#define DEFAULT_REF_TIME "0001-01-01T00:00:00Z"
#define DEFAULT_ONE "1"
#define DEFAULT_ZERO "0"
#define DEFAULT_REPORT_SAMPLE "24"
#define DEFAULT_SAMPLE_INTERVAL "3600"
#define BASE 10
#define MIN_SAMPLE_INTERVAL "60"

#ifdef INTERNAL_SAMPLING_RATE
#define INTERNAL_SAMPLE_RATE INTERNAL_SAMPLING_RATE
#else
#define INTERNAL_SAMPLE_RATE 3
#endif


#define FREE(x) do { if(x) {free(x); x = NULL;} } while (0)

enum calculation_mode_type {
	CAL_MODE_LATEST,
	CAL_MODE_AVERAGE,
	CAL_MODE_MINIMUM,
	CAL_MODE_MAXIMUM,
	__CAL_MODE_LAST
};

/*
 * Structure for parameters inside each sample set
 * */
typedef struct parameters {
	struct list_head list;
	bool enable;
	bool forced;
	bool param_numeric;
	int calculation_mode;
	int low_threshold;
	int high_threshold;
	time_t sample_seconds[INTERNAL_SAMPLE_RATE];
	char *values[INTERNAL_SAMPLE_RATE];
	char name[MAX_NAME_STR];
	char reference[MAX_REF_STR];
} param;

/*
 * Structure for sample sets inside global periodic stats
 * */
typedef struct sample_sets {
	struct uloop_timeout utimer;
	struct uloop_timeout ftimer;
	struct list_head params;
	struct list_head list;
	bool enable;
	bool forced;
	uint32_t smpl_intrvl;
	uint32_t rprt_smpls;
	uint32_t fetch_samples;
	time_t time_ref;
	time_t next_sample_time;
	int int_smpl_count;
	uint32_t rep_smpl_count;
	time_t start_time;
	time_t sample_seconds[INTERNAL_SAMPLE_RATE];
	char name[MAX_NAME_STR];
} smpl_set;

int pstats_global_init(struct list_head *ss_list);
void pstats_global_free(struct list_head *ss_list);
void pstats_get_stats(struct list_head *global_list);
void uci_set_option_value(const char *sec_name, const char *option, const char *value);
void pstats_force_sampling_cb(struct uloop_timeout *timeout);
void periodicstats_force_sample(const char *sample_set_name);
int pstats_get_sample_set_instance(const char *sample_set);
void uci_get_option_value_dmmap(const char *sec_name, const char *option, char **value);


#endif /* PERIODICSTATS_H */
