/* gcc -Ilibdpp -Llibdpp -o a a.c -ldpp -leasy */
#include <stdio.h>
#include <stddef.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>

#include <easy/easy.h>
#include <wifidefs.h>
#include <wifiutils.h>

#include "util.h"
#include "dpp_api.h"

void *ap_key;

int get_ap_pubkey(void **B, size_t *Blen)
{
	void *k = ecc_key_gen("P-256");

	ap_key = k;

	if (k) {
		void *out;
		size_t olen = 0;

		//B-key
		out = ecc_key_get_spki(k, &olen);
		if (out) {
			*B = out;
			*Blen = olen;
			bufprintf("B = DER-SubjectPublicKeyInfo = ", out, olen);
			return 0;
		}
	}

	return -1;
}

int derive_k(uint8_t *z, size_t zlen, uint8_t *k, unsigned int hashlen)
{
	const char *info = "bootstrap transfer key";
	uint8_t salt[64], prk[64];
	const uint8_t *addr[1];
	size_t addrlen[1];
	int res;

	if (hashlen != 32)	/* for prime256v1 */
		return -1;

	/* k = HKDF(<>, "bootstrap transfer key", z) */

	/* HKDF-Extract(<>, z) */
	memset(salt, 0, hashlen);
	addr[0] = (const uint8_t *)z;
	addrlen[0] = zlen;

	res = PLATFORM_HMAC_SHA256(salt, hashlen, 1, addr, addrlen, prk);
	if (res < 0)
		return -1;

	bufprintf("HKDF-Extract(<>, IKM=z)", prk, hashlen);

	/* HKDF-Expand(PRK, info, L) */
	res = hmac_sha256_kdf(prk, hashlen, NULL, (const uint8_t *)info, strlen(info), k, hashlen);
	memset(prk, 0, hashlen);
	if (res < 0)
		return -1;

	bufprintf("k = HKDF-Expand(PRK, info, L)", k, hashlen);
	return 0;
}

int dpp_process_pa_vsie_frame(uint8_t *frame, size_t framelen, char **uri)
{
	size_t wrapped_len, unwrapped_len;
	unsigned int hashlen = 32;
	size_t epk1_len = 0;
	uint8_t *unwrapped;
	uint8_t *pos;
	uint8_t k[64];
	size_t zlen;
	void *epk1;
	uint8_t *z;
	int ret;

	/* validate frame leading bytes */
	if (!frame || !frame[0] == 0x4 || !frame[1] == 0x9 ||
	    memcmp(&frame[2], "\xb4\x56\xfa", 3) || !frame[5] == 0xa) {
		printf("Ignore invalid dpp-pa-vsie-frame\n");
		return -1;
	}
	pos = frame + 6;

	epk1_len = buf_get_le16(pos);
	pos += 2;
	epk1 = ecc_key_gen_from_spki(pos, epk1_len);
	if (!epk1) {
		printf("error getting epk1 from frame (epk1_len = %zd)\n", epk1_len);
		return -1;
	}

	pos += epk1_len;
	wrapped_len = buf_get_le16(pos);
	pos += 2;
	printf("Ciphertext len = %zd\n", wrapped_len);
	if (wrapped_len < AES_BLOCK_SIZE) {
		printf("invalid wrapped len\n");
		return -1;
	}

	/* ECDH */
	if (ecc_ecdh(ap_key, epk1, &z, &zlen) < 0) {
		printf("ecdh error!\n");
		return -1;
	}

	bufprintf("DPP: ECDH shared secret (z')", z, zlen);

	/* KHDF-SHA256 */
	if (derive_k(z, zlen, k, hashlen) < 0) {
		printf("error derive_k\n");
		return -1;
	}

	/* Decrypt C */
	unwrapped_len = wrapped_len - AES_BLOCK_SIZE;
	unwrapped = calloc(1, unwrapped_len + 1); 	// +1 for '\0' of uri
	if (!unwrapped) {
		printf("error calloc unwrapped\n");
		return -1;
	}

	ret = AES_SIV_DECRYPT(k, 32, pos, wrapped_len, 0, NULL, NULL, unwrapped);
	if (ret) {
		printf("AES-SIV decrypt failed\n");
		free(unwrapped);
		return -1;
	}
	bufprintf("AES-SIV plaintext", unwrapped, unwrapped_len);

	if (uri)
		*uri = unwrapped;

	printf("URI (len = %zd): %s\n", unwrapped_len, unwrapped);
	printf("SUCCESS !!!\n");
	return 0;
}

uint8_t *dpp_gen_pa_vsie_frame(uint8_t *payload, size_t len, size_t *olen)
{
	uint8_t *msg, *head;
	uint8_t oui[3] = {0xb4, 0x56, 0xfa};	/* IOPSYS OUI */
	uint8_t ouitype = 0xa;			/* Encoded dpp bootstrap uri */

	*olen = 0;
	msg = calloc(1, len + 6);
	if (!msg)
		return NULL;

	*olen = len + 6;
	head = msg;
	bufptr_put_u8(msg, 0x4);	/* public action */
	bufptr_put_u8(msg, 0x9);	/* public action vendor specific */
	bufptr_put(msg, oui, 3);
	bufptr_put_u8(msg, ouitype);
	memcpy(msg, payload, len);
	//bufprintf("PA-dpp-bs-uri", head, 8);

	return head;
}

uint8_t *dpp_gen_encoded_uri_payload(void *peer_pubkey, void *own_bi_key, const char *uri, size_t *olen)
{
	unsigned int hashlen = 32;	/* for P-256 curve */
	uint8_t *z, *msg = NULL;
	uint8_t *payload = NULL;
	size_t zlen, msglen;
	size_t epk_len = 0;
	size_t plen = 0;
	uint8_t k[64];
	uint8_t *epk;
	int ret;

	if (!uri || !peer_pubkey || !own_bi_key || !olen)
		return NULL;

	*olen = 0;
	printf("URI (len = %zd): %s\n", strlen(uri), uri);

	/* epk */
	epk = ecc_key_get_spki(own_bi_key, &epk_len);
	if (!epk) {
		printf("failed to get epk\n");
		return NULL;
	}

	/* ECDH */
	if (ecc_ecdh(own_bi_key, peer_pubkey, &z, &zlen) < 0) {
		printf("ecdh error!\n");
		return NULL;
	}

	bufprintf("ECDH shared secret (z)", z, zlen);

	/* KHDF-SHA256 */
	if (derive_k(z, zlen, k, hashlen) < 0) {
		printf("error derive_k\n");
		goto out;
	}

	/* AES-SIV encrypt */
	msglen = strlen(uri) + AES_BLOCK_SIZE;
	msg = calloc(1, msglen);
	if (!msg)
		goto out;

	ret = AES_SIV_ENCRYPT(k, 32, uri, strlen(uri), 0, NULL, NULL, msg);
	if (ret) {
		printf("AES-SIV encrypt failed\n");
		goto out;
	}
	//bufprintf("AES-SIV ciphertext", msg, msglen);

	/* construct payload = epk | msg */
	payload = calloc(1, epk_len + msglen);
	if (!payload)
		goto out;

	buf_put_le16(payload, epk_len);
	memcpy(payload + 2, epk, epk_len);
	plen += 2 + epk_len;

	buf_put_le16(&payload[plen], msglen);
	memcpy(&payload[plen + 2], msg, msglen);
	plen += 2 + msglen;
	*olen = plen;

out:
	memset(k, 0, sizeof(k));
	free(z);
	free(msg);

	return payload;
}

int main()
{
	struct dpp_bootstrap_info *bi;
	int ret;
	size_t Blen = 0;
	void *B;	// DER encoded SPKI of AP's pubkey
	void *Bkey;	// EVP_PKEY of B

	/* This should already be available with the Enrollee/Repeater */
	if (get_ap_pubkey(&B, &Blen)) {
		printf("error getting B\n");
		return -1;
	}

	Bkey = ecc_key_gen_from_spki(B, Blen);
	if (!Bkey) {
		printf("error getting Bkey from B\n");
		return -1;
	}

	/* Enrollee - generate bootstrap info and URI */
	ret = dpp_gen_bootstrap_info("P-256", "81/1,81/6,81/11", NULL, "123456789A", &bi);
	if (ret) {
		printf("%s: Failed to gen bootstrap info for self\n", __func__);
		return -1;
	}

	dpp_print_bootstrap_info(bi);

	size_t msglen = 0;
	uint8_t *msg = dpp_gen_encoded_uri_payload(Bkey, bi->key, bi->uri, &msglen);

	if (!msg)
		goto out;

	bufprintf("AES-SIV ciphertext", msg, msglen);

	size_t framelen = 0;
	uint8_t *frame = dpp_gen_pa_vsie_frame(msg, msglen, &framelen);

	if (!frame)
		goto out;

	bufprintf("PA-Frame", frame, framelen);

	printf("########### AP/configurator side processing  ##########\n");
	/* AP/configurator receives frame and processes it */
	char *uri = NULL;

	if (dpp_process_pa_vsie_frame(frame, framelen, &uri) < 0) {
		printf("Error! decode dpp-vsie-frame\n");
	} else {
		printf("Got URI: %s\n", uri);
		free(uri);
	}

out:
	free(msg);
	free(frame);
	return 0;
}
