import pytest
import time
from helpers.container import send_sighup
from helpers.config import config_read, config_set, config_commit, config_get
from helpers.ubus import get_test_bands, verify_ubus_ssid

MAP_GW = "map-gw"
PEER_CONTAINERS = ["map-ext1", "map-ext2"]
MAPCONTROLLER_CONFIG = "mapcontroller"
WIRELESS_CONFIG = "wireless"

NEW_SSID = "MAP-Test-BH-Int"


def config_set_ssid(container, new_ssid, type="fronthaul"):
    """
    Update SSID in mapcontroller 'ap' sections matching 'type'
    """
    uci_dict = config_read(container, MAPCONTROLLER_CONFIG)

    for section in uci_dict:
        if section.startswith("@ap") or section.startswith("ap"):
            section_type = uci_dict[section].get("type", "")
            if section_type == type:
                config_set(container, MAPCONTROLLER_CONFIG, section, "ssid", new_ssid)

    config_commit(container, MAPCONTROLLER_CONFIG)


@pytest.mark.parametrize("container", [MAP_GW] + PEER_CONTAINERS)
def test_multiap_set_backhaul_ssid(container):
    config_set_ssid(MAP_GW, NEW_SSID, type="backhaul")

    result = send_sighup(MAP_GW, "mapcontroller")
    assert result.returncode == 0, f"Failed to SIGHUP mapcontroller: {result.stderr}"

    # Wait for AP auto-configuration
    time.sleep(10)

    # SIGHUP wifimngr to reload with new/changed wireless config (optional)
    for c in [MAP_GW] + PEER_CONTAINERS:
        result = send_sighup(c, "wifimngr")
        assert result.returncode == 0, f"Failed to SIGHUP wifimngr on {c}: {result.stderr}"

    # Wait for wifimngr to be ready
    #time.sleep(3)

    # Verify wireless wifi-iface sections having multiap=2 are correctly updated
    for c in [MAP_GW] + PEER_CONTAINERS:
        wifi_dict = config_read(c, WIRELESS_CONFIG)
        for section in wifi_dict:
            if section.startswith("@wifi-iface") or section.startswith("wifi-iface"):
                multiap_val = wifi_dict[section].get("multiap", None)
                if multiap_val == "1":
                    ssid_val = wifi_dict[section].get("ssid", None)
                    assert ssid_val == NEW_SSID, (
                        f"{c} section {section} has multiap={multiap_val} but SSID='{ssid_val}', "
                        f"expected '{NEW_SSID}'"
                    )

    # FIXME: wifi=TEST returns fixed values now. So, cannot verify ubus output
    #for c in [MAP_GW] + PEER_CONTAINERS:
    #    verify_ubus_ssid(c, NEW_SSID, type="backhaul")

