/*
 * Copyright (C) 2023 IOPSYS Software Solutions AB. All rights reserved.
 */
#if (EASYMESH_VERSION > 2)
#ifdef USE_LIBDPP

#include "dpp.h"

#include <libubox/list.h>
#include <easy/easy.h>
#include <wifidefs.h>
#include <dpp_api.h>
#include <dpputils.h>
#include <string.h>
#include <json-c/json.h>
#include <libubox/blobmsg_json.h>

#include "cmdu.h"
#include "cntlr.h"
#include "cntlr_cmdu.h"
#include "cntlr_map.h"
#include "utils/utils.h"
#include "utils/debug.h"


const char *dpp_frame_type2str(uint8_t type)
{
	if (type == DPP_PA_AUTHENTICATION_REQ)
		return "Authentication Request";
	else if (type == DPP_PA_AUTHENTICATION_RESP)
		return "Authentication Response";
	else if (type == DPP_PA_AUTHENTICATION_CONF)
		return "Authentication Confirm";
	else if (type == DPP_PA_PEER_DISCOVERY_REQ)
		return "Peer Discovery Request";
	else if (type == DPP_PA_PEER_DISCOVERY_RESP)
		return "Peer Discovery Response";
	else if (type == DPP_PA_PKEX_V1_EXCHANGE_REQ)
		return "PKEX Version 1 Exchange Request";
	else if (type == DPP_PA_PKEX_EXCHANGE_RESP)
		return "PKEX Exchange Response";
	else if (type == DPP_PA_PKEX_COMMIT_REVEAL_REQ)
		return "PKEX Commit-Reveal Request";
	else if (type == DPP_PA_PKEX_COMMIT_REVEAL_RESP)
		return "PKEX Commit-Reveal Response";
	else if (type == DPP_PA_CONFIGURATION_RESULT)
		return "Configuration Result";
	else if (type == DPP_PA_CONNECTION_STATUS_RESULT)
		return "Connection Status Result";
	else if (type == DPP_PA_PRESENCE_ANNOUNCEMENT)
		return "Presence Announcement";
	else if (type == DPP_PA_RECONFIG_ANNOUNCEMENT)
		return "Reconfiguration Announcement";
	else if (type == DPP_PA_RECONFIG_AUTH_REQ)
		return "Reconfiguration Authentication Request";
	else if (type == DPP_PA_RECONFIG_AUTH_RESP)
		return "Reconfiguration Authentication Response";
	else if (type == DPP_PA_RECONFIG_AUTH_CONF)
		return "Reconfiguration Authentication Confirm";
	else if (type == DPP_PA_PKEX_EXCHANGE_REQ)
		return "PKEX Exchange Request";
	else if (type == 0x13)
		return "Push Button Presence Announcement";
	else if (type == 0x14)
		return "Push Button Presence Announcement Response";
	else if (type == 0x15)
		return "Private Peer Introduction Query";
	else if (type == 0x16)
		return "Private Peer Introduction Notify";
	else if (type == 0x17)
		return "Private Peer Introduction Update";
	return "Unknown";
}

#ifdef ZEROTOUCH_DPP
int zerotouch_append_keyfile(struct controller *c, char *keystr)
{
	struct json_object *key_json = NULL, *arr = NULL;
	struct json_object *val;
	int rc = ZT_PASSPHRASE_ADD_ERROR;
	int rv, len, i;

	key_json = json_object_from_file(ZEROTOUCH_KEYFILE);
	if (!key_json) {
		dbg("%s: failed to read json:%s error:%s. Generate new\n",
			__func__, ZEROTOUCH_KEYFILE, json_util_get_last_err());

		key_json = json_object_new_object();
		if (!key_json) {
			err("%s: error generating json obj\n", __func__);
			goto out;
		}

		arr = json_object_new_array();
		if (!arr) {
			err("%s: error generating array obj\n", __func__);
			goto out_put_json;
		}

		json_object_object_add(key_json, "keys", arr);
	}

	rv = json_object_object_get_ex(key_json, "keys", &arr);
	if (!rv || !arr) {
		err("%s: couldn't find 'keys' array in file:%s\n",
			__func__, ZEROTOUCH_KEYFILE);
		goto out_put_json;
	}

	if (!json_object_is_type(arr, json_type_array)) {
		warn("%s: file:%s is not expected format (not an array)\n",
			__func__, ZEROTOUCH_KEYFILE);
		goto out_put_json;
	}

	/* Check if key already exists */
	len = json_object_array_length(arr);
	for (i = 0; i < len; i++) {
		struct json_object *t = json_object_array_get_idx(arr, i);
		const char *p = json_object_get_string(t);

		if (p && strcmp(p, keystr) == 0) {
			dbg("%s: key '%s' already exists, skipping\n",
			__func__, keystr);
			rc = ZT_PASSPHRASE_ADD_EXISTS;
			goto out_put_json;
		}
	}

	val = json_object_new_string(keystr);
	if (!val) {
		err("%s: failed to create string object\n", __func__);
		goto out_put_json;
	}

	json_object_array_add(arr, val);

	if (json_object_to_file(ZEROTOUCH_KEYFILE, key_json) != 0)
		err("%s: failed to write file:%s\n", __func__, ZEROTOUCH_KEYFILE);
	else
		dbg("%s: appended key '%s' successfully\n", __func__, keystr);

	json_object_put(key_json);
	return ZT_PASSPHRASE_ADD_SUCCESS;

out_put_json:
	json_object_put(key_json);
out:
	return rc;
}
#endif

int dpp_append_uri_file(struct controller *c, char *uri)
{
	struct json_object *uri_json, *arr, *val;
	int rv, len, i;

	uri_json = json_object_from_file(DPP_URI_FILE);
	if (!uri_json) {
		dbg("%s: failed to read json:%s error:%s. "\
		     "Try to generate new\n", __func__, DPP_URI_FILE,
		     json_util_get_last_err());

		uri_json = json_object_new_object();
		if (!uri_json)
			goto out;

		arr = json_object_new_array();
		if (!arr)
			goto out_uri;

		json_object_object_add(uri_json, "uris", arr);
	}

	rv = json_object_object_get_ex(uri_json, "uris", &arr);
	if (!rv || !arr)
		goto out_uri;

	if (!json_object_is_type(arr, json_type_array)) {
		warn("%s: file:%s is not expected format\n", __func__,
			DPP_URI_FILE);
		goto out_uri;
	}

	len = json_object_array_length(arr);
	for (i = 0; i < len; i++) {
		struct json_object *t;
		const char *p;

		t = json_object_array_get_idx(arr, i);
		if (!t) {
			warn("%s: couldn't get entry:%d from uri array", __func__, i);
			continue;
		}

		p = json_object_get_string(t);
		if (strncmp(p, uri, 128))
			continue;
		goto out;
	}

	val = json_object_new_string(uri);
	if (!val)
		goto out_uri;

	json_object_array_add(arr, val);
	if (json_object_to_file(DPP_URI_FILE, uri_json) != 0)
		err("%s: failed to write file:%s\n", __func__, DPP_URI_FILE);
	else
		dbg("%s: appended key '%s' successfully\n", __func__, uri);
out_uri:
	json_object_put(uri_json);
out:
	return 0;
}

int dpp_cntlr_read_uris(struct controller *c)
{
	struct dpp_bootstrap_info *enrollee_bi;
	struct blob_buf uris = { 0 };
	struct blob_attr *cur;
	static const struct blobmsg_policy attr[] = {
		[0] = { .name = "uris", .type = BLOBMSG_TYPE_ARRAY },
	};
	struct blob_attr *tb[ARRAY_SIZE(attr)];
	int rc = -1;
	int rem;

	blob_buf_init(&uris, 0);

	if (!blobmsg_add_json_from_file(&uris, DPP_URI_FILE)) {
		warn("Failed to parse %s\n", DPP_URI_FILE);
		goto out;
	}

	blobmsg_parse(attr, ARRAY_SIZE(attr), tb, blob_data(uris.head), blob_len(uris.head));
	if (!tb[0])
		goto out;

	blobmsg_for_each_attr(cur, tb[0], rem) {
		const char *uri;
		int ret;

		if (blobmsg_type(cur) != BLOBMSG_TYPE_STRING)
			continue;

		uri = blobmsg_get_string(cur);
		dbg("%s: found URI: %s\n", __func__, uri);

		enrollee_bi = calloc(1, sizeof(*enrollee_bi));
		if (!enrollee_bi) {
			fprintf(stderr, "Failed to allocate enrollee_bi\n");
			return UBUS_STATUS_UNKNOWN_ERROR;
		}

		ret = dpp_build_bootstrap_info_from_uri(uri, enrollee_bi);
		if (ret) {
			fprintf(stderr, "Failed to build bootstrap info\n");
			return UBUS_STATUS_UNKNOWN_ERROR;
		}

		ret = dpp_bootstrap_add(c->dpp, enrollee_bi);
		if (ret) {
			fprintf(stderr, "Failed to add bootstrap\n");
			return UBUS_STATUS_UNKNOWN_ERROR;
		}
	}

out:
	blob_buf_free(&uris);
	return rc;
}

int dpp_frame_handler(void *dpp, uint8_t *smac, enum dpp_event ev,
		      uint8_t *frame, size_t framelen)
{
	cntlr_trace(LOG_DPP, "%s: ----->\n", __func__);

	struct controller *c;
	int ret = 0;

	if (!smac) {
		cntlr_warn(LOG_DPP, "%s: Error: smac = NULL\n", __func__);
		ret = -1;
		goto out;
	}

	cntlr_info(LOG_DPP, "%s: DPP_EVENT: event = %d smac = " MACFMT", frame = %p, len = %zu\n",
		__func__, ev, MAC2STR(smac), frame, framelen);

	switch (ev) {
	case DPP_EVENT_TX_FRAME: {
		struct cmdu_buff *resp;
		int frametype = 0;
		struct node *proxy = NULL;

		frametype = dpp_get_frame_type(frame + 1, framelen - 1);
		if (frametype == 255) {
			cntlr_warn(LOG_DPP, "Invalid frametype\n");
			ret = -1;
			goto out;
		}

		cntlr_dbg(LOG_DPP, "DPP frametype %s\n", dpp_frame_type2str(frametype));

		c = dpp_get_ctx_private_data(dpp);
		if (!c) {
			ret = -1;
			goto out;
		}
#ifdef DEBUG
		dump(frame, framelen, "DPP Frame Response");
#endif /* DEBUG */

		/* if peer has private data, it is the proxy node
		 * thus we can infer its a proxied encap DPP message
		 */
		proxy = dpp_get_peer_private_data(dpp, smac);
		if (proxy) {
			resp = cntlr_gen_proxied_encap_dpp(c, smac, frametype,
							   frame, framelen,
							   NULL, 0);
			if (!resp) {
				cntlr_warn(LOG_DPP, "%s: DPP: ----> Error generating proxy encap\n",
					   __func__);
				ret = -1;
				goto out;
			}
		} else {
			resp = cntlr_gen_direct_encap_dpp(c, smac, frame,
							  framelen);
			if (!resp) {
				cntlr_warn(LOG_DPP, "%s: DPP: ----> Error generating direct encap\n",
					  __func__);
				ret = -1;
				goto out;
			}
		}

		switch (frametype) {
		case DPP_PA_AUTHENTICATION_REQ: {
			if (proxy) {
				struct node *n = NULL;

				list_for_each_entry(n, &c->nodelist, list) {
					memcpy(resp->origin, n->almacaddr, 6);
					cntlr_dbg(LOG_DPP, "|%s:%d| send frame to "MACFMT"\n",
						  __func__, __LINE__, MAC2STR(n->almacaddr));
					send_cmdu(c, resp);
				}
				cmdu_free(resp);
				break;
			}
			/* FALLTHROUGH */
		}
		case DPP_PA_AUTHENTICATION_CONF:
		case DPP_PUB_AF_GAS_INITIAL_RESP:
		case DPP_PA_PEER_DISCOVERY_RESP:
			if (proxy)
				memcpy(resp->origin, proxy->almacaddr, 6);
			else
				memcpy(resp->origin, smac, 6);
			send_cmdu(c, resp);
			cmdu_free(resp);
			break;
		default:
			cntlr_warn(LOG_DPP, "%s: DPP: ----> Unknown frame\n",
				   __func__);
			break;
		}
		break;
	}
	case DPP_EVENT_BAD_AUTH:
		cntlr_warn(LOG_DPP, "%s: DPP: BAD AUTH!\n", __func__);
		/* FALLTHROUGH */
	case DPP_EVENT_TIMEOUT: {
		struct dpp_peer *peer = dpp_lookup_peer(dpp, smac);

		cntlr_info(LOG_DPP, "%s: DPP: EVENT TIMEOUT!\n", __func__);

		if (peer) {
			list_del(&peer->list);
			dpp_free_peer(dpp, peer);
		}
		break;
	}
	default:
		break;
	}

out:
	return ret;
}
#endif
#endif
