/*
 * sta.c - STA management
 *
 * Copyright (C) 2020-2024 IOPSYS Software Solutions AB.
 *
 * See LICENSE file for source code license information.
 *
 */

#include "sta.h"

#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <easy/easy.h>
#include <wifidefs.h>

#include "cntlr.h"
#include "steer_module.h"
#include "timer.h"
#include "utils/debug.h"
#include "wifi_dataelements.h"


extern void cntlr_bcn_metrics_timer_cb(atimer_t *t);
extern void cntlr_btm_req_timer_cb(atimer_t *t);
extern void cntlr_sta_ageout_timer_cb(atimer_t *t);

struct sta *cntlr_find_sta(struct hlist_head *table, uint8_t *macaddr)
{
	int idx = sta_hash(macaddr);
	struct sta *s = NULL;

	if (hlist_empty(&table[idx]))
		return NULL;

	hlist_for_each_entry(s, &table[idx], hlist) {
		if (!memcmp(s->macaddr, macaddr, 6))
			return s;
	}

	return NULL;
}

static struct sta *sta_alloc(uint8_t *macaddr)
{
	struct wifi_sta_element *wse = NULL;
	struct steer_sta *ss;
	struct sta *s;

	s = calloc(1, sizeof(*s) + sizeof(*wse) + sizeof(*ss));
	if (!s)
		return NULL;

	memcpy(s->macaddr, macaddr, 6);
	timer_init(&s->bcn_metrics_timer, cntlr_bcn_metrics_timer_cb);
	timer_init(&s->btm_req_timer, cntlr_btm_req_timer_cb);
	timer_init(&s->ageout_timer, cntlr_sta_ageout_timer_cb);
	time(&s->lookup_time);

	s->de_sta = (struct wifi_sta_element *)(s + 1);
	wse = s->de_sta;
	memcpy(wse->macaddr, macaddr, 6);
	INIT_LIST_HEAD(&wse->meas_reportlist);
	INIT_LIST_HEAD(&wse->umetriclist);
	wse->num_meas_reports = 0;
	wse->mapsta.steer_summary.blacklist_attempt_cnt = STEER_STATS_NO_DATA;
	wse->mapsta.steer_summary.blacklist_success_cnt = STEER_STATS_NO_DATA;
	wse->mapsta.steer_summary.blacklist_failure_cnt = STEER_STATS_NO_DATA;
	wse->mapsta.first = 0;
	wse->mapsta.next = 0;
	wse->mapsta.num_steer_hist = 0;

	s->steer_data = (struct steer_sta *)(wse + 1);
	ss = s->steer_data;
	memcpy(ss->macaddr, macaddr, 6);
	ss->meas_reportlist = &s->de_sta->meas_reportlist;
	ss->unassoc_metriclist = &s->de_sta->umetriclist;

	return s;
}

static void sta_free(struct sta *s)
{
	free(s);
}

struct sta *cntlr_add_sta(void *cntlr, struct hlist_head *table, uint8_t *macaddr)
{
	struct sta *s;
	int idx;

#if 0	//TODO
	if (n->sta_count >= LIMIT_STA_COUNT) {
		time_t least_used_sta_time = (time_t)(~(time_t)0);
		struct sta *least_used_sta = NULL;

		list_for_each_entry(s, &n->stalist, list) {
			if ((uint64_t)s->lookup_time < (uint64_t)least_used_sta_time) {
				least_used_sta = s;
				least_used_sta_time = s->lookup_time;
			}
		}

		if (least_used_sta) {
			cntlr_dbg(LOG_STA, "%s: remove least used STA " MACFMT
					" to add new STA " MACFMT "\n", __func__,
					MAC2STR(least_used_sta->de_sta->macaddr),
					MAC2STR(macaddr));

			node_remove_sta(c, n, least_used_sta);
		} else {
			//Why ?
			dbg("%s: failed to find least used sta\n", __func__);
		}
	}
#endif
	if (WARN_ON(hwaddr_is_zero(macaddr))) {
		/* should never happen */
		return NULL;
	}

	s = sta_alloc(macaddr);
	if (!s)
		return NULL;

	idx = sta_hash(macaddr);
	hlist_add_head(&s->hlist, &table[idx]);
	s->cntlr = cntlr;
	s->state = STA_UNKNOWN;

	info("%s: New client " MACFMT " connected\n",
	     __func__, MAC2STR(macaddr));

	return s;
}

void cntlr_del_sta_hash(struct hlist_head *table, uint8_t *macaddr)
{
	int idx;
	struct hlist_node *tmp = NULL;
	struct sta *s = NULL;
	bool found = false;

	idx = sta_hash(macaddr);
	if (!hlist_empty(&table[idx])) {
		hlist_for_each_entry_safe(s, tmp, &table[idx], hlist) {
			if (!memcmp(s->macaddr, macaddr, 6)) {
				hlist_del(&s->hlist, &table[idx]);
				found = true;
			}
		}
	}

	if (!found) {
		cntlr_warn(LOG_STA, "%s: STA " MACFMT " not found in table\n",
				  __func__, MAC2STR(macaddr));
	}
}

void cntlr_free_sta(struct sta *del)
{
	del->cntlr = NULL;
	sta_free(del);
}

void cntlr_del_sta(struct hlist_head *table, struct sta *del)
{
	cntlr_del_sta_hash(table, del->macaddr);
	cntlr_free_sta(del);
}

int sta_link_metrics_process(struct sta *s)
{
	//TODO:

	return 0;
}

void sta_free_bcn_metrics(struct sta *s)
{
	cntlr_trace(LOG_STA, "%s: --->\n", __func__);
 
	struct wifi_sta_meas_report *b = NULL, *tmp;

	if (!s || !s->de_sta) {
		cntlr_warn(LOG_STA, "%s: Unexpected empty STA reference!\n", __func__);
		return;
	}

	list_for_each_entry_safe(b, tmp, &s->de_sta->meas_reportlist, list) {
		list_del(&b->list);
		free(b);
		s->de_sta->num_meas_reports--;
	}
}

void sta_free_usta_metrics(struct sta *s)
{
	struct unassoc_sta_metrics *u = NULL, *tmp;

	list_for_each_entry_safe(u, tmp, &s->de_sta->umetriclist, list) {
		list_del(&u->list);
		free(u);
	}
}

void sta_free_assoc_frame(struct sta *s)
{
	cntlr_trace(LOG_STA, "%s: --->\n", __func__);

	if (!s || !s->de_sta) {
		cntlr_warn(LOG_STA, "%s: Unexpected empty STA reference!\n", __func__);
		return;
	}

	free(s->de_sta->reassoc_frame);
	s->de_sta->reassoc_framelen = 0;
}

int sta_inc_ref(struct sta *s)
{
	s->nref++;

	if (timer_pending(&s->ageout_timer))
		timer_del(&s->ageout_timer);

	return s->nref;
}

int sta_dec_ref(struct sta *s)
{
	struct controller *c = s->cntlr;

	if (WARN_ON(s->nref == 0))
		return 0;

	s->nref--;
	if (s->nref == 0)
		timer_set(&s->ageout_timer, c->cfg.stale_sta_timeout * 1000);

	return s->nref;
}
