/**
 * @file cntlr_commands.h - mapcontroller commands attributes.
 *
 * Copyright (C) 2023 IOPSYS Software Solutions AB. All rights reserved.
 *
 * Author: anjan.chanda@iopsys.eu
 *
 */

#ifndef CNTLR_COMMANDS_H
#define CNTLR_COMMANDS_H

#include <stddef.h>

#include "cntlr_commands_impl.h"

/**
 * @defgroup cmdattrs Controller command attributes
 * @defgroup cmdlist Controller commands
 */

/**
 * @enum help_attrs
 * @brief help attributes.
 *
 * @ingroup cmdattrs
 * @HELP_ATTR_COMMAND_NAME: command name.
 * @NUM_ATTRS_HELP: Number of help attributes.
 */
enum help_attrs {
	HELP_ATTR_COMMAND_NAME,
	NUM_ATTRS_HELP
};

/**
 * @enum log_attrs
 * @brief logging attributes.
 *
 * @ingroup cmdattrs
 * @LOG_ATTR_FEATURE: feature to log.
 * @LOG_ATTR_LEVEL: log level.
 * @NUM_ATTRS_LOG: Number of log attributes.
 */
enum log_attrs {
	LOG_ATTR_FEATURE,
	LOG_ATTR_LEVEL,
	NUM_ATTRS_LOG
};

/**
 * @enum ap_caps_attrs
 * @brief AP Capability Query attributes.
 *
 * @ingroup cmdattrs
 * @AP_CAPS_ATTR_AGENT: macaddress of Agent.
 * @NUM_ATTRS_AP_CAPS: Number of AP caps attributes.
 */
enum query_ap_caps_attrs {
	AP_CAPS_ATTR_AGENT,
	NUM_ATTRS_AP_CAPS
};

/**
 * @enum query_sta_caps_attrs
 * @brief STA Capability Query attributes
 *
 * @ingroup cmdattrs
 * @STA_CAPS_ATTR_AGENT: Agent's AL-address.
 * @STA_CAPS_ATTR_STA: STA macaddress.
 * @STA_CAPS_ATTR_BSSID: BSSID of an AP interface in Agent.
 * @NUM_ATTRS_STA_CAPS: Number of sta caps attributes.
 */
enum query_sta_caps_attrs {
	STA_CAPS_ATTR_AGENT,
	STA_CAPS_ATTR_STA,
	STA_CAPS_ATTR_BSSID,
	NUM_ATTRS_STA_CAPS
};

/**
 * @enum query_bsta_caps_attrs
 * @brief Backhaul-STA Capability Query attributes
 *
 * @ingroup cmdattrs
 * @BSTA_CAPS_ATTR_AGENT: Agent's AL-address.
 * @NUM_ATTRS_BSTA_CAPS: Number of bsta-caps attributes.
 */
enum query_bsta_caps_attrs {
	BSTA_CAPS_ATTR_AGENT,
	NUM_ATTRS_BSTA_CAPS
};

/**
 * @enum query_ap_metrics_attrs
 * @brief AP Metrics Query attributes
 *
 * @ingroup cmdattrs
 * @AP_METRICS_ATTR_AGENT: Agent's AL-address.
 * @AP_METRICS_ATTR_BSSIDS: array of BSSIDs in Agent.
 * @AP_METRICS_ATTR_RADIOS: array of Radios in Agent.
 */
enum query_ap_metrics_attrs {
	AP_METRICS_ATTR_AGENT,
	AP_METRICS_ATTR_BSSIDS,
	AP_METRICS_ATTR_RADIOS,
	NUM_ATTRS_AP_METRICS
};

/**
 * @enum query_sta_metrics_attrs
 * @brief STA Metrics Query attributes
 *
 * @ingroup cmdattrs
 * @STA_METRICS_ATTR_AGENT: Agent's AL-address.
 * @AP_METRICS_ATTR_STA: macaddress of STA.
 * @NUM_ATTRS_STA_METRICS: number of STA metrics attributes.
 */
enum query_sta_metrics_attrs {
	STA_METRICS_ATTR_AGENT,
	STA_METRICS_ATTR_STA,
	NUM_ATTRS_STA_METRICS
};

/**
 * @enum unassoc_sta_metrics_attrs
 * @brief Unassociated STA Metrics Query attributes
 *
 * @ingroup cmdattrs
 * @UNASSOC_STA_METRICS_ATTR_AGENT: Agent's AL-address.
 * @UNASSOC_STA_METRICS_ATTR_OPCLASS: Opclass where to measure unassociated sta.
 * @UNASSOC_STA_METRICS_ATTR_CHANNEL: Channel in opclass.
 * @UNASSOC_STA_METRICS_ATTR_STALIST: array of STA macaddresses.
 * @NUM_ATTRS_UNASSOC_STA_METRICS: number of unassociated-sta metrics attributes.
 */
enum unassoc_sta_metrics_attrs {
	UNASSOC_STA_METRICS_ATTR_AGENT,
	UNASSOC_STA_METRICS_ATTR_OPCLASS,
	UNASSOC_STA_METRICS_ATTR_CHANNEL,
	UNASSOC_STA_METRICS_ATTR_STALIST,
	NUM_ATTRS_UNASSOC_STA_METRICS
};

/**
 * @enum channel_pref_attrs
 * @brief Channel Preference Query attributes
 *
 * @ingroup cmdattrs
 * @CHANNEL_PREF_ATTR_AGENT: Agent's AL-address.
 * @NUM_ATTRS_CHANNEL_PREF: Number of Channel preference attributes.
 */
enum channel_pref_attrs {
	CHANNEL_PREF_ATTR_AGENT,
	NUM_ATTRS_CHANNEL_PREF,
};

/**
 * @enum query_beacon_metrics_attrs
 * @brief Beacon Metrics Query attributes
 *
 * @ingroup cmdattrs
 * @BCN_METRICS_AGENT: Agent's AL-address.
 * @BCN_METRICS_ATTR_STA: STA macaddress.
 * @BCN_METRICS_ATTR_OPCLASS: Opclass.
 * @BCN_METRICS_ATTR_CHANNEL: Channel.
 * @BCN_METRICS_ATTR_BSSID: BSSID.
 * @BCN_METRICS_ATTR_REPORTING_DETAIL: Reporting detail to be included.
 * @BCN_METRICS_ATTR_SSID: SSID.
 * @BCN_METRICS_ATTR_CHAN_REPORT: include channel report in response.
 * @BCN_METRICS_ATTR_ELEMENT_IDS: include optional element ids.
 * @NUM_ATTRS_BCN_METRICS: Number fo Beacon metrics query attributes.
 */
enum query_beacon_metrics_attrs {
	BCN_METRICS_ATTR_AGENT,
	BCN_METRICS_ATTR_STA,
	BCN_METRICS_ATTR_OPCLASS,
	BCN_METRICS_ATTR_CHANNEL,
	BCN_METRICS_ATTR_BSSID,
	BCN_METRICS_ATTR_REPORTING_DETAIL,
	BCN_METRICS_ATTR_SSID,
	BCN_METRICS_ATTR_CHAN_REPORT,
	BCN_METRICS_ATTR_ELEMENT_IDS,
	NUM_ATTRS_BCN_METRICS,
};

/**
 * @enum channel_sel_attrs
 * @brief Channel Selection Request attributes
 *
 * @ingroup cmdattrs
 * @CHANNEL_SEL_ATTR_AGENT: Agent's AL-address.
 * @CHANNEL_SEL_ATTR_RADIO: Radio
 * @CHANNEL_SEL_ATTR_CHANNEL: Control channel
 * @CHANNEL_SEL_ATTR_BANDWIDTH: Bandwidth
 * @NUM_ATTRS_CHANNEL_SEL: Number of Channel Selection request attributes.
 */
enum channel_sel_attrs {
	CHANNEL_SEL_ATTR_AGENT,
	CHANNEL_SEL_ATTR_RADIO,
	CHANNEL_SEL_ATTR_CHANNEL,
	CHANNEL_SEL_ATTR_BANDWIDTH,
	NUM_ATTRS_CHANNEL_SEL
};

/**
 * @enum channel_clearing_attrs
 * @brief DFS Channel clearing attributes
 *
 * @ingroup cmdattrs
 * @CHANNEL_CLEARING_ATTR_AGENT: Agent's AL-address.
 * @NUM_ATTRS_CHANNEL_CLEARING: Number of Channel clearing attributes.
 */
enum channel_clearing_attrs {
	CHANNEL_CLEARING_ATTR_AGENT,
	NUM_ATTRS_CHANNEL_CLEARING,
};

/**
 * @enum channel_acs_attrs
 * @brief ACS Channel attributes
 *
 * @ingroup cmdattrs
 * @CHANNEL_ACS_ATTR_AGENT: Agent's AL-address.
 * @CHANNEL_ACS_ATTR_BAND: band in Agent.
 * @CHANNEL_ACS_ATTR_SKIP_DFS: skip DFS channels.
 * @CHANNEL_ACS_ATTR_OPCLASS: requested opclass.
 * @CHANNEL_ACS_ATTR_BANDWIDTH: requested bandwidth.
 * @CHANNEL_ACS_ATTR_PREVENT_CAC: prevent DFS CAC.
 * @NUM_ATTRS_CHANNEL_ACS: Number of Channel ACS attributes.
 */
enum channel_acs_attrs {
	CHANNEL_ACS_ATTR_AGENT,
	CHANNEL_ACS_ATTR_BAND,
	CHANNEL_ACS_ATTR_SKIP_DFS,
	CHANNEL_ACS_ATTR_OPCLASS,
	CHANNEL_ACS_ATTR_BANDWIDTH,
	CHANNEL_ACS_ATTR_PREVENT_CAC,
	NUM_ATTRS_CHANNEL_ACS,
};

/**
 * @enum steer_attrs
 * @brief Steer Request attributes
 *
 * @ingroup cmdattrs
 * @STEER_ATTR_AGENT: Agent's AL-address.
 * @STEER_ATTR_STA: STA's macaddress.
 * @STEER_ATTR_FROM_BSSID: Source BSSID in Agent.
 * @STEER_ATTR_TARGET_AGENT: Target Agent's AL-address.
 * @STEER_ATTR_TARGET_BSSID: Target BSSID.
 * @STEER_ATTR_BTM_ABRIDGED: Abridged bit in bssinfo of BTM Request.
 * @STEER_ATTR_DISASSOC_TIMEOUT: Disassoc Immenent and Disassoc timer in TUs.
 * @STEER_ATTR_MBO_REASON: Transition reason code for Agile Multiband STA.
 * @NUM_ATTRS_STEER: Number of Steer Request attributes.
 */
enum steer_attrs {
	STEER_ATTR_AGENT,
	STEER_ATTR_STA,
	STEER_ATTR_FROM_BSSID,
	STEER_ATTR_TARGET_AGENT,
	STEER_ATTR_TARGET_BSSID,
	STEER_ATTR_BTM_ABRIDGED,
	STEER_ATTR_DISASSOC_TIMEOUT,
	STEER_ATTR_MBO_REASON,
	NUM_ATTRS_STEER,
};

/**
 * @enum steer_op_attrs
 * @brief Steer Opportunity Request attributes
 *
 * @ingroup cmdattrs
 * @STEER_OP_ATTR_AGENT: Agent's AL-address.
 * @STEER_OP_ATTR_STA: STA's macaddress.
 * @STEER_OP_ATTR_FROM_BSSID: Source BSSID in Agent.
 * @STEER_OP_ATTR_BTM_ABRIDGED: Abridged bit in bssinfo of BTM Request.
 * @STEER_OP_ATTR_WINDOW: Steering opportunity time window in seconds.
 * @STEER_OP_ATTR_DISASSOC_TIMEOUT: Disassoc Immenent and Disassoc timer in TUs.
 * @STEER_OP_ATTR_MBO_REASON: Transition reason code for Agile Multiband STA.
 * @NUM_ATTRS_STEER_OP: Number of Steer Request attributes.
 */
enum steer_op_attrs {
	STEER_OP_ATTR_AGENT,
	STEER_OP_ATTR_STA,
	STEER_OP_ATTR_FROM_BSSID,
	STEER_OP_ATTR_BTM_ABRIDGED,
	STEER_OP_ATTR_WINDOW,
	STEER_OP_ATTR_DISASSOC_TIMEOUT,
	STEER_OP_ATTR_MBO_REASON,
	NUM_ATTRS_STEER_OP,
};

/**
 * @enum assoc_control_attrs
 * @brief STA Association Control request attributes
 *
 * @ingroup cmdattrs
 * @ASSOC_CONTROL_ATTR_AGENT: Agent's AL-address.
 * @ASSOC_CONTROL_ATTR_BSSID: BSSID in Agent to apply association control.
 * @ASSOC_CONTROL_ATTR_MODE: Mode of association control.
 * @ASSOC_CONTROL_ATTR_VALIDITY_TIME: Amount of time in seconds till association control applies.
 * @ASSOC_CONTROL_ATTR_STA: Array of STA macaddresses for association control.
 * @NUM_ATTRS_ASSOC_CONTROL: Number of Association Control request attributes.
 */
enum assoc_control_attrs {
	ASSOC_CONTROL_ATTR_AGENT,
	ASSOC_CONTROL_ATTR_BSSID,
	ASSOC_CONTROL_ATTR_MODE,
	ASSOC_CONTROL_ATTR_VALIDITY_TIME,
	ASSOC_CONTROL_ATTR_STA,
	NUM_ATTRS_ASSOC_CONTROL,
};

/**
 * @enum steer_backhaul_attrs
 * @brief Backhaul Steer Request attributes
 *
 * @ingroup cmdattrs
 * @BSTA_STEER_ATTR_AGENT: Agent's AL-address.
 * @BSTA_ATTR_TARGET_AGENT: Target Agent's AL-address.
 * @BSTA_ATTR_TARGET_BAND: Radio band in target Agent.
 * @STEER_ATTR_TARGET_BSSID: Target BSSID.
 * @BSTA_STEER_ATTR_BSTA: bSTA's macaddress.
 * @BSTA_STEER_ATTR_TIMEOUT: Timeout in ms.
 * @NUM_ATTRS_BSTA_STEER: Number of bSTA Steer Request attributes.
 */
enum steer_backhaul_attrs {
	BSTA_STEER_ATTR_AGENT,
	BSTA_STEER_ATTR_TARGET_AGENT,
	BSTA_STEER_ATTR_TARGET_BAND,
	BSTA_STEER_ATTR_TARGET_BSSID,
	BSTA_STEER_ATTR_BSTA,
	BSTA_STEER_ATTR_TIMEOUT,
	NUM_ATTRS_BSTA_STEER,
};

/**
 * @enum scan_attrs
 * @brief Scan Request attributes
 *
 * @ingroup cmdattrs
 * @SCAN_ATTR_AGENT: Agent's AL-address.
 * @SCAN_ATTR_BAND: radio frequency band viz. 2, 5 or 6.
 * @SCAN_ATTR_RADIO: array of Radio-ID or radio macaddress.
 * @SCAN_ATTR_OPCLASS: array of opclasses to scan.
 * @SCAN_ATTR_CHANNEL: array of channels to scan.
 * @SCAN_ATTR_FRESH_SCAN: whether fresh scanning should be triggered.
 * @NUM_ATTRS_SCAN: Number of Scan Request attributes.
 */
enum scan_attrs {
	SCAN_ATTR_AGENT,
	SCAN_ATTR_BAND,
	SCAN_ATTR_RADIO,
	SCAN_ATTR_OPCLASS,
	SCAN_ATTR_CHANNEL,
	SCAN_ATTR_FRESH_SCAN,
	NUM_ATTRS_SCAN,
};

/**
 * @enum scanresults_attrs
 * @brief Scan Results attributes
 *
 * @ingroup cmdattrs
 * @SCANRESULTS_ATTR_AGENT: Agent's AL-address.
 * @SCANRESULTS_ATTR_BAND: Radio band 2, 5 or 6.
 * @SCANRESULTS_ATTR_RADIO: Radio-ID or radio macaddress.
 * @NUM_ATTRS_SCANRESULTS: Number of Scan-Results attributes.
 */
enum scanresults_attrs {
	SCANRESULTS_ATTR_AGENT,
	SCANRESULTS_ATTR_BAND,
	SCANRESULTS_ATTR_RADIO,
	NUM_ATTRS_SCANRESULTS,
};

/**
 * @enum cac_start_attrs
 * @brief CAC Request attributes
 *
 * @ingroup cmdattrs
 * @CAC_REQUEST_ATTR_AGENT: Agent's AL-address.
 * @CAC_REQUEST_ATTR_RADIO: radio-ID or radio macaddress in Agent.
 * @CAC_REQUEST_ATTR_OPCLASS: opclass.
 * @CAC_REQUEST_ATTR_CHANNEL: control channel.
 * @CAC_REQUEST_ATTR_METHOD: CAC method.
 * @CAC_REQUEST_ATTR_ACTION: CAC completion action.
 * @NUM_ATTRS_CAC_REQUEST: Number of CAC Request attributes.
 */
enum cac_start_attrs {
	CAC_START_ATTR_AGENT,
	CAC_START_ATTR_RADIO,
	CAC_START_ATTR_OPCLASS,
	CAC_START_ATTR_CHANNEL,
	CAC_START_ATTR_METHOD,
	CAC_START_ATTR_ACTION,
	NUM_ATTRS_CAC_REQUEST,
};

/**
 * @enum cac_stop_attrs
 * @brief CAC Terminate attributes
 *
 * @ingroup cmdattrs
 * @CAC_TERMINATE_ATTR_AGENT: Agent's AL-address.
 * @CAC_TERMINATE_ATTR_RADIO: radio-ID or radio macaddress in Agent.
 * @CAC_TERMINATE_ATTR_OPCLASS: opclass.
 * @CAC_TERMINATE_ATTR_CHANNEL: control channel.
 * @NUM_ATTRS_CAC_TERMINATE: Number of CAC Terminate attributes.
 */
enum cac_stop_attrs {
	CAC_TERMINATE_ATTR_AGENT,
	CAC_TERMINATE_ATTR_RADIO,
	CAC_TERMINATE_ATTR_OPCLASS,
	CAC_TERMINATE_ATTR_CHANNEL,
	NUM_ATTRS_CAC_TERMINATE,
};

/**
 * @enum send_hld_attrs
 * @brief send Higher Layer Data (HLD) attributes
 *
 * @ingroup cmdattrs
 * @HLD_ATTR_AGENT: Agent's AL-address.
 * @HLD_ATTR_PROTOCOL: Higher Layer Data protocol number.
 * @HLD_ATTR_DATA: Higher Layer Data payload.
 * @NUM_ATTRS_HLD: Number of HLD attributes.
 */
enum send_hld_attrs {
	HLD_ATTR_AGENT,
	HLD_ATTR_PROTOCOL,
	HLD_ATTR_DATA,
	NUM_ATTRS_HLD,
};

/**
 * @enum send_combined_metrics_attrs
 * @brief send Combined Infrastructure Metrics attributes
 *
 * @ingroup cmdattrs
 * @COMBINED_METRICS_ATTR_AGENT: Agent's AL-address.
 * @COMBINED_METRICS_ATTR_BSSID: BSSID.
 * @NUM_ATTRS_COMBINED_METRICS: Number of Combined Infrestructure Metrics attributes.
 */
enum send_combined_metrics_attrs {
	COMBINED_METRICS_ATTR_AGENT,
	COMBINED_METRICS_ATTR_BSSID,
	NUM_ATTRS_COMBINED_METRICS,
};

/**
 * @enum topology_query_attrs
 * @brief Topology Query attributes
 *
 * @ingroup cmdattrs
 * @TOPOLOGY_QUERY_ATTR_AGENT: Agent's AL-address.
 * @NUM_ATTRS_TOPOLOGY_QUERY: Number of Topology Query attributes.
 */
enum topology_query_attrs {
	TOPOLOGY_QUERY_ATTR_AGENT,
	NUM_ATTRS_TOPOLOGY_QUERY,
};

/**
 * @enum dump_steer_summary_attrs
 * @brief Dump STA Steering summary attributes
 *
 * @ingroup cmdattrs
 * @STEER_SUMMARY_ATTR_STA: STA macaddress.
 * @NUM_ATTRS_STEER_SUMMARY: Number of Steer summary attributes.
 */
enum dump_steer_summary_attrs {
	STEER_SUMMARY_ATTR_STA,
	NUM_ATTRS_STEER_SUMMARY,
};

/**
 * @enum dump_steer_history_attrs
 * @brief Dump STA Steering history attributes
 *
 * @ingroup cmdattrs
 * @STEER_HISTORY_ATTR_STA: STA macaddress.
 * @NUM_ATTRS_STEER_HISTORY: Number of Steer History attributes.
 */
enum dump_steer_history_attrs {
	STEER_HISTORY_ATTR_STA,
	NUM_ATTRS_STEER_HISTORY,
};

/**
 * @enum dump_unassoc_sta_metrics_attrs
 * @brief Dump Unassociated STA link metrics attributes
 *
 * @ingroup cmdattrs
 * @UNASSOC_STA_METRICS_DUMP_ATTR_STA: STA macaddress.
 * @NUM_ATTRS_UNASSOC_STA_METRICS_DUMP: Number of Unassociated STA metrics dump attributes.
 */
enum dump_unassoc_sta_metrics_attrs {
	UNASSOC_STA_METRICS_DUMP_ATTR_STA,
	NUM_ATTRS_UNASSOC_STA_METRICS_DUMP,
};

/**
 * @enum dump_beacon_metrics_attrs
 * @brief Dump Unassociated STA link metrics attributes
 *
 * @ingroup cmdattrs
 * @BCN_METRICS_DUMP_ATTR_STA: STA macaddress.
 * @NUM_ATTRS_BCN_METRICS_DUMP: Number of Beacon metrics dump attributes.
 */
enum dump_beacon_metrics_attrs {
	BCN_METRICS_DUMP_ATTR_STA,
	NUM_ATTRS_BCN_METRICS_DUMP,
};


/**
 * @enum dump_policy_attrs
 * @brief Dump Policy attributes
 *
 * @ingroup cmdattrs
 * @DUMP_POLICY_AGENT: target agent ID
 * @NUM_ATTRS_DUMP_POLICY: Number of attributes.
 */
enum dump_policy_attrs {
	DUMP_POLICY_AGENT,
	NUM_ATTRS_DUMP_POLICY,
};

#if (EASYMESH_VERSION >= 6)
/**
 * @enum dump_mlo_caps_attrs
 * @brief Dump agent Wi-Fi 7 capabilities
 *
 * @ingroup cmdattrs
 * @MLO_CAPABILITIES_ATTR_AGENT: Agent's AL-address.
 * @NUM_ATTRS_MLO_CAPS_DUMP: Number of Beacon metrics dump attributes.
 */
enum dump_mlo_caps_attrs {
	MLO_CAPABILITIES_ATTR_AGENT,
	NUM_ATTRS_MLO_CAPS_DUMP,
};

#endif

#if (EASYMESH_VERSION > 2)
#ifdef USE_LIBDPP
/**
 * @enum dpp_enrollee_uri_attrs
 * @brief Provide a DPP enrollee bootstrapping information URI for Ethernet onboarding
 *
 * @ingroup cmdattrs
 * @DPP_ENROLLEE_URI_ATTR_URI: DPP enrollee bootstrapping information URI.
 * @DPP_ENROLLEE_URI_ATTR_TYPE: DPP bootstrapping information URI type.
 * @NUM_ATTRS_DPP_ENROLLEE_URI: Number of dpp enrollee uri attributes.

 */
enum dpp_enrollee_uri_attrs {
	DPP_ENROLLEE_URI_ATTR_URI,
	DPP_ENROLLEE_URI_ATTR_TYPE,
	NUM_ATTRS_DPP_ENROLLEE_URI,
};

/**
 * @enum dpp_advertise_cce_attrs
 * @brief Instruct an agent to broadcast CCE in its beacons
 *
 * @ingroup cmdattrs
 * @DPP_ADVERTISE_CCE_ATTR_AGENT: Agent's AL-address.
 * @DPP_ADVERTISE_CCE_ATTR_ENABLE: Enable (1) or disable (0) CCE
 * @NUM_ATTRS_DPP_ADVERTISE_CCE: Number of dpp advertise cce attributes.

 */
enum dpp_advertise_cce_attrs {
	DPP_ADVERTISE_CCE_ATTR_AGENT,
	DPP_ADVERTISE_CCE_ATTR_ENABLE,
	NUM_ATTRS_DPP_ADVERTISE_CCE,
};
#endif
#endif

#ifdef EASYMESH_VENDOR_EXT
/**
 * @enum disassociate_sta_attrs
 * @brief Disassociate STA attributes
 *
 * @ingroup cmdattrs
 * @DISASSOCIATE_STA_ATTR_AGENT: Agent's AL-address.
 * @DISASSOCIATE_STA_ATTR_STA: STA macaddress.
 * @DISASSOCIATE_STA_ATTR_REASON: Disassociate reason.
 * @NUM_ATTRS_DISASSOCIATE_STA: Number of disassociate STA attributes.
 */
enum disassociate_sta_attrs {
	DISASSOCIATE_STA_ATTR_AGENT,
	DISASSOCIATE_STA_ATTR_STA,
	DISASSOCIATE_STA_ATTR_REASON,
	NUM_ATTRS_DISASSOCIATE_STA,
};

/**
 * @enum reset_agent_attrs
 * @brief Reset Agent attributes
 *
 * @ingroup cmdattrs
 * @RESET_AGENT_ATTR_AGENT: Agent's AL-address.
 * @NUM_ATTRS_RESET_AGENT: Number of reset_agent attributes.
 */
enum reset_agent_attrs {
	RESET_AGENT_ATTR_AGENT,
	NUM_ATTRS_RESET_AGENT,
};
#endif

/**
 * Parse command with optional JSON arguments.
 * @param[in] command command name
 * @param[in] args optional arguments to the command provided as JSON string
 * @param[in|out] tb output array of parsed args attributes
 *
 * @return 0 on success, or -1 on error.
 */
//int controller_command_parse(const char *command, const char *args, struct blob_attr **tb);
int controller_command_parse(const char *command, void *args, struct blob_attr **tb);

extern int num_controller_cmds;


struct controller_command *controller_command_lookup(const char *command);

size_t num_controller_commands(void *cntlr);
struct controller_command *controller_command_last(void);
struct controller_command *controller_command_first(void);

#define controller_for_each_command(p)	\
	for (p = controller_command_first(); p != controller_command_last(); p++)


int controller_command_get_policy(const char *command, void **policy, size_t *num_policy);


#endif /* CNTLR_COMMANDS_H */
