/* SPDX-License-Identifier: LGPL-2.1-only */
/*
 * debug.h - defines macros for debugging.
 *
 * Copyright (C) 2020-2024 Iopsys Software Solutions AB. All rights reserved.
 */
#ifndef LIBWIFI_DEBUG_H
#define LIBWIFI_DEBUG_H

#define ERR_LEVEL   (3)
#define WARN_LEVEL  (4)
#define INFO_LEVEL  (6)
#define DBG_LEVEL   (7)

#ifdef __cplusplus
extern "C" {
#endif

extern void log_stderr(int level, const char *fmt, ...);

#define pr_error(...)	log_stderr(ERR_LEVEL, __VA_ARGS__)
#define pr_warn(...)	log_stderr(WARN_LEVEL, __VA_ARGS__)
#define pr_info(...)	log_stderr(INFO_LEVEL, __VA_ARGS__)
#define pr_debug(...)	log_stderr(DBG_LEVEL, __VA_ARGS__)

#define WARN_ON(cond) \
({ \
	typeof(cond) __c = (cond); \
	if (__c) \
		pr_warn("%sL%d@%s: [%s] failed\n", __FILE__, __LINE__, __func__, #cond); \
	__c; \
})

#ifdef LIBEASY_TRACE_TIME
#define ENTER() clock_t _clk_end, _clk_begin; \
		_clk_begin = clock(); \
		pr_error("%s called\n", __func__)

#define EXIT(__res__) _clk_end = clock(); \
			pr_error("%s ret %d consume %u us\n\n", __func__, __res__, \
				 (unsigned int) _clk_end - _clk_begin)

#define EXITV() _clk_end = clock(); \
		pr_error("%s exit consume %u us\n\n", __func__, \
			 (unsigned int) _clk_end - _clk_begin)
#else
#define ENTER()
#define EXIT(__res__)
#define EXITV()
#endif


#ifdef __cplusplus
}
#endif
#endif /* LIBWIFI_DEBUG_H */
