/* SPDX-License-Identifier: LGPL-2.1-only */
/*
 * opclass.c - implements WiFi opclass as per Table-E4 of IEEE802.11 Std.
 *
 * Copyright (C) 2019-2024 Iopsys Software Solutions AB. All rights reserved.
 */
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>

#include <easy/easy.h>
#include "wifiutils.h"


const struct wifi_opclass wifi_opclass_global[] = {
	{ 81, 81, BAND_2, BW20, EXTCH_NONE, {20, 13, {{1, {1}},
						      {2, {2}},
						      {3, {3}},
						      {4, {4}},
						      {5, {5}},
						      {6, {6}},
						      {7, {7}},
						      {8, {8}},
						      {9, {9}},
						      {10, {10}},
						      {11, {11}},
						      {12, {12}},
						      {13, {13}}}}},
	{ 82, 82, BAND_2, BW20, EXTCH_NONE, {20, 1, {{14, {14}}}}},
	{ 83, 83, BAND_2, BW40, EXTCH_ABOVE, {20, 9, {{1, {1}},
						      {2, {2}},
						      {3, {3}},
						      {4, {4}},
						      {5, {5}},
						      {6, {6}},
						      {7, {7}},
						      {8, {8}},
						      {9, {9}}}}},
	{ 84, 84, BAND_2, BW40, EXTCH_BELOW, {20, 9, {{5, {5}},
						      {6, {6}},
						      {7, {7}},
						      {8, {8}},
						      {9, {9}},
						      {10, {10}},
						      {11, {11}},
						      {12, {12}},
						      {13, {13}}}}},

	{ 115, 115, BAND_5, BW20, EXTCH_NONE, {23, 4, {{36, {36}},
						       {40, {40}},
						       {44, {44}},
						       {48, {48}}}}},
	{ 116, 116, BAND_5, BW40, EXTCH_ABOVE, {23, 2, {{36, {36}},
							{44, {44}}}}},
	{ 117, 117, BAND_5, BW40, EXTCH_BELOW, {23, 2, {{40, {40}},
							{48, {48}}}}},
	{ 118, 118, BAND_5, BW20, EXTCH_NONE, {23, 4, {{52, {52}},
						       {56, {56}},
						       {60, {60}},
						       {64, {64}}}}},
	{ 119, 119, BAND_5, BW40, EXTCH_ABOVE, {23, 2, {{52, {52}},
							{60, {60}}}}},
	{ 120, 120, BAND_5, BW40, EXTCH_BELOW, {23, 2, {{56, {56}},
							{64, {64}}}}},
	{ 121, 121, BAND_5, BW20, EXTCH_NONE, {30, 12, {{100, {100}},
							{104, {104}},
							{108, {108}},
							{112, {112}},
							{116, {116}},
							{120, {120}},
							{124, {124}},
							{128, {128}},
							{132, {132}},
							{136, {136}},
							{140, {140}},
							{144, {144}}}}},
	{ 122, 122, BAND_5, BW40, EXTCH_ABOVE, {30, 6, {{100, {100}},
							{108, {108}},
							{116, {116}},
							{124, {124}},
							{132, {132}},
							{140, {140}}}}},
	{ 123, 123, BAND_5, BW40, EXTCH_BELOW, {30, 6, {{104, {104}},
							{112, {112}},
							{120, {120}},
							{128, {128}},
							{136, {136}},
							{144, {144}}}}},
	{ 124, 124, BAND_5, BW20, EXTCH_NONE, {13, 4, {{149, {149}},
						       {153, {153}},
						       {157, {157}},
						       {161, {161}}}}},
	{ 125, 125, BAND_5, BW20, EXTCH_NONE, {13, 8, {{149, {149}},
						       {153, {153}},
						       {157, {157}},
						       {161, {161}},
						       {165, {165}},
						       {169, {169}},
						       {173, {173}},
						       {177, {177}}}}},
	{ 126, 126, BAND_5, BW40, EXTCH_ABOVE, {13, 4, {{149, {149}},
							{157, {157}},
						        {165, {165}},
							{173, {173}}}}},

	{ 127, 127, BAND_5, BW40, EXTCH_BELOW, {13, 4, {{153, {153}},
						        {161, {161}},
						        {169, {169}},
							{177, {177}}}}},

	{ 128, 128, BAND_5, BW80, EXTCH_AUTO, {23, 7, {{36, {36, 40, 44, 48}},
						       {52, {52, 56, 60, 64}},
						       {100, {100, 104, 108, 112}},
						       {116, {116, 120, 124, 128}},
						       {132, {132, 136, 140, 144}},
						       {149, {149, 153, 157, 161}},
						       {165, {165, 169, 173, 177}}}}},

	{ 129, 129, BAND_5, BW160, EXTCH_AUTO, {23, 3, {{36, {36, 40, 44, 48, 52, 56, 60, 64}},
						        {100, {100, 104, 108, 112, 116, 120, 124, 128}},
						        {149, {149, 153, 157, 161, 165, 169, 173, 177}}}}},

	{ 130, 130, BAND_5, BW8080, EXTCH_AUTO, {23, 7, {{36, {36, 40, 44, 48}},
							 {52, {52, 56, 60, 64}},
							 {100, {100, 104, 108, 112}},
							 {116, {116, 120, 124, 128}},
							 {132, {132, 136, 140, 144}},
							 {149, {149, 153, 157, 161}},
							 {165, {165, 169, 173, 177}}}}},

	{ 131, 131, BAND_6, BW20, EXTCH_NONE, {23, 59, {{1, {1}},
						       {5, {5}},
						       {9, {9}},
						       {13, {13}},
						       {17, {17}},
						       {21, {21}},
						       {25, {25}},
						       {29, {29}},
						       {33, {33}},
						       {37, {37}},
						       {41, {41}},
						       {45, {45}},
						       {49, {49}},
						       {53, {53}},
						       {57, {57}},
						       {61, {61}},
						       {65, {65}},
						       {69, {69}},
						       {73, {73}},
						       {77, {77}},
						       {81, {81}},
						       {85, {85}},
						       {89, {89}},
						       {93, {93}},
						       {97, {97}},
						       {101, {101}},
						       {105, {105}},
						       {109, {109}},
						       {113, {113}},
						       {117, {117}},
						       {121, {121}},
						       {125, {125}},
						       {129, {129}},
						       {133, {133}},
						       {137, {137}},
						       {141, {141}},
						       {145, {145}},
						       {149, {149}},
						       {153, {153}},
						       {157, {157}},
						       {161, {161}},
						       {165, {165}},
						       {169, {169}},
						       {173, {173}},
						       {177, {177}},
						       {181, {181}},
						       {185, {185}},
						       {189, {189}},
						       {193, {193}},
						       {197, {197}},
						       {201, {201}},
						       {205, {205}},
						       {209, {209}},
						       {213, {213}},
						       {217, {217}},
						       {221, {221}},
						       {225, {225}},
						       {229, {229}},
						       {233, {233}}}}},

	{ 132, 132, BAND_6, BW40, EXTCH_AUTO, {23, 29, {{1, {1 , 5}},
						       {9, {9, 13}},
						       {17, {17, 21}},
						       {25, {25, 29}},
						       {33, {33, 37}},
						       {41, {41, 45}},
						       {49, {49, 53}},
						       {57, {57, 61}},
						       {65, {65, 69}},
						       {73, {73, 77}},
						       {81, {81, 85}},
						       {89, {89, 93}},
						       {97, {97, 101}},
						       {105, {105, 109}},
						       {113, {113, 117}},
						       {121, {121, 125}},
						       {129, {129, 133}},
						       {137, {137, 141}},
						       {145, {145, 149}},
						       {153, {153, 157}},
						       {161, {161, 165}},
						       {169, {169, 173}},
						       {177, {177, 181}},
						       {185, {185, 189}},
						       {193, {193, 197}},
						       {201, {201, 205}},
						       {209, {209, 213}},
						       {217, {217, 221}},
						       {225, {225, 229}}}}},

	{ 133, 133, BAND_6, BW80, EXTCH_AUTO, {23, 14, {{1, {1, 5, 9, 13}},
						       {17, {17, 21, 25, 29}},
						       {33, {33, 37, 41, 45}},
						       {49, {49, 53, 57, 61}},
						       {65, {65, 69, 73, 77}},
						       {81, {81, 85, 89, 93}},
						       {97, {97, 101, 105, 109}},
						       {113, {113, 117, 121, 125}},
						       {129, {129, 133, 137, 141}},
						       {145, {145, 149, 153, 157}},
						       {161, {161, 165, 169, 173}},
						       {177, {177, 181, 185, 189}},
						       {193, {193, 197, 201, 205}},
						       {209, {209, 213, 217, 221}}}}},

	{ 134, 134, BAND_6, BW160, EXTCH_AUTO, {23, 7, {{1, {1, 5, 9, 13, 17, 21, 25, 29}},
						        {33, {33, 37, 41, 45, 49, 53, 57, 61}},
						        {65, {65, 69, 73, 77, 81, 85, 89, 93}},
						        {97, {97, 101, 105, 109, 113, 117, 121, 125}},
						        {129, {129, 133, 137, 141, 145, 149, 153, 157}},
						        {161, {161, 165, 169, 173, 177, 181, 185, 189}},
						        {193, {193, 197, 201, 205, 209, 213, 217, 221}}}}},

	{ 135, 135, BAND_6, BW8080, EXTCH_AUTO, {23, 14, {{1, {1, 5, 9, 13}},
						       {17, {17, 21, 25, 29}},
						       {33, {33, 37, 41, 45}},
						       {49, {49, 53, 57, 61}},
						       {65, {65, 69, 73, 77}},
						       {81, {81, 85, 89, 93}},
						       {97, {97, 101, 105, 109}},
						       {113, {113, 117, 121, 125}},
						       {129, {129, 133, 137, 141}},
						       {145, {145, 149, 153, 157}},
						       {161, {161, 165, 169, 173}},
						       {177, {177, 181, 185, 189}},
						       {193, {193, 197, 201, 205}},
						       {209, {209, 213, 217, 221}}}}},
	{ 136, 136, BAND_6, BW20, EXTCH_NONE, {23, 1, {{2, {2}}}}},
	{ 137, 137, BAND_6, BW320, EXTCH_AUTO, {23, 6, {{1, {1, 5, 9, 13, 17, 21, 25, 29, 33, 37, 41, 45, 49, 53, 57, 61}},
						        {33, {33, 37, 41, 45, 49, 53, 57, 61, 65, 69, 73, 77, 81, 85, 89, 93}},
						        {65, {65, 69, 73, 77, 81, 85, 89, 93, 97, 101, 105, 109, 113, 117, 121, 125}},
						        {97, {97, 101, 105, 109, 113, 117, 121, 125, 129, 133, 137, 141, 145, 149, 153, 157}},
						        {129, {129, 133, 137, 141, 145, 149, 153, 157, 161, 165, 169, 173, 177, 181, 185, 189}},
						        {161, {161, 165, 169, 173, 177, 181, 185, 189, 193, 197, 201, 205, 209, 213, 217, 221}}}}},

};

size_t wifi_opclass_global_size = sizeof(wifi_opclass_global)/sizeof(wifi_opclass_global[0]);


int wifi_opclass_to_channels(uint32_t opclass, int *num, uint32_t *channels)
{
	struct wifi_opclass *ptr = NULL;
	struct wifi_opclass *tab;
	int tabsize;
	int i;

	if (!num || !channels)
		return -1;

	tab = (struct wifi_opclass *)wifi_opclass_global;
	tabsize = wifi_opclass_global_size;

	for (i = 0; i < tabsize; i++) {
		ptr = tab + i;
		if (ptr->g_opclass == opclass)
			break;
	}

	if (i == tabsize)
		return -1;

	if (*num < ptr->opchannel.num)
		return -1;

	*num = 0;
	for (i = 0; i < ptr->opchannel.num; i++) {
		channels[i] = ptr->opchannel.ch[i].channel;
		*num += 1;
	}

	return 0;
}

int wifi_opclass_bandwidth(uint32_t opclass, enum wifi_bw *bw)
{
	struct wifi_opclass *tab;
	struct wifi_opclass *ptr;
	int i;

	if (!bw)
		return -1;

	*bw = BW_UNKNOWN;
	tab = (struct wifi_opclass *)wifi_opclass_global;

	for (i = 0; i < wifi_opclass_global_size; i++) {
		ptr = tab + i;
		if (ptr->g_opclass == opclass) {
			*bw = ptr->bw;
			return 0;
		}
	}

	return -1;
}

int wifi_opclass_band(uint32_t opclass, enum wifi_band *band)
{
	struct wifi_opclass *tab;
	struct wifi_opclass *ptr;
	int i;

	if (!band)
		return -1;

	*band = BAND_UNKNOWN;
	tab = (struct wifi_opclass *)wifi_opclass_global;

	for (i = 0; i < wifi_opclass_global_size; i++) {
		ptr = tab + i;
		if (ptr->g_opclass == opclass) {
			*band = ptr->band;
			return 0;
		}
	}

	return -1;

}

/* Example:
 *
 *	size_t num = 0;
 *	uint32_t bands = BAND_2 | BAND_5;
 *	uint32_t bws = WIFI_BANDWIDTH_80 | WIFI_BANDWIDTH_40;
 *	int ret;
 *
 *	ret = wifi_get_opclass_e4table(bands, bws, &num, NULL);
 *	if (!ret) {
 *		struct wifi_opclass op[num];
 *
 *		memset(op, 0, num * sizeof(struct wifi_opclass));
 *		ret = wifi_get_opclass_e4table(bands, bws, &num, op);
 *		:
 *	}
 */
int wifi_get_opclass_e4table(uint32_t bands, uint32_t bws, size_t *num_opclass,
			     struct wifi_opclass *o)
{
	struct wifi_opclass *tab = (struct wifi_opclass *)wifi_opclass_global;
	int tabsize = wifi_opclass_global_size;
	int i;

	if (*num_opclass == 0 && !o) {
		/* number of opclass entries unknown to user; return in
		 * num_opclass filtered by requested bands and bandwidths.
		 */
		for (i = 0; i < tabsize; i++) {
			struct wifi_opclass *ptr = tab + i;

			if ((ptr->band & bands) && (!!(BIT(ptr->bw) & bws)))
				*num_opclass += 1;
		}

		return 0;
	}

	*num_opclass = 0;
	for (i = 0; i < tabsize; i++) {
		struct wifi_opclass *ptr = tab + i;

		if ((ptr->band & bands) && (!!(BIT(ptr->bw) & bws))) {
			memcpy(o + *num_opclass, ptr, sizeof(struct wifi_opclass));
			*num_opclass += 1;
		}
	}

	return 0;
}

int wifi_get_opclass_entry(uint32_t opclass, struct wifi_opclass *entry)
{
	struct wifi_opclass *tab = (struct wifi_opclass *)wifi_opclass_global;
	int tabsize = wifi_opclass_global_size;
	int i;

	if (!entry)
		return -1;

	for (i = 0; i < tabsize; i++) {
		struct wifi_opclass *ptr = tab + i;

		if (ptr->opclass == opclass) {
			memset(entry, 0, sizeof(struct wifi_opclass));
			memcpy(entry, ptr, sizeof(struct wifi_opclass));
			return 0;
		}
	}

	return -1;
}

