#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <time.h>
#include <stdarg.h>
#include <ctype.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>
#include <dirent.h>
#include <net/if.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <linux/netlink.h>
#include <linux/pkt_sched.h>
#include <netlink/netlink.h>
#include <netlink/cache.h>
#include <netlink/msg.h>
#include <netlink/utils.h>
#include <netlink/object.h>
#include <netlink/genl/genl.h>
#include <netlink/genl/ctrl.h>
#include <netlink/netlink.h>
#include <netlink/route/rtnl.h>
#include <netlink/route/link.h>
#include <netlink/route/class.h>
#include <netlink/route/qdisc.h>
#include <netlink/route/qdisc/cbq.h>
#include <netlink/route/addr.h>
#include <netlink/cli/utils.h>

#include "../include/qos.h"

#include <easy/easy.h>

#define MAXQES 10

static struct qos_stats queue_stats[MAXQES];

struct queue_desc {
	unsigned ifindex;
	char ifname[IFNAMSIZ];
	int qid;
	struct qos_stats *qstats;
} q_desc[MAXQES];

struct glbl_cntxt {
	int quid;
	int count;
	int ifindex;
	struct nl_sock *sock;
	struct nl_cache *link_cache;
} g;

static void in_class(struct nl_object *, void *);
static void all_classes(uint32_t);
static void in_qdisc(struct nl_object *, void *);
static int all_qdiscs(uint32_t);

/*********************************************
 * Statistics collector from tc .
 * The stats counters on corresponding structs.
 * are populated for all queues at once.
 *********************************************
 */
static  void get_stat(struct nl_object *obj)
{
	struct rtnl_tc *tc = nl_object_priv(obj);
	struct qos_stats *q = q_desc[g.count].qstats;

	if ((int)g.ifindex != rtnl_tc_get_ifindex(tc)) {
		syslog(LOG_ERR, "ifindex did not match. q_desc.idx is %d"
				" rtnl idx is %d", g.ifindex, rtnl_tc_get_ifindex(tc));
		return;
	}
	q->tx_packets = rtnl_tc_get_stat(tc, RTNL_TC_PACKETS);
	q->tx_bytes = rtnl_tc_get_stat(tc, RTNL_TC_BYTES);
	q->tx_dropped_packets = rtnl_tc_get_stat(tc, RTNL_TC_DROPS);
	g.count++;

	if (g.count >= MAXQES) {
		g.count = 0;
	}
}

/***************************************************
 * Traverse the tree rooted at this class.
 ***************************************************
 */
static void in_class(struct nl_object *obj, void *arg)
{
	struct rtnl_tc *tc = nl_object_priv(obj);

	get_stat(obj);

	all_classes(rtnl_tc_get_handle(tc));
	all_qdiscs(rtnl_tc_get_handle(tc));
}

/*******************************************************
 * Iterate on all the classes and find the class
 * corresponding to the parent id supplied to us.
 *******************************************************
 */
static void all_classes(uint32_t parent)
{
	struct rtnl_class *filter;
	struct nl_cache *class_cache;

	filter = rtnl_class_alloc();
	if (!filter) {
		syslog(LOG_ERR, "Unable to allocate nl object");
		return;
	}

	if ((rtnl_class_alloc_cache(g.sock, g.ifindex, &class_cache)) < 0) {
		syslog(LOG_ERR, "Unable to allocate nl object");
		return;
	}

	rtnl_tc_set_parent((struct rtnl_tc *)filter, parent);
	nl_cache_foreach_filter(class_cache, OBJ_CAST(filter), in_class, NULL);

	rtnl_class_put(filter);
	nl_cache_free(class_cache);
}

/*******************************************************
 * Iterate on all the Qdiscs and find the Qdisc
 * corresponding to the parent id supplied to us.
 *******************************************************
 */
static int all_qdiscs(uint32_t parent)
{
	struct nl_cache *qdisc_cache;
	struct rtnl_qdisc *filter;

	if (!(filter = rtnl_qdisc_alloc())) {
		syslog(LOG_ERR, "Unable to allocate rtnl object");
		return -1;
	}

	if ((rtnl_qdisc_alloc_cache(g.sock, &qdisc_cache)) < 0) {
		syslog(LOG_ERR, "Unable to allocate qdisc_cache");
		return -1;
	}

	rtnl_tc_set_ifindex((struct rtnl_tc *)filter, g.ifindex);
	rtnl_tc_set_parent((struct rtnl_tc *)filter, parent);
	nl_cache_foreach_filter(qdisc_cache, OBJ_CAST(filter), in_qdisc, NULL);

	rtnl_qdisc_put(filter);
	nl_cache_free(qdisc_cache);
	return 0;
}

/***************************************************
 * Qdisc parser.
 ***************************************************
 */
static void in_qdisc(struct nl_object *obj, void *arg)
{
	struct rtnl_qdisc *qdisc = nl_object_priv(obj);
	struct rtnl_tc *tc = (struct rtnl_tc *)qdisc;

	if (g.ifindex != rtnl_tc_get_ifindex(tc)) {
		return;
	}

	if (rtnl_tc_get_parent(tc) == TC_H_ROOT) {
		all_classes(TC_H_ROOT);
	}

	all_classes(rtnl_tc_get_handle(tc));

}

/**************************************************************
 * Entry point in to tc parsing state machine.
 * At this point we have the tree rooted at TC_H_ROOT("root").
 * Will parse down this tree until we received the queue stats.
 **************************************************************
 */
static int iterate_On_link_from_ifname(void)
{
	struct rtnl_qdisc *qdisc;
	struct nl_cache *qdisc_cache;

	if ((rtnl_qdisc_alloc_cache(g.sock, &qdisc_cache)) < 0) {
		syslog(LOG_ERR, "Unable to allocate nl object");
		return -1;
	}

	if (!(qdisc = rtnl_qdisc_alloc())) {
		syslog(LOG_ERR, "Unable to allocate rtnl object");
		return -1;
	}

	nl_cache_foreach_filter(qdisc_cache, OBJ_CAST(qdisc), in_qdisc, NULL);

	rtnl_qdisc_put(qdisc);
	nl_cache_free(qdisc_cache);
	return 0;
}

/********************************************************
 * Open, connect  NL socket.
 ********************************************************
 */
static int open_nl_socket(void)
{
	int ret = 0;

	g.sock = nl_socket_alloc();
	if (g.sock == NULL) {
		ret = -errno;
		return ret;
	}
	if ((ret = nl_connect(g.sock, NETLINK_ROUTE)) < 0) {
		syslog(LOG_ERR, "Unable to connect netlink socket.");
		ret = abs(ret);
		goto out;
	}
	return 0;

out:
	nl_socket_free(g.sock);
	return ret;
}


static int close_link_iface(struct nl_sock *s)
{
	nl_socket_free(s);

	return 0;
}

/*******************************************************
 *  Get the queue stats from tc stats using NL socket. 
 *  @param ifname: input parameter for linux interface
 *  @param qid: input parameter for queue id
 *******************************************************
 */
int  qos_get_queue_stats(const char* intname, int q_id)
{
	int ret = 0;
	ret = open_nl_socket();
	if (ret) {
		return ret;
	}
	if ((ret = rtnl_link_alloc_cache(g.sock, AF_UNSPEC, &(g.link_cache))) < 0) {
		syslog(LOG_ERR, "Unable to allocate link_cache");
		ret = -1;
		goto out;
	}

	if ((iterate_On_link_from_ifname()) < 0) {
		ret = -1;
		goto out;
	}

out:
	if (g.link_cache)
		nl_cache_put(g.link_cache);

	(void)close_link_iface(g.sock);
	return ret;
}


static void describe_queue(int q_id, char *intname,
		struct qos_stats *stats)
{
	int i;

	for (i = 0; i < MAXQES ; i++) {
		strncpy(q_desc[i].ifname, intname, IFNAMSIZ - 1);
		q_desc[i].ifindex = (unsigned int)g.ifindex;
		q_desc[i].qid = i;
		q_desc[i].qstats = &queue_stats[i];
		q_desc[i].qstats->tx_packets = 0;
		q_desc[i].qstats->tx_bytes = 0;
		q_desc[i].qstats->tx_dropped_packets = 0;
	}
}

/*******************************************************
 *  To get the queue stats for a linux device:
 *  @param ifname: input parameter for linux interface
 *  @param qid: input parameter for queue id
 *  @param qstats: output parameter pointer to qos_stats
 *  @param is_read_and_reseti: output parameter for stat
 *  fetch was read and reset for driver.
 *******************************************************
 */
static int linux_get_stats(const char *ifname,
		int queue_id, struct qos_stats *stats,
		int *is_read_and_reset)
{
	int ret = 0;
	char iface[IFNAMSIZ];
	//skip the parent class stats and map only child class stats
	int mapped_qid = queue_id + 1;

	strncpy(iface, ifname, IFNAMSIZ - 1);
	g.ifindex = (int)if_nametoindex(ifname);
	g.count = 0;
	g.quid = queue_id;
	g.sock = NULL;
	g.link_cache = NULL;
	describe_queue(queue_id, iface, stats);
	ret = qos_get_queue_stats(ifname, queue_id);

	stats->tx_packets = q_desc[mapped_qid].qstats->tx_packets;
	stats->tx_bytes = q_desc[mapped_qid].qstats->tx_bytes;
	stats->tx_dropped_packets = q_desc[mapped_qid].qstats->tx_dropped_packets;
	*is_read_and_reset = 0;
	return ret;
}

/*******************************************************
 *  To get the number of qos queue for a linux device:
 *  return number of qos queue
 *******************************************************
 */
int linux_get_num_of_queue(const char *ifname)
{
	return 8;
}

#if defined(IOPSYS_MEDIATEK)
const struct qos_ops qos_linux_ops_eth = {
	.ifname = "lan",
	.get_stats = linux_get_stats,
	.get_num_of_queue = linux_get_num_of_queue
};
#elif defined(IOPSYS_LINUX)
const struct qos_ops qos_linux_ops_eth = {
	.ifname = "eth",
	.get_stats = linux_get_stats,
	.get_num_of_queue = linux_get_num_of_queue
};
#endif
