/*
 * ecnt_qos.c - Econet QoS library implementation
 *
 * Copyright (C) 2022 iopsys Software Solutions AB. All rights reserved.
 *
 * Author: maxim.menshikov@iopsys.eu
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <sys/klog.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <sys/syslog.h>
#include <errno.h>
#include <uci.h>

#ifndef SYSLOG_ACTION_SIZE_BUFFER
#define SYSLOG_ACTION_SIZE_BUFFER   (10)
#endif

#ifndef SYSLOG_ACTION_READ_ALL
#define SYSLOG_ACTION_READ_ALL      (3)
#endif

#include "qos.h"

/* Is queue number inverse relative to queue count? */
#define QDMA_WAN_QUEUE_INVERSE      (0)
/* Total number of rings */
#define QDMA_WAN_RING_COUNT         (8)
/* Total number of queues */
#define QDMA_WAN_QUEUE_COUNT        (8)
/* Path to QDMA WAN counters in procfs */
#define QDMA_WAN_QUEUE_STATS_PATH   "/proc/qdma_wan/queue_stats"
/* Path to QDMA LAN counters in procfs */
#define QDMA_LAN_QUEUE_STATS_PATH   "/proc/qdma_lan/queue_stats"
/* Max length of counter name */
#define STAT_COUNTER_NAME_MAX_LEN   (50)
/* Max length of TX queue identifier */
#define TXQ_NAME_MAX_LEN            (50)
/* Max length of read system log */
#define QUEUE_STATS_LINE_MAX_LEN    (256)

/* Static WAN port cache */
static char wan_port[IFNAMSIZ] = {0};

/**
 * Fetch WAN port from UCI if not already cached
 */
static const char* get_wan_port(void)
{
	if (wan_port[0] != '\0') {
		return wan_port;
	}

	struct uci_context *ctx = uci_alloc_context();
	if (!ctx) {
		syslog(LOG_ERR, "Failed to allocate UCI context");
		return NULL;
	}

	struct uci_ptr ptr;
	memset(&ptr, 0, sizeof(ptr));

	char uci_path[] = "network.WAN.name";
	if (uci_lookup_ptr(ctx, &ptr, uci_path, true) == UCI_OK &&
			(ptr.flags & UCI_LOOKUP_COMPLETE) && ptr.o && ptr.o->type == UCI_TYPE_STRING) {
		strncpy(wan_port, ptr.o->v.string, IFNAMSIZ - 1);
		wan_port[IFNAMSIZ - 1] = '\0';
	} else {
		syslog(LOG_ERR, "WAN device not found in network UCI");
	}

	uci_free_context(ctx);
	return wan_port[0] ? wan_port : NULL;
}

/**
 * Read queue data from system log
 *
 * @param[in]  ifname Interface name
 * @param[in]  queue  The queue number
 * @param[out] stats  The pointer to the structure with statistics
 *
 * @return @c 0 on success, any other value on failure.
 */
static int read_from_proc(const char *ifname,
			  int queue,
			  struct qos_stats *stats)
{
	char   buf[QUEUE_STATS_LINE_MAX_LEN];
	char   txq_name[TXQ_NAME_MAX_LEN];
	FILE  *f;
	const char *wan = get_wan_port();

	const char *proc_path = NULL;
	bool is_wan = (wan && strncmp(ifname, wan, IFNAMSIZ) == 0 && strlen(ifname) == strlen(wan));

	if (is_wan) {
		proc_path = QDMA_WAN_QUEUE_STATS_PATH;
	} else {
		proc_path = QDMA_LAN_QUEUE_STATS_PATH;
	}

#if QDMA_WAN_QUEUE_INVERSE
	/* Modes used in qos module have inverse queue numbers */
	if (queue < QDMA_WAN_QUEUE_COUNT && queue != 0)
		queue = QDMA_WAN_QUEUE_COUNT - queue;
#endif

	f = fopen(proc_path, "r");
	if (f == NULL) {
		syslog(LOG_ERR, "failed to open %s: %s", proc_path, strerror(errno));
		return -1;
	}

	while (fgets(buf, sizeof(buf), f) != NULL)
	{
		uint64_t tx_pkts = 0;
		uint64_t tx_bytes = 0;
		uint64_t tx_dropped_pkts = 0;
		uint64_t tx_dropped_bytes = 0;
		char   *tmp;

		/* only process Ring 0 lines */
		if (strstr(buf, "Ring 0") == NULL) {
			continue;
		}

		snprintf(txq_name, TXQ_NAME_MAX_LEN, /* Flawfinder: ignore */
                         "Queue %" PRIu8 " ", (uint8_t)queue);
		tmp = strstr(buf, txq_name);
		if (tmp == NULL ||
		    sscanf(tmp + strlen(txq_name), /* Flawfinder: ignore */
		           "tx Counts: %" PRIu64 ", tx Bytes: %" PRIu64 ", "
		           "tx drop Counts: %" PRIu64 ", tx drop Bytes: %" PRIu64,
		           &tx_pkts,
		           &tx_bytes,
		           &tx_dropped_pkts,
		           &tx_dropped_bytes) != 4)
		{
			tx_pkts = 0;
			tx_bytes = 0;
			tx_dropped_pkts = 0;
			tx_dropped_bytes = 0;
		}

		stats->tx_packets += tx_pkts;
		stats->tx_bytes += tx_bytes;
		stats->tx_dropped_packets += tx_dropped_pkts;
		stats->tx_dropped_bytes += tx_dropped_bytes;
	}

	fclose(f);
	return 0;
}

/**
 *  Get queue stats on Econet platform
 *  @param [in]  ifname 				input parameter for linux interface
 *  @param [in]  qid 					input parameter for queue id
 *  @param [out] qstats 				output parameter pointer to qos_stats
 *  @param [out] is_read_and_reset 		output parameter for stat fetch was read
 *                                      and reset for driver
 */
static int ecnt_get_stats(const char *ifname,
                          int queue_id,
                          struct qos_stats *stats,
                          int *is_read_and_reset)
{
	stats->tx_packets = 0;
	stats->tx_bytes  = 0;
	stats->tx_dropped_packets = 0;
	stats->tx_dropped_bytes = 0;

	/* Ignore interfaces that don't exist */
	if (if_nametoindex(ifname) == 0) {
		syslog(LOG_ERR, "interface doesn't exist: %s", ifname);
		return errno;
	}

	*is_read_and_reset = 0;

	return read_from_proc(ifname, queue_id, stats);
}

int ecnt_get_num_of_queue(const char *ifname)
{
	return 8;
}

const struct qos_ops qos_ecnt_ops_eth = {
	.ifname = "eth",
	.get_stats = ecnt_get_stats,
	.get_num_of_queue = ecnt_get_num_of_queue
};

const struct qos_ops qos_ecnt_ops_nas = {
	.ifname = "nas",
	.get_stats = ecnt_get_stats,
	.get_num_of_queue = ecnt_get_num_of_queue
};

const struct qos_ops qos_ecnt_ops_ae_wan = {
	.ifname = "ae_wan",
	.get_stats = ecnt_get_stats,
	.get_num_of_queue = ecnt_get_num_of_queue
};

const struct qos_ops qos_ecnt_ops_pon = {
	.ifname = "pon",
	.get_stats = ecnt_get_stats,
	.get_num_of_queue = ecnt_get_num_of_queue
};
