#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdbool.h>
#include <assert.h>
#include <errno.h>
#include <arpa/inet.h>															// htns()

#include "libpicoevent.h"

#define BUSMAIL_PACKET_HEADER 0x10
#define CRC_SIZE					(sizeof(uint8_t))

/* Debug function */
void packet_dump(pe_packet_t *p) {
	unsigned int i;

	printf("\n[PACKET %d] - ", p->size);
	for (i = 0; i < p->size; i++) {
		printf("%02x ", p->data[i]);
	}
	printf("\n");
}

/* Create an outgoing packet. */
static int make_tx_packet(uint8_t *out, uint8_t *data, int data_size) {
	pe_mail_t *mail;
	uint8_t crc = 0;
	int i;

	mail = (pe_mail_t*) out;
	mail->frame_header = BUSMAIL_PACKET_HEADER;
	mail->data_size = htons(data_size);

	memcpy(mail->mail_data, data, data_size);
	for (i = 0; i < data_size; i++) crc += data[i];								// Calculate data checksum
	mail->mail_data[data_size] = crc;											// Append crc after data

	return 0;
}

/* Transmit packet to the bus. */
static void send_mail(uint8_t *m, int size, int fd) {
	int len, written_len = 0, chunk_size;

	while(written_len < size) {
		chunk_size = size - written_len;

		len = write(fd, m + written_len, chunk_size);
		if (len == -1 && errno != EINTR && errno != EAGAIN) {
			perror("Error writing to bus fd");
			break;
		}
		else if(len > 0) {
			written_len += len;
		}
		else {
			usleep(0);
		}
	}
}


/* Send a mail to the bus. */
int pe_bus_send(pe_bus_t *bus, uint8_t *data, unsigned int size) {
	pe_mail_t *m;

	if(size > PE_MAX_SIZE)
		return -1;

	m = malloc(sizeof(pe_mail_t) + size);
	if (!m) {
		return -1;
	}

	make_tx_packet((uint8_t *)m, data, size);
	pthread_mutex_lock(&bus->lock);
	send_mail((uint8_t *)m, sizeof(pe_mail_t) + size, bus->fd);
	pthread_mutex_unlock(&bus->lock);
	free(m);

	return 0;
}

/* Read raw data from the bus into the bus buffer. */
int pe_bus_receive(pe_bus_t *bus, pe_event_t *event) {
	int written;

	if (pe_event_count(event) == 0)
		return 0;

	pthread_mutex_lock(&bus->lock);
	written = pe_buffer_write(bus->buf, pe_event_data(event),
		pe_event_count(event));
	pthread_mutex_unlock(&bus->lock);

	if (written != pe_event_count(event)) {
		printf("Warning, incorrect pe buffer write! %d %d\n",
			written, pe_event_count(event));
		return -1;
	}

	return 0;
}


/* Extract a packet from the raw stream of data on the bus. */
static int pe_bus_get(pe_bus_t *bus, pe_packet_t *packetOut) {
	uint32_t i, read = 0;
	uint8_t crc_calc = 0;
	uint16_t data_size;
	pe_mail_t *mail;

	mail = (pe_mail_t*) alloca(sizeof(pe_mail_t) + PE_MAX_SIZE);

	/* Do we have a start of frame? */
	while (pe_buffer_read(bus->buf, (uint8_t*) &mail->frame_header,
			sizeof(mail->frame_header)) > 0) {
		read += sizeof(mail->frame_header);
		if (mail->frame_header == BUSMAIL_PACKET_HEADER) {
			break;
		}
	}

	/* Return if we did not read any data. */
	if (read == 0) {
		return -1;
	}

	/* Do we have a full header? */
	if (pe_buffer_size(bus->buf) < sizeof(pe_mail_t) - sizeof(mail->frame_header)) {
		if (pe_buffer_rewind(bus->buf, sizeof(mail->frame_header) <
				sizeof(mail->frame_header))) {
			printf("Warning, short buffer rewind!\n");
			abort();
		}
		return -1;
	}
	pe_buffer_read(bus->buf, (uint8_t*) &data_size, sizeof(data_size));
	mail->data_size = ntohs(data_size);
	
	/* Do we have a full packet (data + crc)? */
	if (pe_buffer_size(bus->buf) < mail->data_size + CRC_SIZE) {
		if (pe_buffer_rewind(bus->buf, sizeof(mail->frame_header) +
				sizeof(mail->data_size)) < sizeof(mail->frame_header) +
				sizeof(mail->data_size)) {
			printf("Warning, short buffer rewind!\n");
			abort();
		}
		return -1;
	}
	else if (pe_buffer_read(bus->buf, mail->mail_data,
			mail->data_size + CRC_SIZE) < mail->data_size + CRC_SIZE) {
		printf("Warning, short buffer read!\n");
		abort();
	}
	
	/* Calculate checksum over data portion. */
	for (i = 0; i < mail->data_size; i++) {
		crc_calc += mail->mail_data[i];
	}

	if (crc_calc != mail->mail_data[mail->data_size]) {							// Read crc from remote
		printf("Droped bus packet: bad checksum: %x != %x\n",
			crc_calc, mail->mail_data[mail->data_size]);
		abort();
	}

	/* Copy data portion to reciving application buffer. */
	memcpy(packetOut->data, mail->mail_data, mail->data_size);
	packetOut->size = mail->data_size;

	return 0;
}


/* Retreive whole frames from the bus and dispatch them to registered application handlers. */
int pe_bus_dispatch(pe_bus_t *bus) {
	pe_packet_t packet;
	int res;

	while(1) {
		pthread_mutex_lock(&bus->lock);
		res = pe_bus_get(bus, &packet);
		pthread_mutex_unlock(&bus->lock);
		if(res) return 0;

		pe_list_call_each(bus->application_handlers, &packet);
	}
}


/* Register packet handlers to the bus. */
void pe_bus_add_handler(pe_bus_t *bus, void (*app_handler)(pe_packet_t *)) {
	pe_list_add(bus->application_handlers, app_handler);
}


/* Create a new bus. */
pe_bus_t *pe_bus_new(int fd) {
	pe_bus_t *bus = (pe_bus_t *) calloc(1, sizeof(pe_bus_t));

	bus->fd = fd;
	bus->application_handlers = pe_list_new();
	bus->buf = pe_buffer_new(PE_BUS_BUF_SIZE);
	pthread_mutex_init(&bus->lock, NULL);

	return bus;
}
