#ifndef __LIBPICOEVENT__
#define __LIBPICOEVENT__

#include <stdint.h>
#include <pthread.h>
#include <linux/limits.h>														// PIPE_BUF


typedef struct {
	uint8_t *in;
	int count;
} pe_event_t;

typedef struct pe_stream_t {
	int fd;
	int maxEventSize;
	void (*event_handler)(struct pe_stream_t *stream, pe_event_t *event);
} pe_stream_t;

typedef struct {
	struct list_t *first;
	struct list_t *last;
	int count;
	pthread_mutex_t lock;
} pe_list_t;

typedef struct {
	uint8_t * buf_start;			// malloc begin
	uint8_t * buf_end;				// malloc end
	uint8_t * data_start;			// read pointer
	uint8_t * data_end;				// write pointer
	uint32_t count;					// used length
	uint32_t buf_size;				// sizeof malloc
	uint8_t state;
} pe_buffer_t;

typedef struct {
	uint32_t fd;
	pe_buffer_t *buf;
	pthread_mutex_t lock;
	pe_list_t *application_handlers;
} pe_bus_t;

typedef struct __attribute__((__packed__)) {
	uint8_t frame_header;
	uint16_t data_size;
	uint8_t mail_data[1];
} pe_mail_t;

#define PE_MAX_EVENT_SIZE		(PIPE_BUF - sizeof(pe_mail_t))
#define PE_MAX_SIZE				(PE_MAX_EVENT_SIZE)								// Write at most PIPE_BUF to ensure atomic.
#define PE_BUS_BUF_SIZE			(PE_MAX_EVENT_SIZE * 4)


typedef struct __attribute__((__packed__)) {
	uint32_t size;
	uint8_t data[PE_MAX_SIZE];
} pe_packet_t;


/* error functions */
void exit_failure(const char *format, ...);
void err_exit(const char *format, ...);
void exit_succes(const char *format, ...);

/* pe_buffer interface */
pe_buffer_t * pe_buffer_new(int size);
int pe_buffer_dump(pe_buffer_t * self);
uint32_t pe_buffer_write(pe_buffer_t * self, uint8_t *input, uint32_t count);
uint32_t pe_buffer_read(pe_buffer_t * self, uint8_t *buf, uint32_t count);
uint32_t pe_buffer_rewind(pe_buffer_t * self, uint32_t count);
uint32_t pe_buffer_size(pe_buffer_t * self);

/* pe_list interface */
pe_list_t *pe_list_new(void);
void * pe_list_get(pe_list_t *l);
void pe_list_add(pe_list_t *l, void * p);
void pe_list_delete(pe_list_t *l, void * p);
void pe_list_call_each(pe_list_t *l, void * arg);
void *pe_list_find(pe_list_t *head, int (*broker)(void*));

/* pe_stream interface */
pe_stream_t *pe_stream_new(int fd);
int pe_stream_get_fd(pe_stream_t *s);
void *pe_stream_get_handler(pe_stream_t *s);
void pe_stream_add_handler(pe_stream_t *s, int maxEventSize, void (*event_handler)(pe_stream_t *stream, pe_event_t *event));
void pe_stream_remove_handler(pe_stream_t *s, void (*event_handler)(pe_stream_t *stream, pe_event_t *event));

/* pe_event interface */
pe_event_t *pe_event_new(pe_stream_t *stream);
void pe_event_destroy(pe_event_t *e);
uint8_t *pe_event_data(pe_event_t *e);
int pe_event_count(pe_event_t *e);

/* pe_base interface */
int pe_base_new(void);
int pe_base_add_stream(int fd, pe_stream_t *stream);
int pe_base_delete_stream(int fd, pe_stream_t *stream);
void pe_base_dispatch(int fd);

/* 
 *  pe_bus interface
 */

/* Create a new bus. */
pe_bus_t *pe_bus_new(int fd);
/* Register packet handlers to the bus. */
void pe_bus_add_handler(pe_bus_t *bus, void (*app_handler)(pe_packet_t *));
/* Retreive whole frames from the bus and dispatch them to registered application handlers. */
int pe_bus_dispatch(pe_bus_t *bus);
/* Read raw data from the bus into the bus buffer. */
int pe_bus_receive(pe_bus_t *bus, pe_event_t *event);
/* Send a mail to the bus. */
int pe_bus_send(pe_bus_t *bus, uint8_t *data, unsigned int size);

#endif /* __LIBPICOEVENT__ */
