// SPDX-License-Identifier: BSD-3-Clause
/*
 * debug.h - header file for debug and log messages.
 *
 * Copyright (C) 2021-2024 IOPSYS Software Solutions AB. All rights reserved.
 * Copyright (C) 2025 Genexis AB.
 *
 * Author: anjan.chanda@iopsys.eu
 */

#ifndef DEBUG_H
#define DEBUG_H

/* log feature categories */
enum {
	LOG_MAIN,	/* Main process */
	LOG_TIMER,	/* Timers */
	LOG_NEIGH,	/* Neigh table entries */
	LOG_CMDU,	/* CMDU buffer dump in JSON format */
	LOG_WSC,	/* WSC and PBC */
	LOG_CMD,	/* CLI cmd request and response */
	LOG_TX,		/* Transmit */
	LOG_RX,		/* Receive */
	LOG_FRAG,	/* CMDU Fragmentation and Defragmentation */
	LOG_DM,         /* Data-Model related */
	LOG_DEFAULT,	/* uncategorized messages */
};

/* log levels */
enum {
	LOGLEVEL_ERROR,
	LOGLEVEL_WARN,
	LOGLEVEL_INFO,
	LOGLEVEL_DEBUG,
	LOGLEVEL_TRACE,
};

#define LOG_FEATURE(x)	BIT(x)
#define LOG_FEATURE_ALL	BIT(LOG_DEFAULT) | (BIT(LOG_DEFAULT) - 1)

#define DEFAULT_LOGLEVEL	LOGLEVEL_INFO

uint32_t logfeature_to_enum(const char *s);
const char *logfeature_to_string(uint32_t e);

void restart_logging(void *opts);
void stop_logging(void);
void log_message(int feature, int level, const char *fmt, ...)
	__attribute__((__format__(__printf__, 3, 4)));
void log_cmdu(uint8_t *buf, size_t len, const char *ifname, bool is_rx);

#define err(fmt, ...)         log_message(LOG_DEFAULT, 0, fmt, ## __VA_ARGS__)
#define warn(fmt, ...)        log_message(LOG_DEFAULT, 1, fmt, ## __VA_ARGS__)
#define info(fmt, ...)        log_message(LOG_DEFAULT, 2, fmt, ## __VA_ARGS__)
#define dbg(fmt, ...)         log_message(LOG_DEFAULT, 3, fmt, ## __VA_ARGS__)
#define trace(fmt, ...)       log_message(LOG_DEFAULT, 4, fmt, ## __VA_ARGS__)
#define loud(fmt, ...)        log_message(LOG_DEFAULT, 5, fmt, ## __VA_ARGS__)

#define i1905_err(f, fmt, ...)         log_message(f, 0, fmt, ## __VA_ARGS__)
#define i1905_warn(f, fmt, ...)        log_message(f, 1, fmt, ## __VA_ARGS__)
#define i1905_info(f, fmt, ...)        log_message(f, 2, fmt, ## __VA_ARGS__)
#define i1905_dbg(f, fmt, ...)         log_message(f, 3, fmt, ## __VA_ARGS__)
#define i1905_trace(f, fmt, ...)       log_message(f, 4, fmt, ## __VA_ARGS__)
#define i1905_loud(f, fmt, ...)        log_message(f, 5, fmt, ## __VA_ARGS__)

#define logcmdu(b,l,i,t)      log_cmdu(b,l,i,t)

#define warn_on(condition)					\
({								\
	int __rc = !!(condition);				\
	if (__rc)						\
		warn("%s: WARN '%s'\n", __func__, #condition);	\
	__rc;							\
})

#endif /* DEBUG_H */
