/*
 * Copyright (C) 2023 iopsys Software Solutions AB
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1
 * as published by the Free Software Foundation
 *
 *	Author: Amin Ben Romdhane <amin.benromdhane@iopsys.eu>
 */

#include "../bbf_plugin/hosts.h"

static char *AccessPolicy[] = {"Allow", "Deny", NULL};

/* ********** DynamicObj ********** */
DM_MAP_OBJ tDynamicObj[] = {
/* parentobj, nextobject, parameter */
{"Device.", tDeviceObjs, NULL},
{0}
};

/*************************************************************
* ENTRY METHOD
**************************************************************/
/*#Device.Hosts.Host.{i}.!UBUS:hosts/show//hosts*/
static int browseHostsHostInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *res = NULL, *host_obj = NULL, *arrobj = NULL;
	struct dm_data curr_data = {0};
	int idx = 0, i = 0;
	char *inst = NULL;

	dmubus_call("hosts", "show", UBUS_ARGS{0}, 0, &res);

	dmjson_foreach_obj_in_array(res, arrobj, host_obj, idx, 1, "hosts") {

		curr_data.json_object = host_obj;

		inst = handle_instance_without_section(dmctx, parent_node, ++i);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

/*#Device.Hosts.Host.{i}.IPv4Address.{i}.!UBUS:hosts/show//hosts[@i-1].ipv4addr*/
static int browseHostsHostIPv4AddressInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *ip_arr = NULL, *host_obj = ((struct dm_data *)prev_data)->json_object;
	struct dm_data curr_data = {0};
	char *inst = NULL, *ipv4addr = NULL;
	int id = 0, i = 0;

	dmjson_foreach_value_in_array(host_obj, ip_arr, ipv4addr, i, 1, "ipv4addr") {

		curr_data.additional_data = (void *)ipv4addr;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

/*#Device.Hosts.Host.{i}.IPv6Address.{i}.!UBUS:hosts/show//hosts[@i-1].ipv6addr*/
static int browseHostsHostIPv6AddressInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	json_object *ip_arr = NULL, *host_obj = ((struct dm_data *)prev_data)->json_object;
	struct dm_data curr_data = {0};
	char *inst = NULL, *ipv6addr = NULL;
	int id = 0, i = 0;

	dmjson_foreach_value_in_array(host_obj, ip_arr, ipv6addr, i, 1, "ipv6addr") {

		curr_data.additional_data = (void *)ipv6addr;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseHostsAccessControlInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct dm_data *curr_data = NULL;
	LIST_HEAD(dup_list);
	char *inst = NULL;

	synchronize_specific_config_sections_with_dmmap("hosts", "access_control", "dmmap_hosts", &dup_list);
	list_for_each_entry(curr_data, &dup_list, list) {

		inst = handle_instance(dmctx, parent_node, curr_data->dmmap_section, "access_control_instance", "access_control_alias");

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)curr_data, inst) == DM_STOP)
			break;
	}
	free_dmmap_config_dup_list(&dup_list);
	return 0;

}

/*************************************************************
* ADD & DEL OBJ
**************************************************************/
static int addObjHostsAccessControl(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	struct uci_section *ac_s = NULL, *dmmap_s = NULL;
	char ac_name[32] = {0};

	snprintf(ac_name, sizeof(ac_name), "ac_%s", *instance);

	dmuci_add_section("hosts", "access_control", &ac_s);
	dmuci_rename_section_by_section(ac_s, ac_name);
	dmuci_set_value_by_section(ac_s, "enable", "0");
	dmuci_set_value_by_section(ac_s, "access_policy", "Allow");

	dmuci_add_section_bbfdm("dmmap_hosts", "access_control", &dmmap_s);
	dmuci_set_value_by_section(dmmap_s, "section_name", section_name(ac_s));
	dmuci_set_value_by_section(dmmap_s, "access_control_instance", *instance);
	return 0;
}

static int delObjHostsAccessControl(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	struct uci_section *s = NULL, *stmp = NULL;

	switch (del_action) {
		case DEL_INST:
			// AccessControl section
			dmuci_delete_by_section(((struct dm_data *)data)->config_section, NULL, NULL);

			// Dmmap AccessControl section
			dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);

			break;
		case DEL_ALL:
			// AccessControl section
			uci_foreach_sections_safe("hosts", "access_control", stmp, s) {
				dmuci_delete_by_section(s, NULL, NULL);
			}

			// dmmap AccessControl section
			uci_path_foreach_sections_safe(bbfdm, "dmmap_hosts", "access_control", stmp, s) {
				dmuci_delete_by_section(s, NULL, NULL);
			}

			break;
	}
	return 0;
}

/*************************************************************
* GET & SET PARAM
**************************************************************/
/*#Device.Hosts.HostNumberOfEntries!UBUS:hosts/show//hosts*/
static int get_Hosts_HostNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseHostsHostInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_Hosts_AccessControlNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseHostsAccessControlInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

/*#Device.Hosts.Host.{i}.PhysAddress!UBUS:hosts/show//hosts[@i-1].macaddr*/
static int get_HostsHost_PhysAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "macaddr");
	return 0;
}

/*#Device.Hosts.Host.{i}.IPAddress!UBUS:hosts/show//hosts[@i-1].ipaddr*/
static int get_HostsHost_IPAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ipaddr");
	return 0;
}

static int get_HostsHost_DHCPClient(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *address_source = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "address_source");

	if (DM_STRCMP(address_source, "DHCP") == 0) {
		char *linker = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "macaddr");

		return _bbfdm_get_references(ctx, "Device.DHCPv4.Server.Pool.*.Client.", "Chaddr", linker, value);
	} else {
		*value = "";
		return 0;
	}
}

static int get_HostsHost_AssociatedDevice(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *type = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "interface_type");

	if (DM_LSTRCMP(type, "Wi-Fi") == 0) {
		char buf[4096] = {0};

		char *link_macaddr = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "link_macaddr");
		bool is1905 = dmuci_string_to_boolean(dmjson_get_value(((struct dm_data *)data)->json_object, 1, "is1905"));

		if (is1905 && DM_STRLEN(link_macaddr)) {
			bbfdm_get_references(ctx, MATCH_FIRST, "Device.WiFi.DataElements.Network.Device.*.Radio.*.BSS.*.STA.", "MACAddress", link_macaddr, buf, sizeof(buf));
		} else {
			char *macaddr = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "macaddr");

			bbfdm_get_references(ctx, MATCH_FIRST, "Device.WiFi.AccessPoint.*.AssociatedDevice.", "MACAddress", macaddr, buf, sizeof(buf));
			bbfdm_get_references(ctx, MATCH_FIRST, "Device.WiFi.DataElements.Network.Device.*.APMLD.*.STAMLD.", "MLDMACAddress", macaddr, buf, sizeof(buf));
		}

		*value = dmstrdup(buf);
	} else {
		*value = "";
	}

	return 0;
}

static int get_HostsHost_Layer1Interface(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	bool is_local = dmuci_string_to_boolean(dmjson_get_value(((struct dm_data *)data)->json_object, 1, "local"));

	if (is_local) {
		char buf[4096] = {0};
		char *linker = NULL;

		char *type = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "interface_type");

		if (DM_LSTRCMP(type, "Wi-Fi") == 0) {
			bool is1905 = dmuci_string_to_boolean(dmjson_get_value(((struct dm_data *)data)->json_object, 1, "is1905"));
			linker = dmjson_get_value(((struct dm_data *)data)->json_object, 1, is1905 ? "parent_device" : "device");

			struct uci_section *iface_s = get_dup_section_in_config_opt("wireless", "wifi-iface", "ifname", linker);
			dmuci_get_value_by_section_string(iface_s, "device", &linker);

			bbfdm_get_references(ctx, MATCH_FIRST, "Device.WiFi.Radio.", "Name", linker, buf, sizeof(buf));
		} else {
			linker = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "device");
			struct uci_section *device_s = get_dup_section_in_config_opt("network", "device", "name", linker);
			dmuci_get_value_by_section_string(device_s, "ifname", &linker);

			bbfdm_get_references(ctx, MATCH_FIRST, "Device.Ethernet.Interface.", "Name", linker, buf, sizeof(buf));
		}

		*value = dmstrdup(buf);
	} else {
		*value = "";
	}

	return 0;
}

static int get_HostsHost_Layer3Interface(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *linker = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "network");

	return _bbfdm_get_references(ctx, "Device.IP.Interface.", "Name", linker, value);
}

/*#Device.Hosts.Host.{i}.InterfaceType!UBUS:hosts/show//hosts[@i-1].interface_type*/
static int get_HostsHost_InterfaceType(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "interface_type");
	return 0;
}

/*#Device.Hosts.Host.{i}.HostName!UBUS:hosts/show//hosts[@i-1].hostname*/
static int get_HostsHost_HostName(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "hostname");

	if (DM_STRCMP(*value, "*") == 0)
		*value = "";

	return 0;
}

/*#Device.Hosts.Host.{i}.Active!UBUS:hosts/show//hosts[@i-1].active*/
static int get_HostsHost_Active(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "active");
	return 0;
}

/*#Device.Hosts.Host.{i}.ActiveLastChange!UBUS:hosts/show//hosts[@i-1].active_last_change*/
static int get_HostsHost_ActiveLastChange(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "active_last_change");
	return 0;
}

/*#Device.Hosts.Host.{i}.IPv4AddressNumberOfEntries!UBUS:hosts/show//hosts[@i-1].ipv4addr*/
static int get_HostsHost_IPv4AddressNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseHostsHostIPv4AddressInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

/*#Device.Hosts.Host.{i}.IPv6AddressNumberOfEntries!UBUS:hosts/show//hosts[@i-1].ipv6addr*/
static int get_HostsHost_IPv6AddressNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseHostsHostIPv6AddressInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

/*#Device.Hosts.Host.{i}.IPv4Address.{i}.IPAddress!UBUS:hosts/show//hosts[@i-1].ipv4addr[@i-1]*/
static int get_HostsHostIPv4Address_IPAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char *)((struct dm_data *)data)->additional_data;
	return 0;
}

/*#Device.Hosts.Host.{i}.IPv6Address.{i}.IPAddress!UBUS:hosts/show//hosts[@i-1].ipv6addr[@i-1]*/
static int get_HostsHostIPv6Address_IPAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char *)((struct dm_data *)data)->additional_data;
	return 0;
}

/*#Device.Hosts.Host.{i}.WANStats.BytesSent!UBUS:hosts/show//hosts[@i-1].stats.tx_bytes*/
static int get_HostsHostWANStats_BytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "stats", "tx_bytes");
	return 0;
}

/*#Device.Hosts.Host.{i}.WANStats.BytesReceived!UBUS:hosts/show//hosts[@i-1].stats.rx_bytes*/
static int get_HostsHostWANStats_BytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "stats", "rx_bytes");
	return 0;
}

/*#Device.Hosts.Host.{i}.WANStats.PacketsSent!UBUS:hosts/show//hosts[@i-1].stats.tx_packets*/
static int get_HostsHostWANStats_PacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "stats", "tx_packets");
	return 0;
}

/*#Device.Hosts.Host.{i}.WANStats.PacketsReceived!UBUS:hosts/show//hosts[@i-1].stats.rx_packets*/
static int get_HostsHostWANStats_PacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 2, "stats", "rx_packets");
	return 0;
}

static int get_HostsAccessControl_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "access_control_alias", instance, value);
}

static int set_HostsAccessControl_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	return bbf_set_alias(ctx, ((struct dm_data *)data)->dmmap_section, "access_control_alias", instance, value);
}

static int get_HostsAccessControl_PhysAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "macaddr", value);
	return 0;
}

static int set_HostsAccessControl_PhysAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 64, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "macaddr", value);
			break;
	}
	return 0;
}

static int get_HostsAccessControl_HostName(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "host", value);
	return 0;
}

static int set_HostsAccessControl_HostName(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 256, NULL, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "host", value);
			break;
	}
	return 0;
}

static int get_HostsAccessControl_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enable", "0");
	return 0;
}

static int set_HostsAccessControl_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			break;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "enable", b ? "1" : "0");
			break;
	}
	return 0;
}

static int get_HostsAccessControl_AccessPolicy(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "access_policy", value);
	return 0;
}

static int set_HostsAccessControl_AccessPolicy(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, -1, AccessPolicy, NULL))
				return FAULT_9007;
			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "access_policy", value);
			break;
	}
	return 0;
}

static int get_HostsAccessControl_ScheduleRef(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_list *val = NULL;
	struct uci_section *s = NULL;
	char *schedule_alias = NULL;

	dmuci_get_value_by_section_list(((struct dm_data *)data)->config_section, "access_control_schedule", &val);

	if (val) {
		struct uci_element *e = NULL;
		char lbuf[1024] = {0};
		char s_ref[1024] = {0};

		uci_foreach_element(val, e) {
			if (!e->name) {
				continue;
			}
			uci_path_foreach_option_eq(bbfdm, "dmmap_schedules", "schedule", "section_name", e->name, s) {
				dmuci_get_value_by_section_string(s, "schedule_alias", &schedule_alias);
			}

			if (schedule_alias != NULL) {
				bbfdm_get_references(ctx, MATCH_ALL, "Device.Schedules.Schedule.", "Alias", schedule_alias, lbuf, sizeof(lbuf));
				strncat(s_ref, lbuf, sizeof(s_ref) - strlen(s_ref) - 1);
				strncat(s_ref, ",", sizeof(s_ref) - strlen(s_ref) - 1);

			}
		}
		// Remove the trailing , if needed
		size_t len = strlen(s_ref);
		if (len > 0 && s_ref[len - 1] == ',') {
			s_ref[len - 1] = '\0';
		}
		*value = dmstrdup(s_ref);
	}

	return 0;
}

static int set_HostsAccessControl_ScheduleRef(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *allowed_obj[] = {
		"Device.Schedules.Schedule.",
		NULL
	};
	size_t length, i;
	char **arr;
	struct dm_reference reference = {0};

	switch (action)	{
		case VALUECHECK:
			if (DM_STRLEN(value) == 0)
				break;

			if (bbfdm_validate_string_list(ctx, value, -1, -1, -1, -1, -1, NULL, NULL))
				return FAULT_9007;

			arr = strsplit(value, ",", &length);
			for (i = 0; i < length; i++) {
				memset(&reference, 0 , sizeof(struct dm_reference));
				bbfdm_get_reference_linker(ctx, arr[i], &reference);

				if (DM_STRLEN(reference.path) == 0)
					continue;

				if (dm_validate_allowed_objects(ctx, &reference, allowed_obj))
					return FAULT_9007;
			}

			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "access_control_schedule", "");
			if (DM_STRLEN(value) == 0)
				break;

			arr = strsplit(value, ",", &length);
			for (i = 0; i < length; i++) {
				memset(&reference, 0 , sizeof(struct dm_reference));
				bbfdm_get_reference_linker(ctx, arr[i], &reference);

				if (DM_STRLEN(reference.path) == 0 || DM_STRLEN(reference.value) == 0)
					continue;

				// the following logic is necessary since we want to
				// avoid refering to schedule_alias which only exists
				// in dmmap_schedules from hosts uci file. So, we are
				// extracting the section name to populate the uci
				struct uci_section *s = NULL;
				char *section_name = NULL;
				uci_path_foreach_option_eq(bbfdm, "dmmap_schedules", "schedule", "schedule_alias", reference.value, s) {
					dmuci_get_value_by_section_string(s, "section_name", &section_name);
					if (section_name == NULL)
						continue;
					dmuci_add_list_value_by_section(((struct dm_data *)data)->config_section, "access_control_schedule", section_name);
				}
			}

			break;
	}

	return 0;
}

/**********************************************************************************************************************************
*                                            OBJ & LEAF DEFINITION
***********************************************************************************************************************************/
/*** Device. ***/
DMOBJ tDeviceObjs[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys*/
{"Hosts", &DMREAD, NULL, NULL, "file:/etc/config/hosts", NULL, NULL, NULL, tHostsObj, tHostsParams, NULL, BBFDM_BOTH, NULL},
{0}
};

/* *** Device.Hosts. *** */
DMOBJ tHostsObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Host", &DMREAD, NULL, NULL, NULL, browseHostsHostInst, NULL, NULL, tHostsHostObj, tHostsHostParams, NULL, BBFDM_BOTH, NULL},
{"AccessControl", &DMWRITE, addObjHostsAccessControl, delObjHostsAccessControl, NULL, browseHostsAccessControlInst, NULL, NULL, NULL, tHostsAccessControlParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tHostsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"HostNumberOfEntries", &DMREAD, DMT_UNINT, get_Hosts_HostNumberOfEntries, NULL, BBFDM_BOTH},
{"AccessControlNumberOfEntries", &DMREAD, DMT_UNINT, get_Hosts_AccessControlNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.Hosts.Host.{i}. *** */
DMOBJ tHostsHostObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"IPv4Address", &DMREAD, NULL, NULL, NULL, browseHostsHostIPv4AddressInst, NULL, NULL, NULL, tHostsHostIPv4AddressParams, NULL, BBFDM_BOTH, NULL},
{"IPv6Address", &DMREAD, NULL, NULL, NULL, browseHostsHostIPv6AddressInst, NULL, NULL, NULL, tHostsHostIPv6AddressParams, NULL, BBFDM_BOTH, NULL},
{"WANStats", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tHostsHostWANStatsParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tHostsHostParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"PhysAddress", &DMREAD, DMT_STRING, get_HostsHost_PhysAddress, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE|DM_FLAG_LINKER},
{"IPAddress", &DMREAD, DMT_STRING, get_HostsHost_IPAddress, NULL, BBFDM_BOTH},
{"DHCPClient", &DMREAD, DMT_STRING, get_HostsHost_DHCPClient, NULL, BBFDM_BOTH, DM_FLAG_REFERENCE},
{"AssociatedDevice", &DMREAD, DMT_STRING, get_HostsHost_AssociatedDevice, NULL, BBFDM_BOTH, DM_FLAG_REFERENCE},
{"Layer1Interface", &DMREAD, DMT_STRING, get_HostsHost_Layer1Interface, NULL, BBFDM_BOTH, DM_FLAG_REFERENCE},
{"Layer3Interface", &DMREAD, DMT_STRING, get_HostsHost_Layer3Interface, NULL, BBFDM_BOTH, DM_FLAG_REFERENCE},
{"InterfaceType", &DMREAD, DMT_STRING, get_HostsHost_InterfaceType, NULL, BBFDM_BOTH},
{"HostName", &DMREAD, DMT_STRING, get_HostsHost_HostName, NULL, BBFDM_BOTH},
{"Active", &DMREAD, DMT_BOOL, get_HostsHost_Active, NULL, BBFDM_BOTH},
{"ActiveLastChange", &DMREAD, DMT_TIME, get_HostsHost_ActiveLastChange, NULL, BBFDM_BOTH},
{"IPv4AddressNumberOfEntries", &DMREAD, DMT_UNINT, get_HostsHost_IPv4AddressNumberOfEntries, NULL, BBFDM_BOTH},
{"IPv6AddressNumberOfEntries", &DMREAD, DMT_UNINT, get_HostsHost_IPv6AddressNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.Hosts.Host.{i}.IPv4Address.{i}. *** */
DMLEAF tHostsHostIPv4AddressParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"IPAddress", &DMREAD, DMT_STRING, get_HostsHostIPv4Address_IPAddress, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{0}
};

/* *** Device.Hosts.Host.{i}.IPv6Address.{i}. *** */
DMLEAF tHostsHostIPv6AddressParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"IPAddress", &DMREAD, DMT_STRING, get_HostsHostIPv6Address_IPAddress, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{0}
};

/* *** Device.Hosts.Host.{i}.WANStats. *** */
DMLEAF tHostsHostWANStatsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"BytesSent", &DMREAD, DMT_UNLONG, get_HostsHostWANStats_BytesSent, NULL, BBFDM_BOTH},
{"BytesReceived", &DMREAD, DMT_UNLONG, get_HostsHostWANStats_BytesReceived, NULL, BBFDM_BOTH},
{"PacketsSent", &DMREAD, DMT_UNLONG, get_HostsHostWANStats_PacketsSent, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNLONG, get_HostsHostWANStats_PacketsReceived, NULL, BBFDM_BOTH},
{0}
};

DMLEAF tHostsAccessControlParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Alias", &DMWRITE, DMT_STRING, get_HostsAccessControl_Alias, set_HostsAccessControl_Alias, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"PhysAddress", &DMWRITE, DMT_STRING, get_HostsAccessControl_PhysAddress, set_HostsAccessControl_PhysAddress, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"HostName", &DMWRITE, DMT_STRING, get_HostsAccessControl_HostName, set_HostsAccessControl_HostName, BBFDM_BOTH},
{"Enable", &DMWRITE, DMT_BOOL, get_HostsAccessControl_Enable, set_HostsAccessControl_Enable, BBFDM_BOTH},
{"AccessPolicy", &DMWRITE, DMT_STRING, get_HostsAccessControl_AccessPolicy, set_HostsAccessControl_AccessPolicy, BBFDM_BOTH},
{"ScheduleRef", &DMWRITE, DMT_STRING, get_HostsAccessControl_ScheduleRef, set_HostsAccessControl_ScheduleRef, BBFDM_BOTH, DM_FLAG_REFERENCE},
{0}
};

