/*
 * main.c - main entry point.
 *
 * Copyright (C) 2023 IOPSYS Software Solutions AB. All rights reserved.
 *
 * See LICENSE file for license related information.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>

#include "version.h"
#include "util.h"
#include "useropts.h"

const char *PROG_NAME = "hostmngr";
extern int hostmngr_main(void *useropts);


static void usage(void)
{
	fprintf(stdout, "Usage: %s [options]\n", PROG_NAME);
	fprintf(stdout, "\n");
	fprintf(stdout, "options:\n");
	fprintf(stdout, "   -h               usage\n");
	fprintf(stdout, "   -v               print version\n");
	fprintf(stdout, "   -s <socket path> ubus socket\n");
	fprintf(stdout, "   -D               run as a daemon\n");
	fprintf(stdout, "   -c <conf-file>   specify configuration file\n");
	fprintf(stdout, "   -d               debug level; more 'd's mean more verbose\n");
	fprintf(stdout, "   -p <pidfile>     pid file path\n");
	fprintf(stdout, "   -o <file>        log to file\n");
	fprintf(stdout, "   -f               treat above file as fifo\n");
	fprintf(stdout, "\n");
}

static void print_version(void)
{
	printf("%s: version '%s-g%s'\n", PROG_NAME, verstring, githash);
}

int main(int argc, char **argv)
{
	struct hostmngr_useropts opts = {
		.ubus_sockpath = NULL,
		.pidfile = HOSTMNGR_PIDFILE,
		.objname = HOSTS_OBJECT,
		.daemonize = false,
		.conffile = HOSTS_CONFFILE,
		.confpath = HOSTS_CONFFILE_PATH,
		.debug_level = 2,
		.syslogging = false,
		.logfile = NULL,
		.logfile_isfifo = false,
		.alid = NULL,
		.lo = true,
		.num_exclude = 0,
		.num_zone = 0,
		.num_ifname = 0,
	};
	static struct option lopts[] = {
		{ "version", 0, 0, 0 },
		{ "config", 1, 0, 0 },
		/* { "zone", 1, 0, 0 }, */
		{ "ifname", 1, 0, 0 },
		{ "exclude-ifname", 1, 0, 0 },
		{ "help", 0, 0, 0 },
		{ 0, 0, 0, 0 },
	};
	int lidx = 0;
	int ch;

	for (;;) {
		ch = getopt_long(argc, argv, "vhdDp:s:O:d:c:o:fl", lopts, &lidx);
		if (ch == -1)
			break;

		switch (ch) {
		case 0:
			switch (lidx) {
			case 0:
				print_version();
				exit(0);
			case 1:
				if (optarg) {
					opts.conffile = optarg;
					printf("Config file: %s\n", optarg);
				}
				break;
			/*
			case 2:
				if (optarg) {
					if (opts.num_zone < MAX_ZONES) {
						opts.zone[opts.num_zone] = strdup(optarg);
						opts.num_zone++;
						printf("Zone: %s\n", optarg);
					}
				}
				break;
			*/
			case 2:
				if (optarg) {
					if (opts.num_ifname < MAX_IFNAMES) {
						opts.ifname[opts.num_ifname] = strdup(optarg);
						opts.num_ifname++;
						printf("Interface: %s\n", optarg);
					}
				}
				break;
			case 3:
				if (optarg) {
					if (opts.num_exclude < MAX_ALLOWED_EXCLUDE) {
						opts.exclude_ifnames[opts.num_exclude] = strdup(optarg);
						opts.num_exclude++;
						printf("Exclude ifname: %s\n", optarg);
					}
				}
				break;
			case 4:
				usage();
				exit(0);
			default:
				break;
			}
			break;
		case 'v':
			print_version();
			exit(0);
		case 'h':
			usage();
			exit(0);
		case 'd':
			opts.debug_level++;
			break;
		case 's':
			opts.ubus_sockpath = optarg;
			break;
		case 'D':
			opts.daemonize = true;
			break;
		case 'p':
			opts.pidfile = optarg;
			break;
		case 'c':
			opts.conffile = optarg;
			break;
		case 'O':
			opts.objname = optarg;
			break;
		case 'o':
			opts.logfile = optarg;
			break;
		case 'f':
			opts.logfile_isfifo = true;
			break;
		case 'l':
			opts.syslogging = true;
			break;
		default:
			break;
		}
	}

	hostmngr_main(&opts);

	return 0;
}
