/*
 * utils.c - Utility function implementation
 *
 * Copyright (C) 2020 iopsys Software Solutions AB. All rights reserved.
 *
 * Author: yalu.zhang@iopsys.eu
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */
#include <stdarg.h>
#include <stddef.h>
#include <setjmp.h>
#include <cmocka.h>

#include <libubus.h>
#include <libubox/blobmsg_json.h>
#include <libubox/blobmsg.h>

#include <json-validator.h>
#include <json-c/json.h>
#include <json-editor.h>
#include <json-c/json_tokener.h>

#include "dslmngr.h"
#include "utils.h"

#define JSON_GET_OBJ(parent_obj, child_obj, child_name) do { \
			(child_obj) = json_object_get_by_string((parent_obj), (child_name)); \
			assert_non_null(child_obj); \
		} while(0)

struct json_object *json_output = NULL;

const char *interval_types[] = { "total", "showtime", "lastshowtime", "currentday", "quarterhour", NULL };
static const char *stats_counters[] = { "bytes_sent", "bytes_received", "packets_sent", "packets_received",
	"errors_sent", "errors_received", "discard_packets_sent", "discard_packets_received", "total_start",
	"showtime_start", "last_showtime_start", "current_day_start", "quarter_hour_start", NULL };

void validate_dsl_line_status(struct json_object *json_obj)
{
	struct json_object *tmp, *us, *ds;
	struct array_list *array;

	assert_non_null(json_obj);

	tmp = json_object_get_by_string(json_obj, "status");
	assert_non_null(tmp);
	assert_string_equal(json_object_get_string(tmp), "up");

	tmp = json_object_get_by_string(json_obj, "link_status");
	assert_non_null(tmp);
	assert_string_equal(json_object_get_string(tmp), "up");

	tmp = json_object_get_by_string(json_obj, "link_status");
	assert_non_null(tmp);
	assert_string_equal(json_object_get_string(tmp), "up");

	tmp = json_object_get_by_string(json_obj, "standard_used");
	assert_non_null(tmp);
	assert_string_equal(json_object_get_string(tmp), "vdsl2_annexa");

	tmp = json_object_get_by_string(json_obj, "current_profile");
	assert_non_null(tmp);
	assert_string_equal(json_object_get_string(tmp), "17a");

	tmp = json_object_get_by_string(json_obj, "power_management_state");
	assert_non_null(tmp);
	assert_string_equal(json_object_get_string(tmp), "l0");

	tmp = json_object_get_by_string(json_obj, "max_bit_rate");
	assert_non_null(tmp);
	us = json_object_get_by_string(tmp, "us");
	assert_true(us != NULL && json_object_get_int(us) > 65000);
	ds = json_object_get_by_string(tmp, "ds");
	assert_true(us != NULL && json_object_get_int(ds) > 148000);

	tmp = json_object_get_by_string(json_obj, "allowed_profiles");
	assert_non_null(tmp);
	array = json_object_get_array(tmp);
	assert_true(array != NULL && array_list_length(array) == 9);
	tmp = (struct json_object *)array_list_get_idx(array, 0);
	assert_non_null(tmp);
	assert_string_equal(json_object_get_string(tmp), "8a");
	tmp = (struct json_object *)array_list_get_idx(array, 8);
	assert_non_null(tmp);
	assert_string_equal(json_object_get_string(tmp), "35b");

	tmp = json_object_get_by_string(json_obj, "line_encoding");
	assert_non_null(tmp);
	assert_string_equal(json_object_get_string(tmp), "dmt");

	tmp = json_object_get_by_string(json_obj, "success_failure_cause");
	assert_non_null(tmp);
	assert_int_equal(json_object_get_int(tmp), 0);

	tmp = json_object_get_by_string(json_obj, "xtuc_ansi_rev");
	assert_non_null(tmp);
	assert_int_equal(json_object_get_int(tmp), 42129);
}

void validate_dsl_channel_status(struct json_object *json_obj)
{
	struct json_object *tmp, *us, *ds;
	struct array_list *array;

	assert_non_null(json_obj);

	JSON_GET_OBJ(json_obj, tmp, "status");
	assert_string_equal(json_object_get_string(tmp), "up");

	JSON_GET_OBJ(json_obj, tmp, "link_encapsulation_used");
	assert_string_equal(json_object_get_string(tmp), "vdsl2_ptm");

	JSON_GET_OBJ(json_obj, tmp, "curr_rate");
	JSON_GET_OBJ(tmp, us, "us");
	assert_true(json_object_get_int(us) > 60*1000);
	JSON_GET_OBJ(tmp, ds, "ds");
	assert_true(json_object_get_int(ds) > 100*1000);

	JSON_GET_OBJ(json_obj, tmp, "actndr");
	JSON_GET_OBJ(tmp, us, "us");
	assert_true(json_object_get_int(us) > 59*1000);
	JSON_GET_OBJ(tmp, ds, "ds");
	assert_true(json_object_get_int(ds) > 99*1000);

	JSON_GET_OBJ(json_obj, tmp, "link_encapsulation_supported");
	array = json_object_get_array(tmp);
	assert_true(array != NULL && array_list_length(array) == 5);
	tmp = (struct json_object *)array_list_get_idx(array, 0);
	assert_non_null(tmp);
	assert_string_equal(json_object_get_string(tmp), "adsl2_atm");
	tmp = (struct json_object *)array_list_get_idx(array, 4);
	assert_non_null(tmp);
	assert_string_equal(json_object_get_string(tmp), "auto");

	JSON_GET_OBJ(json_obj, tmp, "rfec");
	assert_int_equal(json_object_get_int(tmp), -1);

	JSON_GET_OBJ(json_obj, tmp, "actinprein");
	JSON_GET_OBJ(tmp, us, "us");
	assert_int_equal(json_object_get_int(us), 0);
	JSON_GET_OBJ(tmp, ds, "ds");
	assert_int_equal(json_object_get_int(ds), 0);
}

void validate_dsl_status(struct json_object *json_obj)
{
	struct json_object *tmp, *line, *channel;
	struct array_list *array;

	assert_non_null(json_obj);

	// Validate line status
	JSON_GET_OBJ(json_obj, tmp, "line");
	array = json_object_get_array(tmp);
	assert_true(array != NULL && array_list_length(array) == 1);
	line = (struct json_object *)array_list_get_idx(array, 0);
	validate_dsl_line_status(line);

	// Validate channel status
	JSON_GET_OBJ(line, tmp, "channel");
	array = json_object_get_array(tmp);
	assert_true(array != NULL && array_list_length(array) == 1);
	channel = (struct json_object *)array_list_get_idx(array, 0);
	validate_dsl_channel_status(channel);
}

void validate_dsl_line_stats(struct json_object *json_obj)
{
	struct json_object *tmp;
	const char **counter, **interval;

	assert_non_null(json_obj);

	for (counter = stats_counters; *counter != NULL; counter++) {
		JSON_GET_OBJ(json_obj, tmp, (char *)*counter);
		assert_true(json_object_get_int(tmp) >= 0);
	}

	// Validate all interval statistics
	for (interval = interval_types; *interval != NULL; interval++) {
		JSON_GET_OBJ(json_obj, tmp, (char *)*interval);

		validate_dsl_line_stats_interval(tmp, *interval);
	}
}

void validate_dsl_line_stats_interval(struct json_object *json_obj, const char *interval)
{
	struct json_object *tmp;

	assert_non_null(json_obj);

	JSON_GET_OBJ(json_obj, tmp, "errored_secs");
	assert_true(json_object_get_int(tmp) >= 0);

	JSON_GET_OBJ(json_obj, tmp, "severely_errored_secs");
	assert_true(json_object_get_int(tmp) >= 0);
}

void validate_dsl_channel_stats(struct json_object *json_obj)
{
	struct json_object *tmp;
	const char **counter, **interval;

	assert_non_null(json_obj);

	for (counter = stats_counters; *counter != NULL; counter++) {
		JSON_GET_OBJ(json_obj, tmp, (char *)*counter);
		assert_true(json_object_get_int(tmp) >= 0);
	}

	// Validate all interval statistics
	for (interval = interval_types; *interval != NULL; interval++) {
		JSON_GET_OBJ(json_obj, tmp, (char *)*interval);

		validate_dsl_channel_stats_interval(tmp, *interval);
	}
}

void validate_dsl_channel_stats_interval(struct json_object *json_obj, const char *interval)
{
	struct json_object *tmp;
	int64_t xtur_fec_errors, xtuc_fec_errors, xtur_hec_errors, xtuc_hec_errors;
	int64_t xtur_crc_errors, xtuc_crc_errors;
	const int64_t not_supported = (int64_t)4294967295;

	assert_non_null(json_obj);

	JSON_GET_OBJ(json_obj, tmp, "xtur_fec_errors");
	xtur_fec_errors = json_object_get_int64(tmp);
	if (strcmp(interval, "lastshowtime") == 0)
		assert_true(xtur_fec_errors == not_supported);
	else
		assert_true(xtur_fec_errors == 0);

	JSON_GET_OBJ(json_obj, tmp, "xtuc_fec_errors");
	xtuc_fec_errors = json_object_get_int64(tmp);
	if (strcmp(interval, "lastshowtime") == 0)
		assert_true(xtuc_fec_errors == not_supported);
	else
		assert_true(xtuc_fec_errors == 0);

	JSON_GET_OBJ(json_obj, tmp, "xtur_hec_errors");
	xtur_hec_errors = json_object_get_int64(tmp);
	if (strcmp(interval, "showtime") == 0)
		assert_true(xtur_hec_errors == 0);
	else
		assert_true(xtur_hec_errors == not_supported);

	JSON_GET_OBJ(json_obj, tmp, "xtuc_hec_errors");
	xtuc_hec_errors = json_object_get_int64(tmp);
	if (strcmp(interval, "showtime") == 0)
		assert_true(xtuc_hec_errors == 0);
	else
		assert_true(xtuc_hec_errors == not_supported);

	JSON_GET_OBJ(json_obj, tmp, "xtur_crc_errors");
	xtur_crc_errors = json_object_get_int64(tmp);
	if (strcmp(interval, "lastshowtime") == 0)
		assert_true(xtur_crc_errors == not_supported);
	else
		assert_true(xtur_crc_errors == 0);

	JSON_GET_OBJ(json_obj, tmp, "xtuc_crc_errors");
	xtuc_crc_errors = json_object_get_int64(tmp);
	if (strcmp(interval, "lastshowtime") == 0)
		assert_true(xtuc_crc_errors == not_supported);
	else
		assert_true(xtuc_crc_errors == 0);
}

void validate_dsl_stats(struct json_object *json_obj)
{
	struct json_object *tmp, *line, *channel;
	struct array_list *array;

	assert_non_null(json_obj);

	// Validate line status
	JSON_GET_OBJ(json_obj, tmp, "line");
	array = json_object_get_array(tmp);
	assert_true(array != NULL && array_list_length(array) == 1);
	line = (struct json_object *)array_list_get_idx(array, 0);
	JSON_GET_OBJ(line, tmp, "id");
	assert_true(json_object_get_int(tmp) >= 1);
	validate_dsl_line_stats(line);

	// Validate channel status
	JSON_GET_OBJ(line, tmp, "channel");
	array = json_object_get_array(tmp);
	assert_true(array != NULL && array_list_length(array) == 1);
	channel = (struct json_object *)array_list_get_idx(array, 0);
	JSON_GET_OBJ(channel, tmp, "id");
	assert_true(json_object_get_int(tmp) >= 1);
	validate_dsl_channel_stats(channel);
}
