/*
 * dslmngr_nl.c - converts netlink messages to UBUS events
 *
 * Copyright (C) 2019 iopsys Software Solutions AB. All rights reserved.
 *
 * Author: anjan.chanda@iopsys.eu
 *         yalu.zhang@iopsys.eu
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <signal.h>
#include <netlink/netlink.h>
#include <netlink/socket.h>
#include <netlink/genl/ctrl.h>
#include <netlink/genl/genl.h>
#include <netlink/attr.h>
#include <json-c/json.h>
#include "libubox/blobmsg_json.h"
#include "libubus.h"

#define NETLINK_FAMILY_NAME "easysoc"
#define NETLINK_GROUP_NAME  "notify"

#define MAX_MSG 512

/* nl attributes */
enum {
	XDSL_NL_UNSPEC,
	XDSL_NL_MSG,
	__XDSL_NL_MAX,
};

static struct nla_policy nl_notify_policy[__XDSL_NL_MAX] = {
	[XDSL_NL_MSG] = { .type = NLA_STRING },
};

static struct nlattr *attrs[__XDSL_NL_MAX];

static void
remove_char(char *buf, char ch, size_t buf_len)
{
	char newbuf[strlen(buf)+1];
	int i = 0;
	int j = 0;

	while (buf[i]) {
		newbuf[j] = buf[i];
		if (buf[i] != ch)
			j++;
		i++;
	}
	newbuf[j] = '\0';
	strncpy(buf, newbuf, buf_len);
}

static void
json_get_var(json_object *obj, const char *var, char *value, size_t value_len)
{
	json_object_object_foreach(obj, key, val) {
		if(!strcmp(key, var)) {
			switch (json_object_get_type(val)) {
			case json_type_object:
				break;
			case json_type_array:
				break;
			case json_type_string:
				snprintf(value, value_len, "%s", json_object_get_string(val));
				break;
			case json_type_boolean:
				snprintf(value, value_len, "%d", json_object_get_boolean(val));
				break;
			case json_type_int:
				snprintf(value, value_len, "%d", json_object_get_int(val));
				break;
			default:
				break;
			}
		}
	}
}

static void
json_parse_and_get(char *str, const char *var, char *value, size_t value_len)
{
	json_object *obj;

	obj = json_tokener_parse(str);
	if (!obj || json_object_get_type(obj) != json_type_object) {
		return;
	}
	json_get_var(obj, var, value, value_len);
}

static int dslmngr_hotplug_event(char *message)
{
	char data[MAX_MSG + 1] = {0};
	char event[32 + 1] = {0};
	char mode[16] = {};
	char status[32] = {};
	char cmd[512] = {0};

	sscanf(message, "%32s '%512[^\n]s'", event, data);

	/* ignore non-dsl events */
	if (strcmp(event, "dsl") != 0)
		return 0;

	remove_char(data, '\'', sizeof(data));
	json_parse_and_get(data, "mode", mode, sizeof(mode));
	json_parse_and_get(data, "status", status, sizeof(status));
	snprintf(cmd, sizeof(cmd), "STATUS=%s MODE=%s /sbin/hotplug-call dsl", status, mode);
	system(cmd); /* Flawfinder: ignore */
	return 0;

}

static int dslmngr_ubus_event(struct ubus_context *ctx, char *message)
{
	char data[MAX_MSG + 1] = {0};
	char event[32 + 1] = {0};
	struct blob_buf b;

	sscanf(message, "%32s '%512[^\n]s'", event, data);

	/* ignore non-dsl events */
	if (!strstr(event, "dsl."))
		return 0;

	memset(&b, 0, sizeof(b));
	blob_buf_init(&b, 0);

	if (!blobmsg_add_json_from_string(&b, data)) {
		fprintf(stderr, "Failed to parse message data: %s\n", data);
		return -1;
	}

	ubus_send_event(ctx, event, b.head);
	blob_buf_free(&b);

	return 0;
}

static int dslmngr_nl_to_ubus_event(struct nl_msg *msg, void *arg)
{
	struct nlmsghdr *nlh = nlmsg_hdr(msg);
	struct ubus_context *ctx = (struct ubus_context *)arg;
	int ret;

	if (!genlmsg_valid_hdr(nlh, 0)) {
		fprintf(stderr, "received invalid message\n");
		return 0;
	}

	ret = genlmsg_parse(nlh, 0, attrs, XDSL_NL_MSG, nl_notify_policy);
	if (!ret) {
		if (attrs[XDSL_NL_MSG] ) {
			char message[MAX_MSG] = {0};

			memcpy(message, nla_get_string(attrs[XDSL_NL_MSG]),
						nla_len(attrs[XDSL_NL_MSG]));

			dslmngr_ubus_event(ctx, message);
			dslmngr_hotplug_event(message);
		}
	}
	return 0;
}

int dslmngr_nl_msgs_handler(void *p)
{
	struct ubus_context *ctx = (struct ubus_context *)p;
	struct nl_sock *sock;
	int grp;
	int err;

	sock = nl_socket_alloc();
	if(!sock){
		fprintf(stderr, "Error: nl_socket_alloc\n");
		return -1;
	}

	nl_socket_disable_seq_check(sock);
	err = nl_socket_modify_cb(sock, NL_CB_VALID, NL_CB_CUSTOM,
				dslmngr_nl_to_ubus_event, ctx);
	if (err < 0) {
		fprintf(stderr, "Error: %s\n", nl_geterror(err));
		return -1;
	}

	if ((err = genl_connect(sock)) < 0){
		fprintf(stderr, "Error: %s\n", nl_geterror(err));
		return -1;
	}

	if ((grp = genl_ctrl_resolve_grp(sock,
					NETLINK_FAMILY_NAME,
					NETLINK_GROUP_NAME)) < 0) {
		return -1;
	}

	nl_socket_add_membership(sock, grp);

	while (1) {
		err = nl_recvmsgs_default(sock);
		if (err < 0) {
			fprintf(stderr, "Error: %s (%s grp %s)\n",
					nl_geterror(err),
					NETLINK_FAMILY_NAME,
					NETLINK_GROUP_NAME);
		}
	}

	return 0;
}
