/*
 * unit_test_dslmngr.c - dslmngr unit tests
 *
 * Copyright (C) 2020 iopsys Software Solutions AB. All rights reserved.
 *
 * Author: yalu.zhang@iopsys.eu
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdarg.h>
#include <stddef.h>
#include <setjmp.h>
#include <errno.h>
#include <cmocka.h>

#include <libubus.h>
#include <libubox/blobmsg_json.h>
#include <libubox/blobmsg.h>

#include <json-validator.h>
#include <json-c/json.h>
#include <json-editor.h>
#include <json-c/json_tokener.h>

#include "dslmngr.h"
#include "utils.h"

#define LIBDSL_LOG_INPUT_PARAMS "/tmp/libdsl-input-params.log"

struct test_ctx {
	struct blob_buf bb;
	struct ubus_object dsl;
	struct ubus_object dsl_line;
	struct ubus_object dsl_channel;
	struct ubus_object atm_link;
};

/* Overload ubus_send_reply to prevent segment fault*/
int ubus_send_reply(struct ubus_context *ctx, struct ubus_request_data *req, struct blob_attr *msg)
{
	if (msg) {
		char *json_text;

		json_text = blobmsg_format_json_indent(msg, true, 0);
		//printf("%s\n", json_text);
		json_output = json_tokener_parse(json_text);
		free(json_text);
	}

	if (json_output)
		return UBUS_STATUS_OK;
	return UBUS_STATUS_NO_DATA;
}

static int setup(void **state)
{
	struct test_ctx *ctx = (struct test_ctx *)*state;

	blob_buf_init(&ctx->bb, 0);

	// Delete the log file for libdsl input parameters dump
	unlink(LIBDSL_LOG_INPUT_PARAMS);

	return 0;
}

static int teardown(void **state)
{
	if (json_output) {
		json_object_put(json_output); // Free the output JSON object
		json_output = NULL;
	}

	return 0;
}

static int group_setup(void **state)
{
	struct test_ctx *ctx = calloc(1, sizeof(struct test_ctx));

	if (!ctx)
		return -1;

	ctx->dsl.name = "dsl";
	ctx->dsl_line.name = "dsl.line.1";
	ctx->dsl_channel.name = "dsl.channel.1";
	ctx->atm_link.name = "atm.link.1";
	memset(&ctx->bb, 0, sizeof(struct blob_buf));
	*state = ctx;

	return 0;
}

static int group_teardown(void **state)
{
	struct test_ctx *ctx = (struct test_ctx *)*state;

	blob_buf_free(&ctx->bb);
	free(ctx);

	return 0;
}

static void test_api_dsl_line_status(void **state)
{
	struct test_ctx *ctx = (struct test_ctx *) *state;
	struct blob_buf *bb = &ctx->bb;
	struct ubus_object *obj = &ctx->dsl_line;

	int res = dsl_line_status(NULL, obj, NULL, NULL, bb->head);
	assert_true(res == UBUS_STATUS_OK);

	validate_dsl_line_status(json_output);
}

static void test_api_dsl_channel_status(void **state)
{
	struct test_ctx *ctx = (struct test_ctx *) *state;
	struct blob_buf *bb = &ctx->bb;
	struct ubus_object *obj = &ctx->dsl_channel;

	int res = dsl_channel_status(NULL, obj, NULL, NULL, bb->head);
	assert_true(res == UBUS_STATUS_OK);

	validate_dsl_channel_status(json_output);
}

static void test_api_dsl_status(void **state)
{
	struct test_ctx *ctx = (struct test_ctx *) *state;
	struct blob_buf *bb = &ctx->bb;
	struct ubus_object *obj = &ctx->dsl;

	int res = dsl_status_all(NULL, obj, NULL, NULL, bb->head);
	assert_true(res == UBUS_STATUS_OK);

	validate_dsl_status(json_output);
}

static void test_api_dsl_line_stats(void **state)
{
	struct test_ctx *ctx = (struct test_ctx *) *state;
	struct blob_buf *bb = &ctx->bb;
	struct ubus_object *obj = &ctx->dsl_line;

	int res = dsl_line_stats(NULL, obj, NULL, NULL, bb->head);
	assert_true(res == UBUS_STATUS_OK);

	validate_dsl_line_stats(json_output);
}

static void test_api_dsl_line_stats_interval(void **state)
{
	struct test_ctx *ctx = (struct test_ctx *) *state;
	struct blob_buf *bb = &ctx->bb;
	struct ubus_object *obj = &ctx->dsl_line;
	const char **interval;

	for (interval = interval_types; *interval != NULL; interval++) {
		// Initialize
		setup(state);

		blobmsg_add_string(bb, "interval", *interval);

		int res = dsl_line_stats(NULL, obj, NULL, NULL, bb->head);
		assert_true(res == UBUS_STATUS_OK);

		validate_dsl_line_stats_interval(json_output, *interval);

		// Free the resources
		teardown(state);
	}
}

static void test_api_dsl_channel_stats(void **state)
{
	struct test_ctx *ctx = (struct test_ctx *) *state;
	struct blob_buf *bb = &ctx->bb;
	struct ubus_object *obj = &ctx->dsl_channel;

	int res = dsl_channel_stats(NULL, obj, NULL, NULL, bb->head);
	assert_true(res == UBUS_STATUS_OK);

	validate_dsl_channel_stats(json_output);
}

static void test_api_dsl_channel_stats_interval(void **state)
{
	struct test_ctx *ctx = (struct test_ctx *) *state;
	struct blob_buf *bb = &ctx->bb;
	struct ubus_object *obj = &ctx->dsl_channel;
	const char **interval;

	for (interval = interval_types; *interval != NULL; interval++) {
		// Initialize
		setup(state);

		blobmsg_add_string(bb, "interval", *interval);

		int res = dsl_channel_stats(NULL, obj, NULL, NULL, bb->head);
		assert_true(res == UBUS_STATUS_OK);

		validate_dsl_channel_stats_interval(json_output, *interval);

		// Free the resources
		teardown(state);
	}
}

static void test_api_dsl_stats(void **state)
{
	struct test_ctx *ctx = (struct test_ctx *) *state;
	struct blob_buf *bb = &ctx->bb;
	struct ubus_object *obj = &ctx->dsl;

	int res = dsl_stats_all(NULL, obj, NULL, NULL, bb->head);
	assert_true(res == UBUS_STATUS_OK);

	validate_dsl_stats(json_output);
}

static struct json_object *get_json_obj_from_log()
{
	int fd = -1;
	char *str = NULL;

	// Put the output of ubus call into a string
	fd = open(LIBDSL_LOG_INPUT_PARAMS, O_RDONLY);
	assert_return_code(fd, errno);

	struct stat st;
	int rc = fstat(fd, &st);
	assert_true(rc == 0 && st.st_size > 0);

	str = calloc(1, (size_t)st.st_size + 1);
	assert_non_null(str);

	ssize_t read_len = read(fd, str, (size_t)st.st_size);
	assert_int_equal((int)read_len, (int)st.st_size);

	// Parse the string to a json_object
	char *start = strchr(str, '{');
	assert_non_null(start);
	//puts(start);
	struct json_object *obj = json_tokener_parse(start);

	if (fd >= 0)
		close(fd);
	free(str);

	return obj;
}

static void test_api_dsl_configure(void **state)
{
	const char *xtse_val = "01,02,ab,cd,41,52,f8,6e";
	const char *vdsl2_profiles_val = "8a,8b,8c,8d,17a,30a,35b";
	const char *fast_profiles_val = "106a,212a";
	const bool data_gathering_val = true;
	const unsigned int limit_mask_val = 7;
	const unsigned int us0_mask_val = 3;

	struct test_ctx *ctx = (struct test_ctx *) *state;
	struct blob_buf *bb = &ctx->bb;
	struct ubus_object *obj = &ctx->dsl_line;

	// Prepare the input parameters
	blobmsg_add_string(bb, "xtse", xtse_val);
	blobmsg_add_string(bb, "vdsl2_profiles", vdsl2_profiles_val);
	blobmsg_add_string(bb, "fast_profiles", fast_profiles_val);
	blobmsg_add_u8(bb, "data_gathering", data_gathering_val);
	blobmsg_add_u32(bb, "limit_mask", limit_mask_val);
	blobmsg_add_u32(bb, "us0_mask", us0_mask_val);

	// Call the function to be tested
	int res = dsl_line_configure(NULL, obj, NULL, NULL, bb->head);
	assert_true(res == UBUS_STATUS_OK);

	// Check the results
	struct json_object *jobj = get_json_obj_from_log();
	assert_non_null(jobj);

	struct json_object *tmp = json_object_get_by_string(jobj, "xtse");
	assert_string_equal(json_object_get_string(tmp), xtse_val);

	tmp = json_object_get_by_string(jobj, "vdsl2_profiles");
	assert_string_equal(json_object_get_string(tmp), vdsl2_profiles_val);

	tmp = json_object_get_by_string(jobj, "fast_profiles");
	assert_string_equal(json_object_get_string(tmp), fast_profiles_val);

	tmp = json_object_get_by_string(jobj, "limit_mask");
	assert_int_equal(json_object_get_int(tmp), limit_mask_val);

	tmp = json_object_get_by_string(jobj, "us0_mask");
	assert_int_equal(json_object_get_int(tmp), us0_mask_val);

	// Free the memory
	json_object_put(jobj);
}

static void test_api_dsl_configure_fail(void **state)
{
	const char *xtse_val = "00,00,00,00,00,00,00,00"; // invalid
	const char *vdsl2_profiles_val = "30a,35b";
	const char *fast_profiles_val = "106a";

	struct test_ctx *ctx = (struct test_ctx *) *state;
	struct blob_buf *bb = &ctx->bb;
	struct ubus_object *obj = &ctx->dsl_line;

	// Prepare the input parameters
	blobmsg_add_string(bb, "xtse", xtse_val);
	blobmsg_add_string(bb, "vdsl2_profiles", vdsl2_profiles_val);
	blobmsg_add_string(bb, "fast_profiles", fast_profiles_val);

	// Call the function to be tested
	int res = dsl_line_configure(NULL, obj, NULL, NULL, bb->head);
	assert_true(res != UBUS_STATUS_OK);

	// Check the results
	struct json_object *jobj = get_json_obj_from_log();
	assert_non_null(jobj);

	struct json_object *tmp = json_object_get_by_string(jobj, "xtse");
	assert_string_equal(json_object_get_string(tmp), xtse_val);

	tmp = json_object_get_by_string(jobj, "vdsl2_profiles");
	assert_string_equal(json_object_get_string(tmp), vdsl2_profiles_val);

	tmp = json_object_get_by_string(jobj, "fast_profiles");
	assert_string_equal(json_object_get_string(tmp), fast_profiles_val);

	// Free the memory
	json_object_put(jobj);
}

static void test_api_atm_configure(void **state)
{
	const char *link_type_val = "eoa";
	const unsigned int vpi_val = 8; // invalid
	const unsigned int vci_val = 35;
	const char *qos_class_val = "ubr";
	const unsigned int max_burst_size_val = 8124;

	struct test_ctx *ctx = (struct test_ctx *) *state;
	struct blob_buf *bb = &ctx->bb;
	struct ubus_object *obj = &ctx->atm_link;

	// Prepare the input parameters
	blobmsg_add_string(bb, "link_type", link_type_val);
	blobmsg_add_u32(bb, "vpi", vpi_val);
	blobmsg_add_u32(bb, "vci", vci_val);
	blobmsg_add_string(bb, "qos_class", qos_class_val);
	blobmsg_add_u32(bb, "max_burst_size", max_burst_size_val);

	// Call the function to be tested
	int res = atm_link_configure(NULL, obj, NULL, NULL, bb->head);
	assert_true(res == UBUS_STATUS_OK);

	// Check the results
	struct json_object *jobj = get_json_obj_from_log();
	assert_non_null(jobj);

	struct json_object *tmp = json_object_get_by_string(jobj, "link_type");
	assert_string_equal(json_object_get_string(tmp), link_type_val);

	tmp = json_object_get_by_string(jobj, "vpi");
	assert_int_equal(json_object_get_int(tmp), vpi_val);

	tmp = json_object_get_by_string(jobj, "vci");
	assert_int_equal(json_object_get_int(tmp), vci_val);

	tmp = json_object_get_by_string(jobj, "qos_class");
	assert_string_equal(json_object_get_string(tmp), qos_class_val);

	tmp = json_object_get_by_string(jobj, "max_burst_size");
	assert_int_equal(json_object_get_int(tmp), max_burst_size_val);

	// Free the memory
	json_object_put(jobj);
}

static void test_api_atm_configure_fail(void **state)
{
	const char *link_type_val = "eoa";
	const unsigned int vpi_val = 0; // invalid
	const unsigned int vci_val = 35;
	const char *qos_class_val = "ubr";

	struct test_ctx *ctx = (struct test_ctx *) *state;
	struct blob_buf *bb = &ctx->bb;
	struct ubus_object *obj = &ctx->atm_link;

	// Prepare the input parameters
	blobmsg_add_string(bb, "link_type", link_type_val);
	blobmsg_add_u32(bb, "vpi", vpi_val);
	blobmsg_add_u32(bb, "vci", vci_val);
	blobmsg_add_string(bb, "qos_class", qos_class_val);

	// Call the function to be tested
	int res = atm_link_configure(NULL, obj, NULL, NULL, bb->head);
	assert_true(res != UBUS_STATUS_OK);

	// Check the results
	struct json_object *jobj = get_json_obj_from_log();
	assert_non_null(jobj);

	struct json_object *tmp = json_object_get_by_string(jobj, "link_type");
	assert_string_equal(json_object_get_string(tmp), link_type_val);

	tmp = json_object_get_by_string(jobj, "vpi");
	assert_int_equal(json_object_get_int(tmp), vpi_val);

	tmp = json_object_get_by_string(jobj, "vci");
	assert_int_equal(json_object_get_int(tmp), vci_val);

	tmp = json_object_get_by_string(jobj, "qos_class");
	assert_string_equal(json_object_get_string(tmp), qos_class_val);

	// Free the memory
	json_object_put(jobj);
}

int main(void)
{
	const struct CMUnitTest tests[] = {
		cmocka_unit_test_setup_teardown(test_api_dsl_line_status, setup, teardown),
		cmocka_unit_test_setup_teardown(test_api_dsl_channel_status, setup, teardown),
		cmocka_unit_test_setup_teardown(test_api_dsl_status, setup, teardown),
		cmocka_unit_test_setup_teardown(test_api_dsl_line_stats, setup, teardown),
		cmocka_unit_test_setup_teardown(test_api_dsl_line_stats_interval, setup, teardown),
		cmocka_unit_test_setup_teardown(test_api_dsl_channel_stats, setup, teardown),
		cmocka_unit_test_setup_teardown(test_api_dsl_channel_stats_interval, setup, teardown),
		cmocka_unit_test_setup_teardown(test_api_dsl_stats, setup, teardown),
		cmocka_unit_test_setup_teardown(test_api_dsl_configure, setup, teardown),
		cmocka_unit_test_setup_teardown(test_api_dsl_configure_fail, setup, teardown),
		cmocka_unit_test_setup_teardown(test_api_atm_configure, setup, teardown),
		cmocka_unit_test_setup_teardown(test_api_atm_configure_fail, setup, teardown)
	};

	return cmocka_run_group_tests(tests, group_setup, group_teardown);
}
