/*
 * unit_test_dslmngr.c - dslmngr unit tests
 *
 * Copyright (C) 2020 iopsys Software Solutions AB. All rights reserved.
 *
 * Author: yalu.zhang@iopsys.eu
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdarg.h>
#include <stddef.h>
#include <setjmp.h>
#include <errno.h>
#include <cmocka.h>

#include <libubus.h>
#include <libubox/blobmsg_json.h>
#include <libubox/blobmsg.h>

#include <json-validator.h>
#include <json-c/json.h>
#include <json-editor.h>
#include <json-c/json_tokener.h>

#include "dslmngr.h"
#include "utils.h"

#define UBUS_OUTPUT_FILE "/tmp/dslmngr_ubus_output.txt"

static int group_setup(void **state)
{
	// Start ubusd if it is not running
	system("pidof ubusd || ubusd &");	/* Flawfinder: ignore */

	// Restart dslmngr
	system("kill $(pidof dslmngr) 2>/dev/null; ../../dslmngr &");	/* Flawfinder: ignore */

	return 0;
}

static int group_teardown(void **state)
{
	// Stop dslmngr
	system("kill $(pidof dslmngr)");	/* Flawfinder: ignore */

	// Delete test logs
	unlink(UBUS_OUTPUT_FILE);

	return 0;
}

static int teardown(void **state)
{
	if (json_output) {
		json_object_put(json_output); // Free the output JSON object
		json_output = NULL;
	}

	return 0;
}

static void get_ubus_call_output(const char *ubus_cmd)
{
	char cmd[256];
	int fd = -1;
	char *str = NULL;

	// Put the output of ubus call into a string
	snprintf(cmd, sizeof(cmd), "%s > %s", ubus_cmd, UBUS_OUTPUT_FILE);
	int rc = system(cmd);	/* Flawfinder: ignore */
	assert_return_code(rc, errno);

	fd = open(UBUS_OUTPUT_FILE, O_RDONLY);
	assert_return_code(fd, errno);

	struct stat st;
	rc = fstat(fd, &st);
	assert_true(rc == 0 && st.st_size > 0);

	str = calloc(1, (size_t)st.st_size + 1);
	assert_non_null(str);

	ssize_t read_len = read(fd, str, (size_t)st.st_size);
	assert_int_equal((int)read_len, (int)st.st_size);

	// Parse the string to a json_object
	//puts(str);
	json_output = json_tokener_parse(str);

	if (fd >= 0)
		close(fd);
	if (str)
		free(str);
}

static void test_ubus_dsl_line_status(void **state)
{
	get_ubus_call_output("ubus call dsl.line.1 status");
	validate_dsl_line_status(json_output);
}

static void test_ubus_dsl_channel_status(void **state)
{
	get_ubus_call_output("ubus call dsl.channel.1 status");
	validate_dsl_channel_status(json_output);
}

static void test_ubus_dsl_status(void **state)
{
	get_ubus_call_output("ubus call dsl status");
	validate_dsl_status(json_output);
}

static void test_ubus_dsl_line_stats(void **state)
{
	get_ubus_call_output("ubus call dsl.line.1 stats");
	validate_dsl_line_stats(json_output);
}

static void test_ubus_dsl_line_stats_interval(void **state)
{
	const char **interval;
	char ubus_cmd[256];

	for (interval = interval_types; *interval != NULL; interval++) {
		snprintf(ubus_cmd, sizeof(ubus_cmd), "ubus call dsl.line.1 stats \"{'interval':'%s'}\"", *interval);
		get_ubus_call_output(ubus_cmd);
		validate_dsl_line_stats_interval(json_output, *interval);

		// Free resources
		teardown(state);
	}
}

static void test_ubus_dsl_channel_stats(void **state)
{
	get_ubus_call_output("ubus call dsl.channel.1 stats");
	validate_dsl_channel_stats(json_output);
}

static void test_ubus_dsl_channel_stats_interval(void **state)
{
	const char **interval;
	char ubus_cmd[256];

	for (interval = interval_types; *interval != NULL; interval++) {
		snprintf(ubus_cmd, sizeof(ubus_cmd), "ubus call dsl.channel.1 stats \"{'interval':'%s'}\"", *interval);
		get_ubus_call_output(ubus_cmd);
		validate_dsl_channel_stats_interval(json_output, *interval);

		// Free resources
		teardown(state);
	}
}

static void test_ubus_dsl_stats(void **state)
{
	get_ubus_call_output("ubus call dsl stats");
	validate_dsl_stats(json_output);
}

int main(void)
{
	const struct CMUnitTest tests[] = {
		cmocka_unit_test_teardown(test_ubus_dsl_line_status, teardown),
		cmocka_unit_test_teardown(test_ubus_dsl_channel_status, teardown),
		cmocka_unit_test_teardown(test_ubus_dsl_status, teardown),
		cmocka_unit_test_teardown(test_ubus_dsl_line_stats, teardown),
		cmocka_unit_test_teardown(test_ubus_dsl_line_stats_interval, teardown),
		cmocka_unit_test_teardown(test_ubus_dsl_channel_stats, teardown),
		cmocka_unit_test_teardown(test_ubus_dsl_channel_stats_interval, teardown),
		cmocka_unit_test_teardown(test_ubus_dsl_stats, teardown)
	};

	return cmocka_run_group_tests(tests, group_setup, group_teardown);
}
