/**
 * @file map2.h
 *
 * @copyright
 * Copyright (C) 2020 iopsys Software Solutions AB. All rights reserved.
 *
 * @author Vivek <vivek.dutta@iopsys.eu>
 *
 * @brief This header file consists structures and definitions as it is
 * defined in MultiAP specification
 * See LICENSE file for license related information.
 *
 * Document References:
 * [1]: https://www.wi-fi.org/file/multi-ap-specification
 */

#ifndef MAP2_H_
#define MAP2_H_
#include <stdint.h>
#include <stdbool.h>
#include <linux/types.h>
#include <string.h>

#ifdef __cplusplus
extern "C" {
#endif

#ifndef CMDU_MESSAGE_VERSION_1905_1_2013
#define CMDU_MESSAGE_VERSION_1905_1_2013  (0x00)
#endif

// map1905 extended CMDUs
#define CMDU_TYPE_TOPOLOGY_NOTIFICATION            (0x0001)
#define CMDU_TYPE_TOPOLOGY_QUERY                   (0x0002)
#define CMDU_TYPE_TOPOLOGY_RESPONSE                (0x0003)
#define CMDU_TYPE_AP_AUTOCONFIGURATION_SEARCH      (0x0007)
#define CMDU_TYPE_AP_AUTOCONFIGURATION_RESPONSE    (0x0008)
#define CMDU_TYPE_AP_AUTOCONFIGURATION_WSC         (0x0009)
// MAP-1 specific CMDUs
#define CMDU_1905_ACK                              (0x8000)
#define CMDU_AP_CAPABILITY_QUERY                   (0x8001)
#define CMDU_AP_CAPABILITY_REPORT                  (0x8002)
#define CMDU_POLICY_CONFIG_REQ                     (0x8003)
#define CMDU_CHANNEL_PREFERENCE_QUERY              (0x8004)
#define CMDU_CHANNEL_PREFERENCE_REPORT             (0x8005)
#define CMDU_CHANNEL_SELECTION_REQ                 (0x8006)
#define CMDU_CHANNEL_SELECTION_RESPONSE            (0x8007)
#define CMDU_OPERATING_CHANNEL_REPORT              (0x8008)
#define CMDU_CLIENT_CAPABILITY_QUERY               (0x8009)
#define CMDU_CLIENT_CAPABILITY_REPORT              (0x800a)
#define CMDU_AP_METRICS_QUERY                      (0x800b)
#define CMDU_AP_METRICS_RESPONSE                   (0x800c)
#define CMDU_ASSOC_STA_LINK_METRICS_QUERY          (0x800d)
#define CMDU_ASSOC_STA_LINK_METRICS_RESPONSE       (0x800e)
#define CMDU_UNASSOC_STA_LINK_METRIC_QUERY         (0x800f)
#define CMDU_UNASSOC_STA_LINK_METRIC_RESPONSE      (0x8010)
#define CMDU_BEACON_METRICS_QUERY                  (0x8011)
#define CMDU_BEACON_METRICS_RESPONSE               (0x8012)
#define CMDU_COMBINED_INFRA_METRICS                (0x8013)
#define CMDU_CLIENT_STEERING_REQUEST               (0x8014)
#define CMDU_CLIENT_STEERING_BTM_REPORT            (0x8015)
#define CMDU_CLIENT_ASSOC_CONTROL_REQUEST          (0x8016)
#define CMDU_STEERING_COMPLETED                    (0x8017)
#define CMDU_HIGHER_LAYER_DATA                     (0x8018)
#define CMDU_BACKHAUL_STEER_REQUEST                (0x8019)
#define CMDU_BACKHAUL_STEER_RESPONSE               (0x801a)
// MAP-2 specific CMDUs
#define CMDU_CHANNEL_SCAN_REQUEST                  (0x801b)
#define CMDU_CHANNEL_SCAN_REPORT                   (0x801c)
#define CMDU_CAC_REQUEST                           (0x8020)
#define CMDU_CAC_TERMINATION                       (0x8021)
#define CMDU_CLIENT_DISASSOCIATION_STATS           (0x8022)
#define CMDU_ERROR_RESPONSE                        (0x8024)
#define CMDU_ASSOCIATION_STATUS_NOTIFICATION       (0x8025)
#define CMDU_TUNNELED                              (0x8026)
#define CMDU_BACKHAUL_STA_CAPABILITY_QUERY         (0x8027)
#define CMDU_BACKHAUL_STA_CAPABILITY_REPORT        (0x8028)
#define CMDU_FAILED_CONNECTION                     (0x8033)

// 1905 specific TLV
#define TLV_TYPE_END_OF_MESSAGE                      (0)
#define TLV_TYPE_AL_MAC_ADDRESS_TYPE                 (1)
#define TLV_TYPE_DEVICE_INFORMATION_TYPE             (3)
#define TLV_TYPE_DEVICE_BRIDGING_CAPABILITIES        (4)
#define TLV_TYPE_NON_1905_NEIGHBOR_DEVICE_LIST       (6)
#define TLV_TYPE_NEIGHBOR_DEVICE_LIST                (7)
#define TLV_TYPE_TRANSMITTER_LINK_METRIC             (9)
#define TLV_TYPE_RECEIVER_LINK_METRIC                (10)
#define TLV_TYPE_SEARCHED_ROLE                       (13)
#define TLV_TYPE_AUTOCONFIG_FREQ_BAND                (14)
#define TLV_TYPE_SUPPORTED_ROLE                      (15)
#define TLV_TYPE_SUPPORTED_FREQ_BAND                 (16)
#define TLV_TYPE_WSC                                 (17)
// MULTIAP TLV types
#define MAP_TLV_SUPPORTED_SERVICE                      (0x80)
#define MAP_TLV_SEARCHED_SERVICE                       (0x81)
#define MAP_TLV_AP_RADIO_IDENTIFIER                    (0x82)
#define MAP_TLV_AP_OPERATIONAL_BSS                     (0x83)
#define MAP_TLV_ASSOCIATED_CLIENTS                     (0x84)
#define MAP_TLV_AP_CAPABILITY                          (0xA1)
#define MAP_TLV_AP_RADIO_BASIC_CAPABILITIES            (0x85)
#define MAP_TLV_AP_HT_CAPABILITIES                     (0x86)
#define MAP_TLV_AP_VHT_CAPABILITIES                    (0x87)
#define MAP_TLV_AP_HE_CAPABILITIES                     (0x88)
#define MAP_TLV_STEERING_POLICY                        (0x89)
#define MAP_TLV_METRIC_REPORTING_POLICY                (0x8A)
#define MAP_TLV_CHANNEL_PREFERENCE                     (0x8B)
#define MAP_TLV_RADIO_OPERATION_RESTRICTION            (0x8C)
#define MAP_TLV_TRANSMIT_POWER_LIMIT                   (0x8D)
#define MAP_TLV_CHANNEL_SELECTION_RESPONSE             (0x8E)
#define MAP_TLV_OPERATING_CHANNEL_REPORT               (0x8F)
#define MAP_TLV_CLIENT_INFO                            (0x90)
#define MAP_TLV_CLIENT_CAPABILITY_REPORT               (0x91)
#define MAP_TLV_CLIENT_ASSOCIATION_EVENT               (0x92)
#define MAP_TLV_AP_METRIC_QUERY                        (0x93)
#define MAP_TLV_AP_METRICS                             (0x94)
#define MAP_TLV_STA_MAC_ADDRESS                        (0x95)
#define MAP_TLV_ASSOCIATED_STA_LINK_METRICS            (0x96)
#define MAP_TLV_UNASSOCIATED_STA_LINK_METRICS_QUERY    (0x97)
#define MAP_TLV_UNASSOCIATED_STA_LINK_METRICS_RESPONSE (0x98)
#define MAP_TLV_BEACON_METRICS_QUERY                   (0x99)
#define MAP_TLV_BEACON_METRICS_RESPONSE                (0x9A)
#define MAP_TLV_STEERING_REQUEST                       (0x9B)
#define MAP_TLV_STEERING_BTM_REPORT                    (0x9C)
#define MAP_TLV_CLIENT_ASSOCIATION_CONTROL_REQUEST     (0x9D)
#define MAP_TLV_BACKHAUL_STEERING_REQUEST              (0x9E)
#define MAP_TLV_BACKHAUL_STEERING_RESPONSE             (0x9F)
#define MAP_TLV_HIGHER_LAYER_DATA                      (0xA0)
#define MAP_TLV_ASSOCIATED_STA_TRAFFIC_STATS           (0xA2)
#define MAP_TLV_ERROR_CODE                         (0xA3)
// MultiAP 2.0 profile-2 TLVs
#define MAP_TLV_CHANNEL_SCAN_REPORTING_POLICY      (0xA4)
#define MAP_TLV_CHANNEL_SCAN_CAPABILITY            (0xA5)
#define MAP_TLV_CHANNEL_SCAN_REQ                   (0xA6)
#define MAP_TLV_CHANNEL_SCAN_RES                   (0xA7)
#define MAP_TLV_TIMESTAMP                          (0xA8)
#define MAP_TLV_CAC_REQ                            (0xAD)
#define MAP_TLV_CAC_TERMINATION                    (0xAE)
#define MAP_TLV_CAC_COMPLETION_REPORT              (0xAF)
#define MAP_TLV_CAC_STATUS_REPORT                  (0xB1)
#define MAP_TLV_CAC_CAPABILITY                     (0xB2)
#define MAP_TLV_MULTIAP_PROFILE                    (0xB3)
#define MAP_TLV_PROFILE2_AP_CAPABILITY             (0xB4)
#define MAP_TLV_DEFAULT_8021Q_SETTINGS             (0xB5)
#define MAP_TLV_TRAFFIC_SEPARATION_POLICY          (0xB6)
#define MAP_TLV_PROFILE2_ERR_CODE                  (0xBC)
#define MAP_TLV_AP_RADIO_ADV_CAPABILITY            (0xBE)
#define MAP_TLV_ASSOCIATION_STATUS_NOTIF           (0xBF)
#define MAP_TLV_SOURCE_INFO                        (0xC0)
#define MAP_TLV_TUNNELED_MSG_TYPE                  (0xC1)
#define MAP_TLV_TUNNELED                           (0xC2)
#define MAP_TLV_PROFILE2_STEERING_REQ              (0xC3)
#define MAP_TLV_UNSUCCESS_ASSOCIATION_POLICY       (0xC4)
#define MAP_TLV_METRIC_COLLECTION_INTERVAL         (0xC5)
#define MAP_TLV_RADIO_METRICS                      (0xC6)
#define MAP_TLV_AP_EXTENDED_METRICS                (0xC7)
#define MAP_TLV_ASSOCIATED_STA_EXT_LINK_METRICS    (0xC8)
#define MAP_TLV_STATUS_CODE                        (0xC9)
#define MAP_TLV_REASON_CODE                        (0xCA)
#define MAP_TLV_BACKHAUL_STA_RADIO_CAPABILITY      (0xCB)
#define MAP_TLV_BACKHAUL_BSS_CONFIG                (0xD0)

// NOTE: If new types are introduced in future revisions of the standard,
// update this value so that it always points to the last+1.
#define MAP_TLV_LAST                                   (0xD1)
#define INVALID_TLV                                    (0xFF)

// 1905 related definitions
// Media types as detailed in "Table 6-12"
#define MEDIA_TYPE_IEEE_802_3U_FAST_ETHERNET       (0x0000)
#define MEDIA_TYPE_IEEE_802_3AB_GIGABIT_ETHERNET   (0x0001)
#define MEDIA_TYPE_IEEE_802_11B_2_4_GHZ            (0x0100)
#define MEDIA_TYPE_IEEE_802_11G_2_4_GHZ            (0x0101)
#define MEDIA_TYPE_IEEE_802_11A_5_GHZ              (0x0102)
#define MEDIA_TYPE_IEEE_802_11N_2_4_GHZ            (0x0103)
#define MEDIA_TYPE_IEEE_802_11N_5_GHZ              (0x0104)
#define MEDIA_TYPE_IEEE_802_11AC_5_GHZ             (0x0105)
#define MEDIA_TYPE_IEEE_802_11AD_60_GHZ            (0x0106)
#define MEDIA_TYPE_IEEE_802_11AF_GHZ               (0x0107)
#define MEDIA_TYPE_IEEE_1901_WAVELET               (0x0200)
#define MEDIA_TYPE_IEEE_1901_FFT                   (0x0201)
#define MEDIA_TYPE_MOCA_V1_1                       (0x0300)
#define MEDIA_TYPE_UNKNOWN                         (0xFFFF)

// IEEE802.11 frequency bands used in "Tables 6-22 and 6-24"
#define IEEE80211_ROLE_REGISTRAR                   (0x00)

// IEEE802.11 frequency bands used in "Tables 6-23 and 6-25"
#define IEEE80211_FREQUENCY_BAND_2_4_GHZ           (0x00)
#define IEEE80211_FREQUENCY_BAND_5_GHZ             (0x01)
#define IEEE80211_FREQUENCY_BAND_60_GHZ            (0x02)

#define MULTIAP_PROFILE_1                               0x01
#define MULTIAP_PROFILE_2                               0x02

//Enumeration to be used for determining transmission type
enum {
	TRANSMISSION_UNICAST,
	TRANSMISSION_RELIABLE_MULTICAST,
	TRANSMISSION_RELAYED_MULTICAST
};

/**
 * @brief End of message TLV associated structures ("Section 6.4.1")
 */
struct tlv_eom {
	uint8_t tlv_type; // TLV_TYPE_END_OF_MESSAGE
	uint16_t tlv_len;
};

/**
 * @brief AL MAC address type TLV associated structures ("Section 6.4.3")
 */
struct tlv_al_mac {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t al_mac_address[6];
};

/**
 * @brief Device information type TLV associated structures ("Section 6.4.5")
 */
struct tlv_device_info {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t al_mac_address[6];
	uint8_t local_interfaces_nr;

	struct {
		uint8_t mac_address[6];
		uint16_t media_type;
		uint8_t media_specific_data_size;
		union {
			uint8_t dummy;
			struct {
				uint8_t network_membership[6];
#define IEEE80211_SPECIFIC_INFO_ROLE_AP                   (0x0)
#define IEEE80211_SPECIFIC_INFO_ROLE_NON_AP_NON_PCP_STA   (0x4)
#define IEEE80211_SPECIFIC_INFO_ROLE_WIFI_P2P_CLIENT      (0x8)
#define IEEE80211_SPECIFIC_INFO_ROLE_WIFI_P2P_GROUP_OWNER (0x9)
#define IEEE80211_SPECIFIC_INFO_ROLE_AD_PCP               (0xa)
				uint8_t role;
				uint8_t ap_channel_band;
				uint8_t ap_ch_freq_1;
				uint8_t ap_ch_freq_2;
			} ieee80211;
			struct {
				uint8_t network_identifier[7];
			} ieee1901;
		} media_specific_data;
	} *local_interfaces;
};

/**
 * @brief Device bridging capability TLV associated structures ("Section 6.4.6")
 */
struct tlv_device_bridge_cap {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t bridging_tuples_nr;
	struct {
		uint8_t bridging_tuple_macs_nr;
		struct {
			uint8_t mac_address[6];
		} *bridging_tuple_macs;
	} *bridging_tuples;
};

/**
 * @brief Non-1905 neighbor device list TLV associated structures
 * ("Section 6.4.8")
 */
struct tlv_non1905_neighbor {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t local_mac_address[6];
	uint8_t non_1905_neighbors_nr;
	struct {
		uint8_t mac_address[6];
	} *non_1905_neighbors;
};

/**
 * @brief Neighbor device TLV associated structures ("Section 6.4.9")
 */
struct tlv_neighbor_device {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t local_mac_address[6];
	uint8_t neighbors_nr;
	struct {
		uint8_t mac_address[6];
		uint8_t bridge_flag;
	} *neighbors;
};

/**
 * @brief Transmitter link metric TLV associated structures ("Section 6.4.11")
 */
struct tlv_tx_link_metric {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t local_al_address[6];
	uint8_t neighbor_al_address[6];
	uint8_t transmitter_link_metrics_nr;
	struct {
		uint8_t local_interface_address[6];
		uint8_t neighbor_interface_address[6];
		uint16_t intf_type;
		uint8_t bridge_flag;
		uint32_t packet_errors;
		uint32_t transmitted_packets;
		uint16_t mac_throughput_capacity;
		uint16_t link_availability;
		uint16_t phy_rate;
	} *transmitter_link_metrics;
};

/**
 * @brief Receiver link metric TLV associated structures ("Section 6.4.12")
 */
struct tlv_rx_link_metric {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t local_al_address[6];
	uint8_t neighbor_al_address[6];
	uint8_t receiver_link_metrics_nr;
	struct {
		uint8_t local_interface_address[6];
		uint8_t neighbor_interface_address[6];
		uint16_t intf_type;
		uint32_t packet_errors;
		uint32_t packets_received;
		uint8_t rssi;
	} *receiver_link_metrics;
};

/**
 * @brief Searched role TLV associated structures ("Section 6.4.14")
 */
struct tlv_searched_role {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t role;
};

/**
 * @brief Autoconfig frequency band TLV associated structures ("Section 6.4.15")
 */
struct tlv_autoconf_freq_band {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t freq_band;
};

/**
 * @brief Supported role TLV associated structures ("Section 6.4.16")
 */
struct tlv_supported_role {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t role;
};

/**
 * @brief Supported frequency band TLV associated structures ("Section 6.4.17")
 */
struct tlv_supported_freq_band {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t freq_band;
};

/**
 * @brief Supported frequency band TLV associated structures ("Section 6.4.18")
 */
struct tlv_wsc {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint16_t wsc_frame_size;
	uint8_t *wsc_frame;
};

// MAP related TLVs

/**
 * @brief SupportedService TLV associated structures ("Section 17.2.1 of [1]")
 */
struct tlv_supp_service {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t supported_services_list;
	struct {
#define SUPPORTED_SERVICE_MULTIAP_CONTROLLER  (0x00)
#define SUPPORTED_SERVICE_MULTIAP_AGENT       (0x01)
		uint8_t service;
	} *supported_services;
};

/**
 * @brief SearchedService TLV associated structures ("Section 17.2.2 of [1]")
 */
struct tlv_searched_service {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t searched_services_list;
#define SEARCHED_SERVICE_MULTIAP_CONTROLLER  (0x00)
	uint8_t *service;
};

/**
 * @brief AP Radio Identifier TLV associated structures ("Section 17.2.3 of [1]")
 */
struct tlv_ap_radio_identifier {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t radio_id[6];
};

/**
 * @brief AP Operational BSS TLV associated structures ("Section 17.2.4 of [1]")
 */
struct tlv_ap_oper_bss {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t radios_nr;
	struct {
		uint8_t radio_id[6];
		uint8_t bss_nr;
		struct {
			uint8_t bssid[6];
			uint8_t ssid_len;
			char ssid[33];
		} *bss;
	} *radio;
};


/**
 * @brief Associated Clients TLV associated structures ("Section 17.2.5 of [1]")
 */
struct tlv_assoc_client {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t bss_nr;
	struct {
		uint8_t bssid[6];
		uint16_t assoc_clients_nr;
		struct {
			uint8_t client_addr[6];
			uint16_t uptime;
		} *clients;
	} *bss;
};

/**
 * @brief AP Capability TLV associated structures ("Section 17.2.6 of [1]")
 */
struct tlv_ap_cap {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t op_ch_metric_reporting;
	uint8_t non_op_ch_metric_reporting;
	uint8_t agent_init_rcpi_steering;
	uint8_t reserved;
};

/**
 * @brief AP Radio Basic Capabilities TLV structures ("Section 17.2.7 of [1]")
 */
struct tlv_ap_radio_basic_cap {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t radio_id[6];
	uint8_t max_bss_nr;
	uint8_t operating_classes_nr;
	struct {
		uint8_t op_class;
		uint8_t max_tx_power;
		uint8_t non_op_ch_nr;
		uint8_t *channel;
	} *operating_class;
};

/**
 * @brief AP HT Capabilities TLV associated structures ("Section 17.2.8 of [1]")
 */
struct tlv_ap_ht_cap {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t radio_id[6];
	uint8_t max_tx_streams_supported;
	uint8_t max_rx_streams_supported;
	uint8_t gi_20_support;
	uint8_t gi_40_support;
	uint8_t ht_40_support;
	uint8_t reserved;
};

/**
 * @brief AP VHT Capabilities TLV associated structures ("Section 17.2.9 of [1]")
 */
struct tlv_ap_vht_cap {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t radio_id[6];
	uint16_t vht_tx_mcs_supported;
	uint16_t vht_rx_mcs_supported;
	uint8_t max_tx_streams_supported;
	uint8_t max_rx_streams_supported;
	uint8_t gi_80_support;
	uint8_t gi_160_support;
	uint8_t vht_8080_support;
	uint8_t vht_160_support;
	uint8_t su_beamformer_capable;
	uint8_t mu_beamformer_capable;
	uint8_t reserved;
};

/**
 * @brief AP HE Capabilities TLV associated structures ("Section 17.2.10 of [1]")
 */
struct tlv_ap_he_cap {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t radio_id[6];
	uint8_t he_mcs_len;
	uint8_t *he_mcs_supported;
	uint8_t max_tx_streams_supported;
	uint8_t max_rx_streams_supported;
	uint8_t he_8080_support;
	uint8_t he_160_support;
	uint8_t su_beamformer_capable;
	uint8_t mu_beamformer_capable;
	uint8_t ul_mu_mimo_capable;
	uint8_t ul_mu_mimo_ofdma_capable;
	uint8_t dl_mu_mimo_ofdma_capable;
	uint8_t ul_ofdma_capable;
	uint8_t dl_ofdma_capable;
	uint8_t reserved;
};

/**
 * @brief Steering Policy TLV associated structures ("Section 17.2.11 of [1]")
 */
struct tlv_steering_policy {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t local_disallowed_sta_nr;
	struct {
		uint8_t addr[6];
	} *local_disallowed_sta_macs;
	uint8_t btm_disallowed_sta_nr;
	struct {
		uint8_t addr[6];
	} *btm_disallowed_sta_macs;
	uint8_t control_policy_radio_nr;
	struct {
		uint8_t radio_id[6];
		uint8_t steering_policy;
		uint8_t channel_utilization_thres;
		uint8_t rcpi_steering_thres;
	} *control_policy;
};

/**
 * @brief Metric Reporting Policy TLV associated structures ("Section 17.2.12 of [1]")
 */
struct tlv_metric_report_policy {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t interval;
	uint8_t metric_reporting_policy_radio_nr;
	struct {
		uint8_t radio_id[6];
		uint8_t rcpi_thres;
		uint8_t rcpi_hysteresis_margin;
		uint8_t channel_utilization_thres;
		uint8_t is_assoc_sta_traffic_stats;
		uint8_t is_assoc_sta_link_metrics;
		uint8_t reserved;
	} *metric_reporting_policy;
};

/**
 * @brief Channel Preference TLV associated structures ("Section 17.2.13 of [1]")
 */
struct tlv_channel_pref {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t radio_id[6];
	uint8_t ch_preference_op_class_nr;
	struct {
		uint8_t op_class;
		uint8_t channel_nr;
		uint8_t *channel_list;
		uint8_t preference;
		uint8_t preference_reason;
	} *op_class;
};

/**
 * @brief Radio Operation Restriction TLV structures ("Section 17.2.14 of [1]")
 */
struct tlv_radio_oper_restrict {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t radio_id[6];
	uint8_t op_restricted_op_class_nr;
	struct {
		uint8_t op_class;
		uint8_t channel_nr;
		struct {
			uint8_t channel;
			uint8_t min_freq_sep;
		} *restricted_channels;
	} *restricted_op_class;
};

/**
 * @brief Transmit Power Limit TLV associated structures ("Section 17.2.15 of [1]")
 */
struct tlv_tx_power_limit {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t radio_id[6];
	uint8_t tx_power_limit;
};

/**
 * @brief Channel Selection Response TLV structures ("Section 17.2.16 of [1]")
 */
struct tlv_ch_selection_resp {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t radio_id[6];
	uint8_t response_code;
};

/**
 * @brief Operating Channel Report TLV associated structures ("Section 17.2.17 of [1]")
 */
struct tlv_oper_ch_report {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t radio_id[6];
	uint8_t op_ch_op_class_nr;
	struct {
		uint8_t op_class;
		uint8_t channel;
	} *op_ch_op_class;
	uint8_t curr_tx_power;
};

/**
 * @brief Client Info TLV associated structures ("Section 17.2.18 of [1]")
 */
struct tlv_client_info {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t bssid[6];
	uint8_t client_addr[6];
};

/**
 * @brief Client Capability Report TLV associated structures ("Section 17.2.19 of [1]")
 */
struct tlv_client_cap_report {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t result_code;
	uint8_t *frame_body;
};

/**
 * @brief Client Association Event TLV associated structures ("Section 17.2.20 of [1]")
 */
struct tlv_client_assoc_event {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t client_addr[6];
	uint8_t bssid[6];
	uint8_t assoc_event;
	uint8_t reserved;
};

/**
 * @brief AP Metric Query TLV associated structures ("Section 17.2.21 of [1]")
 */
struct tlv_ap_metric_query {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t bssid_nr;
	struct {
		uint8_t bssid[6];
	} *ap_metric_query_bssid;
};

/**
 * @brief AP Metrics TLV associated structures ("Section 17.2.22 of [1]")
 */
struct tlv_ap_metrics {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t bssid[6];
	uint8_t channel_utilization;
	uint16_t total_assoc_stas;
	uint8_t is_ac_be;
	uint8_t is_ac_bk;
	uint8_t is_ac_vo;
	uint8_t is_ac_vi;
	uint8_t reserved;
	uint8_t service_param_info_be[3];
	uint8_t service_param_info_bk[3];
	uint8_t service_param_info_vo[3];
	uint8_t service_param_info_vi[3];
};

/**
 * @brief STA MAC Address Type TLV associated structures ("Section 17.2.23 of [1]")
 */
struct tlv_sta_mac {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t addr[6];
};

/**
 * @brief Associated STA Link Metrics TLV structures ("Section 17.2.24 of [1]")
 */
struct tlv_assoc_sta_link_metrics {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t addr[6];
	uint8_t bssid_nr;
	struct {
		uint8_t bssid[6];
		uint32_t time_delta;
		uint32_t dl_mac_data_rate;
		uint32_t ul_mac_data_rate;
		uint8_t ul_rcpi;
	} *sta_link_metrics_bssid;
};


/**
 * @brief Unssociated STA Link Metrics Query TLV structures ("Section 17.2.25 of [1]")
 */
struct tlv_unassoc_sta_link_metrics_query {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t op_class;
	uint8_t ch_nr;
	struct {
		uint8_t channel;
		uint8_t sta_mac_nr;
		struct {
			uint8_t addr[6];
		} *sta_mac;
	} *unassoc_sta_link_metrics_query_ch;
};

/**
 * @brief  Unssociated STA Link Metrics Response TLV ("Section 17.2.26 of [1]")
 */
struct tlv_unassoc_sta_link_metrics_resp {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t op_class;
	uint8_t sta_nr;
	struct {
		uint8_t addr[6];
		uint8_t channel;
		uint32_t time_delta;
		uint8_t ul_rcpi;
	} *unassoc_sta_link_metrics_sta;
};

/**
 * @brief  Beacon Metrics Query TLV associated structures ("Section 17.2.27 of [1]")
 */
struct tlv_beacon_metrics_query {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t sta_addr[6];
	uint8_t op_class;
	uint8_t channel;
	uint8_t bssid[6];
	uint8_t reporting_detail;
	uint8_t ssid_len;
	char ssid[33];
	uint8_t ch_reports_nr;
	struct {
		uint8_t report_len;
		uint8_t op_class;
		uint8_t *ch_list;
	} *ch_report;
	uint8_t elem_ids_nr;
	uint8_t *elem_list;
};

/**
 * @brief  Beacon Metrics Response TLV associated structures ("Section 17.2.28 of [1]")
 */
struct tlv_beacon_metrics_resp {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t sta_addr[6];
	uint8_t reserved;
	uint8_t report_elems_nr;
	uint8_t *report_elem;
};

/**
 * @brief  Steering Request TLV associated structures ("Section 17.2.29 of [1]")
 */
struct tlv_steer_Req {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t bssid[6];
	uint8_t request_mode;
	uint8_t btm_disassoc_imminent;
	uint8_t btm_abridged;
	uint8_t reserved;
	uint16_t steer_opp_window;
	uint16_t btm_disassoc_timer;
	uint8_t sta_list_cnt;
	struct {
		uint8_t addr[6];
	} *steering_req_macs;
	uint8_t target_bssid_list_cnt;
	struct {
		uint8_t bssid[6];
		uint8_t op_class;
		uint8_t channel;
	} *steering_req_target_bssids;
};

/**
 * @brief  Steering BTM Report TLV associated structures ("Section 17.2.30 of [1]")
 */
struct tlv_steer_btm_report {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t bssid[6];
	uint8_t sta_mac_addr[6];
	uint8_t status_code;
	uint8_t target_bssid[6];
};

/**
 * @brief Client Association Control Request TLV structures ("Section 17.2.31 of [1]")
 */
struct tlv_client_assoc_control_req {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t bssid[6];
	uint8_t assoc_control;
	uint16_t validity_period;
	uint8_t sta_list_cnt;
	struct {
		uint8_t addr[6];
	} *client_assoc_ctrl_req_stas;
};

/**
 * @brief  Backhaul Steering Request TLV structures ("Section 17.2.32 of [1]")
 */
struct tlv_backhaul_steer_req {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t addr[6];
	uint8_t bssid[6];
	uint8_t op_class;
	uint8_t channel;
};

/**
 * @brief  Backhaul Steering Response TLV structures ("Section 17.2.33 of [1]")
 */
struct tlv_backhaul_steer_resp {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t addr[6];
	uint8_t bssid[6];
	uint8_t res_code;
};

/**
 * @brief  Higher Layer Data TLV associated structures ("Section 17.2.34 of [1]")
 */
struct tlv_higher_layer_data {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t protocol;
	uint8_t *payload;
};

/**
 * @brief Associated STA Traffic Stats TLV structures ("Section 17.2.35 of [1]")
 */
struct tlv_assoc_sta_traffic_stats {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t addr[6];
	uint32_t bytes_sent;
	uint32_t bytes_received;
	uint32_t packets_sent;
	uint32_t packets_received;
	uint32_t tx_packets_err;
	uint32_t rx_packets_err;
	uint32_t retransmission_cnt;
};

/**
 * @brief  Error Code TLV associated structures ("Section 17.2.36 of [1]")
 */
struct tlv_error_code {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t reason_code;
	uint8_t addr[6];
};

/**
 * @brief  TLV channel scan reporting policy structures ("Section 17.2.37 of [1]")
 */
struct tlv_ch_scan_rep_policy {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t ch_scans;
};


/**
 * @brief  TLV channel scan capabilities structures ("Section 17.2.38 of [1]")
 */
struct tlv_ch_scan_cap {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t nbr_radios;
	struct {
		uint8_t radio_id[6];
		uint8_t on_boot;
		// BIT 7
		uint8_t scan_impact;
		// BIT 6-5
		uint8_t reserve;
		// BIT 4-0
		uint32_t min_scan_interval;
		uint8_t nbr_op_classes;
		struct {
			uint8_t op_class;
			uint8_t nbr_ch;
			uint8_t *ch;
		} *op_class;
	} *radio;
};

/**
 * @brief  TLV channel scan req structures ("Section 17.2.39 of [1]")
 */
struct tlv_ch_scan_req {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t fresh_scan;
	// BIT 7 // BIT 6-0 reserved
	uint8_t nbr_radios;
	struct {
		uint8_t radio_id[6];
		uint8_t nbr_op_class;
		struct {
			uint8_t op_class;
			uint8_t nbr_ch;
			uint8_t *ch;
		} *op_class_data;
	} *radio_data;
};

/**
 * @brief  TLV channel scan result structures ("Section 17.2.40 of [1]")
 */
struct tlv_ch_scan_res {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t radio_id[6];
	uint8_t op_class;
	uint8_t ch;
#define CH_SCAN_STATUS_SUCCESS                  0x00
#define CH_SCAN_STATUS_SCAN_NOT_SUPPORTED       0x01
#define CH_SCAN_STATUS_TOO_SOON                 0x02
#define CH_SCAN_STATUS_TOO_BUSY                 0x03
#define CH_SCAN_STATUS_SCAN_NOT_COMPLETED       0x04
#define CH_SCAN_STATUS_SCAN_ABORTED             0x05
#define CH_SCAN_STATUS_BOOT_SCAN_ONLY           0x06
	uint8_t scan_res;
	uint8_t time_len;
	//string '\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\.\d+Z|[\+\-]\d{2}:\d{2})'
	char *timestamp;
	uint8_t utilization;
	uint8_t noise;
	uint16_t nbr_neighbors;
	struct {
		uint8_t bssid[6];
		uint8_t ssid_len;
		char *ssid;
		// ssid_len
		uint8_t signal_strength;
		uint8_t ch_band_len;
		char *ch_bandwidth;
		// string "20", "80+80" etc
		uint8_t bss_load_elm_present;
		// BIT 7 // BIT 6-0 reserverd
		uint8_t ch_util;
		uint16_t sta_count;
	} *neighbor_data;
	uint32_t agg_scan_duration;
	/**< time in millisec */
	uint8_t scan_type;
	// BIT7 BIT6-0 reserved
};

/**
 * @brief  TLV timestamp structures ("Section 17.2.41 of [1]")
 */
struct tlv_timestamp {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t time_len;
	//string '\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\.\d+Z|[\+\-]\d{2}:\d{2})'
	char *time;
};

/**
 * @brief  TLV CAC request structures ("Section 17.2.42 of [1]")
 */
struct tlv_cac_req {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t nbr_radios;
	struct {
		uint8_t radio_id[6];
		uint8_t op_class;
		uint8_t ch;
#define CAC_REQ_METHOD_CONT_CAC                 0x00
#define CAC_REQ_METHOD_CONT_DEDICATED_RADIO     0x01
#define CAC_REQ_METHOD_MIMO_DIM_REDUCED         0x02
#define CAC_REQ_METHOD_TIME_SLICE_CAC           0x03
		uint8_t cac_method;
		// BIT7-5
#define CAC_REQ_ACTION_REMAIN_ON_CH             0x0
#define CAC_REQ_ACTION_RETURN_RADIO             0x1
		uint8_t cac_comp_action;
		// BIT 4-3 BIT2-0 reserved
	} *radio_data;
};

/**
 * @brief  TLV CAC termination structures ("Section 17.2.43 of [1]")
 */
struct tlv_cac_term {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t nbr_radio;
	struct {
		uint8_t radio_id[6];
		uint8_t op_class;
		uint8_t ch;
	} *radio_data;
};

/**
 * @brief  TLV CAC Completion Report structures ("Section 17.2.44 of [1]")
 */
struct tlv_cac_comp_report {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t nbr_radios;
	struct {
		uint8_t radio_id[6];
		uint8_t op_class;
		uint8_t ch;
#define CAC_COMP_REPORT_STATUS_SUCCESSFUL                       0x00
#define CAC_COMP_REPORT_STATUS_RADAR_DETECTED                   0x01
#define CAC_COMP_REPORT_STATUS_CAC_NOT_SUPPORTED                0x02
#define CAC_COMP_REPORT_STATUS_TOO_BUSY                         0x03
#define CAC_COMP_REPORT_STATUS_NON_CONFORMANT                   0x04
#define CAC_COMP_REPORT_STATUS_OTHER                            0x05
		uint8_t completion_status;
		uint8_t nbr_pairs;
		struct {
			uint8_t op_class_detected;
			uint8_t ch_detected;
		} *pair_data;
	} *radio_data;
};

struct _cacStatusRep {
	uint8_t op_class;
	uint8_t ch;
	uint16_t time;
};
/**
 * @brief  TLV CAC status Report structures ("Section 17.2.45 of [1]")
 */
struct tlv_cac_status_report {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t nbr_available_ch;
	struct _cacStatusRep *ch_data;
	/**< time in minutes */
	uint8_t nbr_pairs_duration;
	struct _cacStatusRep *duration_pair_data;
	/**< time in seconds */
	uint8_t nbr_pairs_coundown;
	struct _cacStatusRep *count_pair_data;
	/**< time in seconds */
};

/**
 * @brief  TLV CAC Capability structures ("Section 17.2.46 of [1]")
 */
struct tlv_cac_cap {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint16_t country_code;
	/**< as per ISO-3166 */
	uint8_t nbr_radios;
	struct {
		uint8_t radio_id[6];
		uint8_t nbr_cac_types;
		struct {
#define CAC_CAP_METHOD_CONTI_CAC                        0x00
#define CAC_CAP_METHOD_DEDICATED_RADIO                  0x01
#define CAC_CAP_METHOD_MIMO_DIM_REDUCED                 0x02
#define CAC_CAP_METHOD_TIME_SLICED                      0x03
			uint8_t cac_method_supported;
			uint32_t duration;
			/**< TLV has 3 octets for this field, value is in secs */
			uint8_t nbr_op_classes;
			struct {
				uint8_t op_class;
				uint8_t nbr_ch;
				uint8_t *ch;
			} *op_class_data;
		} *cac_data;
	} *radio_data;
};

/**
 * @brief  TLV MultiAP profile structures ("Section 17.2.47 of [1]")
 */
struct tlv_map_profile {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t profile;
};

/**
 * @brief  TLV Profile-2 AP Capability structures ("Section 17.2.48 of [1]")
 */
struct tlv_profile2_ap_cap {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint16_t reserved;
	uint8_t byte_counter;
	// BIT 7-6 // BIT 5-0
	uint8_t max_vids;
};

/**
 * @brief  TLV Default 802.1Q settings structures ("Section 17.2.49 of [1]")
 */
struct tlv_default_8021q_settings {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint16_t primary_vid;
	uint8_t pcp;
	// BIT 7-5 // BITs 4-0 reserved
};

/**
 * @brief  TLV Traffic Separation Policy structures ("Section 17.2.50 of [1]")
 */
struct tlv_traffic_sep_policy {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t nbr_ssid;
	struct {
		uint8_t ssid_len;
		char *ssid;
		// ssid_len
		uint16_t vid;
	} *data;
};

/**
 * @brief  TLV Profile2 Error Code structures ("Section 17.2.51 of [1]")
 */
struct tlv_profile2_err_code {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t reason_code;
	uint8_t bssid[6];
};

/**
 * @brief  TLV AP Radio Advanced Capability structures ("Section 17.2.52 of [1]")
 */
struct tlv_ap_radio_adv_cap {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t radio_id[6];
	uint8_t combined_front_back;
	// BIT 7
	uint8_t combined_p1_p2;
	// BIT 6 // BIT5-0 reserved
};

/**
 * @brief  TLV Association status notif structures ("Section 17.2.53 of [1]")
 */
struct tlv_assoc_status_notif {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t nbr_bssid;
	struct {
		uint8_t bssid[6];
		uint8_t association_allowance;
	} *data;
};

/**
 * @brief  TLV Source Info structures ("Section 17.2.54 of [1]")
 */
struct tlv_source_info {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t mac[6];
};

/**
 * @brief  TLV Tunneled message Type structures ("Section 17.2.55 of [1]")
 */
struct tlv_tunnel_msg_type {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t tunnel_protocol_type;
};

/**
 * @brief  TLV Tunneled structures ("Section 17.2.56 of [1]")
 */
struct tlv_tunneled {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t *frame_body;
	/**< 802.11 request frame body */
};

/**
 * @brief  TLV Profile2 steering req structures ("Section 17.2.57 of [1]")
 */
struct tlv_profile2_steer_req {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t bssid[6];
	uint8_t req_mode;
	// BIT7
	uint8_t btm_imminent_bit;
	// BIT 6
	uint8_t btm_abridged_bit;
	// BIT 5 // BIT 4-0 reserved
	uint16_t steering_opt_win;
	uint16_t btm_dis_timer;
	uint8_t sta_list_count;
	struct {
		uint8_t mac[6];
	} *sta_mac;
	uint8_t target_bssid_count;
	struct {
		uint8_t bssid[6];
		uint8_t bss_op_class;
		uint8_t bss_ch;
		uint8_t reason_code;
	} *target_data;
};

/**
 * @brief  TLV Unsuccessful Association Policy structures ("Section 17.2.58 of [1]")
 */
struct tlv_unsuccess_assoc_policy {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t report;
	// BIT 7 // BIT 6-0 reserved
	uint32_t max_reporting_rate;
	/**< rate per minute */
};

/**
 * @brief  TLV Metric Collection Interval structures ("Section 17.2.59 of [1]")
 */
struct tlv_metric_collection_interval {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint32_t collection_interval;
};

/**
 * @brief  TLV Radio Metrics structures ("Section 17.2.60 of [1]")
 */
struct tlv_radio_metrics {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t radio_id[6];
	uint8_t noise;
	uint8_t transmit;
	uint8_t receive_self;
	uint8_t receive_other;
};

/**
 * @brief  TLV AP Extended Metrics structures ("Section 17.2.61 of [1]")
 */
struct tlv_ap_ext_metrics {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t bssid[6];
	uint32_t uni_bytes_sent;
	uint32_t uni_bytes_recv;
	uint32_t multi_bytes_sent;
	uint32_t multi_bytes_recv;
	uint32_t bro_bytes_sent;
	uint32_t bro_bytes_recv;
};

/**
 * @brief  TLV Associated STA Ext Link Metrics structures ("Section 17.2.62 of [1]")
 */
struct tlv_assoc_sta_ext_link_metric {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t mac[6];
	uint8_t nbr_bssid;
	struct {
		uint8_t bssid[6];
		uint32_t last_data_dwn_rate;
		uint32_t last_data_up_rate;
		uint32_t uti_recv;
		uint32_t uti_transmit;
	} *data;
};

/**
 * @brief  TLV Status Code structures ("Section 17.2.63 of [1]")
 */
struct tlv_status_code {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint16_t status_code;
};

/**
 * @brief  TLV Reason code structures ("Section 17.2.64 of [1]")
 */
struct tlv_reason_code {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint16_t reason_code;
};

/**
 * @brief TLV Backhaul STA Radio Capabilities structures ("Section 17.2.65 of [1]")
 */
struct tlv_backhaul_sta_radio_cap {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t radio_id[6];
	uint8_t mac_include;
	// bit 7 // BIT 6-0 reserved
	uint8_t mac[6];
};

/**
 * @brief TLV Backhaul BSS Config structures ("Section 17.2.66 of [1]")
 */
struct tlv_backhaul_bss_config {
	uint8_t tlv_type;
	uint16_t tlv_len;
	uint8_t bssid[6];
	uint8_t p1;
	// BIT 7
	uint8_t p2;
	// BIT 6 // BIT 5-0 reserved
};

#define IFNAMESIZE 16
/**
 * @brief C structure to store cmdu information along with TLV C structures
 */
struct cmdu_cstruct {
	uint8_t message_version;
	uint16_t message_type;
	uint16_t message_id;
	char intf_name[IFNAMESIZE];
	uint8_t origin[6];
	uint8_t relay_indicator;
	uint8_t num_tlvs;
	uint8_t **tlvs;
	/**< Contains num_tlvs number of tlv c structures
	 * each corresponds to one specific tlv.
	 */
};

/**
 * @brief C structure to store cmdu information along with TLV binary streams
 */
struct cmdu_stream {
	uint8_t message_version;
	uint16_t message_type;
	uint16_t message_id;
	uint8_t relay_indicator;
	uint8_t nbr_tlv_streams;
	/**< number of tlv binary streams present in the cmdu */

	uint8_t **tlv_stream_list;
	/**< Contains nbr_tlv_streams number of binary streams
	 * each corresponds to one specific tlv.
	 * The "end of message" TLV is not included in this list.
	 */
};

#ifdef __cplusplus
}
#endif

#endif				// END MAP2_H_
