/*
 * Copyright (C) 2023-2024 IOPSYS Software Solutions AB. All rights reserved.
 *
 *
 * See LICENSE file for license related information.
 *
 * Author: Amin Ben Romdhane <amin.benromdhane@iopsys.eu>
 *
 */

#include <libbbfdm-api/dmcommon.h>
#include <libbbfdm-api/bbfdm_api.h>
#include <libbbfdm-ubus/bbfdm-ubus.h>

#include <easymesh.h> // for CHANNEL_PREF_MASK & CHANNEL_PREF_REASON

#include "../utils.h"
#include "../wifi_dataelements.h"

#define DPP_URISFILE "/etc/multiap/dpp_uris.json"

enum set_ssid_operations {
	SET_SSID_ADD,
	SET_SSID_REMOVE,
	SET_SSID_CHANGE,
	SET_SSID_INVALID
};

typedef struct set_ssid_operate_param {
	char *ssid;
	char *new_ssid;
	char *enab;
	char *key;
	char *band;
	char *akm;
	char *adv;
	char *mfp;
	char *haul;
	char *network_type;
	char *mld_enab;
	char *ts;
} set_ssid_param;

typedef struct validate_ssid_param {
	int op;
	int status;
	char *ssid;
} validate_ssid;

struct radio_info {
	struct list_head list;
	char radio_name[10];
	char band[5];
};

extern struct wifi_data_element *bbfdm_wifi_de; /* Global data element structure */
extern uint32_t bbfdm_collect_interval; /* Automatic refresh interval in seconds */

const char *to_base64_str(const uint8_t *src, size_t src_len, char *out, size_t out_len);
const char *operation_mode_to_str(enum operation_mode op_mode);
const char *scan_status_to_str(uint8_t scan_status);

static struct list_head dev_list = {0}; /* Global list for supported radios*/
static bool mld_capable = false; /* Global variable for mld capability */

/*************************************************************
* COMMON FUNCTIONS
**************************************************************/
static inline char *dm_to_macaddr_str(const uint8_t macaddr[6])
{
	char macaddr_str[18] = {0};
	hwaddr_ntoa(macaddr, macaddr_str);
	return dmstrdup(macaddr_str);
}

static char *get_default_key(void)
{
	char *def_key = NULL;

	json_object *json_obj = json_object_from_file(BOARD_JSON_FILE);
	if (!json_obj) {
		return dmstrdup("1234567890");
	}

	char *key = dmjson_get_value(json_obj, 5, "wlan", "defaults", "ssids", "all", "key");
	if (DM_STRLEN(key)) {
		def_key = dmstrdup(key);
	} else {
		def_key = dmstrdup("1234567890");
	}

	json_object_put(json_obj);

	return def_key;
}

static char *get_device_from_band(const char *band)
{
	struct radio_info *node = NULL;

	if (DM_STRLEN(band) == 0)
		return NULL;

	list_for_each_entry(node, &dev_list, list) {
		if (DM_STRNCMP(node->band, band, strlen(band)) == 0)
			return node->radio_name;
	}

	return NULL;
}

static void remove_unused_mld_sections(void)
{
	char *cur_mld = NULL;
	struct uci_section *ssid_s = NULL, *s = NULL, *stmp = NULL;

	if (mld_capable == false)
		return;

	uci_foreach_sections_safe("mapcontroller", "mld", s, stmp) {
		dmuci_get_value_by_section_string(s, "id", &cur_mld);

		ssid_s = get_dup_section_in_dmmap_opt("WiFi", "SSID", "mld_id", cur_mld);
		if (ssid_s != NULL)
			continue;

		// No SSID has assigned this mld unit so delete
		dmuci_delete_by_section(s, NULL, NULL);
	}
}

static void reload_mapcontroller_service(unsigned int proto, bool is_commit, bool reload_required)
{
	struct blob_buf bb = {0};

	memset(&bb, 0, sizeof(struct blob_buf));

	blob_buf_init(&bb, 0);

	void *array = blobmsg_open_array(&bb, "services");
	blobmsg_add_string(&bb, NULL, "mapcontroller");
	blobmsg_close_array(&bb, array);

	blobmsg_add_string(&bb, "proto", (proto == BBFDM_USP) ? "usp" : "both");
	blobmsg_add_u8(&bb, "reload", reload_required);

	dmubus_call_blob_msg_timeout("bbf.config", is_commit ? "commit" : "revert", &bb, 10000);

	blob_buf_free(&bb);
}

static char *find_radio_band(const char *agent_id, const char *radio_mac)
{
	struct uci_section *sec = NULL;
	char *band = NULL;

	uci_foreach_option_eq("mapcontroller", "radio", "agent_id", agent_id, sec) {
		char *macaddr = NULL;

		dmuci_get_value_by_section_string(sec, "macaddr", &macaddr);

		if (DM_STRCMP(macaddr, radio_mac) == 0) {
			dmuci_get_value_by_section_string(sec, "band", &band);
			break;
		}
	}

	return band;
}

static char *find_radio_mac(const char *agent_id, const char *radio_band)
{
	struct uci_section *sec = NULL;
	char *macaddr = NULL;

	uci_foreach_option_eq("mapcontroller", "radio", "agent_id", agent_id, sec) {
		char *band = NULL;

		dmuci_get_value_by_section_string(sec, "band", &band);

		if (DM_STRCMP(band, radio_band) == 0) {
			dmuci_get_value_by_section_string(sec, "macaddr", &macaddr);
			break;
		}
	}

	return macaddr;
}


static char *validate_bss(const char *bss_id, bool is_backhaul_bss)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;
	struct wifi_network_device *dev = NULL;

	if (!net || !bss_id)
		return NULL;

	list_for_each_entry(dev, &net->devicelist, list) { // Device List
		const struct wifi_radio_element *radio = NULL;
		char *dev_id = NULL;

		dev_id = dm_to_macaddr_str(dev->macaddr); // Device ID
		if (DM_STRLEN(dev_id) == 0)
			continue;

		list_for_each_entry(radio, &dev->radiolist, list) { // Radio List
			const struct wifi_bss_element *bss = NULL;
			char *radio_mac = NULL;

			radio_mac = dm_to_macaddr_str(radio->macaddr); // Radio MAC Address
			if (DM_STRLEN(radio_mac) == 0)
				continue;

			list_for_each_entry(bss, &radio->bsslist, list) { // BSS List
				char *bssid = dm_to_macaddr_str(bss->bssid); // BSS ID

				if (DM_STRCASECMP(bssid, bss_id) == 0 && bss->is_bbss == is_backhaul_bss) {
					return find_radio_band(dev_id, radio_mac);
				}
			} // BSS List
		} // Radio List
	} // Device List

	return NULL;
}

static char *get_backhaul_sta_from_radio(struct wifi_network_device *dev, const char *mac_addr)
{
	const struct wifi_radio_element *radio = NULL;

	list_for_each_entry(radio, &dev->radiolist, list) { // Radio List
		char *radio_mac = NULL;

		radio_mac = dm_to_macaddr_str(radio->macaddr); // Radio MAC Address
		if (DM_STRLEN(radio_mac) == 0)
			continue;

		if (DM_STRCASECMP(radio_mac, mac_addr) == 0) {
			return dm_to_macaddr_str(radio->bsta.macaddr);
		}
	} // Radio List

	return NULL;
}

static int fill_bsslist_from_radio(struct wifi_network_device *dev, const char *ruid, struct blob_buf *bb)
{
	const struct wifi_radio_element *radio = NULL;
	bool radio_found = false;

	if (!dev || DM_STRLEN(ruid) == 0 || !bb)
		return 0;

	list_for_each_entry(radio, &dev->radiolist, list) { // Radio List
		const struct wifi_bss_element *bss = NULL;
		char *radio_mac = NULL;
		char radio_id[32];

		to_base64_str(radio->macaddr, sizeof(radio->macaddr), radio_id, sizeof(radio_id));

		if (DM_STRCMP(radio_id, ruid) != 0) // Radio ID
			continue;

		radio_mac = dm_to_macaddr_str(radio->macaddr); // Radio MAC Address
		if (DM_STRLEN(radio_mac) == 0)
			continue;

		radio_found = true;

		// Add radio list
		void *radiolist = blobmsg_open_array(bb, "radiolist");
		blobmsg_add_string(bb, NULL, radio_mac);
		blobmsg_close_array(bb, radiolist);

		// Add bss list
		void *bsslist = blobmsg_open_array(bb, "bsslist");
		list_for_each_entry(bss, &radio->bsslist, list) { // BSS List
			blobmsg_add_string(bb, NULL, dm_to_macaddr_str(bss->bssid));
		} // BSS List
		blobmsg_close_array(bb, bsslist);
	} // Radio List

	return radio_found ? 0 : -1;
}

static char *get_AKMs_uci_name(const char *value)
{
	if (DM_LSTRCMP(value, "psk") == 0) {
		return "psk2";
	} else if(DM_LSTRCMP(value, "dpp") == 0) {
		return "dpp";
	} else if(DM_LSTRCMP(value, "sae") == 0) {
		return "sae";
	} else if(DM_LSTRCMP(value, "psk+sae") == 0) {
		return "sae-mixed";
	} else if(DM_LSTRCMP(value, "dpp+sae") == 0) {
		return "dpp+sae";
	} else if(DM_LSTRCMP(value, "dpp+psk+sae") == 0) {
		return "dpp+sae-mixed";
	}

	return "sae-mixed";
}

static char *get_AKMs_dm_name(const char *value)
{
	if (DM_LSTRCMP(value, "psk2") == 0) {
		return dmstrdup("psk");
	} else if(DM_LSTRCMP(value, "dpp") == 0) {
		return dmstrdup("dpp");
	} else if(DM_LSTRCMP(value, "sae") == 0) {
		return dmstrdup("sae");
	} else if(DM_LSTRCMP(value, "sae-mixed") == 0) {
		return dmstrdup("psk+sae");
	} else if(DM_LSTRCMP(value, "dpp+sae") == 0) {
		return dmstrdup("dpp+sae");
	} else if(DM_LSTRCMP(value, "dpp+sae-mixed") == 0) {
		return dmstrdup("dpp+psk+sae");
	}

	return dmstrdup("");
}

static char *get_haultype_uci_value(const char *value)
{
	if (DM_LSTRCMP(value, "Backhaul") == 0) {
		return "backhaul";
	}

	return "fronthaul";
}

static int get_requested_operation(const char *req)
{
	if (DM_LSTRCMP(req, "Add") == 0)
		return SET_SSID_ADD;

	if (DM_LSTRCMP(req, "Remove") == 0)
		return SET_SSID_REMOVE;

	if (DM_LSTRCMP(req, "Change") == 0)
		return SET_SSID_CHANGE;

	return SET_SSID_INVALID;
}

static int validate_band_value(struct dmctx *ctx, const char *band)
{
	char *band_list[] = {"2.4", "5", "6", "All", NULL};

	if (DM_STRLEN(band) == 0)
		return 0;

	if (bbfdm_validate_string_list(ctx, band, -1, -1, -1, -1, -1, band_list, NULL))
		return -1;

	/* if "All" is present then no other values are allowed in list */
	if (DM_STRSTR(band, "All") != NULL && DM_STRLEN(band) > 3)
		return -1;

	if (DM_STRSTR(band, "All") == NULL) {
		size_t band_arr_length = 0, i;
		char **band_arr = strsplit(band, ",", &band_arr_length);

		/* Only allow supported bands */
		for (i = 0; i < band_arr_length; i++) {
			bool valid = false;
			struct radio_info *node = NULL;

			list_for_each_entry(node, &dev_list, list) {
				if (node->band[0] == band_arr[i][0]) {
					valid = true;
					break;
				}
			}

			if (valid == false) {
				bbfdm_set_fault_message(ctx, "Device not supports band (%s).", band_arr[i]);
				return -1;
			}
		}
	}

	return 0;
}

static int validate_akms_value(struct dmctx *ctx, const char *akms)
{
	char *akms_list[] = {"psk", "dpp", "sae", "psk+sae", "dpp+sae", "dpp+psk+sae", NULL};

	if (DM_STRLEN(akms) == 0)
		return 0;

	if (bbfdm_validate_string_list(ctx, akms, -1, -1, -1, -1, -1, akms_list, NULL)) {
		bbfdm_set_fault_message(ctx, "Only supports psk,dpp,sae,psk+sae,dpp+sae,dpp+psk+sae.");
		return -1;
	}

	return 0;
}

static int validate_haultype_value(struct dmctx *ctx, const char *haul)
{
	char *haul_list[] = {"Fronthaul", "Backhaul", NULL};

	if (DM_STRLEN(haul) == 0)
		return 0;

	if (bbfdm_validate_string(ctx, haul, -1, -1, haul_list, NULL)) {
		bbfdm_set_fault_message(ctx, "Invalid haultype (%s).", haul);
		return -1;
	}

	return 0;
}

static void init_dev_list(void)
{
	struct uci_section *sec = NULL;
	struct bbfdm_ctx dm_ctx = {0};
	struct radio_info *node = NULL;
	char mlo_capable[5] = {0};

	memset(&dev_list, 0, sizeof(struct list_head));
	INIT_LIST_HEAD(&dev_list);

	memset(&dm_ctx, 0, sizeof(struct bbfdm_ctx));
	bbfdm_init_ctx(&dm_ctx);

	BBFDM_UCI_FOREACH_SECTION(&dm_ctx, "wireless", "wifi-device", sec) {
		char *dev = bbfdm_section_name(sec);
		if (strlen(dev) == 0)
			continue;

		node = (struct radio_info *)malloc(sizeof(struct radio_info));
		if (node == NULL)
			return;

		memset(node, 0, sizeof(struct radio_info));
		snprintf(node->radio_name, sizeof(node->radio_name), "%s", dev);
		bbfdm_uci_get_buf(&dm_ctx, "wireless", dev, "band", "", node->band, sizeof(node->band));
		bbfdm_uci_get_buf(&dm_ctx, "wireless", dev, "mlo_capable", "0", mlo_capable, sizeof(mlo_capable));

		INIT_LIST_HEAD(&node->list);
		list_add_tail(&node->list, &dev_list);
	}

	bbfdm_free_ctx(&dm_ctx);
	mld_capable = dmuci_string_to_boolean(mlo_capable);
}

static void free_dev_list(void)
{
	struct radio_info *node = NULL, *ntmp = NULL;

	list_for_each_entry_safe(node, ntmp, &dev_list, list) {
		list_del(&node->list);
		free(node);
	}
}

static void dmmap_synchronize_WiFi_DataElements_ProvisionedDPP(struct dmctx *dmctx)
{
	json_object *res = NULL, *arr = NULL;
	char *instance = NULL;
	int len, i;

	res = json_object_from_file(DPP_URISFILE);
	if (!res)
		return;

	if (!json_object_object_get_ex(res, "enrollees", &arr) ||
			!arr || !json_object_is_type(arr, json_type_array)) {
		json_object_put(res);
		return;
	}

	len = json_object_array_length(arr);
	for (i = 0; i < len; i++) {
		struct json_object *enrollee_obj, *alias_obj, *uri_obj;
		struct uci_section *dpp_s;
		const char *alias, *uri = NULL;

		enrollee_obj = json_object_array_get_idx(arr, i);
		if (!enrollee_obj || !json_object_is_type(enrollee_obj, json_type_object))
			continue;

		if (!json_object_object_get_ex(enrollee_obj, "alias", &alias_obj) ||
				!alias_obj || !json_object_is_type(alias_obj, json_type_string))
			continue;

		alias = json_object_get_string(alias_obj);
		if (!DM_STRLEN(alias))
			continue;

		if (json_object_object_get_ex(enrollee_obj, "uri", &uri_obj) &&
				uri_obj && json_object_is_type(uri_obj, json_type_string)) {
			uri = json_object_get_string(uri_obj);
		}

		/* Check if already exists */
		dpp_s = get_dup_section_in_dmmap_opt("WiFi", "ProvisionedDPP", "Alias", alias);
		if (dpp_s) {
			/* Update URI */
			dmuci_set_value_by_section(dpp_s, "uri", uri ? uri : "");
			continue;
		}

		/* Create new entry */
		dpp_s = create_dmmap_obj(dmctx, 0, "WiFi", "ProvisionedDPP", NULL, &instance);
		if (dpp_s && DM_STRLEN(instance)) {
			struct json_object *key_obj;

			dmuci_set_value_by_section(dpp_s, "Alias", alias);
			dmuci_set_value_by_section(dpp_s, "uri", uri);

			if (json_object_object_get_ex(enrollee_obj, "key", &key_obj) &&
					key_obj && json_object_is_type(key_obj, json_type_string)) {
				const char *key;

				key = json_object_get_string(key_obj);
				dmuci_set_value_by_section(dpp_s, "key", key);
			}
		}
	}

	json_object_put(res);
}

/*************************************************************
* ADD & DEL OBJ
**************************************************************/
static int addObjWiFiDataElementsNetworkProvisionedDPP(char *refparam, struct dmctx *ctx, void *data, char **instance)
{
	return 0;
}

static int delObjWiFiDataElementsNetworkProvisionedDPP(char *refparam, struct dmctx *ctx, void *data, char *instance, unsigned char del_action)
{
	char *uri;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "uri", &uri);
	if (DM_STRLEN(uri)) {
		bbfdm_set_fault_message(ctx,
			"DPP instance contains a defined URI. Instances with active credentials cannot be deleted.");
		return FAULT_9001;
	}

#ifdef DECOLLECTOR_VENDOR_EXTENSIONS
	char *key;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "key", &key);
	if (DM_STRLEN(key)) {
		bbfdm_set_fault_message(ctx,
			"DPP instance contains a defined Passphrase. Instances with active credentials cannot be deleted.");
		return FAULT_9001;
	}
#endif

	dmuci_delete_by_section(((struct dm_data *)data)->dmmap_section, NULL, NULL);
	return 0;
}

/*************************************************************
* ENTRY METHOD
**************************************************************/
static int browseWiFiDataElementsNetworkProvisionedDPPInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	dmmap_synchronize_WiFi_DataElements_ProvisionedDPP(dmctx);

	return generic_browse(dmctx, parent_node, prev_data, prev_instance);
}

static int browseWiFiDataElementsNetworkSSIDInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	uci_foreach_sections("mapcontroller", "ap", curr_data.config_section) {
		char *ssid = NULL, *band = NULL;

		dmuci_get_value_by_section_string(curr_data.config_section, "ssid", &ssid);
		dmuci_get_value_by_section_string(curr_data.config_section, "band", &band);

		if (DM_STRLEN(ssid) == 0 || DM_STRLEN(band) == 0)
			continue;

		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static struct uci_section *find_device_uci_section(const char *unique_key)
{
	struct uci_section *s = NULL;

	uci_foreach_sections("mapcontroller", "node", s) {
		char *agent_id = NULL;

		dmuci_get_value_by_section_string(s, "agent_id", &agent_id);

		// Device found ==> return the current device uci section
		if (DM_STRCMP(agent_id, unique_key) == 0)
			return s;
	}

	return NULL;
}

static struct uci_section *find_radio_uci_section(const char *agent_id, const char *unique_key)
{
	struct uci_section *s = NULL;

	uci_foreach_option_eq("mapcontroller", "radio", "agent_id", agent_id, s) {
		char *macaddr = NULL;

		dmuci_get_value_by_section_string(s, "macaddr", &macaddr);

		// Radio found ==> return the current radio uci section
		if (DM_STRCMP(macaddr, unique_key) == 0)
			return s;
	}

	return NULL;
}

static int browseWiFiDataElementsNetworkDeviceInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;

	if (net) {
		struct wifi_network_device *dev = NULL;
		struct dm_data curr_data = {0};
		char *inst = NULL;
		int id = 0;

		list_for_each_entry(dev, &net->devicelist, list) {
			char *dev_id = dm_to_macaddr_str(dev->macaddr); // ID

			curr_data.additional_data = (void *)dev;
			curr_data.config_section = find_device_uci_section(dev_id);

			inst = handle_instance_without_section(dmctx, parent_node, ++id);
			if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
				return 0;
		}
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceDefault8021QInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	DM_LINK_INST_OBJ(dmctx, parent_node, prev_data, "1");
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceSSIDtoVIDMappingInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;

	if (net) {
		struct wifi_network_ssid *ssid = NULL;
		struct dm_data curr_data = {0};
		char *inst = NULL;
		int id = 0;

		list_for_each_entry(ssid, &net->ssidlist, list) {
			curr_data.additional_data = (void *)ssid;
			inst = handle_instance_without_section(dmctx, parent_node, ++id);
			if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
				return 0;
		}
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceCACStatusInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_cac_status *cac_status_entry = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(cac_status_entry, &dev->cac_statuslist, list) {
		curr_data.additional_data = (void *)cac_status_entry;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceIEEE1905SecurityInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)prev_data)->additional_data;

	if (dev->i1905_seccap.caps_valid) {
		DM_LINK_INST_OBJ(dmctx, parent_node, prev_data, "1");
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceBackhaulDownInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_bh_down *bh_down = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(bh_down, &dev->bh_downlist, list) {
		curr_data.additional_data = (void *)bh_down;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_radio_element *radio = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	char *dev_id;
	int id = 0;

	dev_id = dm_to_macaddr_str(dev->macaddr); // ID

	list_for_each_entry(radio, &dev->radiolist, list) {
		char *radio_mac = dm_to_macaddr_str(radio->macaddr); // Radio MAC Address

		curr_data.additional_data = (void *)radio;
		curr_data.config_section = find_radio_uci_section(dev_id, radio_mac);

		inst = handle_instance_without_section(dmctx, parent_node, ++id);

		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}

	return 0;
}

static int browseWiFiDataElementsNetworkDeviceAPMLDInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_apmld_element *apmld = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(apmld, &dev->apmldlist, list) {
		curr_data.additional_data = (void *)apmld;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceCACStatusCACAvailableChannelInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_cac_status *cac_status_entry = (struct wifi_cac_status *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_cac_available_channel *cac_avail_entry = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(cac_avail_entry, &cac_status_entry->available_chlist, list) {
		curr_data.additional_data = (void *)cac_avail_entry;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannelInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_cac_status *cac_status_entry = (struct wifi_cac_status *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_cac_nop_channel *cac_nop_entry = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(cac_nop_entry, &cac_status_entry->nop_chlist, list) {
		curr_data.additional_data = (void *)cac_nop_entry;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceCACStatusCACActiveChannelInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_cac_status *cac_status_entry = (struct wifi_cac_status *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_cac_active_channel *cac_active_entry = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(cac_active_entry, &cac_status_entry->cac_chlist, list) {
		curr_data.additional_data = (void *)cac_active_entry;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfileInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)prev_data)->additional_data;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i;

	for (i = 0; i < dev->multi_ap_device.backhaul.cur_opclass.num_opclass; i++) {
		curr_data.additional_data = &dev->multi_ap_device.backhaul.cur_opclass.opclass[i];
		curr_data.config_section = (struct uci_section *)&dev->multi_ap_device.backhaul.cur_opclass.entry_time;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfileInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)prev_data)->additional_data;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i;

	for (i = 0; i < radio->cur_opclass.num_opclass; i++) {
		curr_data.additional_data = &radio->cur_opclass.opclass[i];
		curr_data.config_section = (struct uci_section *)&radio->cur_opclass.entry_time;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioOpClassPreferenceInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)prev_data)->additional_data;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i;

	for (i = 0; i < radio->pref_opclass.num_opclass &&
				i < ARRAY_SIZE(radio->pref_opclass.opclass); i++) {
		curr_data.additional_data = &radio->pref_opclass.opclass[i];
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioBSSInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_bss_element *bss = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(bss, &radio->bsslist, list) {
		curr_data.additional_data = (void *)bss;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioScanResultInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_scanres_element *sres = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(sres, &radio->scanlist, list) {
		curr_data.additional_data = (void *)sres;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioUnassociatedSTAInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_unassoc_sta_element *usta = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(usta, &radio->unassoc_stalist, list) {
		curr_data.additional_data = (void *)usta;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLMRFreqSeparationInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)prev_data)->additional_data;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	for (i = 0; i < MAX_NUM_MLO_FREQ_SEP && i < radio->wifi7caps_ap.num_freqsep_emlmr; i++) {
		curr_data.additional_data = &radio->wifi7caps_ap.emlmr_freqsep[i];
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLSRFreqSeparationInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)prev_data)->additional_data;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	for (i = 0; i < MAX_NUM_MLO_FREQ_SEP && i < radio->wifi7caps_ap.num_freqsep_emlsr; i++) {
		curr_data.additional_data = &radio->wifi7caps_ap.emlsr_freqsep[i];
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleSTRFreqSeparationInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)prev_data)->additional_data;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	for (i = 0; i < MAX_NUM_MLO_FREQ_SEP && i < radio->wifi7caps_ap.num_freqsep_str; i++) {
		curr_data.additional_data = &radio->wifi7caps_ap.str_freqsep[i];
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLMRFreqSeparationInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)prev_data)->additional_data;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	for (i = 0; i < MAX_NUM_MLO_FREQ_SEP && i < radio->wifi7caps_bsta.num_freqsep_emlmr; i++) {
		curr_data.additional_data = &radio->wifi7caps_bsta.emlmr_freqsep[i];
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLSRFreqSeparationInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)prev_data)->additional_data;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	for (i = 0; i < MAX_NUM_MLO_FREQ_SEP && i < radio->wifi7caps_bsta.num_freqsep_emlsr; i++) {
		curr_data.additional_data = &radio->wifi7caps_bsta.emlsr_freqsep[i];
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleSTRFreqSeparationInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)prev_data)->additional_data;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	for (i = 0; i < MAX_NUM_MLO_FREQ_SEP && i < radio->wifi7caps_bsta.num_freqsep_str; i++) {
		curr_data.additional_data = &radio->wifi7caps_bsta.str_freqsep[i];
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesAKMFrontHaulInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_akm_suite *bss_akm = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(bss_akm, &radio->fbss_akmlist, list) {
		curr_data.additional_data = (void *)bss_akm;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesAKMBackhaulInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_akm_suite *bss_akm = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(bss_akm, &radio->bbss_akmlist, list) {
		curr_data.additional_data = (void *)bss_akm;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfileInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)prev_data)->additional_data;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	for (i = 0; i < radio->supp_opclass.num_opclass; i++) {
		curr_data.additional_data = &radio->supp_opclass.opclass[i];
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistoryInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)prev_data)->additional_data;
	json_object *res = NULL, *sta_obj = NULL, *sta_arr = NULL, *inst_arr = NULL, *inst_obj = NULL;
	struct dm_data curr_data = {0};
	char *sta_mac = NULL, *inst = NULL;
	int sta_id = 0, inst_id = 0, ob = 0;

	sta_mac = dm_to_macaddr_str(sta->macaddr);
	if (!DM_STRLEN(sta_mac))
		return 0;

	dmubus_call("map.controller", "dump_steer_history", UBUS_ARGS{0}, 0, &res);

	dmjson_foreach_obj_in_array(res, sta_arr, sta_obj, sta_id, 1, "sta") {
		char *mac = dmjson_get_value(sta_obj, 1, "macaddr");

		if (DM_STRCMP(mac, sta_mac) != 0)
			continue;

		dmjson_foreach_obj_in_array(sta_obj, inst_arr, inst_obj, inst_id, 1, "history") {
			curr_data.json_object = inst_obj;
			inst = handle_instance_without_section(dmctx, parent_node, ++ob);
			if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
				break;
		}
		break;
	}

	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioBSSSTAInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_sta_element *sta = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(sta, &bss->stalist, list) {
		curr_data.additional_data = (void *)sta;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_scanres_element *sres = (struct wifi_scanres_element *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_scanres_opclass_element *op = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry_reverse(op, &sres->opclass_scanlist, list) {
		curr_data.additional_data = (void *)op;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_scanres_opclass_element *op = (struct wifi_scanres_opclass_element *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_scanres_channel_element *ch = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(ch, &op->channel_scanlist, list) {
		curr_data.additional_data = (void *)ch;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSSInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_scanres_channel_element *ch = (struct wifi_scanres_channel_element *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_scanres_neighbor_element *nbr = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(nbr, &ch->nbrlist, list) {
		curr_data.additional_data = (void *)nbr;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioScanCapabilityOpClassChannelsInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)prev_data)->additional_data;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	for (i = 0; i < radio->scan_caps.opclass.num_opclass; i++) {
		curr_data.additional_data = &radio->scan_caps.opclass.opclass[i];
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_radio_cac_capabilities *cac_caps = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(cac_caps, &radio->cac_capslist, list) {
		curr_data.additional_data = (void *)cac_caps;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodOpClassChannelsInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_radio_cac_capabilities *cac_caps = (struct wifi_radio_cac_capabilities *)((struct dm_data *)prev_data)->additional_data;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0, i = 0;

	for (i = 0; i < cac_caps->opclasses.num_opclass; i++) {
		curr_data.additional_data = &cac_caps->opclasses.opclass[i];
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			break;
	}
	return 0;
}


static int browseWiFiDataElementsNetworkDeviceAPMLDLinkToOpClassMapInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_apmld_element *apmld = (struct wifi_apmld_element *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_linkopcmap_element *linkopcmap = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(linkopcmap, &apmld->linkopcmaplist, list) {
		curr_data.additional_data = (void *)linkopcmap;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceAPMLDAffiliatedAPInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_apmld_element *apmld = (struct wifi_apmld_element *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_affiliated_ap_element *ap = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(ap, &apmld->aplist, list) {
		curr_data.additional_data = (void *)ap;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceAPMLDSTAMLDInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_apmld_element *apmld = (struct wifi_apmld_element *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_stamld_element *stamld = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(stamld, &apmld->stamldlist, list) {
		curr_data.additional_data = (void *)stamld;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

static int browseWiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTAInst(struct dmctx *dmctx, DMNODE *parent_node, void *prev_data, char *prev_instance)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)prev_data)->additional_data;
	const struct wifi_affiliated_sta_element *sta = NULL;
	struct dm_data curr_data = {0};
	char *inst = NULL;
	int id = 0;

	list_for_each_entry(sta, &stamld->stalist, list) {
		curr_data.additional_data = (void *)sta;
		inst = handle_instance_without_section(dmctx, parent_node, ++id);
		if (DM_LINK_INST_OBJ(dmctx, parent_node, (void *)&curr_data, inst) == DM_STOP)
			return 0;
	}
	return 0;
}

/**************************************************************************
* SET AND GET ALIAS
***************************************************************************/
static int get_WiFiDataElementsNetwork_ID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;
	if (net) {
		char network_id[37] = {0};
		uuid_btostr((uint8_t *)net->id, network_id);
		*value = dmstrdup(network_id);
	}
	return 0;
}

static int get_WiFiDataElementsNetwork_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char timestamp[32] = {0};

	*value = dmstrdup(get_timestamp(NULL, timestamp, sizeof(timestamp)));
	return 0;
}

static int get_WiFiDataElementsNetwork_ControllerID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;
	if (net) {
		*value = dm_to_macaddr_str(net->cntlr_id);
	}
	return 0;
}

static int get_WiFiDataElementsNetwork_DeviceNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;
	dmasprintf(value, "%u", net ? net->num_devices : 0);
	return 0;
}

static int get_WiFiDataElementsNetwork_ColocatedAgentID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;
	if (net) {
		*value = dm_to_macaddr_str(net->coagent_id);
	}
	return 0;
}

static int get_WiFiDataElementsNetwork_SSIDNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;
	dmasprintf(value, "%u", net ? net->num_ssid : 0);
	return 0;
}

static int get_WiFiDataElementsNetwork_ProvisionedDPPNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, NULL);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_WiFiDataElementsNetworkSSID_SSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "ssid", value);
	return 0;
}

static int get_WiFiDataElementsNetworkSSID_Band(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *tmp;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "band", &tmp);

	if (DM_STRCMP(tmp, "2") == 0) {
		*value = dmstrdup("2.4");
		return 0;
	}

	*value = dmstrdup(tmp);
	return 0;
}

static int get_WiFiDataElementsNetworkSSID_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "enabled", "1");
	return 0;
}

static int get_WiFiDataElementsNetworkSSID_AKMs(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *res = NULL;
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "encryption", &res);

	*value = get_AKMs_dm_name(res);
	return 0;
}

static int get_WiFiDataElementsNetworkSSID_Haul(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *tmp;

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "type", &tmp);

	if (DM_STRCMP(tmp, "backhaul") == 0) {
		*value = dmstrdup("Backhaul");
	} else {
		*value = dmstrdup("Fronthaul");
	}

	return 0;
}

static int get_WiFiDataElementsNetworkSSID_Type(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "network_type", value);
	return 0;
}

static int get_WiFiDataElementsNetworkProvisionedDPP_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	return bbf_get_alias(ctx, ((struct dm_data *)data)->dmmap_section, "Alias", instance, value);
}

static int set_WiFiDataElementsNetworkProvisionedDPP_Alias(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char *uri = NULL;
#ifdef DECOLLECTOR_VENDOR_EXTENSIONS
	char *key = NULL;
#endif

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 2048, NULL, NULL))
				return FAULT_9007;

			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "uri", &uri);
			if (DM_STRLEN(uri)) {
				bbfdm_set_fault_message(ctx,
					"DPP URI is already defined for this instance. Once defined, the Alias value cannot be changed.");
				return FAULT_9001;
			}

#ifdef DECOLLECTOR_VENDOR_EXTENSIONS
			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "key", &key);
			if (DM_STRLEN(key)) {
				bbfdm_set_fault_message(ctx,
					"DPP Passphrase is already defined for this instance. Once defined, the Alias value cannot be changed.");
				return FAULT_9001;
			}
#endif

			break;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "Alias", value);
			break;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkProvisionedDPP_DPPURI(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "uri", value);
	return 0;
}

static int set_WiFiDataElementsNetworkProvisionedDPP_DPPURI(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	json_object *res = NULL;
	char *alias = NULL;
	char *uri = NULL;
#ifdef DECOLLECTOR_VENDOR_EXTENSIONS
	char *key = NULL;
#endif

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, -1, 2048, NULL, NULL))
				return FAULT_9007;

			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "uri", &uri);
			if (DM_STRLEN(uri)) {
				bbfdm_set_fault_message(ctx,
					"DPP URI is already configured as '%s'. Once defined, the URI value cannot be changed.", uri);
				return FAULT_9001;
			}

#ifdef DECOLLECTOR_VENDOR_EXTENSIONS
			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "key", &key);
			if (DM_STRLEN(key)) {
				bbfdm_set_fault_message(ctx,
					"A Passphrase is already defined for this instance. Only one of 'URI' or 'Passphrase' can be set.");
				return FAULT_9001;
			}
#endif

			break;
		case VALUESET:
			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "Alias", &alias);
			if (DM_STRLEN(alias) == 0) {
				/* If Alias is empty, then initialize it */
				dmasprintf(&alias, "cpe-%s", instance);
				dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "Alias", alias);
			}

			dmubus_call("map.controller", "dpp_enrollee_uri", UBUS_ARGS{{"uri", value, String}, {"alias", alias, String}}, 2, &res);
			if (!res) {
				bbfdm_set_fault_message(ctx,
					"Failed to apply DPP URI: No response received from 'map.controller'. Requested URI: '%s'", value);
				return FAULT_9001;
			} else {
				char *status = dmjson_get_value(res, 1, "status");
				if (DM_STRCMP(status, "ok") != 0) {
					bbfdm_set_fault_message(ctx,
						"Failed to apply DPP URI: The controller rejected the URI value '%s'.", value);
					return FAULT_9001;
				}
			}

			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "uri", value);
			break;
	}
	return 0;
}

#ifdef DECOLLECTOR_VENDOR_EXTENSIONS
static int get_WiFiDataElementsNetworkProvisionedDPP_Passphrase(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "key", value);
	return 0;
}

static int set_WiFiDataElementsNetworkProvisionedDPP_Passphrase(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	json_object *res = NULL;
	char *alias = NULL;
	char *key = NULL;
	char *uri = NULL;

	switch (action)	{
		case VALUECHECK:
			if (bbfdm_validate_string(ctx, value, 8, 63, NULL, NULL))
				return FAULT_9007;

			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "key", &key);
			if (DM_STRLEN(key)) {
				bbfdm_set_fault_message(ctx, "Passphrase is already defined and cannot be changed.");
				return FAULT_9001;
			}

			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "uri", &uri);
			if (DM_STRLEN(uri)) {
				bbfdm_set_fault_message(ctx,
					"A DPP URI is already defined for this instance. Only one of 'Passphrase' or 'URI' can be configured.");
				return FAULT_9001;
			}

			break;
		case VALUESET:
			dmuci_get_value_by_section_string(((struct dm_data *)data)->dmmap_section, "Alias", &alias);
			if (DM_STRLEN(alias) == 0) {
				/* If Alias is empty, then initialize it */
				dmasprintf(&alias, "cpe-%s", instance);
				dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "Alias", alias);
			}

			dmubus_call("map.controller", "zerotouch_set_key", UBUS_ARGS{{"key", value, String}, {"alias", alias, String}}, 2, &res);
			if (!res) {
				bbfdm_set_fault_message(ctx,
					"Failed to apply Passphrase: No response from 'map.controller'. Requested Passphrase value: '%s'", value);
				return FAULT_9001;
			} else {
				char *status = dmjson_get_value(res, 1, "status");
				if (DM_STRCMP(status, "ok") != 0) {
					bbfdm_set_fault_message(ctx,
						"Failed to apply Passphrase: The controller returned error status for value '%s'.", value);
					return FAULT_9001;
				}
			}

			dmuci_set_value_by_section(((struct dm_data *)data)->dmmap_section, "key", value);
			break;
	}
	return 0;
}
#endif

static int get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_NoCandidateAPFailures(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;
	dmasprintf(value, "%"PRIu64, net ? net->steer_summary.no_candidate_cnt : 0);
	return 0;
}

static int get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BlacklistAttempts(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;
	dmasprintf(value, "%"PRIu64, net ? net->steer_summary.blacklist_attempt_cnt : 0);
	return 0;
}

static int get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BlacklistSuccesses(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;
	dmasprintf(value, "%"PRIu64, net ? net->steer_summary.blacklist_success_cnt : 0);
	return 0;
}

static int get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BlacklistFailures(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;
	dmasprintf(value, "%"PRIu64, net ? net->steer_summary.blacklist_failure_cnt : 0);
	return 0;
}

static int get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BTMAttempts(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;
	dmasprintf(value, "%"PRIu64, net ? net->steer_summary.btm_attempt_cnt : 0);
	return 0;
}

static int get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BTMSuccesses(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;
	dmasprintf(value, "%"PRIu64, net ? net->steer_summary.btm_success_cnt : 0);
	return 0;
}

static int get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BTMFailures(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;
	dmasprintf(value, "%"PRIu64, net ? net->steer_summary.btm_failure_cnt : 0);
	return 0;
}

static int get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BTMQueryResponses(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;
	dmasprintf(value, "%"PRIu64, net ? net->steer_summary.btm_query_resp_cnt : 0);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_ID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(dev->macaddr);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_MultiAPCapabilities(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	char ap_caps_encoded[8] = {0};

	to_base64_str(&dev->multiap_caps, sizeof(dev->multiap_caps), ap_caps_encoded, sizeof(ap_caps_encoded));
	*value = dmstrdup(ap_caps_encoded);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_CollectionInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmasprintf(value, "%u", bbfdm_collect_interval * 1000); // in ms
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_ReportUnsuccessfulAssociations(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", dev->report.sta_assocfails);
	return 0;
}

static int set_WiFiDataElementsNetworkDevice_ReportUnsuccessfulAssociations(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "report_sta_assocfails", b ? "1" : "0");
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_APMetricsReportingInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->report.ap_metrics_int);
	return 0;
}

static int set_WiFiDataElementsNetworkDevice_APMetricsReportingInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,"255"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "report_metric_periodic", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_Manufacturer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(dev->manufacturer);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_SerialNumber(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(dev->serial);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_ManufacturerModel(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(dev->model);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_SoftwareVersion(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(dev->swversion);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_ExecutionEnv(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(dev->execenv);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_MaxPrioritizationRules(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->max_prules);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_PrioritizationSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", dev->support_sp);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_MaxVIDs(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%zu", dev->max_vids);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_CountryCode(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(dev->country_code[0] == 0 ? "XX" : dev->country_code);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_LocalSteeringDisallowedSTAList(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_list *uci_list = NULL;
	dmuci_get_value_by_section_list(((struct dm_data *)data)->config_section, "steer_exclude", &uci_list);
	*value = dmuci_list_to_string(uci_list, ",");
	return 0;
}

static int set_WiFiDataElementsNetworkDevice_LocalSteeringDisallowedSTAList(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	size_t length = 0;
	char **arr = NULL;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string_list(ctx, value, -1, -1, -1, -1, 17, NULL, MACAddress))
				return FAULT_9007;
			return 0;
		case VALUESET:
			arr = strsplit(value, ",", &length);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "steer_exclude", "");
			for (int i = 0; i < length; i++)
				dmuci_add_list_value_by_section(((struct dm_data *)data)->config_section, "steer_exclude", arr[i]);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_BTMSteeringDisallowedSTAList(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct uci_list *uci_list = NULL;
	dmuci_get_value_by_section_list(((struct dm_data *)data)->config_section, "steer_exclude_btm", &uci_list);
	*value = dmuci_list_to_string(uci_list, ",");
	return 0;
}

static int set_WiFiDataElementsNetworkDevice_BTMSteeringDisallowedSTAList(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	size_t length = 0;
	char **arr = NULL;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_string_list(ctx, value, -1, -1, -1, -1, 17, NULL, MACAddress))
				return FAULT_9007;
			return 0;
		case VALUESET:
			arr = strsplit(value, ",", &length);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "steer_exclude_btm", "");
			for (int i = 0; i < length; i++)
				dmuci_add_list_value_by_section(((struct dm_data *)data)->config_section, "steer_exclude_btm", arr[i]);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_DFSEnable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", dev->dfs_enabled);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_ReportIndependentScans(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", dev->report.independent_scans);
	return 0;
}

static int set_WiFiDataElementsNetworkDevice_ReportIndependentScans(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "report_scan", b ? "1" : "0");
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_MaxUnsuccessfulAssociationReportingRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->report.sta_assocfails_rate);
	return 0;
}

static int set_WiFiDataElementsNetworkDevice_MaxUnsuccessfulAssociationReportingRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "report_sta_assocfails_rate", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_STASteeringState(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", dev->sta_steering_state);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_CoordinatedCACAllowed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", dev->coordinated_cac_allowed);
	return 0;
}

static int set_WiFiDataElementsNetworkDevice_CoordinatedCACAllowed(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "coordinated_cac", b ? "1" : "0");
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_TrafficSeparationAllowed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_option_value_fallback_def("mapcontroller", "controller", "enable_ts", "0");
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_ControllerOperationMode(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(operation_mode_to_str(dev->multi_ap_device.controller_opmode));
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_BackhaulMACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(dev->multi_ap_device.backhaul.upstream_bbss_macaddr);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_BackhaulALID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(dev->multi_ap_device.backhaul.upstream_device_macaddr);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_BackhaulMediaType(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(i1905_media_type_to_str(dev->multi_ap_device.backhaul.media_type));
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_BackhaulPHYRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->multi_ap_device.backhaul.stats.phyrate);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_TrafficSeparationCapability(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", dev->support_ts);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_EasyConnectCapability(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", dev->support_dpp);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_RadioNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->num_radios);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_Default8021QNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;
	dmasprintf(value, "%u", net ? net->primary_vid ? 1 : 0 : 0);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_SSIDtoVIDMappingNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network *net = bbfdm_wifi_de ? &bbfdm_wifi_de->network : NULL;
	dmasprintf(value, "%zu", list_count(&net->ssidlist));
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_CACStatusNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%zu", list_count(&dev->cac_statuslist));
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_IEEE1905SecurityNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->i1905_seccap.caps_valid ? 1 : 0);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_MaxNumMLDs(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->max_mlds);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_APMLDMaxLinks(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->max_apmld_links);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_bSTAMLDMaxLinks(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->max_bstamld_links);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_TIDLinkMapCapability(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->ttlm_cap);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_APMLDNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->num_apmld);
	return 0;
}

static int get_WiFiDataElementsNetworkDevice_BackhaulDownNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->num_bh_down);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_ID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	char radio_id[18] = {0};

	to_base64_str(radio->macaddr, sizeof(radio->macaddr), radio_id, sizeof(radio_id));
	*value = dmstrdup(radio_id);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_Enabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->enabled);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_Noise(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->anpi);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_Utilization(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->total_utilization);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_Transmit(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->tx_utilization);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_ReceiveSelf(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->rx_utilization);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_ReceiveOther(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->other_utilization);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_TrafficSeparationCombinedFronthaul(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->ts_combined_fronthaul);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_TrafficSeparationCombinedBackhaul(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->ts_combined_backhaul);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_SteeringPolicy(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "steer_policy", "0");
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadio_SteeringPolicy(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,"2"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "steer_policy", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_ChannelUtilizationThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "util_threshold", "0");
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadio_ChannelUtilizationThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,"255"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "util_threshold", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_RCPISteeringThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "rcpi_threshold", "0");
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadio_RCPISteeringThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,"220"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "rcpi_threshold", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_STAReportingRCPIThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "report_rcpi_threshold", "0");
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadio_STAReportingRCPIThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,"220"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "report_rcpi_threshold", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_STAReportingRCPIHysteresisMarginOverride(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "report_rcpi_hysteresis_margin", "0");
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadio_STAReportingRCPIHysteresisMarginOverride(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "report_rcpi_hysteresis_margin", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_ChannelUtilizationReportingThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "report_util_threshold", "0");
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadio_ChannelUtilizationReportingThreshold(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,NULL}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "report_util_threshold", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_AssociatedSTATrafficStatsInclusionPolicy(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "include_sta_stats", "1");
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadio_AssociatedSTATrafficStatsInclusionPolicy(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "include_sta_stats", b ? "1" : "0");
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_AssociatedSTALinkMetricsInclusionPolicy(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmuci_get_value_by_section_fallback_def(((struct dm_data *)data)->config_section, "include_sta_metric", "1");
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceRadio_AssociatedSTALinkMetricsInclusionPolicy(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "include_sta_metric", b ? "1" : "0");
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_ChipsetVendor(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(radio->vendor);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_APMetricsWiFi6(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->report.include_wifi6_metrics);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_MaxBSS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->max_bssnum);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_CurrentOperatingClassProfileNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->cur_opclass.num_opclass);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_UnassociatedSTANumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->num_unassoc_sta);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_BSSNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->num_bss);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_ScanResultNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->num_scanresult);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadio_OpClassPreferenceNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->pref_opclass.num_opclass);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBackhaulSta_MACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(radio->bsta.macaddr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilities_HTCapabilities(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	char caps_b64[18] = {0};

	to_base64_str(&radio->caps.ht, sizeof(radio->caps.ht), caps_b64, sizeof(caps_b64));
	*value = dmstrdup(caps_b64);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilities_VHTCapabilities(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	char caps_b64[18] = {0};

	to_base64_str(radio->caps.vht, sizeof(radio->caps.vht), caps_b64, sizeof(caps_b64));
	*value = dmstrdup(caps_b64);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilities_CapableOperatingClassProfileNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->supp_opclass.num_opclass);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilities_AKMFrontHaulNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->num_fbss_akm);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilities_AKMBackhaulNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->num_bbss_akm);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_HE160(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.he160);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_HE8080(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.he8080);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MCSNSS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	char caps_b64[32] = {0};

	/* fallback to '0000', when valid MCSNSS is absent */
	if (!radio->wifi6caps_ap.mcs_nss_len)
		radio->wifi6caps_ap.mcs_nss_len = 4;

	/* Encode 4, 8 or 12 bytes of MCS NSS */
	to_base64_str(radio->wifi6caps_ap.mcs_nss_12, radio->wifi6caps_ap.mcs_nss_len, caps_b64, sizeof(caps_b64));
	*value = dmstrdup(caps_b64);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_SUBeamformer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.su_beamformer);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_SUBeamformee(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.su_beamformee);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MUBeamformer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.mu_beamformer);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_Beamformee80orLess(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.beamformee_le80);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_BeamformeeAbove80(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.beamformee_gt80);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_ULMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.ul_mumimo);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_ULOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.ul_ofdma);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_DLOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.dl_ofdma);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MaxDLMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->wifi6caps_ap.max_dl_mumimo);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MaxULMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->wifi6caps_ap.max_ul_mumimo);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MaxDLOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->wifi6caps_ap.max_dl_ofdma);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MaxULOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->wifi6caps_ap.max_ul_ofdma);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_RTS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.rts);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MURTS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.mu_rts);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MultiBSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.multi_bssid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MUEDCA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.mu_edca);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_TWTRequestor(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.twt_requester);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_TWTResponder(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.twt_responder);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_SpatialReuse(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.spatial_reuse);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_AnticipatedChannelUsage(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_ap.anticipated_ch_usage);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_HE160(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.he160);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_HE8080(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.he8080);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MCSNSS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	char caps_b64[32] = {0};

	/* fallback to '0000', when valid MCSNSS is absent */
	if (!radio->wifi6caps_bsta.mcs_nss_len)
		radio->wifi6caps_bsta.mcs_nss_len = 4;

	/* Encode 4, 8 or 12 bytes of MCS NSS */
	to_base64_str(radio->wifi6caps_bsta.mcs_nss_12, radio->wifi6caps_bsta.mcs_nss_len, caps_b64, sizeof(caps_b64));
	*value = dmstrdup(caps_b64);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_SUBeamformer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.su_beamformer);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_SUBeamformee(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.su_beamformee);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MUBeamformer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.mu_beamformer);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_Beamformee80orLess(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.beamformee_le80);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_BeamformeeAbove80(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.beamformee_gt80);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_ULMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.ul_mumimo);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_ULOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.ul_ofdma);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_DLOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.dl_ofdma);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MaxDLMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->wifi6caps_bsta.max_dl_mumimo);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MaxULMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->wifi6caps_bsta.max_ul_mumimo);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MaxDLOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->wifi6caps_bsta.max_dl_ofdma);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MaxULOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->wifi6caps_bsta.max_ul_ofdma);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_RTS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.rts);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MURTS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.mu_rts);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MultiBSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.multi_bssid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MUEDCA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.mu_edca);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_TWTRequestor(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.twt_requester);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_TWTResponder(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.twt_responder);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_SpatialReuse(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.spatial_reuse);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_AnticipatedChannelUsage(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi6caps_bsta.anticipated_ch_usage);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_EMLMRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi7caps_ap.emlmr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_EMLSRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi7caps_ap.emlsr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_STRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi7caps_ap.str);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_NSTRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi7caps_ap.nstr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_TIDLinkMapNegotiation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi7caps_ap.tidlinkmap);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_EMLMRFreqSeparationNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->wifi7caps_ap.num_freqsep_emlmr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_EMLSRFreqSeparationNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->wifi7caps_ap.num_freqsep_emlsr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_STRFreqSeparationNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->wifi7caps_ap.num_freqsep_str);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLMRFreqSeparation_RUID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi7_mlo_freqsep *freqsep = (struct wifi7_mlo_freqsep *)((struct dm_data *)data)->additional_data;
	char ruid[32] = {0};

	to_base64_str(freqsep->ruid, 6, ruid, sizeof(ruid));
	*value = dmstrdup(ruid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLMRFreqSeparation_FreqSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi7_mlo_freqsep *freqsep = (struct wifi7_mlo_freqsep *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", freqsep->sep);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLSRFreqSeparation_RUID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi7_mlo_freqsep *freqsep = (struct wifi7_mlo_freqsep *)((struct dm_data *)data)->additional_data;
	char ruid[32] = {0};

	to_base64_str(freqsep->ruid, 6, ruid, sizeof(ruid));
	*value = dmstrdup(ruid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLSRFreqSeparation_FreqSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi7_mlo_freqsep *freqsep = (struct wifi7_mlo_freqsep *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", freqsep->sep);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleSTRFreqSeparation_RUID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi7_mlo_freqsep *freqsep = (struct wifi7_mlo_freqsep *)((struct dm_data *)data)->additional_data;
	char ruid[32] = {0};

	to_base64_str(freqsep->ruid, 6, ruid, sizeof(ruid));
	*value = dmstrdup(ruid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleSTRFreqSeparation_FreqSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi7_mlo_freqsep *freqsep = (struct wifi7_mlo_freqsep *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", freqsep->sep);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_EMLMRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi7caps_bsta.emlmr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_EMLSRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi7caps_bsta.emlsr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_STRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi7caps_bsta.str);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_NSTRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi7caps_bsta.nstr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_TIDLinkMapNegotiation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->wifi7caps_bsta.tidlinkmap);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_EMLMRFreqSeparationNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->wifi7caps_bsta.num_freqsep_emlmr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_EMLSRFreqSeparationNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->wifi7caps_bsta.num_freqsep_emlsr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_STRFreqSeparationNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->wifi7caps_bsta.num_freqsep_str);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLMRFreqSeparation_RUID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi7_mlo_freqsep *freqsep = (struct wifi7_mlo_freqsep *)((struct dm_data *)data)->additional_data;
	char ruid[32] = {0};

	to_base64_str(freqsep->ruid, 6, ruid, sizeof(ruid));
	*value = dmstrdup(ruid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLMRFreqSeparation_FreqSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi7_mlo_freqsep *freqsep = (struct wifi7_mlo_freqsep *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", freqsep->sep);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLSRFreqSeparation_RUID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi7_mlo_freqsep *freqsep = (struct wifi7_mlo_freqsep *)((struct dm_data *)data)->additional_data;
	char ruid[32] = {0};

	to_base64_str(freqsep->ruid, 6, ruid, sizeof(ruid));
	*value = dmstrdup(ruid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLSRFreqSeparation_FreqSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi7_mlo_freqsep *freqsep = (struct wifi7_mlo_freqsep *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", freqsep->sep);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleSTRFreqSeparation_RUID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi7_mlo_freqsep *freqsep = (struct wifi7_mlo_freqsep *)((struct dm_data *)data)->additional_data;
	char ruid[32] = {0};

	to_base64_str(freqsep->ruid, 6, ruid, sizeof(ruid));
	*value = dmstrdup(ruid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleSTRFreqSeparation_FreqSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi7_mlo_freqsep *freqsep = (struct wifi7_mlo_freqsep *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", freqsep->sep);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesAKMFrontHaul_OUI(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_akm_suite *bss_akm = (struct wifi_akm_suite *)((struct dm_data *)data)->additional_data;
	char oui[16] = {0};

	to_base64_str(bss_akm->oui, 3, oui, sizeof(oui));
	*value = dmstrdup(oui);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesAKMFrontHaul_Type(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_akm_suite *bss_akm = (struct wifi_akm_suite *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", bss_akm->type);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesAKMBackhaul_OUI(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_akm_suite *bss_akm = (struct wifi_akm_suite *)((struct dm_data *)data)->additional_data;
	char oui[16] = {0};

	to_base64_str(bss_akm->oui, 3, oui, sizeof(oui));
	*value = dmstrdup(oui);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesAKMBackhaul_Type(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_akm_suite *bss_akm = (struct wifi_akm_suite *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", bss_akm->type);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfile_Class(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", opclass->id);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfile_MaxTxPower(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", opclass->max_txpower);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfile_NonOperable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	char buf[1024] = {0};
	uint32_t pos = 0;
	int i;

	for (i = 0; i < opclass->num_channel; i++) {
		if (opclass->channel[i].preference == WIFI_RADIO_OPCLASS_NON_OPERABLE) {
			pos += snprintf(&buf[pos], sizeof(buf) - pos, "%u,", opclass->channel[i].channel);
		}
	}

	if (pos > 0)
		buf[pos - 1] = 0; // Remove trailing comma

	*value = dmstrdup(buf);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfile_NumberOfNonOperChan(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	uint32_t num_excluded_channels = 0;
	int i;

	for (i = 0; i < opclass->num_channel; i++) {
		if (opclass->channel[i].preference == WIFI_RADIO_OPCLASS_NON_OPERABLE)
			num_excluded_channels++;
	}

	dmasprintf(value, "%u", num_excluded_channels);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_Class(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", opclass->id);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_Channel(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", opclass->channel[0].channel);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_TxPower(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", opclass->max_txpower);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct timespec *entry_time = (struct timespec *)((struct dm_data *)data)->config_section;
	char tsp[32] = {0};

	get_timestamp(&entry_time->tv_sec, tsp, sizeof(tsp));
	*value = dmstrdup(tsp);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioOpClassPreference_OpClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", opclass->id);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioOpClassPreference_ChannelList(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	char buf[1024] = {0};
	uint32_t pos = 0;
	int i;

	for (i = 0; i < opclass->num_channel; i++) {
		pos += snprintf(&buf[pos], sizeof(buf) - pos, "%u,", opclass->channel[i].channel);
	}

	if (pos > 0)
		buf[pos - 1] = 0; // Remove trailing comma

	*value = dmstrdup(buf);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioOpClassPreference_Preference(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", (opclass->channel[0].preference & CHANNEL_PREF_MASK) >> 4);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioOpClassPreference_ReasonCode(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", opclass->channel[0].preference & CHANNEL_PREF_REASON);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_BSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(bss->bssid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_SSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup((char *)bss->ssid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_Enabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", bss->enabled);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_LastChange(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", bss->uptime);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(bss->tsp);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_UnicastBytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, bss->tx_ucast_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_UnicastBytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, bss->rx_ucast_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_MulticastBytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, bss->tx_mcast_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_MulticastBytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, bss->rx_mcast_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_BroadcastBytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, bss->tx_bcast_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_BroadcastBytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, bss->rx_bcast_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_ByteCounterUnits(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", bss->unit);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_EstServiceParametersBE(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	char est_be[16] = {0};

	to_base64_str(bss->est_wmm_be, 3, est_be, sizeof(est_be));
	*value = dmstrdup(est_be);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_EstServiceParametersBK(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	char est_bk[16] = {0};

	to_base64_str(bss->est_wmm_bk, 3, est_bk, sizeof(est_bk));
	*value = dmstrdup(est_bk);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_EstServiceParametersVI(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	char est_vi[16] = {0};

	to_base64_str(bss->est_wmm_vi, 3, est_vi, sizeof(est_vi));
	*value = dmstrdup(est_vi);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_EstServiceParametersVO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	char est_vo[16] = {0};

	to_base64_str(bss->est_wmm_vo, 3, est_vo, sizeof(est_vo));
	*value = dmstrdup(est_vo);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_BackhaulUse(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", bss->is_bbss);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_FronthaulUse(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", bss->is_fbss);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_R1disallowed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", bss->r1_disallowed);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_R2disallowed(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", bss->r2_disallowed);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_MultiBSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", bss->multi_bssid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_TransmittedBSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", bss->transmitted_bssid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSS_STANumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bss_element *bss = (struct wifi_bss_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", bss->num_stations);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTA_SteeringHistoryNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	int cnt = get_number_of_entries(ctx, data, instance, browseWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistoryInst);
	dmasprintf(value, "%d", cnt);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_MACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(sta->macaddr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(sta->tsp);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_HTCapabilities(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	char caps_b64[32] = {0};

	to_base64_str(&sta->caps.ht, sizeof(sta->caps.ht), caps_b64, sizeof(caps_b64));
	*value = dmstrdup(caps_b64);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_VHTCapabilities(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	char caps_b64[32] = {0};

	to_base64_str(sta->caps.vht, sizeof(sta->caps.vht), caps_b64, sizeof(caps_b64));
	*value = dmstrdup(caps_b64);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_ClientCapabilities(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	char *client_caps_b64 = NULL;

	if (sta->reassoc_framelen) {
		/* Calculate min. size of output buffer as required by base64_encode */
		uint32_t client_cap_b64_len = ((sta->reassoc_framelen * 4) / 3) + 4;
		client_cap_b64_len += (client_cap_b64_len / 72) + 1;

		client_caps_b64 = dmcalloc(client_cap_b64_len, sizeof(char));
		if (client_caps_b64) {
			to_base64_str(sta->reassoc_frame, sta->reassoc_framelen, client_caps_b64, client_cap_b64_len);
		}
	}

	*value = dmstrdup(client_caps_b64 ? client_caps_b64 : "");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_LastDataDownlinkRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->dl_rate);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_LastDataUplinkRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->ul_rate);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_UtilizationReceive(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->dl_utilization);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_UtilizationTransmit(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->ul_utilization);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_EstMACDataRateDownlink(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->dl_est_thput);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_EstMACDataRateUplink(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->ul_est_thput);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_SignalStrength(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->rcpi);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_LastConnectTime(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->conn_time);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_BytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->tx_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_BytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->rx_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_PacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->tx_pkts);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_PacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->rx_pkts);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_ErrorsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->tx_errors);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_ErrorsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->rx_errors);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_RetransCount(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->rtx_pkts);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_MeasurementReport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	struct wifi_sta_meas_report *sta_meas_report = NULL;
	char buf[1024] = {0};
	uint32_t pos = 0;

	list_for_each_entry(sta_meas_report, &sta->meas_reportlist, list) {
		/* Calculate min. size of output buffer as required by base64_encode */
		uint32_t b64_len = ((sta_meas_report->element_len * 4) / 3) + 4;
		char *b64_buf = NULL;
		const char *b64_ret = NULL;

		b64_len += (b64_len / 72) + 1;

		b64_buf = dmmalloc(b64_len);

		if (!b64_buf)
			continue;

		b64_ret = to_base64_str(sta_meas_report->element_data, sta_meas_report->element_len, b64_buf, b64_len);

		pos += snprintf(&buf[pos], sizeof(buf) - pos, "%s,", b64_ret ? b64_ret : "");
	}

	if (pos > 0)
		buf[pos - 1] = 0; // Remove trailing comma

	*value = dmstrdup(buf);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_NumberOfMeasureReports(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->num_meas_reports);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_IPV4Address(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	char ip_addr[INET6_ADDRSTRLEN] = {0};

	*value = dmstrdup(inet_ntop(AF_INET, &sta->ipv4_addr.addr.ip4, ip_addr, INET_ADDRSTRLEN));
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_IPV6Address(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	char ip_addr[INET6_ADDRSTRLEN] = {0};

	*value = dmstrdup(inet_ntop(AF_INET6, &sta->ipv6_addr.addr.ip6, ip_addr, INET6_ADDRSTRLEN));
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_Hostname(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(sta->hostname);
	return 0;
}

#ifdef DECOLLECTOR_VENDOR_EXTENSIONS
static int get_WiFiDataElementsNetworkDeviceRadioBSSSTA_QoERating(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%f", sta->rating);
	return 0;
}
#endif

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_NoCandidateAPFailures(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->mapsta.stats.no_candidate_cnt);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BlacklistAttempts(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->mapsta.stats.blacklist_attempt_cnt);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BlacklistSuccesses(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->mapsta.stats.blacklist_success_cnt);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BlacklistFailures(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->mapsta.stats.blacklist_failure_cnt);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BTMAttempts(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->mapsta.stats.btm_attempt_cnt);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BTMSuccesses(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->mapsta.stats.btm_success_cnt);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BTMFailures(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->mapsta.stats.btm_failure_cnt);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BTMQueryResponses(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->mapsta.stats.btm_query_resp_cnt);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_LastSteerTime(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->mapsta.stats.last_tsp.tv_sec);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_Time(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "time");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_APOrigin(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "ap");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_TriggerEvent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *trigger = NULL;

	trigger = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "trigger");

	if (DM_STRCMP(trigger, "link_quality") == 0) {
		dmasprintf(value, "%s", "Wi-Fi Link Quality");
	} else if (DM_STRCMP(trigger, "channel_util") == 0) {
		dmasprintf(value, "%s", "Wi-Fi Channel Utilization");
	} else if (DM_STRCMP(trigger, "bk_link_util") == 0) {
		dmasprintf(value, "%s", "Backhaul Link Utilization");
	} else {
		dmasprintf(value, "%s", "Unknown");
	}

	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_SteeringApproach(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	char *method = NULL;

	method = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "method");

	if (DM_STRCMP(method, "btm") == 0) {
		dmasprintf(value, "%s", "BTM Request");
	} else if (DM_STRCMP(method, "assoc_ctl") == 0) {
		dmasprintf(value, "%s", "Blacklist");
	} else if (DM_STRCMP(method, "async_btm") == 0) {
		dmasprintf(value, "%s", "Async BTM Query");
	} else {
		dmasprintf(value, "%s", method);
	}

	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_APDestination(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "target_ap");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_SteeringDuration(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmjson_get_value(((struct dm_data *)data)->json_object, 1, "duration");
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_HE160(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", sta->wifi6caps.he160);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_HE8080(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", sta->wifi6caps.he8080);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MCSNSS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	char caps_b64[32] = {0};

	/* fallback to '0000', when valid MCSNSS is absent */
	if (!sta->wifi6caps.mcs_nss_len)
		sta->wifi6caps.mcs_nss_len = 4;

	/* Encode 4, 8 or 12 bytes of MCS NSS */
	to_base64_str(sta->wifi6caps.mcs_nss_12, sta->wifi6caps.mcs_nss_len, caps_b64, sizeof(caps_b64));
	*value = dmstrdup(caps_b64);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_SUBeamformer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", sta->wifi6caps.su_beamformer);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_SUBeamformee(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", sta->wifi6caps.su_beamformee);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MUBeamformer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", sta->wifi6caps.mu_beamformer);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_Beamformee80orLess(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", sta->wifi6caps.beamformee_le80);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_BeamformeeAbove80(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", sta->wifi6caps.beamformee_gt80);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_ULMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", sta->wifi6caps.ul_mumimo);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_ULOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", sta->wifi6caps.ul_ofdma);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MaxDLMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->wifi6caps.max_dl_mumimo);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MaxULMUMIMO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->wifi6caps.max_ul_mumimo);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MaxDLOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->wifi6caps.max_dl_ofdma);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MaxULOFDMA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->wifi6caps.max_ul_ofdma);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_RTS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", sta->wifi6caps.rts);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MURTS(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", sta->wifi6caps.mu_rts);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MultiBSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", sta->wifi6caps.multi_bssid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MUEDCA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", sta->wifi6caps.mu_edca);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_TWTRequestor(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", sta->wifi6caps.twt_requester);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_TWTResponder(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", sta->wifi6caps.twt_responder);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_SpatialReuse(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", sta->wifi6caps.spatial_reuse);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_AnticipatedChannelUsage(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_sta_element *sta = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", sta->wifi6caps.anticipated_ch_usage);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResult_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_element *sres = (struct wifi_scanres_element *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(sres->tsp);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResult_OpClassScanNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_element *sres = (struct wifi_scanres_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sres->num_opclass_scanned);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScan_OperatingClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_opclass_element *op = (struct wifi_scanres_opclass_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", op->opclass);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScan_ChannelScanNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_opclass_element *op = (struct wifi_scanres_opclass_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", op->num_channels_scanned);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_Channel(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_channel_element *ch = (struct wifi_scanres_channel_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", ch->channel);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_channel_element *ch = (struct wifi_scanres_channel_element *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(ch->tsp);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_Utilization(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_channel_element *ch = (struct wifi_scanres_channel_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", ch->utilization);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_Noise(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_channel_element *ch = (struct wifi_scanres_channel_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", ch->anpi);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_ScanStatus(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_channel_element *ch = (struct wifi_scanres_channel_element *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(scan_status_to_str(ch->scan_status));
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_NeighborBSSNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_channel_element *ch = (struct wifi_scanres_channel_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", ch->num_neighbors);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_BSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_neighbor_element *nbr = (struct wifi_scanres_neighbor_element *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(nbr->bssid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_SSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_neighbor_element *nbr = (struct wifi_scanres_neighbor_element *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(nbr->ssid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_SignalStrength(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_neighbor_element *nbr = (struct wifi_scanres_neighbor_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", nbr->rssi);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_ChannelBandwidth(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_neighbor_element *nbr = (struct wifi_scanres_neighbor_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", nbr->bw);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_ChannelUtilization(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_neighbor_element *nbr = (struct wifi_scanres_neighbor_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", nbr->utilization);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_StationCount(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_neighbor_element *nbr = (struct wifi_scanres_neighbor_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", nbr->num_stations);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_MLDMACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_neighbor_element *nbr = (struct wifi_scanres_neighbor_element *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(nbr->mld_macaddr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_ReportingBSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_neighbor_element *nbr = (struct wifi_scanres_neighbor_element *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(nbr->rpt_bssid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_MultiBSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_neighbor_element *nbr = (struct wifi_scanres_neighbor_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", (nbr->multibss_type == 0) ? 0 : 1);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_BSSLoadElementPresent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_neighbor_element *nbr = (struct wifi_scanres_neighbor_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", nbr->bssload_present);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_BSSColor(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_scanres_neighbor_element *nbr = (struct wifi_scanres_neighbor_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", nbr->bss_color);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanCapability_OnBootOnly(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", radio->scan_caps.boot_only);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanCapability_Impact(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->scan_caps.impact);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanCapability_MinimumInterval(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->scan_caps.interval);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanCapability_OpClassChannelsNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", radio->scan_caps.opclass.num_opclass);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanCapabilityOpClassChannels_OpClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", opclass->id);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioScanCapabilityOpClassChannels_ChannelList(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	char buf[1024] = {0};
	uint32_t pos = 0;
	int i;

	for (i = 0; i < opclass->num_channel; i++) {
		pos += snprintf(&buf[pos], sizeof(buf) - pos, "%u,", opclass->channel[i].channel);
	}

	if (pos > 0)
		buf[pos - 1] = 0; // Remove trailing comma

	*value = dmstrdup(buf);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCACCapability_CACMethodNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_element *radio = (struct wifi_radio_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%zu", list_count(&radio->cac_capslist));
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethod_Method(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_cac_capabilities *cac_caps = (struct wifi_radio_cac_capabilities *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", cac_caps->method);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethod_NumberOfSeconds(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_cac_capabilities *cac_caps = (struct wifi_radio_cac_capabilities *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", cac_caps->num_seconds);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethod_OpClassChannelsNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_cac_capabilities *cac_caps = (struct wifi_radio_cac_capabilities *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", cac_caps->opclasses.num_opclass);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodOpClassChannels_OpClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", opclass->id);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodOpClassChannels_ChannelList(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	char buf[1024] = {0};
	uint32_t pos = 0;
	int i;

	for (i = 0; i < opclass->num_channel; i++) {
		pos += snprintf(&buf[pos], sizeof(buf) - pos, "%u,", opclass->channel[i].channel);
	}

	if (pos > 0)
		buf[pos - 1] = 0; // Remove trailing comma

	*value = dmstrdup(buf);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioUnassociatedSTA_MACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_unassoc_sta_element *usta = (struct wifi_unassoc_sta_element *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(usta->macaddr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceRadioUnassociatedSTA_SignalStrength(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_unassoc_sta_element *usta = (struct wifi_unassoc_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", usta->rcpi);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDevice_LastContactTime(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(dev->multi_ap_device.last_contacttime);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDevice_AssocIEEE1905DeviceRef(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	char *dev_id = dm_to_macaddr_str(dev->macaddr);

	_bbfdm_get_references(ctx, "Device.IEEE1905.AL.NetworkTopology.IEEE1905Device.", "IEEE1905Id", dev_id, value);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDevice_EasyMeshAgentOperationMode(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(operation_mode_to_str(dev->multi_ap_device.agent_opmode));
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfile_Class(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", opclass->id);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfile_Channel(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", opclass->channel[0].channel);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfile_TxPower(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_radio_opclass_entry *opclass = (struct wifi_radio_opclass_entry *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", opclass->max_txpower);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfile_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct timespec *entry_time = (struct timespec *)((struct dm_data *)data)->config_section;
	char tsp[32] = {0};

	get_timestamp(&entry_time->tv_sec, tsp, sizeof(tsp));
	*value = dmstrdup(tsp);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_BytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, dev->multi_ap_device.backhaul.stats.tx_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_BytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, dev->multi_ap_device.backhaul.stats.rx_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_PacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->multi_ap_device.backhaul.stats.tx_pkts);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_PacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->multi_ap_device.backhaul.stats.rx_pkts);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_ErrorsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->multi_ap_device.backhaul.stats.tx_errors);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_ErrorsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->multi_ap_device.backhaul.stats.rx_errors);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_SignalStrength(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->multi_ap_device.backhaul.stats.rcpi);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_LastDataDownlinkRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->multi_ap_device.backhaul.stats.dl_rate);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_LastDataUplinkRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->multi_ap_device.backhaul.stats.ul_rate);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(dev->multi_ap_device.backhaul.stats.tsp);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceDefault8021Q_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = dmstrdup("1");
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceDefault8021Q_Enable(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	bool b;

	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_boolean(ctx, value))
				return FAULT_9007;
			return 0;
		case VALUESET:
			string_to_bool(value, &b);
			//TODO
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceDefault8021Q_PrimaryVID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_option_value_string("mapcontroller", "controller", "primary_vid", value);
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceDefault8021Q_PrimaryVID(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,"4095"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value("mapcontroller", "controller", "primary_vid", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceDefault8021Q_DefaultPCP(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	dmuci_get_option_value_string("mapcontroller", "controller", "primary_pcp", value);
	return 0;
}

static int set_WiFiDataElementsNetworkDeviceDefault8021Q_DefaultPCP(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	switch (action) {
		case VALUECHECK:
			if (bbfdm_validate_unsignedInt(ctx, value, RANGE_ARGS{{NULL,"7"}}, 1))
				return FAULT_9007;
			return 0;
		case VALUESET:
			dmuci_set_value("mapcontroller", "controller", "primary_pcp", value);
			return 0;
	}
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceSSIDtoVIDMapping_SSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_ssid *ssid = (struct wifi_network_ssid *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(ssid->ssid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceSSIDtoVIDMapping_VID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_ssid *ssid = (struct wifi_network_ssid *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", ssid->vid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatus_TimeStamp(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_cac_status *cac_status_entry = (struct wifi_cac_status *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(cac_status_entry->tsp);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatus_CACAvailableChannelNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_cac_status *cac_status_entry = (struct wifi_cac_status *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%zu", list_count(&cac_status_entry->available_chlist));
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatus_CACNonOccupancyChannelNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_cac_status *cac_status_entry = (struct wifi_cac_status *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%zu", list_count(&cac_status_entry->nop_chlist));
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatus_CACActiveChannelNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_cac_status *cac_status_entry = (struct wifi_cac_status *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%zu", list_count(&cac_status_entry->cac_chlist));
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACAvailableChannel_OpClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_cac_available_channel *cac_avail_entry = (struct wifi_cac_available_channel *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", cac_avail_entry->opclass);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACAvailableChannel_Channel(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_cac_available_channel *cac_avail_entry = (struct wifi_cac_available_channel *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", cac_avail_entry->channel);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACAvailableChannel_Minutes(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_cac_available_channel *cac_avail_entry = (struct wifi_cac_available_channel *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", cac_avail_entry->cleared);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannel_OpClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_cac_nop_channel *cac_nop_entry = (struct wifi_cac_nop_channel *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", cac_nop_entry->opclass);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannel_Channel(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_cac_nop_channel *cac_nop_entry = (struct wifi_cac_nop_channel *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", cac_nop_entry->channel);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannel_Seconds(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_cac_nop_channel *cac_nop_entry = (struct wifi_cac_nop_channel *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", cac_nop_entry->remaining);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACActiveChannel_OpClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_cac_active_channel *cac_active_entry = (struct wifi_cac_active_channel *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", cac_active_entry->opclass);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACActiveChannel_Channel(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_cac_active_channel *cac_active_entry = (struct wifi_cac_active_channel *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", cac_active_entry->channel);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceCACStatusCACActiveChannel_Countdown(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_cac_active_channel *cac_active_entry = (struct wifi_cac_active_channel *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", cac_active_entry->remaining);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceIEEE1905Security_OnboardingProtocol(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->i1905_seccap.onboarding_protocol);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceIEEE1905Security_IntegrityAlgorithm(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->i1905_seccap.integrity);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceIEEE1905Security_EncryptionAlgorithm(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", dev->i1905_seccap.encryption);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceBackhaulDown_BackhaulDownALID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bh_down *bh_down = (struct wifi_bh_down *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(bh_down->al_macaddr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceBackhaulDown_BackhaulDownMACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_bh_down *bh_down = (struct wifi_bh_down *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(bh_down->bss_macaddr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLD_MLDMACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_apmld_element *apmld = (struct wifi_apmld_element *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(apmld->mld_macaddr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLD_AffiliatedAPNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_apmld_element *apmld = (struct wifi_apmld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", apmld->num_ap);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLD_STAMLDNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_apmld_element *apmld = (struct wifi_apmld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", apmld->num_sta);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLD_LinkToOpClassMapNumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_apmld_element *apmld = (struct wifi_apmld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", apmld->num_linkopcmap);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_EMLMREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_apmld_element *apmld = (struct wifi_apmld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", apmld->emlmr_enabled);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_EMLSREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_apmld_element *apmld = (struct wifi_apmld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", apmld->emlsr_enabled);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_STREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_apmld_element *apmld = (struct wifi_apmld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", apmld->str_enabled);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_NSTREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_apmld_element *apmld = (struct wifi_apmld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", apmld->nstr_enabled);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_TIDLinkMapNegotiation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_apmld_element *apmld = (struct wifi_apmld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", apmld->ttlm_enabled);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDLinkToOpClassMap_LinkID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_linkopcmap_element *linkopcmap = (struct wifi_linkopcmap_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", linkopcmap->linkid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDLinkToOpClassMap_OpClass(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_linkopcmap_element *linkopcmap = (struct wifi_linkopcmap_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", linkopcmap->opclass);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_BSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_ap_element *ap = (struct wifi_affiliated_ap_element *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(ap->bssid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_LinkID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_ap_element *ap = (struct wifi_affiliated_ap_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", ap->linkid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_RUID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_ap_element *ap = (struct wifi_affiliated_ap_element *)((struct dm_data *)data)->additional_data;
	char ruid[32] = {0};

	to_base64_str(ap->ruid, 6, ruid, sizeof(ruid));
	*value = dmstrdup(ruid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_PacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_ap_element *ap = (struct wifi_affiliated_ap_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, ap->tx_packets);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_PacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_ap_element *ap = (struct wifi_affiliated_ap_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, ap->rx_packets);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_UnicastBytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_ap_element *ap = (struct wifi_affiliated_ap_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, ap->tx_ucast_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_UnicastBytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_ap_element *ap = (struct wifi_affiliated_ap_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, ap->rx_ucast_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_ErrorsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_ap_element *ap = (struct wifi_affiliated_ap_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, ap->tx_errors);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_MulticastBytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_ap_element *ap = (struct wifi_affiliated_ap_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, ap->tx_mcast_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_MulticastBytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_ap_element *ap = (struct wifi_affiliated_ap_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, ap->rx_mcast_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_BroadcastBytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_ap_element *ap = (struct wifi_affiliated_ap_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, ap->tx_bcast_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_BroadcastBytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_ap_element *ap = (struct wifi_affiliated_ap_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, ap->rx_bcast_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_EstServiceParametersBE(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_ap_element *ap = (struct wifi_affiliated_ap_element *)((struct dm_data *)data)->additional_data;
	char est_be[16] = {0};

	to_base64_str(ap->est_wmm_be, 3, est_be, sizeof(est_be));
	*value = dmstrdup(est_be);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_EstServiceParametersBK(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_ap_element *ap = (struct wifi_affiliated_ap_element *)((struct dm_data *)data)->additional_data;
	char est_bk[16] = {0};

	to_base64_str(ap->est_wmm_bk, 3, est_bk, sizeof(est_bk));
	*value = dmstrdup(est_bk);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_EstServiceParametersVI(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_ap_element *ap = (struct wifi_affiliated_ap_element *)((struct dm_data *)data)->additional_data;
	char est_vi[16] = {0};

	to_base64_str(ap->est_wmm_vi, 3, est_vi, sizeof(est_vi));
	*value = dmstrdup(est_vi);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_EstServiceParametersVO(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_ap_element *ap = (struct wifi_affiliated_ap_element *)((struct dm_data *)data)->additional_data;
	char est_vo[16] = {0};

	to_base64_str(ap->est_wmm_vo, 3, est_vo, sizeof(est_vo));
	*value = dmstrdup(est_vo);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_MLDMACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(stamld->mld_macaddr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_Hostname(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	*value = dmstrdup(stamld->hostname);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_IPv4Address(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	char ip_addr[INET6_ADDRSTRLEN] = {0};

	*value = dmstrdup(inet_ntop(AF_INET, &stamld->ipv4_addr.addr.ip4, ip_addr, INET_ADDRSTRLEN));
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_IPv6Address(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	char ip_addr[INET6_ADDRSTRLEN] = {0};

	*value = dmstrdup(inet_ntop(AF_INET6, &stamld->ipv6_addr.addr.ip6, ip_addr, INET6_ADDRSTRLEN));
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_IsbSTA(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", stamld->is_bsta);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_LastConnectTime(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", stamld->conn_time);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_BytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, stamld->tx_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_BytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, stamld->rx_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_PacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", stamld->tx_packets);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_PacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", stamld->rx_packets);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_ErrorsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", stamld->tx_errors);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_ErrorsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", stamld->rx_errors);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_RetransCount(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", stamld->rtx_packets);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_AffiliatedSTANumberOfEntries(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", stamld->num_sta);
	return 0;
}

#ifdef DECOLLECTOR_VENDOR_EXTENSIONS
static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_QoERating(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%f", stamld->rating);
	return 0;
}
#endif

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_EMLMRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", stamld->emlmr_supported);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_EMLSRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", stamld->emlsr_supported);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_STRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", stamld->str_supported);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_NSTRSupport(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", stamld->nstr_supported);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_TIDLinkMapNegotiation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", stamld->ttlm_supported);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_EMLMREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", stamld->emlmr_enabled);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_EMLSREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", stamld->emlsr_enabled);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_STREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", stamld->str_enabled);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_NSTREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", stamld->nstr_enabled);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_TIDLinkMapNegotiation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_stamld_element *stamld = (struct wifi_stamld_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", stamld->ttlm_enabled);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_MACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_sta_element *sta = (struct wifi_affiliated_sta_element *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(sta->macaddr);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_BSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_sta_element *sta = (struct wifi_affiliated_sta_element *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(sta->bssid);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_BytesSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_sta_element *sta = (struct wifi_affiliated_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->tx_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_BytesReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_sta_element *sta = (struct wifi_affiliated_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->rx_bytes);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_PacketsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_sta_element *sta = (struct wifi_affiliated_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->tx_packets);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_PacketsReceived(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_sta_element *sta = (struct wifi_affiliated_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->rx_packets);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_ErrorsSent(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_sta_element *sta = (struct wifi_affiliated_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->tx_errors);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_SignalStrength(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_sta_element *sta = (struct wifi_affiliated_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->rcpi);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_EstMACDataRateDownlink(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_sta_element *sta = (struct wifi_affiliated_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->dl_est_thput);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_EstMACDataRateUplink(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_sta_element *sta = (struct wifi_affiliated_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->ul_est_thput);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_LastDataDownlinkRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_sta_element *sta = (struct wifi_affiliated_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->dl_rate);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_LastDataUplinkRate(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_sta_element *sta = (struct wifi_affiliated_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%u", sta->ul_rate);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_UtilizationReceive(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_sta_element *sta = (struct wifi_affiliated_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->dl_utilization);
	return 0;
}

static int get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_UtilizationTransmit(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_affiliated_sta_element *sta = (struct wifi_affiliated_sta_element *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%"PRIu64, sta->ul_utilization);
	return 0;
}

static int get_WiFiDataElementsNetworkDevicebSTAMLD_MLDMACAddress(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(dev->bstamld.mld_macaddr);
	return 0;
}

static int get_WiFiDataElementsNetworkDevicebSTAMLD_BSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	*value = dm_to_macaddr_str(dev->bstamld.bssid);
	return 0;
}

static int get_WiFiDataElementsNetworkDevicebSTAMLD_AffiliatedbSTAList(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	char buf[1024] = {0};
	uint32_t pos = 0;

	if (dev->bstamld.num_aff_bsta) {
		struct wifi_affiliated_sta_element *sta = NULL;

		list_for_each_entry(sta, &dev->bstamld.affbstalist, list) {
			char macaddr[18] = {0};

			hwaddr_ntoa(sta->macaddr, macaddr);
			pos += snprintf(&buf[pos], sizeof(buf) - pos, "%s,", macaddr);
		}
	}

	if (pos > 0)
		buf[pos - 1] = 0; // Remove trailing comma

	*value = dmstrdup(buf);
	return 0;
}

static int get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_EMLMREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", dev->bstamld.emlmr_enabled);
	return 0;
}

static int get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_EMLSREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", dev->bstamld.emlsr_enabled);
	return 0;
}

static int get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_STREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", dev->bstamld.str_enabled);
	return 0;
}

static int get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_NSTREnabled(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", dev->bstamld.nstr_enabled);
	return 0;
}

static int get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_TIDLinkMapNegotiation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	dmasprintf(value, "%d", dev->bstamld.ttlm_enabled);
	return 0;
}

/*************************************************************
 * OPERATE COMMANDS
 *************************************************************/
static operation_args WiFiDataElementsNetwork_SetTrafficSeparation_args = {
	.in = (const char *[]) {
		"Enable",
		"SSIDtoVIDMapping.{i}.SSID",
		"SSIDtoVIDMapping.{i}.VID",
		NULL
	},
	.out = (const char *[]) {
		"Status",
		NULL
	}
};

static int get_operate_args_WiFiDataElementsNetwork_SetTrafficSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char *)&WiFiDataElementsNetwork_SetTrafficSeparation_args;
	return 0;
}

static int operate_WiFiDataElementsNetwork_SetTrafficSeparation(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct wifi_operate_args {
		char *arg1;
		char *arg2;
	} operate_args[16] = {0};
	char status[32];
	bool b = false;

	snprintf(status, sizeof(status), "%s", "Success");

	char *enable = dmjson_get_value((json_object *)value, 1, "Enable");
	if (!enable || *enable == '\0' || bbfdm_validate_boolean(ctx, enable)) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	for (int i = 0; i < ARRAY_SIZE(operate_args); i++) {

		char ssid[32] = {0};
		char vid[32] = {0};

		snprintf(ssid, sizeof(ssid), "SSIDtoVIDMapping.%d.SSID", i+1);
		snprintf(vid, sizeof(vid), "SSIDtoVIDMapping.%d.VID", i+1);

		operate_args[i].arg1 = dmjson_get_value((json_object *)value, 1, ssid);
		operate_args[i].arg2 = dmjson_get_value((json_object *)value, 1, vid);

		// Instance number must be assigned sequentially without gaps, if one is empty then break the loop
		if (*(operate_args[i].arg1) == '\0')
			break;

		struct uci_section *s = NULL;

		uci_foreach_option_eq("mapcontroller", "ap", "ssid", operate_args[i].arg1, s) {

			// If VID is not empty then update it
			if (*(operate_args[i].arg2) != '\0')
				dmuci_set_value_by_section(s, "vid", operate_args[i].arg2);
		}
	}

	string_to_bool(enable, &b);
	dmuci_set_value("mapcontroller", "controller", "enable_ts", b ? "1" : "0");

end:
	fill_blob_param(&ctx->bb, "Status", status, DMT_TYPE[DMT_STRING], 0);
	return 0;
}

static operation_args WiFiDataElementsNetwork_SetSSID_args = {
	.in = (const char *[]) {
		"SSID",
		CUSTOM_PREFIX"NewSSID",
		"Enable",
		"AddRemoveChange",
		"PassPhrase",
		"Band",
		"AKMsAllowed",
		"HaulType",
		"Type",
		CUSTOM_PREFIX"EnableMLD",
		CUSTOM_PREFIX"TrafficSeparation",
		NULL
	},
	.out = (const char *[]) {
		"Status",
		NULL
	}
};

static int get_operate_args_WiFiDataElementsNetwork_SetSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char *)&WiFiDataElementsNetwork_SetSSID_args;
	return 0;
}

static bool duplicate_band_exist(char **band_arr, int band_arr_length)
{
	if (band_arr == NULL)
		return false;

	for (int i = 0; i < (band_arr_length - 1); i++) {
		for (int j = i + 1; j < band_arr_length; j++) {
			if (DM_STRCMP(band_arr[i], band_arr[j]) == 0)
				return true;
		}
	}

	return false;
}

static bool validate_requested_ssid(const char *ssid, unsigned *idx)
{
	struct uci_section *s = NULL;
	char *curr_ssid = NULL;
	unsigned count = 1;

	if (ssid == NULL || idx == NULL)
		return false;

	/* Check if ssid is already added */
	uci_path_foreach_sections(bbfdm, "WiFi", "SSID", s) {
		dmuci_get_value_by_section_string(s, "ssid", &curr_ssid);

		if (DM_STRCMP(curr_ssid, ssid) == 0) {
			return false;
		}

		count = count + 1;
	}

	*idx = count;
	return true;
}

static int find_free_vlan_id(void)
{
	struct uci_section *ap_sec = NULL;
	char vid_str[8] = {0};
	char *primary_vid;
	int vid;

	primary_vid = dmuci_get_option_value_fallback_def("mapcontroller", "controller", "primary_vid", "1");

	for (vid = 4094; vid >= 2; vid--) {
		snprintf(vid_str, sizeof(vid_str), "%d", vid);
		ap_sec = get_dup_section_in_config_opt("mapcontroller", "ap", "vid", vid_str);
		if (ap_sec == NULL && DM_STRCMP(vid_str, primary_vid) != 0)
			return vid;
	}

	return -1; // no free VLAN found
}


static char *process_set_ssid_add_req(struct dmctx *ctx, set_ssid_param *op_param)
{
	struct uci_section *s = NULL;
	char curr_band[90] = {0};
	char passwd[256] = {0};
	char **band_arr = NULL;
	size_t band_arr_length = 0;
	char **enc_arr = NULL;
	size_t enc_arr_length = 0;
	char *haul_type = NULL;
	char vid_str[8] = {0};
	char mld_id[3] = {0};
	unsigned idx = 1;
	bool enable = true;
	bool mld_enable = false;
	bool ts_enable = false;

	if (op_param == NULL) {
		bbfdm_set_fault_message(ctx, "Internal Error");
		return "Error_Other";
	}

	if (DM_STRLEN(op_param->enab) != 0) {
		string_to_bool(op_param->enab, &enable);
	}

	/* If not mld capabled device ignore mld enable config */
	if (mld_capable && DM_STRLEN(op_param->mld_enab) != 0) {
		string_to_bool(op_param->mld_enab, &mld_enable);
	}

	if (DM_STRLEN(op_param->ts) != 0) {
		string_to_bool(op_param->ts, &ts_enable);
	}

	if (DM_STRLEN(op_param->band) == 0 || DM_LSTRCMP(op_param->band, "All") == 0) {
		struct radio_info *node = NULL;
		int len = 0;

		list_for_each_entry(node, &dev_list, list) {
			if (node->band[0] == '2') {
				len = DM_STRLEN(curr_band);
				snprintf(curr_band+len, sizeof(curr_band)-len, "%s,", "2.4");
			} else {
				len = DM_STRLEN(curr_band);
				snprintf(curr_band+len, sizeof(curr_band)-len, "%c,", node->band[0]);
			}
		}

		len = DM_STRLEN(curr_band);
		if (len) {
			curr_band[len-1] = '\0';
		}
	} else {
		snprintf(curr_band, sizeof(curr_band), "%s", op_param->band);
	}

	band_arr = strsplit(curr_band, ",", &band_arr_length);

	/* Band should not have duplicate value to avoid multiple instance with
	 * same SSID and Band */
	if (duplicate_band_exist(band_arr, band_arr_length)) {
		bbfdm_set_fault_message(ctx, "Band should not contain duplicate value");
		return "Error_Invalid_Input";
	}

	if (DM_STRLEN(op_param->akm) != 0) {
		enc_arr = strsplit(op_param->akm, ",", &enc_arr_length);
	}

	if (DM_STRLEN(op_param->key) == 0) {
		char *def_key = get_default_key();
		if (DM_STRLEN(def_key) == 0)
			return "Error_Other";

		snprintf(passwd, sizeof(passwd), "%s", def_key);
	} else {
		snprintf(passwd, sizeof(passwd), "%s", op_param->key);
	}

	if (enc_arr_length > band_arr_length) {
		return "Error_Invalid_Input";
	}

	/* Reject if requested SSID can not be configured */
	if (!validate_requested_ssid(op_param->ssid, &idx)) {
		bbfdm_set_fault_message(ctx, "SSID (%s) already exist", op_param->ssid);
		return "Error_Invalid_Input";
	}

	haul_type = get_haultype_uci_value(op_param->haul);

	if (ts_enable && DM_STRCMP(haul_type, "fronthaul") == 0) {
		int vid;

		vid = find_free_vlan_id();
		if (vid < 0) {
			bbfdm_set_fault_message(ctx, "No free VLAN ID in range 2–4094");
			return "Error_Other";
		}

		snprintf(vid_str, sizeof(vid_str), "%d", vid);
	}

	/* If MLD config enabled then find an unused mld id */
	if (mld_enable == true) {
		/* First remove unused MLD section if any */
		remove_unused_mld_sections();

		for (int id = 1; id <= 24; id++) {
			char mld_str[3] = {0};
			struct uci_section *mld_sec = NULL;

			snprintf(mld_str, sizeof(mld_str), "%d", id);
			mld_sec = get_dup_section_in_config_opt("mapcontroller", "mld", "id", mld_str);
			if (mld_sec == NULL) {
				snprintf(mld_id, sizeof(mld_id), "%s", mld_str);
				break;
			}
		}

		if (DM_STRLEN(mld_id) == 0) {
			bbfdm_set_fault_message(ctx, "No free MLD unit in range of 1 to 24");
			return "Error_Other";
		} else {
			dmuci_add_section("mapcontroller", "mld", &s);
			dmuci_set_value_by_section(s, "ssid", op_param->ssid);
			dmuci_set_value_by_section(s, "key", passwd);
			dmuci_set_value_by_section(s, "type", haul_type);
			dmuci_set_value_by_section(s, "id", mld_id);
		}
	}

	for (int i = 0; i < band_arr_length; i++) {
		char sec_name[64];
		char *encryp = NULL;
		char *device = NULL;
		char *instance = NULL;

		snprintf(sec_name, sizeof(sec_name), "ap_%s_%c_%u", op_param->ssid, band_arr[i][0], idx+i);
		replace_special_char(sec_name, '_');

		encryp = (enc_arr != NULL && i < enc_arr_length) ? get_AKMs_uci_name(enc_arr[i]) : "sae-mixed";

		/* Add section in mapcontroller */
		dmuci_add_section("mapcontroller", "ap", &s);
		dmuci_rename_section_by_section(s, sec_name);

		dmuci_set_value_by_section(s, "ssid", op_param->ssid);
		dmuci_set_value_by_section(s, "key", passwd);
		dmuci_set_value_by_section(s, "encryption", encryp);
		dmuci_set_value_by_section(s, "type", haul_type);
		dmuci_set_value_by_section(s, "enabled", (enable == true) ? "1" : "0");
		dmuci_set_value_by_section(s, "mld_id", (mld_enable == true) ? mld_id : "");
		dmuci_set_value_by_section(s, "vid", vid_str);
		dmuci_set_value_by_section(s, "network_type", DM_STRLEN(op_param->network_type) ? op_param->network_type : "");

		if (DM_LSTRCMP(band_arr[i], "2.4") == 0) {
			dmuci_set_value_by_section(s, "band", "2");
			device = get_device_from_band("2");
		} else {
			dmuci_set_value_by_section(s, "band", band_arr[i]);
			device = get_device_from_band(band_arr[i]);
		}

		if (DM_LSTRCMP(op_param->haul, "Backhaul") == 0) {
			// No need to create a dmmap section for Backhaul AP node
			continue;
		}

		// Device.WiFi.AccessPoint.{i}.
		create_dmmap_obj(ctx, 0, "WiFi", "AccessPoint", s, &instance);

		// Device.WiFi.SSID.{i}.
		struct uci_section *ssid_s = create_dmmap_obj(ctx, 0, "WiFi", "SSID", s, &instance);
		if (ssid_s && DM_STRLEN(instance)) {
			char name[64];

			snprintf(name, sizeof(name), "%s_%c_%u", op_param->ssid, band_arr[i][0], idx+i);

			dmuci_set_value_by_section(ssid_s, "enabled", (enable == true) ? "1" : "0");
			dmuci_set_value_by_section(ssid_s, "ssid", op_param->ssid);
			dmuci_set_value_by_section(ssid_s, "Name", name);
			dmuci_set_value_by_section(ssid_s, "device", device ? device : "");
			dmuci_set_value_by_section(ssid_s, "mld_id", (mld_enable == true) ? mld_id : "");
		}
	}

	return "Success";
}

static char *process_set_ssid_remove_req(struct dmctx *ctx, set_ssid_param *op_param)
{
	struct uci_section *s = NULL, *tmp = NULL, *dmmap = NULL;
	char *curr_ssid = NULL, *mld = NULL;
	char ap_sec_name[64] = {0};
	bool ap_deleted = false;

	if (op_param == NULL) {
		bbfdm_set_fault_message(ctx, "Internal Error");
		return "Error_Other";
	}

	/* Remove from mapcontroller */
	uci_foreach_sections_safe("mapcontroller", "ap", tmp, s) {
		dmuci_get_value_by_section_string(s, "ssid", &curr_ssid);
		if (DM_STRCMP(op_param->ssid, curr_ssid) != 0) {
			continue;
		}

		dmuci_get_value_by_section_string(s, "mld_id", &mld);

		// delete this section
		dmuci_delete_by_section(s, NULL, NULL);

		snprintf(ap_sec_name, sizeof(ap_sec_name), "mapcontroller.%s", section_name(s));

		dmmap = get_dup_section_in_dmmap_opt("WiFi", "AccessPoint", "__section_name__", ap_sec_name);
		if (dmmap) {
			dmuci_delete_by_section(dmmap, NULL, NULL);
		}

		ap_deleted = true;
	}

	if (ap_deleted == false) {
		bbfdm_set_fault_message(ctx, "No such SSID exists");
		return "Error_Invalid_Input";
	}

	/* Remove from dmmap */
	uci_path_foreach_sections_safe(bbfdm, "WiFi", "SSID", tmp, s) {
		dmuci_get_value_by_section_string(s, "ssid", &curr_ssid);
		if (DM_STRCMP(op_param->ssid, curr_ssid) != 0) {
			continue;
		}

		// delete this section
		dmuci_delete_by_section(s, NULL, NULL);
	}

	/* If mld was configured then after deleting all SSIDs we should remove MLD section */
	if (DM_STRLEN(mld) != 0) {
		s = get_dup_section_in_config_opt("mapcontroller", "mld", "id", mld);
		if (s) {
			dmuci_delete_by_section(s, NULL, NULL);
		}
	}

	return "Success";
}

static char *process_set_ssid_change_req(struct dmctx *ctx, set_ssid_param *op_param)
{
	struct uci_section *s = NULL;
	char *curr_ssid = NULL;
	unsigned ap_count = 0;

	if (op_param == NULL) {
		bbfdm_set_fault_message(ctx, "Internal Error");
		return "Error_Other";
	}

	/* Check if ssid is present */
	uci_foreach_sections("mapcontroller", "ap", s) {
		dmuci_get_value_by_section_string(s, "ssid", &curr_ssid);

		if (DM_STRCMP(curr_ssid, op_param->ssid) == 0) {
			ap_count++;
		}
	}

	if (ap_count == 0) {
		bbfdm_set_fault_message(ctx, "No such SSID exists");
		return "Error_Invalid_Input";
	}

	if (DM_STRLEN(op_param->band) != 0) {
		bbfdm_set_fault_message(ctx, "Once assigned, band can not be modified");
		return "Error_Invalid_Input";
	}

	if (DM_STRLEN(op_param->akm) != 0) {
		bbfdm_set_fault_message(ctx, "Once assigned, AKMsAllowed can not be modified");
		return "Error_Invalid_Input";
	}

	if (DM_STRLEN(op_param->mld_enab) != 0) {
		bbfdm_set_fault_message(ctx, "Once assigned, MLD setting can not be modified");
		return "Error_Invalid_Input";
	}

	if (DM_STRLEN(op_param->new_ssid) != 0) {
		/* There should not exist any ssid with new value */
		unsigned idx = 0;
		if (!validate_requested_ssid(op_param->new_ssid, &idx)) {
			bbfdm_set_fault_message(ctx, "SSID can't be changed to (%s), since it's already exist", op_param->new_ssid);
			return "Error_Invalid_Input";
		}
	}

	uci_foreach_sections("mapcontroller", "ap", s) {
		dmuci_get_value_by_section_string(s, "ssid", &curr_ssid);
		if (DM_STRCMP(op_param->ssid, curr_ssid) != 0) {
			continue;
		}

		// modify this section
		bool enable = true;
		if (DM_STRLEN(op_param->enab) != 0) {
			string_to_bool(op_param->enab, &enable);
			dmuci_set_value_by_section(s, "enabled", (enable == true) ? "1" : "0");
		}

		if (DM_STRLEN(op_param->key) != 0) {
			dmuci_set_value_by_section(s, "key", op_param->key);
		}

		if (DM_STRLEN(op_param->haul) != 0) {
			dmuci_set_value_by_section(s, "type", get_haultype_uci_value(op_param->haul));
		}

		if (DM_STRLEN(op_param->new_ssid) != 0) {
			dmuci_set_value_by_section(s, "ssid", op_param->new_ssid);
		}

		if (DM_STRLEN(op_param->network_type) != 0) {
			dmuci_set_value_by_section(s, "network_type", op_param->network_type);
		}

	}

	/* Modify in dmmap */
	uci_path_foreach_sections(bbfdm, "WiFi", "SSID", s) {
		dmuci_get_value_by_section_string(s, "ssid", &curr_ssid);
		if (DM_STRCMP(op_param->ssid, curr_ssid) != 0) {
			continue;
		}

		// modify this section
		bool enable = true;
		if (DM_STRLEN(op_param->enab) != 0) {
			string_to_bool(op_param->enab, &enable);
			dmuci_set_value_by_section(s, "enabled", (enable == true) ? "1" : "0");
		}

		if (DM_STRLEN(op_param->new_ssid) != 0) {
			dmuci_set_value_by_section(s, "ssid", op_param->new_ssid);
		}
	}

	if (mld_capable == false)
		return "Success";

	/* if there exist a MLD section with key ssid then MLD section should be updated */
	struct uci_section *mld_sec = get_dup_section_in_config_opt("mapcontroller", "mld", "ssid", op_param->ssid);
	if (mld_sec != NULL) {
		if (DM_STRLEN(op_param->key) != 0) {
			dmuci_set_value_by_section(mld_sec, "key", op_param->key);
		}

		if (DM_STRLEN(op_param->haul) != 0) {
			dmuci_set_value_by_section(mld_sec, "type", get_haultype_uci_value(op_param->haul));
		}

		if (DM_STRLEN(op_param->new_ssid) != 0) {
			dmuci_set_value_by_section(mld_sec, "ssid", op_param->new_ssid);
		}
	}

	return "Success";
}

static void wifi_validate_easymesh_ssid(struct ubus_request *req, int type, struct blob_attr *msg)
{
	validate_ssid *param = (validate_ssid *)req->priv;
	json_object *dev_arr = NULL, *dev_obj = NULL;
	bool success = true;
	int j = 0;

	if (!param || !msg)
		return;

	param->status = -1;

	char *json_str = blobmsg_format_json_indent(msg, true, -1);
	if (!json_str) {
		BBF_ERR("Error: Failed to format JSON from blobmsg in (%s)", __func__);
		return;
	}

	json_object *dump_output = json_tokener_parse(json_str);
	free(json_str);

	if (!dump_output) {
		BBF_ERR("Error: Failed to parse JSON string in (%s)", __func__);
		return;
	}

	json_object *data_arr = NULL;
	if (!json_object_object_get_ex(dump_output, "data", &data_arr) ||
	    json_object_get_type(data_arr) != json_type_array) {
		json_object_put(dump_output);
		return;
	}

	json_object *data_obj = json_object_array_get_idx(data_arr, 0);
	if (!data_obj) {
		json_object_put(dump_output);
		return;
	}

	dmjson_foreach_obj_in_array(data_obj, dev_arr, dev_obj, j, 2, "wfa-dataelements:Network", "DeviceList") { // Device List
		struct json_object *radio_arr = NULL, *radio_obj = NULL;
		bool found_ssid = false;
		int k = 0;

		dmjson_foreach_obj_in_array(dev_obj, radio_arr, radio_obj, k, 1, "RadioList") { // Radio List
			json_object *bss_arr = NULL, *bss_obj = NULL;
			char mac[32] = {0};
			int l = 0;

			const char *radio_id = dmjson_get_value(radio_obj, 1, "ID");
			if (!radio_id)
				continue;

			char *decoded = dm_base64_decode(radio_id);

			if (decoded && DM_STRLEN(radio_id) == 8)
				string_to_mac(decoded, 6, mac, sizeof(mac));

			if (DM_STRLEN(mac) == 0)
				continue;

			dmjson_foreach_obj_in_array(radio_obj, bss_arr, bss_obj, l, 1, "BSSList") { // BSS List
				const char *ssid = dmjson_get_value(bss_obj, 1, "SSID");
				if (DM_STRCMP(ssid, param->ssid) == 0) {
					found_ssid = true;
					break;
				}
			} // BSS List

			if (found_ssid)
				break;
		} // Radio List

		if ((param->op == SET_SSID_REMOVE && found_ssid) ||
		   (param->op != SET_SSID_REMOVE && !found_ssid)) {
			success = false;
			break;
		}
	}

	json_object_put(dump_output);
	param->status = success ? 0 : -1;
}

static void check_easymesh_agent_ssid(validate_ssid *param)
{
	struct uci_section *ap = get_dup_section_in_config_opt("mapcontroller", "ap", "ssid", param->ssid);
	if (ap && (param->op == SET_SSID_ADD || param->op == SET_SSID_CHANGE)) {
		bool enable = true;
		char *val = dmuci_get_value_by_section_fallback_def(ap, "enabled", "1");
		string_to_bool(val, &enable);

		if (!enable) {
			// No need to wait for network validation if SSID is disabled
			param->status = 0;
			return;
		}
	}

	struct blob_buf bb = {0};

	memset(&bb, 0, sizeof(struct blob_buf));
	blob_buf_init(&bb, 0);

	// Retry up to 8 times (total 80s wait)
	for (int i = 0; i < 8; i++) {
		sleep(10);

		/*
		 * It is mandatory to call "wifi.dataelements dump" UBUS method to validate the SSID,
		 * instead of using the internal global data elements structure (bbfdm_wifi_de).
		 * This is because the operate commands are executed inside a forked process, and therefore
		 * the current network data elements are not accessible from that global internal structure.
		 * Hence, calling "ubus call wifi.dataelements dump" each time is required to retrieve
		 * the latest network state before validating the new SSID.
		 */
		BBFDM_UBUS_INVOKE_SYNC("wifi.dataelements", "dump", bb.head, 3000, wifi_validate_easymesh_ssid, param);

		if (param->status == 0)
			break;
	}

	blob_buf_free(&bb);
}

static int operate_WiFiDataElementsNetwork_SetSSID(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	set_ssid_param op_param;
	char status[32] = {0};

	char *add_remove_change = dmjson_get_value((json_object *)value, 1, "AddRemoveChange");
	int op = get_requested_operation(add_remove_change);

	if (op == SET_SSID_INVALID) {
		bbfdm_set_fault_message(ctx, "Invalid AddRemoveChange value: (%s)", add_remove_change);
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	memset(&op_param, 0, sizeof(set_ssid_param));

	op_param.ssid = dmjson_get_value((json_object *)value, 1, "SSID");
	op_param.new_ssid = dmjson_get_value((json_object *)value, 1, CUSTOM_PREFIX"NewSSID");
	op_param.enab = dmjson_get_value((json_object *)value, 1, "Enable");
	op_param.key = dmjson_get_value((json_object *)value, 1, "PassPhrase");
	op_param.band = dmjson_get_value((json_object *)value, 1, "Band");
	op_param.akm = dmjson_get_value((json_object *)value, 1, "AKMsAllowed");
	op_param.haul = dmjson_get_value((json_object *)value, 1, "HaulType");
	op_param.network_type = dmjson_get_value((json_object *)value, 1, "Type");
	op_param.mld_enab = dmjson_get_value((json_object *)value, 1, CUSTOM_PREFIX"EnableMLD");
	op_param.ts = dmjson_get_value((json_object *)value, 1, CUSTOM_PREFIX"TrafficSeparation");

	if (bbfdm_validate_string(ctx, op_param.ssid, 1, 32, NULL, NULL) ||
			bbfdm_validate_string(ctx, op_param.key, -1, 64, NULL, NULL)||
			validate_band_value(ctx, op_param.band) != 0 ||
			validate_akms_value(ctx, op_param.akm) != 0 ||
			validate_haultype_value(ctx, op_param.haul) != 0) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	if (bbfdm_validate_string(ctx, op_param.new_ssid, 1, 32, NULL, NULL) &&
			op == SET_SSID_CHANGE ) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	if (DM_STRLEN(op_param.enab) != 0 && bbfdm_validate_boolean(ctx, op_param.enab) != 0) {
		bbfdm_set_fault_message(ctx, "Invalid Enable value: (%s)", op_param.enab);
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	if (DM_STRLEN(op_param.mld_enab) != 0 && bbfdm_validate_boolean(ctx, op_param.mld_enab) != 0) {
		bbfdm_set_fault_message(ctx, "Invalid Enable value: (%s)", op_param.mld_enab);
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	if (DM_STRLEN(op_param.ts) != 0 && bbfdm_validate_boolean(ctx, op_param.ts) != 0) {
		bbfdm_set_fault_message(ctx, "Invalid Enable value: (%s)", op_param.ts);
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	switch (op) {
	case SET_SSID_ADD:
		snprintf(status, sizeof(status), "%s", process_set_ssid_add_req(ctx, &op_param));
		break;
	case SET_SSID_REMOVE:
		snprintf(status, sizeof(status), "%s", process_set_ssid_remove_req(ctx, &op_param));
		break;
	case SET_SSID_CHANGE:
		snprintf(status, sizeof(status), "%s", process_set_ssid_change_req(ctx, &op_param));
		break;
	}

	if (DM_LSTRCMP(status, "Success") != 0) {
		// Revert mapcontroller config changes
		reload_mapcontroller_service(ctx->dm_type, false, false);
	} else {
		// Commit mapcontroller config changes
		reload_mapcontroller_service(ctx->dm_type, true, true);

		validate_ssid v_param;
		v_param.op = op;
		v_param.status = -1;
		v_param.ssid = op_param.ssid;

		if (op == SET_SSID_CHANGE && DM_STRLEN(op_param.new_ssid) != 0) {
			/* SSID would be changed with new_ssid */
			v_param.ssid = op_param.new_ssid;
		}

		check_easymesh_agent_ssid(&v_param);
		if (v_param.status != 0) {
			snprintf(status, sizeof(status), "%s", "Error_Other");
		}
	}

end:
	fill_blob_param(&ctx->bb, "Status", status, DMT_TYPE[DMT_STRING], 0);
	return 0;
}

static operation_args WiFiDataElementsNetworkDevice_SetSTASteeringState_args = {
	.in = (const char *[]) {
		"Disallowed",
		NULL
	},
	.out = (const char *[]) {
		"Status",
		NULL
	}
};

static int get_operate_args_WiFiDataElementsNetworkDevice_SetSTASteeringState(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char *)&WiFiDataElementsNetworkDevice_SetSTASteeringState_args;
	return 0;
}

static int operate_WiFiDataElementsNetworkDevice_SetSTASteeringState(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char status[32];
	bool b = false;

	snprintf(status, sizeof(status), "%s", "Success");

	char *disallowed = dmjson_get_value((json_object *)value, 1, "Disallowed");
	if (!disallowed || *disallowed == '\0' || bbfdm_validate_boolean(ctx, disallowed)) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	string_to_bool(disallowed, &b);
	dmuci_set_value_by_section(((struct dm_data *)data)->config_section, "steer_disallow", b ? "1" : "0");

end:
	fill_blob_param(&ctx->bb, "Status", status, DMT_TYPE[DMT_STRING], 0);
	return 0;
}

static operation_args WiFiDataElementsNetworkDevice_InitiateWPSPBC_args = {
	.out = (const char *[]) {
		"Status",
		NULL
	}
};

static int get_operate_args_WiFiDataElementsNetworkDevice_InitiateWPSPBC(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char *)&WiFiDataElementsNetworkDevice_InitiateWPSPBC_args;
	return 0;
}

static int operate_WiFiDataElementsNetworkDevice_InitiateWPSPBC(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	/* This is a temporary solution as pushback operation to network devices is not supported
	 * right now. This should be done later through mapcontroller. So as of now it is being
	 * supported on controller GW device only */
	if (DM_LSTRCMP(instance, "1") != 0) {
		bbfdm_set_fault_message(ctx, "InitiateWPSPBC() is only supported on controller gateway device");
		return USP_FAULT_REQUEST_DENIED;
	}

	dmubus_call_set("map.agent", "WPSPBC", UBUS_ARGS{0}, 0);
	fill_blob_param(&ctx->bb, "Status", "Success", DMT_TYPE[DMT_STRING], 0);
	return 0;
}

static operation_args wifidataelementsnetworkdevice_refreshapmetrics_args = {
    .in = (const char *[]) {
        "RUID",
        NULL
    },
    .out = (const char *[]) {
        "Status",
        NULL
    }
};

static int get_operate_args_WiFiDataElementsNetworkDevice_RefreshAPMetrics(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
    *value = (char *)&wifidataelementsnetworkdevice_refreshapmetrics_args;
    return 0;
}

static int operate_WiFiDataElementsNetworkDevice_RefreshAPMetrics(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	char *agent_id = NULL, *ruid = NULL;
	struct blob_buf bb = {0};
	json_object *res = NULL;
	char status[32];
	int err = 0;

	snprintf(status, sizeof(status), "%s", "Success");

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "agent_id", &agent_id);
	if (DM_STRLEN(agent_id) == 0) {
		snprintf(status, sizeof(status), "%s", "Error_Other");
		goto end;
	}

	ruid = dmjson_get_value((json_object *)value, 1, "RUID");

	blob_buf_init(&bb, 0);

	blobmsg_add_string(&bb, "agent", agent_id);

	if (fill_bsslist_from_radio(dev, ruid, &bb)) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	err = dmubus_call_blob_msg("map.controller", "query_ap_metrics", &bb, &res);

	blob_buf_free(&bb);

	if (err || res == NULL) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	char *res_status = dmjson_get_value((json_object *)res, 1, "status");
	if (DM_STRCMP(res_status, "ok") != 0)
		snprintf(status, sizeof(status), "%s", "Error_Other");

end:
	fill_blob_param(&ctx->bb, "Status", status, DMT_TYPE[DMT_STRING], 0);
	return 0;
}

static operation_args wifidataelementsnetworkdeviceradio_channelscanrequest_args = {
    .in = (const char *[]) {
        "OpClass",
        "ChannelList",
        NULL
    },
    .out = (const char *[]) {
        "Status",
        NULL
    }
};

static int get_operate_args_WiFiDataElementsNetworkDeviceRadio_ChannelScanRequest(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
    *value = (char *)&wifidataelementsnetworkdeviceradio_channelscanrequest_args;
    return 0;
}

static int operate_WiFiDataElementsNetworkDeviceRadio_ChannelScanRequest(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	char status[32] = {0};
	char *agent_id = NULL;
	char *macaddr = NULL;
	char *pch = NULL;
	char *spch = NULL;

	snprintf(status, sizeof(status), "%s", "Success");

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "agent_id", &agent_id);
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "macaddr", &macaddr);

	if ((bbfdm_validate_string(ctx, agent_id, -1, 17, NULL, MACAddress)) ||
		(bbfdm_validate_string(ctx, macaddr, -1, 17, NULL, MACAddress))) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	char *channel_list = dmjson_get_value((json_object *)value, 1, "ChannelList");

	if (bbfdm_validate_unsignedInt_list(ctx, channel_list, -1, -1, -1, RANGE_ARGS{{NULL,"255"}}, 1)) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	struct json_object *in_args = json_object_new_object();
	json_object_object_add(in_args, "agent", json_object_new_string(agent_id));
	json_object_object_add(in_args, "radio", json_object_new_string(macaddr));

	if (DM_STRLEN(channel_list) != 0) {
		struct json_object *channel_array = json_object_new_array();

		for (pch = strtok_r(channel_list, ",", &spch); pch != NULL; pch = strtok_r(NULL, ",", &spch)) {
			struct json_object *val_array = json_object_new_array();
			json_object_array_add(val_array, json_object_new_int(DM_STRTOL(pch)));
			json_object_array_add(channel_array, val_array);
		}

		json_object_object_add(in_args, "channel", channel_array);
	}

	int res = dmubus_call_blob_set("map.controller", "scan", in_args);
	if (res) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
	}

	json_object_put(in_args);

end:
	fill_blob_param(&ctx->bb, "Status", status, DMT_TYPE[DMT_STRING], 0);
    return 0;
}

static operation_args wifidataelementsnetworkdeviceradio_channelselectionrequest_args = {
	.in = (const char *[]) {
		"Class.{i}.OpClass",
		"Class.{i}.Channel.{i}.Channel",
		"Class.{i}.Channel.{i}.Preference",
		NULL
	},
	.out = (const char *[]) {
		"Status",
		NULL
	}
};

static int get_operate_args_WiFiDataElementsNetworkDeviceRadio_ChannelSelectionRequest(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
    *value = (char *)&wifidataelementsnetworkdeviceradio_channelselectionrequest_args;
    return 0;
}

static int operate_WiFiDataElementsNetworkDeviceRadio_ChannelSelectionRequest(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	// ubus call map.controller send_channel_pref '{"agent":"96:3f:0c:bc:a8:20","radioid":"94:3f:0c:bc:a8:23","opclass":[122,122],"channel":[[108,116],[134,156]],"pref":[15,9]}'
#define MAX_OPCLASS_ARGS        64
#define MAX_CHANNEL_PER_OPCLASS 16

	struct channel_request {
		int num_opclass;
		struct opclass {
			unsigned char opclass;
			int num_channels;
			struct op_channel {
				int num_chan;
				unsigned int chan[MAX_CHANNEL_PER_OPCLASS];
				unsigned char preference;
			} channels[MAX_CHANNEL_PER_OPCLASS];
		} classes[MAX_OPCLASS_ARGS];
	};

	struct channel_request req = {0};
	struct json_object *res = NULL;
	struct blob_buf bb = {0};
	char *agent_id = NULL;
	char *radio_id = NULL;
	char status[32];
	int err = 0;

	snprintf(status, sizeof(status), "%s", "Success");

	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "agent_id", &agent_id);
	dmuci_get_value_by_section_string(((struct dm_data *)data)->config_section, "macaddr", &radio_id);

	if ((bbfdm_validate_string(ctx, agent_id, -1, 17, NULL, MACAddress)) ||
		(bbfdm_validate_string(ctx, radio_id, -1, 17, NULL, MACAddress))) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	// --- Parse input into structure ---
	for (int i = 0; i < MAX_OPCLASS_ARGS; i++) {
		char *p_value = NULL;
		char buf[128] = {0};

		snprintf(buf, sizeof(buf), "Class.%d.OpClass", i + 1);
		p_value = dmjson_get_value((json_object *)value, 1, buf);

		if (DM_STRLEN(p_value) == 0) {
			if (i == 0) { // must have at least one
				snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
				goto end;
			}
			break; // stop sequentially
		} else {
			if (bbfdm_validate_unsignedInt(ctx, p_value, RANGE_ARGS{{NULL,"255"}}, 1)) {
				snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
				goto end;
			}
		}

		struct opclass *cls = &req.classes[req.num_opclass++];
		cls->opclass = DM_STRTOUL(p_value);

		for (int j = 0; j <= MAX_CHANNEL_PER_OPCLASS; j++) {
			snprintf(buf, sizeof(buf), "Class.%d.Channel.%d.Preference", i + 1, j + 1);
			p_value = dmjson_get_value((json_object *)value, 1, buf);

			if (DM_STRLEN(p_value) == 0) {
				if (j == 0) { // must have at least one
					snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
					goto end;
				}
				break; // stop sequentially
			} else {
				if (bbfdm_validate_unsignedInt(ctx, p_value, RANGE_ARGS{{NULL,"15"}}, 1)) {
					snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
					goto end;
				}
			}

			struct op_channel *ch = &cls->channels[cls->num_channels++];
			ch->preference = DM_STRTOUL(p_value);

			// Parse channel list (can be comma-separated)
			snprintf(buf, sizeof(buf), "Class.%d.Channel.%d.Channel", i + 1, j + 1);
			p_value = dmjson_get_value((json_object *)value, 1, buf);

			if (DM_STRLEN(p_value) == 0) {
				if (j == 0) { // must have at least one
					snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
					goto end;
				}
				break; // stop sequentially
			}

			char *pch, *spch = NULL;
			for (pch = strtok_r(p_value, ",", &spch);
				 pch != NULL && ch->num_chan < MAX_CHANNEL_PER_OPCLASS;
				 pch = strtok_r(NULL, ",", &spch)) {

				if (bbfdm_validate_unsignedInt(ctx, pch, RANGE_ARGS{{NULL,"255"}}, 1)) {
					snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
					goto end;
				}

				ch->chan[ch->num_chan++] = DM_STRTOUL(pch);
			}
		}
	}

	// --- Serialize to blob_buf ---
	blob_buf_init(&bb, 0);

	blobmsg_add_string(&bb, "agent", agent_id);
	blobmsg_add_string(&bb, "radioid", radio_id);

	void *opclass = blobmsg_open_array(&bb, "opclass");
	for (int i = 0; i < req.num_opclass; i++) {
		for (int j = 0; j < req.classes[i].num_channels; j++) {
			blobmsg_add_u32(&bb, NULL, req.classes[i].opclass);
		}
	}
	blobmsg_close_array(&bb, opclass);

	void *channel = blobmsg_open_array(&bb, "channel");
	for (int i = 0; i < req.num_opclass; i++) {
		for (int j = 0; j < req.classes[i].num_channels; j++) {
			void *ch_list = blobmsg_open_array(&bb, NULL);
			for (int k = 0; k < req.classes[i].channels[j].num_chan; k++) {
				blobmsg_add_u32(&bb, NULL, req.classes[i].channels[j].chan[k]);
			}
			blobmsg_close_array(&bb, ch_list);
		}
	}
	blobmsg_close_array(&bb, channel);

	void *pref = blobmsg_open_array(&bb, "pref");
	for (int i = 0; i < req.num_opclass; i++) {
		for (int j = 0; j < req.classes[i].num_channels; j++) {
			blobmsg_add_u32(&bb, NULL, req.classes[i].channels[j].preference);
		}
	}
	blobmsg_close_array(&bb, pref);

	// --- Call ubus ---
	err = dmubus_call_blob_msg("map.controller", "send_channel_pref", &bb, &res);
	blob_buf_free(&bb);

	if (err || res == NULL) {
		snprintf(status, sizeof(status), "%s",
				(err == UBUS_STATUS_TIMEOUT) ? "Error_Timeout" : "Error_Other");
		goto end;
	}

	if (res) {
		char *res_status = dmjson_get_value((json_object *)res, 1, "status");
		if (DM_STRCMP(res_status, "ok") != 0)
			snprintf(status, sizeof(status), "%s", "Error_Other");
		json_object_put(res);
	}

end:
	fill_blob_param(&ctx->bb, "Status", status, DMT_TYPE[DMT_STRING], 0);
	return 0;
}

static operation_args wifidataelementsnetworkdevicemultiapdevicebackhaul_steerwifibackhaul_args = {
    .in = (const char *[]) {
        "TargetBSS",
        "TimeOut",
        NULL
    },
    .out = (const char *[]) {
        "Status",
        NULL
    }
};

static int get_operate_args_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaul_SteerWiFiBackhaul(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
    *value = (char *)&wifidataelementsnetworkdevicemultiapdevicebackhaul_steerwifibackhaul_args;
    return 0;
}

static int operate_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaul_SteerWiFiBackhaul(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	json_object *res = NULL;
	char *bss_band = NULL;
	char status[32];

	snprintf(status, sizeof(status), "%s", "Success");

	char *target_bss = dmjson_get_value((json_object *)value, 1, "TargetBSS");
	char *time_out = dmjson_get_value((json_object *)value, 1, "TimeOut");
	if (DM_STRLEN(target_bss) == 0 || DM_STRLEN(time_out) == 0) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	if (DM_STRLEN(target_bss) == 0 ||
			(bss_band = validate_bss(target_bss, true)) == NULL) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	char *agent = dm_to_macaddr_str(dev->macaddr);
	char *radio_mac = find_radio_mac(agent, bss_band);
	char *bsta = get_backhaul_sta_from_radio(dev, radio_mac);
	if (DM_STRLEN(agent) == 0 || DM_STRLEN(bsta) == 0) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	dmubus_call_timeout("map.controller", "steer_backhaul", UBUS_ARGS{{"agent", agent, String},
															{"bsta", bsta, String},
															{"target_bssid", target_bss, String},
															{"timeout", time_out, Integer}}, 4, (int)DM_STRTOUL(time_out), &res);

	if (res == NULL) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	char *res_status = dmjson_get_value((json_object *)res, 1, "status");
	if (DM_STRCMP(res_status, "ok") != 0)
		snprintf(status, sizeof(status), "%s", "Error_Other");

end:
	fill_blob_param(&ctx->bb, "Status", status, DMT_TYPE[DMT_STRING], 0);

	if (res != NULL) {
		json_object_put(res);
		res = NULL;
	}
	return 0;
}

static operation_args wifidataelementsnetworkdeviceradiobsssta_clientsteer_args = {
	.in = (const char *[]) {
		"BTMAbridged",
		"TargetBSSID",
		"BTMDisassociationTimer",
		NULL
	},
	.out = (const char *[]) {
		"Status",
		NULL
	}
};

static int get_operate_args_WiFiDataElementsNetworkDeviceRadioBSSSTA_ClientSteer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
	*value = (char *)&wifidataelementsnetworkdeviceradiobsssta_clientsteer_args;
	return 0;
}

static int operate_WiFiDataElementsNetworkDeviceRadioBSSSTA_ClientSteer(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	struct wifi_sta_element *sta_entry = (struct wifi_sta_element *)((struct dm_data *)data)->additional_data;
	json_object *res = NULL;
	char status[32];

	snprintf(status, sizeof(status), "%s", "Success");

	char *target_bss = dmjson_get_value((json_object *)value, 1, "TargetBSSID");
	char *diass_timer = dmjson_get_value((json_object *)value, 1, "BTMDisassociationTimer");
	char *btm_abridged = dmjson_get_value((json_object *)value, 1, "BTMAbridged");

	if (DM_STRLEN(target_bss) == 0 ||
			validate_bss(target_bss, false) == NULL) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	if (DM_STRLEN(diass_timer) != 0) {
		char *endval = NULL;

		errno = 0;
		strtoul(diass_timer, &endval, 10);
		if ((*endval != 0) || (errno != 0)) {
			snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
			goto end;
		}
	} else {
		diass_timer = "0";
	}

	if (DM_STRLEN(btm_abridged) != 0) {
		bool abridged = false;

		if (string_to_bool(btm_abridged, &abridged) != 0) {
			snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
			goto end;
		}

		btm_abridged = abridged ? "1" : "0";
	} else {
		btm_abridged = "0";
	}


	char *sta = dm_to_macaddr_str(sta_entry->macaddr);
	if (DM_STRLEN(sta) == 0) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	dmubus_call("map.controller", "steer", UBUS_ARGS{{"sta", sta, String},
							{"target_bssid", target_bss, String},
							{"disassoc_tmo", diass_timer, Integer},
							{"abridged", btm_abridged, Integer}},
							4, &res);

	if (res == NULL) {
		snprintf(status, sizeof(status), "%s", "Error_Invalid_Input");
		goto end;
	}

	char *res_status = dmjson_get_value((json_object *)res, 1, "status");
	if (DM_STRCMP(res_status, "ok") != 0)
		snprintf(status, sizeof(status), "%s", "Error_Other");

end:
	fill_blob_param(&ctx->bb, "Status", status, DMT_TYPE[DMT_STRING], 0);
	return 0;
}

/*************************************************************
 * EVENTS
 *************************************************************/
static event_args wifidataelementsnetwork_agentonboard_args = {
    .name = "", //TODO
    .param = (const char *[]) {
        "DeviceID",
        "BackhaulSta",
        "BackhaulMACAddress",
        "BackhaulALID",
        NULL
    }
};

static int get_event_args_WiFiDataElementsNetwork_AgentOnboard(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
    *value = (char *)&wifidataelementsnetwork_agentonboard_args;
    return 0;
}

static event_args wifidataelementsnetwork_controllerbtmsteer_args = {
    .name = "map.controller",
    .param = (const char *[]) {
        "BSSID",
        "STAMAC",
        "BTMStatusCode",
        NULL
    }
};

static int get_event_args_WiFiDataElementsNetwork_ControllerBTMSteer(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
    *value = (char *)&wifidataelementsnetwork_controllerbtmsteer_args;
    return 0;
}

static const char *btm_status_reasons[] = { /* IEEE Std 802.11-2012 - Table 8-253 */
	"Accept", /* WNM_BSS_TM_ACCEPT */
	"Reject_Unspecified", /* WNM_BSS_TM_REJECT_UNSPECIFIED */
	"Reject_Insufficient_Response", /* WNM_BSS_TM_REJECT_INSUFFICIENT_BEACON */
	"Reject_Insufficient_Capacity", /* WNM_BSS_TM_REJECT_INSUFFICIENT_CAPABITY */
	"Reject_Undesired", /* WNM_BSS_TM_REJECT_UNDESIRED */
	"Reject_Delay", /* WNM_BSS_TM_REJECT_DELAY_REQUEST */
	"Reject_STA_List", /* WNM_BSS_TM_REJECT_STA_CANDIDATE_LIST_PROVIDED */
	"Reject_No_BSS", /* WNM_BSS_TM_REJECT_NO_SUITABLE_CANDIDATES */
	"Reject_Leaving_ESS" /* WNM_BSS_TM_REJECT_LEAVING_ESS */
};

static const char *get_btm_steer_status_reason(const char *status)
{
	if (DM_STRLEN(status) == 0)
		return "Reject_Unspecified";

	unsigned int code = DM_STRTOUL(status);

	if (code < ARRAY_SIZE(btm_status_reasons))
		return btm_status_reasons[code];

	return "Reject_Unspecified";
}

static int event_WiFiDataElementsNetwork_ControllerBTMSteer(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	/* { "map.controller": {"event":"client_steer_report","data":{"bssid":"94:3f:0c:bc:a8:23","sta_mac":"22:15:68:bb:c4:d1","target_bssid":"fe:d4:37:71:bf:78","status":0}} } */
	char *event_type = NULL;

	switch (action)	{
		case EVENT_CHECK:
			event_type = dmjson_get_value((json_object *)value, 1, "event");
			if (DM_STRCMP(event_type, "client_steer_report") != 0)
				return -1;
			break;
		case EVENT_RUN:
		{
			char *bssid = dmjson_get_value((json_object *)value, 2, "data", "bssid");
			char *sta_mac = dmjson_get_value((json_object *)value, 2, "data", "sta_mac");
			char *status = dmjson_get_value((json_object *)value, 2, "data", "status");

			fill_blob_param(&ctx->bb, "BSSID", bssid, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "STAMAC", sta_mac, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "BTMStatusCode", get_btm_steer_status_reason(status), DMT_TYPE[DMT_STRING], 0);
			break;
		}
	}

	return 0;
}

static event_args wifidataelementsnetwork_backhaulchange_args = {
    .name = "map.controller",
    .param = (const char *[]) {
        "MACAddress",
        "TargetBSSID",
        "ResultCode",
        "ReasonCode",
        NULL
    }
};

static int get_event_args_WiFiDataElementsNetworkDevice_BackhaulChange(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
    *value = (char *)&wifidataelementsnetwork_backhaulchange_args;
    return 0;
}

// Reason codes for backhaul steering
enum BackhaulSteerReason {
    BSR_NOT_ASSOCIATED = 2,
    BSR_CAPABILITY_REPORT_FAILURE = 3,
    BSR_REJECT_CHANNEL = 4,
    BSR_REJECT_WEAK = 5,
    BSR_REJECT_ASSOCIATION = 6
};

static const char *get_backhaul_steer_reason_code(const char *result)
{
    if (DM_STRCMP(result, "0") == 0)
        return "Associated";

    unsigned int reason_code = DM_STRTOUL(result);

    switch (reason_code) {
        case BSR_NOT_ASSOCIATED:
            return "Not_Associated"; /* STA not associated with any BSS operated by the Multi-AP Agent */
        case BSR_CAPABILITY_REPORT_FAILURE:
            return "Capability_Report_Failure"; /* Client capability report unspecified failure */
        case BSR_REJECT_CHANNEL:
            return "Reject_Channel"; /* Backhaul steering request rejected because the backhaul STA cannot operate on the channel specified */
        case BSR_REJECT_WEAK:
            return "Reject_Weak"; /* Backhaul steering request rejected because the target BSS signal is too weak or not found */
        case BSR_REJECT_ASSOCIATION:
            return "Reject_Association"; /* Backhaul steering request authentication or association rejected by the target BSS */
        default:
            return "Not_Associated"; /* STA not associated with any BSS operated by the Multi-AP Agent */
    }
}

static int event_WiFiDataElementsNetworkDevice_BackhaulChange(char *refparam, struct dmctx *ctx, void *data, char *instance, char *value, int action)
{
	/* { "map.controller": {"event":"backhaul_steer_result","data":{"agent":"5a:00:32:fb:d6:c0","macaddr":"58:00:32:fb:d6:ce","target_bssid":"96:3f:0c:bc:a8:22","result":0}} } */
	struct wifi_network_device *dev = (struct wifi_network_device *)((struct dm_data *)data)->additional_data;
	char *agent_id = NULL, *event_agent = NULL, *event_type = NULL;

	switch (action)	{
		case EVENT_CHECK:
			event_type = dmjson_get_value((json_object *)value, 1, "event");
			if (DM_STRCMP(event_type, "backhaul_steer_result") != 0)
				return -1;

			agent_id = dm_to_macaddr_str(dev->macaddr);
			event_agent = dmjson_get_value((json_object *)value, 2, "data", "agent");
			if (DM_STRCMP(agent_id, event_agent) != 0)
				return -1;

			break;
		case EVENT_RUN:
		{
			char *macaddr = dmjson_get_value((json_object *)value, 2, "data", "macaddr");
			char *target_bssid = dmjson_get_value((json_object *)value, 2, "data", "target_bssid");
			char *result = dmjson_get_value((json_object *)value, 2, "data", "result");

			fill_blob_param(&ctx->bb, "MACAddress", macaddr, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "TargetBSSID", target_bssid, DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "ResultCode", (DM_STRCMP(result, "0") == 0) ? "Success" : "Failure", DMT_TYPE[DMT_STRING], 0);
			fill_blob_param(&ctx->bb, "ReasonCode", get_backhaul_steer_reason_code(result), DMT_TYPE[DMT_STRING], 0);
			break;
		}
	}

	return 0;
}

static event_args wifidataelementsassociationevent_associated_args = {
	.name = "",
    .param = (const char *[]) {
        "TimeStamp",
        "BSSID",
        "MACAddress",
        "StatusCode",
        "HTCapabilities",
        "VHTCapabilities",
        NULL
    }
};

static int get_event_args_WiFiDataElementsAssociationEvent_Associated(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
    *value = (char *)&wifidataelementsassociationevent_associated_args;
    return 0;
}

void bbfdm_send_WiFiDataElementsAssociationEvent_Associated_event(struct wifi_assoc_event *e)
{
	struct blob_buf bb;
	char buf[18] = {0};

	memset(&bb, 0, sizeof(struct blob_buf));
	blob_buf_init(&bb, 0);

	blobmsg_add_string(&bb, "name", "Device.WiFi.DataElements.AssociationEvent.Associated!");
	void *arr = blobmsg_open_array(&bb, "input");

	fill_blob_param(&bb, "TimeStamp", e->tsp, DMT_TYPE[DMT_STRING], 0);

	hwaddr_ntoa(e->bssid, buf);
	fill_blob_param(&bb, "BSSID", buf, DMT_TYPE[DMT_STRING], 0);

	hwaddr_ntoa(e->macaddr, buf);
	fill_blob_param(&bb, "MACAddress", buf, DMT_TYPE[DMT_STRING], 0);

	snprintf(buf, sizeof(buf), "%u", e->status_code);
	fill_blob_param(&bb, "StatusCode", buf, DMT_TYPE[DMT_UNINT], 0);

	to_base64_str(&e->caps.ht, sizeof(e->caps.ht), buf, sizeof(buf));
	fill_blob_param(&bb, "HTCapabilities", buf, DMT_TYPE[DMT_BASE64], 0);

	to_base64_str(e->caps.vht, sizeof(e->caps.vht), buf, sizeof(buf));
	fill_blob_param(&bb, "VHTCapabilities", buf, DMT_TYPE[DMT_BASE64], 0);

	blobmsg_close_array(&bb, arr);

	BBFDM_UBUS_SEND_EVENT("bbfdm.event", bb.head);

	blob_buf_free(&bb);
}

static event_args wifidataelementsdisassociationevent_disassociated_args = {
	.name = "",
    .param = (const char *[]) {
        "BSSID",
        "MACAddress",
        "ReasonCode",
        "BytesSent",
        "BytesReceived",
        "PacketsSent",
        "PacketsReceived",
        "ErrorsSent",
        "ErrorsReceived",
        "RetransCount",
        "TimeStamp",
        "LastDataDownlinkRate",
        "LastDataUplinkRate",
        "UtilizationReceive",
        "UtilizationTransmit",
        "EstMACDataRateDownlink",
        "EstMACDataRateUplink",
        "SignalStrength",
        "LastConnectTime",
        "Noise",
        NULL
    }
};

static int get_event_args_WiFiDataElementsDisassociationEvent_Disassociated(char *refparam, struct dmctx *ctx, void *data, char *instance, char **value)
{
    *value = (char *)&wifidataelementsdisassociationevent_disassociated_args;
    return 0;
}

void bbfdm_send_WiFiDataElementsAssociationEvent_Disassociated_event(struct wifi_disassoc_event *e)
{
	struct blob_buf bb;
	char path[64] = {0};
	char buf[18] = {0};

	memset(&bb, 0, sizeof(struct blob_buf));
	blob_buf_init(&bb, 0);

	blobmsg_add_string(&bb, "name", "Device.WiFi.DataElements.DisassociationEvent.Disassociated!");
	void *arr = blobmsg_open_array(&bb, "input");

	fill_blob_param(&bb, "TimeStamp", e->tsp, DMT_TYPE[DMT_STRING], 0);

	hwaddr_ntoa(e->bssid, buf);
	fill_blob_param(&bb, "BSSID", dm_to_macaddr_str(e->bssid), DMT_TYPE[DMT_STRING], 0);

	hwaddr_ntoa(e->bssid, buf);
	fill_blob_param(&bb, "MACAddress", dm_to_macaddr_str(e->macaddr), DMT_TYPE[DMT_STRING], 0);

	snprintf(buf, sizeof(buf), "%u", e->reason_code);
	fill_blob_param(&bb, "ReasonCode", buf, DMT_TYPE[DMT_UNINT], 0);

	snprintf(buf, sizeof(buf), "%"PRIu64, e->sta.tx_bytes); /* Flawfinder: ignore */
	fill_blob_param(&bb, "BytesSent", buf, DMT_TYPE[DMT_UNLONG], 0);

	snprintf(buf, sizeof(buf), "%"PRIu64, e->sta.rx_bytes); /* Flawfinder: ignore */
	fill_blob_param(&bb, "BytesReceived", buf, DMT_TYPE[DMT_UNLONG], 0);

	snprintf(buf, sizeof(buf), "%u", e->sta.tx_pkts);
	fill_blob_param(&bb, "PacketsSent", buf, DMT_TYPE[DMT_UNLONG], 0);

	snprintf(buf, sizeof(buf), "%u", e->sta.rx_pkts);
	fill_blob_param(&bb, "PacketsReceived", buf, DMT_TYPE[DMT_UNLONG], 0);

	snprintf(buf, sizeof(buf), "%u", e->sta.tx_errors);
	fill_blob_param(&bb, "ErrorsSent", buf, DMT_TYPE[DMT_UNINT], 0);

	snprintf(buf, sizeof(buf), "%u", e->sta.rx_errors);
	fill_blob_param(&bb, "ErrorsReceived", buf, DMT_TYPE[DMT_UNINT], 0);

	snprintf(buf, sizeof(buf), "%u", e->sta.rtx_pkts);
	fill_blob_param(&bb, "RetransCount", buf, DMT_TYPE[DMT_UNINT], 0);

	snprintf(buf, sizeof(buf), "%u", e->sta.dl_rate);
	fill_blob_param(&bb, "LastDataDownlinkRate", buf, DMT_TYPE[DMT_UNINT], 0);

	snprintf(buf, sizeof(buf), "%u", e->sta.ul_rate);
	fill_blob_param(&bb, "LastDataUplinkRate", buf, DMT_TYPE[DMT_UNINT], 0);

	snprintf(buf, sizeof(buf), "%lu", e->sta.dl_utilization);
	fill_blob_param(&bb, "UtilizationReceive", buf, DMT_TYPE[DMT_UNLONG], 0);

	snprintf(buf, sizeof(buf), "%lu", e->sta.ul_utilization);
	fill_blob_param(&bb, "UtilizationTransmit", buf, DMT_TYPE[DMT_UNLONG], 0);

	snprintf(buf, sizeof(buf), "%u", 0);
	fill_blob_param(&bb, "EstMACDataRateDownlink", buf, DMT_TYPE[DMT_UNINT], 0);

	snprintf(buf, sizeof(buf), "%u", 0);
	fill_blob_param(&bb, "EstMACDataRateUplink", buf, DMT_TYPE[DMT_UNINT], 0);

	snprintf(buf, sizeof(buf), "%u", e->sta.rcpi);
	fill_blob_param(&bb, "SignalStrength", buf, DMT_TYPE[DMT_UNINT], 0);

	snprintf(buf, sizeof(buf), "%u", e->sta.conn_time);
	fill_blob_param(&bb, "LastConnectTime", buf, DMT_TYPE[DMT_UNINT], 0);

	snprintf(buf, sizeof(buf), "%u", 0);
	fill_blob_param(&bb, "Noise", buf, DMT_TYPE[DMT_UNINT], 0);

	for (int i = 0; i < e->num_affiliated_sta; i++) {
		snprintf(path, sizeof(path), "DisassociationLinkStats.%d.MACAddress", i + 1);
		fill_blob_param(&bb, path, dm_to_macaddr_str(e->affsta[i].macaddr), DMT_TYPE[DMT_STRING], 0);

		snprintf(path, sizeof(path), "DisassociationLinkStats.%d.BytesSent", i + 1);
		snprintf(buf, sizeof(buf), "%"PRIu64, e->affsta[i].tx_bytes); /* Flawfinder: ignore */
		fill_blob_param(&bb, path, buf, DMT_TYPE[DMT_UNLONG], 0);

		snprintf(path, sizeof(path), "DisassociationLinkStats.%d.BytesReceived", i + 1);
		snprintf(buf, sizeof(buf), "%"PRIu64, e->affsta[i].rx_bytes); /* Flawfinder: ignore */
		fill_blob_param(&bb, path, buf, DMT_TYPE[DMT_UNLONG], 0);

		snprintf(path, sizeof(path), "DisassociationLinkStats.%d.PacketsSent", i + 1);
		snprintf(buf, sizeof(buf), "%"PRIu64, e->affsta[i].tx_packets); /* Flawfinder: ignore */
		fill_blob_param(&bb, path, buf, DMT_TYPE[DMT_UNLONG], 0);

		snprintf(path, sizeof(path), "DisassociationLinkStats.%d.PacketsReceived", i + 1);
		snprintf(buf, sizeof(buf), "%"PRIu64, e->affsta[i].rx_packets); /* Flawfinder: ignore */
		fill_blob_param(&bb, path, buf, DMT_TYPE[DMT_UNLONG], 0);

		snprintf(path, sizeof(path), "DisassociationLinkStats.%d.ErrorsSent", i + 1);
		snprintf(buf, sizeof(buf), "%"PRIu64, e->affsta[i].tx_errors); /* Flawfinder: ignore */
		fill_blob_param(&bb, path, buf, DMT_TYPE[DMT_UNLONG], 0);
	}

	blobmsg_close_array(&bb, arr);

	BBFDM_UBUS_SEND_EVENT("bbfdm.event", bb.head);

	blob_buf_free(&bb);
}

/*************************************************************
* Init, Clean & Sync Module
**************************************************************/
int init_wifi_dataelement_module(void *data)
{
	struct dmctx bbf_ctx = {0};

	bbfdm_create_empty_file("/etc/bbfdm/dmmap/WiFi");

	bbf_ctx_init(&bbf_ctx, NULL);
	dmmap_synchronize_WiFi_DataElements_ProvisionedDPP(&bbf_ctx);
	dmuci_commit_package_bbfdm("WiFi");
	bbf_ctx_clean(&bbf_ctx);

	init_dev_list();

	return 0;
}

int clean_wifi_dataelement_module(void *data)
{
	free_dev_list();
	return 0;
}

/**********************************************************************************************************************************
*                                            OBJ & LEAF DEFINITION
***********************************************************************************************************************************/
/* *** Device.WiFi.DataElements.Network.Device.{i}.Default8021Q.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceDefault8021QParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Enable", &DMWRITE, DMT_BOOL, get_WiFiDataElementsNetworkDeviceDefault8021Q_Enable, set_WiFiDataElementsNetworkDeviceDefault8021Q_Enable, BBFDM_BOTH},
{"PrimaryVID", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceDefault8021Q_PrimaryVID, set_WiFiDataElementsNetworkDeviceDefault8021Q_PrimaryVID, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"DefaultPCP", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceDefault8021Q_DefaultPCP, set_WiFiDataElementsNetworkDeviceDefault8021Q_DefaultPCP, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.SSIDtoVIDMapping.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceSSIDtoVIDMappingParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"SSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceSSIDtoVIDMapping_SSID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"VID", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceSSIDtoVIDMapping_VID, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.CACStatus.{i}.CACAvailableChannel.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceCACStatusCACAvailableChannelParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"OpClass", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACAvailableChannel_OpClass, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Channel", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACAvailableChannel_Channel, NULL, BBFDM_BOTH},
{"Minutes", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACAvailableChannel_Minutes, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.CACStatus.{i}.CACNonOccupancyChannel.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannelParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"OpClass", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannel_OpClass, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Channel", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannel_Channel, NULL, BBFDM_BOTH},
{"Seconds", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannel_Seconds, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.CACStatus.{i}.CACActiveChannel.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceCACStatusCACActiveChannelParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"OpClass", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACActiveChannel_OpClass, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Channel", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACActiveChannel_Channel, NULL, BBFDM_BOTH},
{"Countdown", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatusCACActiveChannel_Countdown, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.CACStatus.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceCACStatusObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"CACAvailableChannel", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceCACStatusCACAvailableChannelInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceCACStatusCACAvailableChannelParams, NULL, BBFDM_BOTH, NULL},
{"CACNonOccupancyChannel", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannelInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceCACStatusCACNonOccupancyChannelParams, NULL, BBFDM_BOTH, NULL},
{"CACActiveChannel", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceCACStatusCACActiveChannelInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceCACStatusCACActiveChannelParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceCACStatusParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceCACStatus_TimeStamp, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"CACAvailableChannelNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatus_CACAvailableChannelNumberOfEntries, NULL, BBFDM_BOTH},
{"CACNonOccupancyChannelNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatus_CACNonOccupancyChannelNumberOfEntries, NULL, BBFDM_BOTH},
{"CACActiveChannelNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceCACStatus_CACActiveChannelNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.IEEE1905Security.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceIEEE1905SecurityParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"OnboardingProtocol", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceIEEE1905Security_OnboardingProtocol, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"IntegrityAlgorithm", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceIEEE1905Security_IntegrityAlgorithm, NULL, BBFDM_BOTH},
{"EncryptionAlgorithm", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceIEEE1905Security_EncryptionAlgorithm, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.BackhaulDown.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceBackhaulDownParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"BackhaulDownALID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceBackhaulDown_BackhaulDownALID, NULL, BBFDM_BOTH},
{"BackhaulDownMACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceBackhaulDown_BackhaulDownMACAddress, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.MultiAPDevice.Backhaul.CurrentOperatingClassProfile.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfileParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"Class", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfile_Class, NULL, BBFDM_BOTH},
{"Channel", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfile_Channel, NULL, BBFDM_BOTH},
{"TxPower", &DMREAD, DMT_INT, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfile_TxPower, NULL, BBFDM_BOTH},
{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfile_TimeStamp, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.MultiAPDevice.Backhaul.Stats. *** */
DMLEAF tWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStatsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"BytesSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_BytesSent, NULL, BBFDM_BOTH},
{"BytesReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_BytesReceived, NULL, BBFDM_BOTH},
{"PacketsSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_PacketsSent, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_PacketsReceived, NULL, BBFDM_BOTH},
{"ErrorsSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_ErrorsSent, NULL, BBFDM_BOTH},
{"ErrorsReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_ErrorsReceived, NULL, BBFDM_BOTH},
//{"LinkUtilization", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_LinkUtilization, NULL, BBFDM_BOTH},
{"SignalStrength", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_SignalStrength, NULL, BBFDM_BOTH},
{"LastDataDownlinkRate", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_LastDataDownlinkRate, NULL, BBFDM_BOTH},
{"LastDataUplinkRate", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_LastDataUplinkRate, NULL, BBFDM_BOTH},
{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStats_TimeStamp, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.MultiAPDevice.Backhaul. *** */
DMOBJ tWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"CurrentOperatingClassProfile", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfileInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulCurrentOperatingClassProfileParams, NULL, BBFDM_BOTH, NULL},
{"Stats", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulStatsParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
//{"CurrentOperatingClassProfileNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaul_CurrentOperatingClassProfileNumberOfEntries, NULL, BBFDM_BOTH},
{"SteerWiFiBackhaul()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaul_SteerWiFiBackhaul, operate_WiFiDataElementsNetworkDeviceMultiAPDeviceBackhaul_SteerWiFiBackhaul, BBFDM_USP},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.MultiAPDevice. *** */
DMOBJ tWiFiDataElementsNetworkDeviceMultiAPDeviceObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Backhaul", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulObj, tWiFiDataElementsNetworkDeviceMultiAPDeviceBackhaulParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceMultiAPDeviceParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"LastContactTime", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceMultiAPDevice_LastContactTime, NULL, BBFDM_BOTH},
{"AssocIEEE1905DeviceRef", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceMultiAPDevice_AssocIEEE1905DeviceRef, NULL, BBFDM_BOTH, DM_FLAG_REFERENCE},
{"EasyMeshAgentOperationMode", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceMultiAPDevice_EasyMeshAgentOperationMode, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}.ChannelScan.{i}.NeighborBSS.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSSParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"BSSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_BSSID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"SSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_SSID, NULL, BBFDM_BOTH},
{"SignalStrength", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_SignalStrength, NULL, BBFDM_BOTH},
{"ChannelBandwidth", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_ChannelBandwidth, NULL, BBFDM_BOTH},
{"ChannelUtilization", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_ChannelUtilization, NULL, BBFDM_BOTH},
{"StationCount", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_StationCount, NULL, BBFDM_BOTH},
{"MLDMACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_MLDMACAddress, NULL, BBFDM_BOTH},
{"ReportingBSSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_ReportingBSSID, NULL, BBFDM_BOTH},
{"MultiBSSID", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_MultiBSSID, NULL, BBFDM_BOTH},
{"BSSLoadElementPresent", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_BSSLoadElementPresent, NULL, BBFDM_BOTH},
{"BSSColor", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSS_BSSColor, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}.ChannelScan.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"NeighborBSS", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSSInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanNeighborBSSParams, NULL, BBFDM_BOTH},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Channel", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_Channel, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_TimeStamp, NULL, BBFDM_BOTH},
{"Utilization", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_Utilization, NULL, BBFDM_BOTH},
{"Noise", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_Noise, NULL, BBFDM_BOTH},
{"ScanStatus", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_ScanStatus, NULL, BBFDM_BOTH},
{"NeighborBSSNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScan_NeighborBSSNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}.OpClassScan.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"ChannelScan", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanInst, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanObj, tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanChannelScanParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"OperatingClass", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScan_OperatingClass, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"ChannelScanNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResultOpClassScan_ChannelScanNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanResult.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioScanResultObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"OpClassScan", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanInst, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanObj, tWiFiDataElementsNetworkDeviceRadioScanResultOpClassScanParams, NULL, BBFDM_BOTH},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioScanResultParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceRadioScanResult_TimeStamp, NULL, BBFDM_BOTH},
{"OpClassScanNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanResult_OpClassScanNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BackhaulSta. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioBackhaulStaParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"MACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBackhaulSta_MACAddress, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanCapability.OpClassChannels.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioScanCapabilityOpClassChannelsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"OpClass", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanCapabilityOpClassChannels_OpClass, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"ChannelList", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioScanCapabilityOpClassChannels_ChannelList, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.ScanCapability. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioScanCapabilityObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"OpClassChannels", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioScanCapabilityOpClassChannelsInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioScanCapabilityOpClassChannelsParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioScanCapabilityParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"OnBootOnly", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioScanCapability_OnBootOnly, NULL, BBFDM_BOTH},
{"Impact", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanCapability_Impact, NULL, BBFDM_BOTH},
{"MinimumInterval", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanCapability_MinimumInterval, NULL, BBFDM_BOTH},
{"OpClassChannelsNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioScanCapability_OpClassChannelsNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.CACCapability.CACMethod.{i}.OpClassChannels.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodOpClassChannelsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"OpClass", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodOpClassChannels_OpClass, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"ChannelList", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodOpClassChannels_ChannelList, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.CACCapability.CACMethod.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"OpClassChannels", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodOpClassChannelsInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodOpClassChannelsParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Method", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethod_Method, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"NumberOfSeconds", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethod_NumberOfSeconds, NULL, BBFDM_BOTH},
{"OpClassChannelsNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethod_OpClassChannelsNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.CACCapability. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioCACCapabilityObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"CACMethod", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodInst, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodObj, tWiFiDataElementsNetworkDeviceRadioCACCapabilityCACMethodParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioCACCapabilityParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"CACMethodNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCACCapability_CACMethodNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi6APRole. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRoleParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"HE160", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_HE160, NULL, BBFDM_BOTH},
{"HE8080", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_HE8080, NULL, BBFDM_BOTH},
{"MCSNSS", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MCSNSS, NULL, BBFDM_BOTH},
{"SUBeamformer", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_SUBeamformer, NULL, BBFDM_BOTH},
{"SUBeamformee", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_SUBeamformee, NULL, BBFDM_BOTH},
{"MUBeamformer", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MUBeamformer, NULL, BBFDM_BOTH},
{"Beamformee80orLess", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_Beamformee80orLess, NULL, BBFDM_BOTH},
{"BeamformeeAbove80", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_BeamformeeAbove80, NULL, BBFDM_BOTH},
{"ULMUMIMO", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_ULMUMIMO, NULL, BBFDM_BOTH},
{"ULOFDMA", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_ULOFDMA, NULL, BBFDM_BOTH},
{"DLOFDMA", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_DLOFDMA, NULL, BBFDM_BOTH},
{"MaxDLMUMIMO", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MaxDLMUMIMO, NULL, BBFDM_BOTH},
{"MaxULMUMIMO", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MaxULMUMIMO, NULL, BBFDM_BOTH},
{"MaxDLOFDMA", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MaxDLOFDMA, NULL, BBFDM_BOTH},
{"MaxULOFDMA", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MaxULOFDMA, NULL, BBFDM_BOTH},
{"RTS", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_RTS, NULL, BBFDM_BOTH},
{"MURTS", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MURTS, NULL, BBFDM_BOTH},
{"MultiBSSID", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MultiBSSID, NULL, BBFDM_BOTH},
{"MUEDCA", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_MUEDCA, NULL, BBFDM_BOTH},
{"TWTRequestor", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_TWTRequestor, NULL, BBFDM_BOTH},
{"TWTResponder", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_TWTResponder, NULL, BBFDM_BOTH},
{"SpatialReuse", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_SpatialReuse, NULL, BBFDM_BOTH},
{"AnticipatedChannelUsage", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRole_AnticipatedChannelUsage, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi6bSTARole. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARoleParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"HE160", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_HE160, NULL, BBFDM_BOTH},
{"HE8080", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_HE8080, NULL, BBFDM_BOTH},
{"MCSNSS", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MCSNSS, NULL, BBFDM_BOTH},
{"SUBeamformer", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_SUBeamformer, NULL, BBFDM_BOTH},
{"SUBeamformee", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_SUBeamformee, NULL, BBFDM_BOTH},
{"MUBeamformer", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MUBeamformer, NULL, BBFDM_BOTH},
{"Beamformee80orLess", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_Beamformee80orLess, NULL, BBFDM_BOTH},
{"BeamformeeAbove80", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_BeamformeeAbove80, NULL, BBFDM_BOTH},
{"ULMUMIMO", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_ULMUMIMO, NULL, BBFDM_BOTH},
{"ULOFDMA", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_ULOFDMA, NULL, BBFDM_BOTH},
{"DLOFDMA", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_DLOFDMA, NULL, BBFDM_BOTH},
{"MaxDLMUMIMO", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MaxDLMUMIMO, NULL, BBFDM_BOTH},
{"MaxULMUMIMO", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MaxULMUMIMO, NULL, BBFDM_BOTH},
{"MaxDLOFDMA", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MaxDLOFDMA, NULL, BBFDM_BOTH},
{"MaxULOFDMA", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MaxULOFDMA, NULL, BBFDM_BOTH},
{"RTS", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_RTS, NULL, BBFDM_BOTH},
{"MURTS", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MURTS, NULL, BBFDM_BOTH},
{"MultiBSSID", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MultiBSSID, NULL, BBFDM_BOTH},
{"MUEDCA", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_MUEDCA, NULL, BBFDM_BOTH},
{"TWTRequestor", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_TWTRequestor, NULL, BBFDM_BOTH},
{"TWTResponder", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_TWTResponder, NULL, BBFDM_BOTH},
{"SpatialReuse", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_SpatialReuse, NULL, BBFDM_BOTH},
{"AnticipatedChannelUsage", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARole_AnticipatedChannelUsage, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7APRole.EMLMRFreqSeparation.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLMRFreqSeparationParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLMRFreqSeparation_RUID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"FreqSeparation", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLMRFreqSeparation_FreqSeparation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7APRole.EMLSRFreqSeparation.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLSRFreqSeparationParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLSRFreqSeparation_RUID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"FreqSeparation", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLSRFreqSeparation_FreqSeparation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7APRole.STRFreqSeparation.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleSTRFreqSeparationParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleSTRFreqSeparation_RUID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"FreqSeparation", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleSTRFreqSeparation_FreqSeparation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7APRole.NSTRFreqSeparation.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleNSTRFreqSeparationParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
//{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleNSTRFreqSeparation_RUID, NULL, BBFDM_BOTH},
//{"FreqSeparation", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleNSTRFreqSeparation_FreqSeparation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7APRole. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys */
{"EMLMRFreqSeparation", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLMRFreqSeparationInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLMRFreqSeparationParams, NULL, BBFDM_BOTH, NULL},
{"EMLSRFreqSeparation", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLSRFreqSeparationInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleEMLSRFreqSeparationParams, NULL, BBFDM_BOTH, NULL},
{"STRFreqSeparation", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleSTRFreqSeparationInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleSTRFreqSeparationParams, NULL, BBFDM_BOTH, NULL},
//{"NSTRFreqSeparation", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleNSTRFreqSeparationInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleNSTRFreqSeparationParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"EMLMRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_EMLMRSupport, NULL, BBFDM_BOTH},
{"EMLSRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_EMLSRSupport, NULL, BBFDM_BOTH},
{"STRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_STRSupport, NULL, BBFDM_BOTH},
{"NSTRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_NSTRSupport, NULL, BBFDM_BOTH},
{"TIDLinkMapNegotiation", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_TIDLinkMapNegotiation, NULL, BBFDM_BOTH},
{"EMLMRFreqSeparationNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_EMLMRFreqSeparationNumberOfEntries, NULL, BBFDM_BOTH},
{"EMLSRFreqSeparationNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_EMLSRFreqSeparationNumberOfEntries, NULL, BBFDM_BOTH},
{"STRFreqSeparationNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_STRFreqSeparationNumberOfEntries, NULL, BBFDM_BOTH},
//{"NSTRFreqSeparationNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRole_NSTRFreqSeparationNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7bSTARole.EMLMRFreqSeparation.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLMRFreqSeparationParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLMRFreqSeparation_RUID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"FreqSeparation", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLMRFreqSeparation_FreqSeparation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7bSTARole.EMLSRFreqSeparation.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLSRFreqSeparationParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLSRFreqSeparation_RUID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"FreqSeparation", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLSRFreqSeparation_FreqSeparation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7bSTARole.STRFreqSeparation.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleSTRFreqSeparationParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleSTRFreqSeparation_RUID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"FreqSeparation", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleSTRFreqSeparation_FreqSeparation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7bSTARole.NSTRFreqSeparation.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleNSTRFreqSeparationParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
//{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleNSTRFreqSeparation_RUID, NULL, BBFDM_BOTH},
//{"FreqSeparation", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleNSTRFreqSeparation_FreqSeparation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.WiFi7bSTARole. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys */
{"EMLMRFreqSeparation", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLMRFreqSeparationInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLMRFreqSeparationParams, NULL, BBFDM_BOTH, NULL},
{"EMLSRFreqSeparation", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLSRFreqSeparationInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleEMLSRFreqSeparationParams, NULL, BBFDM_BOTH, NULL},
{"STRFreqSeparation", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleSTRFreqSeparationInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleSTRFreqSeparationParams, NULL, BBFDM_BOTH, NULL},
//{"NSTRFreqSeparation", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleNSTRFreqSeparationInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleNSTRFreqSeparationParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"EMLMRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_EMLMRSupport, NULL, BBFDM_BOTH},
{"EMLSRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_EMLSRSupport, NULL, BBFDM_BOTH},
{"STRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_STRSupport, NULL, BBFDM_BOTH},
{"NSTRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_NSTRSupport, NULL, BBFDM_BOTH},
{"TIDLinkMapNegotiation", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_TIDLinkMapNegotiation, NULL, BBFDM_BOTH},
{"EMLMRFreqSeparationNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_EMLMRFreqSeparationNumberOfEntries, NULL, BBFDM_BOTH},
{"EMLSRFreqSeparationNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_EMLSRFreqSeparationNumberOfEntries, NULL, BBFDM_BOTH},
{"STRFreqSeparationNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_STRFreqSeparationNumberOfEntries, NULL, BBFDM_BOTH},
//{"NSTRFreqSeparationNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARole_NSTRFreqSeparationNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.AKMFrontHaul.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesAKMFrontHaulParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"OUI", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesAKMFrontHaul_OUI, NULL, BBFDM_BOTH},
{"Type", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesAKMFrontHaul_Type, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.AKMBackhaul.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesAKMBackhaulParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"OUI", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesAKMBackhaul_OUI, NULL, BBFDM_BOTH},
{"Type", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesAKMBackhaul_Type, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities.CapableOperatingClassProfile.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfileParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Class", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfile_Class, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"MaxTxPower", &DMREAD, DMT_INT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfile_MaxTxPower, NULL, BBFDM_BOTH},
{"NonOperable", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfile_NonOperable, NULL, BBFDM_BOTH},
{"NumberOfNonOperChan", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfile_NumberOfNonOperChan, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.Capabilities. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioCapabilitiesObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"WiFi6APRole", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6APRoleParams, NULL, BBFDM_BOTH, NULL},
{"WiFi6bSTARole", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi6bSTARoleParams, NULL, BBFDM_BOTH, NULL},
{"WiFi7APRole", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleObj, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7APRoleParams, NULL, BBFDM_BOTH, NULL},
{"WiFi7bSTARole", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleObj, tWiFiDataElementsNetworkDeviceRadioCapabilitiesWiFi7bSTARoleParams, NULL, BBFDM_BOTH, NULL},
{"AKMFrontHaul", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesAKMFrontHaulInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesAKMFrontHaulParams, NULL, BBFDM_BOTH, NULL},
{"AKMBackhaul", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesAKMBackhaulInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesAKMBackhaulParams, NULL, BBFDM_BOTH, NULL},
{"CapableOperatingClassProfile", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfileInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesCapableOperatingClassProfileParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioCapabilitiesParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"HTCapabilities", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilities_HTCapabilities, NULL, BBFDM_BOTH},
{"VHTCapabilities", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioCapabilities_VHTCapabilities, NULL, BBFDM_BOTH},
{"CapableOperatingClassProfileNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilities_CapableOperatingClassProfileNumberOfEntries, NULL, BBFDM_BOTH},
{"AKMFrontHaulNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilities_AKMFrontHaulNumberOfEntries, NULL, BBFDM_BOTH},
{"AKMBackhaulNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCapabilities_AKMBackhaulNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.CurrentOperatingClassProfile.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfileParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Class", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_Class, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Channel", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_Channel, NULL, BBFDM_BOTH},
{"TxPower", &DMREAD, DMT_INT, get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_TxPower, NULL, BBFDM_BOTH},
// {"TransmitPowerLimit", &DMREAD, DMT_INT, get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_TransmitPowerLimit, NULL, BBFDM_BOTH},
{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfile_TimeStamp, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.OpClassPreference.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioOpClassPreferenceParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"OpClass", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioOpClassPreference_OpClass, NULL, BBFDM_BOTH},
{"ChannelList", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioOpClassPreference_ChannelList, NULL, BBFDM_BOTH},
{"Preference", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioOpClassPreference_Preference, NULL, BBFDM_BOTH},
{"ReasonCode", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioOpClassPreference_ReasonCode, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.MultiAPSTA.SteeringSummaryStats. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStatsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"NoCandidateAPFailures", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_NoCandidateAPFailures, NULL, BBFDM_BOTH},
{"BlacklistAttempts", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BlacklistAttempts, NULL, BBFDM_BOTH},
{"BlacklistSuccesses", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BlacklistSuccesses, NULL, BBFDM_BOTH},
{"BlacklistFailures", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BlacklistFailures, NULL, BBFDM_BOTH},
{"BTMAttempts", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BTMAttempts, NULL, BBFDM_BOTH},
{"BTMSuccesses", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BTMSuccesses, NULL, BBFDM_BOTH},
{"BTMFailures", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BTMFailures, NULL, BBFDM_BOTH},
{"BTMQueryResponses", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_BTMQueryResponses, NULL, BBFDM_BOTH},
{"LastSteerTime", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStats_LastSteerTime, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.MultiAPSTA.SteeringHistory.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistoryParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Time", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_Time, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"APOrigin", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_APOrigin, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"TriggerEvent", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_TriggerEvent, NULL, BBFDM_BOTH},
{"SteeringApproach", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_SteeringApproach, NULL, BBFDM_BOTH},
{"APDestination", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_APDestination, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"SteeringDuration", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistory_SteeringDuration, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.MultiAPSTA. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTAObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"SteeringSummaryStats", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringSummaryStatsParams, NULL, BBFDM_BOTH, NULL},
{"SteeringHistory", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistoryInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTASteeringHistoryParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTAParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"SteeringHistoryNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTA_SteeringHistoryNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}.WiFi6Capabilities. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6CapabilitiesParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"HE160", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_HE160, NULL, BBFDM_BOTH},
{"HE8080", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_HE8080, NULL, BBFDM_BOTH},
{"MCSNSS", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MCSNSS, NULL, BBFDM_BOTH},
{"SUBeamformer", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_SUBeamformer, NULL, BBFDM_BOTH},
{"SUBeamformee", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_SUBeamformee, NULL, BBFDM_BOTH},
{"MUBeamformer", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MUBeamformer, NULL, BBFDM_BOTH},
{"Beamformee80orLess", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_Beamformee80orLess, NULL, BBFDM_BOTH},
{"BeamformeeAbove80", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_BeamformeeAbove80, NULL, BBFDM_BOTH},
{"ULMUMIMO", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_ULMUMIMO, NULL, BBFDM_BOTH},
{"ULOFDMA", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_ULOFDMA, NULL, BBFDM_BOTH},
{"MaxDLMUMIMO", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MaxDLMUMIMO, NULL, BBFDM_BOTH},
{"MaxULMUMIMO", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MaxULMUMIMO, NULL, BBFDM_BOTH},
{"MaxDLOFDMA", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MaxDLOFDMA, NULL, BBFDM_BOTH},
{"MaxULOFDMA", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MaxULOFDMA, NULL, BBFDM_BOTH},
{"RTS", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_RTS, NULL, BBFDM_BOTH},
{"MURTS", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MURTS, NULL, BBFDM_BOTH},
{"MultiBSSID", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MultiBSSID, NULL, BBFDM_BOTH},
{"MUEDCA", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_MUEDCA, NULL, BBFDM_BOTH},
{"TWTRequestor", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_TWTRequestor, NULL, BBFDM_BOTH},
{"TWTResponder", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_TWTResponder, NULL, BBFDM_BOTH},
{"SpatialReuse", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_SpatialReuse, NULL, BBFDM_BOTH},
{"AnticipatedChannelUsage", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6Capabilities_AnticipatedChannelUsage, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}.STA.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioBSSSTAObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"MultiAPSTA", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTAObj, tWiFiDataElementsNetworkDeviceRadioBSSSTAMultiAPSTAParams, NULL, BBFDM_BOTH, NULL},
{"WiFi6Capabilities", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSSTAWiFi6CapabilitiesParams, NULL, BBFDM_BOTH, NULL},
//{"TIDQueueSizes", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioBSSSTATIDQueueSizesInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSSTATIDQueueSizesParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioBSSSTAParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"MACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_MACAddress, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE|DM_FLAG_LINKER},
{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_TimeStamp, NULL, BBFDM_BOTH},
{"HTCapabilities", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_HTCapabilities, NULL, BBFDM_BOTH},
{"VHTCapabilities", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_VHTCapabilities, NULL, BBFDM_BOTH},
{"ClientCapabilities", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_ClientCapabilities, NULL, BBFDM_BOTH},
{"LastDataDownlinkRate", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_LastDataDownlinkRate, NULL, BBFDM_BOTH},
{"LastDataUplinkRate", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_LastDataUplinkRate, NULL, BBFDM_BOTH},
{"UtilizationReceive", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_UtilizationReceive, NULL, BBFDM_BOTH},
{"UtilizationTransmit", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_UtilizationTransmit, NULL, BBFDM_BOTH},
{"EstMACDataRateDownlink", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_EstMACDataRateDownlink, NULL, BBFDM_BOTH},
{"EstMACDataRateUplink", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_EstMACDataRateUplink, NULL, BBFDM_BOTH},
{"SignalStrength", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_SignalStrength, NULL, BBFDM_BOTH},
{"LastConnectTime", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_LastConnectTime, NULL, BBFDM_BOTH},
{"BytesSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_BytesSent, NULL, BBFDM_BOTH},
{"BytesReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_BytesReceived, NULL, BBFDM_BOTH},
{"PacketsSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_PacketsSent, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_PacketsReceived, NULL, BBFDM_BOTH},
{"ErrorsSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_ErrorsSent, NULL, BBFDM_BOTH},
{"ErrorsReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_ErrorsReceived, NULL, BBFDM_BOTH},
{"RetransCount", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_RetransCount, NULL, BBFDM_BOTH},
{"MeasurementReport", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_MeasurementReport, NULL, BBFDM_BOTH},
{"NumberOfMeasureReports", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_NumberOfMeasureReports, NULL, BBFDM_BOTH},
{"IPV4Address", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_IPV4Address, NULL, BBFDM_BOTH},
{"IPV6Address", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_IPV6Address, NULL, BBFDM_BOTH},
{"Hostname", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_Hostname, NULL, BBFDM_BOTH},
{"ClientSteer()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDeviceRadioBSSSTA_ClientSteer, operate_WiFiDataElementsNetworkDeviceRadioBSSSTA_ClientSteer, BBFDM_USP},
// {"CellularDataPreference", &DMWRITE, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_CellularDataPreference, set_WiFiDataElementsNetworkDeviceRadioBSSSTA_CellularDataPreference, BBFDM_BOTH},
// {"ReAssociationDelay", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_ReAssociationDelay, set_WiFiDataElementsNetworkDeviceRadioBSSSTA_ReAssociationDelay, BBFDM_BOTH},
//{"TIDQueueSizesNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_TIDQueueSizesNumberOfEntries, NULL, BBFDM_BOTH},
#ifdef DECOLLECTOR_VENDOR_EXTENSIONS
{CUSTOM_PREFIX"QoERating", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSSSTA_QoERating, NULL, BBFDM_BOTH},
#endif
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.BSS.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioBSSObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
// {"QMDescriptor", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioBSSQMDescriptorInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSQMDescriptorParams, NULL, BBFDM_BOTH, NULL},
// {"MultiAPSteering", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSMultiAPSteeringParams, NULL, BBFDM_BOTH, NULL},
{"STA", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioBSSSTAInst, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSSTAObj, tWiFiDataElementsNetworkDeviceRadioBSSSTAParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioBSSParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"BSSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSS_BSSID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"SSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSS_SSID, NULL, BBFDM_BOTH},
{"Enabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_Enabled, NULL, BBFDM_BOTH},
{"LastChange", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSS_LastChange, NULL, BBFDM_BOTH},
{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetworkDeviceRadioBSS_TimeStamp, NULL, BBFDM_BOTH},
{"UnicastBytesSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSS_UnicastBytesSent, NULL, BBFDM_BOTH},
{"UnicastBytesReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSS_UnicastBytesReceived, NULL, BBFDM_BOTH},
{"MulticastBytesSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSS_MulticastBytesSent, NULL, BBFDM_BOTH},
{"MulticastBytesReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSS_MulticastBytesReceived, NULL, BBFDM_BOTH},
{"BroadcastBytesSent", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSS_BroadcastBytesSent, NULL, BBFDM_BOTH},
{"BroadcastBytesReceived", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceRadioBSS_BroadcastBytesReceived, NULL, BBFDM_BOTH},
{"ByteCounterUnits", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSS_ByteCounterUnits, NULL, BBFDM_BOTH},
// {"Profile1bSTAsDisallowed", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_Profile1bSTAsDisallowed, NULL, BBFDM_BOTH},
// {"Profile2bSTAsDisallowed", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_Profile2bSTAsDisallowed, NULL, BBFDM_BOTH},
// {"AssociationAllowanceStatus", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSS_AssociationAllowanceStatus, NULL, BBFDM_BOTH},
{"EstServiceParametersBE", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioBSS_EstServiceParametersBE, NULL, BBFDM_BOTH},
{"EstServiceParametersBK", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioBSS_EstServiceParametersBK, NULL, BBFDM_BOTH},
{"EstServiceParametersVI", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioBSS_EstServiceParametersVI, NULL, BBFDM_BOTH},
{"EstServiceParametersVO", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadioBSS_EstServiceParametersVO, NULL, BBFDM_BOTH},
{"BackhaulUse", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_BackhaulUse, NULL, BBFDM_BOTH},
{"FronthaulUse", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_FronthaulUse, NULL, BBFDM_BOTH},
{"R1disallowed", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_R1disallowed, NULL, BBFDM_BOTH},
{"R2disallowed", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_R2disallowed, NULL, BBFDM_BOTH},
{"MultiBSSID", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_MultiBSSID, NULL, BBFDM_BOTH},
{"TransmittedBSSID", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadioBSS_TransmittedBSSID, NULL, BBFDM_BOTH},
// {"FronthaulAKMsAllowed", &DMWRITE, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSS_FronthaulAKMsAllowed, set_WiFiDataElementsNetworkDeviceRadioBSS_FronthaulAKMsAllowed, BBFDM_BOTH},
// {"BackhaulAKMsAllowed", &DMWRITE, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioBSS_BackhaulAKMsAllowed, set_WiFiDataElementsNetworkDeviceRadioBSS_BackhaulAKMsAllowed, BBFDM_BOTH},
{"STANumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSS_STANumberOfEntries, NULL, BBFDM_BOTH},
// {"QMDescriptorNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioBSS_QMDescriptorNumberOfEntries, NULL, BBFDM_BOTH},
//{"SetQMDescriptors()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDeviceRadioBSS_SetQMDescriptors, operate_WiFiDataElementsNetworkDeviceRadioBSS_SetQMDescriptors, BBFDM_USP},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}.UnassociatedSTA.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceRadioUnassociatedSTAParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"MACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadioUnassociatedSTA_MACAddress, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"SignalStrength", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadioUnassociatedSTA_SignalStrength, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.Radio.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceRadioObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"ScanResult", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioScanResultInst, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioScanResultObj, tWiFiDataElementsNetworkDeviceRadioScanResultParams, NULL, BBFDM_BOTH, NULL},
{"BackhaulSta", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBackhaulStaParams, NULL, BBFDM_BOTH, NULL},
{"ScanCapability", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioScanCapabilityObj, tWiFiDataElementsNetworkDeviceRadioScanCapabilityParams, NULL, BBFDM_BOTH, NULL},
{"CACCapability", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCACCapabilityObj, tWiFiDataElementsNetworkDeviceRadioCACCapabilityParams, NULL, BBFDM_BOTH, NULL},
{"Capabilities", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCapabilitiesObj, tWiFiDataElementsNetworkDeviceRadioCapabilitiesParams, NULL, BBFDM_BOTH, NULL},
{"CurrentOperatingClassProfile", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfileInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioCurrentOperatingClassProfileParams, NULL, BBFDM_BOTH, NULL},
//{"DisAllowedOpClassChannels", &DMWRITE, addObjWiFiDataElementsNetworkDeviceRadioDisAllowedOpClassChannels, delObjWiFiDataElementsNetworkDeviceRadioDisAllowedOpClassChannels, NULL, browseWiFiDataElementsNetworkDeviceRadioDisAllowedOpClassChannelsInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioDisAllowedOpClassChannelsParams, NULL, BBFDM_BOTH, NULL},
{"OpClassPreference", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioOpClassPreferenceInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioOpClassPreferenceParams, NULL, BBFDM_BOTH, NULL},
//{"SpatialReuse", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioSpatialReuseParams, NULL, BBFDM_BOTH, NULL},
{"BSS", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioBSSInst, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioBSSObj, tWiFiDataElementsNetworkDeviceRadioBSSParams, NULL, BBFDM_BOTH, NULL},
{"UnassociatedSTA", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioUnassociatedSTAInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioUnassociatedSTAParams, NULL, BBFDM_BOTH, NULL},
//{"MultiAPRadio", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioMultiAPRadioParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceRadioParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"ID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceRadio_ID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Enabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadio_Enabled, NULL, BBFDM_BOTH},
{"Noise", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_Noise, NULL, BBFDM_BOTH},
{"Utilization", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_Utilization, NULL, BBFDM_BOTH},
{"Transmit", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_Transmit, NULL, BBFDM_BOTH},
{"ReceiveSelf", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_ReceiveSelf, NULL, BBFDM_BOTH},
{"ReceiveOther", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_ReceiveOther, NULL, BBFDM_BOTH},
{"TrafficSeparationCombinedFronthaul", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadio_TrafficSeparationCombinedFronthaul, NULL, BBFDM_BOTH},
{"TrafficSeparationCombinedBackhaul", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadio_TrafficSeparationCombinedBackhaul, NULL, BBFDM_BOTH},
{"SteeringPolicy", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_SteeringPolicy, set_WiFiDataElementsNetworkDeviceRadio_SteeringPolicy, BBFDM_BOTH},
{"ChannelUtilizationThreshold", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_ChannelUtilizationThreshold, set_WiFiDataElementsNetworkDeviceRadio_ChannelUtilizationThreshold, BBFDM_BOTH},
{"RCPISteeringThreshold", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_RCPISteeringThreshold, set_WiFiDataElementsNetworkDeviceRadio_RCPISteeringThreshold, BBFDM_BOTH},
{"STAReportingRCPIThreshold", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_STAReportingRCPIThreshold, set_WiFiDataElementsNetworkDeviceRadio_STAReportingRCPIThreshold, BBFDM_BOTH},
{"STAReportingRCPIHysteresisMarginOverride", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_STAReportingRCPIHysteresisMarginOverride, set_WiFiDataElementsNetworkDeviceRadio_STAReportingRCPIHysteresisMarginOverride, BBFDM_BOTH},
{"ChannelUtilizationReportingThreshold", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_ChannelUtilizationReportingThreshold, set_WiFiDataElementsNetworkDeviceRadio_ChannelUtilizationReportingThreshold, BBFDM_BOTH},
{"AssociatedSTATrafficStatsInclusionPolicy", &DMWRITE, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadio_AssociatedSTATrafficStatsInclusionPolicy, set_WiFiDataElementsNetworkDeviceRadio_AssociatedSTATrafficStatsInclusionPolicy, BBFDM_BOTH},
{"AssociatedSTALinkMetricsInclusionPolicy", &DMWRITE, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadio_AssociatedSTALinkMetricsInclusionPolicy, set_WiFiDataElementsNetworkDeviceRadio_AssociatedSTALinkMetricsInclusionPolicy, BBFDM_BOTH},
{"ChipsetVendor", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceRadio_ChipsetVendor, NULL, BBFDM_BOTH},
{"APMetricsWiFi6", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceRadio_APMetricsWiFi6, NULL, BBFDM_BOTH},
{"MaxBSS", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_MaxBSS, NULL, BBFDM_BOTH},
{"CurrentOperatingClassProfileNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_CurrentOperatingClassProfileNumberOfEntries, NULL, BBFDM_BOTH},
{"UnassociatedSTANumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_UnassociatedSTANumberOfEntries, NULL, BBFDM_BOTH},
{"BSSNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_BSSNumberOfEntries, NULL, BBFDM_BOTH},
{"ScanResultNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_ScanResultNumberOfEntries, NULL, BBFDM_BOTH},
{"OpClassPreferenceNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceRadio_OpClassPreferenceNumberOfEntries, NULL, BBFDM_BOTH},
{"ChannelScanRequest()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDeviceRadio_ChannelScanRequest, operate_WiFiDataElementsNetworkDeviceRadio_ChannelScanRequest, BBFDM_USP},
{"ChannelSelectionRequest()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDeviceRadio_ChannelSelectionRequest, operate_WiFiDataElementsNetworkDeviceRadio_ChannelSelectionRequest, BBFDM_USP},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.APMLD.{i}.STAMLD.{i}.WiFi7Capabilities. *** */
DMLEAF tWiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7CapabilitiesParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"EMLMRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_EMLMRSupport, NULL, BBFDM_BOTH},
{"EMLSRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_EMLSRSupport, NULL, BBFDM_BOTH},
{"STRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_STRSupport, NULL, BBFDM_BOTH},
{"NSTRSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_NSTRSupport, NULL, BBFDM_BOTH},
{"TIDLinkMapNegotiation", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7Capabilities_TIDLinkMapNegotiation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.APMLD.{i}.STAMLD.{i}.STAMLDConfig. *** */
DMLEAF tWiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfigParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"EMLMREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_EMLMREnabled, NULL, BBFDM_BOTH},
{"EMLSREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_EMLSREnabled, NULL, BBFDM_BOTH},
{"STREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_STREnabled, NULL, BBFDM_BOTH},
{"NSTREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_NSTREnabled, NULL, BBFDM_BOTH},
{"TIDLinkMapNegotiation", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfig_TIDLinkMapNegotiation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.APMLD.{i}.STAMLD.{i}.AffiliatedSTA.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTAParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"MACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_MACAddress, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"BSSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_BSSID, NULL, BBFDM_BOTH},
{"BytesSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_BytesSent, NULL, BBFDM_BOTH},
{"BytesReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_BytesReceived, NULL, BBFDM_BOTH},
{"PacketsSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_PacketsSent, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_PacketsReceived, NULL, BBFDM_BOTH},
{"ErrorsSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_ErrorsSent, NULL, BBFDM_BOTH},
{"SignalStrength", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_SignalStrength, NULL, BBFDM_BOTH},
{"EstMACDataRateDownlink", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_EstMACDataRateDownlink, NULL, BBFDM_BOTH},
{"EstMACDataRateUplink", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_EstMACDataRateUplink, NULL, BBFDM_BOTH},
{"LastDataDownlinkRate", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_LastDataDownlinkRate, NULL, BBFDM_BOTH},
{"LastDataUplinkRate", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_LastDataUplinkRate, NULL, BBFDM_BOTH},
{"UtilizationReceive", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_UtilizationReceive, NULL, BBFDM_BOTH},
{"UtilizationTransmit", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTA_UtilizationTransmit, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.APMLD.{i}.STAMLD.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceAPMLDSTAMLDObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys */
{"WiFi7Capabilities", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceAPMLDSTAMLDWiFi7CapabilitiesParams, NULL, BBFDM_BOTH, NULL},
{"STAMLDConfig", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceAPMLDSTAMLDSTAMLDConfigParams, NULL, BBFDM_BOTH, NULL},
{"AffiliatedSTA", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTAInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceAPMLDSTAMLDAffiliatedSTAParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceAPMLDSTAMLDParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"MLDMACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_MLDMACAddress, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE|DM_FLAG_LINKER},
{"Hostname", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_Hostname, NULL, BBFDM_BOTH},
{"IPv4Address", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_IPv4Address, NULL, BBFDM_BOTH},
{"IPv6Address", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_IPv6Address, NULL, BBFDM_BOTH},
{"IsbSTA", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_IsbSTA, NULL, BBFDM_BOTH},
{"LastConnectTime", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_LastConnectTime, NULL, BBFDM_BOTH},
{"BytesSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_BytesSent, NULL, BBFDM_BOTH},
{"BytesReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_BytesReceived, NULL, BBFDM_BOTH},
{"PacketsSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_PacketsSent, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_PacketsReceived, NULL, BBFDM_BOTH},
{"ErrorsSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_ErrorsSent, NULL, BBFDM_BOTH},
{"ErrorsReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_ErrorsReceived, NULL, BBFDM_BOTH},
{"RetransCount", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_RetransCount, NULL, BBFDM_BOTH},
{"AffiliatedSTANumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_AffiliatedSTANumberOfEntries, NULL, BBFDM_BOTH},
#ifdef DECOLLECTOR_VENDOR_EXTENSIONS
{CUSTOM_PREFIX"QoERating", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDSTAMLD_QoERating, NULL, BBFDM_BOTH},
#endif
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.APMLD.{i}.APMLDConfig. *** */
DMLEAF tWiFiDataElementsNetworkDeviceAPMLDAPMLDConfigParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"EMLMREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_EMLMREnabled, NULL, BBFDM_BOTH},
{"EMLSREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_EMLSREnabled, NULL, BBFDM_BOTH},
{"STREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_STREnabled, NULL, BBFDM_BOTH},
{"NSTREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_NSTREnabled, NULL, BBFDM_BOTH},
{"TIDLinkMapNegotiation", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDeviceAPMLDAPMLDConfig_TIDLinkMapNegotiation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.APMLD.{i}.LinkToOpClassMap.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceAPMLDLinkToOpClassMapParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"LinkID", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDLinkToOpClassMap_LinkID, NULL, BBFDM_BOTH},
{"OpClass", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDLinkToOpClassMap_OpClass, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.APMLD.{i}.AffiliatedAP.{i}. *** */
DMLEAF tWiFiDataElementsNetworkDeviceAPMLDAffiliatedAPParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"BSSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_BSSID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"LinkID", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_LinkID, NULL, BBFDM_BOTH},
{"RUID", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_RUID, NULL, BBFDM_BOTH},
//{"DisabledSubChannels", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_DisabledSubChannels, NULL, BBFDM_BOTH},
{"PacketsSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_PacketsSent, NULL, BBFDM_BOTH},
{"PacketsReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_PacketsReceived, NULL, BBFDM_BOTH},
{"UnicastBytesSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_UnicastBytesSent, NULL, BBFDM_BOTH},
{"UnicastBytesReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_UnicastBytesReceived, NULL, BBFDM_BOTH},
{"ErrorsSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_ErrorsSent, NULL, BBFDM_BOTH},
{"MulticastBytesSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_MulticastBytesSent, NULL, BBFDM_BOTH},
{"MulticastBytesReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_MulticastBytesReceived, NULL, BBFDM_BOTH},
{"BroadcastBytesSent", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_BroadcastBytesSent, NULL, BBFDM_BOTH},
{"BroadcastBytesReceived", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_BroadcastBytesReceived, NULL, BBFDM_BOTH},
{"EstServiceParametersBE", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_EstServiceParametersBE, NULL, BBFDM_BOTH},
{"EstServiceParametersBK", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_EstServiceParametersBK, NULL, BBFDM_BOTH},
{"EstServiceParametersVI", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_EstServiceParametersVI, NULL, BBFDM_BOTH},
{"EstServiceParametersVO", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDeviceAPMLDAffiliatedAP_EstServiceParametersVO, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.APMLD.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceAPMLDObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys */
{"APMLDConfig", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceAPMLDAPMLDConfigParams, NULL, BBFDM_BOTH, NULL},
{"LinkToOpClassMap", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceAPMLDLinkToOpClassMapInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceAPMLDLinkToOpClassMapParams, NULL, BBFDM_BOTH, NULL},
{"AffiliatedAP", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceAPMLDAffiliatedAPInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceAPMLDAffiliatedAPParams, NULL, BBFDM_BOTH, NULL},
{"STAMLD", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceAPMLDSTAMLDInst, NULL, NULL, tWiFiDataElementsNetworkDeviceAPMLDSTAMLDObj, tWiFiDataElementsNetworkDeviceAPMLDSTAMLDParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceAPMLDParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"MLDMACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDeviceAPMLD_MLDMACAddress, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"AffiliatedAPNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLD_AffiliatedAPNumberOfEntries, NULL, BBFDM_BOTH},
{"STAMLDNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLD_STAMLDNumberOfEntries, NULL, BBFDM_BOTH},
{"LinkToOpClassMapNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDeviceAPMLD_LinkToOpClassMapNumberOfEntries, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.bSTAMLD.bSTAMLDConfig. *** */
DMLEAF tWiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfigParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"EMLMREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_EMLMREnabled, NULL, BBFDM_BOTH},
{"EMLSREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_EMLSREnabled, NULL, BBFDM_BOTH},
{"STREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_STREnabled, NULL, BBFDM_BOTH},
{"NSTREnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_NSTREnabled, NULL, BBFDM_BOTH},
{"TIDLinkMapNegotiation", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfig_TIDLinkMapNegotiation, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}.bSTAMLD. *** */
DMOBJ tWiFiDataElementsNetworkDevicebSTAMLDObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys */
{"bSTAMLDConfig", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDevicebSTAMLDbSTAMLDConfigParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDevicebSTAMLDParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"MLDMACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevicebSTAMLD_MLDMACAddress, NULL, BBFDM_BOTH},
{"BSSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevicebSTAMLD_BSSID, NULL, BBFDM_BOTH},
{"AffiliatedbSTAList", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevicebSTAMLD_AffiliatedbSTAList, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.SSID.{i}. *** */
DMLEAF tWiFiDataElementsNetworkSSIDParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"SSID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkSSID_SSID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE},
{"Band", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkSSID_Band, NULL, BBFDM_BOTH},
{"Enable", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkSSID_Enable, NULL, BBFDM_BOTH},
{"AKMsAllowed", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkSSID_AKMs, NULL, BBFDM_BOTH},
//{"AdvertisementEnabled", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkSSID_AdvEnabled, NULL, BBFDM_BOTH},
//{"MFPConfig", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkSSID_MFP, NULL, BBFDM_BOTH},
{"HaulType", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkSSID_Haul, NULL, BBFDM_BOTH},
{"Type", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkSSID_Type, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.ProvisionedDPP.{i}. *** */
DMLEAF tWiFiDataElementsNetworkProvisionedDPPParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type */
{"Alias", &DMWRITE, DMT_STRING, get_WiFiDataElementsNetworkProvisionedDPP_Alias, set_WiFiDataElementsNetworkProvisionedDPP_Alias, BBFDM_BOTH},
{"DPPURI", &DMWRITE, DMT_STRING, get_WiFiDataElementsNetworkProvisionedDPP_DPPURI, set_WiFiDataElementsNetworkProvisionedDPP_DPPURI, BBFDM_BOTH},
#ifdef DECOLLECTOR_VENDOR_EXTENSIONS
{CUSTOM_PREFIX"Passphrase", &DMWRITE, DMT_STRING, get_WiFiDataElementsNetworkProvisionedDPP_Passphrase, set_WiFiDataElementsNetworkProvisionedDPP_Passphrase, BBFDM_BOTH, DM_FLAG_SECURE},
#endif
{0}
};

/* *** Device.WiFi.DataElements.Network.MultiAPSteeringSummaryStats. *** */
DMLEAF tWiFiDataElementsNetworkMultiAPSteeringSummaryStatsParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"NoCandidateAPFailures", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_NoCandidateAPFailures, NULL, BBFDM_BOTH},
{"BlacklistAttempts", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BlacklistAttempts, NULL, BBFDM_BOTH},
{"BlacklistSuccesses", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BlacklistSuccesses, NULL, BBFDM_BOTH},
{"BlacklistFailures", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BlacklistFailures, NULL, BBFDM_BOTH},
{"BTMAttempts", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BTMAttempts, NULL, BBFDM_BOTH},
{"BTMSuccesses", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BTMSuccesses, NULL, BBFDM_BOTH},
{"BTMFailures", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BTMFailures, NULL, BBFDM_BOTH},
{"BTMQueryResponses", &DMREAD, DMT_UNLONG, get_WiFiDataElementsNetworkMultiAPSteeringSummaryStats_BTMQueryResponses, NULL, BBFDM_BOTH},
{0}
};

/* *** Device.WiFi.DataElements.Network.Device.{i}. *** */
DMOBJ tWiFiDataElementsNetworkDeviceObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Default8021Q", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceDefault8021QInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceDefault8021QParams, NULL, BBFDM_BOTH, NULL},
{"SSIDtoVIDMapping", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceSSIDtoVIDMappingInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceSSIDtoVIDMappingParams, NULL, BBFDM_BOTH, NULL},
{"CACStatus", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceCACStatusInst, NULL, NULL, tWiFiDataElementsNetworkDeviceCACStatusObj, tWiFiDataElementsNetworkDeviceCACStatusParams, NULL, BBFDM_BOTH, NULL},
//{"SPRule", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceSPRuleInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceSPRuleParams, NULL, BBFDM_BOTH, NULL},
{"IEEE1905Security", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceIEEE1905SecurityInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceIEEE1905SecurityParams, NULL, BBFDM_BOTH, NULL},
//{"AnticipatedChannels", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceAnticipatedChannelsInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceAnticipatedChannelsParams, NULL, BBFDM_BOTH, NULL},
//{"AnticipatedChannelUsage", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceAnticipatedChannelUsageInst, NULL, NULL, tWiFiDataElementsNetworkDeviceAnticipatedChannelUsageObj, tWiFiDataElementsNetworkDeviceAnticipatedChannelUsageParams, NULL, BBFDM_BOTH, NULL},
{"BackhaulDown", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceBackhaulDownInst, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceBackhaulDownParams, NULL, BBFDM_BOTH, NULL},
{"MultiAPDevice", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDeviceMultiAPDeviceObj, tWiFiDataElementsNetworkDeviceMultiAPDeviceParams, NULL, BBFDM_BOTH, NULL},
{"Radio", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceRadioInst, NULL, NULL, tWiFiDataElementsNetworkDeviceRadioObj, tWiFiDataElementsNetworkDeviceRadioParams, NULL, BBFDM_BOTH, NULL},
{"APMLD", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceAPMLDInst, NULL, NULL, tWiFiDataElementsNetworkDeviceAPMLDObj, tWiFiDataElementsNetworkDeviceAPMLDParams, NULL, BBFDM_BOTH, NULL},
{"bSTAMLD", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkDevicebSTAMLDObj, tWiFiDataElementsNetworkDevicebSTAMLDParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkDeviceParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"ID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_ID, NULL, BBFDM_BOTH, DM_FLAG_UNIQUE|DM_FLAG_LINKER},
{"MultiAPCapabilities", &DMREAD, DMT_BASE64, get_WiFiDataElementsNetworkDevice_MultiAPCapabilities, NULL, BBFDM_BOTH},
{"CollectionInterval", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_CollectionInterval, NULL, BBFDM_BOTH},
{"ReportUnsuccessfulAssociations", &DMWRITE, DMT_BOOL, get_WiFiDataElementsNetworkDevice_ReportUnsuccessfulAssociations, set_WiFiDataElementsNetworkDevice_ReportUnsuccessfulAssociations, BBFDM_BOTH},
{"APMetricsReportingInterval", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDevice_APMetricsReportingInterval, set_WiFiDataElementsNetworkDevice_APMetricsReportingInterval, BBFDM_BOTH},
{"Manufacturer", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_Manufacturer, NULL, BBFDM_BOTH},
{"SerialNumber", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_SerialNumber, NULL, BBFDM_BOTH},
{"ManufacturerModel", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_ManufacturerModel, NULL, BBFDM_BOTH},
{"SoftwareVersion", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_SoftwareVersion, NULL, BBFDM_BOTH},
{"ExecutionEnv", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_ExecutionEnv, NULL, BBFDM_BOTH},
// {"DSCPMap", &DMREAD, DMT_HEXBIN, get_WiFiDataElementsNetworkDevice_DSCPMap, NULL, BBFDM_BOTH},
{"MaxPrioritizationRules", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_MaxPrioritizationRules, NULL, BBFDM_BOTH},
{"PrioritizationSupport", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevice_PrioritizationSupport, NULL, BBFDM_BOTH},
{"MaxVIDs", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_MaxVIDs, NULL, BBFDM_BOTH},
{"CountryCode", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_CountryCode, NULL, BBFDM_BOTH},
{"LocalSteeringDisallowedSTAList", &DMWRITE, DMT_STRING, get_WiFiDataElementsNetworkDevice_LocalSteeringDisallowedSTAList, set_WiFiDataElementsNetworkDevice_LocalSteeringDisallowedSTAList, BBFDM_BOTH},
{"BTMSteeringDisallowedSTAList", &DMWRITE, DMT_STRING, get_WiFiDataElementsNetworkDevice_BTMSteeringDisallowedSTAList, set_WiFiDataElementsNetworkDevice_BTMSteeringDisallowedSTAList, BBFDM_BOTH},
{"DFSEnable", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevice_DFSEnable, NULL, BBFDM_BOTH},
{"ReportIndependentScans", &DMWRITE, DMT_BOOL, get_WiFiDataElementsNetworkDevice_ReportIndependentScans, set_WiFiDataElementsNetworkDevice_ReportIndependentScans, BBFDM_BOTH},
{"MaxUnsuccessfulAssociationReportingRate", &DMWRITE, DMT_UNINT, get_WiFiDataElementsNetworkDevice_MaxUnsuccessfulAssociationReportingRate, set_WiFiDataElementsNetworkDevice_MaxUnsuccessfulAssociationReportingRate, BBFDM_BOTH},
{"STASteeringState", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevice_STASteeringState, NULL, BBFDM_BOTH},
{"CoordinatedCACAllowed", &DMWRITE, DMT_BOOL, get_WiFiDataElementsNetworkDevice_CoordinatedCACAllowed, set_WiFiDataElementsNetworkDevice_CoordinatedCACAllowed, BBFDM_BOTH},
{"TrafficSeparationAllowed", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevice_TrafficSeparationAllowed, NULL, BBFDM_BOTH},
// {"ServicePrioritizationAllowed", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevice_ServicePrioritizationAllowed, NULL, BBFDM_BOTH},
{"ControllerOperationMode", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_ControllerOperationMode, NULL, BBFDM_BOTH},
{"BackhaulMACAddress", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_BackhaulMACAddress, NULL, BBFDM_BOTH},
{"BackhaulALID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_BackhaulALID, NULL, BBFDM_BOTH},
{"BackhaulMediaType", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_BackhaulMediaType, NULL, BBFDM_BOTH},
{"BackhaulPHYRate", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_BackhaulPHYRate, NULL, BBFDM_BOTH},
{"TrafficSeparationCapability", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevice_TrafficSeparationCapability, NULL, BBFDM_BOTH},
{"EasyConnectCapability", &DMREAD, DMT_BOOL, get_WiFiDataElementsNetworkDevice_EasyConnectCapability, NULL, BBFDM_BOTH},
{"RadioNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_RadioNumberOfEntries, NULL, BBFDM_BOTH},
{"Default8021QNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_Default8021QNumberOfEntries, NULL, BBFDM_BOTH},
{"SSIDtoVIDMappingNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_SSIDtoVIDMappingNumberOfEntries, NULL, BBFDM_BOTH},
{"CACStatusNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_CACStatusNumberOfEntries, NULL, BBFDM_BOTH},
{"IEEE1905SecurityNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_IEEE1905SecurityNumberOfEntries, NULL, BBFDM_BOTH},
//{"SPRuleNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_SPRuleNumberOfEntries, NULL, BBFDM_BOTH},
//{"AnticipatedChannelsNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_AnticipatedChannelsNumberOfEntries, NULL, BBFDM_BOTH},
//{"AnticipatedChannelUsageNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_AnticipatedChannelUsageNumberOfEntries, NULL, BBFDM_BOTH},
{"MaxNumMLDs", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_MaxNumMLDs, NULL, BBFDM_BOTH},
{"APMLDMaxLinks", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_APMLDMaxLinks, NULL, BBFDM_BOTH},
{"bSTAMLDMaxLinks", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_bSTAMLDMaxLinks, NULL, BBFDM_BOTH},
{"TIDLinkMapCapability", &DMREAD, DMT_STRING, get_WiFiDataElementsNetworkDevice_TIDLinkMapCapability, NULL, BBFDM_BOTH},
{"APMLDNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_APMLDNumberOfEntries, NULL, BBFDM_BOTH},
{"BackhaulDownNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetworkDevice_BackhaulDownNumberOfEntries, NULL, BBFDM_BOTH},
{"SetSTASteeringState()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDevice_SetSTASteeringState, operate_WiFiDataElementsNetworkDevice_SetSTASteeringState, BBFDM_USP},
{"InitiateWPSPBC()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDevice_InitiateWPSPBC, operate_WiFiDataElementsNetworkDevice_InitiateWPSPBC, BBFDM_USP},
{"RefreshAPMetrics()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDevice_RefreshAPMetrics, operate_WiFiDataElementsNetworkDevice_RefreshAPMetrics, BBFDM_USP},
//{"SetDFSState()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDevice_SetDFSState, operate_WiFiDataElementsNetworkDevice_SetDFSState, BBFDM_USP},
//{"SetAnticipatedChannelPreference()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetworkDevice_SetAnticipatedChannelPreference, operate_WiFiDataElementsNetworkDevice_SetAnticipatedChannelPreference, BBFDM_USP},
{"BackhaulChange!", &DMREAD, DMT_EVENT, get_event_args_WiFiDataElementsNetworkDevice_BackhaulChange, event_WiFiDataElementsNetworkDevice_BackhaulChange, BBFDM_USP},
{0}
};

/* *** Device.WiFi.DataElements.Network. *** */
DMOBJ tWiFiDataElementsNetworkObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"SSID", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkSSIDInst, NULL, NULL, NULL, tWiFiDataElementsNetworkSSIDParams, NULL, BBFDM_BOTH, NULL},
{"ProvisionedDPP", &DMWRITE, addObjWiFiDataElementsNetworkProvisionedDPP, delObjWiFiDataElementsNetworkProvisionedDPP, NULL, browseWiFiDataElementsNetworkProvisionedDPPInst, NULL, NULL, NULL, tWiFiDataElementsNetworkProvisionedDPPParams, NULL, BBFDM_BOTH, NULL},
{"MultiAPSteeringSummaryStats", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkMultiAPSteeringSummaryStatsParams, NULL, BBFDM_BOTH, NULL},
{"Device", &DMREAD, NULL, NULL, NULL, browseWiFiDataElementsNetworkDeviceInst, NULL, NULL, tWiFiDataElementsNetworkDeviceObj, tWiFiDataElementsNetworkDeviceParams, NULL, BBFDM_BOTH, NULL},
{0}
};

DMLEAF tWiFiDataElementsNetworkParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"ID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetwork_ID, NULL, BBFDM_BOTH},
{"TimeStamp", &DMREAD, DMT_TIME, get_WiFiDataElementsNetwork_TimeStamp, NULL, BBFDM_BOTH},
{"ControllerID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetwork_ControllerID, NULL, BBFDM_BOTH},
{"DeviceNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetwork_DeviceNumberOfEntries, NULL, BBFDM_BOTH},
//{"MSCSDisallowedStaList", &DMREAD, DMT_STRING, get_WiFiDataElementsNetwork_MSCSDisallowedStaList, NULL, BBFDM_BOTH},
//{"SCSDisallowedStaList", &DMREAD, DMT_STRING, get_WiFiDataElementsNetwork_SCSDisallowedStaList, NULL, BBFDM_BOTH},
{"ColocatedAgentID", &DMREAD, DMT_STRING, get_WiFiDataElementsNetwork_ColocatedAgentID, NULL, BBFDM_BOTH},
{"SSIDNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetwork_SSIDNumberOfEntries, NULL, BBFDM_BOTH},
{"ProvisionedDPPNumberOfEntries", &DMREAD, DMT_UNINT, get_WiFiDataElementsNetwork_ProvisionedDPPNumberOfEntries, NULL, BBFDM_BOTH},
{"SetTrafficSeparation()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetwork_SetTrafficSeparation, operate_WiFiDataElementsNetwork_SetTrafficSeparation, BBFDM_USP},
//{"SetServicePrioritization()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetwork_SetServicePrioritization, operate_WiFiDataElementsNetwork_SetServicePrioritization, BBFDM_USP},
//{"SetPreferredBackhauls()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetwork_SetPreferredBackhauls, operate_WiFiDataElementsNetwork_SetPreferredBackhauls, BBFDM_USP},
{"SetSSID()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetwork_SetSSID, operate_WiFiDataElementsNetwork_SetSSID, BBFDM_USP},
//{"SetMSCSDisallowed()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetwork_SetMSCSDisallowed, operate_WiFiDataElementsNetwork_SetMSCSDisallowed, BBFDM_USP},
//{"SetSCSDisallowed()", &DMASYNC, DMT_COMMAND, get_operate_args_WiFiDataElementsNetwork_SetSCSDisallowed, operate_WiFiDataElementsNetwork_SetSCSDisallowed, BBFDM_USP},
{"AgentOnboard!", &DMREAD, DMT_EVENT, get_event_args_WiFiDataElementsNetwork_AgentOnboard, NULL, BBFDM_USP},
{"ControllerBTMSteer!", &DMREAD, DMT_EVENT, get_event_args_WiFiDataElementsNetwork_ControllerBTMSteer, event_WiFiDataElementsNetwork_ControllerBTMSteer, BBFDM_USP},
{0}
};

/* *** Device.WiFi.DataElements.AssociationEvent. *** */
DMLEAF tWiFiDataElementsAssociationEventParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Associated!", &DMREAD, DMT_EVENT, get_event_args_WiFiDataElementsAssociationEvent_Associated, NULL, BBFDM_USP},
{0}
};

/* *** Device.WiFi.DataElements.DisassociationEvent. *** */
DMLEAF tWiFiDataElementsDisassociationEventParams[] = {
/* PARAM, permission, type, getvalue, setvalue, bbfdm_type, version*/
{"Disassociated!", &DMREAD, DMT_EVENT, get_event_args_WiFiDataElementsDisassociationEvent_Disassociated, NULL, BBFDM_USP},
{0}
};

/* *** Device.WiFi.DataElements. *** */
DMOBJ tWiFiDataElementsObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"Network", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsNetworkObj, tWiFiDataElementsNetworkParams, NULL, BBFDM_BOTH},
{"AssociationEvent", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsAssociationEventParams, NULL, BBFDM_BOTH, NULL},
{"DisassociationEvent", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsDisassociationEventParams, NULL, BBFDM_BOTH, NULL},
//{"FailedConnectionEvent", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsFailedConnectionEventParams, NULL, BBFDM_BOTH, NULL},
{0}
};

/* *** Device.WiFi. *** */
DMOBJ tDeviceWiFiDataElementsObj[] = {
/* OBJ, permission, addobj, delobj, checkdep, browseinstobj, nextdynamicobj, dynamicleaf, nextobj, leaf, linker, bbfdm_type, uniqueKeys, version*/
{"DataElements", &DMREAD, NULL, NULL, NULL, NULL, NULL, NULL, tWiFiDataElementsObj, NULL, NULL, BBFDM_BOTH, NULL},
{0}
};

/* ********** DynamicObj ********** */
DM_MAP_OBJ tDynamicObj[] = {
/* parentobj, nextobject, parameter */
{"Device.WiFi.", tDeviceWiFiDataElementsObj, NULL, init_wifi_dataelement_module, clean_wifi_dataelement_module, NULL},
{0}
};
